/*
 This file is part of GNU Taler
 (C) 2021 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { URL, j2s } from "@gnu-taler/taler-util";
import {
  ExchangeService,
  GlobalTestState,
  MERCHANT_DEFAULT_AUTH,
  MerchantService,
  getTestHarnessPaytoForLabel,
  harnessHttpLib,
  setupDb,
} from "../harness/harness.js";

export async function runMerchantCategoriesTest(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    httpPort: 8083,
    database: db.connStr,
  });

  // We add the exchange to the config, but note that the exchange won't be started.
  merchant.addExchange(exchange);

  await merchant.start();
  await merchant.pingUntilAvailable();

  // Base URL for the default instance.
  const baseUrl = merchant.makeInstanceBaseUrl();

  {
    const r = await harnessHttpLib.fetch(new URL("config", baseUrl).href);
    const data = await r.json();
    console.log(data);
    t.assertDeepEqual(data.currency, "TESTKUDOS");
  }

  // Instances should initially be empty
  {
    const r = await harnessHttpLib.fetch(
      new URL("management/instances", baseUrl).href,
    );
    const data = await r.json();
    t.assertDeepEqual(data.instances, []);
  }

  const { accessToken: adminAccessToken } =
    await merchant.addInstanceWithWireAccount({
      id: "admin",
      name: "Default Instance",
      paytoUris: [getTestHarnessPaytoForLabel("merchant-default")],
      auth: MERCHANT_DEFAULT_AUTH,
    });

  let myNewCategoryId: number;

  {
    const url = new URL("private/categories", merchant.makeInstanceBaseUrl());
    const res = await harnessHttpLib.fetch(url.href, {
      method: "POST",
      body: {
        name: "Snacks",
        name_i18n: {},
      },
      headers: {
        Authorization: `Bearer ${adminAccessToken}`,
      },
    });

    console.log(res.requestUrl);
    console.log("status", res.status);
    const categoryJson = await res.json();
    console.log(categoryJson);
    t.assertTrue(res.status >= 200 && res.status < 300);
    myNewCategoryId = categoryJson.category_id;
  }

  {
    const url = new URL("private/products", merchant.makeInstanceBaseUrl());
    const res = await harnessHttpLib.fetch(url.href, {
      method: "POST",
      body: {
        product_id: "foo",
        description: "Bla Bla",
        unit: "item",
        price: "TESTKUDOS:6",
        total_stock: -1,
      },
      headers: {
        Authorization: `Bearer ${adminAccessToken}`,
      },
    });
    t.assertTrue(res.status >= 200 && res.status < 300);
  }

  {
    const url = new URL("private/products", merchant.makeInstanceBaseUrl());
    const res = await harnessHttpLib.fetch(url.href, {
      method: "POST",
      body: {
        product_id: "bar",
        description: "Bla Bla",
        unit: "item",
        price: "TESTKUDOS:2",
        total_stock: -1,
        categories: [myNewCategoryId],
      },
      headers: {
        Authorization: `Bearer ${adminAccessToken}`,
      },
    });
    t.assertTrue(res.status >= 200 && res.status < 300);
  }

  {
    const url = new URL("private/products", merchant.makeInstanceBaseUrl());
    const res = await harnessHttpLib.fetch(url.href, {
      method: "POST",
      body: {
        product_id: "baz",
        description: "Eggs",
        unit: "item",
        price: "TESTKUDOS:42",
        total_stock: -1,
      },
      headers: {
        Authorization: `Bearer ${adminAccessToken}`,
      },
    });
    t.assertTrue(res.status >= 200 && res.status < 300);
  }

  {
    const posUrl = new URL("private/pos", merchant.makeInstanceBaseUrl());
    const res = await harnessHttpLib.fetch(posUrl.href, {
      method: "GET",
      headers: {
        Authorization: `Bearer ${adminAccessToken}`,
      },
    });
    const posJson = await res.json();
    console.log(j2s(posJson));
    t.assertTrue(res.status >= 200 && res.status < 300);

    t.assertDeepEqual(posJson.products.length, 3);

    const prodFoo = posJson.products.find((x: any) => x.product_id == "foo");
    console.log(`prod foo`, prodFoo);
    t.assertTrue(!!prodFoo);
    // Only default category
    t.assertDeepEqual(prodFoo.categories, [0]);

    const prodBar = posJson.products.find((x: any) => x.product_id == "bar");
    console.log(`prod bar`, prodBar);
    t.assertTrue(!!prodBar);
    // This should have the one we assigned to it.
    t.assertDeepEqual(prodBar.categories, [myNewCategoryId]);
  }
}

runMerchantCategoriesTest.suites = ["merchant"];
