/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.kura.core.db;

import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import javax.sql.ConnectionPoolDataSource;
import org.eclipse.kura.KuraException;
import org.eclipse.kura.configuration.ConfigurableComponent;
import org.eclipse.kura.core.db.H2DbServiceOptions;
import org.eclipse.kura.crypto.CryptoService;
import org.eclipse.kura.db.H2DbService;
import org.h2.jdbcx.JdbcConnectionPool;
import org.h2.jdbcx.JdbcDataSource;
import org.h2.tools.DeleteDbFiles;
import org.osgi.service.component.ComponentException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class H2DbServiceImpl
implements H2DbService,
ConfigurableComponent {
    private static final String ANONYMOUS_MEM_INSTANCE_JDBC_URL = "jdbc:h2:mem:";
    private static Map<String, H2DbServiceImpl> activeInstances = Collections.synchronizedMap(new HashMap());
    private static final int MAX_LENGTH_INPLACE_LOB_VALUE = 2000000000;
    private static Logger logger = LoggerFactory.getLogger(H2DbServiceImpl.class);
    private H2DbServiceOptions configuration;
    private JdbcDataSource dataSource;
    private JdbcConnectionPool connectionPool;
    private char[] lastSessionPassword = null;
    private CryptoService cryptoService;
    private ScheduledExecutorService executor;
    private ScheduledFuture<?> checkpointTask;
    private ScheduledFuture<?> defragTask;
    private final ReadWriteLock rwLock = new ReentrantReadWriteLock(true);
    private final AtomicInteger pendingUpdates = new AtomicInteger();
    private final ThreadLocal<Boolean> isOnExecutor = new ThreadLocal<Boolean>(){

        @Override
        protected Boolean initialValue() {
            return false;
        }
    };
    private final ThreadPoolExecutor executorService = new ThreadPoolExecutor(0, 10, 0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>());

    static {
        try {
            DriverManager.registerDriver((Driver)new org.h2.Driver());
        }
        catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    public void setCryptoService(CryptoService cryptoService) {
        this.cryptoService = cryptoService;
    }

    public void unsetCryptoService(CryptoService cryptoService) {
        this.cryptoService = null;
    }

    public void activate(Map<String, Object> properties) {
        logger.info("activating...");
        String kuraServicePid = (String)properties.get("kura.service.pid");
        ThreadFactory defaultFactory = this.executorService.getThreadFactory();
        AtomicInteger threadNumber = new AtomicInteger();
        this.executorService.setThreadFactory(r -> {
            Thread result = defaultFactory.newThread(() -> {
                this.isOnExecutor.set(true);
                r.run();
            });
            result.setName("H2DbService_" + kuraServicePid + "_" + threadNumber.getAndIncrement());
            return result;
        });
        this.executor = Executors.newSingleThreadScheduledExecutor();
        this.updated(properties);
        logger.info("activating...done");
    }

    public void updated(Map<String, Object> properties) {
        this.pendingUpdates.incrementAndGet();
        this.executor.submit(() -> this.updateInternal(properties));
    }

    public void deactivate() {
        logger.info("deactivate...");
        this.executor.shutdown();
        try {
            this.executor.awaitTermination(1L, TimeUnit.MINUTES);
        }
        catch (InterruptedException interruptedException) {
            logger.warn("Interrupted while waiting for db shutdown");
            Thread.currentThread().interrupt();
        }
        this.executorService.shutdown();
        this.awaitExecutorServiceTermination();
        try {
            this.shutdownDb();
        }
        catch (SQLException e) {
            logger.warn("got exception while shutting down the database", (Throwable)e);
        }
        logger.info("deactivate...done");
    }

    public Connection getConnection() throws SQLException {
        if (this.pendingUpdates.get() > 0) {
            this.syncWithExecutor();
        }
        Lock lock = this.rwLock.readLock();
        lock.lock();
        try {
            Connection connection = this.getConnectionInternal();
            return connection;
        }
        finally {
            lock.unlock();
        }
    }

    private <T> T withConnectionInternal(H2DbService.ConnectionCallable<T> callable) throws SQLException {
        Lock executorlock = this.rwLock.readLock();
        executorlock.lock();
        Connection connection = null;
        try {
            connection = this.getConnectionInternal();
            Object object = callable.call(connection);
            return (T)object;
        }
        catch (SQLException e) {
            logger.warn("Db operation failed");
            this.rollback(connection);
            throw e;
        }
        finally {
            this.close(connection);
            executorlock.unlock();
        }
    }

    public <T> T withConnection(H2DbService.ConnectionCallable<T> callable) throws SQLException {
        if (this.pendingUpdates.get() > 0) {
            this.syncWithExecutor();
        }
        if (this.isOnExecutor.get().booleanValue()) {
            return this.withConnectionInternal(callable);
        }
        Future<Object> result = this.executorService.submit(() -> this.withConnectionInternal(callable));
        try {
            return (T)result.get();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new SQLException(e);
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof SQLException) {
                throw (SQLException)e.getCause();
            }
            if (e.getCause() instanceof RuntimeException) {
                throw (RuntimeException)e.getCause();
            }
            throw new IllegalStateException(e);
        }
    }

    public void rollback(Connection conn) {
        try {
            if (conn != null) {
                conn.rollback();
            }
        }
        catch (SQLException e) {
            logger.error("Error during Connection rollback.", (Throwable)e);
        }
    }

    public void close(ResultSet ... rss) {
        if (rss != null) {
            ResultSet[] resultSetArray = rss;
            int n = rss.length;
            int n2 = 0;
            while (n2 < n) {
                ResultSet rs = resultSetArray[n2];
                try {
                    if (rs != null) {
                        rs.close();
                    }
                }
                catch (SQLException e) {
                    logger.error("Error during ResultSet closing", (Throwable)e);
                }
                ++n2;
            }
        }
    }

    public void close(Statement ... stmts) {
        if (stmts != null) {
            Statement[] statementArray = stmts;
            int n = stmts.length;
            int n2 = 0;
            while (n2 < n) {
                Statement stmt = statementArray[n2];
                try {
                    if (stmt != null) {
                        stmt.close();
                    }
                }
                catch (SQLException e) {
                    logger.error("Error during Statement closing", (Throwable)e);
                }
                ++n2;
            }
        }
    }

    public void close(Connection conn) {
        try {
            if (conn != null) {
                conn.close();
            }
        }
        catch (SQLException e) {
            logger.error("Error during Connection closing", (Throwable)e);
        }
    }

    private void updateInternal(Map<String, Object> properties) {
        Lock lock = this.rwLock.writeLock();
        lock.lock();
        try {
            try {
                char[] password;
                logger.info("updating...");
                H2DbServiceOptions newConfiguration = new H2DbServiceOptions(properties);
                if (this.configuration != null) {
                    boolean userChanged;
                    boolean urlChanged = !this.configuration.getDbUrl().equals(newConfiguration.getDbUrl());
                    boolean bl = userChanged = !this.configuration.getUser().equalsIgnoreCase(newConfiguration.getUser());
                    if (urlChanged || userChanged) {
                        this.shutdownDb();
                    }
                }
                if (newConfiguration.isRemote()) {
                    throw new IllegalArgumentException("Remote databases are not supported");
                }
                String baseUrl = newConfiguration.getBaseUrl();
                if (baseUrl.equals(ANONYMOUS_MEM_INSTANCE_JDBC_URL)) {
                    throw new IllegalArgumentException("Anonymous in-memory databases instances are not supported");
                }
                if (this.isManagedByAnotherInstance(baseUrl)) {
                    throw new IllegalStateException("Another H2DbService instance is managing the same DB URL, please change the DB URL or deactivate the other instance");
                }
                char[] passwordFromConfig = newConfiguration.getEncryptedPassword();
                char[] cArray = password = this.lastSessionPassword != null ? this.lastSessionPassword : passwordFromConfig;
                if (this.connectionPool == null) {
                    this.openConnectionPool(newConfiguration, this.decryptPassword(password));
                    this.lastSessionPassword = password;
                }
                this.setParameters(newConfiguration);
                if (!newConfiguration.isZipBased() && !Arrays.equals(password, passwordFromConfig)) {
                    String decryptedPassword = this.decryptPassword(passwordFromConfig);
                    this.changePassword(newConfiguration.getUser(), decryptedPassword);
                    this.dataSource.setPassword(decryptedPassword);
                    this.lastSessionPassword = passwordFromConfig;
                }
                if (newConfiguration.isFileBased()) {
                    this.restartCheckpointTask(newConfiguration);
                    this.restartDefragTask(newConfiguration);
                }
                if (this.configuration == null || newConfiguration.getConnectionPoolMaxSize() != this.configuration.getConnectionPoolMaxSize()) {
                    this.executorService.setMaximumPoolSize(newConfiguration.getConnectionPoolMaxSize());
                }
                this.configuration = newConfiguration;
                activeInstances.put(baseUrl, this);
                logger.info("updating...done");
            }
            catch (Exception e) {
                this.disposeConnectionPool();
                this.stopCheckpointTask();
                logger.error("Database initialization failed", (Throwable)e);
                lock.unlock();
                this.pendingUpdates.decrementAndGet();
            }
        }
        finally {
            lock.unlock();
            this.pendingUpdates.decrementAndGet();
        }
    }

    private void awaitExecutorServiceTermination() {
        try {
            this.executorService.awaitTermination(1L, TimeUnit.MINUTES);
        }
        catch (InterruptedException interruptedException) {
            logger.warn("Interrupted while waiting for db shutdown");
            Thread.currentThread().interrupt();
        }
    }

    private void setParameters(H2DbServiceOptions configuration) throws SQLException {
        if (!configuration.isFileBasedLogLevelSpecified()) {
            this.executeInternal("SET TRACE_LEVEL_FILE 0");
        }
        if (configuration.isInMemory()) {
            this.executeInternal("SET MAX_LENGTH_INPLACE_LOB 2000000000");
        }
        this.connectionPool.setMaxConnections(configuration.getConnectionPoolMaxSize());
    }

    private void syncWithExecutor() {
        try {
            this.executor.submit(() -> {}).get();
        }
        catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }

    private Connection getConnectionInternal() throws SQLException {
        if (this.connectionPool == null) {
            throw new SQLException("Database instance not initialized");
        }
        Connection conn = null;
        try {
            conn = this.connectionPool.getConnection();
        }
        catch (SQLException e) {
            logger.error("Error getting connection", (Throwable)e);
            throw e;
        }
        return conn;
    }

    private void executeInternal(String sql) throws SQLException {
        Connection conn = null;
        Statement stmt = null;
        try {
            try {
                conn = this.getConnectionInternal();
                stmt = conn.createStatement();
                stmt.execute(sql);
                conn.commit();
            }
            catch (SQLException e) {
                this.rollback(conn);
                throw e;
            }
        }
        catch (Throwable throwable) {
            this.close(stmt);
            this.close(conn);
            throw throwable;
        }
        this.close(stmt);
        this.close(conn);
    }

    private void shutdownDb() throws SQLException {
        this.lastSessionPassword = null;
        if (this.connectionPool == null) {
            return;
        }
        this.stopDefragTask();
        this.stopCheckpointTask();
        Connection conn = null;
        Statement stmt = null;
        try {
            conn = this.dataSource.getConnection();
            stmt = conn.createStatement();
            stmt.execute("SHUTDOWN");
        }
        catch (Throwable throwable) {
            this.close(stmt);
            this.close(conn);
            throw throwable;
        }
        this.close(stmt);
        this.close(conn);
        this.disposeConnectionPool();
        activeInstances.remove(this.configuration.getBaseUrl());
    }

    private void openConnectionPool(H2DbServiceOptions configuration, String password) {
        logger.info("Opening database with url: {}", (Object)configuration.getDbUrl());
        this.dataSource = new JdbcDataSource();
        this.dataSource.setURL(configuration.getDbUrl());
        this.dataSource.setUser(configuration.getUser());
        this.dataSource.setPassword(password);
        this.connectionPool = JdbcConnectionPool.create((ConnectionPoolDataSource)this.dataSource);
        this.openDatabase(configuration, true);
    }

    private void openDatabase(H2DbServiceOptions configuration, boolean deleteDbOnError) {
        Connection conn = null;
        try {
            try {
                conn = this.getConnectionInternal();
            }
            catch (SQLException e) {
                logger.error("Failed to open database", (Throwable)e);
                if (!deleteDbOnError || !configuration.isFileBased()) {
                    this.disposeConnectionPool();
                    throw new ComponentException((Throwable)e);
                }
                logger.warn("Deleting database files...");
                this.deleteDbFiles(configuration);
                logger.warn("Deleting database files...done");
                this.openDatabase(configuration, false);
                this.close(conn);
            }
        }
        finally {
            this.close(conn);
        }
    }

    private void deleteDbFiles(H2DbServiceOptions configuration) {
        try {
            String directory = configuration.getDbDirectory();
            String dbName = configuration.getDatabaseName();
            if (directory == null || dbName == null) {
                logger.warn("Failed to determine database directory or name, not deleting db");
                return;
            }
            DeleteDbFiles.execute((String)directory, (String)dbName, (boolean)false);
        }
        catch (Exception e) {
            logger.warn("Failed to remove DB files", (Throwable)e);
        }
    }

    private void disposeConnectionPool() {
        if (this.connectionPool != null) {
            this.connectionPool.dispose();
            this.connectionPool = null;
        }
    }

    private String decryptPassword(char[] encryptedPassword) throws KuraException {
        char[] decodedPasswordChars = this.cryptoService.decryptAes(encryptedPassword);
        return new String(decodedPasswordChars);
    }

    private void changePassword(String user, String newPassword) throws SQLException {
        this.executeInternal("ALTER USER " + user + " SET PASSWORD '" + newPassword + "'");
    }

    private boolean isManagedByAnotherInstance(String baseUrl) {
        H2DbServiceImpl owner = activeInstances.get(baseUrl);
        return owner != null && owner != this;
    }

    private void restartCheckpointTask(H2DbServiceOptions config) {
        this.stopCheckpointTask();
        long delaySeconds = config.getCheckpointIntervalSeconds();
        if (delaySeconds <= 0L) {
            return;
        }
        this.checkpointTask = this.executor.scheduleWithFixedDelay(new CheckpointTask(), delaySeconds, delaySeconds, TimeUnit.SECONDS);
    }

    private void stopCheckpointTask() {
        if (this.checkpointTask != null) {
            this.checkpointTask.cancel(false);
            this.checkpointTask = null;
        }
    }

    private void restartDefragTask(H2DbServiceOptions config) {
        this.stopDefragTask();
        long delayMinutes = config.getDefragIntervalMinutes();
        if (delayMinutes <= 0L) {
            return;
        }
        this.checkpointTask = this.executor.scheduleWithFixedDelay(new DefragTask(config), delayMinutes, delayMinutes, TimeUnit.MINUTES);
    }

    private void stopDefragTask() {
        if (this.defragTask != null) {
            this.defragTask.cancel(false);
            this.defragTask = null;
        }
    }

    private class CheckpointTask
    implements Runnable {
        private CheckpointTask() {
        }

        @Override
        public void run() {
            try {
                logger.info("performing checkpoint...");
                H2DbServiceImpl.this.executeInternal("CHECKPOINT SYNC");
                logger.info("performing checkpoint...done");
            }
            catch (SQLException e) {
                logger.error("checkpoint failed", (Throwable)e);
            }
        }
    }

    private class DefragTask
    implements Runnable {
        private final H2DbServiceOptions configuration;

        public DefragTask(H2DbServiceOptions configuration) {
            this.configuration = configuration;
        }

        private void shutdownDefrag() throws SQLException {
            Connection conn = null;
            Statement stmt = null;
            try {
                conn = H2DbServiceImpl.this.dataSource.getConnection();
                stmt = conn.createStatement();
                stmt.execute("SHUTDOWN DEFRAG");
            }
            catch (Throwable throwable) {
                H2DbServiceImpl.this.close(stmt);
                H2DbServiceImpl.this.close(conn);
                throw throwable;
            }
            H2DbServiceImpl.this.close(stmt);
            H2DbServiceImpl.this.close(conn);
        }

        @Override
        public void run() {
            Lock lock = H2DbServiceImpl.this.rwLock.writeLock();
            lock.lock();
            try {
                try {
                    logger.info("shutting down and defragmenting db...");
                    this.shutdownDefrag();
                    H2DbServiceImpl.this.disposeConnectionPool();
                    String password = H2DbServiceImpl.this.decryptPassword(this.configuration.getEncryptedPassword());
                    H2DbServiceImpl.this.openConnectionPool(this.configuration, password);
                    logger.info("shutting down and defragmenting db...done");
                }
                catch (Exception e) {
                    logger.error("failed to shutdown and defrag db", (Throwable)e);
                    lock.unlock();
                }
            }
            finally {
                lock.unlock();
            }
        }
    }
}

