/*
 * Copyright (c) 2013, 2018, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package java.time.format;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

/**
 * A helper class to map a zone name to metazone and back to the
 * appropriate zone id for the particular locale.
 * <p>
 * The zid<->metazone mappings are based on CLDR metaZones.xml.
 * The alias mappings are based on Link entries in tzdb data files and
 * CLDR's supplementalMetadata.xml.
 */
class ZoneName {

    public static String toZid(String zid, Locale locale) {
        String mzone = zidToMzone.get(zid);
        if (mzone == null && aliases.containsKey(zid)) {
            zid = aliases.get(zid);
            mzone = zidToMzone.get(zid);
        }
        if (mzone != null) {
            Map<String, String> map = mzoneToZidL.get(mzone);
            if (map != null && map.containsKey(locale.getCountry())) {
                zid = map.get(locale.getCountry());
            } else {
                zid = mzoneToZid.get(mzone);
            }
        }
        return toZid(zid);
    }

    public static String toZid(String zid) {
        if (aliases.containsKey(zid)) {
            return aliases.get(zid);
        }
        return zid;
    }

    private static final String[] zidMap = new String[] {
        // From metaZones.xml
        "Africa/Abidjan", "GMT", "Atlantic/Reykjavik",
        "Africa/Accra", "GMT", "Atlantic/Reykjavik",
        "Africa/Addis_Ababa", "Africa_Eastern", "Africa/Nairobi",
        "Africa/Algiers", "Europe_Central", "Europe/Paris",
        "Africa/Asmara", "Africa_Eastern", "Africa/Nairobi",
        "Africa/Asmera", "Africa_Eastern", "Africa/Nairobi",
        "Africa/Bamako", "GMT", "Atlantic/Reykjavik",
        "Africa/Bangui", "Africa_Western", "Africa/Lagos",
        "Africa/Banjul", "GMT", "Atlantic/Reykjavik",
        "Africa/Bissau", "GMT", "Atlantic/Reykjavik",
        "Africa/Blantyre", "Africa_Central", "Africa/Maputo",
        "Africa/Brazzaville", "Africa_Western", "Africa/Lagos",
        "Africa/Bujumbura", "Africa_Central", "Africa/Maputo",
        "Africa/Cairo", "Europe_Eastern", "Europe/Bucharest",
        "Africa/Ceuta", "Europe_Central", "Europe/Paris",
        "Africa/Conakry", "GMT", "Atlantic/Reykjavik",
        "Africa/Dakar", "GMT", "Atlantic/Reykjavik",
        "Africa/Dar_es_Salaam", "Africa_Eastern", "Africa/Nairobi",
        "Africa/Djibouti", "Africa_Eastern", "Africa/Nairobi",
        "Africa/Douala", "Africa_Western", "Africa/Lagos",
        "Africa/Freetown", "GMT", "Atlantic/Reykjavik",
        "Africa/Gaborone", "Africa_Central", "Africa/Maputo",
        "Africa/Harare", "Africa_Central", "Africa/Maputo",
        "Africa/Johannesburg", "Africa_Southern", "Africa/Johannesburg",
        "Africa/Juba", "Africa_Central", "Africa/Maputo",
        "Africa/Kampala", "Africa_Eastern", "Africa/Nairobi",
        "Africa/Khartoum", "Africa_Central", "Africa/Maputo",
        "Africa/Kigali", "Africa_Central", "Africa/Maputo",
        "Africa/Kinshasa", "Africa_Western", "Africa/Lagos",
        "Africa/Lagos", "Africa_Western", "Africa/Lagos",
        "Africa/Libreville", "Africa_Western", "Africa/Lagos",
        "Africa/Lome", "GMT", "Atlantic/Reykjavik",
        "Africa/Luanda", "Africa_Western", "Africa/Lagos",
        "Africa/Lubumbashi", "Africa_Central", "Africa/Maputo",
        "Africa/Lusaka", "Africa_Central", "Africa/Maputo",
        "Africa/Malabo", "Africa_Western", "Africa/Lagos",
        "Africa/Maputo", "Africa_Central", "Africa/Maputo",
        "Africa/Maseru", "Africa_Southern", "Africa/Johannesburg",
        "Africa/Mbabane", "Africa_Southern", "Africa/Johannesburg",
        "Africa/Mogadishu", "Africa_Eastern", "Africa/Nairobi",
        "Africa/Monrovia", "GMT", "Atlantic/Reykjavik",
        "Africa/Nairobi", "Africa_Eastern", "Africa/Nairobi",
        "Africa/Ndjamena", "Africa_Western", "Africa/Lagos",
        "Africa/Niamey", "Africa_Western", "Africa/Lagos",
        "Africa/Nouakchott", "GMT", "Atlantic/Reykjavik",
        "Africa/Ouagadougou", "GMT", "Atlantic/Reykjavik",
        "Africa/Porto-Novo", "Africa_Western", "Africa/Lagos",
        "Africa/Sao_Tome", "GMT", "Atlantic/Reykjavik",
        "Africa/Timbuktu", "GMT", "Atlantic/Reykjavik",
        "Africa/Tripoli", "Europe_Eastern", "Europe/Bucharest",
        "Africa/Tunis", "Europe_Central", "Europe/Paris",
        "Africa/Windhoek", "Africa_Central", "Africa/Maputo",
        "America/Adak", "Hawaii_Aleutian", "Pacific/Honolulu",
        "America/Anchorage", "Alaska", "America/Juneau",
        "America/Anguilla", "Atlantic", "America/Halifax",
        "America/Antigua", "Atlantic", "America/Halifax",
        "America/Araguaina", "Brasilia", "America/Sao_Paulo",
        "America/Argentina/Buenos_Aires", "Argentina", "America/Buenos_Aires",
        "America/Argentina/Catamarca", "Argentina", "America/Buenos_Aires",
        "America/Argentina/ComodRivadavia", "Argentina", "America/Buenos_Aires",
        "America/Argentina/Cordoba", "Argentina", "America/Buenos_Aires",
        "America/Argentina/Jujuy", "Argentina", "America/Buenos_Aires",
        "America/Argentina/La_Rioja", "Argentina", "America/Buenos_Aires",
        "America/Argentina/Mendoza", "Argentina", "America/Buenos_Aires",
        "America/Argentina/Rio_Gallegos", "Argentina", "America/Buenos_Aires",
        "America/Argentina/Salta", "Argentina", "America/Buenos_Aires",
        "America/Argentina/San_Juan", "Argentina", "America/Buenos_Aires",
        "America/Argentina/San_Luis", "Argentina", "America/Buenos_Aires",
        "America/Argentina/Tucuman", "Argentina", "America/Buenos_Aires",
        "America/Argentina/Ushuaia", "Argentina", "America/Buenos_Aires",
        "America/Aruba", "Atlantic", "America/Halifax",
        "America/Asuncion", "Paraguay", "America/Asuncion",
        "America/Atikokan", "America_Eastern", "America/New_York",
        "America/Atka", "Hawaii_Aleutian", "Pacific/Honolulu",
        "America/Bahia", "Brasilia", "America/Sao_Paulo",
        "America/Bahia_Banderas", "America_Central", "America/Chicago",
        "America/Barbados", "Atlantic", "America/Halifax",
        "America/Belem", "Brasilia", "America/Sao_Paulo",
        "America/Belize", "America_Central", "America/Chicago",
        "America/Blanc-Sablon", "Atlantic", "America/Halifax",
        "America/Boa_Vista", "Amazon", "America/Manaus",
        "America/Bogota", "Colombia", "America/Bogota",
        "America/Boise", "America_Mountain", "America/Denver",
        "America/Buenos_Aires", "Argentina", "America/Buenos_Aires",
        "America/Cambridge_Bay", "America_Mountain", "America/Denver",
        "America/Campo_Grande", "Amazon", "America/Manaus",
        "America/Cancun", "America_Eastern", "America/New_York",
        "America/Caracas", "Venezuela", "America/Caracas",
        "America/Catamarca", "Argentina", "America/Buenos_Aires",
        "America/Cayenne", "French_Guiana", "America/Cayenne",
        "America/Cayman", "America_Eastern", "America/New_York",
        "America/Chicago", "America_Central", "America/Chicago",
        "America/Chihuahua", "Mexico_Pacific", "America/Mazatlan",
        "America/Coral_Harbour", "America_Eastern", "America/New_York",
        "America/Cordoba", "Argentina", "America/Buenos_Aires",
        "America/Costa_Rica", "America_Central", "America/Chicago",
        "America/Creston", "America_Mountain", "America/Denver",
        "America/Cuiaba", "Amazon", "America/Manaus",
        "America/Curacao", "Atlantic", "America/Halifax",
        "America/Danmarkshavn", "GMT", "Atlantic/Reykjavik",
        "America/Dawson", "America_Mountain", "America/Denver",
        "America/Dawson_Creek", "America_Mountain", "America/Denver",
        "America/Denver", "America_Mountain", "America/Denver",
        "America/Detroit", "America_Eastern", "America/New_York",
        "America/Dominica", "Atlantic", "America/Halifax",
        "America/Edmonton", "America_Mountain", "America/Denver",
        "America/Eirunepe", "Acre", "America/Rio_Branco",
        "America/El_Salvador", "America_Central", "America/Chicago",
        "America/Ensenada", "America_Pacific", "America/Los_Angeles",
        "America/Fort_Nelson", "America_Mountain", "America/Denver",
        "America/Fort_Wayne", "America_Eastern", "America/New_York",
        "America/Fortaleza", "Brasilia", "America/Sao_Paulo",
        "America/Glace_Bay", "Atlantic", "America/Halifax",
        "America/Godthab", "Greenland_Western", "America/Godthab",
        "America/Goose_Bay", "Atlantic", "America/Halifax",
        "America/Grand_Turk", "America_Eastern", "America/New_York",
        "America/Grenada", "Atlantic", "America/Halifax",
        "America/Guadeloupe", "Atlantic", "America/Halifax",
        "America/Guatemala", "America_Central", "America/Chicago",
        "America/Guayaquil", "Ecuador", "America/Guayaquil",
        "America/Guyana", "Guyana", "America/Guyana",
        "America/Halifax", "Atlantic", "America/Halifax",
        "America/Havana", "Cuba", "America/Havana",
        "America/Hermosillo", "Mexico_Pacific", "America/Mazatlan",
        "America/Indiana/Indianapolis", "America_Eastern", "America/New_York",
        "America/Indiana/Knox", "America_Central", "America/Chicago",
        "America/Indiana/Marengo", "America_Eastern", "America/New_York",
        "America/Indiana/Petersburg", "America_Eastern", "America/New_York",
        "America/Indiana/Tell_City", "America_Central", "America/Chicago",
        "America/Indiana/Vevay", "America_Eastern", "America/New_York",
        "America/Indiana/Vincennes", "America_Eastern", "America/New_York",
        "America/Indiana/Winamac", "America_Eastern", "America/New_York",
        "America/Indianapolis", "America_Eastern", "America/New_York",
        "America/Inuvik", "America_Mountain", "America/Denver",
        "America/Iqaluit", "America_Eastern", "America/New_York",
        "America/Jamaica", "America_Eastern", "America/New_York",
        "America/Jujuy", "Argentina", "America/Buenos_Aires",
        "America/Juneau", "Alaska", "America/Juneau",
        "America/Kentucky/Louisville", "America_Eastern", "America/New_York",
        "America/Kentucky/Monticello", "America_Eastern", "America/New_York",
        "America/Knox_IN", "America_Central", "America/Chicago",
        "America/Kralendijk", "Atlantic", "America/Halifax",
        "America/La_Paz", "Bolivia", "America/La_Paz",
        "America/Lima", "Peru", "America/Lima",
        "America/Los_Angeles", "America_Pacific", "America/Los_Angeles",
        "America/Louisville", "America_Eastern", "America/New_York",
        "America/Lower_Princes", "Atlantic", "America/Halifax",
        "America/Maceio", "Brasilia", "America/Sao_Paulo",
        "America/Managua", "America_Central", "America/Chicago",
        "America/Manaus", "Amazon", "America/Manaus",
        "America/Marigot", "Atlantic", "America/Halifax",
        "America/Martinique", "Atlantic", "America/Halifax",
        "America/Matamoros", "America_Central", "America/Chicago",
        "America/Mazatlan", "Mexico_Pacific", "America/Mazatlan",
        "America/Mendoza", "Argentina", "America/Buenos_Aires",
        "America/Menominee", "America_Central", "America/Chicago",
        "America/Merida", "America_Central", "America/Chicago",
        "America/Metlakatla", "Alaska", "America/Juneau",
        "America/Mexico_City", "America_Central", "America/Chicago",
        "America/Miquelon", "Pierre_Miquelon", "America/Miquelon",
        "America/Moncton", "Atlantic", "America/Halifax",
        "America/Monterrey", "America_Central", "America/Chicago",
        "America/Montevideo", "Uruguay", "America/Montevideo",
        "America/Montserrat", "Atlantic", "America/Halifax",
        "America/Nassau", "America_Eastern", "America/New_York",
        "America/New_York", "America_Eastern", "America/New_York",
        "America/Nipigon", "America_Eastern", "America/New_York",
        "America/Nome", "Alaska", "America/Juneau",
        "America/Noronha", "Noronha", "America/Noronha",
        "America/North_Dakota/Beulah", "America_Central", "America/Chicago",
        "America/North_Dakota/Center", "America_Central", "America/Chicago",
        "America/North_Dakota/New_Salem", "America_Central", "America/Chicago",
        "America/Nuuk", "Greenland_Western", "America/Godthab",
        "America/Ojinaga", "America_Mountain", "America/Denver",
        "America/Panama", "America_Eastern", "America/New_York",
        "America/Pangnirtung", "America_Eastern", "America/New_York",
        "America/Paramaribo", "Suriname", "America/Paramaribo",
        "America/Phoenix", "America_Mountain", "America/Denver",
        "America/Port-au-Prince", "America_Eastern", "America/New_York",
        "America/Port_of_Spain", "Atlantic", "America/Halifax",
        "America/Porto_Acre", "Acre", "America/Rio_Branco",
        "America/Porto_Velho", "Amazon", "America/Manaus",
        "America/Puerto_Rico", "Atlantic", "America/Halifax",
        "America/Rainy_River", "America_Central", "America/Chicago",
        "America/Rankin_Inlet", "America_Central", "America/Chicago",
        "America/Recife", "Brasilia", "America/Sao_Paulo",
        "America/Regina", "America_Central", "America/Chicago",
        "America/Resolute", "America_Central", "America/Chicago",
        "America/Rio_Branco", "Acre", "America/Rio_Branco",
        "America/Rosario", "Argentina", "America/Buenos_Aires",
        "America/Santa_Isabel", "Mexico_Northwest", "America/Santa_Isabel",
        "America/Santarem", "Brasilia", "America/Sao_Paulo",
        "America/Santiago", "Chile", "America/Santiago",
        "America/Santo_Domingo", "Atlantic", "America/Halifax",
        "America/Sao_Paulo", "Brasilia", "America/Sao_Paulo",
        "America/Scoresbysund", "Greenland_Eastern", "America/Scoresbysund",
        "America/Shiprock", "America_Mountain", "America/Denver",
        "America/Sitka", "Alaska", "America/Juneau",
        "America/St_Barthelemy", "Atlantic", "America/Halifax",
        "America/St_Johns", "Newfoundland", "America/St_Johns",
        "America/St_Kitts", "Atlantic", "America/Halifax",
        "America/St_Lucia", "Atlantic", "America/Halifax",
        "America/St_Thomas", "Atlantic", "America/Halifax",
        "America/St_Vincent", "Atlantic", "America/Halifax",
        "America/Swift_Current", "America_Central", "America/Chicago",
        "America/Tegucigalpa", "America_Central", "America/Chicago",
        "America/Thule", "Atlantic", "America/Halifax",
        "America/Thunder_Bay", "America_Eastern", "America/New_York",
        "America/Tijuana", "America_Pacific", "America/Los_Angeles",
        "America/Toronto", "America_Eastern", "America/New_York",
        "America/Tortola", "Atlantic", "America/Halifax",
        "America/Vancouver", "America_Pacific", "America/Los_Angeles",
        "America/Virgin", "Atlantic", "America/Halifax",
        "America/Whitehorse", "America_Mountain", "America/Denver",
        "America/Winnipeg", "America_Central", "America/Chicago",
        "America/Yakutat", "Alaska", "America/Juneau",
        "America/Yellowknife", "America_Mountain", "America/Denver",
        "Antarctica/Casey", "Casey", "Antarctica/Casey",
        "Antarctica/Davis", "Davis", "Antarctica/Davis",
        "Antarctica/DumontDUrville", "DumontDUrville", "Antarctica/DumontDUrville",
        "Antarctica/Macquarie", "Australia_Eastern", "Australia/Sydney",
        "Antarctica/Mawson", "Mawson", "Antarctica/Mawson",
        "Antarctica/McMurdo", "New_Zealand", "Pacific/Auckland",
        "Antarctica/Rothera", "Rothera", "Antarctica/Rothera",
        "Antarctica/South_Pole", "New_Zealand", "Pacific/Auckland",
        "Antarctica/Syowa", "Syowa", "Antarctica/Syowa",
        "Antarctica/Troll", "GMT", "Atlantic/Reykjavik",
        "Antarctica/Vostok", "Vostok", "Antarctica/Vostok",
        "Arctic/Longyearbyen", "Europe_Central", "Europe/Paris",
        "Asia/Aden", "Arabian", "Asia/Riyadh",
        "Asia/Almaty", "Kazakhstan_Eastern", "Asia/Almaty",
        "Asia/Amman", "Europe_Eastern", "Europe/Bucharest",
        "Asia/Anadyr", "Anadyr", "Asia/Anadyr",
        "Asia/Aqtau", "Kazakhstan_Western", "Asia/Aqtobe",
        "Asia/Aqtobe", "Kazakhstan_Western", "Asia/Aqtobe",
        "Asia/Ashgabat", "Turkmenistan", "Asia/Ashgabat",
        "Asia/Ashkhabad", "Turkmenistan", "Asia/Ashgabat",
        "Asia/Atyrau", "Kazakhstan_Western", "Asia/Aqtobe",
        "Asia/Baghdad", "Arabian", "Asia/Riyadh",
        "Asia/Bahrain", "Arabian", "Asia/Riyadh",
        "Asia/Baku", "Azerbaijan", "Asia/Baku",
        "Asia/Bangkok", "Indochina", "Asia/Bangkok",
        "Asia/Beirut", "Europe_Eastern", "Europe/Bucharest",
        "Asia/Bishkek", "Kyrgystan", "Asia/Bishkek",
        "Asia/Brunei", "Brunei", "Asia/Brunei",
        "Asia/Calcutta", "India", "Asia/Calcutta",
        "Asia/Chita", "Yakutsk", "Asia/Yakutsk",
        "Asia/Choibalsan", "Mongolia", "Asia/Ulaanbaatar",
        "Asia/Chongqing", "China", "Asia/Shanghai",
        "Asia/Chungking", "China", "Asia/Shanghai",
        "Asia/Colombo", "India", "Asia/Calcutta",
        "Asia/Dacca", "Bangladesh", "Asia/Dhaka",
        "Asia/Damascus", "Europe_Eastern", "Europe/Bucharest",
        "Asia/Dhaka", "Bangladesh", "Asia/Dhaka",
        "Asia/Dili", "East_Timor", "Asia/Dili",
        "Asia/Dubai", "Gulf", "Asia/Dubai",
        "Asia/Dushanbe", "Tajikistan", "Asia/Dushanbe",
        "Asia/Gaza", "Europe_Eastern", "Europe/Bucharest",
        "Asia/Harbin", "China", "Asia/Shanghai",
        "Asia/Hebron", "Europe_Eastern", "Europe/Bucharest",
        "Asia/Ho_Chi_Minh", "Indochina", "Asia/Bangkok",
        "Asia/Hong_Kong", "Hong_Kong", "Asia/Hong_Kong",
        "Asia/Hovd", "Hovd", "Asia/Hovd",
        "Asia/Irkutsk", "Irkutsk", "Asia/Irkutsk",
        "Asia/Istanbul", "Turkey", "Europe/Istanbul",
        "Asia/Jakarta", "Indonesia_Western", "Asia/Jakarta",
        "Asia/Jayapura", "Indonesia_Eastern", "Asia/Jayapura",
        "Asia/Jerusalem", "Israel", "Asia/Jerusalem",
        "Asia/Kabul", "Afghanistan", "Asia/Kabul",
        "Asia/Kamchatka", "Kamchatka", "Asia/Kamchatka",
        "Asia/Karachi", "Pakistan", "Asia/Karachi",
        "Asia/Kashgar", "Urumqi", "Asia/Urumqi",
        "Asia/Kathmandu", "Nepal", "Asia/Katmandu",
        "Asia/Katmandu", "Nepal", "Asia/Katmandu",
        "Asia/Khandyga", "Yakutsk", "Asia/Yakutsk",
        "Asia/Kolkata", "India", "Asia/Calcutta",
        "Asia/Krasnoyarsk", "Krasnoyarsk", "Asia/Krasnoyarsk",
        "Asia/Kuala_Lumpur", "Malaysia", "Asia/Kuching",
        "Asia/Kuching", "Malaysia", "Asia/Kuching",
        "Asia/Kuwait", "Arabian", "Asia/Riyadh",
        "Asia/Macao", "China", "Asia/Shanghai",
        "Asia/Macau", "China", "Asia/Shanghai",
        "Asia/Magadan", "Magadan", "Asia/Magadan",
        "Asia/Makassar", "Indonesia_Central", "Asia/Makassar",
        "Asia/Manila", "Philippines", "Asia/Manila",
        "Asia/Muscat", "Gulf", "Asia/Dubai",
        "Asia/Nicosia", "Europe_Eastern", "Europe/Bucharest",
        "Asia/Novokuznetsk", "Krasnoyarsk", "Asia/Krasnoyarsk",
        "Asia/Novosibirsk", "Novosibirsk", "Asia/Novosibirsk",
        "Asia/Omsk", "Omsk", "Asia/Omsk",
        "Asia/Oral", "Kazakhstan_Western", "Asia/Aqtobe",
        "Asia/Phnom_Penh", "Indochina", "Asia/Bangkok",
        "Asia/Pontianak", "Indonesia_Western", "Asia/Jakarta",
        "Asia/Pyongyang", "Korea", "Asia/Seoul",
        "Asia/Qatar", "Arabian", "Asia/Riyadh",
        "Asia/Qostanay", "Kazakhstan_Eastern", "Asia/Almaty",
        "Asia/Qyzylorda", "Kazakhstan_Western", "Asia/Aqtobe",
        "Asia/Rangoon", "Myanmar", "Asia/Rangoon",
        "Asia/Riyadh", "Arabian", "Asia/Riyadh",
        "Asia/Saigon", "Indochina", "Asia/Bangkok",
        "Asia/Sakhalin", "Sakhalin", "Asia/Sakhalin",
        "Asia/Samarkand", "Uzbekistan", "Asia/Tashkent",
        "Asia/Seoul", "Korea", "Asia/Seoul",
        "Asia/Shanghai", "China", "Asia/Shanghai",
        "Asia/Singapore", "Singapore", "Asia/Singapore",
        "Asia/Taipei", "Taipei", "Asia/Taipei",
        "Asia/Tashkent", "Uzbekistan", "Asia/Tashkent",
        "Asia/Tbilisi", "Georgia", "Asia/Tbilisi",
        "Asia/Tehran", "Iran", "Asia/Tehran",
        "Asia/Tel_Aviv", "Israel", "Asia/Jerusalem",
        "Asia/Thimbu", "Bhutan", "Asia/Thimphu",
        "Asia/Thimphu", "Bhutan", "Asia/Thimphu",
        "Asia/Tokyo", "Japan", "Asia/Tokyo",
        "Asia/Ujung_Pandang", "Indonesia_Central", "Asia/Makassar",
        "Asia/Ulaanbaatar", "Mongolia", "Asia/Ulaanbaatar",
        "Asia/Ulan_Bator", "Mongolia", "Asia/Ulaanbaatar",
        "Asia/Urumqi", "Urumqi", "Asia/Urumqi",
        "Asia/Ust-Nera", "Vladivostok", "Asia/Vladivostok",
        "Asia/Vientiane", "Indochina", "Asia/Bangkok",
        "Asia/Vladivostok", "Vladivostok", "Asia/Vladivostok",
        "Asia/Yakutsk", "Yakutsk", "Asia/Yakutsk",
        "Asia/Yangon", "Myanmar", "Asia/Rangoon",
        "Asia/Yekaterinburg", "Yekaterinburg", "Asia/Yekaterinburg",
        "Asia/Yerevan", "Armenia", "Asia/Yerevan",
        "Atlantic/Azores", "Azores", "Atlantic/Azores",
        "Atlantic/Bermuda", "Atlantic", "America/Halifax",
        "Atlantic/Canary", "Europe_Western", "Atlantic/Canary",
        "Atlantic/Cape_Verde", "Cape_Verde", "Atlantic/Cape_Verde",
        "Atlantic/Faeroe", "Europe_Western", "Atlantic/Canary",
        "Atlantic/Faroe", "Europe_Western", "Atlantic/Canary",
        "Atlantic/Jan_Mayen", "Europe_Central", "Europe/Paris",
        "Atlantic/Madeira", "Europe_Western", "Atlantic/Canary",
        "Atlantic/Reykjavik", "GMT", "Atlantic/Reykjavik",
        "Atlantic/South_Georgia", "South_Georgia", "Atlantic/South_Georgia",
        "Atlantic/St_Helena", "GMT", "Atlantic/Reykjavik",
        "Atlantic/Stanley", "Falkland", "Atlantic/Stanley",
        "Australia/ACT", "Australia_Eastern", "Australia/Sydney",
        "Australia/Adelaide", "Australia_Central", "Australia/Adelaide",
        "Australia/Brisbane", "Australia_Eastern", "Australia/Sydney",
        "Australia/Broken_Hill", "Australia_Central", "Australia/Adelaide",
        "Australia/Canberra", "Australia_Eastern", "Australia/Sydney",
        "Australia/Currie", "Australia_Eastern", "Australia/Sydney",
        "Australia/Darwin", "Australia_Central", "Australia/Adelaide",
        "Australia/Eucla", "Australia_CentralWestern", "Australia/Eucla",
        "Australia/Hobart", "Australia_Eastern", "Australia/Sydney",
        "Australia/LHI", "Lord_Howe", "Australia/Lord_Howe",
        "Australia/Lindeman", "Australia_Eastern", "Australia/Sydney",
        "Australia/Lord_Howe", "Lord_Howe", "Australia/Lord_Howe",
        "Australia/Melbourne", "Australia_Eastern", "Australia/Sydney",
        "Australia/NSW", "Australia_Eastern", "Australia/Sydney",
        "Australia/North", "Australia_Central", "Australia/Adelaide",
        "Australia/Perth", "Australia_Western", "Australia/Perth",
        "Australia/Queensland", "Australia_Eastern", "Australia/Sydney",
        "Australia/South", "Australia_Central", "Australia/Adelaide",
        "Australia/Sydney", "Australia_Eastern", "Australia/Sydney",
        "Australia/Tasmania", "Australia_Eastern", "Australia/Sydney",
        "Australia/Victoria", "Australia_Eastern", "Australia/Sydney",
        "Australia/West", "Australia_Western", "Australia/Perth",
        "Australia/Yancowinna", "Australia_Central", "Australia/Adelaide",
        "Brazil/Acre", "Acre", "America/Rio_Branco",
        "Brazil/DeNoronha", "Noronha", "America/Noronha",
        "Brazil/East", "Brasilia", "America/Sao_Paulo",
        "Brazil/West", "Amazon", "America/Manaus",
        "CST6CDT", "America_Central", "America/Chicago",
        "Canada/Atlantic", "Atlantic", "America/Halifax",
        "Canada/Central", "America_Central", "America/Chicago",
        "Canada/Eastern", "America_Eastern", "America/New_York",
        "Canada/Mountain", "America_Mountain", "America/Denver",
        "Canada/Newfoundland", "Newfoundland", "America/St_Johns",
        "Canada/Pacific", "America_Pacific", "America/Los_Angeles",
        "Canada/Saskatchewan", "America_Central", "America/Chicago",
        "Canada/Yukon", "America_Mountain", "America/Denver",
        "Chile/Continental", "Chile", "America/Santiago",
        "Chile/EasterIsland", "Easter", "Pacific/Easter",
        "Cuba", "Cuba", "America/Havana",
        "EST5EDT", "America_Eastern", "America/New_York",
        "Egypt", "Europe_Eastern", "Europe/Bucharest",
        "Eire", "GMT", "Atlantic/Reykjavik",
        "Etc/GMT", "GMT", "Atlantic/Reykjavik",
        "Etc/GMT+0", "GMT", "Atlantic/Reykjavik",
        "Etc/GMT-0", "GMT", "Atlantic/Reykjavik",
        "Etc/GMT0", "GMT", "Atlantic/Reykjavik",
        "Etc/Greenwich", "GMT", "Atlantic/Reykjavik",
        "Europe/Amsterdam", "Europe_Central", "Europe/Paris",
        "Europe/Andorra", "Europe_Central", "Europe/Paris",
        "Europe/Athens", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Belfast", "GMT", "Atlantic/Reykjavik",
        "Europe/Belgrade", "Europe_Central", "Europe/Paris",
        "Europe/Berlin", "Europe_Central", "Europe/Paris",
        "Europe/Bratislava", "Europe_Central", "Europe/Paris",
        "Europe/Brussels", "Europe_Central", "Europe/Paris",
        "Europe/Bucharest", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Budapest", "Europe_Central", "Europe/Paris",
        "Europe/Busingen", "Europe_Central", "Europe/Paris",
        "Europe/Chisinau", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Copenhagen", "Europe_Central", "Europe/Paris",
        "Europe/Dublin", "GMT", "Atlantic/Reykjavik",
        "Europe/Gibraltar", "Europe_Central", "Europe/Paris",
        "Europe/Guernsey", "GMT", "Atlantic/Reykjavik",
        "Europe/Helsinki", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Isle_of_Man", "GMT", "Atlantic/Reykjavik",
        "Europe/Istanbul", "Turkey", "Europe/Istanbul",
        "Europe/Jersey", "GMT", "Atlantic/Reykjavik",
        "Europe/Kaliningrad", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Kiev", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Lisbon", "Europe_Western", "Atlantic/Canary",
        "Europe/Ljubljana", "Europe_Central", "Europe/Paris",
        "Europe/London", "GMT", "Atlantic/Reykjavik",
        "Europe/Luxembourg", "Europe_Central", "Europe/Paris",
        "Europe/Madrid", "Europe_Central", "Europe/Paris",
        "Europe/Malta", "Europe_Central", "Europe/Paris",
        "Europe/Mariehamn", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Minsk", "Moscow", "Europe/Moscow",
        "Europe/Monaco", "Europe_Central", "Europe/Paris",
        "Europe/Moscow", "Moscow", "Europe/Moscow",
        "Europe/Nicosia", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Oslo", "Europe_Central", "Europe/Paris",
        "Europe/Paris", "Europe_Central", "Europe/Paris",
        "Europe/Podgorica", "Europe_Central", "Europe/Paris",
        "Europe/Prague", "Europe_Central", "Europe/Paris",
        "Europe/Riga", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Rome", "Europe_Central", "Europe/Paris",
        "Europe/Samara", "Samara", "Europe/Samara",
        "Europe/San_Marino", "Europe_Central", "Europe/Paris",
        "Europe/Sarajevo", "Europe_Central", "Europe/Paris",
        "Europe/Simferopol", "Moscow", "Europe/Moscow",
        "Europe/Skopje", "Europe_Central", "Europe/Paris",
        "Europe/Sofia", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Stockholm", "Europe_Central", "Europe/Paris",
        "Europe/Tallinn", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Tirane", "Europe_Central", "Europe/Paris",
        "Europe/Tiraspol", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Uzhgorod", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Vaduz", "Europe_Central", "Europe/Paris",
        "Europe/Vatican", "Europe_Central", "Europe/Paris",
        "Europe/Vienna", "Europe_Central", "Europe/Paris",
        "Europe/Vilnius", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Volgograd", "Volgograd", "Europe/Volgograd",
        "Europe/Warsaw", "Europe_Central", "Europe/Paris",
        "Europe/Zagreb", "Europe_Central", "Europe/Paris",
        "Europe/Zaporozhye", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Zurich", "Europe_Central", "Europe/Paris",
        "GB", "GMT", "Atlantic/Reykjavik",
        "GB-Eire", "GMT", "Atlantic/Reykjavik",
        "GMT", "GMT", "Atlantic/Reykjavik",
        "GMT0", "GMT", "Atlantic/Reykjavik",
        "Greenwich", "GMT", "Atlantic/Reykjavik",
        "Hongkong", "Hong_Kong", "Asia/Hong_Kong",
        "Iceland", "GMT", "Atlantic/Reykjavik",
        "Indian/Antananarivo", "Africa_Eastern", "Africa/Nairobi",
        "Indian/Chagos", "Indian_Ocean", "Indian/Chagos",
        "Indian/Christmas", "Christmas", "Indian/Christmas",
        "Indian/Cocos", "Cocos", "Indian/Cocos",
        "Indian/Comoro", "Africa_Eastern", "Africa/Nairobi",
        "Indian/Kerguelen", "French_Southern", "Indian/Kerguelen",
        "Indian/Mahe", "Seychelles", "Indian/Mahe",
        "Indian/Maldives", "Maldives", "Indian/Maldives",
        "Indian/Mauritius", "Mauritius", "Indian/Mauritius",
        "Indian/Mayotte", "Africa_Eastern", "Africa/Nairobi",
        "Indian/Reunion", "Reunion", "Indian/Reunion",
        "Iran", "Iran", "Asia/Tehran",
        "Israel", "Israel", "Asia/Jerusalem",
        "Jamaica", "America_Eastern", "America/New_York",
        "Japan", "Japan", "Asia/Tokyo",
        "Kwajalein", "Marshall_Islands", "Pacific/Majuro",
        "Libya", "Europe_Eastern", "Europe/Bucharest",
        "MST7MDT", "America_Mountain", "America/Denver",
        "Mexico/BajaNorte", "America_Pacific", "America/Los_Angeles",
        "Mexico/BajaSur", "Mexico_Pacific", "America/Mazatlan",
        "Mexico/General", "America_Central", "America/Chicago",
        "NZ", "New_Zealand", "Pacific/Auckland",
        "NZ-CHAT", "Chatham", "Pacific/Chatham",
        "Navajo", "America_Mountain", "America/Denver",
        "PRC", "China", "Asia/Shanghai",
        "PST8PDT", "America_Pacific", "America/Los_Angeles",
        "Pacific/Apia", "Apia", "Pacific/Apia",
        "Pacific/Auckland", "New_Zealand", "Pacific/Auckland",
        "Pacific/Chatham", "Chatham", "Pacific/Chatham",
        "Pacific/Chuuk", "Truk", "Pacific/Truk",
        "Pacific/Easter", "Easter", "Pacific/Easter",
        "Pacific/Efate", "Vanuatu", "Pacific/Efate",
        "Pacific/Enderbury", "Phoenix_Islands", "Pacific/Enderbury",
        "Pacific/Fakaofo", "Tokelau", "Pacific/Fakaofo",
        "Pacific/Fiji", "Fiji", "Pacific/Fiji",
        "Pacific/Funafuti", "Tuvalu", "Pacific/Funafuti",
        "Pacific/Galapagos", "Galapagos", "Pacific/Galapagos",
        "Pacific/Gambier", "Gambier", "Pacific/Gambier",
        "Pacific/Guadalcanal", "Solomon", "Pacific/Guadalcanal",
        "Pacific/Guam", "Chamorro", "Pacific/Saipan",
        "Pacific/Honolulu", "Hawaii_Aleutian", "Pacific/Honolulu",
        "Pacific/Johnston", "Hawaii_Aleutian", "Pacific/Honolulu",
        "Pacific/Kiritimati", "Line_Islands", "Pacific/Kiritimati",
        "Pacific/Kosrae", "Kosrae", "Pacific/Kosrae",
        "Pacific/Kwajalein", "Marshall_Islands", "Pacific/Majuro",
        "Pacific/Majuro", "Marshall_Islands", "Pacific/Majuro",
        "Pacific/Marquesas", "Marquesas", "Pacific/Marquesas",
        "Pacific/Midway", "Samoa", "Pacific/Pago_Pago",
        "Pacific/Nauru", "Nauru", "Pacific/Nauru",
        "Pacific/Niue", "Niue", "Pacific/Niue",
        "Pacific/Norfolk", "Norfolk", "Pacific/Norfolk",
        "Pacific/Noumea", "New_Caledonia", "Pacific/Noumea",
        "Pacific/Pago_Pago", "Samoa", "Pacific/Pago_Pago",
        "Pacific/Palau", "Palau", "Pacific/Palau",
        "Pacific/Pitcairn", "Pitcairn", "Pacific/Pitcairn",
        "Pacific/Pohnpei", "Ponape", "Pacific/Ponape",
        "Pacific/Ponape", "Ponape", "Pacific/Ponape",
        "Pacific/Port_Moresby", "Papua_New_Guinea", "Pacific/Port_Moresby",
        "Pacific/Rarotonga", "Cook", "Pacific/Rarotonga",
        "Pacific/Saipan", "Chamorro", "Pacific/Saipan",
        "Pacific/Samoa", "Samoa", "Pacific/Pago_Pago",
        "Pacific/Tahiti", "Tahiti", "Pacific/Tahiti",
        "Pacific/Tarawa", "Gilbert_Islands", "Pacific/Tarawa",
        "Pacific/Tongatapu", "Tonga", "Pacific/Tongatapu",
        "Pacific/Truk", "Truk", "Pacific/Truk",
        "Pacific/Wake", "Wake", "Pacific/Wake",
        "Pacific/Wallis", "Wallis", "Pacific/Wallis",
        "Pacific/Yap", "Truk", "Pacific/Truk",
        "Poland", "Europe_Central", "Europe/Paris",
        "Portugal", "Europe_Western", "Atlantic/Canary",
        "ROK", "Korea", "Asia/Seoul",
        "Singapore", "Singapore", "Asia/Singapore",
        "Turkey", "Turkey", "Europe/Istanbul",
        "US/Alaska", "Alaska", "America/Juneau",
        "US/Aleutian", "Hawaii_Aleutian", "Pacific/Honolulu",
        "US/Arizona", "America_Mountain", "America/Denver",
        "US/Central", "America_Central", "America/Chicago",
        "US/East-Indiana", "America_Eastern", "America/New_York",
        "US/Eastern", "America_Eastern", "America/New_York",
        "US/Hawaii", "Hawaii_Aleutian", "Pacific/Honolulu",
        "US/Indiana-Starke", "America_Central", "America/Chicago",
        "US/Michigan", "America_Eastern", "America/New_York",
        "US/Mountain", "America_Mountain", "America/Denver",
        "US/Pacific", "America_Pacific", "America/Los_Angeles",
        "US/Samoa", "Samoa", "Pacific/Pago_Pago",
        "W-SU", "Moscow", "Europe/Moscow",

        // From tzdb
        "Africa/Khartoum", "Africa_Central", "Africa/Maputo", // tzdata2017c
        "Africa/Windhoek", "Africa_Central", "Africa/Maputo", // tzdata2017c
        "Africa/Sao_Tome", "Africa_Western", "Africa/Lagos",  // tzdata2018c
    };
    private static final String[] mzoneMap = new String[] {
        // From metaZones.xml
        "Africa_Central", "BI", "Africa/Bujumbura",
        "Africa_Central", "BW", "Africa/Gaborone",
        "Africa_Central", "CD", "Africa/Lubumbashi",
        "Africa_Central", "MW", "Africa/Blantyre",
        "Africa_Central", "RW", "Africa/Kigali",
        "Africa_Central", "ZM", "Africa/Lusaka",
        "Africa_Central", "ZW", "Africa/Harare",
        "Africa_Eastern", "DJ", "Africa/Djibouti",
        "Africa_Eastern", "ER", "Africa/Asmera",
        "Africa_Eastern", "ET", "Africa/Addis_Ababa",
        "Africa_Eastern", "KM", "Indian/Comoro",
        "Africa_Eastern", "MG", "Indian/Antananarivo",
        "Africa_Eastern", "SO", "Africa/Mogadishu",
        "Africa_Eastern", "TZ", "Africa/Dar_es_Salaam",
        "Africa_Eastern", "UG", "Africa/Kampala",
        "Africa_Eastern", "YT", "Indian/Mayotte",
        "Africa_Southern", "LS", "Africa/Maseru",
        "Africa_Southern", "SZ", "Africa/Mbabane",
        "Africa_Western", "AO", "Africa/Luanda",
        "Africa_Western", "BJ", "Africa/Porto-Novo",
        "Africa_Western", "CD", "Africa/Kinshasa",
        "Africa_Western", "CF", "Africa/Bangui",
        "Africa_Western", "CG", "Africa/Brazzaville",
        "Africa_Western", "CM", "Africa/Douala",
        "Africa_Western", "GA", "Africa/Libreville",
        "Africa_Western", "GQ", "Africa/Malabo",
        "Africa_Western", "NE", "Africa/Niamey",
        "Africa_Western", "TD", "Africa/Ndjamena",
        "America_Central", "BZ", "America/Belize",
        "America_Central", "CA", "America/Winnipeg",
        "America_Central", "CR", "America/Costa_Rica",
        "America_Central", "GT", "America/Guatemala",
        "America_Central", "HN", "America/Tegucigalpa",
        "America_Central", "MX", "America/Mexico_City",
        "America_Central", "SV", "America/El_Salvador",
        "America_Eastern", "BS", "America/Nassau",
        "America_Eastern", "CA", "America/Toronto",
        "America_Eastern", "HT", "America/Port-au-Prince",
        "America_Eastern", "JM", "America/Jamaica",
        "America_Eastern", "KY", "America/Cayman",
        "America_Eastern", "PA", "America/Panama",
        "America_Mountain", "CA", "America/Edmonton",
        "America_Mountain", "MX", "America/Hermosillo",
        "America_Pacific", "CA", "America/Vancouver",
        "America_Pacific", "MX", "America/Tijuana",
        "Arabian", "BH", "Asia/Bahrain",
        "Arabian", "IQ", "Asia/Baghdad",
        "Arabian", "KW", "Asia/Kuwait",
        "Arabian", "QA", "Asia/Qatar",
        "Arabian", "YE", "Asia/Aden",
        "Atlantic", "AG", "America/Antigua",
        "Atlantic", "AI", "America/Anguilla",
        "Atlantic", "AW", "America/Aruba",
        "Atlantic", "BB", "America/Barbados",
        "Atlantic", "BM", "Atlantic/Bermuda",
        "Atlantic", "BQ", "America/Kralendijk",
        "Atlantic", "CW", "America/Curacao",
        "Atlantic", "DM", "America/Dominica",
        "Atlantic", "GD", "America/Grenada",
        "Atlantic", "GL", "America/Thule",
        "Atlantic", "GP", "America/Guadeloupe",
        "Atlantic", "KN", "America/St_Kitts",
        "Atlantic", "LC", "America/St_Lucia",
        "Atlantic", "MF", "America/Marigot",
        "Atlantic", "MQ", "America/Martinique",
        "Atlantic", "MS", "America/Montserrat",
        "Atlantic", "PR", "America/Puerto_Rico",
        "Atlantic", "SX", "America/Lower_Princes",
        "Atlantic", "TT", "America/Port_of_Spain",
        "Atlantic", "VC", "America/St_Vincent",
        "Atlantic", "VG", "America/Tortola",
        "Atlantic", "VI", "America/St_Thomas",
        "Chamorro", "GU", "Pacific/Guam",
        "Europe_Central", "AD", "Europe/Andorra",
        "Europe_Central", "AL", "Europe/Tirane",
        "Europe_Central", "AT", "Europe/Vienna",
        "Europe_Central", "BA", "Europe/Sarajevo",
        "Europe_Central", "BE", "Europe/Brussels",
        "Europe_Central", "CH", "Europe/Zurich",
        "Europe_Central", "CZ", "Europe/Prague",
        "Europe_Central", "DE", "Europe/Berlin",
        "Europe_Central", "DK", "Europe/Copenhagen",
        "Europe_Central", "ES", "Europe/Madrid",
        "Europe_Central", "GI", "Europe/Gibraltar",
        "Europe_Central", "HR", "Europe/Zagreb",
        "Europe_Central", "HU", "Europe/Budapest",
        "Europe_Central", "IT", "Europe/Rome",
        "Europe_Central", "LI", "Europe/Vaduz",
        "Europe_Central", "LU", "Europe/Luxembourg",
        "Europe_Central", "MC", "Europe/Monaco",
        "Europe_Central", "ME", "Europe/Podgorica",
        "Europe_Central", "MK", "Europe/Skopje",
        "Europe_Central", "MT", "Europe/Malta",
        "Europe_Central", "NL", "Europe/Amsterdam",
        "Europe_Central", "NO", "Europe/Oslo",
        "Europe_Central", "PL", "Europe/Warsaw",
        "Europe_Central", "RS", "Europe/Belgrade",
        "Europe_Central", "SE", "Europe/Stockholm",
        "Europe_Central", "SI", "Europe/Ljubljana",
        "Europe_Central", "SJ", "Arctic/Longyearbyen",
        "Europe_Central", "SK", "Europe/Bratislava",
        "Europe_Central", "SM", "Europe/San_Marino",
        "Europe_Central", "TN", "Africa/Tunis",
        "Europe_Central", "VA", "Europe/Vatican",
        "Europe_Central", "XK", "Europe/Belgrade",
        "Europe_Eastern", "AX", "Europe/Mariehamn",
        "Europe_Eastern", "BG", "Europe/Sofia",
        "Europe_Eastern", "CY", "Asia/Nicosia",
        "Europe_Eastern", "EG", "Africa/Cairo",
        "Europe_Eastern", "FI", "Europe/Helsinki",
        "Europe_Eastern", "GR", "Europe/Athens",
        "Europe_Eastern", "JO", "Asia/Amman",
        "Europe_Eastern", "LB", "Asia/Beirut",
        "Europe_Eastern", "SY", "Asia/Damascus",
        "Europe_Further_Eastern", "RU", "Europe/Kaliningrad",
        "Europe_Western", "FO", "Atlantic/Faeroe",
        "GMT", "BF", "Africa/Ouagadougou",
        "GMT", "CI", "Africa/Abidjan",
        "GMT", "GB", "Europe/London",
        "GMT", "GH", "Africa/Accra",
        "GMT", "GM", "Africa/Banjul",
        "GMT", "GN", "Africa/Conakry",
        "GMT", "IE", "Europe/Dublin",
        "GMT", "ML", "Africa/Bamako",
        "GMT", "MR", "Africa/Nouakchott",
        "GMT", "SH", "Atlantic/St_Helena",
        "GMT", "SL", "Africa/Freetown",
        "GMT", "SN", "Africa/Dakar",
        "GMT", "TG", "Africa/Lome",
        "Gulf", "OM", "Asia/Muscat",
        "India", "LK", "Asia/Colombo",
        "Indochina", "KH", "Asia/Phnom_Penh",
        "Indochina", "LA", "Asia/Vientiane",
        "New_Zealand", "AQ", "Antarctica/McMurdo",

        // From tzdb
        "Africa_Western", "ST", "Africa/Sao_Tome", // tzdata2018c
    };
    private static final String[] aliasMap = new String[] {
        // From supplementalMetadata.xml
        "Africa/Timbuktu", "Africa/Bamako",
        "America/Montreal", "America/Toronto",
        "America/Shiprock", "America/Denver",
        "Antarctica/South_Pole", "Pacific/Auckland",
        "Asia/Chongqing", "Asia/Shanghai",
        "Asia/Harbin", "Asia/Shanghai",
        "Asia/Kashgar", "Asia/Urumqi",
        "Atlantic/Jan_Mayen", "Europe/Oslo",
        "EST", "America/Indianapolis",
        "Europe/Belfast", "Europe/London",
        "HST", "Pacific/Honolulu",
        "MST", "America/Phoenix",
        "Pacific/Yap", "Pacific/Truk",
        "SystemV/AST4", "America/Puerto_Rico",
        "SystemV/AST4ADT", "America/Halifax",
        "SystemV/CST6", "America/Regina",
        "SystemV/CST6CDT", "America/Chicago",
        "SystemV/EST5", "America/Indianapolis",
        "SystemV/EST5EDT", "America/New_York",
        "SystemV/HST10", "Pacific/Honolulu",
        "SystemV/MST7", "America/Phoenix",
        "SystemV/MST7MDT", "America/Denver",
        "SystemV/PST8", "Pacific/Pitcairn",
        "SystemV/PST8PDT", "America/Los_Angeles",
        "SystemV/YST9", "Pacific/Gambier",
        "SystemV/YST9YDT", "America/Anchorage",

        // From tzdb
        "Africa/Accra", "Africa/Abidjan",
        "Africa/Addis_Ababa", "Africa/Nairobi",
        "Africa/Asmara", "Africa/Nairobi",
        "Africa/Asmera", "Africa/Nairobi",
        "Africa/Bamako", "Africa/Abidjan",
        "Africa/Bangui", "Africa/Lagos",
        "Africa/Banjul", "Africa/Abidjan",
        "Africa/Blantyre", "Africa/Maputo",
        "Africa/Brazzaville", "Africa/Lagos",
        "Africa/Bujumbura", "Africa/Maputo",
        "Africa/Conakry", "Africa/Abidjan",
        "Africa/Dakar", "Africa/Abidjan",
        "Africa/Dar_es_Salaam", "Africa/Nairobi",
        "Africa/Djibouti", "Africa/Nairobi",
        "Africa/Douala", "Africa/Lagos",
        "Africa/Freetown", "Africa/Abidjan",
        "Africa/Gaborone", "Africa/Maputo",
        "Africa/Harare", "Africa/Maputo",
        "Africa/Kampala", "Africa/Nairobi",
        "Africa/Kigali", "Africa/Maputo",
        "Africa/Kinshasa", "Africa/Lagos",
        "Africa/Libreville", "Africa/Lagos",
        "Africa/Lome", "Africa/Abidjan",
        "Africa/Luanda", "Africa/Lagos",
        "Africa/Lubumbashi", "Africa/Maputo",
        "Africa/Lusaka", "Africa/Maputo",
        "Africa/Malabo", "Africa/Lagos",
        "Africa/Maseru", "Africa/Johannesburg",
        "Africa/Mbabane", "Africa/Johannesburg",
        "Africa/Mogadishu", "Africa/Nairobi",
        "Africa/Niamey", "Africa/Lagos",
        "Africa/Nouakchott", "Africa/Abidjan",
        "Africa/Ouagadougou", "Africa/Abidjan",
        "Africa/Porto-Novo", "Africa/Lagos",
        "Africa/Timbuktu", "Africa/Abidjan",
        "America/Anguilla", "America/Puerto_Rico",
        "America/Antigua", "America/Puerto_Rico",
        "America/Argentina/ComodRivadavia", "America/Argentina/Catamarca",
        "America/Aruba", "America/Puerto_Rico",
        "America/Atikokan", "America/Panama",
        "America/Atka", "America/Adak",
        "America/Blanc-Sablon", "America/Puerto_Rico",
        "America/Buenos_Aires", "America/Argentina/Buenos_Aires",
        "America/Catamarca", "America/Argentina/Catamarca",
        "America/Cayman", "America/Panama",
        "America/Coral_Harbour", "America/Panama",
        "America/Cordoba", "America/Argentina/Cordoba",
        "America/Creston", "America/Phoenix",
        "America/Curacao", "America/Puerto_Rico",
        "America/Dominica", "America/Puerto_Rico",
        "America/Ensenada", "America/Tijuana",
        "America/Fort_Wayne", "America/Indiana/Indianapolis",
        "America/Godthab", "America/Nuuk",
        "America/Grenada", "America/Puerto_Rico",
        "America/Guadeloupe", "America/Puerto_Rico",
        "America/Indianapolis", "America/Indiana/Indianapolis",
        "America/Jujuy", "America/Argentina/Jujuy",
        "America/Knox_IN", "America/Indiana/Knox",
        "America/Kralendijk", "America/Puerto_Rico",
        "America/Louisville", "America/Kentucky/Louisville",
        "America/Lower_Princes", "America/Puerto_Rico",
        "America/Marigot", "America/Puerto_Rico",
        "America/Mendoza", "America/Argentina/Mendoza",
        "America/Montreal", "America/Toronto",
        "America/Montserrat", "America/Puerto_Rico",
        "America/Nassau", "America/Toronto",
        "America/Port_of_Spain", "America/Puerto_Rico",
        "America/Porto_Acre", "America/Rio_Branco",
        "America/Rosario", "America/Argentina/Cordoba",
        "America/Santa_Isabel", "America/Tijuana",
        "America/Shiprock", "America/Denver",
        "America/St_Barthelemy", "America/Puerto_Rico",
        "America/St_Kitts", "America/Puerto_Rico",
        "America/St_Lucia", "America/Puerto_Rico",
        "America/St_Thomas", "America/Puerto_Rico",
        "America/St_Vincent", "America/Puerto_Rico",
        "America/Tortola", "America/Puerto_Rico",
        "America/Virgin", "America/Puerto_Rico",
        "Antarctica/DumontDUrville", "Pacific/Port_Moresby",
        "Antarctica/McMurdo", "Pacific/Auckland",
        "Antarctica/South_Pole", "Pacific/Auckland",
        "Antarctica/Syowa", "Asia/Riyadh",
        "Arctic/Longyearbyen", "Europe/Oslo",
        "Asia/Aden", "Asia/Riyadh",
        "Asia/Ashkhabad", "Asia/Ashgabat",
        "Asia/Bahrain", "Asia/Qatar",
        "Asia/Calcutta", "Asia/Kolkata",
        "Asia/Chongqing", "Asia/Shanghai",
        "Asia/Chungking", "Asia/Shanghai",
        "Asia/Dacca", "Asia/Dhaka",
        "Asia/Harbin", "Asia/Shanghai",
        "Asia/Istanbul", "Europe/Istanbul",
        "Asia/Kashgar", "Asia/Urumqi",
        "Asia/Katmandu", "Asia/Kathmandu",
        "Asia/Kuwait", "Asia/Riyadh",
        "Asia/Macao", "Asia/Macau",
        "Asia/Muscat", "Asia/Dubai",
        "Asia/Phnom_Penh", "Asia/Bangkok",
        "Asia/Rangoon", "Asia/Yangon",
        "Asia/Saigon", "Asia/Ho_Chi_Minh",
        "Asia/Tel_Aviv", "Asia/Jerusalem",
        "Asia/Thimbu", "Asia/Thimphu",
        "Asia/Ujung_Pandang", "Asia/Makassar",
        "Asia/Ulan_Bator", "Asia/Ulaanbaatar",
        "Asia/Vientiane", "Asia/Bangkok",
        "Atlantic/Faeroe", "Atlantic/Faroe",
        "Atlantic/Jan_Mayen", "Europe/Oslo",
        "Atlantic/St_Helena", "Africa/Abidjan",
        "Australia/ACT", "Australia/Sydney",
        "Australia/Canberra", "Australia/Sydney",
        "Australia/Currie", "Australia/Hobart",
        "Australia/LHI", "Australia/Lord_Howe",
        "Australia/NSW", "Australia/Sydney",
        "Australia/North", "Australia/Darwin",
        "Australia/Queensland", "Australia/Brisbane",
        "Australia/South", "Australia/Adelaide",
        "Australia/Tasmania", "Australia/Hobart",
        "Australia/Victoria", "Australia/Melbourne",
        "Australia/West", "Australia/Perth",
        "Australia/Yancowinna", "Australia/Broken_Hill",
        "Brazil/Acre", "America/Rio_Branco",
        "Brazil/DeNoronha", "America/Noronha",
        "Brazil/East", "America/Sao_Paulo",
        "Brazil/West", "America/Manaus",
        "Canada/Atlantic", "America/Halifax",
        "Canada/Central", "America/Winnipeg",
        "Canada/Eastern", "America/Toronto",
        "Canada/Mountain", "America/Edmonton",
        "Canada/Newfoundland", "America/St_Johns",
        "Canada/Pacific", "America/Vancouver",
        "Canada/Saskatchewan", "America/Regina",
        "Canada/Yukon", "America/Whitehorse",
        "Chile/Continental", "America/Santiago",
        "Chile/EasterIsland", "Pacific/Easter",
        "Cuba", "America/Havana",
        "Egypt", "Africa/Cairo",
        "Eire", "Europe/Dublin",
        "Etc/GMT+0", "Etc/GMT",
        "Etc/GMT-0", "Etc/GMT",
        "Etc/GMT0", "Etc/GMT",
        "Etc/Greenwich", "Etc/GMT",
        "Etc/UCT", "Etc/UTC",
        "Etc/Universal", "Etc/UTC",
        "Etc/Zulu", "Etc/UTC",
        "Europe/Belfast", "Europe/London",
        "Europe/Bratislava", "Europe/Prague",
        "Europe/Busingen", "Europe/Zurich",
        "Europe/Guernsey", "Europe/London",
        "Europe/Isle_of_Man", "Europe/London",
        "Europe/Jersey", "Europe/London",
        "Europe/Ljubljana", "Europe/Belgrade",
        "Europe/Mariehamn", "Europe/Helsinki",
        "Europe/Nicosia", "Asia/Nicosia",
        "Europe/Podgorica", "Europe/Belgrade",
        "Europe/San_Marino", "Europe/Rome",
        "Europe/Sarajevo", "Europe/Belgrade",
        "Europe/Skopje", "Europe/Belgrade",
        "Europe/Tiraspol", "Europe/Chisinau",
        "Europe/Vaduz", "Europe/Zurich",
        "Europe/Vatican", "Europe/Rome",
        "Europe/Zagreb", "Europe/Belgrade",
        "GB", "Europe/London",
        "GB-Eire", "Europe/London",
        "GMT", "Etc/GMT",
        "GMT+0", "Etc/GMT",
        "GMT-0", "Etc/GMT",
        "GMT0", "Etc/GMT",
        "Greenwich", "Etc/GMT",
        "Hongkong", "Asia/Hong_Kong",
        "Iceland", "Atlantic/Reykjavik",
        "Indian/Antananarivo", "Africa/Nairobi",
        "Indian/Comoro", "Africa/Nairobi",
        "Indian/Mayotte", "Africa/Nairobi",
        "Iran", "Asia/Tehran",
        "Israel", "Asia/Jerusalem",
        "Jamaica", "America/Jamaica",
        "Japan", "Asia/Tokyo",
        "Kwajalein", "Pacific/Kwajalein",
        "Libya", "Africa/Tripoli",
        "Mexico/BajaNorte", "America/Tijuana",
        "Mexico/BajaSur", "America/Mazatlan",
        "Mexico/General", "America/Mexico_City",
        "Mideast/Riyadh87", "Asia/Riyadh87",
        "Mideast/Riyadh88", "Asia/Riyadh88",
        "Mideast/Riyadh89", "Asia/Riyadh89",
        "NZ", "Pacific/Auckland",
        "NZ-CHAT", "Pacific/Chatham",
        "Navajo", "America/Denver",
        "PRC", "Asia/Shanghai",
        "Pacific/Enderbury", "Pacific/Kanton",
        "Pacific/Johnston", "Pacific/Honolulu",
        "Pacific/Midway", "Pacific/Pago_Pago",
        "Pacific/Ponape", "Pacific/Pohnpei",
        "Pacific/Saipan", "Pacific/Guam",
        "Pacific/Samoa", "Pacific/Pago_Pago",
        "Pacific/Truk", "Pacific/Chuuk",
        "Pacific/Yap", "Pacific/Chuuk",
        "Poland", "Europe/Warsaw",
        "Portugal", "Europe/Lisbon",
        "ROC", "Asia/Taipei",
        "ROK", "Asia/Seoul",
        "Singapore", "Asia/Singapore",
        "Turkey", "Europe/Istanbul",
        "UCT", "Etc/UTC",
        "US/Alaska", "America/Anchorage",
        "US/Aleutian", "America/Adak",
        "US/Arizona", "America/Phoenix",
        "US/Central", "America/Chicago",
        "US/East-Indiana", "America/Indiana/Indianapolis",
        "US/Eastern", "America/New_York",
        "US/Hawaii", "Pacific/Honolulu",
        "US/Indiana-Starke", "America/Indiana/Knox",
        "US/Michigan", "America/Detroit",
        "US/Mountain", "America/Denver",
        "US/Pacific", "America/Los_Angeles",
        "US/Samoa", "Pacific/Pago_Pago",
        "UTC", "Etc/UTC",
        "Universal", "Etc/UTC",
        "W-SU", "Europe/Moscow",
        "Zulu", "Etc/UTC",
    };

    private static final Map<String, String> zidToMzone = new HashMap<>();
    private static final Map<String, String> mzoneToZid = new HashMap<>();
    private static final Map<String, Map<String, String>> mzoneToZidL = new HashMap<>();
    private static final Map<String, String> aliases = new HashMap<>();

    static {
        for (int i = 0; i < zidMap.length; i += 3) {
            zidToMzone.put(zidMap[i], zidMap[i + 1]);
            mzoneToZid.put(zidMap[i + 1], zidMap[i + 2]);
        }

        for (int i = 0; i < mzoneMap.length; i += 3) {
            String mzone = mzoneMap[i];
            Map<String, String> map = mzoneToZidL.get(mzone);
            if (map == null) {
                map = new HashMap<>();
                mzoneToZidL.put(mzone, map);
            }
            map.put(mzoneMap[i + 1], mzoneMap[i + 2]);
        }

        for (int i = 0; i < aliasMap.length; i += 2) {
            aliases.put(aliasMap[i], aliasMap[i + 1]);
        }
    }
}
