/* fileviewerbg.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2010-2012 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "fileviewerbg.hh"
#include "fileviewer.hh"
#include "worker.h"
#include <aguix/textview.h>
#include "textstoragefile.h"
#include "nwc_fsentry.hh"
#include <aguix/util.h>
#include <aguix/refcount.hh>
#include "filenameshrinker.hh"
#include <aguix/searchtextstorage.hh>
#include <aguix/backgroundmessagehandler.hh>
#include <aguix/choosebutton.h>
#include <aguix/button.h>
#include <aguix/stringgadget.h>
#include "worker_locale.h"
#include "wconfig.h"

FileViewerBG::FileViewerBG() : win( NULL ),
                               m_wrap_mode( false ),
                               m_showlinenumbers_mode( false ),
                               ts( NULL ),
                               tv( NULL ),
                               search_sg( NULL ),
                               okb( NULL ),
                               wcb( NULL ),
                               slncb( NULL ),
                               jumplineb( NULL ),
                               readmoreb( NULL )

{
}

FileViewerBG::~FileViewerBG()
{
    m_destroy_callback = NULL;
    if ( win != NULL ) delete win;
    if ( ts != NULL ) delete ts;
}

void FileViewerBG::view( const std::list<std::string> &filelist,
                         int initial_line_number,
                         bool highlight_initial_line,
                         RefCount< FileViewerDestroyCallback > destroy_callback )
{
    AGUIX *aguix = Worker::getAGUIX();
    const int cfix = AContainer::ACONT_MINH +
        AContainer::ACONT_MINW +
        AContainer::ACONT_MAXH +
        AContainer::ACONT_MAXW;
    const int cincw = AContainer::ACONT_MINH +
        AContainer::ACONT_MINW +
        AContainer::ACONT_MAXH;
    const int cincwnr = cincw +
        AContainer::ACONT_NORESIZE;

    m_filelist = filelist;
    m_destroy_callback = destroy_callback;
    
    ts = NULL;
    tv = NULL;
    std::list<std::string>::const_iterator it1;
    int w, h, sw, sh;
    
    win = new AWindow( aguix, 10, 10, 10, 10, 0, catalog.getLocaleCom( 49 ) );
    win->create();
    /*  if ( aguix->getTransientWindow() != NULL ) {
        win->setTransientForAWindow( aguix->getTransientWindow() );
        }*/

    AContainer *ac1 = win->setContainer( new AContainer( win, 1, 3 ), true );
    ac1->setMinSpace( 5 );
    ac1->setMaxSpace( 5 );

    AContainer *ac1_1 = ac1->add( new AContainer( win, 3, 1 ), 0, 0 );
    ac1_1->setMinSpace( 5 );
    ac1_1->setMaxSpace( 5 );
    ac1_1->setBorderWidth( 0 );
    ac1_1->add( new Text( aguix, 0, 0, catalog.getLocale( 404 ), 1 ), 0, 0, AContainer::CO_FIXNR );
    Text *fnt = (Text*)ac1_1->add( new Text( aguix, 0, 0, "", 1 ), 1, 0, AContainer::ACONT_MINH + AContainer::ACONT_MAXH );
    readmoreb = (Button*)ac1_1->add( new Button( aguix, 0, 0, catalog.getLocale( 763 ), 1, 0, 0 ),
                                     2, 0, cfix );

    fnt->setTextShrinker( RefCount<TextShrinker>( new FileNameShrinker() ) );

    it1 = filelist.begin();
  
    RefCount<AFontWidth> lencalc( new AFontWidth( aguix, aguix->getFont( wconfig->getFont( 4 ).c_str() ) ) );

    if ( buildTextView( *it1, ac1, fnt, lencalc ) != 0 ) {
        delete win;
	win = NULL;
        if ( ts != NULL ) delete ts;
	ts = NULL;
        return;
    }

    ac1->readLimits();
    ac1_1->readLimits();

    AContainer *ac1_3 = ac1->add( new AContainer( win, 5, 1 ), 0, 2 );
    ac1_3->setMinSpace( 5 );
    ac1_3->setMaxSpace( -1 );
    ac1_3->setBorderWidth( 0 );
  
    /*nextb = (Button*)ac1_3->add( new Button( aguix, 0, 0, "Next", 1, 0, 2 ),
      0, 0, cfix );
      prevb = (Button*)ac1_3->add( new Button( aguix, 0, 0, "Prev", 1, 0, 2 ),
      1, 0, cfix );*/
    wcb = (ChooseButton*)ac1_3->add( new ChooseButton( aguix, 0, 0,
                                                       m_wrap_mode, catalog.getLocale( 637 ),
                                                       LABEL_RIGHT, 1, 0 ),
                                     0, 0, cincwnr );
    slncb = (ChooseButton*)ac1_3->add( new ChooseButton( aguix, 0, 0,
                                                         m_showlinenumbers_mode, catalog.getLocale( 715 ),
                                                         LABEL_RIGHT, 1, 0 ),
                                       1, 0, cincwnr );
    jumplineb = (Button*)ac1_3->add( new Button( aguix, 0, 0, catalog.getLocale( 716 ), 1, 0, 2 ),
                                     2, 0, cfix );

    AContainer *ac1_3_1 = ac1_3->add( new AContainer( win, 2, 1 ), 3, 0 );
    ac1_3_1->setMinSpace( 5 );
    ac1_3_1->setMaxSpace( 5 );
    ac1_3_1->setBorderWidth( 0 );
  
    ac1_3_1->add( new Text( aguix, 0, 0, catalog.getLocale( 790 ), 1 ), 0, 0, AContainer::CO_FIXNR + AContainer::ACONT_CENTER );
    search_sg = (StringGadget*)ac1_3_1->add( new StringGadget( aguix, 0, 0, 60, "", 1 ),
                                             1, 0, cincw );

    okb = (Button*)ac1_3->add( new Button( aguix, 0, 0, catalog.getLocale( 633 ), 1, 0, 2 ),
                               4, 0, cfix );

    win->setDoTabCycling( true );
    win->contMaximize( true );
  
    w = win->getWidth();
    h = win->getHeight();

    int rx, ry, rw, rh;

    aguix->getLargestDimensionOfCurrentScreen( &rx, &ry,
                                               &rw, &rh );

    sw = rw;
    sh = rh;
    sw = (int)( (double)sw * 0.8 );
    sh = (int)( (double)sh * 0.8 );
    if ( sw < 200 ) sw = 200;
    if ( sh < 100 ) sh = 100;
    if ( w < sw ) w = sw;
    if ( h < sh ) h = sh;
    win->resize( w, h );

    tv->setLineWrap( wcb->getState() );
    tv->setShowLineNumbers( slncb->getState() );
    tv->jumpToLine( initial_line_number, highlight_initial_line );

    if ( ts->incompleteFile() == false ) {
        readmoreb->hide();
    }

    win->useStippleBackground();
    win->centerScreen();
    win->show();

    win->applyFocus( tv );
}

void FileViewerBG::handleAGUIXMessage( AGMessage &msg )
{
    AGUIX *aguix = Worker::getAGUIX();
    int endmode = -1;
    //  bool ignore_key_release = true;

    switch ( msg.type ) {
        case AG_CLOSEWINDOW:
            if ( msg.closewindow.window == win->getWindow() ) {
                endmode = 0;
            }
            break;
	case AG_CHOOSECLICKED:
            if ( msg.choose.button == wcb ) {
                tv->setLineWrap( msg.choose.state );
            } else if ( msg.choose.button == slncb ) {
                tv->setShowLineNumbers( msg.choose.state );
            }
            break;
        case AG_BUTTONCLICKED:
            if ( msg.button.button == okb ) {
                endmode = 0;
                /*	  } else if ( msg.button.button == nextb ) {
                          it1++;
                          if ( it1 == filelist.end() ) it1 = filelist.begin();
                          buildTextView( *it1, &ts, &tv, ac1 );
                          } else if ( msg.button.button == prevb ) {
                          if ( it1 == filelist.begin() )
                          it1 = filelist.end();
                          it1--;
                          buildTextView( *it1, &ts, &tv, ac1 );*/
            } else if ( msg.button.button == jumplineb ) {
                jumpToLine();
            } else if ( msg.button.button == readmoreb ) {
                ts->readMore();
                tv->textStorageChanged();
                if ( ts->incompleteFile() == false ) {
                    readmoreb->hide();
                }
            }
            break;
            /*        case AG_KEYPRESSED:
                      ignore_key_release = false;
                      break;*/
	case AG_KEYPRESSED:
            if ( /*ignore_key_release == false && */win->isParent( msg.key.window, false ) == true ) {
                wcb->setState( tv->getLineWrap() );
                slncb->setState( tv->getShowLineNumbers() );
                switch ( msg.key.key ) {
                    case XK_q:
                    case XK_F3:
                    case XK_F10:
                        endmode = 0;
                        break;
                    case XK_Escape:
                        if ( search_sg->isActive() == false ) {
                            //TODO that is time critical, it may already be inactive
                            endmode = 0;
                        }
                        break;
                    case XK_l:
                        if ( ( msg.key.keystate & ( ControlMask | Mod1Mask ) ) != 0 ) {
                            jumpToLine();
                        }
                        break;
                    case XK_r:
                        if ( ts->incompleteFile() == true ) {
                            ts->readMore();
                            tv->textStorageChanged();
                            if ( ts->incompleteFile() == false ) {
                                readmoreb->hide();
                            }
                        }
                        break;
                    case XK_slash:
                        if ( search_sg->isActive() == true ) {
                            nextMatch();
                        } else {
                            std::pair<int, int> rl = ts->getRealLinePair( tv->getYOffset() );
                            ts_search->setSearchStartLine( rl.first );
                            search_sg->setText( "" );
                            search_sg->activate();
                        }
                        break;
                    case XK_s:
                    case XK_f:
                    case XK_i:
                        if ( ( msg.key.keystate & ( ControlMask | Mod1Mask ) ) != 0 ) {
                            if ( search_sg->isActive() == true ) {
                                nextMatch();
                            } else {
                                std::pair<int, int> rl = ts->getRealLinePair( tv->getYOffset() );
                                ts_search->setSearchStartLine( rl.first );
                                search_sg->setText( "" );
                                search_sg->activate();
                            }
                        }
                        break;
                    case XK_n:
                        if ( KEYSTATEMASK( msg.key.keystate ) == 0 ) {
                            nextMatch();
                        }
                        break;
                    case XK_Return:
                        nextMatch();
                        break;
                }
            }
            break;
        case AG_STRINGGADGET_OK:
            if ( msg.stringgadget.sg == search_sg ) {
                search_sg->activate();
            }
        case AG_STRINGGADGET_CONTENTCHANGE:
            if ( msg.stringgadget.sg == search_sg ) {
                nextMatch();
            }
            break;
        case AG_TEXTVIEW_END_REACHED:
            if ( msg.textview.textview == tv &&
                 ts->incompleteFile() ) {
                ts->readMore();
                tv->textStorageChanged();
                if ( ts->incompleteFile() == false ) {
                    readmoreb->hide();
                }
            }
            break;
    }

    if ( endmode == 0 ) {
        aguix->unregisterBGHandler( win );
        FileViewer::setGlobalLineWrap( tv->getLineWrap() );
        FileViewer::setGlobalShowLineNumbers( tv->getShowLineNumbers() );
    }
}

AWindow *FileViewerBG::getAWindow()
{
    return win;
}

int FileViewerBG::buildTextView( const std::string &filename,
                                 AContainer *ac1,
                                 class Text *fnt,
                                 const RefCount<AWidth> &lencalc )
{
    AGUIX *aguix = Worker::getAGUIX();
    const int cmin = AContainer::ACONT_MINH +
        AContainer::ACONT_MINW;
  
    if ( ac1 == NULL ) return 1;

    NWC::FSEntry fse( filename );
    loff_t file_length = 0;
    int initial_size = ( m_wrap_mode == true ) ? ( 512 * 1024 ) : ( 2 * 1024 * 1024 );
    bool continue_viewing = true;
  
    if ( fse.isReg() == true ) {
        if ( fse.isLink() == false ) {
            file_length = fse.stat_size();
        } else {
            file_length = fse.stat_dest_size();
        }
    }

    if ( file_length > initial_size ) {
        std::string l1, l2;
        int cancel_button;

        l1 = AGUIXUtils::bytes_to_human_readable_f( file_length );
        l2 = AGUIXUtils::bytes_to_human_readable_f( (double)initial_size );

        char *textstr = (char*)_allocsafe( strlen( catalog.getLocale( 764 ) ) +
                                           l1.length() + l2.length() + 1 );
        sprintf( textstr, catalog.getLocale( 764 ), l1.c_str(), l2.c_str() );

        std::string buttontext = catalog.getLocale( 765 );
        buttontext += "|";
        buttontext += catalog.getLocale( 766 );

        if ( m_wrap_mode == true ) {
            buttontext += "|";
            buttontext += catalog.getLocale( 767 );

            cancel_button = 3;
        } else {
            cancel_button = 2;
        }

        buttontext += "|";
        buttontext += catalog.getLocale( 8 );

        int erg = Worker::getRequester()->request( catalog.getLocale( 123 ),
                                                   textstr,
                                                   buttontext.c_str(),
                                                   win,
                                                   Requester::REQUEST_NONE );
        if ( erg == 1 ) {
            if ( file_length > 1 * 1024 * 1024 * 1024 ) {
                initial_size = 1 * 1024 * 1024 * 1024;
            } else {
                initial_size = file_length;
            }
        } else if ( erg == cancel_button ) {
            continue_viewing = false;
        } else if ( erg == 2 ) {
            if ( file_length > 1 * 1024 * 1024 * 1024 ) {
                initial_size = 1 * 1024 * 1024 * 1024;
            } else {
                initial_size = file_length;
            }
            m_wrap_mode = false;
        }
        _freesafe( textstr );
    }

    if ( continue_viewing == false ) {
        return 2;
    }
  
    if ( tv != NULL ) {
        delete tv;
    }
    if ( ts != NULL ) {
        delete ts;
    }
    ts = new TextStorageFile( filename, lencalc, initial_size );
    tv = new TextView( aguix, 0, 0, 100, 100, 0, "", *ts );

    ts_search = new SearchTextStorage( *ts );

    tv->setDisplayFocus( true );
    tv->setFont( wconfig->getFont( 4 ).c_str() );
    ac1->add( tv, 0, 1, cmin );
    tv->create();
    if ( fnt != NULL ) fnt->setText( filename.c_str() );
    ac1->rearrange();
    tv->show();

    return 0;
}

void FileViewerBG::jumpToLine()
{
    int erg;
    std::string buttons;
    char *return_str = NULL;

    buttons = catalog.getLocale( 716 );
    buttons += "|";
    buttons += catalog.getLocale( 8 );

    //TODO use current line number as default but I need the interface in tv for this first
    erg = win->string_request( catalog.getLocale( 123 ),
                               catalog.getLocale( 717 ),
                               "",
                               buttons.c_str(),
                               &return_str );
    if ( erg == 0 && return_str != NULL ) {
        int line_nr = 0;
        line_nr = atoi( return_str ) - 1;
        tv->jumpToLine( line_nr, true );
    }
    if ( return_str != NULL )
        _freesafe( return_str );
}

void FileViewerBG::nextMatch()
{
    ts_search->search( search_sg->getText() );
    ts_search->highlightCurMatch( *tv );
    tv->makeSelectionVisible();
}

void FileViewerBG::setLineWrap( bool nv )
{
    m_wrap_mode = nv;
}

bool FileViewerBG::getLineWrap() const
{
    return m_wrap_mode;
}

void FileViewerBG::setShowLineNumbers( bool nv )
{
    m_showlinenumbers_mode = nv;
}

bool FileViewerBG::getShowLineNumbers() const
{
    return m_showlinenumbers_mode;
}
