/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

#ifndef _PERIODIC_ACTION_THREAD_H
#define _PERIODIC_ACTION_THREAD_H

#include "Thread.h"
#include "Event.h"
#include "Semaphore.h"

#define SECOND 1000
#define MINUTE (60 * SECOND)

class PeriodicActionThread : public Thread
{
public:
	//ctor. nResumeDelay is the delay after calling resume() and before doing the next action.
	PeriodicActionThread(const char *pName,
                         unsigned long nActionInterval = UNDEFINED_INTERVAL,
						 unsigned long nResumeDelay = 0);

	//dtor
	virtual ~PeriodicActionThread();

	//Stop the periodic action thread
	virtual bool stop();
	//Suspend the thread and wait until resume() is called.
	void suspend();
	//Resume the suspended thread after suspend() was called.
	void resume();
	//Stop waiting for the next periodic action, and do the action now.
	void actNow();

	//Set the action interval
	void setActionInterval(unsigned long nActionInterval)
	{ m_nActionInterval = nActionInterval; }

	//Get the action interval
	unsigned long getActionInterval() const
	{ return m_nActionInterval; }

protected:

	//The action to be performed. To be implemented by derived class.
	virtual void action() = 0;
	//The actual function that runs the thread.
	virtual void run();
	//Stop the periodic action thread from within itself. Can be used by derived classes.
	void internalStop();
	//Cleanup function, called when stopping thread. To be implemented by derived class.
	virtual void cleanup() = 0;

private:
	//How long we wait for the thread to terminate when stop() is called.
	static const unsigned long WAIT_STOP_TIMEOUT;
	static const unsigned long UNDEFINED_INTERVAL;

	const char *m_pName;
	unsigned long m_nActionInterval;
	unsigned long m_nResumeDelay;

	Event m_ChangeStateEvent;
	Semaphore m_StateSemaphore;
	enum State {STATE_RUNNING, STATE_SUSPENDED, STATE_RESUMING, STATE_STOPPING};
	//Change the internal state of the periodic action thread's state machine
	void ChangeState(State newState);
	//Get the internal state of the periodic action thread's state machine
	State GetState();

	State m_CurrentState;
	bool m_bRunning;
};

#endif //_PERIODIC_ACTION_THREAD_H
