use ui;
use progvis:net;

on Ui:

/**
 * Data for the entire Problem dialog. Allows asking for different types of problems.
 */
class ProblemData {
	private Str[] names;
	private View[] views;

	init() {
		init {}
	}

	void add(Str name, View view) {
		names << name;
		views << view;
	}

	Nat count() { names.count; }
	Str name(Nat id) { names[id]; }
	View view(Nat id) { views[id]; }
	Str[] allNames() { names; }
}

/**
 * Data for some view. Controls what kind of information shall be displayed.
 *
 * Does not actually store any data, but encapsulates the queries that need to be done to acquire the data.
 */
class View {
	protected Client client;

	init(Client client) {
		init { client = client; }
	}

	value Row {
		// Some ID used to refer to this row.
		Int id;

		// Data shown to the user.
		Str[] data;

		init(Int id, Str[] data) {
			init { id = id; data = data; }
		}
	}

	Str[] labels() : abstract;
	Row[] rows() : abstract;

	protected Str[] problemLabels() {
		["ID", "Problem name", "Author", "Solved"];
	}

	protected Row[] getProblems(Request req) {
		if (ans = client.query(req) as ProblemResponse) {
			ans.problems.sort();
			Row[] data;
			for (r in ans.problems) {
				data << Row(r.id, [r.id.toS, r.title, r.author, r.solutions.toS]);
			}
			return data;
		} else {
			return [];
		}
	}

	// Find an appropriate panel for when a row has been clicked.
	Panel details(Int id) : abstract;
}

/**
 * View for problems that we can solve (e.g. unsolved problems).
 */
class UnsolvedView extends View {
	private Callback cb;
	private Bool improved;

	init(Client client, Callback cb, Bool improved) {
		init(client) {
			cb = cb;
			improved = improved;
		}
	}

	Str[] labels() : override {
		problemLabels();
	}

	Row[] rows() : override {
		getProblems(ChallengesRequest(false, improved));
	}

	Panel details(Int id) : override {
		Problem? parent;
		if (response = client.query(StatsRequest(id, false)) as StatsResponse) {
			parent = response.parent;
		}

		StatsPanel(id, parent, true, cb);
	}
}

/**
 * View for problems that we have solved.
 */
class SolvedView extends View {
	private Callback cb;

	init(Client client, Callback cb) {
		init(client) {
			cb = cb;
		}
	}

	Str[] labels() : override {
		problemLabels();
	}

	Row[] rows() : override {
		getProblems(ChallengesRequest(true, false));
	}

	Panel details(Int id) : override {
		if (response = client.query(StatsRequest(id, true)) as StatsResponse) {
			MultiPanel root;
			root.add(StatsPanel(id, response.parent, false, cb));
			root.add(TablePanel(SolutionView(client, response.solutions, cb)));
			root;
		} else {
			StatsPanel(-1, null, false, cb);
		}
	}
}

/**
 * View for own problems.
 */
class OwnView extends SolvedView {
	init(Client client, Callback cb) {
		init(client, cb) {}
	}

	Row[] rows() : override {
		getProblems(OwnProblemsRequest());
	}
}

/**
 * View for solutions.
 */
class SolutionView extends View {
	private Solution[] solutions;
	private Callback cb;

	init(Client client, Solution[] solutions, Callback cb) {
		init(client) { solutions = solutions; cb = cb; }
		solutions.sort();
	}

	Str[] labels() : override {
		["Author", "Solution"];
	}

	Row[] rows() : override {
		Row[] result;
		for (r in solutions) {
			Str type = if (r.solution.empty) {
				"Looks correct to me!";
			} else if (r.improved.empty) {
				"I found \"${r.type}\"...";
			} else {
				"I found \"${r.type}\" and tried to fix it";
			};
			result << Row(r.id, [r.author, type]);
		}
		result;
	}

	Panel details(Int id) : override {
		// TODO: What to show here?
		// We should have "show problem" (if available), "show improvement" (if available).
		for (s in solutions) {
			if (s.id == id)
				return SolutionPanel(s, cb);
		}

		// Good for "zero height".
		return MultiPanel();
	}
}
