
#include <usual/string.h>

#include "tinytest.h"
#include "tinytest_macros.h"

#define str_check(a, b) tt_str_op(a, ==, b)

#include <usual/crypto/hmac.h>
#include <usual/crypto/md5.h>
#include <usual/crypto/sha1.h>
#include <usual/crypto/sha256.h>
#include <usual/crypto/sha512.h>
#include <usual/crypto/keccak.h>
#include <usual/crypto/digest.h>
#include <usual/cxalloc.h>

static const char *mkhex(const uint8_t *src, int len)
{
	static char buf[1024 + 1];
	static const char hextbl[] = "0123456789abcdef";
	int i;
	for (i = 0; i < len; i++) {
		buf[i*2] = hextbl[src[i] >> 4];
		buf[i*2+1] = hextbl[src[i] & 15];
	}
	buf[i*2] = 0;
	return buf;
}

static int hexval(char v)
{
        if (v >= '0' && v <= '9')
                return v - '0';
        if (v >= 'a' && v <= 'f')
                return v - 'a' + 10;
        if (v >= 'A' && v <= 'F')
                return v - 'A' + 10;
        return -1;
}

static uint8_t *fromhex(const char *input, int len)
{
        uint8_t *res;
        const char *s = input;
        int i, b, b1, b2;

        res = malloc(len+1);
        if (!res)
                return NULL;

        for (i = 0; i < len; i++) {
                if (*s == '\0')
                        s = input;
                b1 = hexval(*s++);
                b2 = hexval(*s++);
                b = (b1 << 4) | b2;
                if (b < 0) {
			free(res);
			return NULL;
		}
                res[i] = b;
        }

        return res;
}

static const char *run_hash(const char *str, const char *hexstr, const struct DigestInfo *impl)
{
	struct DigestContext *ctx;
	uint8_t res[512];
	uint8_t res2[512];
	int i, len, step;
	int reslen;
	uint8_t *buf = NULL;

	if (hexstr) {
		len = strlen(hexstr) / 2;
		buf = fromhex(hexstr, len);
		if (!buf)
			return "NOMEM";
		str = (char *)buf;
	} else {
		len = strlen(str);
	}

	ctx = digest_new(impl, USUAL_ALLOC);
	if (!ctx)
		return "NOMEM";
	reslen = digest_result_len(ctx);

	digest_update(ctx, str, len);
	digest_final(ctx, res);

	digest_reset(ctx);
	step = 3;
	for (i = 0; i < len; i += step)
		digest_update(ctx, str+i,
			      (i + step <= len) ? (step) : (len - i));
	digest_final(ctx, res2);

	digest_free(ctx);

	if (buf)
		free(buf);

	if (memcmp(res, res2, reslen) != 0)
		return "FAIL";
	
	return mkhex(res, reslen);
}

/*
 * MD5
 */

static const char *run_md5(const char *str)
{
	return run_hash(str, NULL, digest_MD5());
}

static void test_md5(void *ptr)
{
	str_check(run_md5(""), "d41d8cd98f00b204e9800998ecf8427e");
	str_check(run_md5("a"), "0cc175b9c0f1b6a831c399e269772661");
	str_check(run_md5("abc"), "900150983cd24fb0d6963f7d28e17f72");
	str_check(run_md5("message digest"), "f96b697d7cb7938d525a2f31aaf161d0");
	str_check(run_md5("abcdefghijklmnopqrstuvwxyz"), "c3fcd3d76192e4007dfb496cca67e13b");
	str_check(run_md5("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"), "d174ab98d277d9f5a5611c2c9f419d9f");
	str_check(run_md5("12345678901234567890123456789012345678901234567890123456789012345678901234567890"), "57edf4a22be3c955ac49da2e2107b67a");
end:;
}

/*
 * SHA1
 */

static const char *run_sha1(const char *str)
{
	return run_hash(str, NULL, digest_SHA1());
}

static void test_sha1(void *ptr)
{
	str_check(run_sha1(""), "da39a3ee5e6b4b0d3255bfef95601890afd80709");
	str_check(run_sha1("a"), "86f7e437faa5a7fce15d1ddcb9eaeaea377667b8");
	str_check(run_sha1("abc"), "a9993e364706816aba3e25717850c26c9cd0d89d");
	str_check(run_sha1("message digest"), "c12252ceda8be8994d5fa0290a47231c1d16aae3");
	str_check(run_sha1("abcdefghijklmnopqrstuvwxyz"), "32d10c7b8cf96570ca04ce37f2a19d84240d3a89");
	str_check(run_sha1("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"), "761c457bf73b14d27e9e9265c46f4b4dda11f940");
	str_check(run_sha1("12345678901234567890123456789012345678901234567890123456789012345678901234567890"), "50abf5706a150990a08b2c5ea40fa0e585554732");
end:;
}

/*
 * SHA224
 */

static const char *run_sha224(const char *str)
{
	return run_hash(str, NULL, digest_SHA224());
}

static void test_sha224(void *ptr)
{
	str_check(run_sha224(""), "d14a028c2a3a2bc9476102bb288234c415a2b01f828ea62ac5b3e42f");
	str_check(run_sha224("a"), "abd37534c7d9a2efb9465de931cd7055ffdb8879563ae98078d6d6d5");
	str_check(run_sha224("abc"), "23097d223405d8228642a477bda255b32aadbce4bda0b3f7e36c9da7");
	str_check(run_sha224("abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"), "75388b16512776cc5dba5da1fd890150b0c6455cb4f58b1952522525");
	str_check(run_sha224("12345678901234567890123456789012345678901234567890123456789012345678901234567890"), "b50aecbe4e9bb0b57bc5f3ae760a8e01db24f203fb3cdcd13148046e");
end:;
}

/*
 * SHA256
 */

static const char *run_sha256(const char *str)
{
	return run_hash(str, NULL, digest_SHA256());
}

static void test_sha256(void *ptr)
{
	str_check(run_sha256(""), "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855");
	str_check(run_sha256("a"), "ca978112ca1bbdcafac231b39a23dc4da786eff8147c4e72b9807785afee48bb");
	str_check(run_sha256("abc"), "ba7816bf8f01cfea414140de5dae2223b00361a396177a9cb410ff61f20015ad");
	str_check(run_sha256("abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"), "248d6a61d20638b8e5c026930c3e6039a33ce45964ff2167f6ecedd419db06c1");
	str_check(run_sha256("12345678901234567890123456789012345678901234567890123456789012345678901234567890"), "f371bc4a311f2b009eef952dd83ca80e2b60026c8e935592d0f9c308453c813e");
end:;
}

/*
 * SHA384
 */

static const char *run_sha384(const char *str)
{
	return run_hash(str, NULL, digest_SHA384());
}

static void test_sha384(void *ptr)
{
	str_check(run_sha384(""), "38b060a751ac96384cd9327eb1b1e36a21fdb71114be07434c0cc7bf63f6e1da274edebfe76f65fbd51ad2f14898b95b");
	str_check(run_sha384("a"), "54a59b9f22b0b80880d8427e548b7c23abd873486e1f035dce9cd697e85175033caa88e6d57bc35efae0b5afd3145f31");
	str_check(run_sha384("abc"), "cb00753f45a35e8bb5a03d699ac65007272c32ab0eded1631a8b605a43ff5bed8086072ba1e7cc2358baeca134c825a7");
	str_check(run_sha384("abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"),
		  "3391fdddfc8dc7393707a65b1b4709397cf8b1d162af05abfe8f450de5f36bc6b0455a8520bc4e6f5fe95b1fe3c8452b");
	str_check(run_sha384("abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu"),
		  "09330c33f71147e83d192fc782cd1b4753111b173b3b05d22fa08086e3b0f712fcc7c71a557e2db966c3e9fa91746039");
	str_check(run_sha384("abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz"),
		  "3d208973ab3508dbbd7e2c2862ba290ad3010e4978c198dc4d8fd014e582823a89e16f9b2a7bbc1ac938e2d199e8bea4");
end:;
}

/*
 * SHA512
 */

static const char *run_sha512(const char *str)
{
	return run_hash(str, NULL, digest_SHA512());
}

static void test_sha512(void *ptr)
{
	str_check(run_sha512(""), "cf83e1357eefb8bdf1542850d66d8007d620e4050b5715dc83f4a921d36ce9ce47d0d13c5d85f2b0ff8318d2877eec2f63b931bd47417a81a538327af927da3e");
	str_check(run_sha512("a"), "1f40fc92da241694750979ee6cf582f2d5d7d28e18335de05abc54d0560e0f5302860c652bf08d560252aa5e74210546f369fbbbce8c12cfc7957b2652fe9a75");
	str_check(run_sha512("abc"), "ddaf35a193617abacc417349ae20413112e6fa4e89a97ea20a9eeee64b55d39a2192992a274fc1a836ba3c23a3feebbd454d4423643ce80e2a9ac94fa54ca49f");
	str_check(run_sha512("abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"),
		  "204a8fc6dda82f0a0ced7beb8e08a41657c16ef468b228a8279be331a703c33596fd15c13b1b07f9aa1d3bea57789ca031ad85c7a71dd70354ec631238ca3445");
	str_check(run_sha512("abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz"),
		  "930d0cefcb30ff1133b6898121f1cf3d27578afcafe8677c5257cf069911f75d8f5831b56ebfda67b278e66dff8b84fe2b2870f742a580d8edb41987232850c9");
	str_check(run_sha512("abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu"),
		  "8e959b75dae313da8cf4f72814fc143f8f7779c6eb9f7fa17299aeadb6889018501d289e4900f7e4331b99dec4b5433ac7d329eeb6dd26545e96e55b874be909");
end:;
}

/*
 * Keccak-224
 */

static const char *run_keccak224(const char *hex)
{
	return run_hash(NULL, hex, digest_KECCAK224());
}

static void test_keccak224(void *ptr)
{
	str_check(run_keccak224(""), "f71837502ba8e10837bdd8d365adb85591895602fc552b48b7390abd");
	str_check(run_keccak224("CC"), "a9cab59eb40a10b246290f2d6086e32e3689faf1d26b470c899f2802");
	str_check(run_keccak224("41FB"), "615ba367afdc35aac397bc7eb5d58d106a734b24986d5d978fefd62c");
	str_check(run_keccak224("1F877C"), "6f9d2898efd096baaaaab2e97482ddb6389b8e6caa964b7a0e347e13");
	str_check(run_keccak224("C1ECFDFC"), "e405869da1464a705700a3cbce131aabeeba9c8d2fe6576b21bcbe16");
	str_check(run_keccak224("21F134AC57"), "5573da2b02216a860389a581f6e9fb8d805e9e02f6fa911701eee298");
	str_check(run_keccak224("C6F50BB74E29"), "163c9060163aa66b8b7c0cfaa65d934bff219bcbc267187caba0042f");
	str_check(run_keccak224("119713CC83EEEF"), "cfc04c6f8463ddab24cdf8b8652bd11df23dd1b95f118328dd01580e");
	str_check(run_keccak224("4A4F202484512526"), "7a5c2cb3f999dd00eff7399963314ca647dd0e5ae1bddec611f8338d");
	str_check(run_keccak224("B32D95B0B9AAD2A8816DE6D06D1F86008505BD8C14124F6E9A163B5A2ADE55F835D0EC3880EF50700D3B25E42CC0AF050CCD1BE5E555B23087E04D7BF9813622780C7313A1954F8740B6EE2D3F71F768DD417F520482BD3A08D4F222B4EE9DBD015447B33507DD50F3AB4247C5DE9A8ABD62A8DECEA01E3B87C8B927F5B08BEB37674C6F8E380C04"),
		  "42275c296937745758ff2b7bee9a897191ae87e42bd10198d9466c19");
	str_check(run_keccak224("04410E31082A47584B406F051398A6ABE74E4DA59BB6F85E6B49E8A1F7F2CA00DFBA5462C2CD2BFDE8B64FB21D70C083F11318B56A52D03B81CAC5EEC29EB31BD0078B6156786DA3D6D8C33098C5C47BB67AC64DB14165AF65B44544D806DDE5F487D5373C7F9792C299E9686B7E5821E7C8E2458315B996B5677D926DAC57B3F22DA873C601016A0D"),
		  "143f9055eb1f736729c77721fb65ed5ee142f6e969132fb22989c11f");
	str_check(run_keccak224("3A3A819C48EFDE2AD914FBF00E18AB6BC4F14513AB27D0C178A188B61431E7F5623CB66B23346775D386B50E982C493ADBBFC54B9A3CD383382336A1A0B2150A15358F336D03AE18F666C7573D55C4FD181C29E6CCFDE63EA35F0ADF5885CFC0A3D84A2B2E4DD24496DB789E663170CEF74798AA1BBCD4574EA0BBA40489D764B2F83AADC66B148B4A0CD95246C127D5871C4F11418690A5DDF01246A0C80A43C70088B6183639DCFDA4125BD113A8F49EE23ED306FAAC576C3FB0C1E256671D817FC2534A52F5B439F72E424DE376F4C565CCA82307DD9EF76DA5B7C4EB7E085172E328807C02D011FFBF33785378D79DC266F6A5BE6BB0E4A92ECEEBAEB1"),
		  "5af56987ea9cf11fcd0eac5ebc14b037365e9b1123e31cb2dfc7929a");
end:;
}

/*
 * Keccak-256
 */

static const char *run_keccak256(const char *hex)
{
	return run_hash(NULL, hex, digest_KECCAK256());
}

static void test_keccak256(void *ptr)
{
	str_check(run_keccak256(""), "c5d2460186f7233c927e7db2dcc703c0e500b653ca82273b7bfad8045d85a470");
	str_check(run_keccak256("CC"), "eead6dbfc7340a56caedc044696a168870549a6a7f6f56961e84a54bd9970b8a");
	str_check(run_keccak256("41FB"), "a8eaceda4d47b3281a795ad9e1ea2122b407baf9aabcb9e18b5717b7873537d2");
	str_check(run_keccak256("1F877C"), "627d7bc1491b2ab127282827b8de2d276b13d7d70fb4c5957fdf20655bc7ac30");
	str_check(run_keccak256("C1ECFDFC"), "b149e766d7612eaf7d55f74e1a4fdd63709a8115b14f61fcd22aa4abc8b8e122");
	str_check(run_keccak256("21F134AC57"), "67f05544dbe97d5d6417c1b1ea9bc0e3a99a541381d1cd9b08a9765687eb5bb4");
	str_check(run_keccak256("C6F50BB74E29"), "923062c4e6f057597220d182dbb10e81cd25f60b54005b2a75dd33d6dac518d0");
	str_check(run_keccak256("119713CC83EEEF"), "feb8405dcd315d48c6cbf7a3504996de8e25cc22566efec67433712eda99894f");
	str_check(run_keccak256("4A4F202484512526"), "e620d8f2982b24fedaaa3baa9b46c3f9ce204ee356666553ecb35e15c3ff9bf9");
	str_check(run_keccak256("B32D95B0B9AAD2A8816DE6D06D1F86008505BD8C14124F6E9A163B5A2ADE55F835D0EC3880EF50700D3B25E42CC0AF050CCD1BE5E555B23087E04D7BF9813622780C7313A1954F8740B6EE2D3F71F768DD417F520482BD3A08D4F222B4EE9DBD015447B33507DD50F3AB4247C5DE9A8ABD62A8DECEA01E3B87C8B927F5B08BEB37674C6F8E380C04"),
		  "e717a7769448abbe5fef8187954a88ac56ded1d22e63940ab80d029585a21921");
	str_check(run_keccak256("04410E31082A47584B406F051398A6ABE74E4DA59BB6F85E6B49E8A1F7F2CA00DFBA5462C2CD2BFDE8B64FB21D70C083F11318B56A52D03B81CAC5EEC29EB31BD0078B6156786DA3D6D8C33098C5C47BB67AC64DB14165AF65B44544D806DDE5F487D5373C7F9792C299E9686B7E5821E7C8E2458315B996B5677D926DAC57B3F22DA873C601016A0D"),
		  "a95d50b50b4545f0947441df74a1e9d74622eb3baa49c1bbfc3a0cce6619c1aa");
	str_check(run_keccak256("3A3A819C48EFDE2AD914FBF00E18AB6BC4F14513AB27D0C178A188B61431E7F5623CB66B23346775D386B50E982C493ADBBFC54B9A3CD383382336A1A0B2150A15358F336D03AE18F666C7573D55C4FD181C29E6CCFDE63EA35F0ADF5885CFC0A3D84A2B2E4DD24496DB789E663170CEF74798AA1BBCD4574EA0BBA40489D764B2F83AADC66B148B4A0CD95246C127D5871C4F11418690A5DDF01246A0C80A43C70088B6183639DCFDA4125BD113A8F49EE23ED306FAAC576C3FB0C1E256671D817FC2534A52F5B439F72E424DE376F4C565CCA82307DD9EF76DA5B7C4EB7E085172E328807C02D011FFBF33785378D79DC266F6A5BE6BB0E4A92ECEEBAEB1"),
		  "348fb774adc970a16b1105669442625e6adaa8257a89effdb5a802f161b862ea");
end:;
}

/*
 * Keccak-384
 */

static const char *run_keccak384(const char *hex)
{
	return run_hash(NULL, hex, digest_KECCAK384());
}

static void test_keccak384(void *ptr)
{
	str_check(run_keccak384(""), "2c23146a63a29acf99e73b88f8c24eaa7dc60aa771780ccc006afbfa8fe2479b2dd2b21362337441ac12b515911957ff");
	str_check(run_keccak384("CC"), "1b84e62a46e5a201861754af5dc95c4a1a69caf4a796ae405680161e29572641f5fa1e8641d7958336ee7b11c58f73e9");
	str_check(run_keccak384("41FB"), "495cce2714cd72c8c53c3363d22c58b55960fe26be0bf3bbc7a3316dd563ad1db8410e75eefea655e39d4670ec0b1792");
	str_check(run_keccak384("1F877C"), "b0665c345f45e6de145b0190335ef5d5aa59e0b49fc1425d5eae7355ea442284cb8a2152d565ebdf2810eccab15af04f");
	str_check(run_keccak384("C1ECFDFC"), "f1850b2abb24f3fd683c701582789d9e92b6a45f9c345f9dae7f7997c8c910e88003e592e59281cf92c92d6b51a1afd1");
	str_check(run_keccak384("21F134AC57"), "68d437327f158287c304bbaf36f782f497da2c480a1fbb268682362218641f9070a014919ad7331c49beefccb437fe9a");
	str_check(run_keccak384("C6F50BB74E29"), "03566ec003ff55184f0c85beebc6d1ecf5e5d082d8d40137246f8fd42bce097c09418845ef60286fdd894a00fd2d6589");
	str_check(run_keccak384("119713CC83EEEF"), "790d700fa34d6a835be311b639474780148a2f087ac2fa86e8a1a433ec7a04fcbfc5284a3e188b7d91c6d094eafbeecb");
	str_check(run_keccak384("4A4F202484512526"), "638e65758a297cb09ded1ac5b9e8f779802000ab791f67f33c60be36443793adcc8a4a58e98688157a41784f02a4bcb2");
	str_check(run_keccak384("B32D95B0B9AAD2A8816DE6D06D1F86008505BD8C14124F6E9A163B5A2ADE55F835D0EC3880EF50700D3B25E42CC0AF050CCD1BE5E555B23087E04D7BF9813622780C7313A1954F8740B6EE2D3F71F768DD417F520482BD3A08D4F222B4EE9DBD015447B33507DD50F3AB4247C5DE9A8ABD62A8DECEA01E3B87C8B927F5B08BEB37674C6F8E380C04"),
		  "278e83cff1ff6cc4b3ac41f3879da87ae63b535b43815e273687a4cc519855b452cb6af0198bb9fd0f3e43739bc0cdd7");
	str_check(run_keccak384("04410E31082A47584B406F051398A6ABE74E4DA59BB6F85E6B49E8A1F7F2CA00DFBA5462C2CD2BFDE8B64FB21D70C083F11318B56A52D03B81CAC5EEC29EB31BD0078B6156786DA3D6D8C33098C5C47BB67AC64DB14165AF65B44544D806DDE5F487D5373C7F9792C299E9686B7E5821E7C8E2458315B996B5677D926DAC57B3F22DA873C601016A0D"),
		  "aa4b5a5fb94fe19578f33323ba1eefc5b6ed70b34bc70193f386c99f73863611af20581b4b1b3ed776df9e235d3d4e45");
	str_check(run_keccak384("3A3A819C48EFDE2AD914FBF00E18AB6BC4F14513AB27D0C178A188B61431E7F5623CB66B23346775D386B50E982C493ADBBFC54B9A3CD383382336A1A0B2150A15358F336D03AE18F666C7573D55C4FD181C29E6CCFDE63EA35F0ADF5885CFC0A3D84A2B2E4DD24496DB789E663170CEF74798AA1BBCD4574EA0BBA40489D764B2F83AADC66B148B4A0CD95246C127D5871C4F11418690A5DDF01246A0C80A43C70088B6183639DCFDA4125BD113A8F49EE23ED306FAAC576C3FB0C1E256671D817FC2534A52F5B439F72E424DE376F4C565CCA82307DD9EF76DA5B7C4EB7E085172E328807C02D011FFBF33785378D79DC266F6A5BE6BB0E4A92ECEEBAEB1"),
		  "6bff1c8405a3fe594e360e3bccea1ebcd509310dc79b9e45c263783d7a5dd662c6789b18bd567dbdda1554f5bee6a860");
end:;
}

/*
 * Keccak-512
 */

static const char *run_keccak512(const char *hex)
{
	return run_hash(NULL, hex, digest_KECCAK512());
}

static void test_keccak512(void *ptr)
{
	str_check(run_keccak512(""), "0eab42de4c3ceb9235fc91acffe746b29c29a8c366b7c60e4e67c466f36a4304c00fa9caf9d87976ba469bcbe06713b435f091ef2769fb160cdab33d3670680e");
	str_check(run_keccak512("CC"), "8630c13cbd066ea74bbe7fe468fec1dee10edc1254fb4c1b7c5fd69b646e44160b8ce01d05a0908ca790dfb080f4b513bc3b6225ece7a810371441a5ac666eb9");
	str_check(run_keccak512("41FB"), "551da6236f8b96fce9f97f1190e901324f0b45e06dbbb5cdb8355d6ed1dc34b3f0eae7dcb68622ff232fa3cece0d4616cdeb3931f93803662a28df1cd535b731");
	str_check(run_keccak512("1F877C"), "eb7f2a98e00af37d964f7d8c44c1fb6e114d8ee21a7b976ae736539efdc1e3fe43becef5015171e6da30168cae99a82c53fa99042774ef982c01626a540f08c0");
	str_check(run_keccak512("C1ECFDFC"), "952d4c0a6f0ef5ce438c52e3edd345ea00f91cf5da8097c1168a16069e958fc05bad90a0c5fb4dd9ec28e84b226b94a847d6bb89235692ef4c9712f0c7030fae");
	str_check(run_keccak512("21F134AC57"), "2e76d93affd62b92fc4f29cb83efbe4ba21d88426aa7f075bfc20960ea258787898172e17045af43ab1fe445532be0185fbea84d9be788b05f14dbf4856a5254");
	str_check(run_keccak512("C6F50BB74E29"), "40fa8074e1e509b206448fbe757d9494b9b51e8d6e674a67f53c11ef92e96c3ea08b95ebd4172b020010cd6cf29539a34d6bfa002a2042787aa8d879a0f5b54c");
	str_check(run_keccak512("119713CC83EEEF"), "d1116786a3c1ea46a8f22d82abb4c5d06dc0691b2e747ac9726d0b290e6959f7b23428519a656b237695e56403855ec4c98db0cf87f31b6ceabf2b9b8589b713");
	str_check(run_keccak512("4A4F202484512526"), "f326c7c126ddc277922760feef77c9bab6fb5d3430f652593703d7c5e30135cd0b0575257509a624184330d6ab1f508a666391b5d4690426b4e05301891df897");
	str_check(run_keccak512("B32D95B0B9AAD2A8816DE6D06D1F86008505BD8C14124F6E9A163B5A2ADE55F835D0EC3880EF50700D3B25E42CC0AF050CCD1BE5E555B23087E04D7BF9813622780C7313A1954F8740B6EE2D3F71F768DD417F520482BD3A08D4F222B4EE9DBD015447B33507DD50F3AB4247C5DE9A8ABD62A8DECEA01E3B87C8B927F5B08BEB37674C6F8E380C04"),
		  "a6054ffc3d81591be964c4b004a3a21142365b59ee98b2873d488293f93a8d7154bf72100012c60d3c9418f6af8ea66372cb4703f5f6381de6d4b9b98cff1e90");
	str_check(run_keccak512("04410E31082A47584B406F051398A6ABE74E4DA59BB6F85E6B49E8A1F7F2CA00DFBA5462C2CD2BFDE8B64FB21D70C083F11318B56A52D03B81CAC5EEC29EB31BD0078B6156786DA3D6D8C33098C5C47BB67AC64DB14165AF65B44544D806DDE5F487D5373C7F9792C299E9686B7E5821E7C8E2458315B996B5677D926DAC57B3F22DA873C601016A0D"),
		  "b0e54a12fdba0738898f1bbf0ba81f81de77648d8d14c20bdd5d90f300d382e069f5dba7eec6b23168b008b9f39c2b93fd742a5902a5e02728f57712d6a61d4e");
	str_check(run_keccak512("3A3A819C48EFDE2AD914FBF00E18AB6BC4F14513AB27D0C178A188B61431E7F5623CB66B23346775D386B50E982C493ADBBFC54B9A3CD383382336A1A0B2150A15358F336D03AE18F666C7573D55C4FD181C29E6CCFDE63EA35F0ADF5885CFC0A3D84A2B2E4DD24496DB789E663170CEF74798AA1BBCD4574EA0BBA40489D764B2F83AADC66B148B4A0CD95246C127D5871C4F11418690A5DDF01246A0C80A43C70088B6183639DCFDA4125BD113A8F49EE23ED306FAAC576C3FB0C1E256671D817FC2534A52F5B439F72E424DE376F4C565CCA82307DD9EF76DA5B7C4EB7E085172E328807C02D011FFBF33785378D79DC266F6A5BE6BB0E4A92ECEEBAEB1"),
		  "81950e7096d31d4f22e3db71cac725bf59e81af54c7ca9e6aeee71c010fc5467466312a01aa5c137cfb140646941556796f612c9351268737c7e9a2b9631d1fa");
end:;
}

/*
 * Keccak-Stream
 */

static const char *run_keccakS(const char *hex)
{
	struct DigestContext *ctx;
	uint8_t res[512];
	int len;
	unsigned reslen;
	uint8_t *buf = NULL;

	len = strlen(hex) / 2;
	buf = fromhex(hex, len);

	ctx = digest_new(digest_KECCAK_STREAM(), USUAL_ALLOC);
	if (!ctx)
		return "NOMEM";
	digest_update(ctx, buf, len);
	free(buf);

	reslen = 0;
	while (reslen < sizeof(res)) {
		digest_final(ctx, res + reslen);
		reslen += digest_result_len(ctx);
	}

	return mkhex(res, reslen);
}

static void test_keccak_stream(void *ptr)
{
	str_check(run_keccakS(""),
		  "6753e3380c09e385d0339eb6b050a68f66cfd60a73476e6fd6adeb72f5edd7c6f04a5d017a19cbe291935855b4860f69df04c98aa78b407a9ba9826f7266ef14ba6d3f90c4fe154d27c2858ea6db8c117411a1bc5c499410c391b298f37bf636b0f5c31dbd6487a7d3d8cf2a97b619697e66d894299b8b4d80e0498538e18544c3a2fa33f0bfb1cfef8da7875c4967f332c7fc93c050e81fb404f9a91503d6010ee16f50b4ed0bc563ba8431668b003d7e2e6f226cb7fa93bb2e132c861fdc2141457589a63ecf05481126a7c2de941a2fdec71cb70de81887b9014223865e79c4ffe82dae83c1fc484b9a07a7e52b135f4ae3a0e09247ea4e2625e9349b0ac73f24cb418df6dcb49ca37860298ada18aa23595b5096ef789de3edf3826817fff4f71102a01e1d2599f2958d5c186f5b11f5feedb61bb732dbb42d18b1e77258a8f211bf95c9f47f19603ec419ff879aea41a4811344d016bbc4f9496741c469cca425c5be73543219af40796c0b9ff14aeaa70c5e22e4bb1346a3ddfedd8a559104e4704f1227d42918ae3f7404fbf3c6340a486e776aabcc34190f87da4bd954b83386255a0e34df05ca2e781faf6fe66475852481fce20798a56629abfac408760ce64606008a3b568c88aba1c6df3381e0765567ea84b2ce4b441cf1eefaa32125d5139361a632b3008566a2e8af1055cb06ae462b6bf87b34a9770618e6");
	str_check(run_keccakS("CC"),
		  "56b97029b479ff5dd15f17d12983e3b835bb0531d9b8d49b103b025ca53f991741298e961d1fad00fc365c7761bfb278ae473980d612c1629e075a3fdbae7f82b0f0af54df187f358852e19ea4347cf5ceea676a1dce3a47447e237fd74204f9a4b7f7c9cc7cc8b865b1d554e2f5f4a8ee17dbdde7267894558a20972c9eb6cf5f62ce9151437718ed4aff08fa76d803806e6ce47d229aae839369e31888b26429e27bc3756021cb51498bcf2527d4bb04838bc1ceed9985a2a66ff8cb8c2d58b7099304e7f9622c583b093024a5fcde2be781474c159df24d77d328c298f5766a8a0dbf7ae790a509ccf59e0cacd0abf21492e0095a87ecdb55990093917aaa96d7f68b7b859b8094aec0ddb6fb352a6cc1f007fa988ed764f5d6f21f9d8ade9ce7aca4de6570da39d9acceb46d2582fa4c4231de0b736fb341041d24cfae6c0761f43a2cf7383f38742579218afcab53d2e6816640de05644d877558e965b1a28406999f31ccc43ac0b02bc5448b66ad3b6f8de04c0e25845c8671b6f0594909a057f17fd06031707c8b4599889c994a35c193dbf84a7a0919cd054f67ceb7965f420d02da3477efc8b55413c241adcf71cb10fe7e3e720b8c1736837b06e4b27461b71c6cac892437530bbfe05cf426272f80f11709b9db964f5dedab9e757c2f7a972b6a4c2443b03ad787ab1e243660bced739157a434800696841acea4");
	str_check(run_keccakS("41FB"),
		  "cbe96338dd8f04c0694299637aab223b6d60560c6bed7f69923aebb24fc61b84702403d39e7d081f7f7b714e3ba6e6221fe840f57a1e9bd775b90d59c9853695c2b11cd06f1054210d7d8155b908ff4ee14fdf859b6d5aa6bf76903be0af4a2ffd52b2b149da32c8e372f51826d4ca7dcd6516d167a0621aa88986d19a524dd352b9ca08f341d2267671f45e05892e1a5c604bb721bc8952dac20d559dc183656501cc34bc91e2aea930716b20539131ac5f9ae0a630e3691abe6e76935d21f99e3f2e531526360405004bd730388236a1197fe3715315e8ca40b4e5e6a07cab434264515c26451a7c1387d776bc225b851e9f7807c24a23f42fb47eb29697f6cd80cdbfb79a39675092ab582c5a6bb3284cd72a889601dc2745153fac80ff81c6648cb99facfe51862edc8b03c2c3ba5b83eb1d40d3937caf3d8e511485051d3e5431a19c1571b52e796cf032162292ecf2b490cd97c3e2fc2ca339021533cd1aa1c5e8b3f803767ae7585999a2b7d70c7b34324b36399a87c3c73866741cbef9355c1570309544697df9a82da28b6c5ce35556c5bef4e0a24e62f95e543cd3fad6d2ddeea3950e72867d67a8dcd4b338fd8341583fe0e04fff2d6ecdfcd4b41eb8434ee0e31f812b220494202fab9fcba09a9dd26b36637df5607c6d7cbede04868a2d512d11a16c24c4b8d566ce63932b85e7e1a8648f58857629bdc2ee92");
	str_check(run_keccakS("C1ECFDFC"),
		  "968a76720a2fff1ac9629cac7752c0958b6e19bc79fa31f210244d486645798b55f52581855da53a14139dd78e15f54c66bde1bcc5674f46de6164a86933b2cc99682f7118af25b9034cddac018e6d02f3890fa581c79cad5c6c2380a890ea470876e7bede8e8b78aa6f0cb271f54252dd018c7c9d393a06d60a78be8a5014b89eaa282dfec4e737e43a61cfaade58f9bdde9c6125daf34350b2b4e320f35b62dd0675c0515b943630e3f63880423864dbd70814cc9373c521e8f29bb4138388c92c4b6437a65469902e706ccce3777991a47c0ee9701217fb44cb02e674c7539e473d20352a7a875d6cf3a038e655d3d1a75852fc1859835cc181ef0c58b888a6673bb8275cfb9797f5e146a962d8deb535fc7b166af4fc95209a2dd4c515a0c04ddadfb77ba6b8da76c9ae9cce13538608603ed3550c685bacdc0a059f92dad364f8a72dfaf52faf43011c33b6a6235d9ac6611f346d955e1701f37713bb6f98a05337b1943556d497a4a686645fc359375408702ec45617c949d1209824e4627741cf2760e6ae84e0e1d395da4da5748d042bff19351ff20092df201aeab4d6c603693e6ab01dabaf009a1c8a93e713dacd4a0d93695a2f6ef4e59ab9a140a100766b25b86cbb3632aa73fb16f47b9839e4d0e8736a0b81cdd923f0be79c4c7ab1f7e1be0ed5f079f6e04b979a567e81bc0c1a236e3daa2754f195ff76261");
	str_check(run_keccakS("2B6DB7CED8665EBE9DEB080295218426BDAA7C6DA9ADD2088932CDFFBAA1C14129BCCDD70F369EFB149285858D2B1D155D14DE2FDB680A8B027284055182A0CAE275234CC9C92863C1B4AB66F304CF0621CD54565F5BFF461D3B461BD40DF28198E3732501B4860EADD503D26D6E69338F4E0456E9E9BAF3D827AE685FB1D817"),
		  "dcac84568f15cac076854ea692de95e473768a99df9ac2328ee423d02eeb8ee8e1d1706213c4415dc7aafa66476d8ebdddd8bf39e1de05ca76c36e7e975629331f3a33c3ca4091c82004e5891b7e276d4642ea61bde021871c9b5c8cfa82144b7a4144b44ebe6093e95c59305fd36a8741c4f2df65cb0b59f803cfdcf2ce4b8b54857a7f8ad8477a22cffc07f4a5ef76996837c9b3f8ff25375e669e68faebed8ac79673860f45fd6e7ee7ed630cb8582785eed432af1f4ca22f4fd4488dc3882de5268ee4c47be4eaa2e77692880d218a69578fb090ed8426c29217b72070be2a4bf0d7bcb480ff262d2dc3bfa9c9d88ff6f5284d240e988e4f9b956aaa36e804bb38ce1f5aa65e73624940c28cf816f4c4f00751bcc6cdc79131e96294d95d6bc98f58a2a687ea7b0769a6dd4f4988b2381631ec1967ef4fcc9efb7a7519783a33787850a8c752f36a4b3abf7e460d5689009a232bef1c33fb3d99069b0157764e4477e2ad68b3a99a05bd2d38288ddd416ee784bdc99e157dfdec61ecb0c49763f187e947c54a4ecbf5eeb76af5feeb222b0844cc1bb9f69cf3b291671bbe98c89ef5d656dfc77375c39cbc2a7ff6413b3ca99834d1845499a09bc111c8582f567d187147c5bbade2194871126dda67daf170079a618a77b8f06193e06f87d441687d1de6e5cda9c791728f837c945f2eb20327802b37fc6d9c2b125f4067");
	str_check(run_keccakS("3A3A819C48EFDE2AD914FBF00E18AB6BC4F14513AB27D0C178A188B61431E7F5623CB66B23346775D386B50E982C493ADBBFC54B9A3CD383382336A1A0B2150A15358F336D03AE18F666C7573D55C4FD181C29E6CCFDE63EA35F0ADF5885CFC0A3D84A2B2E4DD24496DB789E663170CEF74798AA1BBCD4574EA0BBA40489D764B2F83AADC66B148B4A0CD95246C127D5871C4F11418690A5DDF01246A0C80A43C70088B6183639DCFDA4125BD113A8F49EE23ED306FAAC576C3FB0C1E256671D817FC2534A52F5B439F72E424DE376F4C565CCA82307DD9EF76DA5B7C4EB7E085172E328807C02D011FFBF33785378D79DC266F6A5BE6BB0E4A92ECEEBAEB1"),
		  "9435fc671dfcfcdac149277e2caaa80ed3d4a2359300db892b8093dffa9442bb5c08f242f2fc2cb5f8388032299f1df47a57489a4fc0d66d88e483092320a471897fb6ade67897e5138c45f19174a4b1ae0e510fa390825d17568989c3659fc57b9345d7d93ee588cb2629c5770808195257bbf42b069576d94011989dc6ebc43cfc7cd27b6f9853904f3eb3842bbb37d2bd807f05468f5057f78373b6f34462095a1205c1fca0d15fbcf890ee78ab6f94cb778b5d6f3620e6e6d6ee688eecc619e22e25e0bb5e143a53472e4f1d1f91a8e625087b0f608770c4b9909749ab50ddcdac59bb3c975aba4dceb2b3a2c436ed103ed6d9c62cd63a69a0bdd2baabfbfd63eef34507637f5e8a16a4fcb33d66141781e10bc6262833ec6e2953cedd5f652b76fa042ec0d34ba20f5657e28c08b6b61dfa8da78cf997127e17a35d75ab35542fe6bb9ce5bd06119da6b497ac1ae12947b0c214de28ed5dda7815fb6d5def81025934c877cb91e923191581b508bbabdfe4bb2dd5af6af414bfa28830e4380355bdf2483cabd01b046956b85d5a34f46849ba1cc869f5babd1b41ec775fcb4b5fbad79661daf47dbe7bc6380bc5034bfe626526f3305abe270bbbf29280e58b71db269cf7962d9dc1731bd10d5633b1b10e76791c0fcfddf1c8263f17f8b68b1a0589fe5c9403d272fa133442980588bc1f385c3af240d8f195ab1a3400");
end:;
}

/*
 * HMAC
 */

static const char *run_hmac(const char *key, const char *str, const struct DigestInfo *impl)
{
	struct HMAC *ctx;
	uint8_t res[512];
	int len = strlen(str);
	int reslen;

	ctx = hmac_new(impl, key, strlen(key), USUAL_ALLOC);
	if (!ctx)
		return "NOMEM";
	reslen = hmac_result_len(ctx);

	hmac_update(ctx, str, len);
	hmac_final(ctx, res);

	return mkhex(res, reslen);
}

static const char *run_hmac_sha1(const char *key, const char *str)
{
	return run_hmac(key, str, digest_SHA1());
}

static void test_hmac(void *ptr)
{
	const char *long_key = (
		"quite a very long key, longer than a sha1 block size, "
		"so it needs to be sha-1d before being used as a key");
	const char *text = "The quick brown fox jumps over the lazy dog";

	str_check(run_hmac_sha1("", ""),
			  "fbdb1d1b18aa6c08324b7d64b71fb76370690e1d");

	str_check(run_hmac_sha1("shrt", ""),
			  "41fee95de96c437cf6c2f38363eb38eb0067ff64");

	str_check(run_hmac_sha1(long_key, ""),
		"496ca9bda3e523814ba7f99f68a2035e4de7702a");

	str_check(run_hmac_sha1(long_key, text),
		"924e1ee84da31f5f569a27dd6201533b42c999c6");

	str_check(run_hmac_sha1("key", text),
		"de7c9b85b8b78aa6bc8a7a36f70a90701c9db4d9");
end:;
}

/*
 * Launcher.
 */

struct testcase_t crypto_tests[] = {
	{ "md5", test_md5 },
	{ "sha1", test_sha1 },
	{ "sha224", test_sha224 },
	{ "sha256", test_sha256 },
	{ "sha384", test_sha384 },
	{ "sha512", test_sha512 },
	{ "keccak224", test_keccak224 },
	{ "keccak256", test_keccak256 },
	{ "keccak384", test_keccak384 },
	{ "keccak512", test_keccak512 },
	{ "keccakStream", test_keccak_stream },
	{ "hmac", test_hmac },
	END_OF_TESTCASES
};

