#![feature(test)]
extern crate chrono;
extern crate rfc822_sanitizer;
extern crate test;

use test::Bencher;
use rfc822_sanitizer::*;
use chrono::DateTime;

static VALID_DATES: &[&str] = &[
    "Mon, 10 Jul 2017 16:00:00 -0700",
    "Mon, 17 Jul 2017 17:00:00 -0700",
    "Mon, 24 Jul 2017 16:00:00 -0700",
    "Mon, 31 Jul 2017 16:00:00 -0700",
    "Wed, 30 Aug 2017 01:30:00 -0700",
    "Wed, 20 Sep 2017 10:00:00 +0000",
    "Wed, 13 Sep 2017 10:00:00 +0000",
    "Wed,  9 Aug 2017 10:00:00 +0000",
    "Wed,  2 Aug 2017 10:00:00 +0000",
    "Wed, 26 Jul 2017 10:00:00 +0000",
    "Wed, 19 Jul 2017 10:00:00 +0000",
    "Wed, 12 Jul 2017 10:00:00 +0000",
    "Wed, 28 Jun 2017 10:00:00 +0000",
    "Wed, 21 Jun 2017 10:00:00 +0000",
    "Wed, 14 Jun 2017 10:00:00 +0000",
    "Wed,  7 Jun 2017 10:00:00 +0000",
    "Wed, 31 May 2017 10:00:00 +0000",
    "Wed, 24 May 2017 10:00:00 +0000",
    "Wed, 17 May 2017 10:00:00 +0000",
    "Wed, 10 May 2017 10:00:00 +0000",
    "Wed,  3 May 2017 10:00:00 +0000",
    "Wed, 19 Apr 2017 10:00:00 +0000",
    "Wed, 12 Apr 2017 10:00:00 +0000",
    "Wed,  5 Apr 2017 10:00:00 +0000",
    "Wed, 29 Mar 2017 10:00:00 +0000",
    "Wed, 22 Mar 2017 10:00:00 +0000",
    "Wed, 15 Mar 2017 10:00:00 +0000",
    "Wed,  8 Mar 2017 11:00:00 +0000",
    "Wed,  1 Mar 2017 11:00:00 +0000",
    "Wed, 22 Feb 2017 11:00:00 +0000",
    "Wed, 15 Feb 2017 11:00:00 +0000",
    "Wed,  8 Feb 2017 11:00:00 +0000",
    "Wed,  1 Feb 2017 11:00:00 +0000",
    "Wed, 25 Jan 2017 11:00:00 +0000",
    "Fri, 13 Jan 2017 18:38:00 +0000",
    "Wed, 20 Sep 2017 03:30:00 +0000",
    "Wed, 13 Sep 2017 03:15:00 +0000",
    "Wed,  6 Sep 2017 03:15:00 +0000",
    "Wed, 30 Aug 2017 03:15:00 +0000",
    "Wed, 23 Aug 2017 03:15:00 +0000",
    "Wed, 16 Aug 2017 03:15:00 +0000",
    "Wed,  9 Aug 2017 03:15:00 +0000",
    "Wed,  2 Aug 2017 03:00:00 +0000",
    "Tue, 11 Jul 2017 17:14:45 +0000",
    "Thu,  3 Aug 2017 06:00:00 -0400",
    "Thu, 27 Jul 2017 06:00:00 -0400",
    "Thu, 20 Jul 2017 06:00:00 -0400",
    "Thu, 13 Jul 2017 06:00:00 -0400",
    "Thu,  6 Jul 2017 06:00:00 -0400",
    "Wed, 28 Jun 2017 06:00:00 -0400",
    "Wed, 17 Jul 2013 06:00:03 -0400",
    "Wed,  2 Apr 2014 06:00:03 -0400",
    "Thu, 14 Jan 2016 06:00:03 -0400",
    "Thu, 22 Jun 2017 06:00:00 -0400",
    "Thu, 15 Jun 2017 06:00:00 -0400",
    "Wed,  7 Jun 2017 06:00:00 -0400",
    "Thu,  1 Jun 2017 06:00:00 -0400",
    "Wed, 23 Dec 2015 06:00:03 -0400",
    "Fri, 14 Feb 2014 06:00:03 -0400",
    "Wed,  4 Dec 2013 06:00:03 -0400",
    "Tue, 20 Dec 2016 06:00:00 -0400",
    "Wed, 23 Nov 2016 06:00:00 -0400",
    "Fri,  5 Aug 2016 06:00:00 -0400",
    "Thu,  9 Jun 2016 12:00:00 -0400",
    "Wed, 10 May 2017 06:00:00 -0400",
    "Wed, 22 Feb 2017 06:00:00 -0400",
    "Wed, 15 Feb 2017 06:00:00 -0400",
];

static INVALID_DATES: &[&str] = &[
    "Mon, 10 July 2017 16:00:00 PDT",
    "Mon, 17 July 2017 17:00:00 PDT",
    "Mon, 24 July 2017 16:00:00 PDT",
    "Mon, 31 July 2017 16:00:00 PDT",
    "Thu, 30 Aug 2017 1:30:00 PDT",
    "Wed, 20 Sep 2017 10:00:00 -0000",
    "Wed, 13 Sep 2017 10:00:00 -0000",
    "Wed, 09 Aug 2017 10:00:00 -0000",
    "Wed, 02 Aug 2017 10:00:00 -0000",
    "Wed, 26 Jul 2017 10:00:00 -0000",
    "Wed, 19 Jul 2017 10:00:00 -0000",
    "Wed, 12 Jul 2017 10:00:00 -0000",
    "Wed, 28 Jun 2017 10:00:00 -0000",
    "Wed, 21 Jun 2017 10:00:00 -0000",
    "Wed, 14 Jun 2017 10:00:00 -0000",
    "Wed, 07 Jun 2017 10:00:00 -0000",
    "Wed, 31 May 2017 10:00:00 -0000",
    "Wed, 24 May 2017 10:00:00 -0000",
    "Wed, 17 May 2017 10:00:00 -0000",
    "Wed, 10 May 2017 10:00:00 -0000",
    "Wed, 03 May 2017 10:00:00 -0000",
    "Wed, 19 Apr 2017 10:00:00 -0000",
    "Wed, 12 Apr 2017 10:00:00 -0000",
    "Wed, 05 Apr 2017 10:00:00 -0000",
    "Wed, 29 Mar 2017 10:00:00 -0000",
    "Wed, 22 Mar 2017 10:00:00 -0000",
    "Wed, 15 Mar 2017 10:00:00 -0000",
    "Wed, 08 Mar 2017 11:00:00 -0000",
    "Wed, 01 Mar 2017 11:00:00 -0000",
    "Wed, 22 Feb 2017 11:00:00 -0000",
    "Wed, 15 Feb 2017 11:00:00 -0000",
    "Wed, 08 Feb 2017 11:00:00 -0000",
    "Wed, 01 Feb 2017 11:00:00 -0000",
    "Wed, 25 Jan 2017 11:00:00 -0000",
    "Fri, 13 Jan 2017 18:38:00 -0000",
    "Wed, 20 Sep 2017 03:30:00 -0000",
    "Wed, 13 Sep 2017 03:15:00 -0000",
    "Wed, 06 Sep 2017 03:15:00 -0000",
    "Wed, 30 Aug 2017 03:15:00 -0000",
    "Wed, 23 Aug 2017 03:15:00 -0000",
    "Wed, 16 Aug 2017 03:15:00 -0000",
    "Wed, 09 Aug 2017 03:15:00 -0000",
    "Wed, 02 Aug 2017 03:00:00 -0000",
    "Tue, 11 Jul 2017 17:14:45 -0000",
    "Thu, 03 August 2017 06:00:00 -0400",
    "Thu, 27 July 2017 06:00:00 -0400",
    "Thu, 20 July 2017 06:00:00 -0400",
    "Thu, 13 July 2017 06:00:00 -0400",
    "Thu, 06 July 2017 06:00:00 -0400",
    "Thu, 28 June 2017 06:00:00 -0400",
    "Thu, 17 Jul 2013 06:00:03 -0400",
    "Thu, 02 Apr 2014 06:00:03 -0400",
    "Wed, 14 Jan 2016 06:00:03 -0400",
    "Thu, 22 June 2017 06:00:00 -0400",
    "Thu, 15 June 2017 06:00:00 -0400",
    "Thu, 7 June 2017 06:00:00 -0400",
    "Thu, 1 June 2017 06:00:00 -0400",
    "Thu, 23 Dec 2015 06:00:03 -0400",
    "Thu, 14 Feb 2014 06:00:03 -0400",
    "Thu, 04 Dec 2013 06:00:03 -0400",
    "Thu, 20 Dec 2016 06:00:00 -0400",
    "Thu, 23 Nov 2016 06:00:00 -0400",
    "Thu, 05 Aug 2016 06:00:00 -0400",
    "Fri, 09 Jun 2016 12:00:00 -0400",
    "Thu, 10 May 2017 06:00:00 -0400",
    "Thu, 22 Feb 2017 06:00:00 -0400",
    "Thu, 15 Feb 2017 06:00:00 -0400",
];

static MIXED_DATES: &[&str] = &[
    "Mon, 19 Jun 2017 19:45:01 EST",
    "Thu, 29 Jun 2017 17:30:00 PDT",
    "Thu, 6 July 2017 15:30:00 PDT",
    "Mon, 10 July 2017 16:00:00 PDT",
    "Mon, 17 July 2017 17:00:00 PDT",
    "Mon, 24 July 2017 16:00:00 PDT",
    "Mon, 31 July 2017 16:00:00 PDT",
    "Tue, 8 Aug 2017 16:00:00 PDT",
    "Tue, 15 Aug 2017 17:00:00 PDT",
    "Mon, 28 Aug 2017 15:00:00 PDT",
    "Thu, 30 Aug 2017 1:30:00 PDT",
    "Monday, 25 Sept 2017 5:30:00 PDT",
    "Mon, 25 Sep 2017 17:02:11 GMT",
    "Fri, 22 Sep 2017 11:00:45 GMT",
    "Mon, 18 Sep 2017 11:00:22 GMT",
    "Mon, 11 Sep 2017 11:00:04 GMT",
    "Fri, 08 Sep 2017 11:00:37 GMT",
    "Mon, 04 Sep 2017 11:00:11 GMT",
    "Fri, 01 Sep 2017 11:00:42 GMT",
    "Mon, 28 Aug 2017 13:00:07 GMT",
    "Fri, 25 Aug 2017 11:00:10 GMT",
    "Mon, 21 Aug 2017 11:00:14 GMT",
    "Fri, 18 Aug 2017 11:00:21 GMT",
    "Mon, 14 Aug 2017 12:12:09 GMT",
    "Fri, 11 Aug 2017 11:00:15 GMT",
    "Fri, 04 Aug 2017 11:00:41 GMT",
    "Mon, 31 Jul 2017 11:32:24 GMT",
    "Fri, 28 Jul 2017 11:00:20 GMT",
    "Mon, 24 Jul 2017 11:00:34 GMT",
    "Fri, 21 Jul 2017 11:00:09 GMT",
    "Fri, 14 Jul 2017 11:00:21 GMT",
    "Mon, 10 Jul 2017 11:39:57 GMT",
    "Fri, 07 Jul 2017 11:00:24 GMT",
    "Mon, 03 Jul 2017 13:20:46 GMT",
    "Fri, 30 Jun 2017 11:00:09 GMT",
    "Mon, 26 Jun 2017 11:03:45 GMT",
    "Fri, 23 Jun 2017 11:00:23 GMT",
    "Wed, 20 Sep 2017 10:00:00 -0000",
    "Wed, 13 Sep 2017 10:00:00 -0000",
    "Wed, 09 Aug 2017 10:00:00 -0000",
    "Wed, 02 Aug 2017 10:00:00 -0000",
    "Wed, 26 Jul 2017 10:00:00 -0000",
    "Wed, 19 Jul 2017 10:00:00 -0000",
    "Wed, 12 Jul 2017 10:00:00 -0000",
    "Mon, 19 Jun 2017 10:54:06 GMT",
    "Fri, 16 Jun 2017 11:25:24 GMT",
    "Mon, 12 Jun 2017 11:41:24 GMT",
    "Fri, 09 Jun 2017 11:00:00 GMT",
    "Mon, 05 Jun 2017 11:57:31 GMT",
    "Fri, 02 Jun 2017 11:00:22 GMT",
    "Wed, 31 May 2017 10:10:24 GMT",
    "Mon, 22 May 2017 16:56:43 GMT",
    "Fri, 19 May 2017 11:32:38 GMT",
    "Fri, 12 May 2017 12:09:26 GMT",
    "Wed, 10 May 2017 10:22:58 GMT",
    "Fri, 05 May 2017 16:40:19 GMT",
    "Wed, 28 Jun 2017 10:00:00 -0000",
    "Wed, 21 Jun 2017 10:00:00 -0000",
    "Wed, 14 Jun 2017 10:00:00 -0000",
    "Wed, 07 Jun 2017 10:00:00 -0000",
    "Wed, 31 May 2017 10:00:00 -0000",
    "Wed, 24 May 2017 10:00:00 -0000",
    "Wed, 17 May 2017 10:00:00 -0000",
    "Wed, 10 May 2017 10:00:00 -0000",
    "Mon, 18 Sep 2017 18:49:17 -0700",
    "Mon, 11 Sep 2017 19:36:21 -0700",
    "Thu, 31 Aug 2017 19:44:08 -0700",
    "Mon, 28 Aug 2017 17:18:34 -0700",
    "Thu, 17 Aug 2017 16:15:04 -0700",
    "Thu, 10 Aug 2017 17:29:36 -0700",
    "Thu, 03 Aug 2017 16:08:23 -0700",
    "Thu, 27 Jul 2017 14:41:30 -0700",
    "Thu, 20 Jul 2017 17:03:42 -0700",
    "Thu, 13 Jul 2017 09:52:34 -0700",
    "Thu, 06 Jul 2017 17:16:40 -0700",
    "Fri, 30 Jun 2017 09:56:37 -0700",
    "Thu, 22 Jun 2017 17:40:20 -0700",
    "Thu, 15 Jun 2017 15:28:56 -0700",
    "Thu, 08 Jun 2017 17:37:15 -0700",
    "Thu, 01 Jun 2017 19:53:26 -0700",
    "Thu, 25 May 2017 19:19:07 -0700",
    "Fri, 19 May 2017 15:40:20 -0700",
    "Mon, 08 May 2017 17:21:01 -0700",
    "Mon, 01 May 2017 16:09:40 -0700",
    "Mon, 24 Apr 2017 18:20:50 -0700",
    "Mon, 17 Apr 2017 16:07:15 -0700",
    "Fri, 14 Apr 2017 13:21:24 -0700",
    "Tue, 11 Apr 2017 21:40:52 -0700",
    "Mon, 27 Mar 2017 15:29:31 -0700",
    "Mon, 20 Mar 2017 16:12:10 -0700",
    "Mon, 13 Mar 2017 16:51:08 -0700",
    "Wed, 03 May 2017 10:00:00 -0000",
    "Wed, 19 Apr 2017 10:00:00 -0000",
    "Wed, 12 Apr 2017 10:00:00 -0000",
    "Wed, 05 Apr 2017 10:00:00 -0000",
    "Wed, 29 Mar 2017 10:00:00 -0000",
    "Wed, 22 Mar 2017 10:00:00 -0000",
    "Wed, 15 Mar 2017 10:00:00 -0000",
    "Wed, 08 Mar 2017 11:00:00 -0000",
    "Wed, 01 Mar 2017 11:00:00 -0000",
    "Wed, 22 Feb 2017 11:00:00 -0000",
    "Wed, 15 Feb 2017 11:00:00 -0000",
    "Thu, 21 Sep 2017 20:20:18 +0000",
    "Thu, 07 Sep 2017 08:55:05 +0000",
    "Thu, 24 Aug 2017 09:11:27 +0000",
    "Thu, 10 Aug 2017 06:58:17 +0000",
    "Thu, 27 Jul 2017 17:01:03 +0000",
    "Thu, 13 Jul 2017 16:23:28 +0000",
    "Thu, 29 Jun 2017 08:39:06 +0000",
    "Thu, 15 Jun 2017 14:13:25 +0000",
    "Thu, 18 May 2017 07:48:14 +0000",
    "Thu, 04 May 2017 08:39:11 +0000",
    "Thu, 20 Apr 2017 08:42:57 +0000",
    "Thu, 06 Apr 2017 08:22:36 +0000",
    "Wed, 08 Feb 2017 11:00:00 -0000",
    "Wed, 01 Feb 2017 11:00:00 -0000",
    "Wed, 25 Jan 2017 11:00:00 -0000",
    "Fri, 13 Jan 2017 18:38:00 -0000",
    "Mon, 06 Mar 2017 16:25:12 -0800",
    "Mon, 27 Feb 2017 16:17:50 -0800",
    "Mon, 20 Feb 2017 08:23:34 -0800",
    "Mon, 13 Feb 2017 18:35:04 -0800",
    "Mon, 06 Feb 2017 05:43:02 -0800",
    "Thu, 02 Feb 2017 14:01:48 -0800",
    "Mon, 23 Jan 2017 15:25:56 -0800",
    "Mon, 16 Jan 2017 16:27:59 -0800",
    "Mon, 09 Jan 2017 16:31:44 -0800",
    "Thu, 05 Jan 2017 14:04:34 -0800",
    "Mon, 26 Dec 2016 10:40:13 -0800",
    "Mon, 19 Dec 2016 14:58:25 -0800",
    "Mon, 01 May 2017 15:37:24 GMT",
    "Fri, 28 Apr 2017 10:41:41 GMT",
    "Mon, 24 Apr 2017 14:50:28 GMT",
    "Fri, 21 Apr 2017 15:29:50 GMT",
    "Mon, 17 Apr 2017 10:51:45 GMT",
    "Mon, 10 Apr 2017 11:16:05 GMT",
    "Fri, 07 Apr 2017 11:31:50 GMT",
    "Mon, 03 Apr 2017 15:07:17 GMT",
    "Fri, 31 Mar 2017 10:42:48 GMT",
    "Mon, 27 Mar 2017 11:33:01 GMT",
    "Fri, 24 Mar 2017 11:49:51 GMT",
    "Mon, 20 Mar 2017 15:49:59 GMT",
    "Tue, 14 Mar 2017 13:34:27 GMT",
    "Thu, 23 Mar 2017 11:06:16 +0000",
    "Mon, 20 Mar 2017 10:12:21 +0000",
    "Mon, 12 Dec 2016 16:06:40 -0800",
    "Mon, 05 Dec 2016 16:28:57 -0800",
    "Mon, 28 Nov 2016 15:54:37 -0800",
    "Mon, 21 Nov 2016 16:30:24 -0800",
    "Mon, 14 Nov 2016 14:46:47 -0800",
    "Mon, 07 Nov 2016 16:38:01 -0800",
    "Mon, 31 Oct 2016 16:25:35 -0700",
    "Mon, 24 Oct 2016 16:48:25 -0700",
    "Mon, 17 Oct 2016 16:05:25 -0700",
    "Mon, 10 Oct 2016 14:43:16 -0700",
    "Mon, 03 Oct 2016 16:20:52 -0700",
    "Mon, 26 Sep 2016 15:33:57 -0700",
    "Mon, 19 Sep 2016 16:48:03 -0700",
    "Mon, 12 Sep 2016 16:05:48 -0700",
    "Mon, 05 Sep 2016 09:52:49 -0700",
    "Mon, 29 Aug 2016 15:54:57 -0700",
    "Mon, 22 Aug 2016 15:37:40 -0700",
    "Mon, 15 Aug 2016 17:43:04 -0700",
    "Mon, 08 Aug 2016 17:26:08 -0700",
    "Mon, 01 Aug 2016 15:25:37 -0700",
    "Mon, 25 Jul 2016 15:30:39 -0700",
    "Mon, 18 Jul 2016 18:50:19 -0700",
    "Mon, 11 Jul 2016 17:26:57 -0700",
    "Mon, 04 Jul 2016 19:03:06 -0700",
    "Mon, 27 Jun 2016 16:13:00 -0700",
    "Tue, 21 Jun 2016 14:37:13 -0700",
    "Mon, 13 Jun 2016 17:56:06 -0700",
    "Mon, 06 Jun 2016 17:12:37 -0700",
    "Mon, 30 May 2016 17:28:40 -0700",
    "Mon, 23 May 2016 16:24:44 -0700",
    "Mon, 16 May 2016 15:21:08 -0700",
    "Mon, 09 May 2016 15:55:36 -0700",
    "Mon, 02 May 2016 15:31:30 -0700",
    "Mon, 25 Apr 2016 18:18:06 -0700",
    "Mon, 18 Apr 2016 10:21:15 -0700",
    "Mon, 11 Apr 2016 09:25:56 -0700",
    "Mon, 04 Apr 2016 14:45:44 -0700",
    "Mon, 28 Mar 2016 16:06:41 -0700",
    "Mon, 21 Mar 2016 14:54:34 -0700",
    "Mon, 14 Mar 2016 15:51:23 -0700",
    "Mon, 07 Mar 2016 16:39:02 -0800",
    "Tue, 01 Mar 2016 04:43:08 -0800",
    "Mon, 22 Feb 2016 15:09:53 -0800",
    "Mon, 15 Feb 2016 15:33:44 -0800",
    "Mon, 08 Feb 2016 14:31:35 -0800",
    "Mon, 01 Feb 2016 15:10:14 -0800",
    "Mon, 25 Jan 2016 16:11:58 -0800",
    "Mon, 18 Jan 2016 17:49:22 -0800",
    "Mon, 11 Jan 2016 15:33:30 -0800",
    "Mon, 04 Jan 2016 15:25:27 -0800",
    "Mon, 28 Dec 2015 15:59:46 -0800",
    "Mon, 21 Dec 2015 10:31:50 -0800",
    "Mon, 14 Dec 2015 16:33:53 -0800",
    "Mon, 07 Dec 2015 17:06:41 -0800",
    "Mon, 30 Nov 2015 15:16:42 -0800",
    "Mon, 23 Nov 2015 15:12:07 -0800",
    "Mon, 16 Nov 2015 15:08:52 -0800",
    "Mon, 09 Nov 2015 15:01:16 -0800",
    "Mon, 02 Nov 2015 16:11:21 -0800",
    "Mon, 26 Oct 2015 15:27:22 -0700",
    "Mon, 19 Oct 2015 14:46:16 -0700",
    "Mon, 12 Oct 2015 16:12:35 -0700",
    "Mon, 05 Oct 2015 20:29:22 -0700",
    "Mon, 28 Sep 2015 22:03:18 -0700",
    "Fri, 18 Sep 2015 19:08:17 -0700",
    "Mon, 14 Sep 2015 16:35:31 -0700",
    "Fri, 04 Sep 2015 11:00:52 -0700",
    "Mon, 24 Aug 2015 14:53:33 -0700",
    "Fri, 14 Aug 2015 12:12:47 -0700",
    "Mon, 10 Aug 2015 15:21:00 -0700",
    "Mon, 03 Aug 2015 15:34:42 -0700",
    "Mon, 27 Jul 2015 15:24:30 -0700",
    "Mon, 20 Jul 2015 17:25:52 -0700",
    "Mon, 13 Jul 2015 14:24:51 -0700",
    "Mon, 06 Jul 2015 16:25:32 -0700",
    "Mon, 29 Jun 2015 16:52:35 -0700",
    "Mon, 22 Jun 2015 15:49:23 -0700",
    "Mon, 15 Jun 2015 15:38:18 -0700",
    "Mon, 08 Jun 2015 16:21:13 -0700",
    "Mon, 01 Jun 2015 17:50:05 -0700",
    "Mon, 25 May 2015 15:23:10 -0700",
    "Fri, 22 May 2015 16:36:21 -0700",
    "Mon, 11 May 2015 19:07:50 -0700",
    "Mon, 04 May 2015 14:46:39 -0700",
    "Mon, 27 Apr 2015 21:12:27 -0700",
    "Mon, 20 Apr 2015 15:27:44 -0700",
    "Mon, 13 Apr 2015 15:33:09 -0700",
    "Mon, 06 Apr 2015 15:54:14 -0700",
    "Mon, 30 Mar 2015 15:54:48 -0700",
    "Mon, 23 Mar 2015 15:07:44 -0700",
    "Mon, 16 Mar 2015 15:39:34 -0700",
    "Mon, 09 Mar 2015 15:27:37 -0700",
    "Mon, 02 Mar 2015 14:13:45 -0800",
    "Mon, 23 Feb 2015 15:19:09 -0800",
    "Mon, 16 Feb 2015 16:47:11 -0800",
    "Thu, 23 Feb 2017 10:13:52 +0000",
    "Thu, 09 Feb 2017 11:15:01 +0000",
    "Thu, 26 Jan 2017 09:43:22 +0000",
    "Wed, 11 Jan 2017 19:12:41 +0000",
    "Thu, 22 Dec 2016 21:44:45 +0000",
    "Tue, 13 Dec 2016 09:58:14 +0000",
    "Thu, 01 Dec 2016 08:56:23 +0000",
    "Thu, 17 Nov 2016 15:55:00 +0000",
    "Thu, 03 Nov 2016 11:10:06 +0000",
    "Thu, 20 Oct 2016 09:52:43 +0000",
    "Sat, 15 Oct 2016 08:23:41 +0000",
    "Mon, 09 Feb 2015 15:21:22 -0800",
    "Mon, 02 Feb 2015 15:57:03 -0800",
    "Mon, 26 Jan 2015 16:00:23 -0800",
    "Mon, 19 Jan 2015 15:20:52 -0800",
    "Mon, 12 Jan 2015 19:42:48 -0800",
    "Mon, 05 Jan 2015 17:06:14 -0800",
    "Mon, 29 Dec 2014 14:25:56 -0800",
    "Mon, 22 Dec 2014 12:40:51 -0800",
    "Mon, 15 Dec 2014 19:26:56 -0800",
    "Mon, 08 Dec 2014 15:22:47 -0800",
    "Mon, 01 Dec 2014 16:31:32 -0800",
    "Mon, 24 Nov 2014 16:41:07 -0800",
    "Mon, 17 Nov 2014 17:05:41 -0800",
    "Mon, 10 Nov 2014 15:59:14 -0800",
    "Mon, 03 Nov 2014 15:33:48 -0800",
    "Mon, 27 Oct 2014 21:24:20 -0700",
    "Mon, 20 Oct 2014 14:47:24 -0700",
    "Mon, 13 Oct 2014 15:16:03 -0700",
    "Mon, 06 Oct 2014 16:41:48 -0700",
    "Mon, 29 Sep 2014 15:28:39 -0700",
    "Mon, 22 Sep 2014 16:00:28 -0700",
    "Mon, 15 Sep 2014 14:43:59 -0700",
    "Mon, 08 Sep 2014 15:31:12 -0700",
    "Tue, 02 Sep 2014 16:06:41 -0700",
    "Mon, 25 Aug 2014 15:25:00 -0700",
    "Mon, 18 Aug 2014 14:14:15 -0700",
    "Mon, 11 Aug 2014 15:17:06 -0700",
    "Fri, 08 Aug 2014 12:20:46 -0700",
    "Mon, 04 Aug 2014 15:21:41 -0700",
    "Mon, 21 Jul 2014 15:40:11 -0700",
    "Mon, 14 Jul 2014 14:42:37 -0700",
    "Fri, 22 Sep 2017 12:00:00 +0000",
    "Fri, 08 Sep 2017 12:00:00 +0000",
    "Fri, 25 Aug 2017 12:00:00 +0000",
    "Fri, 11 Aug 2017 12:00:00 +0000",
    "Fri, 28 Jul 2017 12:00:00 +0000",
    "Thu, 22 Sep 2016 09:45:33 +0000",
    "Thu, 08 Sep 2016 13:58:40 +0000",
    "Thu, 11 Aug 2016 14:18:27 +0000",
    "Thu, 28 Jul 2016 09:18:26 +0000",
    "Thu, 14 Jul 2016 09:57:57 +0000",
    "Thu, 30 Jun 2016 15:44:24 +0000",
    "Thu, 16 Jun 2016 13:57:42 +0000",
    "Thu, 02 Jun 2016 17:20:15 +0000",
    "Thu, 26 May 2016 15:45:19 +0000",
    "Thu, 05 May 2016 08:29:23 +0000",
    "Thu, 21 Apr 2016 08:09:05 +0000",
    "Thu, 07 Apr 2016 09:31:16 +0000",
    "Thu, 24 Mar 2016 09:58:18 +0000",
    "Thu, 10 Mar 2016 09:48:46 +0000",
    "Thu, 25 Feb 2016 08:48:21 +0000",
    "Fri, 12 Feb 2016 10:08:23 +0000",
    "Mon, 01 Feb 2016 10:57:10 +0000",
    "Thu, 14 Jan 2016 09:47:12 +0000",
    "Thu, 31 Dec 2015 08:12:56 +0000",
    "Thu, 17 Dec 2015 09:46:36 +0000",
    "Fri, 04 Dec 2015 10:44:20 +0000",
    "Fri, 13 Nov 2015 09:26:07 +0000",
    "Thu, 29 Oct 2015 09:29:45 +0000",
    "Thu, 15 Oct 2015 10:49:31 +0000",
    "Thu, 08 Oct 2015 12:06:12 +0000",
    "Thu, 17 Sep 2015 13:32:24 +0000",
    "Thu, 03 Sep 2015 10:52:03 +0000",
    "Fri, 21 Aug 2015 11:12:31 +0000",
    "Tue, 28 Jul 2015 16:09:22 +0000",
    "Thu, 09 Jul 2015 08:58:05 +0000",
    "Thu, 25 Jun 2015 12:19:27 +0000",
    "Fri, 12 Jun 2015 08:57:59 +0000",
    "Thu, 28 May 2015 09:28:52 +0000",
    "Thu, 14 May 2015 08:16:26 +0000",
    "Thu, 30 Apr 2015 08:26:25 +0000",
    "Thu, 16 Apr 2015 09:39:17 +0000",
    "Thu, 02 Apr 2015 09:22:57 +0000",
    "Thu, 19 Mar 2015 10:31:57 +0000",
    "Fri, 06 Mar 2015 14:35:10 +0000",
    "Thu, 26 Feb 2015 21:22:54 +0000",
    "Thu, 05 Feb 2015 16:10:35 +0000",
    "Thu, 22 Jan 2015 11:20:52 +0000",
    "Thu, 08 Jan 2015 10:50:18 +0000",
    "Tue, 23 Dec 2014 10:21:30 +0000",
    "Thu, 11 Dec 2014 13:24:27 +0000",
    "Thu, 27 Nov 2014 13:33:10 +0000",
    "Thu, 13 Nov 2014 12:46:47 +0000",
    "Thu, 30 Oct 2014 13:11:37 +0000",
    "Sat, 18 Oct 2014 07:49:06 +0000",
    "Thu, 02 Oct 2014 10:05:59 +0000",
    "Fri, 10 Mar 2017 12:32:47 GMT",
    "Mon, 06 Mar 2017 11:00:11 GMT",
    "Fri, 03 Mar 2017 11:35:11 GMT",
    "Mon, 27 Feb 2017 10:21:21 GMT",
    "Fri, 24 Feb 2017 11:37:01 GMT",
    "Tue, 21 Feb 2017 10:44:51 GMT",
    "Fri, 17 Feb 2017 16:44:46 GMT",
    "Mon, 13 Feb 2017 11:00:12 GMT",
    "Fri, 10 Feb 2017 11:00:50 GMT",
    "Mon, 06 Feb 2017 11:58:59 GMT",
    "Fri, 03 Feb 2017 11:00:13 GMT",
    "Mon, 30 Jan 2017 11:27:02 GMT",
    "Fri, 19 Sep 2014 05:09:56 +0000",
    "Thu, 04 Sep 2014 09:51:28 +0000",
    "Thu, 21 Aug 2014 10:04:08 +0000",
    "Thu, 07 Aug 2014 10:02:34 +0000",
    "Fri, 25 Jul 2014 09:03:17 +0000",
    "Thu, 10 Jul 2014 14:37:09 +0000",
    "Thu, 26 Jun 2014 13:07:09 +0000",
    "Thu, 12 Jun 2014 16:16:30 +0000",
    "Thu, 29 May 2014 12:47:18 +0000",
    "Thu, 22 May 2014 14:55:55 +0000",
    "Thu, 08 May 2014 11:38:49 +0000",
    "Thu, 24 Apr 2014 13:29:11 +0000",
    "Thu, 10 Apr 2014 11:40:30 +0000",
    "Fri, 28 Mar 2014 14:46:57 +0000",
    "Thu, 13 Mar 2014 00:29:45 +0000",
    "Thu, 27 Feb 2014 14:02:45 +0000",
    "Thu, 13 Feb 2014 16:16:06 +0000",
    "Mon, 07 Jul 2014 14:03:14 -0700",
    "Mon, 30 Jun 2014 14:17:33 -0700",
    "Mon, 23 Jun 2014 14:51:58 -0700",
    "Mon, 16 Jun 2014 15:05:57 -0700",
    "Mon, 09 Jun 2014 15:13:06 -0700",
    "Mon, 02 Jun 2014 14:53:51 -0700",
    "Mon, 26 May 2014 15:36:19 -0700",
    "Mon, 19 May 2014 18:02:56 -0700",
    "Mon, 12 May 2014 17:31:06 -0700",
    "Mon, 05 May 2014 19:30:05 -0700",
    "Mon, 28 Apr 2014 16:52:09 -0700",
    "Mon, 21 Apr 2014 10:53:10 -0700",
    "Mon, 14 Apr 2014 13:25:20 -0700",
    "Mon, 07 Apr 2014 12:39:22 -0700",
    "Mon, 31 Mar 2014 11:40:54 -0700",
    "Mon, 24 Mar 2014 11:47:49 -0700",
    "Mon, 17 Mar 2014 11:39:02 -0700",
    "Mon, 10 Mar 2014 11:40:53 -0700",
    "Mon, 03 Mar 2014 11:28:04 -0800",
    "Mon, 24 Feb 2014 11:31:19 -0800",
    "Mon, 17 Feb 2014 12:35:31 -0800",
    "Mon, 10 Feb 2014 12:14:40 -0800",
    "Mon, 03 Feb 2014 12:13:39 -0800",
    "Mon, 27 Jan 2014 12:08:09 -0800",
    "Mon, 20 Jan 2014 12:02:01 -0800",
    "Mon, 13 Jan 2014 12:02:34 -0800",
    "Mon, 06 Jan 2014 12:02:04 -0800",
    "Mon, 30 Dec 2013 11:55:13 -0800",
    "Mon, 23 Dec 2013 06:55:16 -0800",
    "Mon, 16 Dec 2013 14:35:21 -0800",
    "Mon, 09 Dec 2013 11:53:07 -0800",
    "Mon, 02 Dec 2013 14:01:30 -0800",
    "Mon, 25 Nov 2013 12:37:28 -0800",
    "Mon, 18 Nov 2013 12:46:59 -0800",
    "Mon, 11 Nov 2013 13:03:36 -0800",
    "Mon, 04 Nov 2013 12:02:45 -0800",
    "Mon, 28 Oct 2013 12:26:27 -0700",
    "Mon, 21 Oct 2013 12:42:56 -0700",
    "Mon, 14 Oct 2013 12:31:47 -0700",
    "Mon, 07 Oct 2013 12:45:05 -0700",
    "Mon, 30 Sep 2013 11:57:40 -0700",
    "Mon, 23 Sep 2013 12:53:25 -0700",
    "Mon, 16 Sep 2013 13:36:22 -0700",
    "Mon, 09 Sep 2013 12:29:11 -0700",
    "Mon, 02 Sep 2013 12:11:00 -0700",
    "Mon, 26 Aug 2013 12:25:39 -0700",
    "Mon, 19 Aug 2013 18:51:14 -0700",
    "Mon, 12 Aug 2013 12:04:49 -0700",
    "Mon, 05 Aug 2013 13:42:44 -0700",
    "Mon, 29 Jul 2013 12:53:50 -0700",
    "Mon, 22 Jul 2013 13:20:10 -0700",
    "Mon, 15 Jul 2013 12:29:24 -0700",
    "Mon, 08 Jul 2013 12:30:37 -0700",
    "Mon, 01 Jul 2013 12:22:30 -0700",
    "Mon, 24 Jun 2013 11:36:33 -0700",
    "Fri, 27 Jan 2017 11:00:24 GMT",
    "Mon, 23 Jan 2017 11:05:17 GMT",
    "Fri, 20 Jan 2017 11:00:04 GMT",
    "Mon, 16 Jan 2017 11:25:49 GMT",
    "Fri, 13 Jan 2017 11:00:16 GMT",
    "Fri, 06 Jan 2017 11:00:16 GMT",
    "Fri, 30 Dec 2016 11:00:31 GMT",
    "Fri, 23 Dec 2016 11:00:14 GMT",
    "Mon, 19 Dec 2016 11:14:32 GMT",
    "Fri, 16 Dec 2016 11:23:08 GMT",
    "Tue, 13 Dec 2016 13:03:30 GMT",
    "Fri, 09 Dec 2016 11:34:02 GMT",
    "Mon, 05 Dec 2016 11:34:46 GMT",
    "Thu, 30 Jan 2014 12:46:32 +0000",
    "Thu, 16 Jan 2014 14:16:28 +0000",
    "Sat, 04 Jan 2014 01:55:12 +0000",
    "Thu, 19 Dec 2013 14:42:42 +0000",
    "Thu, 05 Dec 2013 14:00:18 +0000",
    "Thu, 21 Nov 2013 15:00:32 +0000",
    "Thu, 07 Nov 2013 14:30:56 +0000",
    "Thu, 24 Oct 2013 13:00:04 +0000",
    "Fri, 02 Dec 2016 12:30:33 GMT",
    "Tue, 29 Nov 2016 16:34:51 GMT",
    "Fri, 25 Nov 2016 12:33:57 GMT",
    "Mon, 21 Nov 2016 11:49:41 GMT",
    "Fri, 18 Nov 2016 11:07:38 GMT",
    "Mon, 14 Nov 2016 11:08:55 GMT",
    "Fri, 11 Nov 2016 11:21:28 GMT",
    "Fri, 04 Nov 2016 10:59:04 GMT",
    "Mon, 31 Oct 2016 16:11:31 GMT",
    "Fri, 28 Oct 2016 12:00:54 GMT",
    "Mon, 24 Oct 2016 11:41:44 GMT",
    "Fri, 21 Oct 2016 10:10:27 GMT",
    "Fri, 14 Jul 2017 12:00:00 +0000",
    "Fri, 30 Jun 2017 12:00:00 +0000",
    "Mon, 17 Oct 2016 10:01:38 GMT",
    "Fri, 14 Oct 2016 09:55:14 GMT",
    "Mon, 03 Oct 2016 10:06:24 GMT",
    "Fri, 30 Sep 2016 09:50:25 GMT",
    "Mon, 26 Sep 2016 13:38:25 GMT",
    "Wed, 20 Sep 2017 03:30:00 -0000",
    "Fri, 16 Jun 2017 12:00:00 +0000",
    "Fri, 09 Jun 2017 12:00:00 +0000",
    "Sat, 11 Mar 2017 00:54:00 +0000",
    "Fri, 23 Sep 2016 10:08:34 GMT",
    "Mon, 19 Sep 2016 10:18:45 GMT",
    "Fri, 16 Sep 2016 09:59:25 GMT",
    "Mon, 12 Sep 2016 12:53:04 GMT",
    "Fri, 09 Sep 2016 10:57:49 GMT",
    "Fri, 02 Sep 2016 10:59:13 GMT",
    "Mon, 15 Aug 2016 13:08:12 GMT",
    "Fri, 05 Aug 2016 12:18:15 GMT",
    "Wed, 13 Sep 2017 03:15:00 -0000",
    "Fri, 24 Feb 2017 14:00:00 +0000",
    "Fri, 17 Feb 2017 17:40:57 +0000",
    "Fri, 10 Feb 2017 17:25:36 +0000",
    "Fri, 03 Feb 2017 14:00:40 +0000",
    "Fri, 27 Jan 2017 17:03:24 +0000",
    "Fri, 20 Jan 2017 14:00:14 +0000",
    "Fri, 13 Jan 2017 14:00:24 +0000",
    "Wed, 06 Sep 2017 03:15:00 -0000",
    "Fri, 06 Jan 2017 22:22:00 +0000",
    "Fri, 06 Jan 2017 22:21:18 +0000",
    "Fri, 23 Dec 2016 21:57:02 +0000",
    "Tue, 19 Sep 2017 20:12:14 -0700",
    "Tue, 12 Sep 2017 22:24:42 -0700",
    "Tue, 05 Sep 2017 20:57:27 -0700",
    "Tue, 29 Aug 2017 21:15:30 -0700",
    "Tue, 22 Aug 2017 19:45:55 -0700",
    "Tue, 15 Aug 2017 23:03:23 -0700",
    "Tue, 08 Aug 2017 22:29:26 -0700",
    "Tue, 01 Aug 2017 22:03:48 -0700",
    "Tue, 25 Jul 2017 22:26:36 -0700",
    "Tue, 18 Jul 2017 23:42:59 -0700",
    "Tue, 11 Jul 2017 22:13:14 -0700",
    "Tue, 04 Jul 2017 23:48:01 -0700",
    "Tue, 27 Jun 2017 21:16:27 -0700",
    "Tue, 20 Jun 2017 19:45:54 -0700",
    "Tue, 13 Jun 2017 19:07:28 -0700",
    "Tue, 06 Jun 2017 21:29:20 -0700",
    "Tue, 30 May 2017 20:15:26 -0700",
    "Tue, 23 May 2017 19:11:12 -0700",
    "Tue, 16 May 2017 21:51:30 -0700",
    "Tue, 09 May 2017 20:14:54 -0700",
    "Wed, 03 May 2017 02:31:25 -0700",
    "Tue, 25 Apr 2017 21:50:51 -0700",
    "Tue, 18 Apr 2017 19:37:04 -0700",
    "Wed, 12 Apr 2017 00:32:24 -0700",
    "Tue, 04 Apr 2017 23:15:24 -0700",
    "Tue, 28 Mar 2017 21:00:37 -0700",
    "Tue, 21 Mar 2017 20:08:25 -0700",
    "Tue, 14 Mar 2017 19:09:35 -0700",
    "Tue, 07 Mar 2017 20:12:26 -0800",
    "Tue, 28 Feb 2017 19:57:29 -0800",
    "Tue, 21 Feb 2017 20:55:45 -0800",
    "Tue, 14 Feb 2017 19:51:59 -0800",
    "Tue, 07 Feb 2017 20:55:05 -0800",
    "Tue, 31 Jan 2017 20:17:31 -0800",
    "Tue, 24 Jan 2017 20:30:30 -0800",
    "Tue, 17 Jan 2017 20:02:51 -0800",
    "Tue, 10 Jan 2017 19:59:01 -0800",
    "Wed, 30 Aug 2017 03:15:00 -0000",
    "Sun, 24 Sep 2017 15:00:00 -0700",
    "Sat, 16 Sep 2017 13:00:00 -0700",
    "Sun, 10 Sep 2017 07:00:00 -0700",
    "Sun, 03 Sep 2017 09:15:00 -0700",
    "Sat, 26 Aug 2017 17:00:00 -0700",
    "Sun, 17 Sep 2017 22:00:00 -0700",
    "Wed, 23 Aug 2017 03:15:00 -0000",
    "Tue, 03 Jan 2017 23:06:55 -0800",
    "Tue, 27 Dec 2016 14:57:40 -0800",
    "Tue, 20 Dec 2016 23:40:41 -0800",
    "Tue, 13 Dec 2016 21:29:23 -0800",
    "Tue, 06 Dec 2016 22:55:19 -0800",
    "Wed, 30 Nov 2016 01:46:47 -0800",
    "Tue, 22 Nov 2016 21:43:19 -0800",
    "Tue, 15 Nov 2016 21:05:57 -0800",
    "Wed, 09 Nov 2016 00:06:52 -0800",
    "Tue, 01 Nov 2016 20:25:14 -0700",
    "Tue, 25 Oct 2016 21:34:12 -0700",
    "Tue, 18 Oct 2016 21:15:32 -0700",
    "Tue, 11 Oct 2016 20:12:52 -0700",
    "Tue, 04 Oct 2016 19:41:57 -0700",
    "Wed, 28 Sep 2016 03:14:12 -0700",
    "Tue, 20 Sep 2016 20:22:23 -0700",
    "Tue, 13 Sep 2016 22:56:50 -0700",
    "Tue, 06 Sep 2016 19:01:57 -0700",
    "Tue, 30 Aug 2016 22:14:17 -0700",
    "Tue, 23 Aug 2016 23:15:22 -0700",
    "Tue, 16 Aug 2016 20:48:44 -0700",
    "Tue, 09 Aug 2016 20:21:08 -0700",
    "Tue, 02 Aug 2016 21:27:15 -0700",
    "Tue, 26 Jul 2016 21:02:05 -0700",
    "Tue, 19 Jul 2016 20:21:03 -0700",
    "Tue, 12 Jul 2016 21:38:18 -0700",
    "Tue, 05 Jul 2016 20:43:56 -0700",
    "Tue, 28 Jun 2016 20:28:47 -0700",
    "Tue, 21 Jun 2016 20:19:16 -0700",
    "Tue, 14 Jun 2016 19:42:23 -0700",
    "Tue, 07 Jun 2016 19:18:45 -0700",
    "Tue, 31 May 2016 22:53:11 -0700",
    "Tue, 24 May 2016 20:13:30 -0700",
    "Tue, 17 May 2016 20:15:37 -0700",
    "Tue, 10 May 2016 19:43:38 -0700",
    "Tue, 03 May 2016 20:05:59 -0700",
    "Tue, 26 Apr 2016 21:18:17 -0700",
    "Tue, 19 Apr 2016 21:54:22 -0700",
    "Tue, 12 Apr 2016 19:43:05 -0700",
    "Tue, 05 Apr 2016 19:47:03 -0700",
    "Tue, 29 Mar 2016 19:43:17 -0700",
    "Tue, 22 Mar 2016 20:14:36 -0700",
    "Tue, 15 Mar 2016 20:00:57 -0700",
    "Tue, 08 Mar 2016 18:55:11 -0800",
    "Tue, 01 Mar 2016 19:59:49 -0800",
    "Tue, 23 Feb 2016 18:25:25 -0800",
    "Tue, 16 Feb 2016 19:03:31 -0800",
    "Tue, 09 Feb 2016 19:47:13 -0800",
    "Tue, 02 Feb 2016 19:10:37 -0800",
    "Tue, 26 Jan 2016 19:37:32 -0800",
    "Tue, 19 Jan 2016 21:22:11 -0800",
    "Tue, 12 Jan 2016 19:12:37 -0800",
    "Tue, 05 Jan 2016 19:38:22 -0800",
    "Tue, 29 Dec 2015 18:46:53 -0800",
    "Tue, 22 Dec 2015 10:02:15 -0800",
    "Tue, 15 Dec 2015 20:44:53 -0800",
    "Tue, 08 Dec 2015 20:30:41 -0800",
    "Tue, 01 Dec 2015 19:52:27 -0800",
    "Tue, 24 Nov 2015 20:27:05 -0800",
    "Wed, 18 Nov 2015 08:54:59 -0800",
    "Tue, 10 Nov 2015 20:02:04 -0800",
    "Tue, 03 Nov 2015 18:55:31 -0800",
    "Tue, 27 Oct 2015 18:44:51 -0700",
    "Tue, 20 Oct 2015 19:27:29 -0700",
    "Tue, 13 Oct 2015 21:04:14 -0700",
    "Tue, 06 Oct 2015 18:37:12 -0700",
    "Wed, 30 Sep 2015 10:27:19 -0700",
    "Wed, 23 Sep 2015 11:38:06 -0700",
    "Wed, 16 Sep 2015 18:38:28 -0700",
    "Tue, 08 Sep 2015 18:03:47 -0700",
    "Sat, 19 Aug 2017 19:00:00 -0700",
    "Sat, 12 Aug 2017 14:00:00 -0700",
    "Sun, 06 Aug 2017 09:00:00 -0700",
    "Sat, 29 Jul 2017 12:00:00 -0700",
    "Sat, 22 Jul 2017 13:00:00 -0700",
    "Wed, 16 Aug 2017 03:15:00 -0000",
    "Wed, 09 Aug 2017 03:15:00 -0000",
    "Wed, 02 Aug 2017 03:00:00 -0000",
    "Tue, 11 Jul 2017 17:14:45 -0000",
    "Tue, 01 Sep 2015 17:30:03 -0700",
    "Tue, 25 Aug 2015 19:42:06 -0700",
    "Wed, 19 Aug 2015 07:52:39 -0700",
    "Tue, 11 Aug 2015 19:21:22 -0700",
    "Wed, 05 Aug 2015 20:27:11 -0700",
    "Tue, 28 Jul 2015 18:55:23 -0700",
    "Mon, 20 Jul 2015 20:31:30 -0700",
    "Tue, 14 Jul 2015 19:01:12 -0700",
    "Tue, 07 Jul 2015 17:24:13 -0700",
    "Tue, 30 Jun 2015 19:06:38 -0700",
    "Tue, 23 Jun 2015 19:35:18 -0700",
    "Tue, 16 Jun 2015 21:58:08 -0700",
    "Tue, 09 Jun 2015 18:36:58 -0700",
    "Mon, 01 Jun 2015 17:08:57 -0700",
    "Tue, 26 May 2015 19:09:19 -0700",
    "Tue, 19 May 2015 18:32:23 -0700",
    "Tue, 12 May 2015 20:26:15 -0700",
    "Tue, 05 May 2015 18:30:20 -0700",
    "Tue, 28 Apr 2015 20:21:08 -0700",
    "Tue, 21 Apr 2015 19:30:21 -0700",
    "Tue, 14 Apr 2015 20:09:25 -0700",
    "Tue, 07 Apr 2015 19:01:31 -0700",
    "Tue, 31 Mar 2015 18:15:38 -0700",
    "Tue, 24 Mar 2015 18:29:47 -0700",
    "Tue, 17 Mar 2015 18:40:21 -0700",
    "Tue, 10 Mar 2015 19:47:18 -0700",
    "Tue, 03 Mar 2015 19:44:12 -0800",
    "Sun, 11 Jun 2017 10:00:00 -0700",
    "Sun, 04 Jun 2017 09:00:00 -0700",
    "Sun, 28 May 2017 10:00:00 -0700",
    "Sun, 21 May 2017 10:00:00 -0700",
    "Sun, 03 Sep 2017 23:00:00 -0700",
    "Sat, 15 Jul 2017 08:00:00 -0700",
    "Sun, 09 Jul 2017 10:00:00 -0700",
    "Sun, 02 Jul 2017 10:00:00 -0700",
    "Sun, 25 Jun 2017 10:00:00 -0700",
    "Sun, 18 Jun 2017 10:00:00 -0700",
    "Sun, 14 May 2017 10:00:00 -0700",
    "Sun, 20 Aug 2017 22:00:00 -0700",
    "Sun, 06 Aug 2017 23:55:00 -0700",
    "Fri, 05 May 2017 16:00:00 -0700",
    "Tue, 24 Feb 2015 07:03:35 -0800",
    "Tue, 17 Feb 2015 18:15:54 -0800",
    "Tue, 10 Feb 2015 18:00:45 -0800",
    "Tue, 03 Feb 2015 18:37:55 -0800",
    "Tue, 27 Jan 2015 17:43:33 -0800",
    "Tue, 20 Jan 2015 19:14:46 -0800",
    "Tue, 13 Jan 2015 20:40:57 -0800",
    "Tue, 06 Jan 2015 20:37:54 -0800",
    "Tue, 30 Dec 2014 19:10:27 -0800",
    "Tue, 23 Dec 2014 11:59:18 -0800",
    "Tue, 16 Dec 2014 20:01:41 -0800",
    "Tue, 09 Dec 2014 22:23:20 -0800",
    "Tue, 02 Dec 2014 19:47:38 -0800",
    "Tue, 25 Nov 2014 17:37:56 -0800",
    "Tue, 18 Nov 2014 18:31:26 -0800",
    "Tue, 11 Nov 2014 21:00:51 -0800",
    "Tue, 04 Nov 2014 19:03:49 -0800",
    "Tue, 28 Oct 2014 18:59:14 -0700",
    "Tue, 21 Oct 2014 19:12:58 -0700",
    "Tue, 14 Oct 2014 18:06:13 -0700",
    "Sun, 23 Jul 2017 22:00:00 -0700",
    "Tue, 07 Oct 2014 21:25:47 -0700",
    "Tue, 30 Sep 2014 18:33:48 -0700",
    "Tue, 23 Sep 2014 18:07:36 -0700",
    "Tue, 16 Sep 2014 19:31:51 -0700",
    "Tue, 09 Sep 2014 18:53:38 -0700",
    "Tue, 02 Sep 2014 19:47:06 -0700",
    "Tue, 26 Aug 2014 19:16:18 -0700",
    "Sun, 09 Jul 2017 22:00:00 -0700",
    "Tue, 19 Sep 2017 01:27:07 +0000",
    "Tue, 05 Sep 2017 01:20:45 +0000",
    "Sun, 25 Jun 2017 22:45:00 -0700",
    "Mon, 12 Jun 2017 12:00:00 -0700",
    "Sat, 19 Aug 2017 18:47:28 +0000",
    "Tue, 19 Aug 2014 18:38:08 -0700",
    "Tue, 12 Aug 2014 20:27:38 -0700",
    "Tue, 05 Aug 2014 18:19:30 -0700",
    "Tue, 29 Jul 2014 19:25:17 -0700",
    "Tue, 22 Jul 2014 10:21:36 -0700",
    "Tue, 15 Jul 2014 19:12:11 -0700",
    "Tue, 08 Jul 2014 18:48:59 -0700",
    "Tue, 01 Jul 2014 18:36:20 -0700",
    "Tue, 24 Jun 2014 17:31:14 -0700",
    "Tue, 17 Jun 2014 15:46:15 -0700",
    "Tue, 10 Jun 2014 17:03:08 -0700",
    "Tue, 03 Jun 2014 18:02:35 -0700",
    "Tue, 27 May 2014 17:34:45 -0700",
    "Tue, 20 May 2014 17:52:46 -0700",
    "Tue, 13 May 2014 19:23:29 -0700",
    "Tue, 06 May 2014 18:31:34 -0700",
    "Tue, 29 Apr 2014 17:52:49 -0700",
    "Tue, 22 Apr 2014 18:18:39 -0700",
    "Tue, 15 Apr 2014 16:37:17 -0700",
    "Tue, 08 Apr 2014 17:29:25 -0700",
    "Tue, 01 Apr 2014 21:36:43 -0700",
    "Tue, 25 Mar 2014 18:14:46 -0700",
    "Tue, 18 Mar 2014 20:42:59 -0700",
    "Tue, 11 Mar 2014 17:42:35 -0700",
    "Tue, 04 Mar 2014 17:09:07 -0800",
    "Tue, 25 Feb 2014 17:25:51 -0800",
    "Tue, 18 Feb 2014 18:03:14 -0800",
    "Tue, 11 Feb 2014 17:40:11 -0800",
    "Sun, 13 Aug 2017 11:13:50 +0000",
    "Mon, 07 Aug 2017 18:22:57 +0000",
    "Sun, 23 Jul 2017 15:23:38 +0000",
    "Tue, 04 Feb 2014 17:57:53 -0800",
    "Tue, 28 Jan 2014 22:29:57 -0800",
    "Tue, 21 Jan 2014 17:33:14 -0800",
    "Tue, 14 Jan 2014 19:01:45 -0800",
    "Tue, 07 Jan 2014 17:05:36 -0800",
    "Tue, 11 Jul 2017 01:07:15 +0000",
    "Mon, 25 Sep 2017 08:00:00 -0400",
    "Tue, 31 Dec 2013 18:03:41 -0800",
    "Tue, 24 Dec 2013 09:27:00 -0800",
    "Tue, 17 Dec 2013 17:05:59 -0800",
    "Tue, 10 Dec 2013 18:49:34 -0800",
    "Wed, 04 Dec 2013 07:19:47 -0800",
    "Mon, 26 Jun 2017 22:50:56 +0000",
    "Tue, 13 Jun 2017 00:47:49 +0000",
    "Tue, 30 May 2017 01:29:49 +0000",
    "Fri, 22 Sep 2017 08:00:00 -0400",
    "Thu, 21 Sep 2017 08:00:00 -0400",
    "Wed, 20 Sep 2017 08:00:00 -0400",
    "Tue, 19 Sep 2017 08:00:00 -0400",
    "Mon, 18 Sep 2017 08:00:00 -0400",
    "Tue, 26 Nov 2013 18:07:16 -0800",
    "Fri, 15 Sep 2017 08:00:00 -0400",
    "Tue, 12 Nov 2013 16:30:29 -0800",
    "Tue, 05 Nov 2013 17:02:21 -0800",
    "Thu, 14 Sep 2017 08:00:00 -0400",
    "Tue, 19 Nov 2013 19:48:11 -0800",
    "Tue, 29 Oct 2013 17:21:45 -0700",
    "Wed, 13 Sep 2017 08:00:00 -0400",
    "Thu, 21 Sep 2017 08:00:00 -0400",
    "Tue, 22 Oct 2013 18:05:53 -0700",
    "Thu, 21 Sep 2017 06:00:00 -0400",
    "Thu, 14 Sep 2017 06:00:00 -0400",
    "Thu, 07 Sep 2017 06:00:00 -0400",
    "Thu, 31 Aug 2017 06:00:00 -0400",
    "Thu, 24 Aug 2017 06:00:00 -0400",
    "Thu, 17 Aug 2017 06:00:00 -0400",
    "Thu, 10 Aug 2017 06:00:00 -0400",
    "Thu, 03 August 2017 06:00:00 -0400",
    "Thu, 27 July 2017 06:00:00 -0400",
    "Thu, 20 July 2017 06:00:00 -0400",
    "Thu, 13 July 2017 06:00:00 -0400",
    "Thu, 06 July 2017 06:00:00 -0400",
    "Thu, 28 June 2017 06:00:00 -0400",
    "Thu, 22 June 2017 06:00:00 -0400",
    "Thu, 15 June 2017 06:00:00 -0400",
    "Thu, 7 June 2017 06:00:00 -0400",
    "Thu, 1 June 2017 06:00:00 -0400",
    "Thu, 25 May 2017 06:00:00 -0400",
    "Thu, 18 May 2017 06:00:00 -0400",
    "Thu, 10 May 2017 06:00:00 -0400",
    "Thu, 04 May 2017 06:00:00 -0400",
    "Thu, 27 Apr 2017 06:00:00 -0400",
    "Thu, 20 Apr 2017 06:00:00 -0400",
    "Thu, 13 Apr 2017 06:00:00 -0400",
    "Thu, 06 Apr 2017 06:00:00 -0400",
    "Thu, 30 Mar 2017 06:00:00 -0400",
    "Thu, 23 Mar 2017 06:00:00 -0400",
    "Thu, 16 Mar 2017 06:00:00 -0400",
    "Thu, 09 Mar 2017 06:00:00 -0400",
    "Thu, 02 Mar 2017 06:00:00 -0400",
    "Thu, 22 Feb 2017 06:00:00 -0400",
    "Thu, 15 Feb 2017 06:00:00 -0400",
    "Thu, 09 Feb 2017 06:00:00 -0400",
    "Thu, 02 Feb 2017 06:00:00 -0400",
    "Thu, 26 Jan 2017 06:00:00 -0400",
    "Thu, 19 Jan 2017 06:00:00 -0400",
    "Thu, 12 Jan 2017 06:00:00 -0400",
    "Thu, 29 Dec 2016 06:00:00 -0400",
    "Fri, 23 Dec 2016 06:00:00 -0400",
    "Thu, 20 Dec 2016 06:00:00 -0400",
    "Thu, 15 Dec 2016 06:00:00 -0400",
    "Thu, 08 Dec 2016 06:00:00 -0400",
    "Thu, 01 Dec 2016 06:00:00 -0400",
    "Thu, 23 Nov 2016 06:00:00 -0400",
    "Thu, 17 Nov 2016 06:00:00 -0400",
    "Thu, 10 Nov 2016 06:00:00 -0400",
    "Thu, 03 Nov 2016 06:00:00 -0400",
    "Thu, 13 Oct 2016 06:00:00 -0400",
    "Thu, 13 Oct 2016 06:00:00 -0400",
    "Thu, 06 Oct 2016 06:00:00 -0400",
    "Thu, 29 Sep 2016 06:00:00 -0400",
    "Thu, 22 Sep 2016 06:00:00 -0400",
    "Thu, 15 Sep 2016 06:00:00 -0400",
    "Thu, 08 Sep 2016 06:00:00 -0400",
    "Thu, 01 Sep 2016 06:00:00 -0400",
    "Thu, 25 Aug 2016 06:00:00 -0400",
    "Thu, 18 Aug 2016 06:00:00 -0400",
    "Thu, 11 Aug 2016 06:00:00 -0400",
    "Thu, 05 Aug 2016 06:00:00 -0400",
    "Tue, 19 Sep 2017 08:00:00 -0400",
    "Tue, 15 Oct 2013 16:59:19 -0700",
    "Tue, 12 Sep 2017 08:00:00 -0400",
    "Thu, 28 Jul 2016 06:00:00 -0400",
    "Thu, 21 Jul 2016 06:00:00 -0400",
    "Thu, 14 Jul 2016 06:00:00 -0400",
    "Thu, 07 Jul 2016 06:00:00 -0400",
    "Thu, 30 Jun 2016 06:00:00 -0400",
    "Thu, 23 Jun 2016 06:00:00 -0400",
    "Thu, 16 Jun 2016 06:00:00 -0400",
    "Tue, 24 Sep 2013 17:37:19 -0700",
    "Tue, 17 Sep 2013 18:06:18 -0700",
    "Thu, 21 Sep 2017 08:00:00 -0400",
    "Fri, 09 Jun 2016 12:00:00 -0400",
    "Fri, 03 Jun 2016 06:33:07 -0400",
    "Thu, 26 May 2016 06:33:07 -0400",
    "Thu, 19 May 2016 06:33:07 -0400",
    "Thu, 12 May 2016 17:33:07 -0400",
    "Tue, 10 Sep 2013 17:14:16 -0700",
    "Mon, 05 Jan 2015 06:00:03 -0400",
    "Sun, 04 Jan 2015 06:00:03 -0400",
    "Wed, 31 Dec 2014 06:00:03 -0400",
    "Thu, 18 Dec 2014 06:00:03 -0400",
    "Thu, 11 Dec 2014 06:00:03 -0400",
    "Thu, 04 Dec 2014 06:00:03 -0400",
    "Thu, 20 Nov 2014 06:00:03 -0400",
    "Thu, 13 Nov 2014 06:00:03 -0400",
    "Thu, 06 Nov 2014 06:00:03 -0400",
    "Thu, 30 Oct 2014 06:00:03 -0400",
    "Thu, 23 Oct 2014 06:00:03 -0400",
    "Thu, 16 Oct 2014 06:00:03 -0400",
    "Thu, 09 Oct 2014 06:00:03 -0400",
    "Thu, 02 Oct 2014 06:00:03 -0400",
    "Thu, 25 Sep 2014 06:00:03 -0400",
    "Thu, 18 Sep 2014 06:00:03 -0400",
    "Thu, 11 Sep 2014 06:00:03 -0400",
    "Thu, 04 Sep 2014 06:00:03 -0400",
    "Thu, 28 Aug 2014 06:00:03 -0400",
    "Thu, 21 Aug 2014 06:00:03 -0400",
    "Thu, 14 Aug 2014 06:00:03 -0400",
    "Thu, 07 Aug 2014 06:00:03 -0400",
    "Thu, 31 Jul 2014 06:00:03 -0400",
    "Thu, 24 Jul 2014 06:00:03 -0400",
    "Thu, 17 Jul 2014 06:00:03 -0400",
    "Thu, 10 Jul 2014 06:00:03 -0400",
    "Thu, 03 Jul 2014 06:00:03 -0400",
    "Thu, 26 Jun 2014 06:00:03 -0400",
    "Thu, 19 Jun 2014 06:00:03 -0400",
    "Thu, 12 Jun 2014 06:00:03 -0400",
    "Thu, 05 Jun 2014 06:00:03 -0400",
    "Thu, 29 May 2014 06:00:03 -0400",
    "Thu, 22 May 2014 06:00:03 -0400",
    "Thu, 15 May 2014 06:00:03 -0400",
    "Thu, 08 May 2014 06:00:03 -0400",
    "Thu, 05 May 2016 17:33:07 -0400",
    "Thu, 28 Apr 2016 06:00:03 -0400",
    "Thu, 21 Apr 2016 06:00:03 -0400",
    "Thu, 14 Apr 2016 06:00:03 -0400",
    "Thu, 07 Apr 2016 06:00:03 -0400",
    "Thu, 31 Mar 2016 06:00:03 -0400",
    "Thu, 01 May 2014 06:00:03 -0400",
    "Thu, 24 Apr 2014 06:00:03 -0400",
    "Thu, 17 Apr 2014 06:00:03 -0400",
    "Thu, 10 Apr 2014 06:00:03 -0400",
    "Thu, 02 Apr 2014 06:00:03 -0400",
    "Thu, 27 Mar 2014 06:00:03 -0400",
    "Thu, 20 Mar 2014 06:00:03 -0400",
    "Thu, 13 Mar 2014 06:00:03 -0400",
    "Thu, 06 Mar 2014 06:00:03 -0400",
    "Thu, 27 Feb 2014 06:00:03 -0400",
    "Thu, 20 Feb 2014 06:00:03 -0400",
    "Thu, 14 Feb 2014 06:00:03 -0400",
    "Thu, 06 Feb 2014 06:00:03 -0400",
    "Thu, 30 Jan 2014 06:00:03 -0400",
    "Thu, 23 Jan 2014 06:00:03 -0400",
    "Thu, 16 Jan 2014 06:00:03 -0400",
    "Wed, 08 Jan 2014 06:00:03 -0400",
    "Tue, 07 Jan 2014 06:00:03 -0400",
    "Mon, 06 Jan 2014 06:00:03 -0400",
    "Sun, 05 Jan 2014 06:00:03 -0400",
    "Thu, 02 Jan 2014 06:00:03 -0400",
    "Thu, 19 Dec 2013 06:00:03 -0400",
    "Thu, 12 Dec 2013 06:00:03 -0400",
    "Thu, 04 Dec 2013 06:00:03 -0400",
    "Thu, 21 Nov 2013 06:00:03 -0400",
    "Thu, 14 Nov 2013 06:00:03 -0400",
    "Thu, 07 Nov 2013 06:00:03 -0400",
    "Thu, 24 Mar 2016 06:00:03 -0400",
    "Thu, 17 Mar 2016 06:00:03 -0400",
    "Thu, 10 Mar 2016 06:00:03 -0400",
    "Thu, 03 Mar 2016 06:00:03 -0400",
    "Thu, 25 Feb 2016 06:00:03 -0400",
    "Thu, 18 Feb 2016 06:00:03 -0400",
    "Thu, 11 Feb 2016 06:00:03 -0400",
    "Thu, 04 Feb 2016 06:00:03 -0400",
    "Thu, 28 Jan 2016 06:00:03 -0400",
    "Thu, 21 Jan 2016 06:00:03 -0400",
    "Wed, 14 Jan 2016 06:00:03 -0400",
    "Wed, 06 Jan 2016 06:00:03 -0400",
    "Wed, 06 Jan 2016 01:00:03 -0400",
    "Tue, 05 Jan 2016 01:00:03 -0400",
    "Thu, 31 Dec 2015 06:00:03 -0400",
    "Thu, 23 Dec 2015 06:00:03 -0400",
    "Thu, 17 Dec 2015 06:00:03 -0400",
    "Thu, 10 Dec 2015 06:00:03 -0400",
    "Thu, 03 Dec 2015 06:00:03 -0400",
    "Thu, 19 Nov 2015 06:00:03 -0400",
    "Thu, 12 Nov 2015 06:00:03 -0400",
    "Thu, 05 Nov 2015 06:00:03 -0400",
    "Thu, 29 Oct 2015 06:00:03 -0400",
    "Thu, 22 Oct 2015 06:00:03 -0400",
    "Thu, 15 Oct 2015 06:00:03 -0400",
    "Thu, 08 Oct 2015 06:00:03 -0400",
    "Thu, 01 Oct 2015 06:00:03 -0400",
    "Thu, 24 Sep 2015 06:00:03 -0400",
    "Thu, 17 Sep 2015 06:00:03 -0400",
    "Tue, 08 Oct 2013 17:57:08 -0700",
    "Thu, 31 Oct 2013 06:00:03 -0400",
    "Thu, 24 Oct 2013 06:00:03 -0400",
    "Thu, 17 Oct 2013 06:00:03 -0400",
    "Thu, 10 Oct 2013 06:00:03 -0400",
    "Thu, 03 Oct 2013 06:00:03 -0400",
    "Thu, 26 Sep 2013 06:00:03 -0400",
    "Thu, 19 Sep 2013 06:00:03 -0400",
    "Thu, 12 Sep 2013 06:00:03 -0400",
    "Thu, 05 Sep 2013 06:00:03 -0400",
    "Thu, 29 Aug 2013 06:00:03 -0400",
    "Thu, 22 Aug 2013 06:00:03 -0400",
    "Thu, 15 Aug 2013 06:00:03 -0400",
    "Thu, 08 Aug 2013 06:00:03 -0400",
    "Thu, 01 Aug 2013 06:00:03 -0400",
    "Thu, 25 Jul 2013 06:00:03 -0400",
    "Thu, 17 Jul 2013 06:00:03 -0400",
    "Thu, 11 Jul 2013 06:00:03 -0400",
    "Thu, 10 Sep 2015 06:00:03 -0400",
    "Thu, 03 Sep 2015 06:00:03 -0400",
    "Thu, 27 Aug 2015 06:00:03 -0400",
    "Thu, 20 Aug 2015 06:00:03 -0400",
    "Thu, 13 Aug 2015 06:00:03 -0400",
    "Thu, 06 Aug 2015 06:00:03 -0400",
    "Thu, 30 Jul 2015 06:00:03 -0400",
    "Thu, 23 Jul 2015 06:00:03 -0400",
    "Thu, 16 Jul 2015 06:00:03 -0400",
    "Thu, 09 Jul 2015 06:00:03 -0400",
    "Thu, 02 Jul 2015 06:00:03 -0400",
    "Thu, 25 Jun 2015 06:00:03 -0400",
    "Thu, 04 Jul 2013 06:00:03 -0400",
    "Thu, 27 Jun 2013 06:00:03 -0400",
    "Thu, 20 Jun 2013 06:00:03 -0400",
    "Thu, 13 Jun 2013 06:00:03 -0400",
    "Thu, 06 Jun 2013 06:00:03 -0400",
    "Thu, 30 May 2013 06:00:03 -0400",
    "Thu, 23 May 2013 06:00:03 -0400",
    "Thu, 16 May 2013 06:00:03 -0400",
    "Thu, 09 May 2013 06:00:03 -0400",
    "Thu, 02 May 2013 06:00:03 -0400",
    "Thu, 25 Apr 2013 06:00:03 -0400",
    "Thu, 18 Apr 2013 06:00:03 -0400",
    "Thu, 11 Apr 2013 06:00:03 -0400",
    "Thu, 04 Apr 2013 06:00:03 -0400",
    "Thu, 28 Mar 2013 06:00:03 -0400",
    "Thu, 21 Mar 2013 06:00:03 -0400",
    "Thu, 14 Mar 2013 06:00:03 -0400",
    "Thu, 07 Mar 2013 06:00:03 -0400",
    "Thu, 18 Jun 2015 06:00:03 -0400",
    "Thu, 11 Jun 2015 06:00:03 -0400",
    "Thu, 04 Jun 2015 06:00:03 -0400",
    "Thu, 28 May 2015 06:00:03 -0400",
    "Thu, 21 May 2015 06:00:03 -0400",
    "Thu, 14 May 2015 06:00:03 -0400",
    "Thu, 07 May 2015 06:00:03 -0400",
    "Thu, 30 Apr 2015 06:00:03 -0400",
    "Thu, 23 Apr 2015 06:00:03 -0400",
    "Thu, 16 Apr 2015 06:00:03 -0400",
    "Thu, 09 Apr 2015 06:00:03 -0400",
    "Thu, 02 Apr 2015 06:00:03 -0400",
    "Thu, 26 Mar 2015 06:00:03 -0400",
    "Thu, 19 Mar 2015 06:00:03 -0400",
    "Thu, 12 Mar 2015 06:00:03 -0400",
    "Thu, 05 Mar 2015 06:00:03 -0400",
    "Thu, 26 Feb 2015 06:00:03 -0400",
    "Thu, 19 Feb 2015 06:00:03 -0400",
    "Thu, 12 Feb 2015 06:00:03 -0400",
    "Thu, 05 Feb 2015 06:00:03 -0400",
    "Thu, 29 Jan 2015 06:00:03 -0400",
    "Thu, 22 Jan 2015 06:00:03 -0400",
    "Thu, 15 Jan 2015 06:00:03 -0400",
    "Tue, 06 Jan 2015 06:00:03 -0400",
    "Mon, 19 Aug 2013 18:58:12 -0700",
    "Mon, 12 Aug 2013 19:05:39 -0700",
    "Tue, 03 Sep 2013 18:19:04 -0700",
    "Thu, 28 Feb 2013 06:00:03 -0400",
    "Thu, 21 Feb 2013 06:00:03 -0400",
    "Thu, 14 Feb 2013 06:00:03 -0400",
    "Thu, 07 Feb 2013 06:00:03 -0400",
    "Thu, 31 Jan 2013 06:00:03 -0400",
    "Thu, 24 Jan 2013 06:00:03 -0400",
    "Thu, 17 Jan 2013 06:00:03 -0400",
    "Wed, 09 Jan 2013 06:00:03 -0400",
    "Mon, 07 Jan 2013 06:00:03 -0400",
    "Mon, 26 Aug 2013 18:27:57 -0700",
    "Tue, 01 Oct 2013 17:20:16 -0700",
    "Sun, 06 Jan 2013 06:00:03 -0400",
    "Thu, 03 Jan 2013 06:00:03 -0400",
    "Thu, 27 Dec 2012 06:00:03 -0400",
    "Thu, 20 Dec 2012 06:00:03 -0400",
    "Thu, 30 Aug 2012 06:00:03 -0400",
    "Thu, 23 Aug 2012 06:00:03 -0400",
    "Thu, 16 Aug 2012 06:00:03 -0400",
    "Thu, 28 Jun 2012 06:00:03 -0400",
    "Thu, 01 Nov 2012 06:00:03 -0400",
    "Thu, 25 Oct 2012 06:00:03 -0400",
    "Thu, 18 Oct 2012 06:00:03 -0400",
    "Thu, 11 Oct 2012 06:00:03 -0400",
    "Thu, 13 Dec 2012 06:00:03 -0400",
    "Thu, 06 Dec 2012 06:00:03 -0400",
    "Thu, 21 Jun 2012 06:00:03 -0400",
    "Thu, 04 Oct 2012 06:00:03 -0400",
    "Thu, 27 Sep 2012 06:00:03 -0400",
    "Thu, 29 Nov 2012 06:00:03 -0400",
    "Thu, 15 Nov 2012 06:00:03 -0400",
    "Thu, 08 Nov 2012 06:00:03 -0400",
    "Thu, 20 Sep 2012 06:00:03 -0400",
    "Thu, 14 Jun 2012 06:00:03 -0400",
    "Thu, 07 Jun 2012 06:00:03 -0400",
    "Thu, 31 May 2012 06:00:03 -0400",
    "Thu, 24 May 2012 06:00:03 -0400",
    "Thu, 17 May 2012 06:00:03 -0400",
    "Thu, 06 Sep 2012 06:00:03 -0400",
    "Thu, 13 Sep 2012 06:00:03 -0400",
    "Thu, 09 Aug 2012 06:00:03 -0400",
    "Thu, 19 Jul 2012 06:00:03 -0400",
    "Thu, 03 May 2012 06:00:03 -0400",
    "Sat, 04 Aug 2012 06:00:03 -0400",
    "Thu, 10 May 2012 06:00:03 -0400",
    "Thu, 12 Jul 2012 06:00:03 -0400",
    "Thu, 05 Jul 2012 06:00:03 -0400",
    "Thu, 26 Jul 2012 06:00:03 -0400",
];

#[bench]
fn bench_correct_dates_with_fallback(b: &mut Bencher) {
    b.iter(|| {
        for &d in VALID_DATES {
            let _ = parse_from_rfc2822_with_fallback(d);
        }
    })
}

#[bench]
fn bench_correct_dates_normal_parse(b: &mut Bencher) {
    b.iter(|| {
        for &d in VALID_DATES {
            let _ = DateTime::parse_from_rfc2822(d);
        }
    })
}

#[bench]
fn bench_parse_invalid_dates_with_fallback(b: &mut Bencher) {
    b.iter(|| {
        for &d in INVALID_DATES {
            let _ = parse_from_rfc2822_with_fallback(d);
        }
    })
}

#[bench]
/// Used for performance benchmarking
fn mixed_sample_case(b: &mut Bencher) {
    b.iter(|| {
        for &d in MIXED_DATES {
            let _ = parse_from_rfc2822_with_fallback(d);
        }
    })
}
