% Generated by roxygen2 (4.1.0): do not edit by hand
% Please edit documentation in R/rollit.R
\name{rollit}
\alias{rollit}
\title{Generate your own Weighted C++ Roll Function}
\usage{
rollit(fun = "x", vector = FALSE, const_vars = NULL, combine = "+",
  final_trans = NULL, includes = NULL, depends = NULL, inline = TRUE,
  name = NULL, ...)
}
\arguments{
\item{fun}{A character string defining the function call. The function must be in
terms of \code{x}. The function will be applied individually
to each element being 'roll'ed over, unless \code{vector} is \code{TRUE}.}

\item{vector}{boolean; if \code{TRUE}, then \code{fun} is a scalar-valued
function of a vector, rather than a function to apply to each element
individually.}

\item{const_vars}{Constant variables that will live within
the sourced C++ function. Format is a named \code{list}; e.g, you
could pass \code{list(e=exp(1))} to have \code{e} as a constant variable
available in the function being called. Note that the variable \code{N}
is fixed to be the number of non-zero weights passed, to facilitate the use
of \code{0} weights in terms of skipping elements.}

\item{combine}{character; typically one of \code{"+", "-", "*", "/"}, but
any operator usable as a C++ compound assignment operator is accepted.
This specifies how each element should be combined in the rolling function.
Only used when \code{vector} is \code{FALSE}.}

\item{final_trans}{A final transformation to perform after either 'rolling'
over each element in the vector \code{x} (with \code{vector=FALSE}),
or after applying a scalar-valued function of a vector (with \code{vector=TRUE}).
Must be in terms of \code{x}.}

\item{includes}{Other C++ libraries to include. For example, to include
\code{boost/math.hpp}, you would pass
\code{c("<boost/math.hpp>")}.}

\item{depends}{Other libraries to link to. Linking is done through
Rcpp attributes.}

\item{inline}{boolean; mark the function generated as \code{inline}?
This may or may not increase execution speed.}

\item{name}{string; a name to internally assign to your generated C++ functions.}

\item{...}{Additional arguments passed to \code{sourceCpp}.}
}
\value{
A wrapper \R function that calls compiled C++ files, as generated through
\code{sourceCpp}. See \code{\link{rollit_example}} for more information on the
functions generated by \code{rollit}.
}
\description{
Using this interface, you can define a function that you would like to
be called on each sub-vector you are rolling over. The generated code is
compiled and exposed via \code{sourceCpp}.
}
\details{
By default, we include \code{<RcppArmadillo.h>} in each file; however, you can
include your own libraries with the \code{includes} call.
}
\note{
All functions generated use Rcpp's \code{NumericVector} and
\code{NumericMatrix} to interface with \R vectors and matrices.
Elements within these vectors are
translated as \code{double}s so any function that receives a \code{double}
will work fine.

If you want to just write your own C++ function to wrap into a 'rolling'
interface, see \code{\link{rollit_raw}}.
}
\examples{
\dontrun{
x <- matrix(1:16, nrow=4)

## the squared rolling sum -- we square the sum of our rolled results
rolling_sqsum <- rollit( final_trans="x*x" )

rolling_sqsum( x, 4 )
rolling_sqsum( x, 4, by.column=FALSE )
cbind( as.vector(rolling_sqsum(x, 4)), apply(x, 2, function(x) { sum(x)^2 } ) )

## implement your own variance function
## we can use the sugar function 'mean' to get
## the mean of x

const_vars <- list(m = "mean(x)")
var_fun <- "( (x-m) * (x-m) )/(N-1)"
rolling_var <- rollit( var_fun, const_vars=const_vars )

x <- c(1, 5, 10, 15)
cbind( rolling_var(x, 2), roll_var(x, 2) )

## use a function from cmath

rolling_log10 <- rollit( "log10(x)" )
rolling_log10( 10^(1:5), 2 )

## rolling product
rolling_prod <- rollit( combine="*" )
rolling_prod( 1:10, 2 )

## using weights to specify something like a 'by' argument
rolling_prod( 1:10, 3, weights=c(1,0,1) )

## a benchmark

if( require("microbenchmark") && require("zoo") ) {
  x <- rnorm(1E4)
  microbenchmark(
    rolling_var(x, 100),
    roll_var(x, 100),
    rollapply(x, 100, var),
    times=10
    )
  }}
}
\seealso{
\code{\link{rollit_example}} for an example of the function signature
for functions generated with \code{rollit},
\code{\link{sourceCpp}} for information on how Rcpp
compiles your functions, \code{\link{get_rollit_source}} for
inspection of the generated C++ code, and \code{\link{rollit_raw}} for
wrapping in your own C++ code.
}

