/*--------------------------------------------------------------------------*/
/* ALBERTA:   an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques                                                     */
/*                                                                          */
/* file:     estimator_dowb.c                                               */
/*                                                                          */
/* description:  residual error estimator for elliptic and parabolic        */
/*               problems, REAL_D version (for use with DOWB matrices)      */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*             Claus-Justus Heine                                           */
/*             Abteilung fuer Angewandte Mathematik                         */
/*             Alberta-Ludwigs-Universitaet Freiburg                         */
/*             Hermann-Herder-Str. 10                                       */
/*             D-79104 Freiburg im Breisgau, Germany                        */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                        */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"

/*--------------------------------------------------------------------------*/
/*  residual type estimator for quasi linear elliptic problem:              */
/*   -\div A \nabla u + f(.,u,\nabla u) = 0                                 */
/*--------------------------------------------------------------------------*/

typedef struct ellipt_est_d_data ELLIPT_EST_D_DATA;
struct ellipt_est_d_data
{
  const DOF_REAL_D_VEC *uh;
  const BAS_FCTS       *bas_fcts;

  const REAL_DD (*A)[DIM_OF_WORLD];

  const REAL        *(*f)(const EL_INFO *, const QUAD *, int iq,
			  const REAL_D uh,
			  const REAL_DD grd_uh, REAL_D res);

  const QUAD_FAST  *quad_fast;      /*--  element integration  -------------*/
  const QUAD       *quad;           /*--  face integration     -------------*/

  REAL             *(*rw_est)(EL *);
  REAL             *(*rw_estc)(EL *);

  REAL_D           *uh_el;          /*--  vector for storing uh on el  -----*/

  FLAGS            f_flag;
  int              norm;            /*--  estimated norm: H1_NORM/L2_NORM --*/

  REAL             C0, C1, C2;

  REAL             est_sum;
  REAL             est_max;
};

static REAL h2_from_det(int dim, REAL det)
{
  FUNCNAME("h2_from_det");

  switch(dim) {
  case 1:
    return det*det;
  case 2:
    return det;
  case 3:
    return pow(det, 2.0/3.0);
  default:
    ERROR_EXIT("Illegal dim!\n");
    return 0.0; /* shut up the compiler */
  }
}

/*--------------------------------------------------------------------------*/
/* element residual:  C0*h_S^2*||-div A nabla u^h + r||_L^2(S)  (H^1)       */
/*                    C0*h_S^4*||-div A nabla u^h + r||_L^2(S)  (L^2)       */
/*--------------------------------------------------------------------------*/

static REAL el_res2(const EL_INFO *el_info, const REAL_D *Lambda, REAL det, 
		    const ELLIPT_EST_D_DATA *data)
{
  const REAL_DD   (*D2uhqp)[DIM_OF_WORLD];
  int             dim = el_info->mesh->dim;
  REAL            val, h2 = h2_from_det(dim, det);
  const REAL      *uh_qpi = NULL;
  int             iq, i, j, mu, nu;
  REAL_D          riq = {};
  const REAL_D    *uh_qp = nil, *grd_uh_qpi = nil;
  const REAL_DD   *grd_uh_qp = nil;
  const QUAD_FAST *quad_fast = data->quad_fast;
  const QUAD      *quad = quad_fast->quad;

  if (data->quad_fast->bas_fcts->degree > 1)
    D2uhqp = D2_uh_d_at_qp(data->quad_fast, Lambda,
			   (const REAL_D *)data->uh_el, nil);
  else
    D2uhqp = nil;

  if (data->f) {
    if (data->f_flag & INIT_UH)
      uh_qp = uh_d_at_qp(quad_fast, (const REAL_D *)data->uh_el, nil);
    else
      uh_qpi = nil;

    if (data->f_flag & INIT_GRD_UH)
      grd_uh_qp = grd_uh_d_at_qp(quad_fast, Lambda,
				 (const REAL_D *)data->uh_el, nil);
    else
      grd_uh_qpi = nil;
  }

  for (val = iq = 0; iq < quad->n_points; iq++)
  {
    if (data->f) {
      if (data->f_flag & INIT_UH)     uh_qpi     = uh_qp[iq];
      if (data->f_flag & INIT_GRD_UH) grd_uh_qpi = grd_uh_qp[iq];

      data->f(el_info, quad, iq, uh_qpi,  grd_uh_qpi, riq);
    }
    else
      SET_DOW(0.0, riq);


    if (D2uhqp) {
      for(mu = 0; mu < DIM_OF_WORLD; mu++)
	for (i = 0; i < DIM_OF_WORLD; i++)
	  for(nu = 0; nu < DIM_OF_WORLD; nu++)
	    for (j = 0; j < DIM_OF_WORLD; j++)
	      riq[mu] -= data->A[i][j][mu][nu] * D2uhqp[iq][nu][i][j];
    }
    
    val += quad->w[iq]*NRM2_DOW(riq);
  }

  if (data->norm == L2_NORM)
    val = data->C0*h2*h2*det*val;
  else
    val = data->C0*h2*det*val;
  
  return(val);
}


/*--------------------------------------------------------------------------*/
/*  face residuals:  C1*h_Gamma*||[A(u_h)]||_L^2(Gamma)^2          (H^1)    */
/*                   C1*h_S^2*h_Gamma*||[A(u_h)]||_L^2(Gamma)^2    (L^2)    */
/*  Since det_S = det_Gamma*h_Gamma we use for det_Gamma*h_Gamma the term   */
/*  0.5(det_S + det_S')                                                     */
/*--------------------------------------------------------------------------*/

static REAL jump_res2(const EL_INFO *el_info, int face, const REAL_D *Lambda,
		      REAL det, const ELLIPT_EST_D_DATA *data)
{
  EL_INFO        neigh_info[1];
  int            dim = el_info->mesh->dim;
  int            face_ind_el[dim], face_ind_neigh[dim];
  EL             *neigh = el_info->neigh[face];
  int            opp_v  = el_info->opp_vertex[face];
  int            mu, nu, i, j, i1, i2, iq;
  REAL_DD        jump, grd_uh_el, grd_uh_neigh;
  REAL_D         Lambda_neigh[N_LAMBDA];
  const REAL_D   *uh_neigh;
  REAL           det_neigh = 0.0, lambda[N_LAMBDA], val = 0.0;
  const BAS_FCTS *bas_fcts = data->uh->fe_space->bas_fcts;
  const QUAD     *quad = data->quad;

/*--------------------------------------------------------------------------*/
/* orient the edge/face => same quadrature nodes from both sides!           */
/*--------------------------------------------------------------------------*/

  sort_wall_indices(dim, el_info->el, face, face_ind_el);
  sort_wall_indices(dim, neigh, opp_v, face_ind_neigh);

  neigh_info->mesh = el_info->mesh;
  neigh_info->el = neigh;
  neigh_info->fill_flag = FILL_COORDS;

  for (j = 0; j < DIM_OF_WORLD; j++)
    neigh_info->coord[opp_v][j] = el_info->opp_coord[face][j];

  for (i = 0; i < dim; i++) {
    i1 = face_ind_el[i];
    i2 = face_ind_neigh[i];
    for (j = 0; j < DIM_OF_WORLD; j++)
      neigh_info->coord[i2][j] = el_info->coord[i1][j];
  }

  switch(dim) {
  case 1:
    det_neigh = el_grd_lambda_1d(neigh_info, Lambda_neigh);
    break;
#if DIM_OF_WORLD > 1
  case 2:
    det_neigh = el_grd_lambda_2d(neigh_info, Lambda_neigh);
    break;
#if DIM_OF_WORLD > 2
  case 3:
    det_neigh = el_grd_lambda_3d(neigh_info, Lambda_neigh);
#endif
#endif
  }

  uh_neigh = bas_fcts->get_real_d_vec(neigh, data->uh, nil);

/*--------------------------------------------------------------------------*/
/*  now eval the jump at all quadrature nodes                               */
/*--------------------------------------------------------------------------*/

  for (val = iq = 0; iq < quad->n_points; iq++) {
    lambda[face] = 0.0;
    for (i = 0; i < dim; i++)
      lambda[face_ind_el[i]] = quad->lambda[iq][i];
    eval_grd_uh_d(lambda,  Lambda,
		  (const REAL_D *)data->uh_el, bas_fcts, grd_uh_el);

    lambda[opp_v] = 0.0;
    for (i = 0; i < dim; i++)
      lambda[face_ind_neigh[i]] = quad->lambda[iq][i];
    eval_grd_uh_d(lambda, (const REAL_D *)Lambda_neigh, uh_neigh, bas_fcts, 
		  grd_uh_neigh);
    
    for (mu = 0; mu < DIM_OF_WORLD; mu++)
      for (i = 0; i < DIM_OF_WORLD; i++)
	for (jump[mu][i] = nu = 0; nu < DIM_OF_WORLD; nu++)
	  for (j = 0; j < DIM_OF_WORLD; j++)
	    jump[mu][i] += data->A[i][j][mu][nu]
	      * (grd_uh_el[nu][j] - grd_uh_neigh[nu][j]);

    val += quad->w[iq]*MSCP_DOW(jump,jump);
  }

  det = 0.5*(det + det_neigh);
  if (data->norm == L2_NORM)
    return(data->C1*h2_from_det(dim, det)*det*val);
  else
    return(data->C1*det*val);
}

/*--------------------------------------------------------------------------*/
/*  neuman residual:  C1*h_Gamma*||A(u_h).normal||_L^2(Gamma)^2    (H^1)    */
/*                    C1*h_S^2*h_Gamma*||A(u_h).normal]||_L^2(Gamma)^2 (L^2)*/
/*  Since det_S = det_Gamma*h_Gamma we use for det_Gamma*h_Gamma the term   */
/*  det_S                                                                   */
/*--------------------------------------------------------------------------*/

static REAL neumann_res2(const EL_INFO *el_info, int face, 
			 const REAL_D *Lambda, REAL det, 
			 const ELLIPT_EST_D_DATA *data)
{
  int            i, j, mu, nu, iq;
  int            dim = el_info->mesh->dim;
  REAL           lambda[N_LAMBDA];
  REAL_D         n_A_grd_uh;
  REAL           val;
  REAL_D         normal;
  REAL_DD        grd_uh, A_grd_uh;
  const QUAD     *quad = data->quad;

  switch(dim) {
  case 1:
    get_wall_normal_1d(el_info, face, normal);
    break;
#if DIM_OF_WORLD > 1
  case 2:
    get_wall_normal_2d(el_info, face, normal);
    break;
#if DIM_OF_WORLD > 2
  case 3:
    get_wall_normal_3d(el_info, face, normal);
#endif
#endif
  }

  for (val = iq = 0; iq < quad->n_points; iq++)
  {
    for (i = 0; i < face; i++)
      lambda[i] = quad->lambda[iq][i];
    lambda[face] = 0.0;
    for (i = face+1; i < N_LAMBDA; i++)
      lambda[i] = quad->lambda[iq][i-1];

    eval_grd_uh_d(lambda, (const REAL_D *)Lambda,
		  (const REAL_D *)data->uh_el, data->bas_fcts, grd_uh);

    for (mu = 0; mu < DIM_OF_WORLD; mu++)
      for (i = 0; i < DIM_OF_WORLD; i++)
	for (A_grd_uh[mu][i] = nu = 0; nu < DIM_OF_WORLD; nu++)
	  for (j = 0; j < DIM_OF_WORLD; j++)
	    A_grd_uh[mu][i] += data->A[i][j][mu][nu]*grd_uh[nu][j];

    SET_DOW(0.0, n_A_grd_uh);
    MV_DOW(A_grd_uh, normal, n_A_grd_uh);
    val += quad->w[iq]*NRM2_DOW(n_A_grd_uh);
  }

  if (data->norm == L2_NORM)
    return(data->C1*h2_from_det(dim, det)*det*val);
  else
    return(data->C1*det*val);
}


#define DATA ((ELLIPT_EST_D_DATA *)data)
static void clear_indicator_fct(const EL_INFO *el_info, void *data)
{
  el_info->el->mark = 1;
  if (DATA->rw_est)  *(DATA->rw_est(el_info->el)) = 0.0;
  if (DATA->rw_estc) *(DATA->rw_estc(el_info->el)) = 0.0;
  return;
}

static void ellipt_est_fct(const EL_INFO *el_info, void *data)
{
  EL           *el = el_info->el;
  REAL          det = 0.0, est_el;
  REAL_D        Lambda[N_LAMBDA];
  int           face;
  int           dim = el_info->mesh->dim;
  const S_CHAR *bound = 
    ((dim == 3) ? el_info->face_bound : el_info->edge_bound);
  EL           *neigh;

/*--- if rw_est, then there might already be contributions from jumps ------*/
  est_el = DATA->rw_est ? *(DATA->rw_est(el)) : 0.0;

  DATA->bas_fcts->get_real_d_vec(el, DATA->uh, DATA->uh_el);

  switch(dim) {
  case 1:
    det = el_grd_lambda_1d(el_info, Lambda);
    break;
#if DIM_OF_WORLD > 1
  case 2:
    det = el_grd_lambda_2d(el_info, Lambda);
    break;
#if DIM_OF_WORLD > 2
  case 3:
    det = el_grd_lambda_3d(el_info, Lambda);
#endif
#endif
  }


/*---  element residual  ---------------------------------------------------*/
  if (DATA->C0)
    est_el += el_res2(el_info, (const REAL_D *)Lambda, det, DATA);

/*---  face residuals  -----------------------------------------------------*/

  if (dim > 1 && DATA->C1)
  {
    for (face = 0; face < N_NEIGH(dim); face++)
    {
      if ((neigh = el_info->neigh[face]))
      {
/*--------------------------------------------------------------------------*/
/*  if rw_est is nil, compute jump for both neighbouring elements           */
/*                    only this allows correct computation of est_max       */
/*  if rw_est is not nil, compute jump only once for each edge/face         */
/*                    if neigh->mark: estimate not computed on neighbour!   */
/*  contribution for the element and for neighbour: 0.5*jump!               */
/*--------------------------------------------------------------------------*/
	if (!DATA->rw_est  ||  neigh->mark)
	{
	  REAL est = jump_res2(el_info, face, 
			       (const REAL_D *)Lambda, det, DATA);

	  est_el += est;
/*--  if rw_est, add neighbour contribution to neigbour indicator  ---------*/
	  if (DATA->rw_est) *(DATA->rw_est(neigh)) += est;
	}
      }
      else if (IS_NEUMANN(bound[face]))
      {
	est_el += neumann_res2(el_info, face, 
			       (const REAL_D *)Lambda, det, DATA);
      }
    }
  }

/*--  if rw_est, write indicator to element  -------------------------------*/
  if (DATA->rw_est) *(DATA->rw_est(el)) = est_el;

  DATA->est_sum += est_el;
  DATA->est_max = MAX(DATA->est_max, est_el);

  el_info->el->mark = 0; /*--- all contributions are computed!  ------------*/
  return;
}

#undef DATA

/*--------------------------------------------------------------------------*/

REAL ellipt_est_d(const DOF_REAL_D_VEC *uh, ADAPT_STAT *adapt,
		  REAL *(rw_est)(EL *), REAL *(rw_estc)(EL *),
		  int degree, int norm, REAL C[3],
		  const REAL_DD A[DIM_OF_WORLD][DIM_OF_WORLD],
		  const REAL *(*f)(const EL_INFO *,
				   const QUAD *, int qp,
				   const REAL_D uh, const REAL_DD grd_uh,
				   REAL_D res),
		  FLAGS f_flag)
{
  FUNCNAME("ellipt_est_d");
  ELLIPT_EST_D_DATA  ell_data[1] = {};
  static WORKSPACE   ws = {0, nil};
  FLAGS              fill_flag;
  const QUAD         *quad;
  int                dim;

  if (!(ell_data->uh = uh))
  {
    MSG("no discrete solution; doing nothing\n");
    return(0.0);
  }
  dim = uh->fe_space->mesh->dim;

  if(dim > 1) /* We need a vertex index to orient walls. */
    get_vertex_admin(uh->fe_space->mesh);

  ell_data->bas_fcts    = uh->fe_space->bas_fcts;

  ell_data->A           = A;
  ell_data->f           = f;
  ell_data->f_flag      = f_flag;

  if (degree < 0) degree = 2*ell_data->bas_fcts->degree;
  quad = get_quadrature(dim, degree);
  if (ell_data->bas_fcts->degree > 1)
    fill_flag = INIT_PHI|INIT_GRD_PHI|INIT_D2_PHI;
  else
    fill_flag = INIT_PHI|INIT_GRD_PHI;

  ell_data->quad_fast = get_quad_fast(ell_data->bas_fcts, quad, fill_flag);
  if(dim > 1)
    ell_data->quad      = get_quadrature(dim-1, degree);

  ell_data->rw_est    = rw_est;
  ell_data->rw_estc   = rw_estc;

  REALLOC_WORKSPACE(&ws, ell_data->bas_fcts->n_bas_fcts*sizeof(REAL_D));
  ell_data->uh_el = (REAL_D *)ws.work;
  
  ell_data->norm  = norm;
  if (C)
  {
    ell_data->C0    = C[0] > 1.e-25 ? SQR(C[0]) : 0.0;
    if(dim > 1) {
      ell_data->C1    = C[1] > 1.e-25 ? SQR(C[1]) : 0.0;
      ell_data->C2    = C[2] > 1.e-25 ? SQR(C[2]) : 0.0;
    }
  }
  else
  {
    if(dim == 1)
      ell_data->C0 = 1.0;
    else
      ell_data->C0 = ell_data->C1 = ell_data->C2 = 1.0;
  }

  if (rw_est) /*---  clear error indicators   -----------------------------*/
    mesh_traverse(uh->fe_space->mesh, -1, CALL_LEAF_EL, 
		  clear_indicator_fct, ell_data);

  ell_data->est_sum = ell_data->est_max = 0.0;
  if(dim == 1)
    fill_flag = FILL_COORDS|CALL_LEAF_EL;
  else
    fill_flag = FILL_NEIGH|FILL_COORDS|FILL_OPP_COORDS|FILL_BOUND|CALL_LEAF_EL;

  mesh_traverse(uh->fe_space->mesh, -1, fill_flag, ellipt_est_fct, ell_data);

  ell_data->est_sum = sqrt(ell_data->est_sum);
  if (adapt)
  {
    adapt->err_sum = ell_data->est_sum;
    adapt->err_max = ell_data->est_max;
  }

  return(ell_data->est_sum);
}

/*--------------------------------------------------------------------------*/
/*                                                                          */
/* error estimator for (nonlinear) heat equation:                           */
/*              u_t - div A grad u + f(x,t,u,grad u) = 0                    */
/*                                                                          */
/* eta_h = C[0]*||h^2 ((U - Uold)/tau - div A grad U + f(x,t,U,grad U))||   */
/*           + C[1]*|||h^1.5 [A grad U]|||                                  */
/* eta_c = C[2]*||(Uold - Uold_coarse)/tau||                                */
/* eta_t = C[3]*||U - Uold||                                                */
/*                                                                          */
/* heat_est() return value is the TIME DISCRETIZATION ESTIMATE, eta_t       */
/*                                                                          */
/*--------------------------------------------------------------------------*/

typedef struct heat_est_d_data HEAT_EST_D_DATA;
struct heat_est_d_data
{
  const DOF_REAL_D_VEC *uh, *uh_old;
  const BAS_FCTS     *bas_fcts;

  const REAL_DD     (*A)[DIM_OF_WORLD];

  const REAL        *(*f)(const EL_INFO *, const QUAD *, int iq,
			  REAL time,
			  const REAL_D uh,
			  const REAL_DD grd_uh,
			  REAL_D res);

  const QUAD_FAST  *quad_fast;      /*--  element integration  -------------*/
  const QUAD       *quad;           /*--  face integration     -------------*/

  REAL             *(*rw_est)(EL *);
  REAL             *(*rw_estc)(EL *);

  REAL_D           *uh_el;          /*--  vector for storing uh on el  -----*/
  REAL_D           *uh_old_el;      /*--  vector for storing uh on el  -----*/
  REAL_D           *uh_qp;          /*--  vector for storing uh at quadpts -*/
  REAL_D           *uh_old_qp;      /*--  vector for storing uh at quadpts -*/

  REAL             time, timestep;

  REAL             C0, C1, C2, C3;  /*--  interior,jump,coarsen,time coefs -*/
  REAL             est_sum;
  REAL             est_max;
  REAL             est_t_sum;
  FLAGS            f_flag;
};


/*--------------------------------------------------------------------------*/
/* element residual:  C0*h_S^2*|| U_t -div A nabla u^h + r ||_L^2(S)        */
/*--------------------------------------------------------------------------*/
/* time residual:  C3*|| U - Uold ||_L^2(S)                                 */
/*--------------------------------------------------------------------------*/

static REAL heat_el_res2(const EL_INFO *el_info, const REAL_D *Lambda,
			 REAL det, HEAT_EST_D_DATA *data)
{
  const REAL_DD   (*D2uhqp)[DIM_OF_WORLD];
  int             dim = el_info->mesh->dim;
  REAL            val, h2 = h2_from_det(dim, det);
  const REAL      *uh_qpi = nil;
  REAL_D          riq;
  int             iq, i, j, mu, nu;
  const REAL_D    *uh_qp = nil, *uh_old_qp = nil, *grd_uh_qpi = nil;
  const REAL_DD   *grd_uh_qp = nil;
  const QUAD_FAST *quad_fast = data->quad_fast;
  const QUAD      *quad = quad_fast->quad;

  uh_old_qp = uh_d_at_qp(quad_fast,
			 (const REAL_D *)data->uh_old_el, data->uh_old_qp);
  uh_qp     = uh_d_at_qp(quad_fast,
			 (const REAL_D *)data->uh_el, data->uh_qp);

  if (data->C3) {
    for (val = iq = 0; iq < quad->n_points; iq++)
    {
      AXPBY_DOW(1.0, uh_qp[iq], -1.0, uh_old_qp[iq], riq);
      val += quad->w[iq]*NRM2_DOW(riq);
    }
    data->est_t_sum += data->C3*det*val;
  }

  if (!(data->C0)) return(0.0);


  if (data->quad_fast->bas_fcts->degree > 1)
    D2uhqp = D2_uh_d_at_qp(data->quad_fast, Lambda,
			   (const REAL_D *)data->uh_el, nil);
  else
    D2uhqp = nil;

  if (data->f)
  {
    if (!(data->f_flag & INIT_UH))
      uh_qpi = nil;

    if (data->f_flag & INIT_GRD_UH)
      grd_uh_qp = grd_uh_d_at_qp(quad_fast, Lambda,
				 (const REAL_D *)data->uh_el, nil);
    else
      grd_uh_qpi = nil;
  }

  for (val = iq = 0; iq < quad->n_points; iq++)
  {
    AXPBY_DOW(1.0, uh_qp[iq], -1.0, uh_old_qp[iq], riq);
    AX_DOW(data->timestep, riq);

    if (data->f)
    {
      if (data->f_flag & INIT_UH)     uh_qpi     = uh_qp[iq];
      if (data->f_flag & INIT_GRD_UH) grd_uh_qpi = grd_uh_qp[iq];
      AXPY_DOW(1.0, data->f(el_info, quad, iq, data->time,
			    uh_qpi,  grd_uh_qpi, nil),
	       riq);
    }

    if (D2uhqp) {
      for(mu = 0; mu < DIM_OF_WORLD; mu++)
	for (i = 0; i < DIM_OF_WORLD; i++)
	  for(nu = 0; nu < DIM_OF_WORLD; nu++)
	    for (j = 0; j < DIM_OF_WORLD; j++)
	      riq[mu] -= data->A[i][j][mu][nu] * D2uhqp[iq][nu][i][j];
    }
    val += quad->w[iq]*NRM2_DOW(riq);
  }

  val = data->C0*h2*h2*det*val;

  return(val);
}


/*--------------------------------------------------------------------------*/
/*  face residuals:  C1*h_Gamma*||[A(u_h)]||_L^2(Gamma)^2                   */
/*  Since det_S = det_Gamma*h_Gamma we use for det_Gamma*h_Gamma the term   */
/*  0.5(det_S + det_S')                                                     */
/*--------------------------------------------------------------------------*/

static REAL heat_jump_res2(const EL_INFO *el_info, int face,
			   const REAL_D *Lambda,
			   REAL det, HEAT_EST_D_DATA *data)
{
  EL_INFO        neigh_info[1];
  int		 dim = el_info->mesh->dim;
  int            face_ind_el[dim], face_ind_neigh[dim];
  EL             *neigh = el_info->neigh[face];
  int            opp_v  = el_info->opp_vertex[face];
  int            i, j, mu, nu, i1, i2, iq;
  REAL_DD        jump, grd_uh_el, grd_uh_neigh;
  REAL_D         Lambda_neigh[N_LAMBDA];
  const REAL_D   *uh_neigh;
  REAL           det_neigh = 0.0, lambda[N_LAMBDA], val = 0.0;
  const BAS_FCTS *bas_fcts = data->uh->fe_space->bas_fcts;
  const QUAD     *quad = data->quad;

/*--------------------------------------------------------------------------*/
/* orient the edge/face => same quadrature nodes from both sides!           */
/*--------------------------------------------------------------------------*/

  sort_wall_indices(dim, el_info->el, face, face_ind_el);
  sort_wall_indices(dim, neigh, opp_v, face_ind_neigh);

  neigh_info->mesh = el_info->mesh;
  neigh_info->el = neigh;
  neigh_info->fill_flag = FILL_COORDS;

  for (j = 0; j < DIM_OF_WORLD; j++)
    neigh_info->coord[opp_v][j] = el_info->opp_coord[face][j];

  for (i = 0; i < dim; i++)
  {
    i1 = face_ind_el[i];
    i2 = face_ind_neigh[i];
    for (j = 0; j < DIM_OF_WORLD; j++)
      neigh_info->coord[i2][j] = el_info->coord[i1][j];
  }

  switch(dim) {
  case 1:
    det_neigh = el_grd_lambda_1d(neigh_info, Lambda_neigh);
    break;
#if DIM_OF_WORLD > 1
  case 2:
    det_neigh = el_grd_lambda_2d(neigh_info, Lambda_neigh);
    break;
#if DIM_OF_WORLD > 2
  case 3:
    det_neigh = el_grd_lambda_3d(neigh_info, Lambda_neigh);
#endif
#endif
  }

  uh_neigh = bas_fcts->get_real_d_vec(neigh, data->uh, nil);

/*--------------------------------------------------------------------------*/
/*  now eval the jump at all quadrature nodes                               */
/*--------------------------------------------------------------------------*/

  for (val = iq = 0; iq < quad->n_points; iq++)
  {
    lambda[face] = 0.0;
    for (i = 0; i < dim; i++)
      lambda[face_ind_el[i]] = quad->lambda[iq][i];
    eval_grd_uh_d(lambda, (const REAL_D *)Lambda,
		  (const REAL_D *)data->uh_el, bas_fcts, grd_uh_el);

    lambda[opp_v] = 0.0;
    for (i = 0; i < dim; i++)
      lambda[face_ind_neigh[i]] = quad->lambda[iq][i];
    eval_grd_uh_d(lambda, (const REAL_D *)Lambda_neigh, uh_neigh, bas_fcts, 
		  grd_uh_neigh);

    for (mu = 0; mu < DIM_OF_WORLD; mu++)
      for (i = 0; i < DIM_OF_WORLD; i++)
	for (jump[mu][i] = nu = 0; nu < DIM_OF_WORLD; nu++)
	  for (j = 0; j < DIM_OF_WORLD; j++)
	    jump[mu][i] += data->A[i][j][mu][nu]
	      * (grd_uh_el[nu][j] - grd_uh_neigh[nu][j]);

    val += quad->w[iq]*MSCP_DOW(jump,jump);
  }

  det = 0.5*(det + det_neigh);

  return(data->C1*h2_from_det(dim, det)*det*val);
}

/*--------------------------------------------------------------------------*/
/*  neuman residual:  C1*h_Gamma*||A(u_h).normal||_L^2(Gamma)^2             */
/*  Since det_S = det_Gamma*h_Gamma we use for det_Gamma*h_Gamma the term   */
/*  det_S                                                                   */
/*--------------------------------------------------------------------------*/

static REAL heat_neumann_res2(const EL_INFO *el_info, int face, 
			      const REAL_D *Lambda, REAL det, 
			      HEAT_EST_D_DATA *data)
{
  int            i, j, mu, nu, iq;
  int            dim = el_info->mesh->dim;
  REAL           lambda[N_LAMBDA];
  REAL_D         n_A_grd_uh;
  REAL           val;
  REAL_D         normal;
  REAL_DD        grd_uh, A_grd_uh;
  const QUAD     *quad = data->quad;

  switch(dim) {
  case 1:
    get_wall_normal_1d(el_info, face, normal);
    break;
#if DIM_OF_WORLD > 1
  case 2:
    get_wall_normal_2d(el_info, face, normal);
    break;
#if DIM_OF_WORLD > 2
  case 3:
    get_wall_normal_3d(el_info, face, normal);
#endif
#endif
  }

  for (val = iq = 0; iq < quad->n_points; iq++)
  {
    for (i = 0; i < face; i++)
      lambda[i] = quad->lambda[iq][i];
    lambda[face] = 0.0;
    for (i = face+1; i < N_LAMBDA; i++)
      lambda[i] = quad->lambda[iq][i-1];

    eval_grd_uh_d(lambda, (const REAL_D *)Lambda,
		  (const REAL_D *)data->uh_el, data->bas_fcts, grd_uh);

    for (mu = 0; mu < DIM_OF_WORLD; mu++)
      for (i = 0; i < DIM_OF_WORLD; i++)
	for (A_grd_uh[mu][i] = nu = 0; nu < DIM_OF_WORLD; nu++)
	  for (j = 0; j < DIM_OF_WORLD; j++)
	    A_grd_uh[mu][i] += data->A[i][j][mu][nu]*grd_uh[nu][j];

    SET_DOW(0.0, n_A_grd_uh);
    MV_DOW(A_grd_uh, normal, n_A_grd_uh);
    val += quad->w[iq]*NRM2_DOW(n_A_grd_uh);
  }

  return(data->C1*h2_from_det(dim, det)*det*val);
}


/*--------------------------------------------------------------------------*/

#define DATA ((HEAT_EST_D_DATA *)data)

static void heat_clear_indicator_fct(const EL_INFO *el_info, void *data)
{
  el_info->el->mark = 1;
  if (DATA->rw_est)  *(DATA->rw_est(el_info->el)) = 0.0;
  if (DATA->rw_estc) *(DATA->rw_estc(el_info->el)) = 0.0;
  return;
}

static void heat_est_fct(const EL_INFO *el_info, void *data)
{
  EL           *el = el_info->el;
  REAL          det = 0.0, est_el;
  REAL_D        Lambda[N_LAMBDA];
  int           face;
  int           dim = el_info->mesh->dim;
  const S_CHAR *bound = 
    ((dim == 3) ? el_info->face_bound : el_info->edge_bound);
  EL           *neigh;

/*--- if rw_est, then there might already be contributions from jumps ------*/
  est_el = DATA->rw_est ? *(DATA->rw_est(el)) : 0.0;

  DATA->bas_fcts->get_real_d_vec(el, DATA->uh, DATA->uh_el);
  DATA->bas_fcts->get_real_d_vec(el, DATA->uh_old, DATA->uh_old_el);

  switch(dim) {
  case 1:
    det = el_grd_lambda_1d(el_info, Lambda);
    break;
#if DIM_OF_WORLD > 1
  case 2:
    det = el_grd_lambda_2d(el_info, Lambda);
    break;
#if DIM_OF_WORLD > 2
  case 3:
    det = el_grd_lambda_3d(el_info, Lambda);
#endif
#endif
  }

/*---  element and time residual  ------------------------------------------*/
  if (DATA->C0 || DATA->C3)
    est_el += heat_el_res2(el_info, (const REAL_D *)Lambda, det, DATA);

/*---  face residuals  -----------------------------------------------------*/
  if (dim > 1 && DATA->C1)
  {
    for (face = 0; face < N_NEIGH(dim); face++)
    {
      if ((neigh = el_info->neigh[face]))
      {
/*--------------------------------------------------------------------------*/
/*  if rw_est is nil, compute jump for both neighbouring elements           */
/*                    only this allows correct computation of est_max       */
/*  if rw_est is not nil, compute jump only once for each edge/face         */
/*                    if neigh->mark: estimate not computed on neighbour!   */
/*  contribution for the element and for neighbour: 0.5*jump!               */
/*--------------------------------------------------------------------------*/
	if (!DATA->rw_est  ||  neigh->mark)
	{
	  REAL est = heat_jump_res2(el_info, face, 
				    (const REAL_D *)Lambda, det, DATA);

	  est_el += est;
/*--  if rw_est, add neighbour contribution to neigbour indicator  ---------*/
	  if (DATA->rw_est) *(DATA->rw_est(neigh)) += est;
	}
      }
      else if (IS_NEUMANN(bound[face]))
      {
	est_el += heat_neumann_res2(el_info, face, 
				    (const REAL_D *)Lambda, det, DATA);
      }
    }
  }

#if 0
/*--  if rw_estc, calculate coarsening error estimate  ---------------------*/
  if (DATA->rw_estc && DATA->C2)
  {
    *(DATA->rw_estc(el)) = heat_estc_el(el_info, det, DATA);
  }
#endif

/*--  if rw_est, write indicator to element  -------------------------------*/
  if (DATA->rw_est) *(DATA->rw_est(el)) = est_el;

  DATA->est_sum += est_el;
  DATA->est_max  = MAX(DATA->est_max, est_el);

  el_info->el->mark = 0; /*--- all contributions are computed!  ------------*/

  return;
}

#undef DATA

/*--------------------------------------------------------------------------*/

REAL heat_est_d(const DOF_REAL_D_VEC *uh, ADAPT_INSTAT *adapt,
		REAL *(rw_est)(EL *), REAL *(rw_estc)(EL *),
		int degree, REAL C[4], 
		const DOF_REAL_D_VEC *uh_old,
		const REAL_DD A[DIM_OF_WORLD][DIM_OF_WORLD],
		const REAL *(*f)(const EL_INFO *, const QUAD *, int iq, REAL t,
				 const REAL_D u, const REAL_DD grd_u, 
				 REAL_D res),
		FLAGS f_flag)
{
  FUNCNAME("heat_est_d");
  HEAT_EST_D_DATA    heat_data[1] = {};
  static WORKSPACE   ws = {0, nil};
  FLAGS              fill_flag;
  const QUAD         *quad;
  int                dim;

  if (!(heat_data->uh = uh))
  {
    MSG("no discrete solution; doing nothing\n");
    return(0.0);
  }
  dim = uh->fe_space->mesh->dim;

  if (!(heat_data->uh_old = uh_old))
  {
    MSG("no discrete solution from previous timestep; doing nothing\n");
    /* here, initial error could be calculated... */
    return(0.0);
  }

  if(dim > 1) /* We need a vertex index to orient walls. */
    get_vertex_admin(uh->fe_space->mesh);

  heat_data->bas_fcts = uh->fe_space->bas_fcts;
  heat_data->A        = A;

  heat_data->f        = f;
  heat_data->f_flag   = f_flag;

  if (degree < 0) degree = 2*heat_data->bas_fcts->degree;
  quad = get_quadrature(dim, degree);
  if (heat_data->bas_fcts->degree > 1)
    fill_flag = INIT_PHI|INIT_GRD_PHI|INIT_D2_PHI;
  else
    fill_flag = INIT_PHI|INIT_GRD_PHI;

  heat_data->quad_fast = get_quad_fast(heat_data->bas_fcts, quad, fill_flag);
  if(dim > 1)
    heat_data->quad      = get_quadrature(dim-1, degree);

  heat_data->rw_est    = rw_est;
  heat_data->rw_estc   = rw_estc;

  REALLOC_WORKSPACE(&ws, 2*(heat_data->bas_fcts->n_bas_fcts
			    + heat_data->quad_fast->n_points)*sizeof(REAL_D));
  heat_data->uh_el     = (REAL_D *)ws.work;
  heat_data->uh_old_el = heat_data->uh_el + heat_data->bas_fcts->n_bas_fcts;
  heat_data->uh_qp     =
    heat_data->uh_old_el + heat_data->bas_fcts->n_bas_fcts;
  heat_data->uh_old_qp = heat_data->uh_qp + heat_data->quad_fast->n_points;

#if 0
  heat_data->uh_qp     = MEM_ALLOC(6, REAL);
  heat_data->uh_old_qp = MEM_ALLOC(6, REAL);
#endif

  
  if (C)
  {
    heat_data->C0    = C[0] > 1.e-25 ? SQR(C[0]) : 0.0;
    heat_data->C1    = C[1] > 1.e-25 ? SQR(C[1]) : 0.0;
    heat_data->C2    = C[2] > 1.e-25 ? SQR(C[2]) : 0.0;
    heat_data->C3    = C[3] > 1.e-25 ? SQR(C[3]) : 0.0;
  }
  else
  {
    heat_data->C0 = heat_data->C1 = heat_data->C2 = heat_data->C3 = 1.0;
  }

  heat_data->time = adapt->time;
  heat_data->timestep = adapt->timestep;


  if (rw_est)  /*---  clear error indicators   -----------------------------*/
    mesh_traverse(uh->fe_space->mesh, -1, CALL_LEAF_EL,
		  heat_clear_indicator_fct, heat_data);

  heat_data->est_sum = heat_data->est_max = heat_data->est_t_sum = 0.0;
  if(dim == 1)
    fill_flag = FILL_COORDS|CALL_LEAF_EL;
  else
    fill_flag = FILL_NEIGH|FILL_COORDS|FILL_OPP_COORDS|FILL_BOUND|CALL_LEAF_EL;

  mesh_traverse(uh->fe_space->mesh, -1, fill_flag, heat_est_fct, heat_data);

  heat_data->est_sum   = sqrt(heat_data->est_sum);
  heat_data->est_t_sum = sqrt(heat_data->est_t_sum);
  if (adapt)
  {
    adapt->adapt_space->err_sum = heat_data->est_sum;
    adapt->adapt_space->err_max = heat_data->est_max;
/*     adapt->err_t = heat_data->est_t_sum; */
  }

  return(heat_data->est_t_sum);
}
