/**************************************************************************
*   Copyright (C) 2007-2010 by Thomas Thelliez aka jblud                  *
*   Contact : <admin.kontrol@gmail.com>                                   *
*                                                                         *
* This program is free software; you can redistribute it and/or           *
* modify it either under the terms of the GNU Lesser General Public       *
* License version 3 as published by the Free Software Foundation          *
* (the "LGPL") or, at your option, any later version.                     *
* If you do not alter this notice, a recipient may use your version       *
* of this file under the LGPL.                                            *
*                                                                         *
* You should have received a copy of the LGPL along with this library     *
* in the file COPYING; if not, write to the Free Software                 *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA *
*                                                                         *
* This software is distributed on an "AS IS" basis, WITHOUT WARRANTY      *
* OF ANY KIND, either express or implied. See the LGPL for                *
* the specific language governing rights and limitations.                 *
**************************************************************************/

#include "auth_session.h"

auth_session::auth_session(QTcpSocket *socket,
                           int client_nbr,
                           QString server_password,
                           QObject *parent)
                               : tcp_socket(socket),
                               nbr(client_nbr),
                               password(server_password),
                               QObject(parent)
{
    OpenSSL_add_all_algorithms();
    ERR_load_crypto_strings();
    receiving = false;
    set_socket(tcp_socket);
    QObject::connect(tcp_socket,
                     SIGNAL(disconnected()), this,
                     SLOT(disconnected_client()));
    QObject::connect(tcp_socket, SIGNAL(error(QAbstractSocket::SocketError)),
                     this, SLOT(socket_error(QAbstractSocket::SocketError)));
    auth_status = TOKEN_WELCOME;
    xml = welcome_token_exchange();
    token = welcome_token_exchange_parser(xml);
    strcat(token, password.toUtf8().data());
}

void auth_session::set_socket(QTcpSocket* qSocket)
{
    tcp_socket = qSocket;
}

QTcpSocket* auth_session::get_socket()
{
    return tcp_socket;
}

void auth_session::read_results()
{
    if (auth_status == AUTH_SUCCESS) {
        // If server received something then decrypt
        // and send message to app with signal.
        QString data = get_data_from_socket();
        if (!data.isEmpty()) {
            QString decrypted = auth_aes_decrypt(data);
            emit received_stream(decrypted, nbr);
        }
    } else {
        handle_crypted_auth();
    }
}

QString auth_session::get_data_from_socket()
{
    if (!receiving) {
        QString sLine = QString("");
        if (tcp_socket->bytesAvailable() > HEADER_SIZE) {
            sLine.append(tcp_socket->read(HEADER_SIZE));
        }
        QStringList list = sLine.split(SIZE_DELIMITER);
        if (list.size() > 1) {
            packet_size = remove_header_padding(list.at(0).toInt());
            current_bytes = new QByteArray;
            receiving = true;
            current_bytes->append(QByteArray(list.at(1).toUtf8()));
            current_size = current_bytes->length();
        } else {send_error(); return "";}
    }
    if (packet_size > current_size) {
        while (tcp_socket->bytesAvailable() > 0) {
            current_bytes->append(tcp_socket->read(10000));
            current_size = current_bytes->length();
            QApplication::processEvents();
            emit update_progress_size(current_size, packet_size);
        }
    }
    if (packet_size == current_size) {
        QString result = QString(current_bytes->data());
        delete current_bytes;
        receiving = false;
        current_bytes = NULL;
        current_size = 0;
        packet_size = 0;
        if (result.contains(ERROR,
                            Qt::CaseInsensitive)) {
            send_error();
        }
        return result;
    }
    return "";
}

void auth_session::write_data_to_socket(QString data)
{
    tcp_socket->write(QString(add_header_padding(data.length()) + QString(SIZE_DELIMITER) + data).toUtf8());
    tcp_socket->flush();
}

void auth_session::disconnected_client()
{
    emit remove_client(nbr);
}

/*
    #define TOKEN_WELCOME 0
    #define RSA_PUBLIC_KEY_EXCHANGE 1
    #define PASSWORD_VALIDATION 2
    #define AES_KEY_RECEPTION 3
    #define AUTH_SUCCESS 4
*/
void auth_session::handle_crypted_auth()
{
    QString result;
    switch( auth_status )
    {
    case TOKEN_WELCOME : {
            // Sending token converting it in b64 format before sending over network.
            write_data_to_socket(QString(xml));
            auth_status = RSA_PUBLIC_KEY_RECEPTION;
            emit emit_tcp_state(TOKEN_WELCOME, nbr);
            break;
        }
    case RSA_PUBLIC_KEY_RECEPTION :
        // decode from base64.
        result = get_data_from_socket();
        if (!result.isEmpty()) {
            if (result.contains(XML_START_ELEMENT, Qt::CaseInsensitive)
                && result.contains(XML_END_ELEMENT, Qt::CaseInsensitive)) {
                // Saving client RSA public key.
                client_pub_key = key_exchange_parser(result.toUtf8().data());
                if (strlen(client_pub_key) == 0)
                    fprintf (stderr, "%s:%d Error transfering key...\n",
                             __FILE__, __LINE__);
                rsa_client_pub = char_array_to_RSA_key(
                        client_pub_key, PUBLIC_KEY, NULL);
                if (rsa_client_pub == NULL) {
                    fprintf (stderr, "%s:%d Invalid RSA Key.\n",
                             __FILE__, __LINE__);
                    send_error();
                    break;
                }
                // Preparing AES (256) key to secure and encrypt messages between workstations.
                int i = 0;
                for (i = 0; i < 32; i++) {
                    char c = (char)(rand() % 26+65);
                    aes_key[i] = c;
                }
                aes_key[32] = '\0';

                char *xml =  symetric_key_exchange(aes_key);
                unsigned char cleartext[strlen(xml)];

                memcpy(cleartext, xml, strlen(xml));

                // Encrypt with the saved RSA public Key
                unsigned char *encrypted =
                        (unsigned char *)malloc(RSA_size(rsa_client_pub));
                if (RSA_public_encrypt (
                        strlen((char *)cleartext),
                        cleartext, encrypted,
                        rsa_client_pub, RSA_PKCS1_PADDING )==-1) {
                    int error = ERR_get_error();
                    fprintf (stderr, "%s %s:%d \n",
                             ERR_lib_error_string( error ),
                             __FILE__, __LINE__);
                    fprintf (stderr, "%s %s:%d \n",
                             ERR_func_error_string( error ),
                             __FILE__, __LINE__);
                    fprintf (stderr, "%s %s:%d \n",
                             ERR_reason_error_string( error ),
                             __FILE__, __LINE__);
                    send_error();
                    break;
                }
                // To prevent encryption error : check
                // if encrypted length is at least as long as the
                // RSA key length.
                while (strlen((char *)encrypted) < RSA_BUFFER_LEN) {
                    // Encrypt with the saved Key.
                    if (RSA_public_encrypt (
                            strlen((char *)cleartext),
                            cleartext, encrypted, rsa_client_pub,
                            RSA_PKCS1_PADDING )==-1) {
                        fprintf (stderr, "%s:%d Encryption failed\n",
                                 __FILE__, __LINE__);
                        send_error();
                        break;
                    }
                }
                // Sending AES key to remote client.
                write_data_to_socket(QString(base64(
                        encrypted, strlen((char*)encrypted))));
                emit emit_tcp_state(RSA_PUBLIC_KEY_RECEPTION, nbr);
                auth_status = PASSWORD_VALIDATION;
            } else {
                fprintf (stderr, "%s:%d Key exchange bad format.\n",
                         __FILE__, __LINE__);
                send_error();
            }
        }
        break;
        case PASSWORD_VALIDATION : {
                result = get_data_from_socket();
                if (!result.isEmpty() && !result.contains(ERROR,
                                                          Qt::CaseInsensitive)) {
                    // Receiving password encrypted with
                    // AES symetric algorithm.
                    QString decrypted = auth_aes_decrypt(result);
                    if (decrypted.contains(XML_START_ELEMENT,
                                           Qt::CaseInsensitive)
                        && decrypted.contains(
                                XML_END_ELEMENT, Qt::CaseInsensitive)) {
                        b64_hashed_password_from_client =
                                password_exchange_parser(
                                        decrypted.toUtf8().data());

                        // SHA1 hash of token+password.
                        unsigned char sha1sum[SHA_DIGEST_LENGTH];

                        SHA1((unsigned char*)token, strlen(token), sha1sum);
                        b64_hashed_password = base64(
                                (unsigned char*)sha1sum, sizeof(sha1sum));

                        if (QString::compare(b64_hashed_password,
                                             b64_hashed_password_from_client,
                                             Qt::CaseInsensitive) == 0) {
                            emit emit_tcp_state(PASSWORD_VALIDATION, nbr);
                            auth_status = AUTH_SUCCESS;
                            write_data_to_socket(auth_aes_encrypt(
                                    SUCCESS));
                            emit auth_suceeded(nbr);
                        } else {
                            write_data_to_socket(auth_aes_encrypt(
                                    FAILED));
                            emit emit_error(nbr);
                            tcp_socket->close();
                            tcp_socket->disconnect();
                        }
                    } else {
                        send_error();
                        fprintf (stderr,
                                 "%s:%d Bad AES encryption, password verification : XML bad format.\n",
                                 __FILE__, __LINE__);
                    }
                }
                break;
            }
    }
}

void auth_session::send_error()
{
    write_data_to_socket(QString(ERROR));
    emit emit_error(nbr);
    tcp_socket->close();
    tcp_socket->disconnect();
}

QString auth_session::auth_aes_encrypt(char* message)
{
    // AES Encryption
    std::string key_encrypt((char*) aes_key);
    std::string std_xml((char*) message);
    string str = END_DELIMITER;
    std_xml.append(str);

    std::string encrypted(aes_encrypt(encode(std_xml),
                                      key_encrypt));
    return QString::fromStdString(encrypted);

}

QString auth_session::auth_aes_decrypt(QString message)
{
    // AES Decryption
    std::string key_decrypt((char*) aes_key);

    std::string decrypted(aes_decrypt(
            message.toStdString(), key_decrypt));

    QString message_final = QString(decode(decrypted).c_str());
    message_final = message_final.mid(0, message_final.indexOf(END));
    return message_final;
}

void auth_session::socket_error(QAbstractSocket::SocketError socketError)
{
    emit remove_client(nbr);
}
