------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2000-2006 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-statements.ads,v 1.15 2007/04/25 10:08:47 vagul Exp $

with OCI.Thick.Connections;
with Ada.Strings.Unbounded;
package OCI.Thick.Statements is

   use Connections;
   use Ada.Strings.Unbounded;

   type Statement is new Handle_Reference with private;

   type VStrings is array (Positive range <>) of Unbounded_String;

   type Statement_Type is
     (Stmt_Unknown,
      Stmt_Select,
      Stmt_Update,
      Stmt_Delete,
      Stmt_Insert,
      Stmt_Create,
      Stmt_Drop,
      Stmt_Alter,
      Stmt_Begin,
      Stmt_Declare);

   function Prepare
     (Connect : in Connection;
      Code    : in String)
      return  Statement;

   -------------------------------------------------
   -- Undepended from Connection statement stile. --
   -------------------------------------------------

   function Prepare (Code : in String) return Statement;

   procedure Execute (Connect : in Connection; Stmt : in out Statement);

   procedure Describe (Connect : in Connection; Stmt : in out Statement);

   -----------------------------------------------
   -- Depended from Connection statement stile. --
   -- Like ODBC                                 --
   -----------------------------------------------

   procedure Execute (Stmt : in out Statement);

   procedure Execute (Stmt : in out Statement; Success : out Boolean);

   procedure Execute_And_Commit
     (Connect : in     Connection;
      Stmt    : in out Statement);

   procedure Execute_And_Commit (Stmt : in out Statement);

   procedure Execute_And_Commit
     (Stmt    : in out Statement;
      Success :    out Boolean);

   procedure Describe (Stmt : in out Statement);

   procedure Describe (Stmt : in out Statement; Success : out Boolean);

   function Execute
     (Connect : in Connection;
      Code    : in String)
      return  Statement;

   procedure Execute (Connect : in Connection; Code : in String);

   function Type_Of_Statement (Stmt : in Statement) return Statement_Type;

   function Described (Stmt : in Statement) return Boolean;

   function Executed (Stmt : in Statement) return Boolean;

   function Is_Executing (Stmt : in Statement) return Boolean;

   procedure Set_Connection
     (Stmt    : in out Statement;
      Connect : in     Connection);

   function Get_Connection (Stmt : in Statement) return Connection;

   function Fetch (Stmt : in Statement) return Boolean;

   procedure Cancel (Stmt : in Statement);

   function Parse_Error_Offset (Stmt : in Statement) return Natural;

   function Number_Of_Columns (Stmt : in Statement) return Natural;

   function Rows_Processed (Stmt : in Statement) return Natural;
   --  Returns the number of rows processed so far.

   function Bind_Names (Stmt : in Statement) return VStrings;

   procedure Set_Blocking (Item : in out Statement; Mode : in Boolean);
   --  Set the statement connection to the spetsific blocking mode.

   pragma Inline (Executed, Get_Connection);

private

   for Statement_Type use
     (Stmt_Unknown => 0, --  Undocumented OCI Stmt type.
      Stmt_Select  => OCI_STMT_SELECT,
      Stmt_Update  => OCI_STMT_UPDATE,
      Stmt_Delete  => OCI_STMT_DELETE,
      Stmt_Insert  => OCI_STMT_INSERT,
      Stmt_Create  => OCI_STMT_CREATE,
      Stmt_Drop    => OCI_STMT_DROP,
      Stmt_Alter   => OCI_STMT_ALTER,
      Stmt_Begin   => OCI_STMT_BEGIN,
      Stmt_Declare => OCI_STMT_DECLARE);
   for Statement_Type'Size use Ub2'Size;

   type Statement is new Handle_Reference with record
      Connect : Connection;
      Executed, Described, Executing : Boolean := False;
   end record;

   procedure Destroy (Object : in out Statement);

end OCI.Thick.Statements;
