/* Ergo, version 3.3, a program for linear scaling electronic structure
 * calculations.
 * Copyright (C) 2013 Elias Rudberg, Emanuel H. Rubensson, and Pawel Salek.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Primary academic reference:
 * Kohn−Sham Density Functional Theory Electronic Structure Calculations 
 * with Linearly Scaling Computational Time and Memory Usage,
 * Elias Rudberg, Emanuel H. Rubensson, and Pawel Salek,
 * J. Chem. Theory Comput. 7, 340 (2011),
 * <http://dx.doi.org/10.1021/ct100611z>
 * 
 * For further information about Ergo, see <http://www.ergoscf.org>.
 */
#ifdef USE_CHUNKS_AND_TASKS

#include <cstring>
#include <cstdio>
#include "DistrBasisSetChunk.h"

CHT_CHUNK_TYPE_IMPLEMENTATION((DistrBasisSetChunk));

template<typename T>
size_t write_vector_to_buffer(char* destPtr, const std::vector<T> & vec) {
  int n = vec.size();
  char* p = destPtr;
  memcpy(p, &n, sizeof(int));
  p += sizeof(int);
  memcpy(p, &vec[0], n*sizeof(T));
  p += n*sizeof(T);
  return p - destPtr;
}

template<typename T>
size_t read_vector_from_buffer(const char* srcPtr, std::vector<T> & vec) {
  const char* p = srcPtr;
  int n;
  memcpy(&n, p, sizeof(int));
  p += sizeof(int);
  vec.resize(n);
  memcpy(&vec[0], p, n*sizeof(T));
  p += n*sizeof(T);
  return p - srcPtr;
}

template<typename T>
size_t get_vector_size(const std::vector<T> & vec) {
  return sizeof(int) + vec.size() * sizeof(T);
}

void DistrBasisSetChunk::writeToBuffer ( char * dataBuffer, 
					 size_t const bufferSize ) const {
  // First store one int number: noOfBasisFuncs.
  char* p = dataBuffer;
  if(bufferSize < getSize())
    throw std::runtime_error("Error: bufferSize too small.");
  memcpy(p, &noOfBasisFuncs, sizeof(int));
  p += sizeof(int);
  // Now the basisInfo struct.
  size_t sizeSoFar = p - dataBuffer;
  basisInfo.write_to_buffer(p, bufferSize - sizeSoFar);
  p += basisInfo.get_size();
  // basisFuncIndexList
  p += write_vector_to_buffer(p, basisFuncIndexList);
  // basisFuncExtentList
  p += write_vector_to_buffer(p, basisFuncExtentList);
  // maxExtent
  memcpy(p, &maxExtent, sizeof(ergo_real));
  p += sizeof(ergo_real);
  // Now boundingBoxForCenters.
  memcpy(p, &boundingBoxForCenters, sizeof(BoxStruct));
  p += sizeof(BoxStruct);
  // Now cid_child_chunks.
  memcpy(p, cid_child_chunks, 2*sizeof(cht::ChunkID));
  p += 2*sizeof(cht::ChunkID);
  // DONE!
}

size_t DistrBasisSetChunk::getSize() const {
  return sizeof(int) 
    + get_vector_size(basisFuncIndexList)
    + get_vector_size(basisFuncExtentList)
    + sizeof(ergo_real)
    + basisInfo.get_size()
    + sizeof(BoxStruct)
    + 2*sizeof(cht::ChunkID);
}

void DistrBasisSetChunk::assignFromBuffer ( char const * dataBuffer, 
					    size_t const bufferSize) {
  // First get noOfBasisFuncs.
  const char* p = dataBuffer;
  if(bufferSize < 2*sizeof(int) + sizeof(BoxStruct) + 2*sizeof(cht::ChunkID))
    throw std::runtime_error("Error: bufferSize too small.");
  memcpy(&noOfBasisFuncs, p, sizeof(int));
  p += sizeof(int);
  // Now get the basisInfo struct.
  size_t sizeSoFar = p - dataBuffer;
  basisInfo.assign_from_buffer(p, bufferSize - sizeSoFar);
  p += basisInfo.get_size();
  // basisFuncIndexList
  p += read_vector_from_buffer(p, basisFuncIndexList);
  // basisFuncExtentList
  p += read_vector_from_buffer(p, basisFuncExtentList);
  // maxExtent
  memcpy(&maxExtent, p, sizeof(ergo_real));
  p += sizeof(ergo_real);
  // Now get boundingBoxForCenters.
  memcpy(&boundingBoxForCenters, p, sizeof(BoxStruct));
  p += sizeof(BoxStruct);
  // Now get cid_child_chunks.
  memcpy(cid_child_chunks, p, 2*sizeof(cht::ChunkID));
  p += 2*sizeof(cht::ChunkID);
  // Ok, now we have exctracted all needed info.
  if((p-dataBuffer) > bufferSize)
    throw std::runtime_error("Error: (p > bufferSize).");
  // DONE!
}

size_t DistrBasisSetChunk::memoryUsage() const {
  return getSize();
}

void DistrBasisSetChunk::getChildChunks(std::list<cht::ChunkID> & childChunkIDs) const {
  // Check if this is lowest level in hierarchy.
  if(noOfBasisFuncs == basisInfo.noOfBasisFuncs) {
    // This is lowest level. No child chunks to delete in this case.
  }
  else {
    childChunkIDs.push_back(cid_child_chunks[0]);
    childChunkIDs.push_back(cid_child_chunks[1]);
  }
}


#endif
