/**
 * @fileoverview Test cases for the numeric tick-generating functions.
 * These were generated by adding logging code to the old ticker functions. The
 * tests serve to track existing behavior should it change in the future.
 *
 * @author danvdk@gmail.com (Dan Vanderkam)
 */

import Dygraph from '../../src/dygraph';
import * as DygraphTickers from '../../src/dygraph-tickers';
import DEFAULT_ATTRS from '../../src/dygraph-default-attrs';

describe("numeric-ticker-tests", function() {

cleanupAfterEach();

var createOptionsViewForAxis = function(axis, dict) {
  return function (x) {
    if (dict && dict.hasOwnProperty(x)) {
      return dict[x];
    }
    if (DEFAULT_ATTRS.axes[axis].hasOwnProperty(x)) {
      return DEFAULT_ATTRS.axes[axis][x];
    }
    if (DEFAULT_ATTRS.hasOwnProperty(x)) {
      return DEFAULT_ATTRS[x];
    }
    if (x == 'axisLabelFormatter') return null;
    throw "mysterious " + axis + "-axis option: " + x;
  };
};

it('testBasicNumericTicker', function() {
  var opts = {"logscale":null,"labelsKMG2":false,"labelsKMB":false};
  var options = createOptionsViewForAxis('y', opts);

  var ticks = DygraphTickers.numericTicks(-0.4, 4.4, 320, options);
  var expected_ticks = [
    {"v":-0.5,"label":"-0.5"},
    {"v":0,"label":"0"},
    {"v":0.5,"label":"0.5"},
    {"v":1,"label":"1"},
    {"v":1.5,"label":"1.5"},
    {"v":2,"label":"2"},
    {"v":2.5,"label":"2.5"},
    {"v":3,"label":"3"},
    {"v":3.5,"label":"3.5"},
    {"v":4,"label":"4"},
    {"v":4.5,"label":"4.5"}];
  assert.deepEqual(expected_ticks, ticks);

  ticks = DygraphTickers.numericTicks(1, 84, 540, options);
  var expected_ticks = [
    {"v":0,"label":"0"},
    {"v":5,"label":"5"},
    {"v":10,"label":"10"},
    {"v":15,"label":"15"},
    {"v":20,"label":"20"},
    {"v":25,"label":"25"},
    {"v":30,"label":"30"},
    {"v":35,"label":"35"},
    {"v":40,"label":"40"},
    {"v":45,"label":"45"},
    {"v":50,"label":"50"},
    {"v":55,"label":"55"},
    {"v":60,"label":"60"},
    {"v":65,"label":"65"},
    {"v":70,"label":"70"},
    {"v":75,"label":"75"},
    {"v":80,"label":"80"},
    {"v":85,"label":"85"}
  ];
  assert.deepEqual(expected_ticks, ticks);
});

/*
it('testAllNumericTickers', function() {
  assert.deepEqual([{"v":-0.5,"label":"-0.5"},{"v":0,"label":"0"},{"v":0.5,"label":"0.5"},{"v":1,"label":"1"},{"v":1.5,"label":"1.5"},{"v":2,"label":"2"},{"v":2.5,"label":"2.5"},{"v":3,"label":"3"},{"v":3.5,"label":"3.5"},{"v":4,"label":"4"}], Dygraph.numericTicks(-0.4, 4.4, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":-1.5,"label":"-1.5"},{"v":-1,"label":"-1"},{"v":-0.5,"label":"-0.5"},{"v":0,"label":"0"},{"v":0.5,"label":"0.5"},{"v":1,"label":"1"}], Dygraph.numericTicks(-1.4157430939856124, 1.4157430939856124, 400, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":-10,"label":"-10"},{"v":-8,"label":"-8"},{"v":-6,"label":"-6"},{"v":-4,"label":"-4"},{"v":-2,"label":"-2"},{"v":0,"label":"0"},{"v":2,"label":"2"},{"v":4,"label":"4"},{"v":6,"label":"6"},{"v":8,"label":"8"}], Dygraph.numericTicks(-10, 9.98046875, 400, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":-200,"label":"-200"},{"v":0,"label":"0"},{"v":200,"label":"200"},{"v":400,"label":"400"},{"v":600,"label":"600"},{"v":800,"label":"800"},{"v":-17999000,"label":"1000"}], Dygraph.numericTicks(-101.10000000000001, 1100.1, 300, createOptionsViewForAxis('y',{"logscale":false,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":-20,"label":"-20"},{"v":-10,"label":"-10"},{"v":0,"label":"0"},{"v":10,"label":"10"},{"v":20,"label":"20"},{"v":30,"label":"30"},{"v":40,"label":"40"}], Dygraph.numericTicks(-11.687459005175139, 42.287459005175144, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":-15,"label":"-15"},{"v":-10,"label":"-10"},{"v":-5,"label":"-5"},{"v":0,"label":"0"},{"v":5,"label":"5"},{"v":10,"label":"10"}], Dygraph.numericTicks(-12, 12, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":-15,"label":"-15"},{"v":-10,"label":"-10"},{"v":-5,"label":"-5"},{"v":0,"label":"0"},{"v":5,"label":"5"},{"v":10,"label":"10"}], Dygraph.numericTicks(-13.19792086872138, 13.197062407353386, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":-220,"label":"-220"},{"v":-200,"label":"-200"},{"v":-180,"label":"-180"},{"v":-160,"label":"-160"},{"v":-140,"label":"-140"},{"v":-120,"label":"-120"}], Dygraph.numericTicks(-220, -100, 200, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":-40,"label":"-40"},{"v":-20,"label":"-20"},{"v":0,"label":"0"},{"v":20,"label":"20"},{"v":40,"label":"40"},{"v":60,"label":"60"},{"v":80,"label":"80"},{"v":100,"label":"100"},{"v":120,"label":"120"}], Dygraph.numericTicks(-32.8, 132.8, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":-40,"label":"-40"},{"v":-30,"label":"-30"},{"v":-20,"label":"-20"},{"v":-10,"label":"-10"},{"v":0,"label":"0"},{"v":10,"label":"10"},{"v":20,"label":"20"},{"v":30,"label":"30"},{"v":40,"label":"40"},{"v":50,"label":"50"},{"v":60,"label":"60"},{"v":70,"label":"70"},{"v":80,"label":"80"}], Dygraph.numericTicks(-34.309, 89.279, 400, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":-60,"label":"-60"},{"v":-40,"label":"-40"},{"v":-20,"label":"-20"},{"v":0,"label":"0"},{"v":20,"label":"20"},{"v":40,"label":"40"}], Dygraph.numericTicks(-60, 60, 200, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":-60,"label":"-60"},{"v":-40,"label":"-40"},{"v":-20,"label":"-20"},{"v":0,"label":"0"},{"v":20,"label":"20"},{"v":40,"label":"40"}], Dygraph.numericTicks(-60, 60, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":0.0001,"label":"1.00e-4"},{"v":0.0002,"label":"2.00e-4"},{"v":-17999999,"label":"3.00e-4"},{"v":0.0004,"label":"4.00e-4"},{"v":0.0005,"label":"5.00e-4"}], Dygraph.numericTicks(0, 0.00055, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":0},{"v":0.0001,"label":0.0001},{"v":0.0002,"label":0.0002},{"v":-17999999,"label":0.0003},{"v":0.0004,"label":0.0004},{"v":0.0005,"label":0.0005}], Dygraph.numericTicks(0, 0.00055, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":0.2,"label":"0.2"},{"v":0.4,"label":"0.4"},{"v":-17999999,"label":"0.6"},{"v":0.8,"label":"0.8"}], Dygraph.numericTicks(0, 1, 200, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":0.2,"label":"0.2"},{"v":0.4,"label":"0.4"},{"v":-17999999,"label":"0.6"},{"v":0.8,"label":"0.8"}], Dygraph.numericTicks(0, 1, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":0.1,"label":"0.1"},{"v":0.2,"label":"0.2"},{"v":-17999999,"label":"0.3"},{"v":0.4,"label":"0.4"},{"v":0.5,"label":"0.5"},{"v":-17999999,"label":"0.6"},{"v":-17999999,"label":"0.7"},{"v":0.8,"label":"0.8"},{"v":0.9,"label":"0.9"},{"v":1,"label":"1"},{"v":1.1,"label":"1.1"},{"v":-17999998,"label":"1.2"}], Dygraph.numericTicks(0, 1.2, 400, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":10,"label":"10"},{"v":20,"label":"20"},{"v":30,"label":"30"},{"v":40,"label":"40"},{"v":50,"label":"50"},{"v":60,"label":"60"},{"v":70,"label":"70"},{"v":80,"label":"80"},{"v":90,"label":"90"}], Dygraph.numericTicks(0, 100, 400, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":20,"label":"20"},{"v":40,"label":"40"},{"v":60,"label":"60"},{"v":80,"label":"80"},{"v":100,"label":"100"}], Dygraph.numericTicks(0, 104.53192180924043, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":20,"label":"20"},{"v":40,"label":"40"},{"v":60,"label":"60"},{"v":80,"label":"80"},{"v":100,"label":"100"}], Dygraph.numericTicks(0, 109.9856877755916, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":2,"label":"2"},{"v":4,"label":"4"},{"v":6,"label":"6"},{"v":8,"label":"8"},{"v":10,"label":"10"}], Dygraph.numericTicks(0, 11, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":20,"label":"20"},{"v":40,"label":"40"},{"v":60,"label":"60"},{"v":80,"label":"80"},{"v":100,"label":"100"}], Dygraph.numericTicks(0, 110, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":20,"label":"20"},{"v":40,"label":"40"},{"v":60,"label":"60"},{"v":80,"label":"80"},{"v":100,"label":"100"}], Dygraph.numericTicks(0, 110, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":10,"label":"10"},{"v":20,"label":"20"},{"v":30,"label":"30"},{"v":40,"label":"40"},{"v":50,"label":"50"},{"v":60,"label":"60"},{"v":70,"label":"70"},{"v":80,"label":"80"},{"v":90,"label":"90"},{"v":100,"label":"100"}], Dygraph.numericTicks(0, 110, 350, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":200,"label":"200"},{"v":400,"label":"400"},{"v":600,"label":"600"},{"v":800,"label":"800"},{"v":-17999000,"label":"1000"}], Dygraph.numericTicks(0, 1100, 300, createOptionsViewForAxis('y',{"logscale":false,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":-17000000,"label":"1M"},{"v":-16000000,"label":"2M"},{"v":-15000000,"label":"3M"},{"v":-14000000,"label":"4M"},{"v":-13000000,"label":"5M"},{"v":-12000000,"label":"6M"},{"v":-11000000,"label":"7M"},{"v":-10000000,"label":"8M"},{"v":-9000000,"label":"9M"},{"v":-8000000,"label":"10M"}], Dygraph.numericTicks(0, 11000000, 480, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":true})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":20,"label":"20"},{"v":40,"label":"40"},{"v":60,"label":"60"},{"v":80,"label":"80"},{"v":100,"label":"100"}], Dygraph.numericTicks(0, 119, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":20,"label":"20"},{"v":40,"label":"40"},{"v":60,"label":"60"},{"v":80,"label":"80"},{"v":100,"label":"100"},{"v":120,"label":"120"}], Dygraph.numericTicks(0, 130.9, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":20,"label":"20"},{"v":40,"label":"40"},{"v":60,"label":"60"},{"v":80,"label":"80"},{"v":100,"label":"100"},{"v":120,"label":"120"}], Dygraph.numericTicks(0, 131, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":-17998000,"label":"2000"},{"v":-17996000,"label":"4000"},{"v":-17994000,"label":"6000"},{"v":-17992000,"label":"8000"},{"v":-17990000,"label":"10000"},{"v":-17988000,"label":"12000"},{"v":-17986000,"label":"14000"},{"v":-17984000,"label":"16000"}], Dygraph.numericTicks(0, 16977.4, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":0.5,"label":"0.5"},{"v":1,"label":"1"},{"v":1.5,"label":"1.5"}], Dygraph.numericTicks(0, 2, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":0.2,"label":"0.2"},{"v":0.4,"label":"0.4"},{"v":-17999999,"label":"0.6"},{"v":0.8,"label":"0.8"},{"v":1,"label":"1"},{"v":-17999998,"label":"1.2"},{"v":-17999998,"label":"1.4"},{"v":1.6,"label":"1.6"},{"v":1.8,"label":"1.8"}], Dygraph.numericTicks(0, 2, 400, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":0.5,"label":"0.5"},{"v":1,"label":"1"},{"v":1.5,"label":"1.5"},{"v":2,"label":"2"}], Dygraph.numericTicks(0, 2.2, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":-17800000,"label":"200K"},{"v":-17600000,"label":"400K"},{"v":-17400000,"label":"600K"},{"v":-17200000,"label":"800K"},{"v":-17000000,"label":"1M"},{"v":-16800000,"label":"1.2M"},{"v":-16600000,"label":"1.4M"},{"v":-16400000,"label":"1.6M"},{"v":-16200000,"label":"1.8M"},{"v":-16000000,"label":"2M"}], Dygraph.numericTicks(0, 2200000, 350, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":true})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":50,"label":"50"},{"v":100,"label":"100"},{"v":150,"label":"150"},{"v":200,"label":"200"}], Dygraph.numericTicks(0, 249, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":500,"label":"500"},{"v":-17999000,"label":"1000"},{"v":1500,"label":"1500"},{"v":-17998000,"label":"2000"},{"v":2500,"label":"2500"}], Dygraph.numericTicks(0, 2747.9970998900817, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":200,"label":"200"},{"v":400,"label":"400"},{"v":600,"label":"600"},{"v":800,"label":"800"},{"v":-17999000,"label":"1K"},{"v":1200,"label":"1.2K"},{"v":1400,"label":"1.4K"},{"v":1600,"label":"1.6K"},{"v":1800,"label":"1.8K"},{"v":-17998000,"label":"2K"},{"v":2200,"label":"2.2K"},{"v":2400,"label":"2.4K"},{"v":2600,"label":"2.6K"}], Dygraph.numericTicks(0, 2747.9970998900817, 480, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":true})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":5,"label":"5"},{"v":10,"label":"10"},{"v":15,"label":"15"},{"v":20,"label":"20"},{"v":25,"label":"25"},{"v":30,"label":"30"}], Dygraph.numericTicks(0, 32.698942321287205, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":-17500000,"label":"500000"},{"v":-17000000,"label":"1.00e+6"},{"v":-16500000,"label":"1.50e+6"},{"v":-16000000,"label":"2.00e+6"},{"v":-15500000,"label":"2.50e+6"},{"v":-15000000,"label":"3.00e+6"}], Dygraph.numericTicks(0, 3263100.6418021005, 480, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":5,"label":"5"},{"v":10,"label":"10"},{"v":15,"label":"15"},{"v":20,"label":"20"},{"v":25,"label":"25"},{"v":30,"label":"30"}], Dygraph.numericTicks(0, 33.16213467701236, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":0.5,"label":"0.5"},{"v":1,"label":"1"},{"v":1.5,"label":"1.5"},{"v":2,"label":"2"},{"v":2.5,"label":"2.5"},{"v":3,"label":"3"},{"v":3.5,"label":"3.5"}], Dygraph.numericTicks(0, 4, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":0.5,"label":"0.5"},{"v":1,"label":"1"},{"v":1.5,"label":"1.5"},{"v":2,"label":"2"},{"v":2.5,"label":"2.5"},{"v":3,"label":"3"},{"v":3.5,"label":"3.5"},{"v":4,"label":"4"}], Dygraph.numericTicks(0, 4.4, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":5,"label":"5"},{"v":10,"label":"10"},{"v":15,"label":"15"},{"v":20,"label":"20"},{"v":25,"label":"25"},{"v":30,"label":"30"},{"v":35,"label":"35"},{"v":40,"label":"40"}], Dygraph.numericTicks(0, 42, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":true})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":8,"label":"8"},{"v":16,"label":"16"},{"v":24,"label":"24"},{"v":32,"label":"32"},{"v":40,"label":"40"}], Dygraph.numericTicks(0, 42, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":true,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":0},{"v":8,"label":8},{"v":16,"label":16},{"v":24,"label":24},{"v":32,"label":32},{"v":40,"label":40}], Dygraph.numericTicks(0, 42, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":true,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":999982000000,"label":"1T"},{"v":1999982000000,"label":"2T"},{"v":2999982000000,"label":"3T"},{"v":3999982000000,"label":"4T"}], Dygraph.numericTicks(0, 4837851162214.3, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":true})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":549755813888,"label":"512G"},{"v":1099511627776,"label":"1T"},{"v":1649267441664,"label":"1.5T"},{"v":2199023255552,"label":"2T"},{"v":2748779069440,"label":"2.5T"},{"v":3298534883328,"label":"3T"},{"v":3848290697216,"label":"3.5T"},{"v":4398046511104,"label":"4T"}], Dygraph.numericTicks(0, 4837851162214.3, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":true,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":0},{"v":549755813888,"label":"512G"},{"v":1099511627776,"label":"1T"},{"v":1649267441664,"label":"1.5T"},{"v":2199023255552,"label":"2T"},{"v":2748779069440,"label":"2.5T"},{"v":3298534883328,"label":"3T"},{"v":3848290697216,"label":"3.5T"},{"v":4398046511104,"label":"4T"}], Dygraph.numericTicks(0, 4837851162214.3, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":true,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":-17999000,"label":"1000"},{"v":-17998000,"label":"2000"},{"v":-17997000,"label":"3000"},{"v":-17996000,"label":"4000"},{"v":-17995000,"label":"5000"}], Dygraph.numericTicks(0, 5451.6, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":100,"label":"100"},{"v":200,"label":"200"},{"v":300,"label":"300"},{"v":400,"label":"400"},{"v":500,"label":"500"}], Dygraph.numericTicks(0, 550, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":10,"label":"10"},{"v":20,"label":"20"},{"v":30,"label":"30"},{"v":40,"label":"40"},{"v":50,"label":"50"},{"v":60,"label":"60"}], Dygraph.numericTicks(0, 64.9, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":100,"label":"100"},{"v":200,"label":"200"},{"v":300,"label":"300"},{"v":400,"label":"400"},{"v":500,"label":"500"},{"v":600,"label":"600"}], Dygraph.numericTicks(0, 667.9, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":1,"label":"1"},{"v":2,"label":"2"},{"v":3,"label":"3"},{"v":4,"label":"4"},{"v":5,"label":"5"},{"v":6,"label":"6"},{"v":7,"label":"7"}], Dygraph.numericTicks(0, 7.7, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":1,"label":"1"},{"v":2,"label":"2"},{"v":3,"label":"3"},{"v":4,"label":"4"},{"v":5,"label":"5"},{"v":6,"label":"6"},{"v":7,"label":"7"}], Dygraph.numericTicks(0, 7.9347329768293005, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":10,"label":"10"},{"v":20,"label":"20"},{"v":30,"label":"30"},{"v":40,"label":"40"},{"v":50,"label":"50"},{"v":60,"label":"60"},{"v":70,"label":"70"}], Dygraph.numericTicks(0, 72.6, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":20,"label":"20"},{"v":40,"label":"40"},{"v":60,"label":"60"},{"v":80,"label":"80"}], Dygraph.numericTicks(0, 99, 200, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":20,"label":"20"},{"v":40,"label":"40"},{"v":60,"label":"60"},{"v":80,"label":"80"}], Dygraph.numericTicks(0, 99, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":10,"label":"10"},{"v":20,"label":"20"},{"v":30,"label":"30"},{"v":40,"label":"40"},{"v":50,"label":"50"},{"v":60,"label":"60"},{"v":70,"label":"70"},{"v":80,"label":"80"},{"v":90,"label":"90"}], Dygraph.numericTicks(0, 99, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":200,"label":"200"},{"v":400,"label":"400"},{"v":600,"label":"600"},{"v":800,"label":"800"}], Dygraph.numericTicks(0, 999, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0.000001,"label":"1.00e-6"},{"v":-17999999,"label":""},{"v":-17999999,"label":""},{"v":-17999999,"label":""},{"v":-17999999,"label":""},{"v":-17999999,"label":""},{"v":-17999999,"label":""},{"v":-17999999,"label":""},{"v":-17999999,"label":""},{"v":-17999999,"label":"1.00e-5"},{"v":-17999999,"label":""},{"v":-17999999,"label":""},{"v":-17999999,"label":""},{"v":-17999999,"label":""},{"v":-17999999,"label":""},{"v":-17999999,"label":""},{"v":-17999999,"label":""},{"v":-17999999,"label":""},{"v":0.0001,"label":"1.00e-4"},{"v":0.0002,"label":""},{"v":-17999999,"label":""},{"v":0.0004,"label":""},{"v":0.0005,"label":""},{"v":-17999999,"label":""},{"v":0.0007,"label":""},{"v":0.0008,"label":""},{"v":-17999999,"label":""},{"v":0.001,"label":"1.00e-3"},{"v":0.002,"label":""},{"v":0.003,"label":""},{"v":0.004,"label":""},{"v":0.005,"label":""},{"v":0.006,"label":""},{"v":0.007,"label":""},{"v":0.008,"label":""},{"v":-17999999,"label":""},{"v":0.01,"label":"0.01"},{"v":0.02,"label":""},{"v":0.03,"label":""},{"v":0.04,"label":""},{"v":0.05,"label":""},{"v":0.06,"label":""},{"v":0.07,"label":""},{"v":0.08,"label":""},{"v":0.09,"label":""},{"v":0.1,"label":"0.1"},{"v":0.2,"label":""},{"v":-17999999,"label":""},{"v":0.4,"label":""},{"v":0.5,"label":""},{"v":-17999999,"label":""},{"v":-17999999,"label":""},{"v":0.8,"label":""},{"v":0.9,"label":""},{"v":1,"label":"1"},{"v":2,"label":""},{"v":3,"label":""},{"v":4,"label":""},{"v":5,"label":""},{"v":6,"label":""},{"v":7,"label":""},{"v":8,"label":""},{"v":9,"label":""},{"v":10,"label":"10"},{"v":20,"label":""},{"v":30,"label":""},{"v":40,"label":""},{"v":50,"label":""},{"v":60,"label":""},{"v":70,"label":""},{"v":80,"label":""},{"v":90,"label":""},{"v":100,"label":"100"},{"v":200,"label":""},{"v":300,"label":""},{"v":400,"label":""},{"v":500,"label":""},{"v":600,"label":""},{"v":700,"label":""},{"v":800,"label":""},{"v":900,"label":""},{"v":-17999000,"label":"1000"}], Dygraph.numericTicks(0.000001, 1099.9999999, 300, createOptionsViewForAxis('y',{"logscale":true,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":1,"label":"1"},{"v":2,"label":"2"},{"v":3,"label":"3"},{"v":4,"label":"4"},{"v":5,"label":"5"}], Dygraph.numericTicks(0.6, 5.4, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0.5,"label":"0.5"},{"v":1,"label":"1"},{"v":1.5,"label":"1.5"},{"v":2,"label":"2"},{"v":2.5,"label":"2.5"},{"v":3,"label":"3"},{"v":3.5,"label":"3.5"},{"v":4,"label":"4"},{"v":4.5,"label":"4.5"}], Dygraph.numericTicks(0.6373123361267239, 4.824406504982038, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0.5,"label":"0.5"},{"v":1,"label":"1"},{"v":1.5,"label":"1.5"},{"v":2,"label":"2"},{"v":2.5,"label":"2.5"},{"v":3,"label":"3"},{"v":3.5,"label":"3.5"},{"v":4,"label":"4"},{"v":4.5,"label":"4.5"}], Dygraph.numericTicks(0.6373123361267239, 4.824406504982038, 353, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0.6000000000000001,"label":"0.6"},{"v":0.8,"label":"0.8"},{"v":1,"label":"1"},{"v":-17999998,"label":"1.2"},{"v":-17999998,"label":"1.4"},{"v":1.6,"label":"1.6"},{"v":-17999998,"label":"1.8"},{"v":2,"label":"2"},{"v":2.2,"label":"2.2"},{"v":-17999997,"label":"2.4"},{"v":2.6,"label":"2.6"},{"v":-17999997,"label":"2.8"},{"v":-17999997,"label":"3"},{"v":3.2,"label":"3.2"},{"v":-17999996,"label":"3.4"},{"v":3.6,"label":"3.6"},{"v":-17999996,"label":"3.8"},{"v":4,"label":"4"},{"v":4.2,"label":"4.2"},{"v":4.4,"label":"4.4"},{"v":4.6,"label":"4.6"},{"v":-17999995,"label":"4.8"}], Dygraph.numericTicks(0.6373123361267239, 4.824406504982038, 743, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0.5,"label":"0.5"},{"v":1,"label":"1"},{"v":1.5,"label":"1.5"},{"v":2,"label":"2"},{"v":2.5,"label":"2.5"},{"v":3,"label":"3"},{"v":3.5,"label":"3.5"},{"v":4,"label":"4"},{"v":4.5,"label":"4.5"}], Dygraph.numericTicks(0.6386658954698001, 4.8095173522082, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0.5,"label":"0.5"},{"v":1,"label":"1"},{"v":1.5,"label":"1.5"},{"v":2,"label":"2"},{"v":2.5,"label":"2.5"},{"v":3,"label":"3"},{"v":3.5,"label":"3.5"},{"v":4,"label":"4"}], Dygraph.numericTicks(0.7101014279158788, 4.023726495301334, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":20,"label":"20"},{"v":40,"label":"40"},{"v":60,"label":"60"},{"v":80,"label":"80"},{"v":100,"label":"100"}], Dygraph.numericTicks(1, 109, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":1,"label":"1"},{"v":1.5,"label":"1.5"},{"v":2,"label":"2"},{"v":2.5,"label":"2.5"}], Dygraph.numericTicks(1, 3, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":1,"label":"1"},{"v":1.5,"label":"1.5"},{"v":2,"label":"2"},{"v":2.5,"label":"2.5"},{"v":3,"label":"3"},{"v":3.5,"label":"3.5"}], Dygraph.numericTicks(1, 4, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":1,"label":"1"},{"v":1.5,"label":"1.5"},{"v":2,"label":"2"},{"v":2.5,"label":"2.5"},{"v":3,"label":"3"},{"v":3.5,"label":"3.5"}], Dygraph.numericTicks(1, 4, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":1,"label":"1"},{"v":1.5,"label":"1.5"},{"v":2,"label":"2"},{"v":2.5,"label":"2.5"},{"v":3,"label":"3"},{"v":3.5,"label":"3.5"},{"v":4,"label":"4"},{"v":4.5,"label":"4.5"}], Dygraph.numericTicks(1, 5, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":1,"label":1},{"v":1.5,"label":1.5},{"v":2,"label":2},{"v":2.5,"label":2.5},{"v":3,"label":3},{"v":3.5,"label":3.5},{"v":4,"label":4},{"v":4.5,"label":4.5}], Dygraph.numericTicks(1, 5, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":1,"label":"1"},{"v":2,"label":"2"},{"v":3,"label":"3"},{"v":4,"label":"4"},{"v":5,"label":"5"}], Dygraph.numericTicks(1, 6, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":1,"label":"1"},{"v":2,"label":"2"},{"v":3,"label":"3"},{"v":4,"label":"4"},{"v":5,"label":"5"},{"v":6,"label":"6"}], Dygraph.numericTicks(1, 7, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":1,"label":"1"},{"v":2,"label":"2"},{"v":3,"label":"3"},{"v":4,"label":"4"},{"v":5,"label":"5"},{"v":6,"label":"6"},{"v":7,"label":"7"},{"v":8,"label":"8"}], Dygraph.numericTicks(1, 9, 300, createOptionsViewForAxis('y',{"logscale":false,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":1,"label":"1"},{"v":2,"label":"2"},{"v":3,"label":"3"},{"v":4,"label":"4"},{"v":5,"label":"5"},{"v":6,"label":""},{"v":7,"label":"7"},{"v":8,"label":""},{"v":9,"label":"9"}], Dygraph.numericTicks(1, 9, 300, createOptionsViewForAxis('y',{"logscale":true,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":1,"label":"1"},{"v":2,"label":"2"},{"v":3,"label":"3"},{"v":4,"label":"4"},{"v":5,"label":"5"},{"v":6,"label":"6"},{"v":7,"label":"7"},{"v":8,"label":"8"}], Dygraph.numericTicks(1, 9, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":2,"label":"2"},{"v":4,"label":"4"},{"v":6,"label":"6"},{"v":8,"label":"8"},{"v":10,"label":"10"}], Dygraph.numericTicks(1.2, 10.8, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":1,"label":"1"},{"v":1.5,"label":"1.5"},{"v":2,"label":"2"},{"v":2.5,"label":"2.5"},{"v":3,"label":"3"},{"v":3.5,"label":"3.5"},{"v":4,"label":"4"},{"v":4.5,"label":"4.5"}], Dygraph.numericTicks(1.2872947778969237, 4.765317192093838, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":1,"label":"1"},{"v":2,"label":"2"},{"v":3,"label":"3"},{"v":4,"label":"4"},{"v":5,"label":"5"},{"v":6,"label":"6"},{"v":7,"label":"7"}], Dygraph.numericTicks(1.5, 7.5, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":5,"label":"5"},{"v":10,"label":"10"},{"v":15,"label":"15"},{"v":20,"label":"20"},{"v":25,"label":"25"}], Dygraph.numericTicks(1.7999999999999998, 28.2, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":10,"label":"10"},{"v":10.1,"label":"10.1"},{"v":10.2,"label":"10.2"},{"v":10.3,"label":"10.3"},{"v":10.4,"label":"10.4"},{"v":10.5,"label":"10.5"},{"v":10.6,"label":"10.6"},{"v":10.7,"label":"10.7"},{"v":10.8,"label":"10.8"},{"v":10.9,"label":"10.9"}], Dygraph.numericTicks(10, 11, 480, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":100,"label":"100"},{"v":120,"label":"120"},{"v":140,"label":"140"},{"v":160,"label":"160"},{"v":180,"label":"180"}], Dygraph.numericTicks(100, 200, 200, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":10000,"label":"10000"},{"v":-17988000,"label":"12000"},{"v":-17986000,"label":"14000"},{"v":-17984000,"label":"16000"},{"v":-17982000,"label":"18000"},{"v":-17980000,"label":"20000"},{"v":-17978000,"label":"22000"},{"v":-17976000,"label":"24000"},{"v":-17974000,"label":"26000"},{"v":-17972000,"label":"28000"},{"v":-17970000,"label":"30000"},{"v":-17968000,"label":"32000"},{"v":-17966000,"label":"34000"},{"v":-17964000,"label":"36000"}], Dygraph.numericTicks(10122.8, 36789.2, 480, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":11000,"label":"11000"},{"v":11200,"label":"11200"},{"v":11400,"label":"11400"},{"v":11600,"label":"11600"},{"v":11800,"label":"11800"},{"v":-17988000,"label":"12000"},{"v":12200,"label":"12200"},{"v":12400,"label":"12400"},{"v":12600,"label":"12600"},{"v":12800,"label":"12800"},{"v":-17987000,"label":"13000"},{"v":13200,"label":"13200"},{"v":13400,"label":"13400"}], Dygraph.numericTicks(11110.5, 13579.5, 480, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":162000,"label":"162000"},{"v":-17836000,"label":"164000"},{"v":-17834000,"label":"166000"},{"v":-17832000,"label":"168000"},{"v":-17830000,"label":"170000"},{"v":-17828000,"label":"172000"},{"v":-17826000,"label":"174000"},{"v":-17824000,"label":"176000"},{"v":-17822000,"label":"178000"}], Dygraph.numericTicks(163038.4, 179137.6, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":2,"label":"2"},{"v":2.5,"label":"2.5"},{"v":3,"label":"3"},{"v":3.5,"label":"3.5"}], Dygraph.numericTicks(2, 4, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":2,"label":"2"},{"v":3,"label":"3"},{"v":4,"label":"4"},{"v":5,"label":"5"},{"v":6,"label":"6"},{"v":7,"label":"7"}], Dygraph.numericTicks(2.6, 7.4, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":20,"label":"20"},{"v":30,"label":"30"},{"v":40,"label":"40"},{"v":50,"label":"50"},{"v":60,"label":"60"},{"v":70,"label":"70"},{"v":80,"label":"80"},{"v":90,"label":"90"}], Dygraph.numericTicks(21.7, 97.3, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":20,"label":"20"},{"v":30,"label":"30"},{"v":40,"label":"40"},{"v":50,"label":"50"},{"v":60,"label":"60"},{"v":70,"label":"70"},{"v":80,"label":"80"},{"v":90,"label":"90"}], Dygraph.numericTicks(21.7, 97.3, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":20,"label":"20"},{"v":30,"label":"30"},{"v":40,"label":"40"},{"v":50,"label":"50"},{"v":60,"label":"60"},{"v":70,"label":"70"},{"v":80,"label":"80"},{"v":90,"label":"90"}], Dygraph.numericTicks(24, 96, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"}], Dygraph.numericTicks(26.185714285714287, 90.81428571428572, 20, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":20,"label":"20"},{"v":30,"label":"30"},{"v":40,"label":"40"},{"v":50,"label":"50"},{"v":60,"label":"60"},{"v":70,"label":"70"},{"v":80,"label":"80"},{"v":90,"label":"90"}], Dygraph.numericTicks(26.185714285714287, 90.81428571428572, 200, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false,pixelsPerLabel:20})));
  assert.deepEqual([{"v":25,"label":"25"},{"v":30,"label":"30"},{"v":35,"label":"35"},{"v":40,"label":"40"},{"v":45,"label":"45"},{"v":50,"label":"50"},{"v":55,"label":"55"},{"v":60,"label":"60"},{"v":65,"label":"65"},{"v":70,"label":"70"},{"v":75,"label":"75"},{"v":80,"label":"80"},{"v":85,"label":"85"},{"v":90,"label":"90"}], Dygraph.numericTicks(26.185714285714287, 90.81428571428572, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false,pixelsPerLabel:20})));
  assert.deepEqual([{"v":20,"label":"20"},{"v":30,"label":"30"},{"v":40,"label":"40"},{"v":50,"label":"50"},{"v":60,"label":"60"},{"v":70,"label":"70"},{"v":80,"label":"80"},{"v":90,"label":"90"}], Dygraph.numericTicks(26.185714285714287, 90.81428571428572, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":20,"label":"20"},{"v":30,"label":"30"},{"v":40,"label":"40"},{"v":50,"label":"50"},{"v":60,"label":"60"},{"v":70,"label":"70"},{"v":80,"label":"80"}], Dygraph.numericTicks(28.33333333333333, 88.33333333333334, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":3,"label":"3"},{"v":3.5,"label":"3.5"},{"v":4,"label":"4"},{"v":4.5,"label":"4.5"}], Dygraph.numericTicks(3, 5, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":3000,"label":"3K"},{"v":2500,"label":"2.5K"},{"v":-17998000,"label":"2K"},{"v":1500,"label":"1.5K"},{"v":-17999000,"label":"1K"},{"v":500,"label":"500"}], Dygraph.numericTicks(3000, 0, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":true})));
  assert.deepEqual([{"v":30,"label":"30"},{"v":40,"label":"40"},{"v":50,"label":"50"},{"v":60,"label":"60"},{"v":70,"label":"70"},{"v":80,"label":"80"}], Dygraph.numericTicks(33.11333333333334, 83.75333333333333, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":30,"label":"30"},{"v":40,"label":"40"},{"v":50,"label":"50"},{"v":60,"label":"60"},{"v":70,"label":"70"},{"v":80,"label":"80"}], Dygraph.numericTicks(36.921241050119335, 88.32696897374701, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":50,"label":""},{"v":60,"label":"60"},{"v":70,"label":""},{"v":80,"label":""},{"v":90,"label":""},{"v":100,"label":"100"},{"v":200,"label":""},{"v":300,"label":"300"},{"v":400,"label":""},{"v":500,"label":""},{"v":600,"label":"600"},{"v":700,"label":""},{"v":800,"label":""},{"v":900,"label":""},{"v":-17999000,"label":"1000"},{"v":-17998000,"label":""},{"v":-17997000,"label":"3000"},{"v":-17996000,"label":""},{"v":-17995000,"label":""},{"v":-17994000,"label":"6000"},{"v":-17993000,"label":""},{"v":-17992000,"label":""},{"v":-17991000,"label":""},{"v":-17990000,"label":"10000"}], Dygraph.numericTicks(41.220000000000084, 15576.828000000018, 400, createOptionsViewForAxis('y',{"logscale":true,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":40,"label":"40"},{"v":50,"label":"50"},{"v":60,"label":"60"},{"v":70,"label":"70"},{"v":80,"label":"80"},{"v":90,"label":"90"}], Dygraph.numericTicks(44.5, 98.5, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":5,"label":"5"},{"v":6,"label":""},{"v":7,"label":""},{"v":8,"label":""},{"v":9,"label":""},{"v":10,"label":"10"},{"v":20,"label":"20"},{"v":30,"label":""},{"v":40,"label":""},{"v":50,"label":"50"},{"v":60,"label":""},{"v":70,"label":""},{"v":80,"label":""},{"v":90,"label":""},{"v":100,"label":"100"},{"v":200,"label":"200"},{"v":300,"label":""},{"v":400,"label":""},{"v":500,"label":"500"},{"v":600,"label":""},{"v":700,"label":""},{"v":800,"label":""},{"v":900,"label":""},{"v":-17999000,"label":"1000"}], Dygraph.numericTicks(5, 1099.5, 300, createOptionsViewForAxis('y',{"logscale":true,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":50,"label":"50"},{"v":55,"label":"55"},{"v":60,"label":"60"},{"v":65,"label":"65"},{"v":70,"label":"70"},{"v":75,"label":"75"},{"v":80,"label":"80"}], Dygraph.numericTicks(52.5, 82.5, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":68,"label":"68"},{"v":70,"label":"70"},{"v":72,"label":"72"},{"v":74,"label":"74"},{"v":76,"label":"76"},{"v":78,"label":"78"},{"v":80,"label":"80"}], Dygraph.numericTicks(69, 81, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":0,"label":"0"},{"v":-17980000,"label":"20K"},{"v":-17960000,"label":"40K"},{"v":-17940000,"label":"60K"},{"v":-17920000,"label":"80K"}], Dygraph.numericTicks(7921.099999999999, 81407.9, 240, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":true})));
  assert.deepEqual([{"v":8,"label":"8"},{"v":10,"label":"10"},{"v":12,"label":"12"},{"v":14,"label":"14"},{"v":16,"label":"16"},{"v":18,"label":"18"},{"v":20,"label":"20"}], Dygraph.numericTicks(9, 21, 300, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":8,"label":"8"},{"v":10,"label":"10"},{"v":12,"label":"12"},{"v":14,"label":"14"},{"v":16,"label":"16"},{"v":18,"label":"18"},{"v":20,"label":"20"}], Dygraph.numericTicks(9, 21, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":9,"label":"9"},{"v":10,"label":"10"},{"v":11,"label":"11"},{"v":12,"label":"12"},{"v":13,"label":"13"},{"v":14,"label":"14"},{"v":15,"label":"15"},{"v":16,"label":"16"},{"v":17,"label":"17"},{"v":18,"label":"18"}], Dygraph.numericTicks(9.2, 18.8, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":80,"label":"80"},{"v":100,"label":"100"},{"v":120,"label":"120"},{"v":140,"label":"140"},{"v":160,"label":"160"},{"v":180,"label":"180"},{"v":200,"label":"200"}], Dygraph.numericTicks(90, 210, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":true})));
  assert.deepEqual([{"v":95,"label":"95"},{"v":96,"label":"96"},{"v":97,"label":"97"},{"v":98,"label":"98"},{"v":99,"label":"99"},{"v":100,"label":"100"},{"v":101,"label":"101"},{"v":102,"label":"102"},{"v":103,"label":"103"},{"v":104,"label":"104"}], Dygraph.numericTicks(95.71121718377088, 104.23150357995226, 320, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
  assert.deepEqual([{"v":950,"label":"950"},{"v":-17999000,"label":"1000"},{"v":1050,"label":"1050"},{"v":1100,"label":"1100"},{"v":1150,"label":"1150"},{"v":1200,"label":"1200"}], Dygraph.numericTicks(980.1, 1218.9, 200, createOptionsViewForAxis('y',{"logscale":null,"labelsKMG2":false,"labelsKMB":false})));
});
*/

});
