'use strict';

var middlewareHostHeader = require('@aws-sdk/middleware-host-header');
var middlewareLogger = require('@aws-sdk/middleware-logger');
var middlewareRecursionDetection = require('@aws-sdk/middleware-recursion-detection');
var middlewareUserAgent = require('@aws-sdk/middleware-user-agent');
var configResolver = require('@smithy/config-resolver');
var core = require('@smithy/core');
var middlewareContentLength = require('@smithy/middleware-content-length');
var middlewareEndpoint = require('@smithy/middleware-endpoint');
var middlewareRetry = require('@smithy/middleware-retry');
var smithyClient = require('@smithy/smithy-client');
var httpAuthSchemeProvider = require('./auth/httpAuthSchemeProvider');
var runtimeConfig = require('./runtimeConfig');
var regionConfigResolver = require('@aws-sdk/region-config-resolver');
var protocolHttp = require('@smithy/protocol-http');
var middlewareSerde = require('@smithy/middleware-serde');
var core$1 = require('@aws-sdk/core');
var xmlBuilder = require('@aws-sdk/xml-builder');
var utilWaiter = require('@smithy/util-waiter');

const resolveClientEndpointParameters = (options) => {
    return Object.assign(options, {
        useDualstackEndpoint: options.useDualstackEndpoint ?? false,
        useFipsEndpoint: options.useFipsEndpoint ?? false,
        defaultSigningName: "cloudfront",
    });
};
const commonParams = {
    UseFIPS: { type: "builtInParams", name: "useFipsEndpoint" },
    Endpoint: { type: "builtInParams", name: "endpoint" },
    Region: { type: "builtInParams", name: "region" },
    UseDualStack: { type: "builtInParams", name: "useDualstackEndpoint" },
};

const getHttpAuthExtensionConfiguration = (runtimeConfig) => {
    const _httpAuthSchemes = runtimeConfig.httpAuthSchemes;
    let _httpAuthSchemeProvider = runtimeConfig.httpAuthSchemeProvider;
    let _credentials = runtimeConfig.credentials;
    return {
        setHttpAuthScheme(httpAuthScheme) {
            const index = _httpAuthSchemes.findIndex((scheme) => scheme.schemeId === httpAuthScheme.schemeId);
            if (index === -1) {
                _httpAuthSchemes.push(httpAuthScheme);
            }
            else {
                _httpAuthSchemes.splice(index, 1, httpAuthScheme);
            }
        },
        httpAuthSchemes() {
            return _httpAuthSchemes;
        },
        setHttpAuthSchemeProvider(httpAuthSchemeProvider) {
            _httpAuthSchemeProvider = httpAuthSchemeProvider;
        },
        httpAuthSchemeProvider() {
            return _httpAuthSchemeProvider;
        },
        setCredentials(credentials) {
            _credentials = credentials;
        },
        credentials() {
            return _credentials;
        },
    };
};
const resolveHttpAuthRuntimeConfig = (config) => {
    return {
        httpAuthSchemes: config.httpAuthSchemes(),
        httpAuthSchemeProvider: config.httpAuthSchemeProvider(),
        credentials: config.credentials(),
    };
};

const resolveRuntimeExtensions = (runtimeConfig, extensions) => {
    const extensionConfiguration = Object.assign(regionConfigResolver.getAwsRegionExtensionConfiguration(runtimeConfig), smithyClient.getDefaultExtensionConfiguration(runtimeConfig), protocolHttp.getHttpHandlerExtensionConfiguration(runtimeConfig), getHttpAuthExtensionConfiguration(runtimeConfig));
    extensions.forEach((extension) => extension.configure(extensionConfiguration));
    return Object.assign(runtimeConfig, regionConfigResolver.resolveAwsRegionExtensionConfiguration(extensionConfiguration), smithyClient.resolveDefaultRuntimeConfig(extensionConfiguration), protocolHttp.resolveHttpHandlerRuntimeConfig(extensionConfiguration), resolveHttpAuthRuntimeConfig(extensionConfiguration));
};

class CloudFrontClient extends smithyClient.Client {
    config;
    constructor(...[configuration]) {
        const _config_0 = runtimeConfig.getRuntimeConfig(configuration || {});
        super(_config_0);
        this.initConfig = _config_0;
        const _config_1 = resolveClientEndpointParameters(_config_0);
        const _config_2 = middlewareUserAgent.resolveUserAgentConfig(_config_1);
        const _config_3 = middlewareRetry.resolveRetryConfig(_config_2);
        const _config_4 = configResolver.resolveRegionConfig(_config_3);
        const _config_5 = middlewareHostHeader.resolveHostHeaderConfig(_config_4);
        const _config_6 = middlewareEndpoint.resolveEndpointConfig(_config_5);
        const _config_7 = httpAuthSchemeProvider.resolveHttpAuthSchemeConfig(_config_6);
        const _config_8 = resolveRuntimeExtensions(_config_7, configuration?.extensions || []);
        this.config = _config_8;
        this.middlewareStack.use(middlewareUserAgent.getUserAgentPlugin(this.config));
        this.middlewareStack.use(middlewareRetry.getRetryPlugin(this.config));
        this.middlewareStack.use(middlewareContentLength.getContentLengthPlugin(this.config));
        this.middlewareStack.use(middlewareHostHeader.getHostHeaderPlugin(this.config));
        this.middlewareStack.use(middlewareLogger.getLoggerPlugin(this.config));
        this.middlewareStack.use(middlewareRecursionDetection.getRecursionDetectionPlugin(this.config));
        this.middlewareStack.use(core.getHttpAuthSchemeEndpointRuleSetPlugin(this.config, {
            httpAuthSchemeParametersProvider: httpAuthSchemeProvider.defaultCloudFrontHttpAuthSchemeParametersProvider,
            identityProviderConfigProvider: async (config) => new core.DefaultIdentityProviderConfig({
                "aws.auth#sigv4": config.credentials,
            }),
        }));
        this.middlewareStack.use(core.getHttpSigningPlugin(this.config));
    }
    destroy() {
        super.destroy();
    }
}

class CloudFrontServiceException extends smithyClient.ServiceException {
    constructor(options) {
        super(options);
        Object.setPrototypeOf(this, CloudFrontServiceException.prototype);
    }
}

const ResponseHeadersPolicyAccessControlAllowMethodsValues = {
    ALL: "ALL",
    DELETE: "DELETE",
    GET: "GET",
    HEAD: "HEAD",
    OPTIONS: "OPTIONS",
    PATCH: "PATCH",
    POST: "POST",
    PUT: "PUT",
};
class AccessDenied extends CloudFrontServiceException {
    name = "AccessDenied";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "AccessDenied",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, AccessDenied.prototype);
        this.Message = opts.Message;
    }
}
const ICPRecordalStatus = {
    APPROVED: "APPROVED",
    PENDING: "PENDING",
    SUSPENDED: "SUSPENDED",
};
const Method = {
    DELETE: "DELETE",
    GET: "GET",
    HEAD: "HEAD",
    OPTIONS: "OPTIONS",
    PATCH: "PATCH",
    POST: "POST",
    PUT: "PUT",
};
class IllegalUpdate extends CloudFrontServiceException {
    name = "IllegalUpdate";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "IllegalUpdate",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, IllegalUpdate.prototype);
        this.Message = opts.Message;
    }
}
class InvalidArgument extends CloudFrontServiceException {
    name = "InvalidArgument";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidArgument",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidArgument.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchDistribution extends CloudFrontServiceException {
    name = "NoSuchDistribution";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchDistribution",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchDistribution.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionCNAMEs extends CloudFrontServiceException {
    name = "TooManyDistributionCNAMEs";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionCNAMEs",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionCNAMEs.prototype);
        this.Message = opts.Message;
    }
}
class EntityNotFound extends CloudFrontServiceException {
    name = "EntityNotFound";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "EntityNotFound",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, EntityNotFound.prototype);
        this.Message = opts.Message;
    }
}
class InvalidIfMatchVersion extends CloudFrontServiceException {
    name = "InvalidIfMatchVersion";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidIfMatchVersion",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidIfMatchVersion.prototype);
        this.Message = opts.Message;
    }
}
class PreconditionFailed extends CloudFrontServiceException {
    name = "PreconditionFailed";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "PreconditionFailed",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, PreconditionFailed.prototype);
        this.Message = opts.Message;
    }
}
class BatchTooLarge extends CloudFrontServiceException {
    name = "BatchTooLarge";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "BatchTooLarge",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, BatchTooLarge.prototype);
        this.Message = opts.Message;
    }
}
const ItemSelection = {
    all: "all",
    none: "none",
    whitelist: "whitelist",
};
const EventType = {
    origin_request: "origin-request",
    origin_response: "origin-response",
    viewer_request: "viewer-request",
    viewer_response: "viewer-response",
};
const ViewerProtocolPolicy = {
    allow_all: "allow-all",
    https_only: "https-only",
    redirect_to_https: "redirect-to-https",
};
const CachePolicyCookieBehavior = {
    all: "all",
    allExcept: "allExcept",
    none: "none",
    whitelist: "whitelist",
};
const CachePolicyHeaderBehavior = {
    none: "none",
    whitelist: "whitelist",
};
const CachePolicyQueryStringBehavior = {
    all: "all",
    allExcept: "allExcept",
    none: "none",
    whitelist: "whitelist",
};
class CachePolicyAlreadyExists extends CloudFrontServiceException {
    name = "CachePolicyAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "CachePolicyAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CachePolicyAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class CachePolicyInUse extends CloudFrontServiceException {
    name = "CachePolicyInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "CachePolicyInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CachePolicyInUse.prototype);
        this.Message = opts.Message;
    }
}
const CachePolicyType = {
    custom: "custom",
    managed: "managed",
};
class CannotChangeImmutablePublicKeyFields extends CloudFrontServiceException {
    name = "CannotChangeImmutablePublicKeyFields";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "CannotChangeImmutablePublicKeyFields",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CannotChangeImmutablePublicKeyFields.prototype);
        this.Message = opts.Message;
    }
}
class CannotDeleteEntityWhileInUse extends CloudFrontServiceException {
    name = "CannotDeleteEntityWhileInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "CannotDeleteEntityWhileInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CannotDeleteEntityWhileInUse.prototype);
        this.Message = opts.Message;
    }
}
class CannotUpdateEntityWhileInUse extends CloudFrontServiceException {
    name = "CannotUpdateEntityWhileInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "CannotUpdateEntityWhileInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CannotUpdateEntityWhileInUse.prototype);
        this.Message = opts.Message;
    }
}
const CertificateSource = {
    acm: "acm",
    cloudfront: "cloudfront",
    iam: "iam",
};
const CertificateTransparencyLoggingPreference = {
    Disabled: "disabled",
    Enabled: "enabled",
};
class CNAMEAlreadyExists extends CloudFrontServiceException {
    name = "CNAMEAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "CNAMEAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CNAMEAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
const ConnectionMode = {
    Direct: "direct",
    TenantOnly: "tenant-only",
};
const HttpVersion = {
    http1_1: "http1.1",
    http2: "http2",
    http2and3: "http2and3",
    http3: "http3",
};
const OriginGroupSelectionCriteria = {
    Default: "default",
    MediaQualityBased: "media-quality-based",
};
const IpAddressType = {
    DualStack: "dualstack",
    Ipv4: "ipv4",
    Ipv6: "ipv6",
};
const OriginProtocolPolicy = {
    http_only: "http-only",
    https_only: "https-only",
    match_viewer: "match-viewer",
};
const SslProtocol = {
    SSLv3: "SSLv3",
    TLSv1: "TLSv1",
    TLSv1_1: "TLSv1.1",
    TLSv1_2: "TLSv1.2",
};
const PriceClass = {
    None: "None",
    PriceClass_100: "PriceClass_100",
    PriceClass_200: "PriceClass_200",
    PriceClass_All: "PriceClass_All",
};
const GeoRestrictionType = {
    blacklist: "blacklist",
    none: "none",
    whitelist: "whitelist",
};
const MinimumProtocolVersion = {
    SSLv3: "SSLv3",
    TLSv1: "TLSv1",
    TLSv1_1_2016: "TLSv1.1_2016",
    TLSv1_2016: "TLSv1_2016",
    TLSv1_2_2018: "TLSv1.2_2018",
    TLSv1_2_2019: "TLSv1.2_2019",
    TLSv1_2_2021: "TLSv1.2_2021",
    TLSv1_2_2025: "TLSv1.2_2025",
    TLSv1_3_2025: "TLSv1.3_2025",
};
const SSLSupportMethod = {
    sni_only: "sni-only",
    static_ip: "static-ip",
    vip: "vip",
};
class DistributionAlreadyExists extends CloudFrontServiceException {
    name = "DistributionAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "DistributionAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, DistributionAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior extends CloudFrontServiceException {
    name = "IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior.prototype);
        this.Message = opts.Message;
    }
}
class InconsistentQuantities extends CloudFrontServiceException {
    name = "InconsistentQuantities";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InconsistentQuantities",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InconsistentQuantities.prototype);
        this.Message = opts.Message;
    }
}
class InvalidDefaultRootObject extends CloudFrontServiceException {
    name = "InvalidDefaultRootObject";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidDefaultRootObject",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidDefaultRootObject.prototype);
        this.Message = opts.Message;
    }
}
class InvalidErrorCode extends CloudFrontServiceException {
    name = "InvalidErrorCode";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidErrorCode",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidErrorCode.prototype);
        this.Message = opts.Message;
    }
}
class InvalidForwardCookies extends CloudFrontServiceException {
    name = "InvalidForwardCookies";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidForwardCookies",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidForwardCookies.prototype);
        this.Message = opts.Message;
    }
}
class InvalidFunctionAssociation extends CloudFrontServiceException {
    name = "InvalidFunctionAssociation";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidFunctionAssociation",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidFunctionAssociation.prototype);
        this.Message = opts.Message;
    }
}
class InvalidGeoRestrictionParameter extends CloudFrontServiceException {
    name = "InvalidGeoRestrictionParameter";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidGeoRestrictionParameter",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidGeoRestrictionParameter.prototype);
        this.Message = opts.Message;
    }
}
class InvalidHeadersForS3Origin extends CloudFrontServiceException {
    name = "InvalidHeadersForS3Origin";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidHeadersForS3Origin",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidHeadersForS3Origin.prototype);
        this.Message = opts.Message;
    }
}
class InvalidLambdaFunctionAssociation extends CloudFrontServiceException {
    name = "InvalidLambdaFunctionAssociation";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidLambdaFunctionAssociation",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidLambdaFunctionAssociation.prototype);
        this.Message = opts.Message;
    }
}
class InvalidLocationCode extends CloudFrontServiceException {
    name = "InvalidLocationCode";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidLocationCode",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidLocationCode.prototype);
        this.Message = opts.Message;
    }
}
class InvalidMinimumProtocolVersion extends CloudFrontServiceException {
    name = "InvalidMinimumProtocolVersion";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidMinimumProtocolVersion",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidMinimumProtocolVersion.prototype);
        this.Message = opts.Message;
    }
}
class InvalidOrigin extends CloudFrontServiceException {
    name = "InvalidOrigin";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidOrigin",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidOrigin.prototype);
        this.Message = opts.Message;
    }
}
class InvalidOriginAccessControl extends CloudFrontServiceException {
    name = "InvalidOriginAccessControl";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidOriginAccessControl",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidOriginAccessControl.prototype);
        this.Message = opts.Message;
    }
}
class InvalidOriginAccessIdentity extends CloudFrontServiceException {
    name = "InvalidOriginAccessIdentity";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidOriginAccessIdentity",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidOriginAccessIdentity.prototype);
        this.Message = opts.Message;
    }
}
class InvalidOriginKeepaliveTimeout extends CloudFrontServiceException {
    name = "InvalidOriginKeepaliveTimeout";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidOriginKeepaliveTimeout",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidOriginKeepaliveTimeout.prototype);
        this.Message = opts.Message;
    }
}
class InvalidOriginReadTimeout extends CloudFrontServiceException {
    name = "InvalidOriginReadTimeout";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidOriginReadTimeout",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidOriginReadTimeout.prototype);
        this.Message = opts.Message;
    }
}
class InvalidProtocolSettings extends CloudFrontServiceException {
    name = "InvalidProtocolSettings";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidProtocolSettings",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidProtocolSettings.prototype);
        this.Message = opts.Message;
    }
}
class InvalidQueryStringParameters extends CloudFrontServiceException {
    name = "InvalidQueryStringParameters";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidQueryStringParameters",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidQueryStringParameters.prototype);
        this.Message = opts.Message;
    }
}
class InvalidRelativePath extends CloudFrontServiceException {
    name = "InvalidRelativePath";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidRelativePath",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidRelativePath.prototype);
        this.Message = opts.Message;
    }
}
class InvalidRequiredProtocol extends CloudFrontServiceException {
    name = "InvalidRequiredProtocol";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidRequiredProtocol",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidRequiredProtocol.prototype);
        this.Message = opts.Message;
    }
}
class InvalidResponseCode extends CloudFrontServiceException {
    name = "InvalidResponseCode";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidResponseCode",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidResponseCode.prototype);
        this.Message = opts.Message;
    }
}
class InvalidTTLOrder extends CloudFrontServiceException {
    name = "InvalidTTLOrder";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidTTLOrder",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidTTLOrder.prototype);
        this.Message = opts.Message;
    }
}
class InvalidViewerCertificate extends CloudFrontServiceException {
    name = "InvalidViewerCertificate";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidViewerCertificate",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidViewerCertificate.prototype);
        this.Message = opts.Message;
    }
}
class InvalidWebACLId extends CloudFrontServiceException {
    name = "InvalidWebACLId";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidWebACLId",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidWebACLId.prototype);
        this.Message = opts.Message;
    }
}
class MissingBody extends CloudFrontServiceException {
    name = "MissingBody";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "MissingBody",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, MissingBody.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchCachePolicy extends CloudFrontServiceException {
    name = "NoSuchCachePolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchCachePolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchCachePolicy.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchFieldLevelEncryptionConfig extends CloudFrontServiceException {
    name = "NoSuchFieldLevelEncryptionConfig";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchFieldLevelEncryptionConfig",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchFieldLevelEncryptionConfig.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchOrigin extends CloudFrontServiceException {
    name = "NoSuchOrigin";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchOrigin",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchOrigin.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchOriginRequestPolicy extends CloudFrontServiceException {
    name = "NoSuchOriginRequestPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchOriginRequestPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchOriginRequestPolicy.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchRealtimeLogConfig extends CloudFrontServiceException {
    name = "NoSuchRealtimeLogConfig";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchRealtimeLogConfig",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchRealtimeLogConfig.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchResponseHeadersPolicy extends CloudFrontServiceException {
    name = "NoSuchResponseHeadersPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchResponseHeadersPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchResponseHeadersPolicy.prototype);
        this.Message = opts.Message;
    }
}
class RealtimeLogConfigOwnerMismatch extends CloudFrontServiceException {
    name = "RealtimeLogConfigOwnerMismatch";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "RealtimeLogConfigOwnerMismatch",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, RealtimeLogConfigOwnerMismatch.prototype);
        this.Message = opts.Message;
    }
}
class TooManyCacheBehaviors extends CloudFrontServiceException {
    name = "TooManyCacheBehaviors";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyCacheBehaviors",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyCacheBehaviors.prototype);
        this.Message = opts.Message;
    }
}
class TooManyCertificates extends CloudFrontServiceException {
    name = "TooManyCertificates";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyCertificates",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyCertificates.prototype);
        this.Message = opts.Message;
    }
}
class TooManyCookieNamesInWhiteList extends CloudFrontServiceException {
    name = "TooManyCookieNamesInWhiteList";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyCookieNamesInWhiteList",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyCookieNamesInWhiteList.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributions extends CloudFrontServiceException {
    name = "TooManyDistributions";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributions",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributions.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsAssociatedToCachePolicy extends CloudFrontServiceException {
    name = "TooManyDistributionsAssociatedToCachePolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsAssociatedToCachePolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsAssociatedToCachePolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsAssociatedToFieldLevelEncryptionConfig extends CloudFrontServiceException {
    name = "TooManyDistributionsAssociatedToFieldLevelEncryptionConfig";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsAssociatedToFieldLevelEncryptionConfig",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsAssociatedToFieldLevelEncryptionConfig.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsAssociatedToKeyGroup extends CloudFrontServiceException {
    name = "TooManyDistributionsAssociatedToKeyGroup";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsAssociatedToKeyGroup",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsAssociatedToKeyGroup.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsAssociatedToOriginAccessControl extends CloudFrontServiceException {
    name = "TooManyDistributionsAssociatedToOriginAccessControl";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsAssociatedToOriginAccessControl",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsAssociatedToOriginAccessControl.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsAssociatedToOriginRequestPolicy extends CloudFrontServiceException {
    name = "TooManyDistributionsAssociatedToOriginRequestPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsAssociatedToOriginRequestPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsAssociatedToOriginRequestPolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsAssociatedToResponseHeadersPolicy extends CloudFrontServiceException {
    name = "TooManyDistributionsAssociatedToResponseHeadersPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsAssociatedToResponseHeadersPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsAssociatedToResponseHeadersPolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsWithFunctionAssociations extends CloudFrontServiceException {
    name = "TooManyDistributionsWithFunctionAssociations";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsWithFunctionAssociations",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsWithFunctionAssociations.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsWithLambdaAssociations extends CloudFrontServiceException {
    name = "TooManyDistributionsWithLambdaAssociations";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsWithLambdaAssociations",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsWithLambdaAssociations.prototype);
        this.Message = opts.Message;
    }
}
class TooManyDistributionsWithSingleFunctionARN extends CloudFrontServiceException {
    name = "TooManyDistributionsWithSingleFunctionARN";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyDistributionsWithSingleFunctionARN",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyDistributionsWithSingleFunctionARN.prototype);
        this.Message = opts.Message;
    }
}
class TooManyFunctionAssociations extends CloudFrontServiceException {
    name = "TooManyFunctionAssociations";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyFunctionAssociations",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyFunctionAssociations.prototype);
        this.Message = opts.Message;
    }
}
class TooManyHeadersInForwardedValues extends CloudFrontServiceException {
    name = "TooManyHeadersInForwardedValues";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyHeadersInForwardedValues",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyHeadersInForwardedValues.prototype);
        this.Message = opts.Message;
    }
}
class TooManyKeyGroupsAssociatedToDistribution extends CloudFrontServiceException {
    name = "TooManyKeyGroupsAssociatedToDistribution";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyKeyGroupsAssociatedToDistribution",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyKeyGroupsAssociatedToDistribution.prototype);
        this.Message = opts.Message;
    }
}
class TooManyLambdaFunctionAssociations extends CloudFrontServiceException {
    name = "TooManyLambdaFunctionAssociations";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyLambdaFunctionAssociations",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyLambdaFunctionAssociations.prototype);
        this.Message = opts.Message;
    }
}
class TooManyOriginCustomHeaders extends CloudFrontServiceException {
    name = "TooManyOriginCustomHeaders";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyOriginCustomHeaders",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyOriginCustomHeaders.prototype);
        this.Message = opts.Message;
    }
}
class TooManyOriginGroupsPerDistribution extends CloudFrontServiceException {
    name = "TooManyOriginGroupsPerDistribution";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyOriginGroupsPerDistribution",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyOriginGroupsPerDistribution.prototype);
        this.Message = opts.Message;
    }
}
class TooManyOrigins extends CloudFrontServiceException {
    name = "TooManyOrigins";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyOrigins",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyOrigins.prototype);
        this.Message = opts.Message;
    }
}
class TooManyQueryStringParameters extends CloudFrontServiceException {
    name = "TooManyQueryStringParameters";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyQueryStringParameters",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyQueryStringParameters.prototype);
        this.Message = opts.Message;
    }
}
class TooManyTrustedSigners extends CloudFrontServiceException {
    name = "TooManyTrustedSigners";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyTrustedSigners",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyTrustedSigners.prototype);
        this.Message = opts.Message;
    }
}
class TrustedKeyGroupDoesNotExist extends CloudFrontServiceException {
    name = "TrustedKeyGroupDoesNotExist";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TrustedKeyGroupDoesNotExist",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TrustedKeyGroupDoesNotExist.prototype);
        this.Message = opts.Message;
    }
}
class TrustedSignerDoesNotExist extends CloudFrontServiceException {
    name = "TrustedSignerDoesNotExist";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TrustedSignerDoesNotExist",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TrustedSignerDoesNotExist.prototype);
        this.Message = opts.Message;
    }
}
class EntityAlreadyExists extends CloudFrontServiceException {
    name = "EntityAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "EntityAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, EntityAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class EntityLimitExceeded extends CloudFrontServiceException {
    name = "EntityLimitExceeded";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "EntityLimitExceeded",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, EntityLimitExceeded.prototype);
        this.Message = opts.Message;
    }
}
class InvalidTagging extends CloudFrontServiceException {
    name = "InvalidTagging";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidTagging",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidTagging.prototype);
        this.Message = opts.Message;
    }
}
class UnsupportedOperation extends CloudFrontServiceException {
    name = "UnsupportedOperation";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "UnsupportedOperation",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, UnsupportedOperation.prototype);
        this.Message = opts.Message;
    }
}
class TooManyCachePolicies extends CloudFrontServiceException {
    name = "TooManyCachePolicies";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyCachePolicies",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyCachePolicies.prototype);
        this.Message = opts.Message;
    }
}
class TooManyCookiesInCachePolicy extends CloudFrontServiceException {
    name = "TooManyCookiesInCachePolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyCookiesInCachePolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyCookiesInCachePolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyHeadersInCachePolicy extends CloudFrontServiceException {
    name = "TooManyHeadersInCachePolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyHeadersInCachePolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyHeadersInCachePolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyQueryStringsInCachePolicy extends CloudFrontServiceException {
    name = "TooManyQueryStringsInCachePolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyQueryStringsInCachePolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyQueryStringsInCachePolicy.prototype);
        this.Message = opts.Message;
    }
}
class CloudFrontOriginAccessIdentityAlreadyExists extends CloudFrontServiceException {
    name = "CloudFrontOriginAccessIdentityAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "CloudFrontOriginAccessIdentityAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CloudFrontOriginAccessIdentityAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class TooManyCloudFrontOriginAccessIdentities extends CloudFrontServiceException {
    name = "TooManyCloudFrontOriginAccessIdentities";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyCloudFrontOriginAccessIdentities",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyCloudFrontOriginAccessIdentities.prototype);
        this.Message = opts.Message;
    }
}
class ContinuousDeploymentPolicyAlreadyExists extends CloudFrontServiceException {
    name = "ContinuousDeploymentPolicyAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "ContinuousDeploymentPolicyAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ContinuousDeploymentPolicyAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
const ContinuousDeploymentPolicyType = {
    SingleHeader: "SingleHeader",
    SingleWeight: "SingleWeight",
};
class StagingDistributionInUse extends CloudFrontServiceException {
    name = "StagingDistributionInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "StagingDistributionInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, StagingDistributionInUse.prototype);
        this.Message = opts.Message;
    }
}
class TooManyContinuousDeploymentPolicies extends CloudFrontServiceException {
    name = "TooManyContinuousDeploymentPolicies";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyContinuousDeploymentPolicies",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyContinuousDeploymentPolicies.prototype);
        this.Message = opts.Message;
    }
}
class ContinuousDeploymentPolicyInUse extends CloudFrontServiceException {
    name = "ContinuousDeploymentPolicyInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "ContinuousDeploymentPolicyInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ContinuousDeploymentPolicyInUse.prototype);
        this.Message = opts.Message;
    }
}
class IllegalOriginAccessConfiguration extends CloudFrontServiceException {
    name = "IllegalOriginAccessConfiguration";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "IllegalOriginAccessConfiguration",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, IllegalOriginAccessConfiguration.prototype);
        this.Message = opts.Message;
    }
}
class InvalidDomainNameForOriginAccessControl extends CloudFrontServiceException {
    name = "InvalidDomainNameForOriginAccessControl";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidDomainNameForOriginAccessControl",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidDomainNameForOriginAccessControl.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchContinuousDeploymentPolicy extends CloudFrontServiceException {
    name = "NoSuchContinuousDeploymentPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchContinuousDeploymentPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchContinuousDeploymentPolicy.prototype);
        this.Message = opts.Message;
    }
}
const CustomizationActionType = {
    disable: "disable",
    override: "override",
};
const ValidationTokenHost = {
    CloudFront: "cloudfront",
    SelfHosted: "self-hosted",
};
const DomainStatus = {
    Active: "active",
    Inactive: "inactive",
};
class InvalidAssociation extends CloudFrontServiceException {
    name = "InvalidAssociation";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "InvalidAssociation",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidAssociation.prototype);
        this.Message = opts.Message;
    }
}
const Format = {
    URLEncoded: "URLEncoded",
};
class FieldLevelEncryptionConfigAlreadyExists extends CloudFrontServiceException {
    name = "FieldLevelEncryptionConfigAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "FieldLevelEncryptionConfigAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FieldLevelEncryptionConfigAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchFieldLevelEncryptionProfile extends CloudFrontServiceException {
    name = "NoSuchFieldLevelEncryptionProfile";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchFieldLevelEncryptionProfile",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchFieldLevelEncryptionProfile.prototype);
        this.Message = opts.Message;
    }
}
class QueryArgProfileEmpty extends CloudFrontServiceException {
    name = "QueryArgProfileEmpty";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "QueryArgProfileEmpty",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, QueryArgProfileEmpty.prototype);
        this.Message = opts.Message;
    }
}
class TooManyFieldLevelEncryptionConfigs extends CloudFrontServiceException {
    name = "TooManyFieldLevelEncryptionConfigs";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyFieldLevelEncryptionConfigs",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyFieldLevelEncryptionConfigs.prototype);
        this.Message = opts.Message;
    }
}
class TooManyFieldLevelEncryptionContentTypeProfiles extends CloudFrontServiceException {
    name = "TooManyFieldLevelEncryptionContentTypeProfiles";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyFieldLevelEncryptionContentTypeProfiles",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyFieldLevelEncryptionContentTypeProfiles.prototype);
        this.Message = opts.Message;
    }
}
class TooManyFieldLevelEncryptionQueryArgProfiles extends CloudFrontServiceException {
    name = "TooManyFieldLevelEncryptionQueryArgProfiles";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyFieldLevelEncryptionQueryArgProfiles",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyFieldLevelEncryptionQueryArgProfiles.prototype);
        this.Message = opts.Message;
    }
}
class FieldLevelEncryptionProfileAlreadyExists extends CloudFrontServiceException {
    name = "FieldLevelEncryptionProfileAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "FieldLevelEncryptionProfileAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FieldLevelEncryptionProfileAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class FieldLevelEncryptionProfileSizeExceeded extends CloudFrontServiceException {
    name = "FieldLevelEncryptionProfileSizeExceeded";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "FieldLevelEncryptionProfileSizeExceeded",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FieldLevelEncryptionProfileSizeExceeded.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchPublicKey extends CloudFrontServiceException {
    name = "NoSuchPublicKey";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchPublicKey",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchPublicKey.prototype);
        this.Message = opts.Message;
    }
}
class TooManyFieldLevelEncryptionEncryptionEntities extends CloudFrontServiceException {
    name = "TooManyFieldLevelEncryptionEncryptionEntities";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyFieldLevelEncryptionEncryptionEntities",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyFieldLevelEncryptionEncryptionEntities.prototype);
        this.Message = opts.Message;
    }
}
class TooManyFieldLevelEncryptionFieldPatterns extends CloudFrontServiceException {
    name = "TooManyFieldLevelEncryptionFieldPatterns";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyFieldLevelEncryptionFieldPatterns",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyFieldLevelEncryptionFieldPatterns.prototype);
        this.Message = opts.Message;
    }
}
class TooManyFieldLevelEncryptionProfiles extends CloudFrontServiceException {
    name = "TooManyFieldLevelEncryptionProfiles";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyFieldLevelEncryptionProfiles",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyFieldLevelEncryptionProfiles.prototype);
        this.Message = opts.Message;
    }
}
const FunctionRuntime = {
    cloudfront_js_1_0: "cloudfront-js-1.0",
    cloudfront_js_2_0: "cloudfront-js-2.0",
};
const FunctionStage = {
    DEVELOPMENT: "DEVELOPMENT",
    LIVE: "LIVE",
};
class FunctionAlreadyExists extends CloudFrontServiceException {
    name = "FunctionAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "FunctionAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FunctionAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class FunctionSizeLimitExceeded extends CloudFrontServiceException {
    name = "FunctionSizeLimitExceeded";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "FunctionSizeLimitExceeded",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FunctionSizeLimitExceeded.prototype);
        this.Message = opts.Message;
    }
}
class TooManyFunctions extends CloudFrontServiceException {
    name = "TooManyFunctions";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyFunctions",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyFunctions.prototype);
        this.Message = opts.Message;
    }
}
class TooManyInvalidationsInProgress extends CloudFrontServiceException {
    name = "TooManyInvalidationsInProgress";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyInvalidationsInProgress",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyInvalidationsInProgress.prototype);
        this.Message = opts.Message;
    }
}
class KeyGroupAlreadyExists extends CloudFrontServiceException {
    name = "KeyGroupAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "KeyGroupAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, KeyGroupAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class TooManyKeyGroups extends CloudFrontServiceException {
    name = "TooManyKeyGroups";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyKeyGroups",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyKeyGroups.prototype);
        this.Message = opts.Message;
    }
}
class TooManyPublicKeysInKeyGroup extends CloudFrontServiceException {
    name = "TooManyPublicKeysInKeyGroup";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyPublicKeysInKeyGroup",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyPublicKeysInKeyGroup.prototype);
        this.Message = opts.Message;
    }
}
const ImportSourceType = {
    S3: "S3",
};
class EntitySizeLimitExceeded extends CloudFrontServiceException {
    name = "EntitySizeLimitExceeded";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "EntitySizeLimitExceeded",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, EntitySizeLimitExceeded.prototype);
        this.Message = opts.Message;
    }
}
const RealtimeMetricsSubscriptionStatus = {
    Disabled: "Disabled",
    Enabled: "Enabled",
};
class MonitoringSubscriptionAlreadyExists extends CloudFrontServiceException {
    name = "MonitoringSubscriptionAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "MonitoringSubscriptionAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, MonitoringSubscriptionAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
const OriginAccessControlOriginTypes = {
    lambda: "lambda",
    mediapackagev2: "mediapackagev2",
    mediastore: "mediastore",
    s3: "s3",
};
const OriginAccessControlSigningBehaviors = {
    always: "always",
    never: "never",
    no_override: "no-override",
};
const OriginAccessControlSigningProtocols = {
    sigv4: "sigv4",
};
const OriginCustomHeaderFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.HeaderValue && { HeaderValue: smithyClient.SENSITIVE_STRING }),
});
const CustomHeadersFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Items && { Items: obj.Items.map((item) => OriginCustomHeaderFilterSensitiveLog(item)) }),
});
const OriginFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.CustomHeaders && { CustomHeaders: CustomHeadersFilterSensitiveLog(obj.CustomHeaders) }),
});
const OriginsFilterSensitiveLog = (obj) => ({
    ...obj,
});
const StringSchemaConfigFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Comment && { Comment: smithyClient.SENSITIVE_STRING }),
});
const ParameterDefinitionSchemaFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.StringSchema && { StringSchema: StringSchemaConfigFilterSensitiveLog(obj.StringSchema) }),
});
const ParameterDefinitionFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Definition && { Definition: ParameterDefinitionSchemaFilterSensitiveLog(obj.Definition) }),
});
const TenantConfigFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.ParameterDefinitions && {
        ParameterDefinitions: obj.ParameterDefinitions.map((item) => ParameterDefinitionFilterSensitiveLog(item)),
    }),
});
const DistributionConfigFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Comment && { Comment: smithyClient.SENSITIVE_STRING }),
    ...(obj.TenantConfig && { TenantConfig: TenantConfigFilterSensitiveLog(obj.TenantConfig) }),
});
const DistributionFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.DistributionConfig && { DistributionConfig: DistributionConfigFilterSensitiveLog(obj.DistributionConfig) }),
});
const CopyDistributionResultFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Distribution && { Distribution: DistributionFilterSensitiveLog(obj.Distribution) }),
});
const CreateDistributionRequestFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.DistributionConfig && { DistributionConfig: DistributionConfigFilterSensitiveLog(obj.DistributionConfig) }),
});
const CreateDistributionResultFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Distribution && { Distribution: DistributionFilterSensitiveLog(obj.Distribution) }),
});
const DistributionConfigWithTagsFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.DistributionConfig && { DistributionConfig: DistributionConfigFilterSensitiveLog(obj.DistributionConfig) }),
});
const CreateDistributionWithTagsRequestFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.DistributionConfigWithTags && {
        DistributionConfigWithTags: DistributionConfigWithTagsFilterSensitiveLog(obj.DistributionConfigWithTags),
    }),
});
const CreateDistributionWithTagsResultFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Distribution && { Distribution: DistributionFilterSensitiveLog(obj.Distribution) }),
});
const CreateFunctionRequestFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.FunctionCode && { FunctionCode: smithyClient.SENSITIVE_STRING }),
});

class OriginAccessControlAlreadyExists extends CloudFrontServiceException {
    name = "OriginAccessControlAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "OriginAccessControlAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, OriginAccessControlAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class TooManyOriginAccessControls extends CloudFrontServiceException {
    name = "TooManyOriginAccessControls";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyOriginAccessControls",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyOriginAccessControls.prototype);
        this.Message = opts.Message;
    }
}
const OriginRequestPolicyCookieBehavior = {
    all: "all",
    allExcept: "allExcept",
    none: "none",
    whitelist: "whitelist",
};
const OriginRequestPolicyHeaderBehavior = {
    allExcept: "allExcept",
    allViewer: "allViewer",
    allViewerAndWhitelistCloudFront: "allViewerAndWhitelistCloudFront",
    none: "none",
    whitelist: "whitelist",
};
const OriginRequestPolicyQueryStringBehavior = {
    all: "all",
    allExcept: "allExcept",
    none: "none",
    whitelist: "whitelist",
};
class OriginRequestPolicyAlreadyExists extends CloudFrontServiceException {
    name = "OriginRequestPolicyAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "OriginRequestPolicyAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, OriginRequestPolicyAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class TooManyCookiesInOriginRequestPolicy extends CloudFrontServiceException {
    name = "TooManyCookiesInOriginRequestPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyCookiesInOriginRequestPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyCookiesInOriginRequestPolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyHeadersInOriginRequestPolicy extends CloudFrontServiceException {
    name = "TooManyHeadersInOriginRequestPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyHeadersInOriginRequestPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyHeadersInOriginRequestPolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyOriginRequestPolicies extends CloudFrontServiceException {
    name = "TooManyOriginRequestPolicies";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyOriginRequestPolicies",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyOriginRequestPolicies.prototype);
        this.Message = opts.Message;
    }
}
class TooManyQueryStringsInOriginRequestPolicy extends CloudFrontServiceException {
    name = "TooManyQueryStringsInOriginRequestPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyQueryStringsInOriginRequestPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyQueryStringsInOriginRequestPolicy.prototype);
        this.Message = opts.Message;
    }
}
class PublicKeyAlreadyExists extends CloudFrontServiceException {
    name = "PublicKeyAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "PublicKeyAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, PublicKeyAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class TooManyPublicKeys extends CloudFrontServiceException {
    name = "TooManyPublicKeys";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyPublicKeys",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyPublicKeys.prototype);
        this.Message = opts.Message;
    }
}
class RealtimeLogConfigAlreadyExists extends CloudFrontServiceException {
    name = "RealtimeLogConfigAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "RealtimeLogConfigAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, RealtimeLogConfigAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class TooManyRealtimeLogConfigs extends CloudFrontServiceException {
    name = "TooManyRealtimeLogConfigs";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyRealtimeLogConfigs",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyRealtimeLogConfigs.prototype);
        this.Message = opts.Message;
    }
}
const FrameOptionsList = {
    DENY: "DENY",
    SAMEORIGIN: "SAMEORIGIN",
};
const ReferrerPolicyList = {
    no_referrer: "no-referrer",
    no_referrer_when_downgrade: "no-referrer-when-downgrade",
    origin: "origin",
    origin_when_cross_origin: "origin-when-cross-origin",
    same_origin: "same-origin",
    strict_origin: "strict-origin",
    strict_origin_when_cross_origin: "strict-origin-when-cross-origin",
    unsafe_url: "unsafe-url",
};
class ResponseHeadersPolicyAlreadyExists extends CloudFrontServiceException {
    name = "ResponseHeadersPolicyAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "ResponseHeadersPolicyAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResponseHeadersPolicyAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class TooLongCSPInResponseHeadersPolicy extends CloudFrontServiceException {
    name = "TooLongCSPInResponseHeadersPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooLongCSPInResponseHeadersPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooLongCSPInResponseHeadersPolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyCustomHeadersInResponseHeadersPolicy extends CloudFrontServiceException {
    name = "TooManyCustomHeadersInResponseHeadersPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyCustomHeadersInResponseHeadersPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyCustomHeadersInResponseHeadersPolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyRemoveHeadersInResponseHeadersPolicy extends CloudFrontServiceException {
    name = "TooManyRemoveHeadersInResponseHeadersPolicy";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyRemoveHeadersInResponseHeadersPolicy",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyRemoveHeadersInResponseHeadersPolicy.prototype);
        this.Message = opts.Message;
    }
}
class TooManyResponseHeadersPolicies extends CloudFrontServiceException {
    name = "TooManyResponseHeadersPolicies";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyResponseHeadersPolicies",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyResponseHeadersPolicies.prototype);
        this.Message = opts.Message;
    }
}
class StreamingDistributionAlreadyExists extends CloudFrontServiceException {
    name = "StreamingDistributionAlreadyExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "StreamingDistributionAlreadyExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, StreamingDistributionAlreadyExists.prototype);
        this.Message = opts.Message;
    }
}
class TooManyStreamingDistributionCNAMEs extends CloudFrontServiceException {
    name = "TooManyStreamingDistributionCNAMEs";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyStreamingDistributionCNAMEs",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyStreamingDistributionCNAMEs.prototype);
        this.Message = opts.Message;
    }
}
class TooManyStreamingDistributions extends CloudFrontServiceException {
    name = "TooManyStreamingDistributions";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "TooManyStreamingDistributions",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyStreamingDistributions.prototype);
        this.Message = opts.Message;
    }
}
class IllegalDelete extends CloudFrontServiceException {
    name = "IllegalDelete";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "IllegalDelete",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, IllegalDelete.prototype);
        this.Message = opts.Message;
    }
}
class CloudFrontOriginAccessIdentityInUse extends CloudFrontServiceException {
    name = "CloudFrontOriginAccessIdentityInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "CloudFrontOriginAccessIdentityInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, CloudFrontOriginAccessIdentityInUse.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchCloudFrontOriginAccessIdentity extends CloudFrontServiceException {
    name = "NoSuchCloudFrontOriginAccessIdentity";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchCloudFrontOriginAccessIdentity",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchCloudFrontOriginAccessIdentity.prototype);
        this.Message = opts.Message;
    }
}
class ResourceNotDisabled extends CloudFrontServiceException {
    name = "ResourceNotDisabled";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "ResourceNotDisabled",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResourceNotDisabled.prototype);
        this.Message = opts.Message;
    }
}
class DistributionNotDisabled extends CloudFrontServiceException {
    name = "DistributionNotDisabled";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "DistributionNotDisabled",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, DistributionNotDisabled.prototype);
        this.Message = opts.Message;
    }
}
class ResourceInUse extends CloudFrontServiceException {
    name = "ResourceInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "ResourceInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResourceInUse.prototype);
        this.Message = opts.Message;
    }
}
class FieldLevelEncryptionConfigInUse extends CloudFrontServiceException {
    name = "FieldLevelEncryptionConfigInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "FieldLevelEncryptionConfigInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FieldLevelEncryptionConfigInUse.prototype);
        this.Message = opts.Message;
    }
}
class FieldLevelEncryptionProfileInUse extends CloudFrontServiceException {
    name = "FieldLevelEncryptionProfileInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "FieldLevelEncryptionProfileInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FieldLevelEncryptionProfileInUse.prototype);
        this.Message = opts.Message;
    }
}
class FunctionInUse extends CloudFrontServiceException {
    name = "FunctionInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "FunctionInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, FunctionInUse.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchFunctionExists extends CloudFrontServiceException {
    name = "NoSuchFunctionExists";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchFunctionExists",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchFunctionExists.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchResource extends CloudFrontServiceException {
    name = "NoSuchResource";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchResource",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchResource.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchMonitoringSubscription extends CloudFrontServiceException {
    name = "NoSuchMonitoringSubscription";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchMonitoringSubscription",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchMonitoringSubscription.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchOriginAccessControl extends CloudFrontServiceException {
    name = "NoSuchOriginAccessControl";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchOriginAccessControl",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchOriginAccessControl.prototype);
        this.Message = opts.Message;
    }
}
class OriginAccessControlInUse extends CloudFrontServiceException {
    name = "OriginAccessControlInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "OriginAccessControlInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, OriginAccessControlInUse.prototype);
        this.Message = opts.Message;
    }
}
class OriginRequestPolicyInUse extends CloudFrontServiceException {
    name = "OriginRequestPolicyInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "OriginRequestPolicyInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, OriginRequestPolicyInUse.prototype);
        this.Message = opts.Message;
    }
}
class PublicKeyInUse extends CloudFrontServiceException {
    name = "PublicKeyInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "PublicKeyInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, PublicKeyInUse.prototype);
        this.Message = opts.Message;
    }
}
class RealtimeLogConfigInUse extends CloudFrontServiceException {
    name = "RealtimeLogConfigInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "RealtimeLogConfigInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, RealtimeLogConfigInUse.prototype);
        this.Message = opts.Message;
    }
}
class ResponseHeadersPolicyInUse extends CloudFrontServiceException {
    name = "ResponseHeadersPolicyInUse";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "ResponseHeadersPolicyInUse",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ResponseHeadersPolicyInUse.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchStreamingDistribution extends CloudFrontServiceException {
    name = "NoSuchStreamingDistribution";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchStreamingDistribution",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchStreamingDistribution.prototype);
        this.Message = opts.Message;
    }
}
class StreamingDistributionNotDisabled extends CloudFrontServiceException {
    name = "StreamingDistributionNotDisabled";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "StreamingDistributionNotDisabled",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, StreamingDistributionNotDisabled.prototype);
        this.Message = opts.Message;
    }
}
class NoSuchInvalidation extends CloudFrontServiceException {
    name = "NoSuchInvalidation";
    $fault = "client";
    Message;
    constructor(opts) {
        super({
            name: "NoSuchInvalidation",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NoSuchInvalidation.prototype);
        this.Message = opts.Message;
    }
}
const ManagedCertificateStatus = {
    Expired: "expired",
    Failed: "failed",
    Inactive: "inactive",
    Issued: "issued",
    PendingValidation: "pending-validation",
    Revoked: "revoked",
    ValidationTimedOut: "validation-timed-out",
};
const DistributionResourceType = {
    Distribution: "distribution",
    DistributionTenant: "distribution-tenant",
};
const OriginRequestPolicyType = {
    custom: "custom",
    managed: "managed",
};
const ResponseHeadersPolicyType = {
    custom: "custom",
    managed: "managed",
};
const GetDistributionResultFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Distribution && { Distribution: DistributionFilterSensitiveLog(obj.Distribution) }),
});
const GetDistributionConfigResultFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.DistributionConfig && { DistributionConfig: DistributionConfigFilterSensitiveLog(obj.DistributionConfig) }),
});
const GetFunctionResultFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.FunctionCode && { FunctionCode: smithyClient.SENSITIVE_STRING }),
});
const DistributionSummaryFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Comment && { Comment: smithyClient.SENSITIVE_STRING }),
});
const DistributionListFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Items && { Items: obj.Items.map((item) => DistributionSummaryFilterSensitiveLog(item)) }),
});
const ListDistributionsResultFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.DistributionList && { DistributionList: DistributionListFilterSensitiveLog(obj.DistributionList) }),
});
const ListDistributionsByAnycastIpListIdResultFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.DistributionList && { DistributionList: DistributionListFilterSensitiveLog(obj.DistributionList) }),
});
const ListDistributionsByConnectionModeResultFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.DistributionList && { DistributionList: DistributionListFilterSensitiveLog(obj.DistributionList) }),
});
const ListDistributionsByRealtimeLogConfigResultFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.DistributionList && { DistributionList: DistributionListFilterSensitiveLog(obj.DistributionList) }),
});
const ListDistributionsByWebACLIdResultFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.DistributionList && { DistributionList: DistributionListFilterSensitiveLog(obj.DistributionList) }),
});

class TestFunctionFailed extends CloudFrontServiceException {
    name = "TestFunctionFailed";
    $fault = "server";
    Message;
    constructor(opts) {
        super({
            name: "TestFunctionFailed",
            $fault: "server",
            ...opts,
        });
        Object.setPrototypeOf(this, TestFunctionFailed.prototype);
        this.Message = opts.Message;
    }
}
const DnsConfigurationStatus = {
    Invalid: "invalid-configuration",
    Unknown: "unknown-configuration",
    Valid: "valid-configuration",
};
const TestFunctionRequestFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.EventObject && { EventObject: smithyClient.SENSITIVE_STRING }),
});
const TestResultFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.FunctionExecutionLogs && { FunctionExecutionLogs: smithyClient.SENSITIVE_STRING }),
    ...(obj.FunctionErrorMessage && { FunctionErrorMessage: smithyClient.SENSITIVE_STRING }),
    ...(obj.FunctionOutput && { FunctionOutput: smithyClient.SENSITIVE_STRING }),
});
const TestFunctionResultFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.TestResult && { TestResult: TestResultFilterSensitiveLog(obj.TestResult) }),
});
const UpdateDistributionRequestFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.DistributionConfig && { DistributionConfig: DistributionConfigFilterSensitiveLog(obj.DistributionConfig) }),
});
const UpdateDistributionResultFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Distribution && { Distribution: DistributionFilterSensitiveLog(obj.Distribution) }),
});
const UpdateDistributionWithStagingConfigResultFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.Distribution && { Distribution: DistributionFilterSensitiveLog(obj.Distribution) }),
});
const UpdateFunctionRequestFilterSensitiveLog = (obj) => ({
    ...obj,
    ...(obj.FunctionCode && { FunctionCode: smithyClient.SENSITIVE_STRING }),
});

const se_AssociateAliasCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distribution/{TargetDistributionId}/associate-alias");
    b.p("TargetDistributionId", () => input.TargetDistributionId, "{TargetDistributionId}", false);
    const query = smithyClient.map({
        [_A]: [, smithyClient.expectNonNull(input[_A], `Alias`)],
    });
    let body;
    b.m("PUT").h(headers).q(query).b(body);
    return b.build();
};
const se_AssociateDistributionTenantWebACLCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/distribution-tenant/{Id}/associate-web-acl");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_ADTWACLR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_WACLA] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_WACLA]).n(_WACLA));
    }
    body += bn.toString();
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_AssociateDistributionWebACLCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/distribution/{Id}/associate-web-acl");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_ADWACLR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_WACLA] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_WACLA]).n(_WACLA));
    }
    body += bn.toString();
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_CopyDistributionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_st]: [() => smithyClient.isSerializableHeaderValue(input[_S]), () => input[_S].toString()],
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/distribution/{PrimaryDistributionId}/copy");
    b.p("PrimaryDistributionId", () => input.PrimaryDistributionId, "{PrimaryDistributionId}", false);
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_CDR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_CR] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_CR]).n(_CR));
    }
    if (input[_E] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_E])).n(_E));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateAnycastIpListCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/anycast-ip-list");
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_CAILR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_IC] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_IC])).n(_IC));
    }
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_AILN, input[_N]).n(_N));
    }
    if (input[_T] != null) {
        bn.c(se_Tags(input[_T]).n(_T));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateCachePolicyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/cache-policy");
    let body;
    let contents;
    if (input.CachePolicyConfig !== undefined) {
        contents = se_CachePolicyConfig(input.CachePolicyConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateCloudFrontOriginAccessIdentityCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/origin-access-identity/cloudfront");
    let body;
    let contents;
    if (input.CloudFrontOriginAccessIdentityConfig !== undefined) {
        contents = se_CloudFrontOriginAccessIdentityConfig(input.CloudFrontOriginAccessIdentityConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateConnectionGroupCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/connection-group");
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_CCGR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_AILI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_AILI]).n(_AILI));
    }
    if (input[_E] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_E])).n(_E));
    }
    if (input[_IE] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_IE])).n(_IE));
    }
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_N]).n(_N));
    }
    if (input[_T] != null) {
        bn.c(se_Tags(input[_T]).n(_T));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateContinuousDeploymentPolicyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/continuous-deployment-policy");
    let body;
    let contents;
    if (input.ContinuousDeploymentPolicyConfig !== undefined) {
        contents = se_ContinuousDeploymentPolicyConfig(input.ContinuousDeploymentPolicyConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateDistributionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/distribution");
    let body;
    let contents;
    if (input.DistributionConfig !== undefined) {
        contents = se_DistributionConfig(input.DistributionConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateDistributionTenantCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/distribution-tenant");
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_CDTR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_CGI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_CGI]).n(_CGI));
    }
    if (input[_C] != null) {
        bn.c(se_Customizations(input[_C]).n(_C));
    }
    if (input[_DI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_DI]).n(_DI));
    }
    bn.lc(input, "Domains", "Domains", () => se_DomainList(input[_D]));
    if (input[_E] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_E])).n(_E));
    }
    if (input[_MCR] != null) {
        bn.c(se_ManagedCertificateRequest(input[_MCR]).n(_MCR));
    }
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_N]).n(_N));
    }
    bn.lc(input, "Parameters", "Parameters", () => se_Parameters(input[_P]));
    if (input[_T] != null) {
        bn.c(se_Tags(input[_T]).n(_T));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateDistributionWithTagsCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/distribution");
    const query = smithyClient.map({
        [_WT]: [, ""],
    });
    let body;
    let contents;
    if (input.DistributionConfigWithTags !== undefined) {
        contents = se_DistributionConfigWithTags(input.DistributionConfigWithTags);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).q(query).b(body);
    return b.build();
};
const se_CreateFieldLevelEncryptionConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/field-level-encryption");
    let body;
    let contents;
    if (input.FieldLevelEncryptionConfig !== undefined) {
        contents = se_FieldLevelEncryptionConfig(input.FieldLevelEncryptionConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateFieldLevelEncryptionProfileCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/field-level-encryption-profile");
    let body;
    let contents;
    if (input.FieldLevelEncryptionProfileConfig !== undefined) {
        contents = se_FieldLevelEncryptionProfileConfig(input.FieldLevelEncryptionProfileConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateFunctionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/function");
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_CFR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_FC] != null) {
        bn.c(xmlBuilder.XmlNode.of(_FB, context.base64Encoder(input[_FC])).n(_FC));
    }
    if (input[_FCu] != null) {
        bn.c(se_FunctionConfig(input[_FCu]).n(_FCu));
    }
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_FN, input[_N]).n(_N));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateInvalidationCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/distribution/{DistributionId}/invalidation");
    b.p("DistributionId", () => input.DistributionId, "{DistributionId}", false);
    let body;
    let contents;
    if (input.InvalidationBatch !== undefined) {
        contents = se_InvalidationBatch(input.InvalidationBatch);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateInvalidationForDistributionTenantCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/distribution-tenant/{Id}/invalidation");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    let contents;
    if (input.InvalidationBatch !== undefined) {
        contents = se_InvalidationBatch(input.InvalidationBatch);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateKeyGroupCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/key-group");
    let body;
    let contents;
    if (input.KeyGroupConfig !== undefined) {
        contents = se_KeyGroupConfig(input.KeyGroupConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateKeyValueStoreCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/key-value-store");
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_CKVSR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_Co] != null) {
        bn.c(xmlBuilder.XmlNode.of(_KVSC, input[_Co]).n(_Co));
    }
    if (input[_IS] != null) {
        bn.c(se_ImportSource(input[_IS]).n(_IS));
    }
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_KVSN, input[_N]).n(_N));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateMonitoringSubscriptionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/distributions/{DistributionId}/monitoring-subscription");
    b.p("DistributionId", () => input.DistributionId, "{DistributionId}", false);
    let body;
    let contents;
    if (input.MonitoringSubscription !== undefined) {
        contents = se_MonitoringSubscription(input.MonitoringSubscription);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateOriginAccessControlCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/origin-access-control");
    let body;
    let contents;
    if (input.OriginAccessControlConfig !== undefined) {
        contents = se_OriginAccessControlConfig(input.OriginAccessControlConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateOriginRequestPolicyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/origin-request-policy");
    let body;
    let contents;
    if (input.OriginRequestPolicyConfig !== undefined) {
        contents = se_OriginRequestPolicyConfig(input.OriginRequestPolicyConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreatePublicKeyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/public-key");
    let body;
    let contents;
    if (input.PublicKeyConfig !== undefined) {
        contents = se_PublicKeyConfig(input.PublicKeyConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateRealtimeLogConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/realtime-log-config");
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_CRLCR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    bn.lc(input, "EndPoints", "EndPoints", () => se_EndPointList(input[_EP]));
    bn.lc(input, "Fields", "Fields", () => se_FieldList(input[_F]));
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_N]).n(_N));
    }
    if (input[_SR] != null) {
        bn.c(xmlBuilder.XmlNode.of(_l, String(input[_SR])).n(_SR));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateResponseHeadersPolicyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/response-headers-policy");
    let body;
    let contents;
    if (input.ResponseHeadersPolicyConfig !== undefined) {
        contents = se_ResponseHeadersPolicyConfig(input.ResponseHeadersPolicyConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateStreamingDistributionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/streaming-distribution");
    let body;
    let contents;
    if (input.StreamingDistributionConfig !== undefined) {
        contents = se_StreamingDistributionConfig(input.StreamingDistributionConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_CreateStreamingDistributionWithTagsCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/streaming-distribution");
    const query = smithyClient.map({
        [_WT]: [, ""],
    });
    let body;
    let contents;
    if (input.StreamingDistributionConfigWithTags !== undefined) {
        contents = se_StreamingDistributionConfigWithTags(input.StreamingDistributionConfigWithTags);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).q(query).b(body);
    return b.build();
};
const se_CreateVpcOriginCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/vpc-origin");
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_CVOR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_T] != null) {
        bn.c(se_Tags(input[_T]).n(_T));
    }
    if (input[_VOEC] != null) {
        bn.c(se_VpcOriginEndpointConfig(input[_VOEC]).n(_VOEC));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_DeleteAnycastIpListCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/anycast-ip-list/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeleteCachePolicyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/cache-policy/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeleteCloudFrontOriginAccessIdentityCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/origin-access-identity/cloudfront/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeleteConnectionGroupCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/connection-group/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeleteContinuousDeploymentPolicyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/continuous-deployment-policy/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeleteDistributionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/distribution/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeleteDistributionTenantCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/distribution-tenant/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeleteFieldLevelEncryptionConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/field-level-encryption/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeleteFieldLevelEncryptionProfileCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/field-level-encryption-profile/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeleteFunctionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/function/{Name}");
    b.p("Name", () => input.Name, "{Name}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeleteKeyGroupCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/key-group/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeleteKeyValueStoreCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/key-value-store/{Name}");
    b.p("Name", () => input.Name, "{Name}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeleteMonitoringSubscriptionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distributions/{DistributionId}/monitoring-subscription");
    b.p("DistributionId", () => input.DistributionId, "{DistributionId}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeleteOriginAccessControlCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/origin-access-control/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeleteOriginRequestPolicyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/origin-request-policy/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeletePublicKeyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/public-key/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeleteRealtimeLogConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/delete-realtime-log-config");
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_DRLCR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_ARN] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_ARN]).n(_ARN));
    }
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_N]).n(_N));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_DeleteResponseHeadersPolicyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/response-headers-policy/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeleteStreamingDistributionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/streaming-distribution/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DeleteVpcOriginCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/vpc-origin/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("DELETE").h(headers).b(body);
    return b.build();
};
const se_DescribeFunctionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/function/{Name}/describe");
    b.p("Name", () => input.Name, "{Name}", false);
    const query = smithyClient.map({
        [_St]: [, input[_St]],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_DescribeKeyValueStoreCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/key-value-store/{Name}");
    b.p("Name", () => input.Name, "{Name}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_DisassociateDistributionTenantWebACLCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/distribution-tenant/{Id}/disassociate-web-acl");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_DisassociateDistributionWebACLCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/distribution/{Id}/disassociate-web-acl");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_GetAnycastIpListCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/anycast-ip-list/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetCachePolicyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/cache-policy/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetCachePolicyConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/cache-policy/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetCloudFrontOriginAccessIdentityCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/origin-access-identity/cloudfront/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetCloudFrontOriginAccessIdentityConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/origin-access-identity/cloudfront/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetConnectionGroupCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/connection-group/{Identifier}");
    b.p("Identifier", () => input.Identifier, "{Identifier}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetConnectionGroupByRoutingEndpointCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/connection-group");
    const query = smithyClient.map({
        [_RE]: [, smithyClient.expectNonNull(input[_RE], `RoutingEndpoint`)],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_GetContinuousDeploymentPolicyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/continuous-deployment-policy/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetContinuousDeploymentPolicyConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/continuous-deployment-policy/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetDistributionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distribution/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetDistributionConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distribution/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetDistributionTenantCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distribution-tenant/{Identifier}");
    b.p("Identifier", () => input.Identifier, "{Identifier}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetDistributionTenantByDomainCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distribution-tenant");
    const query = smithyClient.map({
        [_d]: [, smithyClient.expectNonNull(input[_Do], `Domain`)],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_GetFieldLevelEncryptionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/field-level-encryption/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetFieldLevelEncryptionConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/field-level-encryption/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetFieldLevelEncryptionProfileCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/field-level-encryption-profile/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetFieldLevelEncryptionProfileConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/field-level-encryption-profile/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetFunctionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/function/{Name}");
    b.p("Name", () => input.Name, "{Name}", false);
    const query = smithyClient.map({
        [_St]: [, input[_St]],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_GetInvalidationCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distribution/{DistributionId}/invalidation/{Id}");
    b.p("DistributionId", () => input.DistributionId, "{DistributionId}", false);
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetInvalidationForDistributionTenantCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distribution-tenant/{DistributionTenantId}/invalidation/{Id}");
    b.p("DistributionTenantId", () => input.DistributionTenantId, "{DistributionTenantId}", false);
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetKeyGroupCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/key-group/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetKeyGroupConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/key-group/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetManagedCertificateDetailsCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/managed-certificate/{Identifier}");
    b.p("Identifier", () => input.Identifier, "{Identifier}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetMonitoringSubscriptionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distributions/{DistributionId}/monitoring-subscription");
    b.p("DistributionId", () => input.DistributionId, "{DistributionId}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetOriginAccessControlCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/origin-access-control/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetOriginAccessControlConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/origin-access-control/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetOriginRequestPolicyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/origin-request-policy/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetOriginRequestPolicyConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/origin-request-policy/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetPublicKeyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/public-key/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetPublicKeyConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/public-key/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetRealtimeLogConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/get-realtime-log-config");
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_GRLCR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_ARN] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_ARN]).n(_ARN));
    }
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_N]).n(_N));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_GetResponseHeadersPolicyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/response-headers-policy/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetResponseHeadersPolicyConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/response-headers-policy/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetStreamingDistributionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/streaming-distribution/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetStreamingDistributionConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/streaming-distribution/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_GetVpcOriginCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/vpc-origin/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    b.m("GET").h(headers).b(body);
    return b.build();
};
const se_ListAnycastIpListsCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/anycast-ip-list");
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListCachePoliciesCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/cache-policy");
    const query = smithyClient.map({
        [_Ty]: [, input[_Ty]],
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListCloudFrontOriginAccessIdentitiesCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/origin-access-identity/cloudfront");
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListConflictingAliasesCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/conflicting-alias");
    const query = smithyClient.map({
        [_DI]: [, smithyClient.expectNonNull(input[_DI], `DistributionId`)],
        [_A]: [, smithyClient.expectNonNull(input[_A], `Alias`)],
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListConnectionGroupsCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/connection-groups");
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_LCGR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_AF] != null) {
        bn.c(se_ConnectionGroupAssociationFilter(input[_AF]).n(_AF));
    }
    if (input[_M] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_M]).n(_M));
    }
    if (input[_MI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_MI])).n(_MI));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_ListContinuousDeploymentPoliciesCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/continuous-deployment-policy");
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListDistributionsCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distribution");
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListDistributionsByAnycastIpListIdCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distributionsByAnycastIpListId/{AnycastIpListId}");
    b.p("AnycastIpListId", () => input.AnycastIpListId, "{AnycastIpListId}", false);
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListDistributionsByCachePolicyIdCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distributionsByCachePolicyId/{CachePolicyId}");
    b.p("CachePolicyId", () => input.CachePolicyId, "{CachePolicyId}", false);
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListDistributionsByConnectionModeCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distributionsByConnectionMode/{ConnectionMode}");
    b.p("ConnectionMode", () => input.ConnectionMode, "{ConnectionMode}", false);
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListDistributionsByKeyGroupCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distributionsByKeyGroupId/{KeyGroupId}");
    b.p("KeyGroupId", () => input.KeyGroupId, "{KeyGroupId}", false);
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListDistributionsByOriginRequestPolicyIdCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distributionsByOriginRequestPolicyId/{OriginRequestPolicyId}");
    b.p("OriginRequestPolicyId", () => input.OriginRequestPolicyId, "{OriginRequestPolicyId}", false);
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListDistributionsByRealtimeLogConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/distributionsByRealtimeLogConfig");
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_LDBRLCR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_M] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_M]).n(_M));
    }
    if (input[_MI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_MI])).n(_MI));
    }
    if (input[_RLCA] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_RLCA]).n(_RLCA));
    }
    if (input[_RLCN] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_RLCN]).n(_RLCN));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_ListDistributionsByResponseHeadersPolicyIdCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distributionsByResponseHeadersPolicyId/{ResponseHeadersPolicyId}");
    b.p("ResponseHeadersPolicyId", () => input.ResponseHeadersPolicyId, "{ResponseHeadersPolicyId}", false);
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListDistributionsByVpcOriginIdCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distributionsByVpcOriginId/{VpcOriginId}");
    b.p("VpcOriginId", () => input.VpcOriginId, "{VpcOriginId}", false);
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListDistributionsByWebACLIdCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distributionsByWebACLId/{WebACLId}");
    b.p("WebACLId", () => input.WebACLId, "{WebACLId}", false);
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListDistributionTenantsCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/distribution-tenants");
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_LDTR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_AF] != null) {
        bn.c(se_DistributionTenantAssociationFilter(input[_AF]).n(_AF));
    }
    if (input[_M] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_M]).n(_M));
    }
    if (input[_MI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_MI])).n(_MI));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_ListDistributionTenantsByCustomizationCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/distribution-tenants-by-customization");
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_LDTBCR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_CA] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_CA]).n(_CA));
    }
    if (input[_M] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_M]).n(_M));
    }
    if (input[_MI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_MI])).n(_MI));
    }
    if (input[_WACLA] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_WACLA]).n(_WACLA));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_ListDomainConflictsCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/domain-conflicts");
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_LDCR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_Do] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Do]).n(_Do));
    }
    if (input[_DCVR] != null) {
        bn.c(se_DistributionResourceId(input[_DCVR]).n(_DCVR));
    }
    if (input[_M] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_M]).n(_M));
    }
    if (input[_MI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_MI])).n(_MI));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_ListFieldLevelEncryptionConfigsCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/field-level-encryption");
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListFieldLevelEncryptionProfilesCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/field-level-encryption-profile");
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListFunctionsCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/function");
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
        [_St]: [, input[_St]],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListInvalidationsCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distribution/{DistributionId}/invalidation");
    b.p("DistributionId", () => input.DistributionId, "{DistributionId}", false);
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListInvalidationsForDistributionTenantCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/distribution-tenant/{Id}/invalidation");
    b.p("Id", () => input.Id, "{Id}", false);
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListKeyGroupsCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/key-group");
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListKeyValueStoresCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/key-value-store");
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
        [_Sta]: [, input[_Sta]],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListOriginAccessControlsCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/origin-access-control");
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListOriginRequestPoliciesCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/origin-request-policy");
    const query = smithyClient.map({
        [_Ty]: [, input[_Ty]],
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListPublicKeysCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/public-key");
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListRealtimeLogConfigsCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/realtime-log-config");
    const query = smithyClient.map({
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
        [_M]: [, input[_M]],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListResponseHeadersPoliciesCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/response-headers-policy");
    const query = smithyClient.map({
        [_Ty]: [, input[_Ty]],
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListStreamingDistributionsCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/streaming-distribution");
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListTagsForResourceCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/tagging");
    const query = smithyClient.map({
        [_R]: [, smithyClient.expectNonNull(input[_R], `Resource`)],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_ListVpcOriginsCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {};
    b.bp("/2020-05-31/vpc-origin");
    const query = smithyClient.map({
        [_M]: [, input[_M]],
        [_MI]: [() => input.MaxItems !== void 0, () => input[_MI].toString()],
    });
    let body;
    b.m("GET").h(headers).q(query).b(body);
    return b.build();
};
const se_PublishFunctionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/function/{Name}/publish");
    b.p("Name", () => input.Name, "{Name}", false);
    let body;
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_TagResourceCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/tagging");
    const query = smithyClient.map({
        [_O]: [, "Tag"],
        [_R]: [, smithyClient.expectNonNull(input[_R], `Resource`)],
    });
    let body;
    let contents;
    if (input.Tags !== undefined) {
        contents = se_Tags(input.Tags);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).q(query).b(body);
    return b.build();
};
const se_TestFunctionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/function/{Name}/test");
    b.p("Name", () => input.Name, "{Name}", false);
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_TFR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_EO] != null) {
        bn.c(xmlBuilder.XmlNode.of(_FEO, context.base64Encoder(input[_EO])).n(_EO));
    }
    if (input[_St] != null) {
        bn.c(xmlBuilder.XmlNode.of(_FS, input[_St]).n(_St));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_UntagResourceCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/tagging");
    const query = smithyClient.map({
        [_O]: [, "Untag"],
        [_R]: [, smithyClient.expectNonNull(input[_R], `Resource`)],
    });
    let body;
    let contents;
    if (input.TagKeys !== undefined) {
        contents = se_TagKeys(input.TagKeys);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("POST").h(headers).q(query).b(body);
    return b.build();
};
const se_UpdateCachePolicyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/cache-policy/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    let contents;
    if (input.CachePolicyConfig !== undefined) {
        contents = se_CachePolicyConfig(input.CachePolicyConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_UpdateCloudFrontOriginAccessIdentityCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/origin-access-identity/cloudfront/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    let contents;
    if (input.CloudFrontOriginAccessIdentityConfig !== undefined) {
        contents = se_CloudFrontOriginAccessIdentityConfig(input.CloudFrontOriginAccessIdentityConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_UpdateConnectionGroupCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/connection-group/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_UCGR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_AILI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_AILI]).n(_AILI));
    }
    if (input[_E] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_E])).n(_E));
    }
    if (input[_IE] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_IE])).n(_IE));
    }
    body += bn.toString();
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_UpdateContinuousDeploymentPolicyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/continuous-deployment-policy/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    let contents;
    if (input.ContinuousDeploymentPolicyConfig !== undefined) {
        contents = se_ContinuousDeploymentPolicyConfig(input.ContinuousDeploymentPolicyConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_UpdateDistributionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/distribution/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    let contents;
    if (input.DistributionConfig !== undefined) {
        contents = se_DistributionConfig(input.DistributionConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_UpdateDistributionTenantCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/distribution-tenant/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_UDTR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_CGI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_CGI]).n(_CGI));
    }
    if (input[_C] != null) {
        bn.c(se_Customizations(input[_C]).n(_C));
    }
    if (input[_DI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_DI]).n(_DI));
    }
    bn.lc(input, "Domains", "Domains", () => se_DomainList(input[_D]));
    if (input[_E] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_E])).n(_E));
    }
    if (input[_MCR] != null) {
        bn.c(se_ManagedCertificateRequest(input[_MCR]).n(_MCR));
    }
    bn.lc(input, "Parameters", "Parameters", () => se_Parameters(input[_P]));
    body += bn.toString();
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_UpdateDistributionWithStagingConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/distribution/{Id}/promote-staging-config");
    b.p("Id", () => input.Id, "{Id}", false);
    const query = smithyClient.map({
        [_SDI]: [, input[_SDI]],
    });
    let body;
    b.m("PUT").h(headers).q(query).b(body);
    return b.build();
};
const se_UpdateDomainAssociationCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/domain-association");
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_UDAR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_Do] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Do]).n(_Do));
    }
    if (input[_TR] != null) {
        bn.c(se_DistributionResourceId(input[_TR]).n(_TR));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const se_UpdateFieldLevelEncryptionConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/field-level-encryption/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    let contents;
    if (input.FieldLevelEncryptionConfig !== undefined) {
        contents = se_FieldLevelEncryptionConfig(input.FieldLevelEncryptionConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_UpdateFieldLevelEncryptionProfileCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/field-level-encryption-profile/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    let contents;
    if (input.FieldLevelEncryptionProfileConfig !== undefined) {
        contents = se_FieldLevelEncryptionProfileConfig(input.FieldLevelEncryptionProfileConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_UpdateFunctionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/function/{Name}");
    b.p("Name", () => input.Name, "{Name}", false);
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_UFR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_FC] != null) {
        bn.c(xmlBuilder.XmlNode.of(_FB, context.base64Encoder(input[_FC])).n(_FC));
    }
    if (input[_FCu] != null) {
        bn.c(se_FunctionConfig(input[_FCu]).n(_FCu));
    }
    body += bn.toString();
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_UpdateKeyGroupCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/key-group/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    let contents;
    if (input.KeyGroupConfig !== undefined) {
        contents = se_KeyGroupConfig(input.KeyGroupConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_UpdateKeyValueStoreCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/key-value-store/{Name}");
    b.p("Name", () => input.Name, "{Name}", false);
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_UKVSR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_Co] != null) {
        bn.c(xmlBuilder.XmlNode.of(_KVSC, input[_Co]).n(_Co));
    }
    body += bn.toString();
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_UpdateOriginAccessControlCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/origin-access-control/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    let contents;
    if (input.OriginAccessControlConfig !== undefined) {
        contents = se_OriginAccessControlConfig(input.OriginAccessControlConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_UpdateOriginRequestPolicyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/origin-request-policy/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    let contents;
    if (input.OriginRequestPolicyConfig !== undefined) {
        contents = se_OriginRequestPolicyConfig(input.OriginRequestPolicyConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_UpdatePublicKeyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/public-key/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    let contents;
    if (input.PublicKeyConfig !== undefined) {
        contents = se_PublicKeyConfig(input.PublicKeyConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_UpdateRealtimeLogConfigCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/realtime-log-config");
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_URLCR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_ARN] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_ARN]).n(_ARN));
    }
    bn.lc(input, "EndPoints", "EndPoints", () => se_EndPointList(input[_EP]));
    bn.lc(input, "Fields", "Fields", () => se_FieldList(input[_F]));
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_N]).n(_N));
    }
    if (input[_SR] != null) {
        bn.c(xmlBuilder.XmlNode.of(_l, String(input[_SR])).n(_SR));
    }
    body += bn.toString();
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_UpdateResponseHeadersPolicyCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/response-headers-policy/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    let contents;
    if (input.ResponseHeadersPolicyConfig !== undefined) {
        contents = se_ResponseHeadersPolicyConfig(input.ResponseHeadersPolicyConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_UpdateStreamingDistributionCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/streaming-distribution/{Id}/config");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    let contents;
    if (input.StreamingDistributionConfig !== undefined) {
        contents = se_StreamingDistributionConfig(input.StreamingDistributionConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_UpdateVpcOriginCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = smithyClient.map({}, smithyClient.isSerializableHeaderValue, {
        "content-type": "application/xml",
        [_im]: input[_IM],
    });
    b.bp("/2020-05-31/vpc-origin/{Id}");
    b.p("Id", () => input.Id, "{Id}", false);
    let body;
    let contents;
    if (input.VpcOriginEndpointConfig !== undefined) {
        contents = se_VpcOriginEndpointConfig(input.VpcOriginEndpointConfig);
        body = _ve;
        contents.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
        body += contents.toString();
    }
    b.m("PUT").h(headers).b(body);
    return b.build();
};
const se_VerifyDnsConfigurationCommand = async (input, context) => {
    const b = core.requestBuilder(input, context);
    const headers = {
        "content-type": "application/xml",
    };
    b.bp("/2020-05-31/verify-dns-configuration");
    let body;
    body = _ve;
    const bn = new xmlBuilder.XmlNode(_VDCR);
    bn.a("xmlns", "http://cloudfront.amazonaws.com/doc/2020-05-31/");
    if (input[_Do] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Do]).n(_Do));
    }
    if (input[_I] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_I]).n(_I));
    }
    body += bn.toString();
    b.m("POST").h(headers).b(body);
    return b.build();
};
const de_AssociateAliasCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_AssociateDistributionTenantWebACLCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectNonNull(smithyClient.expectObject(await core$1.parseXmlBody(output.body, context)), "body");
    if (data[_Id] != null) {
        contents[_Id] = smithyClient.expectString(data[_Id]);
    }
    if (data[_WACLA] != null) {
        contents[_WACLA] = smithyClient.expectString(data[_WACLA]);
    }
    return contents;
};
const de_AssociateDistributionWebACLCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectNonNull(smithyClient.expectObject(await core$1.parseXmlBody(output.body, context)), "body");
    if (data[_Id] != null) {
        contents[_Id] = smithyClient.expectString(data[_Id]);
    }
    if (data[_WACLA] != null) {
        contents[_WACLA] = smithyClient.expectString(data[_WACLA]);
    }
    return contents;
};
const de_CopyDistributionCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.Distribution = de_Distribution(data);
    return contents;
};
const de_CreateAnycastIpListCommand = async (output, context) => {
    if (output.statusCode !== 202 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.AnycastIpList = de_AnycastIpList(data);
    return contents;
};
const de_CreateCachePolicyCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.CachePolicy = de_CachePolicy(data);
    return contents;
};
const de_CreateCloudFrontOriginAccessIdentityCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.CloudFrontOriginAccessIdentity = de_CloudFrontOriginAccessIdentity(data);
    return contents;
};
const de_CreateConnectionGroupCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.ConnectionGroup = de_ConnectionGroup(data);
    return contents;
};
const de_CreateContinuousDeploymentPolicyCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.ContinuousDeploymentPolicy = de_ContinuousDeploymentPolicy(data);
    return contents;
};
const de_CreateDistributionCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.Distribution = de_Distribution(data);
    return contents;
};
const de_CreateDistributionTenantCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.DistributionTenant = de_DistributionTenant(data);
    return contents;
};
const de_CreateDistributionWithTagsCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.Distribution = de_Distribution(data);
    return contents;
};
const de_CreateFieldLevelEncryptionConfigCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.FieldLevelEncryption = de_FieldLevelEncryption(data);
    return contents;
};
const de_CreateFieldLevelEncryptionProfileCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.FieldLevelEncryptionProfile = de_FieldLevelEncryptionProfile(data);
    return contents;
};
const de_CreateFunctionCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.FunctionSummary = de_FunctionSummary(data);
    return contents;
};
const de_CreateInvalidationCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.Invalidation = de_Invalidation(data);
    return contents;
};
const de_CreateInvalidationForDistributionTenantCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.Invalidation = de_Invalidation(data);
    return contents;
};
const de_CreateKeyGroupCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.KeyGroup = de_KeyGroup(data);
    return contents;
};
const de_CreateKeyValueStoreCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
        [_L]: [, output.headers[_lo]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.KeyValueStore = de_KeyValueStore(data);
    return contents;
};
const de_CreateMonitoringSubscriptionCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.MonitoringSubscription = de_MonitoringSubscription(data);
    return contents;
};
const de_CreateOriginAccessControlCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.OriginAccessControl = de_OriginAccessControl(data);
    return contents;
};
const de_CreateOriginRequestPolicyCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.OriginRequestPolicy = de_OriginRequestPolicy(data);
    return contents;
};
const de_CreatePublicKeyCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.PublicKey = de_PublicKey(data);
    return contents;
};
const de_CreateRealtimeLogConfigCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectNonNull(smithyClient.expectObject(await core$1.parseXmlBody(output.body, context)), "body");
    if (data[_RLC] != null) {
        contents[_RLC] = de_RealtimeLogConfig(data[_RLC]);
    }
    return contents;
};
const de_CreateResponseHeadersPolicyCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.ResponseHeadersPolicy = de_ResponseHeadersPolicy(data);
    return contents;
};
const de_CreateStreamingDistributionCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.StreamingDistribution = de_StreamingDistribution(data);
    return contents;
};
const de_CreateStreamingDistributionWithTagsCommand = async (output, context) => {
    if (output.statusCode !== 201 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.StreamingDistribution = de_StreamingDistribution(data);
    return contents;
};
const de_CreateVpcOriginCommand = async (output, context) => {
    if (output.statusCode !== 202 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_L]: [, output.headers[_lo]],
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.VpcOrigin = de_VpcOrigin(data);
    return contents;
};
const de_DeleteAnycastIpListCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteCachePolicyCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteCloudFrontOriginAccessIdentityCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteConnectionGroupCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteContinuousDeploymentPolicyCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteDistributionCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteDistributionTenantCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteFieldLevelEncryptionConfigCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteFieldLevelEncryptionProfileCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteFunctionCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteKeyGroupCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteKeyValueStoreCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteMonitoringSubscriptionCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteOriginAccessControlCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteOriginRequestPolicyCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeletePublicKeyCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteRealtimeLogConfigCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteResponseHeadersPolicyCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteStreamingDistributionCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_DeleteVpcOriginCommand = async (output, context) => {
    if (output.statusCode !== 202 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.VpcOrigin = de_VpcOrigin(data);
    return contents;
};
const de_DescribeFunctionCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.FunctionSummary = de_FunctionSummary(data);
    return contents;
};
const de_DescribeKeyValueStoreCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.KeyValueStore = de_KeyValueStore(data);
    return contents;
};
const de_DisassociateDistributionTenantWebACLCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectNonNull(smithyClient.expectObject(await core$1.parseXmlBody(output.body, context)), "body");
    if (data[_Id] != null) {
        contents[_Id] = smithyClient.expectString(data[_Id]);
    }
    return contents;
};
const de_DisassociateDistributionWebACLCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectNonNull(smithyClient.expectObject(await core$1.parseXmlBody(output.body, context)), "body");
    if (data[_Id] != null) {
        contents[_Id] = smithyClient.expectString(data[_Id]);
    }
    return contents;
};
const de_GetAnycastIpListCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.AnycastIpList = de_AnycastIpList(data);
    return contents;
};
const de_GetCachePolicyCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.CachePolicy = de_CachePolicy(data);
    return contents;
};
const de_GetCachePolicyConfigCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.CachePolicyConfig = de_CachePolicyConfig(data);
    return contents;
};
const de_GetCloudFrontOriginAccessIdentityCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.CloudFrontOriginAccessIdentity = de_CloudFrontOriginAccessIdentity(data);
    return contents;
};
const de_GetCloudFrontOriginAccessIdentityConfigCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.CloudFrontOriginAccessIdentityConfig = de_CloudFrontOriginAccessIdentityConfig(data);
    return contents;
};
const de_GetConnectionGroupCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.ConnectionGroup = de_ConnectionGroup(data);
    return contents;
};
const de_GetConnectionGroupByRoutingEndpointCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.ConnectionGroup = de_ConnectionGroup(data);
    return contents;
};
const de_GetContinuousDeploymentPolicyCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.ContinuousDeploymentPolicy = de_ContinuousDeploymentPolicy(data);
    return contents;
};
const de_GetContinuousDeploymentPolicyConfigCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.ContinuousDeploymentPolicyConfig = de_ContinuousDeploymentPolicyConfig(data);
    return contents;
};
const de_GetDistributionCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.Distribution = de_Distribution(data);
    return contents;
};
const de_GetDistributionConfigCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.DistributionConfig = de_DistributionConfig(data);
    return contents;
};
const de_GetDistributionTenantCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.DistributionTenant = de_DistributionTenant(data);
    return contents;
};
const de_GetDistributionTenantByDomainCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.DistributionTenant = de_DistributionTenant(data);
    return contents;
};
const de_GetFieldLevelEncryptionCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.FieldLevelEncryption = de_FieldLevelEncryption(data);
    return contents;
};
const de_GetFieldLevelEncryptionConfigCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.FieldLevelEncryptionConfig = de_FieldLevelEncryptionConfig(data);
    return contents;
};
const de_GetFieldLevelEncryptionProfileCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.FieldLevelEncryptionProfile = de_FieldLevelEncryptionProfile(data);
    return contents;
};
const de_GetFieldLevelEncryptionProfileConfigCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.FieldLevelEncryptionProfileConfig = de_FieldLevelEncryptionProfileConfig(data);
    return contents;
};
const de_GetFunctionCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
        [_CT]: [, output.headers[_ct]],
    });
    const data = await smithyClient.collectBody(output.body, context);
    contents.FunctionCode = data;
    return contents;
};
const de_GetInvalidationCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.Invalidation = de_Invalidation(data);
    return contents;
};
const de_GetInvalidationForDistributionTenantCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.Invalidation = de_Invalidation(data);
    return contents;
};
const de_GetKeyGroupCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.KeyGroup = de_KeyGroup(data);
    return contents;
};
const de_GetKeyGroupConfigCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.KeyGroupConfig = de_KeyGroupConfig(data);
    return contents;
};
const de_GetManagedCertificateDetailsCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.ManagedCertificateDetails = de_ManagedCertificateDetails(data);
    return contents;
};
const de_GetMonitoringSubscriptionCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.MonitoringSubscription = de_MonitoringSubscription(data);
    return contents;
};
const de_GetOriginAccessControlCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.OriginAccessControl = de_OriginAccessControl(data);
    return contents;
};
const de_GetOriginAccessControlConfigCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.OriginAccessControlConfig = de_OriginAccessControlConfig(data);
    return contents;
};
const de_GetOriginRequestPolicyCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.OriginRequestPolicy = de_OriginRequestPolicy(data);
    return contents;
};
const de_GetOriginRequestPolicyConfigCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.OriginRequestPolicyConfig = de_OriginRequestPolicyConfig(data);
    return contents;
};
const de_GetPublicKeyCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.PublicKey = de_PublicKey(data);
    return contents;
};
const de_GetPublicKeyConfigCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.PublicKeyConfig = de_PublicKeyConfig(data);
    return contents;
};
const de_GetRealtimeLogConfigCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectNonNull(smithyClient.expectObject(await core$1.parseXmlBody(output.body, context)), "body");
    if (data[_RLC] != null) {
        contents[_RLC] = de_RealtimeLogConfig(data[_RLC]);
    }
    return contents;
};
const de_GetResponseHeadersPolicyCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.ResponseHeadersPolicy = de_ResponseHeadersPolicy(data);
    return contents;
};
const de_GetResponseHeadersPolicyConfigCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.ResponseHeadersPolicyConfig = de_ResponseHeadersPolicyConfig(data);
    return contents;
};
const de_GetStreamingDistributionCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.StreamingDistribution = de_StreamingDistribution(data);
    return contents;
};
const de_GetStreamingDistributionConfigCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.StreamingDistributionConfig = de_StreamingDistributionConfig(data);
    return contents;
};
const de_GetVpcOriginCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.VpcOrigin = de_VpcOrigin(data);
    return contents;
};
const de_ListAnycastIpListsCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.AnycastIpLists = de_AnycastIpListCollection(data);
    return contents;
};
const de_ListCachePoliciesCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.CachePolicyList = de_CachePolicyList(data);
    return contents;
};
const de_ListCloudFrontOriginAccessIdentitiesCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.CloudFrontOriginAccessIdentityList = de_CloudFrontOriginAccessIdentityList(data);
    return contents;
};
const de_ListConflictingAliasesCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.ConflictingAliasesList = de_ConflictingAliasesList(data);
    return contents;
};
const de_ListConnectionGroupsCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectNonNull(smithyClient.expectObject(await core$1.parseXmlBody(output.body, context)), "body");
    if (String(data.ConnectionGroups).trim() === "") {
        contents[_CG] = [];
    }
    else if (data[_CG] != null && data[_CG][_CGS] != null) {
        contents[_CG] = de_ConnectionGroupSummaryList(smithyClient.getArrayIfSingleItem(data[_CG][_CGS]));
    }
    if (data[_NM] != null) {
        contents[_NM] = smithyClient.expectString(data[_NM]);
    }
    return contents;
};
const de_ListContinuousDeploymentPoliciesCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.ContinuousDeploymentPolicyList = de_ContinuousDeploymentPolicyList(data);
    return contents;
};
const de_ListDistributionsCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.DistributionList = de_DistributionList(data);
    return contents;
};
const de_ListDistributionsByAnycastIpListIdCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.DistributionList = de_DistributionList(data);
    return contents;
};
const de_ListDistributionsByCachePolicyIdCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.DistributionIdList = de_DistributionIdList(data);
    return contents;
};
const de_ListDistributionsByConnectionModeCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.DistributionList = de_DistributionList(data);
    return contents;
};
const de_ListDistributionsByKeyGroupCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.DistributionIdList = de_DistributionIdList(data);
    return contents;
};
const de_ListDistributionsByOriginRequestPolicyIdCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.DistributionIdList = de_DistributionIdList(data);
    return contents;
};
const de_ListDistributionsByRealtimeLogConfigCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.DistributionList = de_DistributionList(data);
    return contents;
};
const de_ListDistributionsByResponseHeadersPolicyIdCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.DistributionIdList = de_DistributionIdList(data);
    return contents;
};
const de_ListDistributionsByVpcOriginIdCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.DistributionIdList = de_DistributionIdList(data);
    return contents;
};
const de_ListDistributionsByWebACLIdCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.DistributionList = de_DistributionList(data);
    return contents;
};
const de_ListDistributionTenantsCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectNonNull(smithyClient.expectObject(await core$1.parseXmlBody(output.body, context)), "body");
    if (String(data.DistributionTenantList).trim() === "") {
        contents[_DTL] = [];
    }
    else if (data[_DTL] != null && data[_DTL][_DTS] != null) {
        contents[_DTL] = de_DistributionTenantList(smithyClient.getArrayIfSingleItem(data[_DTL][_DTS]));
    }
    if (data[_NM] != null) {
        contents[_NM] = smithyClient.expectString(data[_NM]);
    }
    return contents;
};
const de_ListDistributionTenantsByCustomizationCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectNonNull(smithyClient.expectObject(await core$1.parseXmlBody(output.body, context)), "body");
    if (String(data.DistributionTenantList).trim() === "") {
        contents[_DTL] = [];
    }
    else if (data[_DTL] != null && data[_DTL][_DTS] != null) {
        contents[_DTL] = de_DistributionTenantList(smithyClient.getArrayIfSingleItem(data[_DTL][_DTS]));
    }
    if (data[_NM] != null) {
        contents[_NM] = smithyClient.expectString(data[_NM]);
    }
    return contents;
};
const de_ListDomainConflictsCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectNonNull(smithyClient.expectObject(await core$1.parseXmlBody(output.body, context)), "body");
    if (String(data.DomainConflicts).trim() === "") {
        contents[_DC] = [];
    }
    else if (data[_DC] != null && data[_DC][_DC] != null) {
        contents[_DC] = de_DomainConflictsList(smithyClient.getArrayIfSingleItem(data[_DC][_DC]));
    }
    if (data[_NM] != null) {
        contents[_NM] = smithyClient.expectString(data[_NM]);
    }
    return contents;
};
const de_ListFieldLevelEncryptionConfigsCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.FieldLevelEncryptionList = de_FieldLevelEncryptionList(data);
    return contents;
};
const de_ListFieldLevelEncryptionProfilesCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.FieldLevelEncryptionProfileList = de_FieldLevelEncryptionProfileList(data);
    return contents;
};
const de_ListFunctionsCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.FunctionList = de_FunctionList(data);
    return contents;
};
const de_ListInvalidationsCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.InvalidationList = de_InvalidationList(data);
    return contents;
};
const de_ListInvalidationsForDistributionTenantCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.InvalidationList = de_InvalidationList(data);
    return contents;
};
const de_ListKeyGroupsCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.KeyGroupList = de_KeyGroupList(data);
    return contents;
};
const de_ListKeyValueStoresCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.KeyValueStoreList = de_KeyValueStoreList(data);
    return contents;
};
const de_ListOriginAccessControlsCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.OriginAccessControlList = de_OriginAccessControlList(data);
    return contents;
};
const de_ListOriginRequestPoliciesCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.OriginRequestPolicyList = de_OriginRequestPolicyList(data);
    return contents;
};
const de_ListPublicKeysCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.PublicKeyList = de_PublicKeyList(data);
    return contents;
};
const de_ListRealtimeLogConfigsCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.RealtimeLogConfigs = de_RealtimeLogConfigs(data);
    return contents;
};
const de_ListResponseHeadersPoliciesCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.ResponseHeadersPolicyList = de_ResponseHeadersPolicyList(data);
    return contents;
};
const de_ListStreamingDistributionsCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.StreamingDistributionList = de_StreamingDistributionList(data);
    return contents;
};
const de_ListTagsForResourceCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.Tags = de_Tags(data);
    return contents;
};
const de_ListVpcOriginsCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.VpcOriginList = de_VpcOriginList(data);
    return contents;
};
const de_PublishFunctionCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.FunctionSummary = de_FunctionSummary(data);
    return contents;
};
const de_TagResourceCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_TestFunctionCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.TestResult = de_TestResult(data);
    return contents;
};
const de_UntagResourceCommand = async (output, context) => {
    if (output.statusCode !== 204 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    await smithyClient.collectBody(output.body, context);
    return contents;
};
const de_UpdateCachePolicyCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.CachePolicy = de_CachePolicy(data);
    return contents;
};
const de_UpdateCloudFrontOriginAccessIdentityCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.CloudFrontOriginAccessIdentity = de_CloudFrontOriginAccessIdentity(data);
    return contents;
};
const de_UpdateConnectionGroupCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.ConnectionGroup = de_ConnectionGroup(data);
    return contents;
};
const de_UpdateContinuousDeploymentPolicyCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.ContinuousDeploymentPolicy = de_ContinuousDeploymentPolicy(data);
    return contents;
};
const de_UpdateDistributionCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.Distribution = de_Distribution(data);
    return contents;
};
const de_UpdateDistributionTenantCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.DistributionTenant = de_DistributionTenant(data);
    return contents;
};
const de_UpdateDistributionWithStagingConfigCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.Distribution = de_Distribution(data);
    return contents;
};
const de_UpdateDomainAssociationCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectNonNull(smithyClient.expectObject(await core$1.parseXmlBody(output.body, context)), "body");
    if (data[_Do] != null) {
        contents[_Do] = smithyClient.expectString(data[_Do]);
    }
    if (data[_RI] != null) {
        contents[_RI] = smithyClient.expectString(data[_RI]);
    }
    return contents;
};
const de_UpdateFieldLevelEncryptionConfigCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.FieldLevelEncryption = de_FieldLevelEncryption(data);
    return contents;
};
const de_UpdateFieldLevelEncryptionProfileCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.FieldLevelEncryptionProfile = de_FieldLevelEncryptionProfile(data);
    return contents;
};
const de_UpdateFunctionCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_et]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.FunctionSummary = de_FunctionSummary(data);
    return contents;
};
const de_UpdateKeyGroupCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.KeyGroup = de_KeyGroup(data);
    return contents;
};
const de_UpdateKeyValueStoreCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.KeyValueStore = de_KeyValueStore(data);
    return contents;
};
const de_UpdateOriginAccessControlCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.OriginAccessControl = de_OriginAccessControl(data);
    return contents;
};
const de_UpdateOriginRequestPolicyCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.OriginRequestPolicy = de_OriginRequestPolicy(data);
    return contents;
};
const de_UpdatePublicKeyCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.PublicKey = de_PublicKey(data);
    return contents;
};
const de_UpdateRealtimeLogConfigCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectNonNull(smithyClient.expectObject(await core$1.parseXmlBody(output.body, context)), "body");
    if (data[_RLC] != null) {
        contents[_RLC] = de_RealtimeLogConfig(data[_RLC]);
    }
    return contents;
};
const de_UpdateResponseHeadersPolicyCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.ResponseHeadersPolicy = de_ResponseHeadersPolicy(data);
    return contents;
};
const de_UpdateStreamingDistributionCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.StreamingDistribution = de_StreamingDistribution(data);
    return contents;
};
const de_UpdateVpcOriginCommand = async (output, context) => {
    if (output.statusCode !== 202 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
        [_ET]: [, output.headers[_e]],
    });
    const data = smithyClient.expectObject(await core$1.parseXmlBody(output.body, context));
    contents.VpcOrigin = de_VpcOrigin(data);
    return contents;
};
const de_VerifyDnsConfigurationCommand = async (output, context) => {
    if (output.statusCode !== 200 && output.statusCode >= 300) {
        return de_CommandError(output, context);
    }
    const contents = smithyClient.map({
        $metadata: deserializeMetadata(output),
    });
    const data = smithyClient.expectNonNull(smithyClient.expectObject(await core$1.parseXmlBody(output.body, context)), "body");
    if (String(data.DnsConfigurationList).trim() === "") {
        contents[_DCL] = [];
    }
    else if (data[_DCL] != null && data[_DCL][_DCn] != null) {
        contents[_DCL] = de_DnsConfigurationList(smithyClient.getArrayIfSingleItem(data[_DCL][_DCn]));
    }
    return contents;
};
const de_CommandError = async (output, context) => {
    const parsedOutput = {
        ...output,
        body: await core$1.parseXmlErrorBody(output.body, context),
    };
    const errorCode = core$1.loadRestXmlErrorCode(output, parsedOutput.body);
    switch (errorCode) {
        case "AccessDenied":
        case "com.amazonaws.cloudfront#AccessDenied":
            throw await de_AccessDeniedRes(parsedOutput);
        case "IllegalUpdate":
        case "com.amazonaws.cloudfront#IllegalUpdate":
            throw await de_IllegalUpdateRes(parsedOutput);
        case "InvalidArgument":
        case "com.amazonaws.cloudfront#InvalidArgument":
            throw await de_InvalidArgumentRes(parsedOutput);
        case "NoSuchDistribution":
        case "com.amazonaws.cloudfront#NoSuchDistribution":
            throw await de_NoSuchDistributionRes(parsedOutput);
        case "TooManyDistributionCNAMEs":
        case "com.amazonaws.cloudfront#TooManyDistributionCNAMEs":
            throw await de_TooManyDistributionCNAMEsRes(parsedOutput);
        case "EntityNotFound":
        case "com.amazonaws.cloudfront#EntityNotFound":
            throw await de_EntityNotFoundRes(parsedOutput);
        case "InvalidIfMatchVersion":
        case "com.amazonaws.cloudfront#InvalidIfMatchVersion":
            throw await de_InvalidIfMatchVersionRes(parsedOutput);
        case "PreconditionFailed":
        case "com.amazonaws.cloudfront#PreconditionFailed":
            throw await de_PreconditionFailedRes(parsedOutput);
        case "CNAMEAlreadyExists":
        case "com.amazonaws.cloudfront#CNAMEAlreadyExists":
            throw await de_CNAMEAlreadyExistsRes(parsedOutput);
        case "DistributionAlreadyExists":
        case "com.amazonaws.cloudfront#DistributionAlreadyExists":
            throw await de_DistributionAlreadyExistsRes(parsedOutput);
        case "IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior":
        case "com.amazonaws.cloudfront#IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior":
            throw await de_IllegalFieldLevelEncryptionConfigAssociationWithCacheBehaviorRes(parsedOutput);
        case "InconsistentQuantities":
        case "com.amazonaws.cloudfront#InconsistentQuantities":
            throw await de_InconsistentQuantitiesRes(parsedOutput);
        case "InvalidDefaultRootObject":
        case "com.amazonaws.cloudfront#InvalidDefaultRootObject":
            throw await de_InvalidDefaultRootObjectRes(parsedOutput);
        case "InvalidErrorCode":
        case "com.amazonaws.cloudfront#InvalidErrorCode":
            throw await de_InvalidErrorCodeRes(parsedOutput);
        case "InvalidForwardCookies":
        case "com.amazonaws.cloudfront#InvalidForwardCookies":
            throw await de_InvalidForwardCookiesRes(parsedOutput);
        case "InvalidFunctionAssociation":
        case "com.amazonaws.cloudfront#InvalidFunctionAssociation":
            throw await de_InvalidFunctionAssociationRes(parsedOutput);
        case "InvalidGeoRestrictionParameter":
        case "com.amazonaws.cloudfront#InvalidGeoRestrictionParameter":
            throw await de_InvalidGeoRestrictionParameterRes(parsedOutput);
        case "InvalidHeadersForS3Origin":
        case "com.amazonaws.cloudfront#InvalidHeadersForS3Origin":
            throw await de_InvalidHeadersForS3OriginRes(parsedOutput);
        case "InvalidLambdaFunctionAssociation":
        case "com.amazonaws.cloudfront#InvalidLambdaFunctionAssociation":
            throw await de_InvalidLambdaFunctionAssociationRes(parsedOutput);
        case "InvalidLocationCode":
        case "com.amazonaws.cloudfront#InvalidLocationCode":
            throw await de_InvalidLocationCodeRes(parsedOutput);
        case "InvalidMinimumProtocolVersion":
        case "com.amazonaws.cloudfront#InvalidMinimumProtocolVersion":
            throw await de_InvalidMinimumProtocolVersionRes(parsedOutput);
        case "InvalidOrigin":
        case "com.amazonaws.cloudfront#InvalidOrigin":
            throw await de_InvalidOriginRes(parsedOutput);
        case "InvalidOriginAccessControl":
        case "com.amazonaws.cloudfront#InvalidOriginAccessControl":
            throw await de_InvalidOriginAccessControlRes(parsedOutput);
        case "InvalidOriginAccessIdentity":
        case "com.amazonaws.cloudfront#InvalidOriginAccessIdentity":
            throw await de_InvalidOriginAccessIdentityRes(parsedOutput);
        case "InvalidOriginKeepaliveTimeout":
        case "com.amazonaws.cloudfront#InvalidOriginKeepaliveTimeout":
            throw await de_InvalidOriginKeepaliveTimeoutRes(parsedOutput);
        case "InvalidOriginReadTimeout":
        case "com.amazonaws.cloudfront#InvalidOriginReadTimeout":
            throw await de_InvalidOriginReadTimeoutRes(parsedOutput);
        case "InvalidProtocolSettings":
        case "com.amazonaws.cloudfront#InvalidProtocolSettings":
            throw await de_InvalidProtocolSettingsRes(parsedOutput);
        case "InvalidQueryStringParameters":
        case "com.amazonaws.cloudfront#InvalidQueryStringParameters":
            throw await de_InvalidQueryStringParametersRes(parsedOutput);
        case "InvalidRelativePath":
        case "com.amazonaws.cloudfront#InvalidRelativePath":
            throw await de_InvalidRelativePathRes(parsedOutput);
        case "InvalidRequiredProtocol":
        case "com.amazonaws.cloudfront#InvalidRequiredProtocol":
            throw await de_InvalidRequiredProtocolRes(parsedOutput);
        case "InvalidResponseCode":
        case "com.amazonaws.cloudfront#InvalidResponseCode":
            throw await de_InvalidResponseCodeRes(parsedOutput);
        case "InvalidTTLOrder":
        case "com.amazonaws.cloudfront#InvalidTTLOrder":
            throw await de_InvalidTTLOrderRes(parsedOutput);
        case "InvalidViewerCertificate":
        case "com.amazonaws.cloudfront#InvalidViewerCertificate":
            throw await de_InvalidViewerCertificateRes(parsedOutput);
        case "InvalidWebACLId":
        case "com.amazonaws.cloudfront#InvalidWebACLId":
            throw await de_InvalidWebACLIdRes(parsedOutput);
        case "MissingBody":
        case "com.amazonaws.cloudfront#MissingBody":
            throw await de_MissingBodyRes(parsedOutput);
        case "NoSuchCachePolicy":
        case "com.amazonaws.cloudfront#NoSuchCachePolicy":
            throw await de_NoSuchCachePolicyRes(parsedOutput);
        case "NoSuchFieldLevelEncryptionConfig":
        case "com.amazonaws.cloudfront#NoSuchFieldLevelEncryptionConfig":
            throw await de_NoSuchFieldLevelEncryptionConfigRes(parsedOutput);
        case "NoSuchOrigin":
        case "com.amazonaws.cloudfront#NoSuchOrigin":
            throw await de_NoSuchOriginRes(parsedOutput);
        case "NoSuchOriginRequestPolicy":
        case "com.amazonaws.cloudfront#NoSuchOriginRequestPolicy":
            throw await de_NoSuchOriginRequestPolicyRes(parsedOutput);
        case "NoSuchRealtimeLogConfig":
        case "com.amazonaws.cloudfront#NoSuchRealtimeLogConfig":
            throw await de_NoSuchRealtimeLogConfigRes(parsedOutput);
        case "NoSuchResponseHeadersPolicy":
        case "com.amazonaws.cloudfront#NoSuchResponseHeadersPolicy":
            throw await de_NoSuchResponseHeadersPolicyRes(parsedOutput);
        case "RealtimeLogConfigOwnerMismatch":
        case "com.amazonaws.cloudfront#RealtimeLogConfigOwnerMismatch":
            throw await de_RealtimeLogConfigOwnerMismatchRes(parsedOutput);
        case "TooManyCacheBehaviors":
        case "com.amazonaws.cloudfront#TooManyCacheBehaviors":
            throw await de_TooManyCacheBehaviorsRes(parsedOutput);
        case "TooManyCertificates":
        case "com.amazonaws.cloudfront#TooManyCertificates":
            throw await de_TooManyCertificatesRes(parsedOutput);
        case "TooManyCookieNamesInWhiteList":
        case "com.amazonaws.cloudfront#TooManyCookieNamesInWhiteList":
            throw await de_TooManyCookieNamesInWhiteListRes(parsedOutput);
        case "TooManyDistributions":
        case "com.amazonaws.cloudfront#TooManyDistributions":
            throw await de_TooManyDistributionsRes(parsedOutput);
        case "TooManyDistributionsAssociatedToCachePolicy":
        case "com.amazonaws.cloudfront#TooManyDistributionsAssociatedToCachePolicy":
            throw await de_TooManyDistributionsAssociatedToCachePolicyRes(parsedOutput);
        case "TooManyDistributionsAssociatedToFieldLevelEncryptionConfig":
        case "com.amazonaws.cloudfront#TooManyDistributionsAssociatedToFieldLevelEncryptionConfig":
            throw await de_TooManyDistributionsAssociatedToFieldLevelEncryptionConfigRes(parsedOutput);
        case "TooManyDistributionsAssociatedToKeyGroup":
        case "com.amazonaws.cloudfront#TooManyDistributionsAssociatedToKeyGroup":
            throw await de_TooManyDistributionsAssociatedToKeyGroupRes(parsedOutput);
        case "TooManyDistributionsAssociatedToOriginAccessControl":
        case "com.amazonaws.cloudfront#TooManyDistributionsAssociatedToOriginAccessControl":
            throw await de_TooManyDistributionsAssociatedToOriginAccessControlRes(parsedOutput);
        case "TooManyDistributionsAssociatedToOriginRequestPolicy":
        case "com.amazonaws.cloudfront#TooManyDistributionsAssociatedToOriginRequestPolicy":
            throw await de_TooManyDistributionsAssociatedToOriginRequestPolicyRes(parsedOutput);
        case "TooManyDistributionsAssociatedToResponseHeadersPolicy":
        case "com.amazonaws.cloudfront#TooManyDistributionsAssociatedToResponseHeadersPolicy":
            throw await de_TooManyDistributionsAssociatedToResponseHeadersPolicyRes(parsedOutput);
        case "TooManyDistributionsWithFunctionAssociations":
        case "com.amazonaws.cloudfront#TooManyDistributionsWithFunctionAssociations":
            throw await de_TooManyDistributionsWithFunctionAssociationsRes(parsedOutput);
        case "TooManyDistributionsWithLambdaAssociations":
        case "com.amazonaws.cloudfront#TooManyDistributionsWithLambdaAssociations":
            throw await de_TooManyDistributionsWithLambdaAssociationsRes(parsedOutput);
        case "TooManyDistributionsWithSingleFunctionARN":
        case "com.amazonaws.cloudfront#TooManyDistributionsWithSingleFunctionARN":
            throw await de_TooManyDistributionsWithSingleFunctionARNRes(parsedOutput);
        case "TooManyFunctionAssociations":
        case "com.amazonaws.cloudfront#TooManyFunctionAssociations":
            throw await de_TooManyFunctionAssociationsRes(parsedOutput);
        case "TooManyHeadersInForwardedValues":
        case "com.amazonaws.cloudfront#TooManyHeadersInForwardedValues":
            throw await de_TooManyHeadersInForwardedValuesRes(parsedOutput);
        case "TooManyKeyGroupsAssociatedToDistribution":
        case "com.amazonaws.cloudfront#TooManyKeyGroupsAssociatedToDistribution":
            throw await de_TooManyKeyGroupsAssociatedToDistributionRes(parsedOutput);
        case "TooManyLambdaFunctionAssociations":
        case "com.amazonaws.cloudfront#TooManyLambdaFunctionAssociations":
            throw await de_TooManyLambdaFunctionAssociationsRes(parsedOutput);
        case "TooManyOriginCustomHeaders":
        case "com.amazonaws.cloudfront#TooManyOriginCustomHeaders":
            throw await de_TooManyOriginCustomHeadersRes(parsedOutput);
        case "TooManyOriginGroupsPerDistribution":
        case "com.amazonaws.cloudfront#TooManyOriginGroupsPerDistribution":
            throw await de_TooManyOriginGroupsPerDistributionRes(parsedOutput);
        case "TooManyOrigins":
        case "com.amazonaws.cloudfront#TooManyOrigins":
            throw await de_TooManyOriginsRes(parsedOutput);
        case "TooManyQueryStringParameters":
        case "com.amazonaws.cloudfront#TooManyQueryStringParameters":
            throw await de_TooManyQueryStringParametersRes(parsedOutput);
        case "TooManyTrustedSigners":
        case "com.amazonaws.cloudfront#TooManyTrustedSigners":
            throw await de_TooManyTrustedSignersRes(parsedOutput);
        case "TrustedKeyGroupDoesNotExist":
        case "com.amazonaws.cloudfront#TrustedKeyGroupDoesNotExist":
            throw await de_TrustedKeyGroupDoesNotExistRes(parsedOutput);
        case "TrustedSignerDoesNotExist":
        case "com.amazonaws.cloudfront#TrustedSignerDoesNotExist":
            throw await de_TrustedSignerDoesNotExistRes(parsedOutput);
        case "EntityAlreadyExists":
        case "com.amazonaws.cloudfront#EntityAlreadyExists":
            throw await de_EntityAlreadyExistsRes(parsedOutput);
        case "EntityLimitExceeded":
        case "com.amazonaws.cloudfront#EntityLimitExceeded":
            throw await de_EntityLimitExceededRes(parsedOutput);
        case "InvalidTagging":
        case "com.amazonaws.cloudfront#InvalidTagging":
            throw await de_InvalidTaggingRes(parsedOutput);
        case "UnsupportedOperation":
        case "com.amazonaws.cloudfront#UnsupportedOperation":
            throw await de_UnsupportedOperationRes(parsedOutput);
        case "CachePolicyAlreadyExists":
        case "com.amazonaws.cloudfront#CachePolicyAlreadyExists":
            throw await de_CachePolicyAlreadyExistsRes(parsedOutput);
        case "TooManyCachePolicies":
        case "com.amazonaws.cloudfront#TooManyCachePolicies":
            throw await de_TooManyCachePoliciesRes(parsedOutput);
        case "TooManyCookiesInCachePolicy":
        case "com.amazonaws.cloudfront#TooManyCookiesInCachePolicy":
            throw await de_TooManyCookiesInCachePolicyRes(parsedOutput);
        case "TooManyHeadersInCachePolicy":
        case "com.amazonaws.cloudfront#TooManyHeadersInCachePolicy":
            throw await de_TooManyHeadersInCachePolicyRes(parsedOutput);
        case "TooManyQueryStringsInCachePolicy":
        case "com.amazonaws.cloudfront#TooManyQueryStringsInCachePolicy":
            throw await de_TooManyQueryStringsInCachePolicyRes(parsedOutput);
        case "CloudFrontOriginAccessIdentityAlreadyExists":
        case "com.amazonaws.cloudfront#CloudFrontOriginAccessIdentityAlreadyExists":
            throw await de_CloudFrontOriginAccessIdentityAlreadyExistsRes(parsedOutput);
        case "TooManyCloudFrontOriginAccessIdentities":
        case "com.amazonaws.cloudfront#TooManyCloudFrontOriginAccessIdentities":
            throw await de_TooManyCloudFrontOriginAccessIdentitiesRes(parsedOutput);
        case "ContinuousDeploymentPolicyAlreadyExists":
        case "com.amazonaws.cloudfront#ContinuousDeploymentPolicyAlreadyExists":
            throw await de_ContinuousDeploymentPolicyAlreadyExistsRes(parsedOutput);
        case "StagingDistributionInUse":
        case "com.amazonaws.cloudfront#StagingDistributionInUse":
            throw await de_StagingDistributionInUseRes(parsedOutput);
        case "TooManyContinuousDeploymentPolicies":
        case "com.amazonaws.cloudfront#TooManyContinuousDeploymentPolicies":
            throw await de_TooManyContinuousDeploymentPoliciesRes(parsedOutput);
        case "ContinuousDeploymentPolicyInUse":
        case "com.amazonaws.cloudfront#ContinuousDeploymentPolicyInUse":
            throw await de_ContinuousDeploymentPolicyInUseRes(parsedOutput);
        case "IllegalOriginAccessConfiguration":
        case "com.amazonaws.cloudfront#IllegalOriginAccessConfiguration":
            throw await de_IllegalOriginAccessConfigurationRes(parsedOutput);
        case "InvalidDomainNameForOriginAccessControl":
        case "com.amazonaws.cloudfront#InvalidDomainNameForOriginAccessControl":
            throw await de_InvalidDomainNameForOriginAccessControlRes(parsedOutput);
        case "NoSuchContinuousDeploymentPolicy":
        case "com.amazonaws.cloudfront#NoSuchContinuousDeploymentPolicy":
            throw await de_NoSuchContinuousDeploymentPolicyRes(parsedOutput);
        case "InvalidAssociation":
        case "com.amazonaws.cloudfront#InvalidAssociation":
            throw await de_InvalidAssociationRes(parsedOutput);
        case "FieldLevelEncryptionConfigAlreadyExists":
        case "com.amazonaws.cloudfront#FieldLevelEncryptionConfigAlreadyExists":
            throw await de_FieldLevelEncryptionConfigAlreadyExistsRes(parsedOutput);
        case "NoSuchFieldLevelEncryptionProfile":
        case "com.amazonaws.cloudfront#NoSuchFieldLevelEncryptionProfile":
            throw await de_NoSuchFieldLevelEncryptionProfileRes(parsedOutput);
        case "QueryArgProfileEmpty":
        case "com.amazonaws.cloudfront#QueryArgProfileEmpty":
            throw await de_QueryArgProfileEmptyRes(parsedOutput);
        case "TooManyFieldLevelEncryptionConfigs":
        case "com.amazonaws.cloudfront#TooManyFieldLevelEncryptionConfigs":
            throw await de_TooManyFieldLevelEncryptionConfigsRes(parsedOutput);
        case "TooManyFieldLevelEncryptionContentTypeProfiles":
        case "com.amazonaws.cloudfront#TooManyFieldLevelEncryptionContentTypeProfiles":
            throw await de_TooManyFieldLevelEncryptionContentTypeProfilesRes(parsedOutput);
        case "TooManyFieldLevelEncryptionQueryArgProfiles":
        case "com.amazonaws.cloudfront#TooManyFieldLevelEncryptionQueryArgProfiles":
            throw await de_TooManyFieldLevelEncryptionQueryArgProfilesRes(parsedOutput);
        case "FieldLevelEncryptionProfileAlreadyExists":
        case "com.amazonaws.cloudfront#FieldLevelEncryptionProfileAlreadyExists":
            throw await de_FieldLevelEncryptionProfileAlreadyExistsRes(parsedOutput);
        case "FieldLevelEncryptionProfileSizeExceeded":
        case "com.amazonaws.cloudfront#FieldLevelEncryptionProfileSizeExceeded":
            throw await de_FieldLevelEncryptionProfileSizeExceededRes(parsedOutput);
        case "NoSuchPublicKey":
        case "com.amazonaws.cloudfront#NoSuchPublicKey":
            throw await de_NoSuchPublicKeyRes(parsedOutput);
        case "TooManyFieldLevelEncryptionEncryptionEntities":
        case "com.amazonaws.cloudfront#TooManyFieldLevelEncryptionEncryptionEntities":
            throw await de_TooManyFieldLevelEncryptionEncryptionEntitiesRes(parsedOutput);
        case "TooManyFieldLevelEncryptionFieldPatterns":
        case "com.amazonaws.cloudfront#TooManyFieldLevelEncryptionFieldPatterns":
            throw await de_TooManyFieldLevelEncryptionFieldPatternsRes(parsedOutput);
        case "TooManyFieldLevelEncryptionProfiles":
        case "com.amazonaws.cloudfront#TooManyFieldLevelEncryptionProfiles":
            throw await de_TooManyFieldLevelEncryptionProfilesRes(parsedOutput);
        case "FunctionAlreadyExists":
        case "com.amazonaws.cloudfront#FunctionAlreadyExists":
            throw await de_FunctionAlreadyExistsRes(parsedOutput);
        case "FunctionSizeLimitExceeded":
        case "com.amazonaws.cloudfront#FunctionSizeLimitExceeded":
            throw await de_FunctionSizeLimitExceededRes(parsedOutput);
        case "TooManyFunctions":
        case "com.amazonaws.cloudfront#TooManyFunctions":
            throw await de_TooManyFunctionsRes(parsedOutput);
        case "BatchTooLarge":
        case "com.amazonaws.cloudfront#BatchTooLarge":
            throw await de_BatchTooLargeRes(parsedOutput);
        case "TooManyInvalidationsInProgress":
        case "com.amazonaws.cloudfront#TooManyInvalidationsInProgress":
            throw await de_TooManyInvalidationsInProgressRes(parsedOutput);
        case "KeyGroupAlreadyExists":
        case "com.amazonaws.cloudfront#KeyGroupAlreadyExists":
            throw await de_KeyGroupAlreadyExistsRes(parsedOutput);
        case "TooManyKeyGroups":
        case "com.amazonaws.cloudfront#TooManyKeyGroups":
            throw await de_TooManyKeyGroupsRes(parsedOutput);
        case "TooManyPublicKeysInKeyGroup":
        case "com.amazonaws.cloudfront#TooManyPublicKeysInKeyGroup":
            throw await de_TooManyPublicKeysInKeyGroupRes(parsedOutput);
        case "EntitySizeLimitExceeded":
        case "com.amazonaws.cloudfront#EntitySizeLimitExceeded":
            throw await de_EntitySizeLimitExceededRes(parsedOutput);
        case "MonitoringSubscriptionAlreadyExists":
        case "com.amazonaws.cloudfront#MonitoringSubscriptionAlreadyExists":
            throw await de_MonitoringSubscriptionAlreadyExistsRes(parsedOutput);
        case "OriginAccessControlAlreadyExists":
        case "com.amazonaws.cloudfront#OriginAccessControlAlreadyExists":
            throw await de_OriginAccessControlAlreadyExistsRes(parsedOutput);
        case "TooManyOriginAccessControls":
        case "com.amazonaws.cloudfront#TooManyOriginAccessControls":
            throw await de_TooManyOriginAccessControlsRes(parsedOutput);
        case "OriginRequestPolicyAlreadyExists":
        case "com.amazonaws.cloudfront#OriginRequestPolicyAlreadyExists":
            throw await de_OriginRequestPolicyAlreadyExistsRes(parsedOutput);
        case "TooManyCookiesInOriginRequestPolicy":
        case "com.amazonaws.cloudfront#TooManyCookiesInOriginRequestPolicy":
            throw await de_TooManyCookiesInOriginRequestPolicyRes(parsedOutput);
        case "TooManyHeadersInOriginRequestPolicy":
        case "com.amazonaws.cloudfront#TooManyHeadersInOriginRequestPolicy":
            throw await de_TooManyHeadersInOriginRequestPolicyRes(parsedOutput);
        case "TooManyOriginRequestPolicies":
        case "com.amazonaws.cloudfront#TooManyOriginRequestPolicies":
            throw await de_TooManyOriginRequestPoliciesRes(parsedOutput);
        case "TooManyQueryStringsInOriginRequestPolicy":
        case "com.amazonaws.cloudfront#TooManyQueryStringsInOriginRequestPolicy":
            throw await de_TooManyQueryStringsInOriginRequestPolicyRes(parsedOutput);
        case "PublicKeyAlreadyExists":
        case "com.amazonaws.cloudfront#PublicKeyAlreadyExists":
            throw await de_PublicKeyAlreadyExistsRes(parsedOutput);
        case "TooManyPublicKeys":
        case "com.amazonaws.cloudfront#TooManyPublicKeys":
            throw await de_TooManyPublicKeysRes(parsedOutput);
        case "RealtimeLogConfigAlreadyExists":
        case "com.amazonaws.cloudfront#RealtimeLogConfigAlreadyExists":
            throw await de_RealtimeLogConfigAlreadyExistsRes(parsedOutput);
        case "TooManyRealtimeLogConfigs":
        case "com.amazonaws.cloudfront#TooManyRealtimeLogConfigs":
            throw await de_TooManyRealtimeLogConfigsRes(parsedOutput);
        case "ResponseHeadersPolicyAlreadyExists":
        case "com.amazonaws.cloudfront#ResponseHeadersPolicyAlreadyExists":
            throw await de_ResponseHeadersPolicyAlreadyExistsRes(parsedOutput);
        case "TooLongCSPInResponseHeadersPolicy":
        case "com.amazonaws.cloudfront#TooLongCSPInResponseHeadersPolicy":
            throw await de_TooLongCSPInResponseHeadersPolicyRes(parsedOutput);
        case "TooManyCustomHeadersInResponseHeadersPolicy":
        case "com.amazonaws.cloudfront#TooManyCustomHeadersInResponseHeadersPolicy":
            throw await de_TooManyCustomHeadersInResponseHeadersPolicyRes(parsedOutput);
        case "TooManyRemoveHeadersInResponseHeadersPolicy":
        case "com.amazonaws.cloudfront#TooManyRemoveHeadersInResponseHeadersPolicy":
            throw await de_TooManyRemoveHeadersInResponseHeadersPolicyRes(parsedOutput);
        case "TooManyResponseHeadersPolicies":
        case "com.amazonaws.cloudfront#TooManyResponseHeadersPolicies":
            throw await de_TooManyResponseHeadersPoliciesRes(parsedOutput);
        case "StreamingDistributionAlreadyExists":
        case "com.amazonaws.cloudfront#StreamingDistributionAlreadyExists":
            throw await de_StreamingDistributionAlreadyExistsRes(parsedOutput);
        case "TooManyStreamingDistributionCNAMEs":
        case "com.amazonaws.cloudfront#TooManyStreamingDistributionCNAMEs":
            throw await de_TooManyStreamingDistributionCNAMEsRes(parsedOutput);
        case "TooManyStreamingDistributions":
        case "com.amazonaws.cloudfront#TooManyStreamingDistributions":
            throw await de_TooManyStreamingDistributionsRes(parsedOutput);
        case "CannotDeleteEntityWhileInUse":
        case "com.amazonaws.cloudfront#CannotDeleteEntityWhileInUse":
            throw await de_CannotDeleteEntityWhileInUseRes(parsedOutput);
        case "IllegalDelete":
        case "com.amazonaws.cloudfront#IllegalDelete":
            throw await de_IllegalDeleteRes(parsedOutput);
        case "CachePolicyInUse":
        case "com.amazonaws.cloudfront#CachePolicyInUse":
            throw await de_CachePolicyInUseRes(parsedOutput);
        case "CloudFrontOriginAccessIdentityInUse":
        case "com.amazonaws.cloudfront#CloudFrontOriginAccessIdentityInUse":
            throw await de_CloudFrontOriginAccessIdentityInUseRes(parsedOutput);
        case "NoSuchCloudFrontOriginAccessIdentity":
        case "com.amazonaws.cloudfront#NoSuchCloudFrontOriginAccessIdentity":
            throw await de_NoSuchCloudFrontOriginAccessIdentityRes(parsedOutput);
        case "ResourceNotDisabled":
        case "com.amazonaws.cloudfront#ResourceNotDisabled":
            throw await de_ResourceNotDisabledRes(parsedOutput);
        case "DistributionNotDisabled":
        case "com.amazonaws.cloudfront#DistributionNotDisabled":
            throw await de_DistributionNotDisabledRes(parsedOutput);
        case "ResourceInUse":
        case "com.amazonaws.cloudfront#ResourceInUse":
            throw await de_ResourceInUseRes(parsedOutput);
        case "FieldLevelEncryptionConfigInUse":
        case "com.amazonaws.cloudfront#FieldLevelEncryptionConfigInUse":
            throw await de_FieldLevelEncryptionConfigInUseRes(parsedOutput);
        case "FieldLevelEncryptionProfileInUse":
        case "com.amazonaws.cloudfront#FieldLevelEncryptionProfileInUse":
            throw await de_FieldLevelEncryptionProfileInUseRes(parsedOutput);
        case "FunctionInUse":
        case "com.amazonaws.cloudfront#FunctionInUse":
            throw await de_FunctionInUseRes(parsedOutput);
        case "NoSuchFunctionExists":
        case "com.amazonaws.cloudfront#NoSuchFunctionExists":
            throw await de_NoSuchFunctionExistsRes(parsedOutput);
        case "NoSuchResource":
        case "com.amazonaws.cloudfront#NoSuchResource":
            throw await de_NoSuchResourceRes(parsedOutput);
        case "NoSuchMonitoringSubscription":
        case "com.amazonaws.cloudfront#NoSuchMonitoringSubscription":
            throw await de_NoSuchMonitoringSubscriptionRes(parsedOutput);
        case "NoSuchOriginAccessControl":
        case "com.amazonaws.cloudfront#NoSuchOriginAccessControl":
            throw await de_NoSuchOriginAccessControlRes(parsedOutput);
        case "OriginAccessControlInUse":
        case "com.amazonaws.cloudfront#OriginAccessControlInUse":
            throw await de_OriginAccessControlInUseRes(parsedOutput);
        case "OriginRequestPolicyInUse":
        case "com.amazonaws.cloudfront#OriginRequestPolicyInUse":
            throw await de_OriginRequestPolicyInUseRes(parsedOutput);
        case "PublicKeyInUse":
        case "com.amazonaws.cloudfront#PublicKeyInUse":
            throw await de_PublicKeyInUseRes(parsedOutput);
        case "RealtimeLogConfigInUse":
        case "com.amazonaws.cloudfront#RealtimeLogConfigInUse":
            throw await de_RealtimeLogConfigInUseRes(parsedOutput);
        case "ResponseHeadersPolicyInUse":
        case "com.amazonaws.cloudfront#ResponseHeadersPolicyInUse":
            throw await de_ResponseHeadersPolicyInUseRes(parsedOutput);
        case "NoSuchStreamingDistribution":
        case "com.amazonaws.cloudfront#NoSuchStreamingDistribution":
            throw await de_NoSuchStreamingDistributionRes(parsedOutput);
        case "StreamingDistributionNotDisabled":
        case "com.amazonaws.cloudfront#StreamingDistributionNotDisabled":
            throw await de_StreamingDistributionNotDisabledRes(parsedOutput);
        case "NoSuchInvalidation":
        case "com.amazonaws.cloudfront#NoSuchInvalidation":
            throw await de_NoSuchInvalidationRes(parsedOutput);
        case "TestFunctionFailed":
        case "com.amazonaws.cloudfront#TestFunctionFailed":
            throw await de_TestFunctionFailedRes(parsedOutput);
        case "CannotChangeImmutablePublicKeyFields":
        case "com.amazonaws.cloudfront#CannotChangeImmutablePublicKeyFields":
            throw await de_CannotChangeImmutablePublicKeyFieldsRes(parsedOutput);
        case "CannotUpdateEntityWhileInUse":
        case "com.amazonaws.cloudfront#CannotUpdateEntityWhileInUse":
            throw await de_CannotUpdateEntityWhileInUseRes(parsedOutput);
        default:
            const parsedBody = parsedOutput.body;
            return throwDefaultError({
                output,
                parsedBody: parsedBody.Error,
                errorCode,
            });
    }
};
const throwDefaultError = smithyClient.withBaseException(CloudFrontServiceException);
const de_AccessDeniedRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new AccessDenied({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_BatchTooLargeRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new BatchTooLarge({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_CachePolicyAlreadyExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new CachePolicyAlreadyExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_CachePolicyInUseRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new CachePolicyInUse({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_CannotChangeImmutablePublicKeyFieldsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new CannotChangeImmutablePublicKeyFields({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_CannotDeleteEntityWhileInUseRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new CannotDeleteEntityWhileInUse({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_CannotUpdateEntityWhileInUseRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new CannotUpdateEntityWhileInUse({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_CloudFrontOriginAccessIdentityAlreadyExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new CloudFrontOriginAccessIdentityAlreadyExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_CloudFrontOriginAccessIdentityInUseRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new CloudFrontOriginAccessIdentityInUse({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_CNAMEAlreadyExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new CNAMEAlreadyExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_ContinuousDeploymentPolicyAlreadyExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new ContinuousDeploymentPolicyAlreadyExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_ContinuousDeploymentPolicyInUseRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new ContinuousDeploymentPolicyInUse({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_DistributionAlreadyExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new DistributionAlreadyExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_DistributionNotDisabledRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new DistributionNotDisabled({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_EntityAlreadyExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new EntityAlreadyExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_EntityLimitExceededRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new EntityLimitExceeded({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_EntityNotFoundRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new EntityNotFound({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_EntitySizeLimitExceededRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new EntitySizeLimitExceeded({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_FieldLevelEncryptionConfigAlreadyExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new FieldLevelEncryptionConfigAlreadyExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_FieldLevelEncryptionConfigInUseRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new FieldLevelEncryptionConfigInUse({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_FieldLevelEncryptionProfileAlreadyExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new FieldLevelEncryptionProfileAlreadyExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_FieldLevelEncryptionProfileInUseRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new FieldLevelEncryptionProfileInUse({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_FieldLevelEncryptionProfileSizeExceededRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new FieldLevelEncryptionProfileSizeExceeded({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_FunctionAlreadyExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new FunctionAlreadyExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_FunctionInUseRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new FunctionInUse({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_FunctionSizeLimitExceededRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new FunctionSizeLimitExceeded({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_IllegalDeleteRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new IllegalDelete({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_IllegalFieldLevelEncryptionConfigAssociationWithCacheBehaviorRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_IllegalOriginAccessConfigurationRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new IllegalOriginAccessConfiguration({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_IllegalUpdateRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new IllegalUpdate({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InconsistentQuantitiesRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InconsistentQuantities({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidArgumentRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidArgument({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidAssociationRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidAssociation({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidDefaultRootObjectRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidDefaultRootObject({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidDomainNameForOriginAccessControlRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidDomainNameForOriginAccessControl({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidErrorCodeRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidErrorCode({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidForwardCookiesRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidForwardCookies({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidFunctionAssociationRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidFunctionAssociation({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidGeoRestrictionParameterRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidGeoRestrictionParameter({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidHeadersForS3OriginRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidHeadersForS3Origin({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidIfMatchVersionRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidIfMatchVersion({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidLambdaFunctionAssociationRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidLambdaFunctionAssociation({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidLocationCodeRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidLocationCode({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidMinimumProtocolVersionRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidMinimumProtocolVersion({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidOriginRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidOrigin({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidOriginAccessControlRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidOriginAccessControl({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidOriginAccessIdentityRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidOriginAccessIdentity({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidOriginKeepaliveTimeoutRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidOriginKeepaliveTimeout({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidOriginReadTimeoutRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidOriginReadTimeout({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidProtocolSettingsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidProtocolSettings({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidQueryStringParametersRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidQueryStringParameters({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidRelativePathRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidRelativePath({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidRequiredProtocolRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidRequiredProtocol({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidResponseCodeRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidResponseCode({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidTaggingRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidTagging({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidTTLOrderRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidTTLOrder({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidViewerCertificateRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidViewerCertificate({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_InvalidWebACLIdRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new InvalidWebACLId({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_KeyGroupAlreadyExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new KeyGroupAlreadyExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_MissingBodyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new MissingBody({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_MonitoringSubscriptionAlreadyExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new MonitoringSubscriptionAlreadyExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_NoSuchCachePolicyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new NoSuchCachePolicy({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_NoSuchCloudFrontOriginAccessIdentityRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new NoSuchCloudFrontOriginAccessIdentity({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_NoSuchContinuousDeploymentPolicyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new NoSuchContinuousDeploymentPolicy({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_NoSuchDistributionRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new NoSuchDistribution({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_NoSuchFieldLevelEncryptionConfigRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new NoSuchFieldLevelEncryptionConfig({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_NoSuchFieldLevelEncryptionProfileRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new NoSuchFieldLevelEncryptionProfile({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_NoSuchFunctionExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new NoSuchFunctionExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_NoSuchInvalidationRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new NoSuchInvalidation({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_NoSuchMonitoringSubscriptionRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new NoSuchMonitoringSubscription({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_NoSuchOriginRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new NoSuchOrigin({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_NoSuchOriginAccessControlRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new NoSuchOriginAccessControl({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_NoSuchOriginRequestPolicyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new NoSuchOriginRequestPolicy({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_NoSuchPublicKeyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new NoSuchPublicKey({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_NoSuchRealtimeLogConfigRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new NoSuchRealtimeLogConfig({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_NoSuchResourceRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new NoSuchResource({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_NoSuchResponseHeadersPolicyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new NoSuchResponseHeadersPolicy({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_NoSuchStreamingDistributionRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new NoSuchStreamingDistribution({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_OriginAccessControlAlreadyExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new OriginAccessControlAlreadyExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_OriginAccessControlInUseRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new OriginAccessControlInUse({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_OriginRequestPolicyAlreadyExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new OriginRequestPolicyAlreadyExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_OriginRequestPolicyInUseRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new OriginRequestPolicyInUse({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_PreconditionFailedRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new PreconditionFailed({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_PublicKeyAlreadyExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new PublicKeyAlreadyExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_PublicKeyInUseRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new PublicKeyInUse({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_QueryArgProfileEmptyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new QueryArgProfileEmpty({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_RealtimeLogConfigAlreadyExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new RealtimeLogConfigAlreadyExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_RealtimeLogConfigInUseRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new RealtimeLogConfigInUse({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_RealtimeLogConfigOwnerMismatchRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new RealtimeLogConfigOwnerMismatch({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_ResourceInUseRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new ResourceInUse({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_ResourceNotDisabledRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new ResourceNotDisabled({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_ResponseHeadersPolicyAlreadyExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new ResponseHeadersPolicyAlreadyExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_ResponseHeadersPolicyInUseRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new ResponseHeadersPolicyInUse({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_StagingDistributionInUseRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new StagingDistributionInUse({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_StreamingDistributionAlreadyExistsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new StreamingDistributionAlreadyExists({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_StreamingDistributionNotDisabledRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new StreamingDistributionNotDisabled({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TestFunctionFailedRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TestFunctionFailed({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooLongCSPInResponseHeadersPolicyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooLongCSPInResponseHeadersPolicy({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyCacheBehaviorsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyCacheBehaviors({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyCachePoliciesRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyCachePolicies({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyCertificatesRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyCertificates({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyCloudFrontOriginAccessIdentitiesRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyCloudFrontOriginAccessIdentities({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyContinuousDeploymentPoliciesRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyContinuousDeploymentPolicies({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyCookieNamesInWhiteListRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyCookieNamesInWhiteList({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyCookiesInCachePolicyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyCookiesInCachePolicy({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyCookiesInOriginRequestPolicyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyCookiesInOriginRequestPolicy({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyCustomHeadersInResponseHeadersPolicyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyCustomHeadersInResponseHeadersPolicy({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyDistributionCNAMEsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyDistributionCNAMEs({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyDistributionsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyDistributions({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyDistributionsAssociatedToCachePolicyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyDistributionsAssociatedToCachePolicy({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyDistributionsAssociatedToFieldLevelEncryptionConfigRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyDistributionsAssociatedToFieldLevelEncryptionConfig({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyDistributionsAssociatedToKeyGroupRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyDistributionsAssociatedToKeyGroup({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyDistributionsAssociatedToOriginAccessControlRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyDistributionsAssociatedToOriginAccessControl({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyDistributionsAssociatedToOriginRequestPolicyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyDistributionsAssociatedToOriginRequestPolicy({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyDistributionsAssociatedToResponseHeadersPolicyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyDistributionsAssociatedToResponseHeadersPolicy({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyDistributionsWithFunctionAssociationsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyDistributionsWithFunctionAssociations({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyDistributionsWithLambdaAssociationsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyDistributionsWithLambdaAssociations({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyDistributionsWithSingleFunctionARNRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyDistributionsWithSingleFunctionARN({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyFieldLevelEncryptionConfigsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyFieldLevelEncryptionConfigs({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyFieldLevelEncryptionContentTypeProfilesRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyFieldLevelEncryptionContentTypeProfiles({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyFieldLevelEncryptionEncryptionEntitiesRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyFieldLevelEncryptionEncryptionEntities({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyFieldLevelEncryptionFieldPatternsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyFieldLevelEncryptionFieldPatterns({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyFieldLevelEncryptionProfilesRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyFieldLevelEncryptionProfiles({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyFieldLevelEncryptionQueryArgProfilesRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyFieldLevelEncryptionQueryArgProfiles({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyFunctionAssociationsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyFunctionAssociations({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyFunctionsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyFunctions({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyHeadersInCachePolicyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyHeadersInCachePolicy({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyHeadersInForwardedValuesRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyHeadersInForwardedValues({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyHeadersInOriginRequestPolicyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyHeadersInOriginRequestPolicy({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyInvalidationsInProgressRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyInvalidationsInProgress({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyKeyGroupsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyKeyGroups({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyKeyGroupsAssociatedToDistributionRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyKeyGroupsAssociatedToDistribution({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyLambdaFunctionAssociationsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyLambdaFunctionAssociations({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyOriginAccessControlsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyOriginAccessControls({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyOriginCustomHeadersRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyOriginCustomHeaders({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyOriginGroupsPerDistributionRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyOriginGroupsPerDistribution({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyOriginRequestPoliciesRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyOriginRequestPolicies({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyOriginsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyOrigins({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyPublicKeysRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyPublicKeys({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyPublicKeysInKeyGroupRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyPublicKeysInKeyGroup({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyQueryStringParametersRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyQueryStringParameters({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyQueryStringsInCachePolicyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyQueryStringsInCachePolicy({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyQueryStringsInOriginRequestPolicyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyQueryStringsInOriginRequestPolicy({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyRealtimeLogConfigsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyRealtimeLogConfigs({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyRemoveHeadersInResponseHeadersPolicyRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyRemoveHeadersInResponseHeadersPolicy({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyResponseHeadersPoliciesRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyResponseHeadersPolicies({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyStreamingDistributionCNAMEsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyStreamingDistributionCNAMEs({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyStreamingDistributionsRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyStreamingDistributions({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TooManyTrustedSignersRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TooManyTrustedSigners({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TrustedKeyGroupDoesNotExistRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TrustedKeyGroupDoesNotExist({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_TrustedSignerDoesNotExistRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new TrustedSignerDoesNotExist({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const de_UnsupportedOperationRes = async (parsedOutput, context) => {
    const contents = smithyClient.map({});
    const data = parsedOutput.body.Error;
    if (data[_Me] != null) {
        contents[_Me] = smithyClient.expectString(data[_Me]);
    }
    const exception = new UnsupportedOperation({
        $metadata: deserializeMetadata(parsedOutput),
        ...contents,
    });
    return smithyClient.decorateServiceException(exception, parsedOutput.body.Error);
};
const se_AccessControlAllowHeadersList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_s, entry);
        return n.n(_H);
    });
};
const se_AccessControlAllowMethodsList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_RHPACAMV, entry);
        return n.n(_Met);
    });
};
const se_AccessControlAllowOriginsList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_s, entry);
        return n.n(_Or);
    });
};
const se_AccessControlExposeHeadersList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_s, entry);
        return n.n(_H);
    });
};
const se_Aliases = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_Al);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_AliasList(input[_It]));
    return bn;
};
const se_AliasList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_s, entry);
        return n.n(_CNAME);
    });
};
const se_AllowedMethods = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_AM);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_MethodsList(input[_It]));
    if (input[_CM] != null) {
        bn.c(se_CachedMethods(input[_CM]).n(_CM));
    }
    return bn;
};
const se_AwsAccountNumberList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_s, entry);
        return n.n(_AAN);
    });
};
const se_CacheBehavior = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CB);
    if (input[_PP] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_PP]).n(_PP));
    }
    if (input[_TOI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_TOI]).n(_TOI));
    }
    if (input[_TS] != null) {
        bn.c(se_TrustedSigners(input[_TS]).n(_TS));
    }
    if (input[_TKG] != null) {
        bn.c(se_TrustedKeyGroups(input[_TKG]).n(_TKG));
    }
    bn.cc(input, _VPP);
    if (input[_AM] != null) {
        bn.c(se_AllowedMethods(input[_AM]).n(_AM));
    }
    if (input[_SS] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_SS])).n(_SS));
    }
    if (input[_Com] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_Com])).n(_Com));
    }
    if (input[_LFA] != null) {
        bn.c(se_LambdaFunctionAssociations(input[_LFA]).n(_LFA));
    }
    if (input[_FA] != null) {
        bn.c(se_FunctionAssociations(input[_FA]).n(_FA));
    }
    if (input[_FLEI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_FLEI]).n(_FLEI));
    }
    if (input[_RLCA] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_RLCA]).n(_RLCA));
    }
    if (input[_CPI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_CPI]).n(_CPI));
    }
    if (input[_ORPI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_ORPI]).n(_ORPI));
    }
    if (input[_RHPI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_RHPI]).n(_RHPI));
    }
    if (input[_GC] != null) {
        bn.c(se_GrpcConfig(input[_GC]).n(_GC));
    }
    if (input[_FV] != null) {
        bn.c(se_ForwardedValues(input[_FV]).n(_FV));
    }
    if (input[_MTTL] != null) {
        bn.c(xmlBuilder.XmlNode.of(_l, String(input[_MTTL])).n(_MTTL));
    }
    if (input[_DTTL] != null) {
        bn.c(xmlBuilder.XmlNode.of(_l, String(input[_DTTL])).n(_DTTL));
    }
    if (input[_MTTLa] != null) {
        bn.c(xmlBuilder.XmlNode.of(_l, String(input[_MTTLa])).n(_MTTLa));
    }
    return bn;
};
const se_CacheBehaviorList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_CacheBehavior(entry);
        return n.n(_CB);
    });
};
const se_CacheBehaviors = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CBa);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_CacheBehaviorList(input[_It]));
    return bn;
};
const se_CachedMethods = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CM);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_MethodsList(input[_It]));
    return bn;
};
const se_CachePolicyConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CPC);
    if (input[_Co] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Co]).n(_Co));
    }
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_N]).n(_N));
    }
    if (input[_DTTL] != null) {
        bn.c(xmlBuilder.XmlNode.of(_l, String(input[_DTTL])).n(_DTTL));
    }
    if (input[_MTTLa] != null) {
        bn.c(xmlBuilder.XmlNode.of(_l, String(input[_MTTLa])).n(_MTTLa));
    }
    if (input[_MTTL] != null) {
        bn.c(xmlBuilder.XmlNode.of(_l, String(input[_MTTL])).n(_MTTL));
    }
    if (input[_PICKAFTO] != null) {
        bn.c(se_ParametersInCacheKeyAndForwardedToOrigin(input[_PICKAFTO]).n(_PICKAFTO));
    }
    return bn;
};
const se_CachePolicyCookiesConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CPCC);
    if (input[_CBo] != null) {
        bn.c(xmlBuilder.XmlNode.of(_CPCB, input[_CBo]).n(_CBo));
    }
    if (input[_Coo] != null) {
        bn.c(se_CookieNames(input[_Coo]).n(_Coo));
    }
    return bn;
};
const se_CachePolicyHeadersConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CPHC);
    if (input[_HB] != null) {
        bn.c(xmlBuilder.XmlNode.of(_CPHB, input[_HB]).n(_HB));
    }
    if (input[_He] != null) {
        bn.c(se_Headers(input[_He]).n(_He));
    }
    return bn;
};
const se_CachePolicyQueryStringsConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CPQSC);
    if (input[_QSB] != null) {
        bn.c(xmlBuilder.XmlNode.of(_CPQSB, input[_QSB]).n(_QSB));
    }
    if (input[_QS] != null) {
        bn.c(se_QueryStringNames(input[_QS]).n(_QS));
    }
    return bn;
};
const se_Certificate = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_Ce);
    if (input[_Ar] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Ar]).n(_Ar));
    }
    return bn;
};
const se_CloudFrontOriginAccessIdentityConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CFOAIC);
    if (input[_CR] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_CR]).n(_CR));
    }
    if (input[_Co] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Co]).n(_Co));
    }
    return bn;
};
const se_ConnectionGroupAssociationFilter = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CGAF);
    if (input[_AILI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_AILI]).n(_AILI));
    }
    return bn;
};
const se_ContentTypeProfile = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CTP);
    bn.cc(input, _Fo);
    if (input[_PI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_PI]).n(_PI));
    }
    if (input[_CT] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_CT]).n(_CT));
    }
    return bn;
};
const se_ContentTypeProfileConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CTPC);
    if (input[_FWCTIU] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_FWCTIU])).n(_FWCTIU));
    }
    if (input[_CTPo] != null) {
        bn.c(se_ContentTypeProfiles(input[_CTPo]).n(_CTPo));
    }
    return bn;
};
const se_ContentTypeProfileList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_ContentTypeProfile(entry);
        return n.n(_CTP);
    });
};
const se_ContentTypeProfiles = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CTPo);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_ContentTypeProfileList(input[_It]));
    return bn;
};
const se_ContinuousDeploymentPolicyConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CDPC);
    if (input[_SDDN] != null) {
        bn.c(se_StagingDistributionDnsNames(input[_SDDN]).n(_SDDN));
    }
    if (input[_E] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_E])).n(_E));
    }
    if (input[_TC] != null) {
        bn.c(se_TrafficConfig(input[_TC]).n(_TC));
    }
    return bn;
};
const se_ContinuousDeploymentSingleHeaderConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CDSHC);
    if (input[_H] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_H]).n(_H));
    }
    if (input[_V] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_V]).n(_V));
    }
    return bn;
};
const se_ContinuousDeploymentSingleWeightConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CDSWC);
    if (input[_W] != null) {
        bn.c(xmlBuilder.XmlNode.of(_f, String(input[_W])).n(_W));
    }
    if (input[_SSC] != null) {
        bn.c(se_SessionStickinessConfig(input[_SSC]).n(_SSC));
    }
    return bn;
};
const se_CookieNameList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_s, entry);
        return n.n(_N);
    });
};
const se_CookieNames = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CN);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_CookieNameList(input[_It]));
    return bn;
};
const se_CookiePreference = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CP);
    if (input[_For] != null) {
        bn.c(xmlBuilder.XmlNode.of(_ISt, input[_For]).n(_For));
    }
    if (input[_WN] != null) {
        bn.c(se_CookieNames(input[_WN]).n(_WN));
    }
    return bn;
};
const se_CustomErrorResponse = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CER);
    if (input[_EC] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_EC])).n(_EC));
    }
    if (input[_RPP] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_RPP]).n(_RPP));
    }
    if (input[_RC] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_RC]).n(_RC));
    }
    if (input[_ECMTTL] != null) {
        bn.c(xmlBuilder.XmlNode.of(_l, String(input[_ECMTTL])).n(_ECMTTL));
    }
    return bn;
};
const se_CustomErrorResponseList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_CustomErrorResponse(entry);
        return n.n(_CER);
    });
};
const se_CustomErrorResponses = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CERu);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_CustomErrorResponseList(input[_It]));
    return bn;
};
const se_CustomHeaders = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_CH);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_OriginCustomHeadersList(input[_It]));
    return bn;
};
const se_Customizations = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_C);
    if (input[_WA] != null) {
        bn.c(se_WebAclCustomization(input[_WA]).n(_WA));
    }
    if (input[_Ce] != null) {
        bn.c(se_Certificate(input[_Ce]).n(_Ce));
    }
    if (input[_GR] != null) {
        bn.c(se_GeoRestrictionCustomization(input[_GR]).n(_GR));
    }
    return bn;
};
const se_CustomOriginConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_COC);
    if (input[_HTTPP] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_HTTPP])).n(_HTTPP));
    }
    if (input[_HTTPSP] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_HTTPSP])).n(_HTTPSP));
    }
    bn.cc(input, _OPP);
    if (input[_OSP] != null) {
        bn.c(se_OriginSslProtocols(input[_OSP]).n(_OSP));
    }
    if (input[_ORT] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_ORT])).n(_ORT));
    }
    if (input[_OKT] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_OKT])).n(_OKT));
    }
    bn.cc(input, _IAT);
    return bn;
};
const se_DefaultCacheBehavior = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_DCB);
    if (input[_TOI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_TOI]).n(_TOI));
    }
    if (input[_TS] != null) {
        bn.c(se_TrustedSigners(input[_TS]).n(_TS));
    }
    if (input[_TKG] != null) {
        bn.c(se_TrustedKeyGroups(input[_TKG]).n(_TKG));
    }
    bn.cc(input, _VPP);
    if (input[_AM] != null) {
        bn.c(se_AllowedMethods(input[_AM]).n(_AM));
    }
    if (input[_SS] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_SS])).n(_SS));
    }
    if (input[_Com] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_Com])).n(_Com));
    }
    if (input[_LFA] != null) {
        bn.c(se_LambdaFunctionAssociations(input[_LFA]).n(_LFA));
    }
    if (input[_FA] != null) {
        bn.c(se_FunctionAssociations(input[_FA]).n(_FA));
    }
    if (input[_FLEI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_FLEI]).n(_FLEI));
    }
    if (input[_RLCA] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_RLCA]).n(_RLCA));
    }
    if (input[_CPI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_CPI]).n(_CPI));
    }
    if (input[_ORPI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_ORPI]).n(_ORPI));
    }
    if (input[_RHPI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_RHPI]).n(_RHPI));
    }
    if (input[_GC] != null) {
        bn.c(se_GrpcConfig(input[_GC]).n(_GC));
    }
    if (input[_FV] != null) {
        bn.c(se_ForwardedValues(input[_FV]).n(_FV));
    }
    if (input[_MTTL] != null) {
        bn.c(xmlBuilder.XmlNode.of(_l, String(input[_MTTL])).n(_MTTL));
    }
    if (input[_DTTL] != null) {
        bn.c(xmlBuilder.XmlNode.of(_l, String(input[_DTTL])).n(_DTTL));
    }
    if (input[_MTTLa] != null) {
        bn.c(xmlBuilder.XmlNode.of(_l, String(input[_MTTLa])).n(_MTTLa));
    }
    return bn;
};
const se_DistributionConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_DCi);
    if (input[_CR] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_CR]).n(_CR));
    }
    if (input[_Al] != null) {
        bn.c(se_Aliases(input[_Al]).n(_Al));
    }
    if (input[_DRO] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_DRO]).n(_DRO));
    }
    if (input[_Ori] != null) {
        bn.c(se_Origins(input[_Ori]).n(_Ori));
    }
    if (input[_OG] != null) {
        bn.c(se_OriginGroups(input[_OG]).n(_OG));
    }
    if (input[_DCB] != null) {
        bn.c(se_DefaultCacheBehavior(input[_DCB]).n(_DCB));
    }
    if (input[_CBa] != null) {
        bn.c(se_CacheBehaviors(input[_CBa]).n(_CBa));
    }
    if (input[_CERu] != null) {
        bn.c(se_CustomErrorResponses(input[_CERu]).n(_CERu));
    }
    if (input[_Co] != null) {
        bn.c(xmlBuilder.XmlNode.of(_CTo, input[_Co]).n(_Co));
    }
    if (input[_Lo] != null) {
        bn.c(se_LoggingConfig(input[_Lo]).n(_Lo));
    }
    bn.cc(input, _PC);
    if (input[_E] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_E])).n(_E));
    }
    if (input[_VC] != null) {
        bn.c(se_ViewerCertificate(input[_VC]).n(_VC));
    }
    if (input[_Re] != null) {
        bn.c(se_Restrictions(input[_Re]).n(_Re));
    }
    if (input[_WACLI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_WACLI]).n(_WACLI));
    }
    bn.cc(input, _HV);
    if (input[_IIPVE] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_IIPVE])).n(_IIPVE));
    }
    if (input[_CDPI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_CDPI]).n(_CDPI));
    }
    if (input[_S] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_S])).n(_S));
    }
    if (input[_AILI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_AILI]).n(_AILI));
    }
    if (input[_TCe] != null) {
        bn.c(se_TenantConfig(input[_TCe]).n(_TCe));
    }
    bn.cc(input, _CMo);
    return bn;
};
const se_DistributionConfigWithTags = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_DCWT);
    if (input[_DCi] != null) {
        bn.c(se_DistributionConfig(input[_DCi]).n(_DCi));
    }
    if (input[_T] != null) {
        bn.c(se_Tags(input[_T]).n(_T));
    }
    return bn;
};
const se_DistributionResourceId = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_DRI);
    if (input[_DI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_DI]).n(_DI));
    }
    if (input[_DTI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_DTI]).n(_DTI));
    }
    return bn;
};
const se_DistributionTenantAssociationFilter = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_DTAF);
    if (input[_DI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_DI]).n(_DI));
    }
    if (input[_CGI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_CGI]).n(_CGI));
    }
    return bn;
};
const se_DomainItem = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_DIo);
    if (input[_Do] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Do]).n(_Do));
    }
    return bn;
};
const se_DomainList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_DomainItem(entry);
        return n.n(_m);
    });
};
const se_EncryptionEntities = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_EE);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_EncryptionEntityList(input[_It]));
    return bn;
};
const se_EncryptionEntity = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_EEn);
    if (input[_PKI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_PKI]).n(_PKI));
    }
    if (input[_PIr] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_PIr]).n(_PIr));
    }
    if (input[_FP] != null) {
        bn.c(se_FieldPatterns(input[_FP]).n(_FP));
    }
    return bn;
};
const se_EncryptionEntityList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_EncryptionEntity(entry);
        return n.n(_EEn);
    });
};
const se_EndPoint = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_EPn);
    if (input[_ST] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_ST]).n(_ST));
    }
    if (input[_KSC] != null) {
        bn.c(se_KinesisStreamConfig(input[_KSC]).n(_KSC));
    }
    return bn;
};
const se_EndPointList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_EndPoint(entry);
        return n.n(_m);
    });
};
const se_FieldLevelEncryptionConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_FLEC);
    if (input[_CR] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_CR]).n(_CR));
    }
    if (input[_Co] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Co]).n(_Co));
    }
    if (input[_QAPC] != null) {
        bn.c(se_QueryArgProfileConfig(input[_QAPC]).n(_QAPC));
    }
    if (input[_CTPC] != null) {
        bn.c(se_ContentTypeProfileConfig(input[_CTPC]).n(_CTPC));
    }
    return bn;
};
const se_FieldLevelEncryptionProfileConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_FLEPC);
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_N]).n(_N));
    }
    if (input[_CR] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_CR]).n(_CR));
    }
    if (input[_Co] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Co]).n(_Co));
    }
    if (input[_EE] != null) {
        bn.c(se_EncryptionEntities(input[_EE]).n(_EE));
    }
    return bn;
};
const se_FieldList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_s, entry);
        return n.n(_Fi);
    });
};
const se_FieldPatternList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_s, entry);
        return n.n(_FPi);
    });
};
const se_FieldPatterns = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_FP);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_FieldPatternList(input[_It]));
    return bn;
};
const se_ForwardedValues = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_FV);
    if (input[_QSu] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_QSu])).n(_QSu));
    }
    if (input[_Coo] != null) {
        bn.c(se_CookiePreference(input[_Coo]).n(_Coo));
    }
    if (input[_He] != null) {
        bn.c(se_Headers(input[_He]).n(_He));
    }
    if (input[_QSCK] != null) {
        bn.c(se_QueryStringCacheKeys(input[_QSCK]).n(_QSCK));
    }
    return bn;
};
const se_FunctionAssociation = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_FAu);
    bn.cc(input, _FARN);
    bn.cc(input, _ETv);
    return bn;
};
const se_FunctionAssociationList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_FunctionAssociation(entry);
        return n.n(_FAu);
    });
};
const se_FunctionAssociations = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_FA);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_FunctionAssociationList(input[_It]));
    return bn;
};
const se_FunctionConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_FCu);
    if (input[_Co] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Co]).n(_Co));
    }
    if (input[_Ru] != null) {
        bn.c(xmlBuilder.XmlNode.of(_FR, input[_Ru]).n(_Ru));
    }
    if (input[_KVSA] != null) {
        bn.c(se_KeyValueStoreAssociations(input[_KVSA]).n(_KVSA));
    }
    return bn;
};
const se_GeoRestriction = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_GRe);
    if (input[_RT] != null) {
        bn.c(xmlBuilder.XmlNode.of(_GRT, input[_RT]).n(_RT));
    }
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_LocationList(input[_It]));
    return bn;
};
const se_GeoRestrictionCustomization = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_GRC);
    if (input[_RT] != null) {
        bn.c(xmlBuilder.XmlNode.of(_GRT, input[_RT]).n(_RT));
    }
    bn.lc(input, "Locations", "Locations", () => se_LocationList(input[_Loc]));
    return bn;
};
const se_GrpcConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_GC);
    if (input[_E] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_E])).n(_E));
    }
    return bn;
};
const se_HeaderList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_s, entry);
        return n.n(_N);
    });
};
const se_Headers = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_He);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_HeaderList(input[_It]));
    return bn;
};
const se_ImportSource = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_IS);
    if (input[_STo] != null) {
        bn.c(xmlBuilder.XmlNode.of(_IST, input[_STo]).n(_STo));
    }
    if (input[_SARN] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_SARN]).n(_SARN));
    }
    return bn;
};
const se_InvalidationBatch = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_IB);
    if (input[_Pa] != null) {
        bn.c(se_Paths(input[_Pa]).n(_Pa));
    }
    if (input[_CR] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_CR]).n(_CR));
    }
    return bn;
};
const se_KeyGroupConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_KGC);
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_N]).n(_N));
    }
    bn.lc(input, "Items", "Items", () => se_PublicKeyIdList(input[_It]));
    if (input[_Co] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Co]).n(_Co));
    }
    return bn;
};
const se_KeyValueStoreAssociation = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_KVSAe);
    bn.cc(input, _KVSARN);
    return bn;
};
const se_KeyValueStoreAssociationList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_KeyValueStoreAssociation(entry);
        return n.n(_KVSAe);
    });
};
const se_KeyValueStoreAssociations = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_KVSA);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_KeyValueStoreAssociationList(input[_It]));
    return bn;
};
const se_KinesisStreamConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_KSC);
    if (input[_RARN] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_RARN]).n(_RARN));
    }
    if (input[_SARNt] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_SARNt]).n(_SARNt));
    }
    return bn;
};
const se_LambdaFunctionAssociation = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_LFAa);
    bn.cc(input, _LFARN);
    bn.cc(input, _ETv);
    if (input[_IBn] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_IBn])).n(_IBn));
    }
    return bn;
};
const se_LambdaFunctionAssociationList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_LambdaFunctionAssociation(entry);
        return n.n(_LFAa);
    });
};
const se_LambdaFunctionAssociations = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_LFA);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_LambdaFunctionAssociationList(input[_It]));
    return bn;
};
const se_LocationList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_s, entry);
        return n.n(_L);
    });
};
const se_LoggingConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_LC);
    if (input[_E] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_E])).n(_E));
    }
    if (input[_ICn] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_ICn])).n(_ICn));
    }
    if (input[_B] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_B]).n(_B));
    }
    if (input[_Pr] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Pr]).n(_Pr));
    }
    return bn;
};
const se_ManagedCertificateRequest = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_MCR);
    bn.cc(input, _VTH);
    if (input[_PDN] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_PDN]).n(_PDN));
    }
    bn.cc(input, _CTLP);
    return bn;
};
const se_MethodsList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_Met, entry);
        return n.n(_Met);
    });
};
const se_MonitoringSubscription = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_MS);
    if (input[_RMSC] != null) {
        bn.c(se_RealtimeMetricsSubscriptionConfig(input[_RMSC]).n(_RMSC));
    }
    return bn;
};
const se_Origin = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_Or);
    if (input[_Id] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Id]).n(_Id));
    }
    if (input[_DN] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_DN]).n(_DN));
    }
    if (input[_OP] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_OP]).n(_OP));
    }
    if (input[_CH] != null) {
        bn.c(se_CustomHeaders(input[_CH]).n(_CH));
    }
    if (input[_SOC] != null) {
        bn.c(se_S3OriginConfig(input[_SOC]).n(_SOC));
    }
    if (input[_COC] != null) {
        bn.c(se_CustomOriginConfig(input[_COC]).n(_COC));
    }
    if (input[_VOC] != null) {
        bn.c(se_VpcOriginConfig(input[_VOC]).n(_VOC));
    }
    if (input[_CAo] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_CAo])).n(_CAo));
    }
    if (input[_CTon] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_CTon])).n(_CTon));
    }
    if (input[_RCT] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_RCT])).n(_RCT));
    }
    if (input[_OS] != null) {
        bn.c(se_OriginShield(input[_OS]).n(_OS));
    }
    if (input[_OACI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_OACI]).n(_OACI));
    }
    return bn;
};
const se_OriginAccessControlConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_OACC);
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_N]).n(_N));
    }
    if (input[_De] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_De]).n(_De));
    }
    if (input[_SP] != null) {
        bn.c(xmlBuilder.XmlNode.of(_OACSP, input[_SP]).n(_SP));
    }
    if (input[_SB] != null) {
        bn.c(xmlBuilder.XmlNode.of(_OACSB, input[_SB]).n(_SB));
    }
    if (input[_OACOT] != null) {
        bn.c(xmlBuilder.XmlNode.of(_OACOTr, input[_OACOT]).n(_OACOT));
    }
    return bn;
};
const se_OriginCustomHeader = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_OCH);
    if (input[_HN] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_HN]).n(_HN));
    }
    if (input[_HVe] != null) {
        bn.c(xmlBuilder.XmlNode.of(_sST, input[_HVe]).n(_HVe));
    }
    return bn;
};
const se_OriginCustomHeadersList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_OriginCustomHeader(entry);
        return n.n(_OCH);
    });
};
const se_OriginGroup = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_OGr);
    if (input[_Id] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Id]).n(_Id));
    }
    if (input[_FCa] != null) {
        bn.c(se_OriginGroupFailoverCriteria(input[_FCa]).n(_FCa));
    }
    if (input[_Mem] != null) {
        bn.c(se_OriginGroupMembers(input[_Mem]).n(_Mem));
    }
    if (input[_SC] != null) {
        bn.c(xmlBuilder.XmlNode.of(_OGSC, input[_SC]).n(_SC));
    }
    return bn;
};
const se_OriginGroupFailoverCriteria = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_OGFC);
    if (input[_SCt] != null) {
        bn.c(se_StatusCodes(input[_SCt]).n(_SCt));
    }
    return bn;
};
const se_OriginGroupList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_OriginGroup(entry);
        return n.n(_OGr);
    });
};
const se_OriginGroupMember = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_OGM);
    if (input[_OI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_OI]).n(_OI));
    }
    return bn;
};
const se_OriginGroupMemberList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_OriginGroupMember(entry);
        return n.n(_OGM);
    });
};
const se_OriginGroupMembers = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_OGMr);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_OriginGroupMemberList(input[_It]));
    return bn;
};
const se_OriginGroups = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_OG);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_OriginGroupList(input[_It]));
    return bn;
};
const se_OriginList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_Origin(entry);
        return n.n(_Or);
    });
};
const se_OriginRequestPolicyConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_ORPC);
    if (input[_Co] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Co]).n(_Co));
    }
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_N]).n(_N));
    }
    if (input[_HC] != null) {
        bn.c(se_OriginRequestPolicyHeadersConfig(input[_HC]).n(_HC));
    }
    if (input[_CC] != null) {
        bn.c(se_OriginRequestPolicyCookiesConfig(input[_CC]).n(_CC));
    }
    if (input[_QSC] != null) {
        bn.c(se_OriginRequestPolicyQueryStringsConfig(input[_QSC]).n(_QSC));
    }
    return bn;
};
const se_OriginRequestPolicyCookiesConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_ORPCC);
    if (input[_CBo] != null) {
        bn.c(xmlBuilder.XmlNode.of(_ORPCB, input[_CBo]).n(_CBo));
    }
    if (input[_Coo] != null) {
        bn.c(se_CookieNames(input[_Coo]).n(_Coo));
    }
    return bn;
};
const se_OriginRequestPolicyHeadersConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_ORPHC);
    if (input[_HB] != null) {
        bn.c(xmlBuilder.XmlNode.of(_ORPHB, input[_HB]).n(_HB));
    }
    if (input[_He] != null) {
        bn.c(se_Headers(input[_He]).n(_He));
    }
    return bn;
};
const se_OriginRequestPolicyQueryStringsConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_ORPQSC);
    if (input[_QSB] != null) {
        bn.c(xmlBuilder.XmlNode.of(_ORPQSB, input[_QSB]).n(_QSB));
    }
    if (input[_QS] != null) {
        bn.c(se_QueryStringNames(input[_QS]).n(_QS));
    }
    return bn;
};
const se_Origins = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_Ori);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_OriginList(input[_It]));
    return bn;
};
const se_OriginShield = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_OS);
    if (input[_E] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_E])).n(_E));
    }
    bn.cc(input, _OSR);
    return bn;
};
const se_OriginSslProtocols = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_OSP);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_SslProtocolsList(input[_It]));
    return bn;
};
const se_Parameter = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_Par);
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_PN, input[_N]).n(_N));
    }
    if (input[_V] != null) {
        bn.c(xmlBuilder.XmlNode.of(_PV, input[_V]).n(_V));
    }
    return bn;
};
const se_ParameterDefinition = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_PD);
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_PN, input[_N]).n(_N));
    }
    if (input[_Def] != null) {
        bn.c(se_ParameterDefinitionSchema(input[_Def]).n(_Def));
    }
    return bn;
};
const se_ParameterDefinitions = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_ParameterDefinition(entry);
        return n.n(_m);
    });
};
const se_ParameterDefinitionSchema = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_PDS);
    if (input[_SSt] != null) {
        bn.c(se_StringSchemaConfig(input[_SSt]).n(_SSt));
    }
    return bn;
};
const se_Parameters = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_Parameter(entry);
        return n.n(_m);
    });
};
const se_ParametersInCacheKeyAndForwardedToOrigin = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_PICKAFTO);
    if (input[_EAEG] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_EAEG])).n(_EAEG));
    }
    if (input[_EAEB] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_EAEB])).n(_EAEB));
    }
    if (input[_HC] != null) {
        bn.c(se_CachePolicyHeadersConfig(input[_HC]).n(_HC));
    }
    if (input[_CC] != null) {
        bn.c(se_CachePolicyCookiesConfig(input[_CC]).n(_CC));
    }
    if (input[_QSC] != null) {
        bn.c(se_CachePolicyQueryStringsConfig(input[_QSC]).n(_QSC));
    }
    return bn;
};
const se_PathList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_s, entry);
        return n.n(_Pat);
    });
};
const se_Paths = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_Pa);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_PathList(input[_It]));
    return bn;
};
const se_PublicKeyConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_PKC);
    if (input[_CR] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_CR]).n(_CR));
    }
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_N]).n(_N));
    }
    if (input[_EK] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_EK]).n(_EK));
    }
    if (input[_Co] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Co]).n(_Co));
    }
    return bn;
};
const se_PublicKeyIdList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_s, entry);
        return n.n(_PK);
    });
};
const se_QueryArgProfile = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_QAP);
    if (input[_QA] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_QA]).n(_QA));
    }
    if (input[_PI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_PI]).n(_PI));
    }
    return bn;
};
const se_QueryArgProfileConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_QAPC);
    if (input[_FWQAPIU] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_FWQAPIU])).n(_FWQAPIU));
    }
    if (input[_QAPu] != null) {
        bn.c(se_QueryArgProfiles(input[_QAPu]).n(_QAPu));
    }
    return bn;
};
const se_QueryArgProfileList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_QueryArgProfile(entry);
        return n.n(_QAP);
    });
};
const se_QueryArgProfiles = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_QAPu);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_QueryArgProfileList(input[_It]));
    return bn;
};
const se_QueryStringCacheKeys = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_QSCK);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_QueryStringCacheKeysList(input[_It]));
    return bn;
};
const se_QueryStringCacheKeysList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_s, entry);
        return n.n(_N);
    });
};
const se_QueryStringNames = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_QSN);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_QueryStringNamesList(input[_It]));
    return bn;
};
const se_QueryStringNamesList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_s, entry);
        return n.n(_N);
    });
};
const se_RealtimeMetricsSubscriptionConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RMSC);
    bn.cc(input, _RMSS);
    return bn;
};
const se_ResponseHeadersPolicyAccessControlAllowHeaders = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPACAH);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_AccessControlAllowHeadersList(input[_It]));
    return bn;
};
const se_ResponseHeadersPolicyAccessControlAllowMethods = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPACAM);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_AccessControlAllowMethodsList(input[_It]));
    return bn;
};
const se_ResponseHeadersPolicyAccessControlAllowOrigins = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPACAO);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_AccessControlAllowOriginsList(input[_It]));
    return bn;
};
const se_ResponseHeadersPolicyAccessControlExposeHeaders = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPACEH);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_AccessControlExposeHeadersList(input[_It]));
    return bn;
};
const se_ResponseHeadersPolicyConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPC);
    if (input[_Co] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Co]).n(_Co));
    }
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_N]).n(_N));
    }
    if (input[_CCo] != null) {
        bn.c(se_ResponseHeadersPolicyCorsConfig(input[_CCo]).n(_CCo));
    }
    if (input[_SHC] != null) {
        bn.c(se_ResponseHeadersPolicySecurityHeadersConfig(input[_SHC]).n(_SHC));
    }
    if (input[_STHC] != null) {
        bn.c(se_ResponseHeadersPolicyServerTimingHeadersConfig(input[_STHC]).n(_STHC));
    }
    if (input[_CHC] != null) {
        bn.c(se_ResponseHeadersPolicyCustomHeadersConfig(input[_CHC]).n(_CHC));
    }
    if (input[_RHC] != null) {
        bn.c(se_ResponseHeadersPolicyRemoveHeadersConfig(input[_RHC]).n(_RHC));
    }
    return bn;
};
const se_ResponseHeadersPolicyContentSecurityPolicy = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPCSP);
    if (input[_Ov] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_Ov])).n(_Ov));
    }
    if (input[_CSP] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_CSP]).n(_CSP));
    }
    return bn;
};
const se_ResponseHeadersPolicyContentTypeOptions = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPCTO);
    if (input[_Ov] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_Ov])).n(_Ov));
    }
    return bn;
};
const se_ResponseHeadersPolicyCorsConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPCC);
    if (input[_ACAO] != null) {
        bn.c(se_ResponseHeadersPolicyAccessControlAllowOrigins(input[_ACAO]).n(_ACAO));
    }
    if (input[_ACAH] != null) {
        bn.c(se_ResponseHeadersPolicyAccessControlAllowHeaders(input[_ACAH]).n(_ACAH));
    }
    if (input[_ACAM] != null) {
        bn.c(se_ResponseHeadersPolicyAccessControlAllowMethods(input[_ACAM]).n(_ACAM));
    }
    if (input[_ACAC] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_ACAC])).n(_ACAC));
    }
    if (input[_ACEH] != null) {
        bn.c(se_ResponseHeadersPolicyAccessControlExposeHeaders(input[_ACEH]).n(_ACEH));
    }
    if (input[_ACMAS] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_ACMAS])).n(_ACMAS));
    }
    if (input[_OO] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_OO])).n(_OO));
    }
    return bn;
};
const se_ResponseHeadersPolicyCustomHeader = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPCH);
    if (input[_H] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_H]).n(_H));
    }
    if (input[_V] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_V]).n(_V));
    }
    if (input[_Ov] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_Ov])).n(_Ov));
    }
    return bn;
};
const se_ResponseHeadersPolicyCustomHeaderList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_ResponseHeadersPolicyCustomHeader(entry);
        return n.n(_RHPCH);
    });
};
const se_ResponseHeadersPolicyCustomHeadersConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPCHC);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_ResponseHeadersPolicyCustomHeaderList(input[_It]));
    return bn;
};
const se_ResponseHeadersPolicyFrameOptions = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPFO);
    if (input[_Ov] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_Ov])).n(_Ov));
    }
    if (input[_FO] != null) {
        bn.c(xmlBuilder.XmlNode.of(_FOL, input[_FO]).n(_FO));
    }
    return bn;
};
const se_ResponseHeadersPolicyReferrerPolicy = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPRP);
    if (input[_Ov] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_Ov])).n(_Ov));
    }
    if (input[_RP] != null) {
        bn.c(xmlBuilder.XmlNode.of(_RPL, input[_RP]).n(_RP));
    }
    return bn;
};
const se_ResponseHeadersPolicyRemoveHeader = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPRH);
    if (input[_H] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_H]).n(_H));
    }
    return bn;
};
const se_ResponseHeadersPolicyRemoveHeaderList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_ResponseHeadersPolicyRemoveHeader(entry);
        return n.n(_RHPRH);
    });
};
const se_ResponseHeadersPolicyRemoveHeadersConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPRHC);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_ResponseHeadersPolicyRemoveHeaderList(input[_It]));
    return bn;
};
const se_ResponseHeadersPolicySecurityHeadersConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPSHC);
    if (input[_XSSP] != null) {
        bn.c(se_ResponseHeadersPolicyXSSProtection(input[_XSSP]).n(_XSSP));
    }
    if (input[_FOr] != null) {
        bn.c(se_ResponseHeadersPolicyFrameOptions(input[_FOr]).n(_FOr));
    }
    if (input[_RP] != null) {
        bn.c(se_ResponseHeadersPolicyReferrerPolicy(input[_RP]).n(_RP));
    }
    if (input[_CSP] != null) {
        bn.c(se_ResponseHeadersPolicyContentSecurityPolicy(input[_CSP]).n(_CSP));
    }
    if (input[_CTO] != null) {
        bn.c(se_ResponseHeadersPolicyContentTypeOptions(input[_CTO]).n(_CTO));
    }
    if (input[_STS] != null) {
        bn.c(se_ResponseHeadersPolicyStrictTransportSecurity(input[_STS]).n(_STS));
    }
    return bn;
};
const se_ResponseHeadersPolicyServerTimingHeadersConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPSTHC);
    if (input[_E] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_E])).n(_E));
    }
    if (input[_SR] != null) {
        bn.c(xmlBuilder.XmlNode.of(_SR, String(input[_SR])).n(_SR));
    }
    return bn;
};
const se_ResponseHeadersPolicyStrictTransportSecurity = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPSTS);
    if (input[_Ov] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_Ov])).n(_Ov));
    }
    if (input[_ISn] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_ISn])).n(_ISn));
    }
    if (input[_Pre] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_Pre])).n(_Pre));
    }
    if (input[_ACMAS] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_ACMAS])).n(_ACMAS));
    }
    return bn;
};
const se_ResponseHeadersPolicyXSSProtection = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_RHPXSSP);
    if (input[_Ov] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_Ov])).n(_Ov));
    }
    if (input[_Pro] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_Pro])).n(_Pro));
    }
    if (input[_MB] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_MB])).n(_MB));
    }
    if (input[_RU] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_RU]).n(_RU));
    }
    return bn;
};
const se_Restrictions = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_Re);
    if (input[_GRe] != null) {
        bn.c(se_GeoRestriction(input[_GRe]).n(_GRe));
    }
    return bn;
};
const se_S3Origin = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_SO);
    if (input[_DN] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_DN]).n(_DN));
    }
    if (input[_OAI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_OAI]).n(_OAI));
    }
    return bn;
};
const se_S3OriginConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_SOC);
    if (input[_OAI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_OAI]).n(_OAI));
    }
    if (input[_ORT] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_ORT])).n(_ORT));
    }
    return bn;
};
const se_SessionStickinessConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_SSC);
    if (input[_ITTL] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_ITTL])).n(_ITTL));
    }
    if (input[_MTTLax] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_MTTLax])).n(_MTTLax));
    }
    return bn;
};
const se_SslProtocolsList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_SPs, entry);
        return n.n(_SPs);
    });
};
const se_StagingDistributionDnsNameList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_s, entry);
        return n.n(_DNn);
    });
};
const se_StagingDistributionDnsNames = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_SDDN);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_StagingDistributionDnsNameList(input[_It]));
    return bn;
};
const se_StatusCodeList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_i, String(entry));
        return n.n(_SCta);
    });
};
const se_StatusCodes = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_SCt);
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_StatusCodeList(input[_It]));
    return bn;
};
const se_StreamingDistributionConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_SDC);
    if (input[_CR] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_CR]).n(_CR));
    }
    if (input[_SO] != null) {
        bn.c(se_S3Origin(input[_SO]).n(_SO));
    }
    if (input[_Al] != null) {
        bn.c(se_Aliases(input[_Al]).n(_Al));
    }
    if (input[_Co] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Co]).n(_Co));
    }
    if (input[_Lo] != null) {
        bn.c(se_StreamingLoggingConfig(input[_Lo]).n(_Lo));
    }
    if (input[_TS] != null) {
        bn.c(se_TrustedSigners(input[_TS]).n(_TS));
    }
    bn.cc(input, _PC);
    if (input[_E] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_E])).n(_E));
    }
    return bn;
};
const se_StreamingDistributionConfigWithTags = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_SDCWT);
    if (input[_SDC] != null) {
        bn.c(se_StreamingDistributionConfig(input[_SDC]).n(_SDC));
    }
    if (input[_T] != null) {
        bn.c(se_Tags(input[_T]).n(_T));
    }
    return bn;
};
const se_StreamingLoggingConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_SLC);
    if (input[_E] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_E])).n(_E));
    }
    if (input[_B] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_B]).n(_B));
    }
    if (input[_Pr] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Pr]).n(_Pr));
    }
    return bn;
};
const se_StringSchemaConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_SSCt);
    if (input[_Co] != null) {
        bn.c(xmlBuilder.XmlNode.of(_sST, input[_Co]).n(_Co));
    }
    if (input[_DV] != null) {
        bn.c(xmlBuilder.XmlNode.of(_PV, input[_DV]).n(_DV));
    }
    if (input[_Req] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_Req])).n(_Req));
    }
    return bn;
};
const se_Tag = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_Ta);
    if (input[_K] != null) {
        bn.c(xmlBuilder.XmlNode.of(_TK, input[_K]).n(_K));
    }
    if (input[_V] != null) {
        bn.c(xmlBuilder.XmlNode.of(_TV, input[_V]).n(_V));
    }
    return bn;
};
const se_TagKeyList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_TK, entry);
        return n.n(_K);
    });
};
const se_TagKeys = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_TKa);
    bn.lc(input, "Items", "Items", () => se_TagKeyList(input[_It]));
    return bn;
};
const se_TagList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = se_Tag(entry);
        return n.n(_Ta);
    });
};
const se_Tags = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_T);
    bn.lc(input, "Items", "Items", () => se_TagList(input[_It]));
    return bn;
};
const se_TenantConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_TCe);
    bn.lc(input, "ParameterDefinitions", "ParameterDefinitions", () => se_ParameterDefinitions(input[_PDa]));
    return bn;
};
const se_TrafficConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_TC);
    if (input[_SWC] != null) {
        bn.c(se_ContinuousDeploymentSingleWeightConfig(input[_SWC]).n(_SWC));
    }
    if (input[_SHCi] != null) {
        bn.c(se_ContinuousDeploymentSingleHeaderConfig(input[_SHCi]).n(_SHCi));
    }
    if (input[_Ty] != null) {
        bn.c(xmlBuilder.XmlNode.of(_CDPT, input[_Ty]).n(_Ty));
    }
    return bn;
};
const se_TrustedKeyGroupIdList = (input, context) => {
    return input
        .filter((e) => e != null)
        .map((entry) => {
        const n = xmlBuilder.XmlNode.of(_s, entry);
        return n.n(_KG);
    });
};
const se_TrustedKeyGroups = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_TKG);
    if (input[_E] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_E])).n(_E));
    }
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_TrustedKeyGroupIdList(input[_It]));
    return bn;
};
const se_TrustedSigners = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_TS);
    if (input[_E] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_E])).n(_E));
    }
    if (input[_Q] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_Q])).n(_Q));
    }
    bn.lc(input, "Items", "Items", () => se_AwsAccountNumberList(input[_It]));
    return bn;
};
const se_ViewerCertificate = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_VC);
    if (input[_CFDC] != null) {
        bn.c(xmlBuilder.XmlNode.of(_b, String(input[_CFDC])).n(_CFDC));
    }
    if (input[_IAMCI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_SCI, input[_IAMCI]).n(_IAMCI));
    }
    if (input[_ACMCA] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_ACMCA]).n(_ACMCA));
    }
    bn.cc(input, _SSLSM);
    bn.cc(input, _MPV);
    if (input[_Ce] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Ce]).n(_Ce));
    }
    bn.cc(input, _CS);
    return bn;
};
const se_VpcOriginConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_VOC);
    if (input[_VOI] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_VOI]).n(_VOI));
    }
    if (input[_ORT] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_ORT])).n(_ORT));
    }
    if (input[_OKT] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_OKT])).n(_OKT));
    }
    return bn;
};
const se_VpcOriginEndpointConfig = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_VOEC);
    if (input[_N] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_N]).n(_N));
    }
    if (input[_Ar] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Ar]).n(_Ar));
    }
    if (input[_HTTPP] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_HTTPP])).n(_HTTPP));
    }
    if (input[_HTTPSP] != null) {
        bn.c(xmlBuilder.XmlNode.of(_i, String(input[_HTTPSP])).n(_HTTPSP));
    }
    bn.cc(input, _OPP);
    if (input[_OSP] != null) {
        bn.c(se_OriginSslProtocols(input[_OSP]).n(_OSP));
    }
    return bn;
};
const se_WebAclCustomization = (input, context) => {
    const bn = new xmlBuilder.XmlNode(_WAC);
    if (input[_Ac] != null) {
        bn.c(xmlBuilder.XmlNode.of(_CAT, input[_Ac]).n(_Ac));
    }
    if (input[_Ar] != null) {
        bn.c(xmlBuilder.XmlNode.of(_s, input[_Ar]).n(_Ar));
    }
    return bn;
};
const de_AccessControlAllowHeadersList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_AccessControlAllowMethodsList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_AccessControlAllowOriginsList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_AccessControlExposeHeadersList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_ActiveTrustedKeyGroups = (output, context) => {
    const contents = {};
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_KG] != null) {
        contents[_It] = de_KGKeyPairIdsList(smithyClient.getArrayIfSingleItem(output[_It][_KG]));
    }
    return contents;
};
const de_ActiveTrustedSigners = (output, context) => {
    const contents = {};
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_Si] != null) {
        contents[_It] = de_SignerList(smithyClient.getArrayIfSingleItem(output[_It][_Si]));
    }
    return contents;
};
const de_Aliases = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_CNAME] != null) {
        contents[_It] = de_AliasList(smithyClient.getArrayIfSingleItem(output[_It][_CNAME]));
    }
    return contents;
};
const de_AliasICPRecordal = (output, context) => {
    const contents = {};
    if (output[_CNAME] != null) {
        contents[_CNAME] = smithyClient.expectString(output[_CNAME]);
    }
    if (output[_ICPRS] != null) {
        contents[_ICPRS] = smithyClient.expectString(output[_ICPRS]);
    }
    return contents;
};
const de_AliasICPRecordals = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_AliasICPRecordal(entry);
    });
};
const de_AliasList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_AllowedMethods = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_Met] != null) {
        contents[_It] = de_MethodsList(smithyClient.getArrayIfSingleItem(output[_It][_Met]));
    }
    if (output[_CM] != null) {
        contents[_CM] = de_CachedMethods(output[_CM]);
    }
    return contents;
};
const de_AnycastIpList = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    if (output[_Ar] != null) {
        contents[_Ar] = smithyClient.expectString(output[_Ar]);
    }
    if (String(output.AnycastIps).trim() === "") {
        contents[_AI] = [];
    }
    else if (output[_AI] != null && output[_AI][_AIn] != null) {
        contents[_AI] = de_AnycastIps(smithyClient.getArrayIfSingleItem(output[_AI][_AIn]));
    }
    if (output[_IC] != null) {
        contents[_IC] = smithyClient.strictParseInt32(output[_IC]);
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    return contents;
};
const de_AnycastIpListCollection = (output, context) => {
    const contents = {};
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_AILS] != null) {
        contents[_It] = de_AnycastIpListSummaries(smithyClient.getArrayIfSingleItem(output[_It][_AILS]));
    }
    if (output[_M] != null) {
        contents[_M] = smithyClient.expectString(output[_M]);
    }
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_IT] != null) {
        contents[_IT] = smithyClient.parseBoolean(output[_IT]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    return contents;
};
const de_AnycastIpListSummaries = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_AnycastIpListSummary(entry);
    });
};
const de_AnycastIpListSummary = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    if (output[_Ar] != null) {
        contents[_Ar] = smithyClient.expectString(output[_Ar]);
    }
    if (output[_IC] != null) {
        contents[_IC] = smithyClient.strictParseInt32(output[_IC]);
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    return contents;
};
const de_AnycastIps = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_AwsAccountNumberList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_CacheBehavior = (output, context) => {
    const contents = {};
    if (output[_PP] != null) {
        contents[_PP] = smithyClient.expectString(output[_PP]);
    }
    if (output[_TOI] != null) {
        contents[_TOI] = smithyClient.expectString(output[_TOI]);
    }
    if (output[_TS] != null) {
        contents[_TS] = de_TrustedSigners(output[_TS]);
    }
    if (output[_TKG] != null) {
        contents[_TKG] = de_TrustedKeyGroups(output[_TKG]);
    }
    if (output[_VPP] != null) {
        contents[_VPP] = smithyClient.expectString(output[_VPP]);
    }
    if (output[_AM] != null) {
        contents[_AM] = de_AllowedMethods(output[_AM]);
    }
    if (output[_SS] != null) {
        contents[_SS] = smithyClient.parseBoolean(output[_SS]);
    }
    if (output[_Com] != null) {
        contents[_Com] = smithyClient.parseBoolean(output[_Com]);
    }
    if (output[_LFA] != null) {
        contents[_LFA] = de_LambdaFunctionAssociations(output[_LFA]);
    }
    if (output[_FA] != null) {
        contents[_FA] = de_FunctionAssociations(output[_FA]);
    }
    if (output[_FLEI] != null) {
        contents[_FLEI] = smithyClient.expectString(output[_FLEI]);
    }
    if (output[_RLCA] != null) {
        contents[_RLCA] = smithyClient.expectString(output[_RLCA]);
    }
    if (output[_CPI] != null) {
        contents[_CPI] = smithyClient.expectString(output[_CPI]);
    }
    if (output[_ORPI] != null) {
        contents[_ORPI] = smithyClient.expectString(output[_ORPI]);
    }
    if (output[_RHPI] != null) {
        contents[_RHPI] = smithyClient.expectString(output[_RHPI]);
    }
    if (output[_GC] != null) {
        contents[_GC] = de_GrpcConfig(output[_GC]);
    }
    if (output[_FV] != null) {
        contents[_FV] = de_ForwardedValues(output[_FV]);
    }
    if (output[_MTTL] != null) {
        contents[_MTTL] = smithyClient.strictParseLong(output[_MTTL]);
    }
    if (output[_DTTL] != null) {
        contents[_DTTL] = smithyClient.strictParseLong(output[_DTTL]);
    }
    if (output[_MTTLa] != null) {
        contents[_MTTLa] = smithyClient.strictParseLong(output[_MTTLa]);
    }
    return contents;
};
const de_CacheBehaviorList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_CacheBehavior(entry);
    });
};
const de_CacheBehaviors = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_CB] != null) {
        contents[_It] = de_CacheBehaviorList(smithyClient.getArrayIfSingleItem(output[_It][_CB]));
    }
    return contents;
};
const de_CachedMethods = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_Met] != null) {
        contents[_It] = de_MethodsList(smithyClient.getArrayIfSingleItem(output[_It][_Met]));
    }
    return contents;
};
const de_CachePolicy = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_CPC] != null) {
        contents[_CPC] = de_CachePolicyConfig(output[_CPC]);
    }
    return contents;
};
const de_CachePolicyConfig = (output, context) => {
    const contents = {};
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_DTTL] != null) {
        contents[_DTTL] = smithyClient.strictParseLong(output[_DTTL]);
    }
    if (output[_MTTLa] != null) {
        contents[_MTTLa] = smithyClient.strictParseLong(output[_MTTLa]);
    }
    if (output[_MTTL] != null) {
        contents[_MTTL] = smithyClient.strictParseLong(output[_MTTL]);
    }
    if (output[_PICKAFTO] != null) {
        contents[_PICKAFTO] = de_ParametersInCacheKeyAndForwardedToOrigin(output[_PICKAFTO]);
    }
    return contents;
};
const de_CachePolicyCookiesConfig = (output, context) => {
    const contents = {};
    if (output[_CBo] != null) {
        contents[_CBo] = smithyClient.expectString(output[_CBo]);
    }
    if (output[_Coo] != null) {
        contents[_Coo] = de_CookieNames(output[_Coo]);
    }
    return contents;
};
const de_CachePolicyHeadersConfig = (output, context) => {
    const contents = {};
    if (output[_HB] != null) {
        contents[_HB] = smithyClient.expectString(output[_HB]);
    }
    if (output[_He] != null) {
        contents[_He] = de_Headers(output[_He]);
    }
    return contents;
};
const de_CachePolicyList = (output, context) => {
    const contents = {};
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_CPS] != null) {
        contents[_It] = de_CachePolicySummaryList(smithyClient.getArrayIfSingleItem(output[_It][_CPS]));
    }
    return contents;
};
const de_CachePolicyQueryStringsConfig = (output, context) => {
    const contents = {};
    if (output[_QSB] != null) {
        contents[_QSB] = smithyClient.expectString(output[_QSB]);
    }
    if (output[_QS] != null) {
        contents[_QS] = de_QueryStringNames(output[_QS]);
    }
    return contents;
};
const de_CachePolicySummary = (output, context) => {
    const contents = {};
    if (output[_Ty] != null) {
        contents[_Ty] = smithyClient.expectString(output[_Ty]);
    }
    if (output[_CPa] != null) {
        contents[_CPa] = de_CachePolicy(output[_CPa]);
    }
    return contents;
};
const de_CachePolicySummaryList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_CachePolicySummary(entry);
    });
};
const de_Certificate = (output, context) => {
    const contents = {};
    if (output[_Ar] != null) {
        contents[_Ar] = smithyClient.expectString(output[_Ar]);
    }
    return contents;
};
const de_CloudFrontOriginAccessIdentity = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_SCUI] != null) {
        contents[_SCUI] = smithyClient.expectString(output[_SCUI]);
    }
    if (output[_CFOAIC] != null) {
        contents[_CFOAIC] = de_CloudFrontOriginAccessIdentityConfig(output[_CFOAIC]);
    }
    return contents;
};
const de_CloudFrontOriginAccessIdentityConfig = (output, context) => {
    const contents = {};
    if (output[_CR] != null) {
        contents[_CR] = smithyClient.expectString(output[_CR]);
    }
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    return contents;
};
const de_CloudFrontOriginAccessIdentityList = (output, context) => {
    const contents = {};
    if (output[_M] != null) {
        contents[_M] = smithyClient.expectString(output[_M]);
    }
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_IT] != null) {
        contents[_IT] = smithyClient.parseBoolean(output[_IT]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_CFOAIS] != null) {
        contents[_It] = de_CloudFrontOriginAccessIdentitySummaryList(smithyClient.getArrayIfSingleItem(output[_It][_CFOAIS]));
    }
    return contents;
};
const de_CloudFrontOriginAccessIdentitySummary = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_SCUI] != null) {
        contents[_SCUI] = smithyClient.expectString(output[_SCUI]);
    }
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    return contents;
};
const de_CloudFrontOriginAccessIdentitySummaryList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_CloudFrontOriginAccessIdentitySummary(entry);
    });
};
const de_ConflictingAlias = (output, context) => {
    const contents = {};
    if (output[_A] != null) {
        contents[_A] = smithyClient.expectString(output[_A]);
    }
    if (output[_DI] != null) {
        contents[_DI] = smithyClient.expectString(output[_DI]);
    }
    if (output[_AIc] != null) {
        contents[_AIc] = smithyClient.expectString(output[_AIc]);
    }
    return contents;
};
const de_ConflictingAliases = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_ConflictingAlias(entry);
    });
};
const de_ConflictingAliasesList = (output, context) => {
    const contents = {};
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_CAon] != null) {
        contents[_It] = de_ConflictingAliases(smithyClient.getArrayIfSingleItem(output[_It][_CAon]));
    }
    return contents;
};
const de_ConnectionGroup = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_Ar] != null) {
        contents[_Ar] = smithyClient.expectString(output[_Ar]);
    }
    if (output[_CTr] != null) {
        contents[_CTr] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_CTr]));
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_T] != null) {
        contents[_T] = de_Tags(output[_T]);
    }
    if (output[_IE] != null) {
        contents[_IE] = smithyClient.parseBoolean(output[_IE]);
    }
    if (output[_RE] != null) {
        contents[_RE] = smithyClient.expectString(output[_RE]);
    }
    if (output[_AILI] != null) {
        contents[_AILI] = smithyClient.expectString(output[_AILI]);
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    if (output[_ID] != null) {
        contents[_ID] = smithyClient.parseBoolean(output[_ID]);
    }
    return contents;
};
const de_ConnectionGroupSummary = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_Ar] != null) {
        contents[_Ar] = smithyClient.expectString(output[_Ar]);
    }
    if (output[_RE] != null) {
        contents[_RE] = smithyClient.expectString(output[_RE]);
    }
    if (output[_CTr] != null) {
        contents[_CTr] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_CTr]));
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_ET] != null) {
        contents[_ET] = smithyClient.expectString(output[_ET]);
    }
    if (output[_AILI] != null) {
        contents[_AILI] = smithyClient.expectString(output[_AILI]);
    }
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    if (output[_ID] != null) {
        contents[_ID] = smithyClient.parseBoolean(output[_ID]);
    }
    return contents;
};
const de_ConnectionGroupSummaryList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_ConnectionGroupSummary(entry);
    });
};
const de_ContentTypeProfile = (output, context) => {
    const contents = {};
    if (output[_Fo] != null) {
        contents[_Fo] = smithyClient.expectString(output[_Fo]);
    }
    if (output[_PI] != null) {
        contents[_PI] = smithyClient.expectString(output[_PI]);
    }
    if (output[_CT] != null) {
        contents[_CT] = smithyClient.expectString(output[_CT]);
    }
    return contents;
};
const de_ContentTypeProfileConfig = (output, context) => {
    const contents = {};
    if (output[_FWCTIU] != null) {
        contents[_FWCTIU] = smithyClient.parseBoolean(output[_FWCTIU]);
    }
    if (output[_CTPo] != null) {
        contents[_CTPo] = de_ContentTypeProfiles(output[_CTPo]);
    }
    return contents;
};
const de_ContentTypeProfileList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_ContentTypeProfile(entry);
    });
};
const de_ContentTypeProfiles = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_CTP] != null) {
        contents[_It] = de_ContentTypeProfileList(smithyClient.getArrayIfSingleItem(output[_It][_CTP]));
    }
    return contents;
};
const de_ContinuousDeploymentPolicy = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_CDPC] != null) {
        contents[_CDPC] = de_ContinuousDeploymentPolicyConfig(output[_CDPC]);
    }
    return contents;
};
const de_ContinuousDeploymentPolicyConfig = (output, context) => {
    const contents = {};
    if (output[_SDDN] != null) {
        contents[_SDDN] = de_StagingDistributionDnsNames(output[_SDDN]);
    }
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    if (output[_TC] != null) {
        contents[_TC] = de_TrafficConfig(output[_TC]);
    }
    return contents;
};
const de_ContinuousDeploymentPolicyList = (output, context) => {
    const contents = {};
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_CDPS] != null) {
        contents[_It] = de_ContinuousDeploymentPolicySummaryList(smithyClient.getArrayIfSingleItem(output[_It][_CDPS]));
    }
    return contents;
};
const de_ContinuousDeploymentPolicySummary = (output, context) => {
    const contents = {};
    if (output[_CDP] != null) {
        contents[_CDP] = de_ContinuousDeploymentPolicy(output[_CDP]);
    }
    return contents;
};
const de_ContinuousDeploymentPolicySummaryList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_ContinuousDeploymentPolicySummary(entry);
    });
};
const de_ContinuousDeploymentSingleHeaderConfig = (output, context) => {
    const contents = {};
    if (output[_H] != null) {
        contents[_H] = smithyClient.expectString(output[_H]);
    }
    if (output[_V] != null) {
        contents[_V] = smithyClient.expectString(output[_V]);
    }
    return contents;
};
const de_ContinuousDeploymentSingleWeightConfig = (output, context) => {
    const contents = {};
    if (output[_W] != null) {
        contents[_W] = smithyClient.strictParseFloat(output[_W]);
    }
    if (output[_SSC] != null) {
        contents[_SSC] = de_SessionStickinessConfig(output[_SSC]);
    }
    return contents;
};
const de_CookieNameList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_CookieNames = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_N] != null) {
        contents[_It] = de_CookieNameList(smithyClient.getArrayIfSingleItem(output[_It][_N]));
    }
    return contents;
};
const de_CookiePreference = (output, context) => {
    const contents = {};
    if (output[_For] != null) {
        contents[_For] = smithyClient.expectString(output[_For]);
    }
    if (output[_WN] != null) {
        contents[_WN] = de_CookieNames(output[_WN]);
    }
    return contents;
};
const de_CustomErrorResponse = (output, context) => {
    const contents = {};
    if (output[_EC] != null) {
        contents[_EC] = smithyClient.strictParseInt32(output[_EC]);
    }
    if (output[_RPP] != null) {
        contents[_RPP] = smithyClient.expectString(output[_RPP]);
    }
    if (output[_RC] != null) {
        contents[_RC] = smithyClient.expectString(output[_RC]);
    }
    if (output[_ECMTTL] != null) {
        contents[_ECMTTL] = smithyClient.strictParseLong(output[_ECMTTL]);
    }
    return contents;
};
const de_CustomErrorResponseList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_CustomErrorResponse(entry);
    });
};
const de_CustomErrorResponses = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_CER] != null) {
        contents[_It] = de_CustomErrorResponseList(smithyClient.getArrayIfSingleItem(output[_It][_CER]));
    }
    return contents;
};
const de_CustomHeaders = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_OCH] != null) {
        contents[_It] = de_OriginCustomHeadersList(smithyClient.getArrayIfSingleItem(output[_It][_OCH]));
    }
    return contents;
};
const de_Customizations = (output, context) => {
    const contents = {};
    if (output[_WA] != null) {
        contents[_WA] = de_WebAclCustomization(output[_WA]);
    }
    if (output[_Ce] != null) {
        contents[_Ce] = de_Certificate(output[_Ce]);
    }
    if (output[_GR] != null) {
        contents[_GR] = de_GeoRestrictionCustomization(output[_GR]);
    }
    return contents;
};
const de_CustomOriginConfig = (output, context) => {
    const contents = {};
    if (output[_HTTPP] != null) {
        contents[_HTTPP] = smithyClient.strictParseInt32(output[_HTTPP]);
    }
    if (output[_HTTPSP] != null) {
        contents[_HTTPSP] = smithyClient.strictParseInt32(output[_HTTPSP]);
    }
    if (output[_OPP] != null) {
        contents[_OPP] = smithyClient.expectString(output[_OPP]);
    }
    if (output[_OSP] != null) {
        contents[_OSP] = de_OriginSslProtocols(output[_OSP]);
    }
    if (output[_ORT] != null) {
        contents[_ORT] = smithyClient.strictParseInt32(output[_ORT]);
    }
    if (output[_OKT] != null) {
        contents[_OKT] = smithyClient.strictParseInt32(output[_OKT]);
    }
    if (output[_IAT] != null) {
        contents[_IAT] = smithyClient.expectString(output[_IAT]);
    }
    return contents;
};
const de_DefaultCacheBehavior = (output, context) => {
    const contents = {};
    if (output[_TOI] != null) {
        contents[_TOI] = smithyClient.expectString(output[_TOI]);
    }
    if (output[_TS] != null) {
        contents[_TS] = de_TrustedSigners(output[_TS]);
    }
    if (output[_TKG] != null) {
        contents[_TKG] = de_TrustedKeyGroups(output[_TKG]);
    }
    if (output[_VPP] != null) {
        contents[_VPP] = smithyClient.expectString(output[_VPP]);
    }
    if (output[_AM] != null) {
        contents[_AM] = de_AllowedMethods(output[_AM]);
    }
    if (output[_SS] != null) {
        contents[_SS] = smithyClient.parseBoolean(output[_SS]);
    }
    if (output[_Com] != null) {
        contents[_Com] = smithyClient.parseBoolean(output[_Com]);
    }
    if (output[_LFA] != null) {
        contents[_LFA] = de_LambdaFunctionAssociations(output[_LFA]);
    }
    if (output[_FA] != null) {
        contents[_FA] = de_FunctionAssociations(output[_FA]);
    }
    if (output[_FLEI] != null) {
        contents[_FLEI] = smithyClient.expectString(output[_FLEI]);
    }
    if (output[_RLCA] != null) {
        contents[_RLCA] = smithyClient.expectString(output[_RLCA]);
    }
    if (output[_CPI] != null) {
        contents[_CPI] = smithyClient.expectString(output[_CPI]);
    }
    if (output[_ORPI] != null) {
        contents[_ORPI] = smithyClient.expectString(output[_ORPI]);
    }
    if (output[_RHPI] != null) {
        contents[_RHPI] = smithyClient.expectString(output[_RHPI]);
    }
    if (output[_GC] != null) {
        contents[_GC] = de_GrpcConfig(output[_GC]);
    }
    if (output[_FV] != null) {
        contents[_FV] = de_ForwardedValues(output[_FV]);
    }
    if (output[_MTTL] != null) {
        contents[_MTTL] = smithyClient.strictParseLong(output[_MTTL]);
    }
    if (output[_DTTL] != null) {
        contents[_DTTL] = smithyClient.strictParseLong(output[_DTTL]);
    }
    if (output[_MTTLa] != null) {
        contents[_MTTLa] = smithyClient.strictParseLong(output[_MTTLa]);
    }
    return contents;
};
const de_Distribution = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_ARN] != null) {
        contents[_ARN] = smithyClient.expectString(output[_ARN]);
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_IPIB] != null) {
        contents[_IPIB] = smithyClient.strictParseInt32(output[_IPIB]);
    }
    if (output[_DN] != null) {
        contents[_DN] = smithyClient.expectString(output[_DN]);
    }
    if (output[_ATS] != null) {
        contents[_ATS] = de_ActiveTrustedSigners(output[_ATS]);
    }
    if (output[_ATKG] != null) {
        contents[_ATKG] = de_ActiveTrustedKeyGroups(output[_ATKG]);
    }
    if (output[_DCi] != null) {
        contents[_DCi] = de_DistributionConfig(output[_DCi]);
    }
    if (String(output.AliasICPRecordals).trim() === "") {
        contents[_AICPR] = [];
    }
    else if (output[_AICPR] != null && output[_AICPR][_AICPRl] != null) {
        contents[_AICPR] = de_AliasICPRecordals(smithyClient.getArrayIfSingleItem(output[_AICPR][_AICPRl]));
    }
    return contents;
};
const de_DistributionConfig = (output, context) => {
    const contents = {};
    if (output[_CR] != null) {
        contents[_CR] = smithyClient.expectString(output[_CR]);
    }
    if (output[_Al] != null) {
        contents[_Al] = de_Aliases(output[_Al]);
    }
    if (output[_DRO] != null) {
        contents[_DRO] = smithyClient.expectString(output[_DRO]);
    }
    if (output[_Ori] != null) {
        contents[_Ori] = de_Origins(output[_Ori]);
    }
    if (output[_OG] != null) {
        contents[_OG] = de_OriginGroups(output[_OG]);
    }
    if (output[_DCB] != null) {
        contents[_DCB] = de_DefaultCacheBehavior(output[_DCB]);
    }
    if (output[_CBa] != null) {
        contents[_CBa] = de_CacheBehaviors(output[_CBa]);
    }
    if (output[_CERu] != null) {
        contents[_CERu] = de_CustomErrorResponses(output[_CERu]);
    }
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    if (output[_Lo] != null) {
        contents[_Lo] = de_LoggingConfig(output[_Lo]);
    }
    if (output[_PC] != null) {
        contents[_PC] = smithyClient.expectString(output[_PC]);
    }
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    if (output[_VC] != null) {
        contents[_VC] = de_ViewerCertificate(output[_VC]);
    }
    if (output[_Re] != null) {
        contents[_Re] = de_Restrictions(output[_Re]);
    }
    if (output[_WACLI] != null) {
        contents[_WACLI] = smithyClient.expectString(output[_WACLI]);
    }
    if (output[_HV] != null) {
        contents[_HV] = smithyClient.expectString(output[_HV]);
    }
    if (output[_IIPVE] != null) {
        contents[_IIPVE] = smithyClient.parseBoolean(output[_IIPVE]);
    }
    if (output[_CDPI] != null) {
        contents[_CDPI] = smithyClient.expectString(output[_CDPI]);
    }
    if (output[_S] != null) {
        contents[_S] = smithyClient.parseBoolean(output[_S]);
    }
    if (output[_AILI] != null) {
        contents[_AILI] = smithyClient.expectString(output[_AILI]);
    }
    if (output[_TCe] != null) {
        contents[_TCe] = de_TenantConfig(output[_TCe]);
    }
    if (output[_CMo] != null) {
        contents[_CMo] = smithyClient.expectString(output[_CMo]);
    }
    return contents;
};
const de_DistributionIdList = (output, context) => {
    const contents = {};
    if (output[_M] != null) {
        contents[_M] = smithyClient.expectString(output[_M]);
    }
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_IT] != null) {
        contents[_IT] = smithyClient.parseBoolean(output[_IT]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_DI] != null) {
        contents[_It] = de_DistributionIdListSummary(smithyClient.getArrayIfSingleItem(output[_It][_DI]));
    }
    return contents;
};
const de_DistributionIdListSummary = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_DistributionList = (output, context) => {
    const contents = {};
    if (output[_M] != null) {
        contents[_M] = smithyClient.expectString(output[_M]);
    }
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_IT] != null) {
        contents[_IT] = smithyClient.parseBoolean(output[_IT]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_DS] != null) {
        contents[_It] = de_DistributionSummaryList(smithyClient.getArrayIfSingleItem(output[_It][_DS]));
    }
    return contents;
};
const de_DistributionSummary = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_ARN] != null) {
        contents[_ARN] = smithyClient.expectString(output[_ARN]);
    }
    if (output[_ET] != null) {
        contents[_ET] = smithyClient.expectString(output[_ET]);
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_DN] != null) {
        contents[_DN] = smithyClient.expectString(output[_DN]);
    }
    if (output[_Al] != null) {
        contents[_Al] = de_Aliases(output[_Al]);
    }
    if (output[_Ori] != null) {
        contents[_Ori] = de_Origins(output[_Ori]);
    }
    if (output[_OG] != null) {
        contents[_OG] = de_OriginGroups(output[_OG]);
    }
    if (output[_DCB] != null) {
        contents[_DCB] = de_DefaultCacheBehavior(output[_DCB]);
    }
    if (output[_CBa] != null) {
        contents[_CBa] = de_CacheBehaviors(output[_CBa]);
    }
    if (output[_CERu] != null) {
        contents[_CERu] = de_CustomErrorResponses(output[_CERu]);
    }
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    if (output[_PC] != null) {
        contents[_PC] = smithyClient.expectString(output[_PC]);
    }
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    if (output[_VC] != null) {
        contents[_VC] = de_ViewerCertificate(output[_VC]);
    }
    if (output[_Re] != null) {
        contents[_Re] = de_Restrictions(output[_Re]);
    }
    if (output[_WACLI] != null) {
        contents[_WACLI] = smithyClient.expectString(output[_WACLI]);
    }
    if (output[_HV] != null) {
        contents[_HV] = smithyClient.expectString(output[_HV]);
    }
    if (output[_IIPVE] != null) {
        contents[_IIPVE] = smithyClient.parseBoolean(output[_IIPVE]);
    }
    if (String(output.AliasICPRecordals).trim() === "") {
        contents[_AICPR] = [];
    }
    else if (output[_AICPR] != null && output[_AICPR][_AICPRl] != null) {
        contents[_AICPR] = de_AliasICPRecordals(smithyClient.getArrayIfSingleItem(output[_AICPR][_AICPRl]));
    }
    if (output[_S] != null) {
        contents[_S] = smithyClient.parseBoolean(output[_S]);
    }
    if (output[_CMo] != null) {
        contents[_CMo] = smithyClient.expectString(output[_CMo]);
    }
    if (output[_AILI] != null) {
        contents[_AILI] = smithyClient.expectString(output[_AILI]);
    }
    return contents;
};
const de_DistributionSummaryList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_DistributionSummary(entry);
    });
};
const de_DistributionTenant = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_DI] != null) {
        contents[_DI] = smithyClient.expectString(output[_DI]);
    }
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_Ar] != null) {
        contents[_Ar] = smithyClient.expectString(output[_Ar]);
    }
    if (String(output.Domains).trim() === "") {
        contents[_D] = [];
    }
    else if (output[_D] != null && output[_D][_m] != null) {
        contents[_D] = de_DomainResultList(smithyClient.getArrayIfSingleItem(output[_D][_m]));
    }
    if (output[_T] != null) {
        contents[_T] = de_Tags(output[_T]);
    }
    if (output[_C] != null) {
        contents[_C] = de_Customizations(output[_C]);
    }
    if (String(output.Parameters).trim() === "") {
        contents[_P] = [];
    }
    else if (output[_P] != null && output[_P][_m] != null) {
        contents[_P] = de_Parameters(smithyClient.getArrayIfSingleItem(output[_P][_m]));
    }
    if (output[_CGI] != null) {
        contents[_CGI] = smithyClient.expectString(output[_CGI]);
    }
    if (output[_CTr] != null) {
        contents[_CTr] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_CTr]));
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    return contents;
};
const de_DistributionTenantList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_DistributionTenantSummary(entry);
    });
};
const de_DistributionTenantSummary = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_DI] != null) {
        contents[_DI] = smithyClient.expectString(output[_DI]);
    }
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_Ar] != null) {
        contents[_Ar] = smithyClient.expectString(output[_Ar]);
    }
    if (String(output.Domains).trim() === "") {
        contents[_D] = [];
    }
    else if (output[_D] != null && output[_D][_m] != null) {
        contents[_D] = de_DomainResultList(smithyClient.getArrayIfSingleItem(output[_D][_m]));
    }
    if (output[_CGI] != null) {
        contents[_CGI] = smithyClient.expectString(output[_CGI]);
    }
    if (output[_C] != null) {
        contents[_C] = de_Customizations(output[_C]);
    }
    if (output[_CTr] != null) {
        contents[_CTr] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_CTr]));
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_ET] != null) {
        contents[_ET] = smithyClient.expectString(output[_ET]);
    }
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    return contents;
};
const de_DnsConfiguration = (output, context) => {
    const contents = {};
    if (output[_Do] != null) {
        contents[_Do] = smithyClient.expectString(output[_Do]);
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    if (output[_Rea] != null) {
        contents[_Rea] = smithyClient.expectString(output[_Rea]);
    }
    return contents;
};
const de_DnsConfigurationList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_DnsConfiguration(entry);
    });
};
const de_DomainConflict = (output, context) => {
    const contents = {};
    if (output[_Do] != null) {
        contents[_Do] = smithyClient.expectString(output[_Do]);
    }
    if (output[_RTe] != null) {
        contents[_RTe] = smithyClient.expectString(output[_RTe]);
    }
    if (output[_RI] != null) {
        contents[_RI] = smithyClient.expectString(output[_RI]);
    }
    if (output[_AIc] != null) {
        contents[_AIc] = smithyClient.expectString(output[_AIc]);
    }
    return contents;
};
const de_DomainConflictsList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_DomainConflict(entry);
    });
};
const de_DomainResult = (output, context) => {
    const contents = {};
    if (output[_Do] != null) {
        contents[_Do] = smithyClient.expectString(output[_Do]);
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    return contents;
};
const de_DomainResultList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_DomainResult(entry);
    });
};
const de_EncryptionEntities = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_EEn] != null) {
        contents[_It] = de_EncryptionEntityList(smithyClient.getArrayIfSingleItem(output[_It][_EEn]));
    }
    return contents;
};
const de_EncryptionEntity = (output, context) => {
    const contents = {};
    if (output[_PKI] != null) {
        contents[_PKI] = smithyClient.expectString(output[_PKI]);
    }
    if (output[_PIr] != null) {
        contents[_PIr] = smithyClient.expectString(output[_PIr]);
    }
    if (output[_FP] != null) {
        contents[_FP] = de_FieldPatterns(output[_FP]);
    }
    return contents;
};
const de_EncryptionEntityList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_EncryptionEntity(entry);
    });
};
const de_EndPoint = (output, context) => {
    const contents = {};
    if (output[_ST] != null) {
        contents[_ST] = smithyClient.expectString(output[_ST]);
    }
    if (output[_KSC] != null) {
        contents[_KSC] = de_KinesisStreamConfig(output[_KSC]);
    }
    return contents;
};
const de_EndPointList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_EndPoint(entry);
    });
};
const de_FieldLevelEncryption = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_FLEC] != null) {
        contents[_FLEC] = de_FieldLevelEncryptionConfig(output[_FLEC]);
    }
    return contents;
};
const de_FieldLevelEncryptionConfig = (output, context) => {
    const contents = {};
    if (output[_CR] != null) {
        contents[_CR] = smithyClient.expectString(output[_CR]);
    }
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    if (output[_QAPC] != null) {
        contents[_QAPC] = de_QueryArgProfileConfig(output[_QAPC]);
    }
    if (output[_CTPC] != null) {
        contents[_CTPC] = de_ContentTypeProfileConfig(output[_CTPC]);
    }
    return contents;
};
const de_FieldLevelEncryptionList = (output, context) => {
    const contents = {};
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_FLES] != null) {
        contents[_It] = de_FieldLevelEncryptionSummaryList(smithyClient.getArrayIfSingleItem(output[_It][_FLES]));
    }
    return contents;
};
const de_FieldLevelEncryptionProfile = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_FLEPC] != null) {
        contents[_FLEPC] = de_FieldLevelEncryptionProfileConfig(output[_FLEPC]);
    }
    return contents;
};
const de_FieldLevelEncryptionProfileConfig = (output, context) => {
    const contents = {};
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_CR] != null) {
        contents[_CR] = smithyClient.expectString(output[_CR]);
    }
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    if (output[_EE] != null) {
        contents[_EE] = de_EncryptionEntities(output[_EE]);
    }
    return contents;
};
const de_FieldLevelEncryptionProfileList = (output, context) => {
    const contents = {};
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_FLEPS] != null) {
        contents[_It] = de_FieldLevelEncryptionProfileSummaryList(smithyClient.getArrayIfSingleItem(output[_It][_FLEPS]));
    }
    return contents;
};
const de_FieldLevelEncryptionProfileSummary = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_EE] != null) {
        contents[_EE] = de_EncryptionEntities(output[_EE]);
    }
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    return contents;
};
const de_FieldLevelEncryptionProfileSummaryList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_FieldLevelEncryptionProfileSummary(entry);
    });
};
const de_FieldLevelEncryptionSummary = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    if (output[_QAPC] != null) {
        contents[_QAPC] = de_QueryArgProfileConfig(output[_QAPC]);
    }
    if (output[_CTPC] != null) {
        contents[_CTPC] = de_ContentTypeProfileConfig(output[_CTPC]);
    }
    return contents;
};
const de_FieldLevelEncryptionSummaryList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_FieldLevelEncryptionSummary(entry);
    });
};
const de_FieldList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_FieldPatternList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_FieldPatterns = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_FPi] != null) {
        contents[_It] = de_FieldPatternList(smithyClient.getArrayIfSingleItem(output[_It][_FPi]));
    }
    return contents;
};
const de_ForwardedValues = (output, context) => {
    const contents = {};
    if (output[_QSu] != null) {
        contents[_QSu] = smithyClient.parseBoolean(output[_QSu]);
    }
    if (output[_Coo] != null) {
        contents[_Coo] = de_CookiePreference(output[_Coo]);
    }
    if (output[_He] != null) {
        contents[_He] = de_Headers(output[_He]);
    }
    if (output[_QSCK] != null) {
        contents[_QSCK] = de_QueryStringCacheKeys(output[_QSCK]);
    }
    return contents;
};
const de_FunctionAssociation = (output, context) => {
    const contents = {};
    if (output[_FARN] != null) {
        contents[_FARN] = smithyClient.expectString(output[_FARN]);
    }
    if (output[_ETv] != null) {
        contents[_ETv] = smithyClient.expectString(output[_ETv]);
    }
    return contents;
};
const de_FunctionAssociationList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_FunctionAssociation(entry);
    });
};
const de_FunctionAssociations = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_FAu] != null) {
        contents[_It] = de_FunctionAssociationList(smithyClient.getArrayIfSingleItem(output[_It][_FAu]));
    }
    return contents;
};
const de_FunctionConfig = (output, context) => {
    const contents = {};
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    if (output[_Ru] != null) {
        contents[_Ru] = smithyClient.expectString(output[_Ru]);
    }
    if (output[_KVSA] != null) {
        contents[_KVSA] = de_KeyValueStoreAssociations(output[_KVSA]);
    }
    return contents;
};
const de_FunctionExecutionLogList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_FunctionList = (output, context) => {
    const contents = {};
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_FSu] != null) {
        contents[_It] = de_FunctionSummaryList(smithyClient.getArrayIfSingleItem(output[_It][_FSu]));
    }
    return contents;
};
const de_FunctionMetadata = (output, context) => {
    const contents = {};
    if (output[_FARN] != null) {
        contents[_FARN] = smithyClient.expectString(output[_FARN]);
    }
    if (output[_St] != null) {
        contents[_St] = smithyClient.expectString(output[_St]);
    }
    if (output[_CTr] != null) {
        contents[_CTr] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_CTr]));
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    return contents;
};
const de_FunctionSummary = (output, context) => {
    const contents = {};
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    if (output[_FCu] != null) {
        contents[_FCu] = de_FunctionConfig(output[_FCu]);
    }
    if (output[_FM] != null) {
        contents[_FM] = de_FunctionMetadata(output[_FM]);
    }
    return contents;
};
const de_FunctionSummaryList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_FunctionSummary(entry);
    });
};
const de_GeoRestriction = (output, context) => {
    const contents = {};
    if (output[_RT] != null) {
        contents[_RT] = smithyClient.expectString(output[_RT]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_L] != null) {
        contents[_It] = de_LocationList(smithyClient.getArrayIfSingleItem(output[_It][_L]));
    }
    return contents;
};
const de_GeoRestrictionCustomization = (output, context) => {
    const contents = {};
    if (output[_RT] != null) {
        contents[_RT] = smithyClient.expectString(output[_RT]);
    }
    if (String(output.Locations).trim() === "") {
        contents[_Loc] = [];
    }
    else if (output[_Loc] != null && output[_Loc][_L] != null) {
        contents[_Loc] = de_LocationList(smithyClient.getArrayIfSingleItem(output[_Loc][_L]));
    }
    return contents;
};
const de_GrpcConfig = (output, context) => {
    const contents = {};
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    return contents;
};
const de_HeaderList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_Headers = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_N] != null) {
        contents[_It] = de_HeaderList(smithyClient.getArrayIfSingleItem(output[_It][_N]));
    }
    return contents;
};
const de_Invalidation = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    if (output[_CTre] != null) {
        contents[_CTre] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_CTre]));
    }
    if (output[_IB] != null) {
        contents[_IB] = de_InvalidationBatch(output[_IB]);
    }
    return contents;
};
const de_InvalidationBatch = (output, context) => {
    const contents = {};
    if (output[_Pa] != null) {
        contents[_Pa] = de_Paths(output[_Pa]);
    }
    if (output[_CR] != null) {
        contents[_CR] = smithyClient.expectString(output[_CR]);
    }
    return contents;
};
const de_InvalidationList = (output, context) => {
    const contents = {};
    if (output[_M] != null) {
        contents[_M] = smithyClient.expectString(output[_M]);
    }
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_IT] != null) {
        contents[_IT] = smithyClient.parseBoolean(output[_IT]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_ISnv] != null) {
        contents[_It] = de_InvalidationSummaryList(smithyClient.getArrayIfSingleItem(output[_It][_ISnv]));
    }
    return contents;
};
const de_InvalidationSummary = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_CTre] != null) {
        contents[_CTre] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_CTre]));
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    return contents;
};
const de_InvalidationSummaryList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_InvalidationSummary(entry);
    });
};
const de_KeyGroup = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_KGC] != null) {
        contents[_KGC] = de_KeyGroupConfig(output[_KGC]);
    }
    return contents;
};
const de_KeyGroupConfig = (output, context) => {
    const contents = {};
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_PK] != null) {
        contents[_It] = de_PublicKeyIdList(smithyClient.getArrayIfSingleItem(output[_It][_PK]));
    }
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    return contents;
};
const de_KeyGroupList = (output, context) => {
    const contents = {};
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_KGS] != null) {
        contents[_It] = de_KeyGroupSummaryList(smithyClient.getArrayIfSingleItem(output[_It][_KGS]));
    }
    return contents;
};
const de_KeyGroupSummary = (output, context) => {
    const contents = {};
    if (output[_KG] != null) {
        contents[_KG] = de_KeyGroup(output[_KG]);
    }
    return contents;
};
const de_KeyGroupSummaryList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_KeyGroupSummary(entry);
    });
};
const de_KeyPairIdList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_KeyPairIds = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_KPI] != null) {
        contents[_It] = de_KeyPairIdList(smithyClient.getArrayIfSingleItem(output[_It][_KPI]));
    }
    return contents;
};
const de_KeyValueStore = (output, context) => {
    const contents = {};
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    if (output[_ARN] != null) {
        contents[_ARN] = smithyClient.expectString(output[_ARN]);
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    return contents;
};
const de_KeyValueStoreAssociation = (output, context) => {
    const contents = {};
    if (output[_KVSARN] != null) {
        contents[_KVSARN] = smithyClient.expectString(output[_KVSARN]);
    }
    return contents;
};
const de_KeyValueStoreAssociationList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_KeyValueStoreAssociation(entry);
    });
};
const de_KeyValueStoreAssociations = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_KVSAe] != null) {
        contents[_It] = de_KeyValueStoreAssociationList(smithyClient.getArrayIfSingleItem(output[_It][_KVSAe]));
    }
    return contents;
};
const de_KeyValueStoreList = (output, context) => {
    const contents = {};
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_KVS] != null) {
        contents[_It] = de_KeyValueStoreSummaryList(smithyClient.getArrayIfSingleItem(output[_It][_KVS]));
    }
    return contents;
};
const de_KeyValueStoreSummaryList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_KeyValueStore(entry);
    });
};
const de_KGKeyPairIds = (output, context) => {
    const contents = {};
    if (output[_KGI] != null) {
        contents[_KGI] = smithyClient.expectString(output[_KGI]);
    }
    if (output[_KPIe] != null) {
        contents[_KPIe] = de_KeyPairIds(output[_KPIe]);
    }
    return contents;
};
const de_KGKeyPairIdsList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_KGKeyPairIds(entry);
    });
};
const de_KinesisStreamConfig = (output, context) => {
    const contents = {};
    if (output[_RARN] != null) {
        contents[_RARN] = smithyClient.expectString(output[_RARN]);
    }
    if (output[_SARNt] != null) {
        contents[_SARNt] = smithyClient.expectString(output[_SARNt]);
    }
    return contents;
};
const de_LambdaFunctionAssociation = (output, context) => {
    const contents = {};
    if (output[_LFARN] != null) {
        contents[_LFARN] = smithyClient.expectString(output[_LFARN]);
    }
    if (output[_ETv] != null) {
        contents[_ETv] = smithyClient.expectString(output[_ETv]);
    }
    if (output[_IBn] != null) {
        contents[_IBn] = smithyClient.parseBoolean(output[_IBn]);
    }
    return contents;
};
const de_LambdaFunctionAssociationList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_LambdaFunctionAssociation(entry);
    });
};
const de_LambdaFunctionAssociations = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_LFAa] != null) {
        contents[_It] = de_LambdaFunctionAssociationList(smithyClient.getArrayIfSingleItem(output[_It][_LFAa]));
    }
    return contents;
};
const de_LocationList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_LoggingConfig = (output, context) => {
    const contents = {};
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    if (output[_ICn] != null) {
        contents[_ICn] = smithyClient.parseBoolean(output[_ICn]);
    }
    if (output[_B] != null) {
        contents[_B] = smithyClient.expectString(output[_B]);
    }
    if (output[_Pr] != null) {
        contents[_Pr] = smithyClient.expectString(output[_Pr]);
    }
    return contents;
};
const de_ManagedCertificateDetails = (output, context) => {
    const contents = {};
    if (output[_CA] != null) {
        contents[_CA] = smithyClient.expectString(output[_CA]);
    }
    if (output[_CSe] != null) {
        contents[_CSe] = smithyClient.expectString(output[_CSe]);
    }
    if (output[_VTH] != null) {
        contents[_VTH] = smithyClient.expectString(output[_VTH]);
    }
    if (String(output.ValidationTokenDetails).trim() === "") {
        contents[_VTD] = [];
    }
    else if (output[_VTD] != null && output[_VTD][_m] != null) {
        contents[_VTD] = de_ValidationTokenDetailList(smithyClient.getArrayIfSingleItem(output[_VTD][_m]));
    }
    return contents;
};
const de_MethodsList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_MonitoringSubscription = (output, context) => {
    const contents = {};
    if (output[_RMSC] != null) {
        contents[_RMSC] = de_RealtimeMetricsSubscriptionConfig(output[_RMSC]);
    }
    return contents;
};
const de_Origin = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_DN] != null) {
        contents[_DN] = smithyClient.expectString(output[_DN]);
    }
    if (output[_OP] != null) {
        contents[_OP] = smithyClient.expectString(output[_OP]);
    }
    if (output[_CH] != null) {
        contents[_CH] = de_CustomHeaders(output[_CH]);
    }
    if (output[_SOC] != null) {
        contents[_SOC] = de_S3OriginConfig(output[_SOC]);
    }
    if (output[_COC] != null) {
        contents[_COC] = de_CustomOriginConfig(output[_COC]);
    }
    if (output[_VOC] != null) {
        contents[_VOC] = de_VpcOriginConfig(output[_VOC]);
    }
    if (output[_CAo] != null) {
        contents[_CAo] = smithyClient.strictParseInt32(output[_CAo]);
    }
    if (output[_CTon] != null) {
        contents[_CTon] = smithyClient.strictParseInt32(output[_CTon]);
    }
    if (output[_RCT] != null) {
        contents[_RCT] = smithyClient.strictParseInt32(output[_RCT]);
    }
    if (output[_OS] != null) {
        contents[_OS] = de_OriginShield(output[_OS]);
    }
    if (output[_OACI] != null) {
        contents[_OACI] = smithyClient.expectString(output[_OACI]);
    }
    return contents;
};
const de_OriginAccessControl = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_OACC] != null) {
        contents[_OACC] = de_OriginAccessControlConfig(output[_OACC]);
    }
    return contents;
};
const de_OriginAccessControlConfig = (output, context) => {
    const contents = {};
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_De] != null) {
        contents[_De] = smithyClient.expectString(output[_De]);
    }
    if (output[_SP] != null) {
        contents[_SP] = smithyClient.expectString(output[_SP]);
    }
    if (output[_SB] != null) {
        contents[_SB] = smithyClient.expectString(output[_SB]);
    }
    if (output[_OACOT] != null) {
        contents[_OACOT] = smithyClient.expectString(output[_OACOT]);
    }
    return contents;
};
const de_OriginAccessControlList = (output, context) => {
    const contents = {};
    if (output[_M] != null) {
        contents[_M] = smithyClient.expectString(output[_M]);
    }
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_IT] != null) {
        contents[_IT] = smithyClient.parseBoolean(output[_IT]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_OACS] != null) {
        contents[_It] = de_OriginAccessControlSummaryList(smithyClient.getArrayIfSingleItem(output[_It][_OACS]));
    }
    return contents;
};
const de_OriginAccessControlSummary = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_De] != null) {
        contents[_De] = smithyClient.expectString(output[_De]);
    }
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_SP] != null) {
        contents[_SP] = smithyClient.expectString(output[_SP]);
    }
    if (output[_SB] != null) {
        contents[_SB] = smithyClient.expectString(output[_SB]);
    }
    if (output[_OACOT] != null) {
        contents[_OACOT] = smithyClient.expectString(output[_OACOT]);
    }
    return contents;
};
const de_OriginAccessControlSummaryList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_OriginAccessControlSummary(entry);
    });
};
const de_OriginCustomHeader = (output, context) => {
    const contents = {};
    if (output[_HN] != null) {
        contents[_HN] = smithyClient.expectString(output[_HN]);
    }
    if (output[_HVe] != null) {
        contents[_HVe] = smithyClient.expectString(output[_HVe]);
    }
    return contents;
};
const de_OriginCustomHeadersList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_OriginCustomHeader(entry);
    });
};
const de_OriginGroup = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_FCa] != null) {
        contents[_FCa] = de_OriginGroupFailoverCriteria(output[_FCa]);
    }
    if (output[_Mem] != null) {
        contents[_Mem] = de_OriginGroupMembers(output[_Mem]);
    }
    if (output[_SC] != null) {
        contents[_SC] = smithyClient.expectString(output[_SC]);
    }
    return contents;
};
const de_OriginGroupFailoverCriteria = (output, context) => {
    const contents = {};
    if (output[_SCt] != null) {
        contents[_SCt] = de_StatusCodes(output[_SCt]);
    }
    return contents;
};
const de_OriginGroupList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_OriginGroup(entry);
    });
};
const de_OriginGroupMember = (output, context) => {
    const contents = {};
    if (output[_OI] != null) {
        contents[_OI] = smithyClient.expectString(output[_OI]);
    }
    return contents;
};
const de_OriginGroupMemberList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_OriginGroupMember(entry);
    });
};
const de_OriginGroupMembers = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_OGM] != null) {
        contents[_It] = de_OriginGroupMemberList(smithyClient.getArrayIfSingleItem(output[_It][_OGM]));
    }
    return contents;
};
const de_OriginGroups = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_OGr] != null) {
        contents[_It] = de_OriginGroupList(smithyClient.getArrayIfSingleItem(output[_It][_OGr]));
    }
    return contents;
};
const de_OriginList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_Origin(entry);
    });
};
const de_OriginRequestPolicy = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_ORPC] != null) {
        contents[_ORPC] = de_OriginRequestPolicyConfig(output[_ORPC]);
    }
    return contents;
};
const de_OriginRequestPolicyConfig = (output, context) => {
    const contents = {};
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_HC] != null) {
        contents[_HC] = de_OriginRequestPolicyHeadersConfig(output[_HC]);
    }
    if (output[_CC] != null) {
        contents[_CC] = de_OriginRequestPolicyCookiesConfig(output[_CC]);
    }
    if (output[_QSC] != null) {
        contents[_QSC] = de_OriginRequestPolicyQueryStringsConfig(output[_QSC]);
    }
    return contents;
};
const de_OriginRequestPolicyCookiesConfig = (output, context) => {
    const contents = {};
    if (output[_CBo] != null) {
        contents[_CBo] = smithyClient.expectString(output[_CBo]);
    }
    if (output[_Coo] != null) {
        contents[_Coo] = de_CookieNames(output[_Coo]);
    }
    return contents;
};
const de_OriginRequestPolicyHeadersConfig = (output, context) => {
    const contents = {};
    if (output[_HB] != null) {
        contents[_HB] = smithyClient.expectString(output[_HB]);
    }
    if (output[_He] != null) {
        contents[_He] = de_Headers(output[_He]);
    }
    return contents;
};
const de_OriginRequestPolicyList = (output, context) => {
    const contents = {};
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_ORPS] != null) {
        contents[_It] = de_OriginRequestPolicySummaryList(smithyClient.getArrayIfSingleItem(output[_It][_ORPS]));
    }
    return contents;
};
const de_OriginRequestPolicyQueryStringsConfig = (output, context) => {
    const contents = {};
    if (output[_QSB] != null) {
        contents[_QSB] = smithyClient.expectString(output[_QSB]);
    }
    if (output[_QS] != null) {
        contents[_QS] = de_QueryStringNames(output[_QS]);
    }
    return contents;
};
const de_OriginRequestPolicySummary = (output, context) => {
    const contents = {};
    if (output[_Ty] != null) {
        contents[_Ty] = smithyClient.expectString(output[_Ty]);
    }
    if (output[_ORP] != null) {
        contents[_ORP] = de_OriginRequestPolicy(output[_ORP]);
    }
    return contents;
};
const de_OriginRequestPolicySummaryList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_OriginRequestPolicySummary(entry);
    });
};
const de_Origins = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_Or] != null) {
        contents[_It] = de_OriginList(smithyClient.getArrayIfSingleItem(output[_It][_Or]));
    }
    return contents;
};
const de_OriginShield = (output, context) => {
    const contents = {};
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    if (output[_OSR] != null) {
        contents[_OSR] = smithyClient.expectString(output[_OSR]);
    }
    return contents;
};
const de_OriginSslProtocols = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_SPs] != null) {
        contents[_It] = de_SslProtocolsList(smithyClient.getArrayIfSingleItem(output[_It][_SPs]));
    }
    return contents;
};
const de_Parameter = (output, context) => {
    const contents = {};
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_V] != null) {
        contents[_V] = smithyClient.expectString(output[_V]);
    }
    return contents;
};
const de_ParameterDefinition = (output, context) => {
    const contents = {};
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_Def] != null) {
        contents[_Def] = de_ParameterDefinitionSchema(output[_Def]);
    }
    return contents;
};
const de_ParameterDefinitions = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_ParameterDefinition(entry);
    });
};
const de_ParameterDefinitionSchema = (output, context) => {
    const contents = {};
    if (output[_SSt] != null) {
        contents[_SSt] = de_StringSchemaConfig(output[_SSt]);
    }
    return contents;
};
const de_Parameters = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_Parameter(entry);
    });
};
const de_ParametersInCacheKeyAndForwardedToOrigin = (output, context) => {
    const contents = {};
    if (output[_EAEG] != null) {
        contents[_EAEG] = smithyClient.parseBoolean(output[_EAEG]);
    }
    if (output[_EAEB] != null) {
        contents[_EAEB] = smithyClient.parseBoolean(output[_EAEB]);
    }
    if (output[_HC] != null) {
        contents[_HC] = de_CachePolicyHeadersConfig(output[_HC]);
    }
    if (output[_CC] != null) {
        contents[_CC] = de_CachePolicyCookiesConfig(output[_CC]);
    }
    if (output[_QSC] != null) {
        contents[_QSC] = de_CachePolicyQueryStringsConfig(output[_QSC]);
    }
    return contents;
};
const de_PathList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_Paths = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_Pat] != null) {
        contents[_It] = de_PathList(smithyClient.getArrayIfSingleItem(output[_It][_Pat]));
    }
    return contents;
};
const de_PublicKey = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_CTr] != null) {
        contents[_CTr] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_CTr]));
    }
    if (output[_PKC] != null) {
        contents[_PKC] = de_PublicKeyConfig(output[_PKC]);
    }
    return contents;
};
const de_PublicKeyConfig = (output, context) => {
    const contents = {};
    if (output[_CR] != null) {
        contents[_CR] = smithyClient.expectString(output[_CR]);
    }
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_EK] != null) {
        contents[_EK] = smithyClient.expectString(output[_EK]);
    }
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    return contents;
};
const de_PublicKeyIdList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_PublicKeyList = (output, context) => {
    const contents = {};
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_PKS] != null) {
        contents[_It] = de_PublicKeySummaryList(smithyClient.getArrayIfSingleItem(output[_It][_PKS]));
    }
    return contents;
};
const de_PublicKeySummary = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_CTr] != null) {
        contents[_CTr] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_CTr]));
    }
    if (output[_EK] != null) {
        contents[_EK] = smithyClient.expectString(output[_EK]);
    }
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    return contents;
};
const de_PublicKeySummaryList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_PublicKeySummary(entry);
    });
};
const de_QueryArgProfile = (output, context) => {
    const contents = {};
    if (output[_QA] != null) {
        contents[_QA] = smithyClient.expectString(output[_QA]);
    }
    if (output[_PI] != null) {
        contents[_PI] = smithyClient.expectString(output[_PI]);
    }
    return contents;
};
const de_QueryArgProfileConfig = (output, context) => {
    const contents = {};
    if (output[_FWQAPIU] != null) {
        contents[_FWQAPIU] = smithyClient.parseBoolean(output[_FWQAPIU]);
    }
    if (output[_QAPu] != null) {
        contents[_QAPu] = de_QueryArgProfiles(output[_QAPu]);
    }
    return contents;
};
const de_QueryArgProfileList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_QueryArgProfile(entry);
    });
};
const de_QueryArgProfiles = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_QAP] != null) {
        contents[_It] = de_QueryArgProfileList(smithyClient.getArrayIfSingleItem(output[_It][_QAP]));
    }
    return contents;
};
const de_QueryStringCacheKeys = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_N] != null) {
        contents[_It] = de_QueryStringCacheKeysList(smithyClient.getArrayIfSingleItem(output[_It][_N]));
    }
    return contents;
};
const de_QueryStringCacheKeysList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_QueryStringNames = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_N] != null) {
        contents[_It] = de_QueryStringNamesList(smithyClient.getArrayIfSingleItem(output[_It][_N]));
    }
    return contents;
};
const de_QueryStringNamesList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_RealtimeLogConfig = (output, context) => {
    const contents = {};
    if (output[_ARN] != null) {
        contents[_ARN] = smithyClient.expectString(output[_ARN]);
    }
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_SR] != null) {
        contents[_SR] = smithyClient.strictParseLong(output[_SR]);
    }
    if (String(output.EndPoints).trim() === "") {
        contents[_EP] = [];
    }
    else if (output[_EP] != null && output[_EP][_m] != null) {
        contents[_EP] = de_EndPointList(smithyClient.getArrayIfSingleItem(output[_EP][_m]));
    }
    if (String(output.Fields).trim() === "") {
        contents[_F] = [];
    }
    else if (output[_F] != null && output[_F][_Fi] != null) {
        contents[_F] = de_FieldList(smithyClient.getArrayIfSingleItem(output[_F][_Fi]));
    }
    return contents;
};
const de_RealtimeLogConfigList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_RealtimeLogConfig(entry);
    });
};
const de_RealtimeLogConfigs = (output, context) => {
    const contents = {};
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_m] != null) {
        contents[_It] = de_RealtimeLogConfigList(smithyClient.getArrayIfSingleItem(output[_It][_m]));
    }
    if (output[_IT] != null) {
        contents[_IT] = smithyClient.parseBoolean(output[_IT]);
    }
    if (output[_M] != null) {
        contents[_M] = smithyClient.expectString(output[_M]);
    }
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    return contents;
};
const de_RealtimeMetricsSubscriptionConfig = (output, context) => {
    const contents = {};
    if (output[_RMSS] != null) {
        contents[_RMSS] = smithyClient.expectString(output[_RMSS]);
    }
    return contents;
};
const de_ResponseHeadersPolicy = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_RHPC] != null) {
        contents[_RHPC] = de_ResponseHeadersPolicyConfig(output[_RHPC]);
    }
    return contents;
};
const de_ResponseHeadersPolicyAccessControlAllowHeaders = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_H] != null) {
        contents[_It] = de_AccessControlAllowHeadersList(smithyClient.getArrayIfSingleItem(output[_It][_H]));
    }
    return contents;
};
const de_ResponseHeadersPolicyAccessControlAllowMethods = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_Met] != null) {
        contents[_It] = de_AccessControlAllowMethodsList(smithyClient.getArrayIfSingleItem(output[_It][_Met]));
    }
    return contents;
};
const de_ResponseHeadersPolicyAccessControlAllowOrigins = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_Or] != null) {
        contents[_It] = de_AccessControlAllowOriginsList(smithyClient.getArrayIfSingleItem(output[_It][_Or]));
    }
    return contents;
};
const de_ResponseHeadersPolicyAccessControlExposeHeaders = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_H] != null) {
        contents[_It] = de_AccessControlExposeHeadersList(smithyClient.getArrayIfSingleItem(output[_It][_H]));
    }
    return contents;
};
const de_ResponseHeadersPolicyConfig = (output, context) => {
    const contents = {};
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_CCo] != null) {
        contents[_CCo] = de_ResponseHeadersPolicyCorsConfig(output[_CCo]);
    }
    if (output[_SHC] != null) {
        contents[_SHC] = de_ResponseHeadersPolicySecurityHeadersConfig(output[_SHC]);
    }
    if (output[_STHC] != null) {
        contents[_STHC] = de_ResponseHeadersPolicyServerTimingHeadersConfig(output[_STHC]);
    }
    if (output[_CHC] != null) {
        contents[_CHC] = de_ResponseHeadersPolicyCustomHeadersConfig(output[_CHC]);
    }
    if (output[_RHC] != null) {
        contents[_RHC] = de_ResponseHeadersPolicyRemoveHeadersConfig(output[_RHC]);
    }
    return contents;
};
const de_ResponseHeadersPolicyContentSecurityPolicy = (output, context) => {
    const contents = {};
    if (output[_Ov] != null) {
        contents[_Ov] = smithyClient.parseBoolean(output[_Ov]);
    }
    if (output[_CSP] != null) {
        contents[_CSP] = smithyClient.expectString(output[_CSP]);
    }
    return contents;
};
const de_ResponseHeadersPolicyContentTypeOptions = (output, context) => {
    const contents = {};
    if (output[_Ov] != null) {
        contents[_Ov] = smithyClient.parseBoolean(output[_Ov]);
    }
    return contents;
};
const de_ResponseHeadersPolicyCorsConfig = (output, context) => {
    const contents = {};
    if (output[_ACAO] != null) {
        contents[_ACAO] = de_ResponseHeadersPolicyAccessControlAllowOrigins(output[_ACAO]);
    }
    if (output[_ACAH] != null) {
        contents[_ACAH] = de_ResponseHeadersPolicyAccessControlAllowHeaders(output[_ACAH]);
    }
    if (output[_ACAM] != null) {
        contents[_ACAM] = de_ResponseHeadersPolicyAccessControlAllowMethods(output[_ACAM]);
    }
    if (output[_ACAC] != null) {
        contents[_ACAC] = smithyClient.parseBoolean(output[_ACAC]);
    }
    if (output[_ACEH] != null) {
        contents[_ACEH] = de_ResponseHeadersPolicyAccessControlExposeHeaders(output[_ACEH]);
    }
    if (output[_ACMAS] != null) {
        contents[_ACMAS] = smithyClient.strictParseInt32(output[_ACMAS]);
    }
    if (output[_OO] != null) {
        contents[_OO] = smithyClient.parseBoolean(output[_OO]);
    }
    return contents;
};
const de_ResponseHeadersPolicyCustomHeader = (output, context) => {
    const contents = {};
    if (output[_H] != null) {
        contents[_H] = smithyClient.expectString(output[_H]);
    }
    if (output[_V] != null) {
        contents[_V] = smithyClient.expectString(output[_V]);
    }
    if (output[_Ov] != null) {
        contents[_Ov] = smithyClient.parseBoolean(output[_Ov]);
    }
    return contents;
};
const de_ResponseHeadersPolicyCustomHeaderList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_ResponseHeadersPolicyCustomHeader(entry);
    });
};
const de_ResponseHeadersPolicyCustomHeadersConfig = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_RHPCH] != null) {
        contents[_It] = de_ResponseHeadersPolicyCustomHeaderList(smithyClient.getArrayIfSingleItem(output[_It][_RHPCH]));
    }
    return contents;
};
const de_ResponseHeadersPolicyFrameOptions = (output, context) => {
    const contents = {};
    if (output[_Ov] != null) {
        contents[_Ov] = smithyClient.parseBoolean(output[_Ov]);
    }
    if (output[_FO] != null) {
        contents[_FO] = smithyClient.expectString(output[_FO]);
    }
    return contents;
};
const de_ResponseHeadersPolicyList = (output, context) => {
    const contents = {};
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_RHPS] != null) {
        contents[_It] = de_ResponseHeadersPolicySummaryList(smithyClient.getArrayIfSingleItem(output[_It][_RHPS]));
    }
    return contents;
};
const de_ResponseHeadersPolicyReferrerPolicy = (output, context) => {
    const contents = {};
    if (output[_Ov] != null) {
        contents[_Ov] = smithyClient.parseBoolean(output[_Ov]);
    }
    if (output[_RP] != null) {
        contents[_RP] = smithyClient.expectString(output[_RP]);
    }
    return contents;
};
const de_ResponseHeadersPolicyRemoveHeader = (output, context) => {
    const contents = {};
    if (output[_H] != null) {
        contents[_H] = smithyClient.expectString(output[_H]);
    }
    return contents;
};
const de_ResponseHeadersPolicyRemoveHeaderList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_ResponseHeadersPolicyRemoveHeader(entry);
    });
};
const de_ResponseHeadersPolicyRemoveHeadersConfig = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_RHPRH] != null) {
        contents[_It] = de_ResponseHeadersPolicyRemoveHeaderList(smithyClient.getArrayIfSingleItem(output[_It][_RHPRH]));
    }
    return contents;
};
const de_ResponseHeadersPolicySecurityHeadersConfig = (output, context) => {
    const contents = {};
    if (output[_XSSP] != null) {
        contents[_XSSP] = de_ResponseHeadersPolicyXSSProtection(output[_XSSP]);
    }
    if (output[_FOr] != null) {
        contents[_FOr] = de_ResponseHeadersPolicyFrameOptions(output[_FOr]);
    }
    if (output[_RP] != null) {
        contents[_RP] = de_ResponseHeadersPolicyReferrerPolicy(output[_RP]);
    }
    if (output[_CSP] != null) {
        contents[_CSP] = de_ResponseHeadersPolicyContentSecurityPolicy(output[_CSP]);
    }
    if (output[_CTO] != null) {
        contents[_CTO] = de_ResponseHeadersPolicyContentTypeOptions(output[_CTO]);
    }
    if (output[_STS] != null) {
        contents[_STS] = de_ResponseHeadersPolicyStrictTransportSecurity(output[_STS]);
    }
    return contents;
};
const de_ResponseHeadersPolicyServerTimingHeadersConfig = (output, context) => {
    const contents = {};
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    if (output[_SR] != null) {
        contents[_SR] = smithyClient.strictParseFloat(output[_SR]);
    }
    return contents;
};
const de_ResponseHeadersPolicyStrictTransportSecurity = (output, context) => {
    const contents = {};
    if (output[_Ov] != null) {
        contents[_Ov] = smithyClient.parseBoolean(output[_Ov]);
    }
    if (output[_ISn] != null) {
        contents[_ISn] = smithyClient.parseBoolean(output[_ISn]);
    }
    if (output[_Pre] != null) {
        contents[_Pre] = smithyClient.parseBoolean(output[_Pre]);
    }
    if (output[_ACMAS] != null) {
        contents[_ACMAS] = smithyClient.strictParseInt32(output[_ACMAS]);
    }
    return contents;
};
const de_ResponseHeadersPolicySummary = (output, context) => {
    const contents = {};
    if (output[_Ty] != null) {
        contents[_Ty] = smithyClient.expectString(output[_Ty]);
    }
    if (output[_RHP] != null) {
        contents[_RHP] = de_ResponseHeadersPolicy(output[_RHP]);
    }
    return contents;
};
const de_ResponseHeadersPolicySummaryList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_ResponseHeadersPolicySummary(entry);
    });
};
const de_ResponseHeadersPolicyXSSProtection = (output, context) => {
    const contents = {};
    if (output[_Ov] != null) {
        contents[_Ov] = smithyClient.parseBoolean(output[_Ov]);
    }
    if (output[_Pro] != null) {
        contents[_Pro] = smithyClient.parseBoolean(output[_Pro]);
    }
    if (output[_MB] != null) {
        contents[_MB] = smithyClient.parseBoolean(output[_MB]);
    }
    if (output[_RU] != null) {
        contents[_RU] = smithyClient.expectString(output[_RU]);
    }
    return contents;
};
const de_Restrictions = (output, context) => {
    const contents = {};
    if (output[_GRe] != null) {
        contents[_GRe] = de_GeoRestriction(output[_GRe]);
    }
    return contents;
};
const de_S3Origin = (output, context) => {
    const contents = {};
    if (output[_DN] != null) {
        contents[_DN] = smithyClient.expectString(output[_DN]);
    }
    if (output[_OAI] != null) {
        contents[_OAI] = smithyClient.expectString(output[_OAI]);
    }
    return contents;
};
const de_S3OriginConfig = (output, context) => {
    const contents = {};
    if (output[_OAI] != null) {
        contents[_OAI] = smithyClient.expectString(output[_OAI]);
    }
    if (output[_ORT] != null) {
        contents[_ORT] = smithyClient.strictParseInt32(output[_ORT]);
    }
    return contents;
};
const de_SessionStickinessConfig = (output, context) => {
    const contents = {};
    if (output[_ITTL] != null) {
        contents[_ITTL] = smithyClient.strictParseInt32(output[_ITTL]);
    }
    if (output[_MTTLax] != null) {
        contents[_MTTLax] = smithyClient.strictParseInt32(output[_MTTLax]);
    }
    return contents;
};
const de_Signer = (output, context) => {
    const contents = {};
    if (output[_AAN] != null) {
        contents[_AAN] = smithyClient.expectString(output[_AAN]);
    }
    if (output[_KPIe] != null) {
        contents[_KPIe] = de_KeyPairIds(output[_KPIe]);
    }
    return contents;
};
const de_SignerList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_Signer(entry);
    });
};
const de_SslProtocolsList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_StagingDistributionDnsNameList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_StagingDistributionDnsNames = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_DNn] != null) {
        contents[_It] = de_StagingDistributionDnsNameList(smithyClient.getArrayIfSingleItem(output[_It][_DNn]));
    }
    return contents;
};
const de_StatusCodeList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.strictParseInt32(entry);
    });
};
const de_StatusCodes = (output, context) => {
    const contents = {};
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_SCta] != null) {
        contents[_It] = de_StatusCodeList(smithyClient.getArrayIfSingleItem(output[_It][_SCta]));
    }
    return contents;
};
const de_StreamingDistribution = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_ARN] != null) {
        contents[_ARN] = smithyClient.expectString(output[_ARN]);
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_DN] != null) {
        contents[_DN] = smithyClient.expectString(output[_DN]);
    }
    if (output[_ATS] != null) {
        contents[_ATS] = de_ActiveTrustedSigners(output[_ATS]);
    }
    if (output[_SDC] != null) {
        contents[_SDC] = de_StreamingDistributionConfig(output[_SDC]);
    }
    return contents;
};
const de_StreamingDistributionConfig = (output, context) => {
    const contents = {};
    if (output[_CR] != null) {
        contents[_CR] = smithyClient.expectString(output[_CR]);
    }
    if (output[_SO] != null) {
        contents[_SO] = de_S3Origin(output[_SO]);
    }
    if (output[_Al] != null) {
        contents[_Al] = de_Aliases(output[_Al]);
    }
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    if (output[_Lo] != null) {
        contents[_Lo] = de_StreamingLoggingConfig(output[_Lo]);
    }
    if (output[_TS] != null) {
        contents[_TS] = de_TrustedSigners(output[_TS]);
    }
    if (output[_PC] != null) {
        contents[_PC] = smithyClient.expectString(output[_PC]);
    }
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    return contents;
};
const de_StreamingDistributionList = (output, context) => {
    const contents = {};
    if (output[_M] != null) {
        contents[_M] = smithyClient.expectString(output[_M]);
    }
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_IT] != null) {
        contents[_IT] = smithyClient.parseBoolean(output[_IT]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_SDS] != null) {
        contents[_It] = de_StreamingDistributionSummaryList(smithyClient.getArrayIfSingleItem(output[_It][_SDS]));
    }
    return contents;
};
const de_StreamingDistributionSummary = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_ARN] != null) {
        contents[_ARN] = smithyClient.expectString(output[_ARN]);
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_DN] != null) {
        contents[_DN] = smithyClient.expectString(output[_DN]);
    }
    if (output[_SO] != null) {
        contents[_SO] = de_S3Origin(output[_SO]);
    }
    if (output[_Al] != null) {
        contents[_Al] = de_Aliases(output[_Al]);
    }
    if (output[_TS] != null) {
        contents[_TS] = de_TrustedSigners(output[_TS]);
    }
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    if (output[_PC] != null) {
        contents[_PC] = smithyClient.expectString(output[_PC]);
    }
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    return contents;
};
const de_StreamingDistributionSummaryList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_StreamingDistributionSummary(entry);
    });
};
const de_StreamingLoggingConfig = (output, context) => {
    const contents = {};
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    if (output[_B] != null) {
        contents[_B] = smithyClient.expectString(output[_B]);
    }
    if (output[_Pr] != null) {
        contents[_Pr] = smithyClient.expectString(output[_Pr]);
    }
    return contents;
};
const de_StringSchemaConfig = (output, context) => {
    const contents = {};
    if (output[_Co] != null) {
        contents[_Co] = smithyClient.expectString(output[_Co]);
    }
    if (output[_DV] != null) {
        contents[_DV] = smithyClient.expectString(output[_DV]);
    }
    if (output[_Req] != null) {
        contents[_Req] = smithyClient.parseBoolean(output[_Req]);
    }
    return contents;
};
const de_Tag = (output, context) => {
    const contents = {};
    if (output[_K] != null) {
        contents[_K] = smithyClient.expectString(output[_K]);
    }
    if (output[_V] != null) {
        contents[_V] = smithyClient.expectString(output[_V]);
    }
    return contents;
};
const de_TagList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_Tag(entry);
    });
};
const de_Tags = (output, context) => {
    const contents = {};
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_Ta] != null) {
        contents[_It] = de_TagList(smithyClient.getArrayIfSingleItem(output[_It][_Ta]));
    }
    return contents;
};
const de_TenantConfig = (output, context) => {
    const contents = {};
    if (String(output.ParameterDefinitions).trim() === "") {
        contents[_PDa] = [];
    }
    else if (output[_PDa] != null && output[_PDa][_m] != null) {
        contents[_PDa] = de_ParameterDefinitions(smithyClient.getArrayIfSingleItem(output[_PDa][_m]));
    }
    return contents;
};
const de_TestResult = (output, context) => {
    const contents = {};
    if (output[_FSu] != null) {
        contents[_FSu] = de_FunctionSummary(output[_FSu]);
    }
    if (output[_CU] != null) {
        contents[_CU] = smithyClient.expectString(output[_CU]);
    }
    if (String(output.FunctionExecutionLogs).trim() === "") {
        contents[_FEL] = [];
    }
    else if (output[_FEL] != null && output[_FEL][_m] != null) {
        contents[_FEL] = de_FunctionExecutionLogList(smithyClient.getArrayIfSingleItem(output[_FEL][_m]));
    }
    if (output[_FEM] != null) {
        contents[_FEM] = smithyClient.expectString(output[_FEM]);
    }
    if (output[_FOu] != null) {
        contents[_FOu] = smithyClient.expectString(output[_FOu]);
    }
    return contents;
};
const de_TrafficConfig = (output, context) => {
    const contents = {};
    if (output[_SWC] != null) {
        contents[_SWC] = de_ContinuousDeploymentSingleWeightConfig(output[_SWC]);
    }
    if (output[_SHCi] != null) {
        contents[_SHCi] = de_ContinuousDeploymentSingleHeaderConfig(output[_SHCi]);
    }
    if (output[_Ty] != null) {
        contents[_Ty] = smithyClient.expectString(output[_Ty]);
    }
    return contents;
};
const de_TrustedKeyGroupIdList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return smithyClient.expectString(entry);
    });
};
const de_TrustedKeyGroups = (output, context) => {
    const contents = {};
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_KG] != null) {
        contents[_It] = de_TrustedKeyGroupIdList(smithyClient.getArrayIfSingleItem(output[_It][_KG]));
    }
    return contents;
};
const de_TrustedSigners = (output, context) => {
    const contents = {};
    if (output[_E] != null) {
        contents[_E] = smithyClient.parseBoolean(output[_E]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_AAN] != null) {
        contents[_It] = de_AwsAccountNumberList(smithyClient.getArrayIfSingleItem(output[_It][_AAN]));
    }
    return contents;
};
const de_ValidationTokenDetail = (output, context) => {
    const contents = {};
    if (output[_Do] != null) {
        contents[_Do] = smithyClient.expectString(output[_Do]);
    }
    if (output[_RTed] != null) {
        contents[_RTed] = smithyClient.expectString(output[_RTed]);
    }
    if (output[_RF] != null) {
        contents[_RF] = smithyClient.expectString(output[_RF]);
    }
    return contents;
};
const de_ValidationTokenDetailList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_ValidationTokenDetail(entry);
    });
};
const de_ViewerCertificate = (output, context) => {
    const contents = {};
    if (output[_CFDC] != null) {
        contents[_CFDC] = smithyClient.parseBoolean(output[_CFDC]);
    }
    if (output[_IAMCI] != null) {
        contents[_IAMCI] = smithyClient.expectString(output[_IAMCI]);
    }
    if (output[_ACMCA] != null) {
        contents[_ACMCA] = smithyClient.expectString(output[_ACMCA]);
    }
    if (output[_SSLSM] != null) {
        contents[_SSLSM] = smithyClient.expectString(output[_SSLSM]);
    }
    if (output[_MPV] != null) {
        contents[_MPV] = smithyClient.expectString(output[_MPV]);
    }
    if (output[_Ce] != null) {
        contents[_Ce] = smithyClient.expectString(output[_Ce]);
    }
    if (output[_CS] != null) {
        contents[_CS] = smithyClient.expectString(output[_CS]);
    }
    return contents;
};
const de_VpcOrigin = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_Ar] != null) {
        contents[_Ar] = smithyClient.expectString(output[_Ar]);
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    if (output[_CTr] != null) {
        contents[_CTr] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_CTr]));
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_VOEC] != null) {
        contents[_VOEC] = de_VpcOriginEndpointConfig(output[_VOEC]);
    }
    return contents;
};
const de_VpcOriginConfig = (output, context) => {
    const contents = {};
    if (output[_VOI] != null) {
        contents[_VOI] = smithyClient.expectString(output[_VOI]);
    }
    if (output[_ORT] != null) {
        contents[_ORT] = smithyClient.strictParseInt32(output[_ORT]);
    }
    if (output[_OKT] != null) {
        contents[_OKT] = smithyClient.strictParseInt32(output[_OKT]);
    }
    return contents;
};
const de_VpcOriginEndpointConfig = (output, context) => {
    const contents = {};
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_Ar] != null) {
        contents[_Ar] = smithyClient.expectString(output[_Ar]);
    }
    if (output[_HTTPP] != null) {
        contents[_HTTPP] = smithyClient.strictParseInt32(output[_HTTPP]);
    }
    if (output[_HTTPSP] != null) {
        contents[_HTTPSP] = smithyClient.strictParseInt32(output[_HTTPSP]);
    }
    if (output[_OPP] != null) {
        contents[_OPP] = smithyClient.expectString(output[_OPP]);
    }
    if (output[_OSP] != null) {
        contents[_OSP] = de_OriginSslProtocols(output[_OSP]);
    }
    return contents;
};
const de_VpcOriginList = (output, context) => {
    const contents = {};
    if (output[_M] != null) {
        contents[_M] = smithyClient.expectString(output[_M]);
    }
    if (output[_NM] != null) {
        contents[_NM] = smithyClient.expectString(output[_NM]);
    }
    if (output[_MI] != null) {
        contents[_MI] = smithyClient.strictParseInt32(output[_MI]);
    }
    if (output[_IT] != null) {
        contents[_IT] = smithyClient.parseBoolean(output[_IT]);
    }
    if (output[_Q] != null) {
        contents[_Q] = smithyClient.strictParseInt32(output[_Q]);
    }
    if (String(output.Items).trim() === "") {
        contents[_It] = [];
    }
    else if (output[_It] != null && output[_It][_VOS] != null) {
        contents[_It] = de_VpcOriginSummaryList(smithyClient.getArrayIfSingleItem(output[_It][_VOS]));
    }
    return contents;
};
const de_VpcOriginSummary = (output, context) => {
    const contents = {};
    if (output[_Id] != null) {
        contents[_Id] = smithyClient.expectString(output[_Id]);
    }
    if (output[_N] != null) {
        contents[_N] = smithyClient.expectString(output[_N]);
    }
    if (output[_Sta] != null) {
        contents[_Sta] = smithyClient.expectString(output[_Sta]);
    }
    if (output[_CTr] != null) {
        contents[_CTr] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_CTr]));
    }
    if (output[_LMT] != null) {
        contents[_LMT] = smithyClient.expectNonNull(smithyClient.parseRfc3339DateTimeWithOffset(output[_LMT]));
    }
    if (output[_Ar] != null) {
        contents[_Ar] = smithyClient.expectString(output[_Ar]);
    }
    if (output[_OEA] != null) {
        contents[_OEA] = smithyClient.expectString(output[_OEA]);
    }
    return contents;
};
const de_VpcOriginSummaryList = (output, context) => {
    return (output || [])
        .filter((e) => e != null)
        .map((entry) => {
        return de_VpcOriginSummary(entry);
    });
};
const de_WebAclCustomization = (output, context) => {
    const contents = {};
    if (output[_Ac] != null) {
        contents[_Ac] = smithyClient.expectString(output[_Ac]);
    }
    if (output[_Ar] != null) {
        contents[_Ar] = smithyClient.expectString(output[_Ar]);
    }
    return contents;
};
const deserializeMetadata = (output) => ({
    httpStatusCode: output.statusCode,
    requestId: output.headers["x-amzn-requestid"] ?? output.headers["x-amzn-request-id"] ?? output.headers["x-amz-request-id"],
    extendedRequestId: output.headers["x-amz-id-2"],
    cfId: output.headers["x-amz-cf-id"],
});
const _A = "Alias";
const _AAN = "AwsAccountNumber";
const _ACAC = "AccessControlAllowCredentials";
const _ACAH = "AccessControlAllowHeaders";
const _ACAM = "AccessControlAllowMethods";
const _ACAO = "AccessControlAllowOrigins";
const _ACEH = "AccessControlExposeHeaders";
const _ACMAS = "AccessControlMaxAgeSec";
const _ACMCA = "ACMCertificateArn";
const _ADTWACLR = "AssociateDistributionTenantWebACLRequest";
const _ADWACLR = "AssociateDistributionWebACLRequest";
const _AF = "AssociationFilter";
const _AI = "AnycastIps";
const _AICPR = "AliasICPRecordals";
const _AICPRl = "AliasICPRecordal";
const _AILI = "AnycastIpListId";
const _AILN = "AnycastIpListName";
const _AILS = "AnycastIpListSummary";
const _AIc = "AccountId";
const _AIn = "AnycastIp";
const _AM = "AllowedMethods";
const _ARN = "ARN";
const _ATKG = "ActiveTrustedKeyGroups";
const _ATS = "ActiveTrustedSigners";
const _Ac = "Action";
const _Al = "Aliases";
const _Ar = "Arn";
const _B = "Bucket";
const _C = "Customizations";
const _CA = "CertificateArn";
const _CAILR = "CreateAnycastIpListRequest";
const _CAT = "CustomizationActionType";
const _CAo = "ConnectionAttempts";
const _CAon = "ConflictingAlias";
const _CB = "CacheBehavior";
const _CBa = "CacheBehaviors";
const _CBo = "CookieBehavior";
const _CC = "CookiesConfig";
const _CCGR = "CreateConnectionGroupRequest";
const _CCo = "CorsConfig";
const _CDP = "ContinuousDeploymentPolicy";
const _CDPC = "ContinuousDeploymentPolicyConfig";
const _CDPI = "ContinuousDeploymentPolicyId";
const _CDPS = "ContinuousDeploymentPolicySummary";
const _CDPT = "ContinuousDeploymentPolicyType";
const _CDR = "CopyDistributionRequest";
const _CDSHC = "ContinuousDeploymentSingleHeaderConfig";
const _CDSWC = "ContinuousDeploymentSingleWeightConfig";
const _CDTR = "CreateDistributionTenantRequest";
const _CER = "CustomErrorResponse";
const _CERu = "CustomErrorResponses";
const _CFDC = "CloudFrontDefaultCertificate";
const _CFOAIC = "CloudFrontOriginAccessIdentityConfig";
const _CFOAIS = "CloudFrontOriginAccessIdentitySummary";
const _CFR = "CreateFunctionRequest";
const _CG = "ConnectionGroups";
const _CGAF = "ConnectionGroupAssociationFilter";
const _CGI = "ConnectionGroupId";
const _CGS = "ConnectionGroupSummary";
const _CH = "CustomHeaders";
const _CHC = "CustomHeadersConfig";
const _CKVSR = "CreateKeyValueStoreRequest";
const _CM = "CachedMethods";
const _CMo = "ConnectionMode";
const _CN = "CookieNames";
const _CNAME = "CNAME";
const _COC = "CustomOriginConfig";
const _CP = "CookiePreference";
const _CPC = "CachePolicyConfig";
const _CPCB = "CachePolicyCookieBehavior";
const _CPCC = "CachePolicyCookiesConfig";
const _CPHB = "CachePolicyHeaderBehavior";
const _CPHC = "CachePolicyHeadersConfig";
const _CPI = "CachePolicyId";
const _CPQSB = "CachePolicyQueryStringBehavior";
const _CPQSC = "CachePolicyQueryStringsConfig";
const _CPS = "CachePolicySummary";
const _CPa = "CachePolicy";
const _CR = "CallerReference";
const _CRLCR = "CreateRealtimeLogConfigRequest";
const _CS = "CertificateSource";
const _CSP = "ContentSecurityPolicy";
const _CSe = "CertificateStatus";
const _CT = "ContentType";
const _CTLP = "CertificateTransparencyLoggingPreference";
const _CTO = "ContentTypeOptions";
const _CTP = "ContentTypeProfile";
const _CTPC = "ContentTypeProfileConfig";
const _CTPo = "ContentTypeProfiles";
const _CTo = "CommentType";
const _CTon = "ConnectionTimeout";
const _CTr = "CreatedTime";
const _CTre = "CreateTime";
const _CU = "ComputeUtilization";
const _CVOR = "CreateVpcOriginRequest";
const _Ce = "Certificate";
const _Co = "Comment";
const _Com = "Compress";
const _Coo = "Cookies";
const _D = "Domains";
const _DC = "DomainConflicts";
const _DCB = "DefaultCacheBehavior";
const _DCL = "DnsConfigurationList";
const _DCVR = "DomainControlValidationResource";
const _DCWT = "DistributionConfigWithTags";
const _DCi = "DistributionConfig";
const _DCn = "DnsConfiguration";
const _DI = "DistributionId";
const _DIo = "DomainItem";
const _DN = "DomainName";
const _DNn = "DnsName";
const _DRI = "DistributionResourceId";
const _DRLCR = "DeleteRealtimeLogConfigRequest";
const _DRO = "DefaultRootObject";
const _DS = "DistributionSummary";
const _DTAF = "DistributionTenantAssociationFilter";
const _DTI = "DistributionTenantId";
const _DTL = "DistributionTenantList";
const _DTS = "DistributionTenantSummary";
const _DTTL = "DefaultTTL";
const _DV = "DefaultValue";
const _De = "Description";
const _Def = "Definition";
const _Do = "Domain";
const _E = "Enabled";
const _EAEB = "EnableAcceptEncodingBrotli";
const _EAEG = "EnableAcceptEncodingGzip";
const _EC = "ErrorCode";
const _ECMTTL = "ErrorCachingMinTTL";
const _EE = "EncryptionEntities";
const _EEn = "EncryptionEntity";
const _EK = "EncodedKey";
const _EO = "EventObject";
const _EP = "EndPoints";
const _EPn = "EndPoint";
const _ET = "ETag";
const _ETv = "EventType";
const _F = "Fields";
const _FA = "FunctionAssociations";
const _FARN = "FunctionARN";
const _FAu = "FunctionAssociation";
const _FB = "FunctionBlob";
const _FC = "FunctionCode";
const _FCa = "FailoverCriteria";
const _FCu = "FunctionConfig";
const _FEL = "FunctionExecutionLogs";
const _FEM = "FunctionErrorMessage";
const _FEO = "FunctionEventObject";
const _FLEC = "FieldLevelEncryptionConfig";
const _FLEI = "FieldLevelEncryptionId";
const _FLEPC = "FieldLevelEncryptionProfileConfig";
const _FLEPS = "FieldLevelEncryptionProfileSummary";
const _FLES = "FieldLevelEncryptionSummary";
const _FM = "FunctionMetadata";
const _FN = "FunctionName";
const _FO = "FrameOption";
const _FOL = "FrameOptionsList";
const _FOr = "FrameOptions";
const _FOu = "FunctionOutput";
const _FP = "FieldPatterns";
const _FPi = "FieldPattern";
const _FR = "FunctionRuntime";
const _FS = "FunctionStage";
const _FSu = "FunctionSummary";
const _FV = "ForwardedValues";
const _FWCTIU = "ForwardWhenContentTypeIsUnknown";
const _FWQAPIU = "ForwardWhenQueryArgProfileIsUnknown";
const _Fi = "Field";
const _Fo = "Format";
const _For = "Forward";
const _GC = "GrpcConfig";
const _GR = "GeoRestrictions";
const _GRC = "GeoRestrictionCustomization";
const _GRLCR = "GetRealtimeLogConfigRequest";
const _GRT = "GeoRestrictionType";
const _GRe = "GeoRestriction";
const _H = "Header";
const _HB = "HeaderBehavior";
const _HC = "HeadersConfig";
const _HN = "HeaderName";
const _HTTPP = "HTTPPort";
const _HTTPSP = "HTTPSPort";
const _HV = "HttpVersion";
const _HVe = "HeaderValue";
const _He = "Headers";
const _I = "Identifier";
const _IAMCI = "IAMCertificateId";
const _IAT = "IpAddressType";
const _IB = "InvalidationBatch";
const _IBn = "IncludeBody";
const _IC = "IpCount";
const _ICPRS = "ICPRecordalStatus";
const _ICn = "IncludeCookies";
const _ID = "IsDefault";
const _IE = "Ipv6Enabled";
const _IIPVE = "IsIPV6Enabled";
const _IM = "IfMatch";
const _IPIB = "InProgressInvalidationBatches";
const _IS = "ImportSource";
const _IST = "ImportSourceType";
const _ISn = "IncludeSubdomains";
const _ISnv = "InvalidationSummary";
const _ISt = "ItemSelection";
const _IT = "IsTruncated";
const _ITTL = "IdleTTL";
const _Id = "Id";
const _It = "Items";
const _K = "Key";
const _KG = "KeyGroup";
const _KGC = "KeyGroupConfig";
const _KGI = "KeyGroupId";
const _KGS = "KeyGroupSummary";
const _KPI = "KeyPairId";
const _KPIe = "KeyPairIds";
const _KSC = "KinesisStreamConfig";
const _KVS = "KeyValueStore";
const _KVSA = "KeyValueStoreAssociations";
const _KVSARN = "KeyValueStoreARN";
const _KVSAe = "KeyValueStoreAssociation";
const _KVSC = "KeyValueStoreComment";
const _KVSN = "KeyValueStoreName";
const _L = "Location";
const _LC = "LoggingConfig";
const _LCGR = "ListConnectionGroupsRequest";
const _LDBRLCR = "ListDistributionsByRealtimeLogConfigRequest";
const _LDCR = "ListDomainConflictsRequest";
const _LDTBCR = "ListDistributionTenantsByCustomizationRequest";
const _LDTR = "ListDistributionTenantsRequest";
const _LFA = "LambdaFunctionAssociations";
const _LFARN = "LambdaFunctionARN";
const _LFAa = "LambdaFunctionAssociation";
const _LMT = "LastModifiedTime";
const _Lo = "Logging";
const _Loc = "Locations";
const _M = "Marker";
const _MB = "ModeBlock";
const _MCR = "ManagedCertificateRequest";
const _MI = "MaxItems";
const _MPV = "MinimumProtocolVersion";
const _MS = "MonitoringSubscription";
const _MTTL = "MinTTL";
const _MTTLa = "MaxTTL";
const _MTTLax = "MaximumTTL";
const _Me = "Message";
const _Mem = "Members";
const _Met = "Method";
const _N = "Name";
const _NM = "NextMarker";
const _O = "Operation";
const _OACC = "OriginAccessControlConfig";
const _OACI = "OriginAccessControlId";
const _OACOT = "OriginAccessControlOriginType";
const _OACOTr = "OriginAccessControlOriginTypes";
const _OACS = "OriginAccessControlSummary";
const _OACSB = "OriginAccessControlSigningBehaviors";
const _OACSP = "OriginAccessControlSigningProtocols";
const _OAI = "OriginAccessIdentity";
const _OCH = "OriginCustomHeader";
const _OEA = "OriginEndpointArn";
const _OG = "OriginGroups";
const _OGFC = "OriginGroupFailoverCriteria";
const _OGM = "OriginGroupMember";
const _OGMr = "OriginGroupMembers";
const _OGSC = "OriginGroupSelectionCriteria";
const _OGr = "OriginGroup";
const _OI = "OriginId";
const _OKT = "OriginKeepaliveTimeout";
const _OO = "OriginOverride";
const _OP = "OriginPath";
const _OPP = "OriginProtocolPolicy";
const _ORP = "OriginRequestPolicy";
const _ORPC = "OriginRequestPolicyConfig";
const _ORPCB = "OriginRequestPolicyCookieBehavior";
const _ORPCC = "OriginRequestPolicyCookiesConfig";
const _ORPHB = "OriginRequestPolicyHeaderBehavior";
const _ORPHC = "OriginRequestPolicyHeadersConfig";
const _ORPI = "OriginRequestPolicyId";
const _ORPQSB = "OriginRequestPolicyQueryStringBehavior";
const _ORPQSC = "OriginRequestPolicyQueryStringsConfig";
const _ORPS = "OriginRequestPolicySummary";
const _ORT = "OriginReadTimeout";
const _OS = "OriginShield";
const _OSP = "OriginSslProtocols";
const _OSR = "OriginShieldRegion";
const _Or = "Origin";
const _Ori = "Origins";
const _Ov = "Override";
const _P = "Parameters";
const _PC = "PriceClass";
const _PD = "ParameterDefinition";
const _PDN = "PrimaryDomainName";
const _PDS = "ParameterDefinitionSchema";
const _PDa = "ParameterDefinitions";
const _PI = "ProfileId";
const _PICKAFTO = "ParametersInCacheKeyAndForwardedToOrigin";
const _PIr = "ProviderId";
const _PK = "PublicKey";
const _PKC = "PublicKeyConfig";
const _PKI = "PublicKeyId";
const _PKS = "PublicKeySummary";
const _PN = "ParameterName";
const _PP = "PathPattern";
const _PV = "ParameterValue";
const _Pa = "Paths";
const _Par = "Parameter";
const _Pat = "Path";
const _Pr = "Prefix";
const _Pre = "Preload";
const _Pro = "Protection";
const _Q = "Quantity";
const _QA = "QueryArg";
const _QAP = "QueryArgProfile";
const _QAPC = "QueryArgProfileConfig";
const _QAPu = "QueryArgProfiles";
const _QS = "QueryStrings";
const _QSB = "QueryStringBehavior";
const _QSC = "QueryStringsConfig";
const _QSCK = "QueryStringCacheKeys";
const _QSN = "QueryStringNames";
const _QSu = "QueryString";
const _R = "Resource";
const _RARN = "RoleARN";
const _RC = "ResponseCode";
const _RCT = "ResponseCompletionTimeout";
const _RE = "RoutingEndpoint";
const _RF = "RedirectFrom";
const _RHC = "RemoveHeadersConfig";
const _RHP = "ResponseHeadersPolicy";
const _RHPACAH = "ResponseHeadersPolicyAccessControlAllowHeaders";
const _RHPACAM = "ResponseHeadersPolicyAccessControlAllowMethods";
const _RHPACAMV = "ResponseHeadersPolicyAccessControlAllowMethodsValues";
const _RHPACAO = "ResponseHeadersPolicyAccessControlAllowOrigins";
const _RHPACEH = "ResponseHeadersPolicyAccessControlExposeHeaders";
const _RHPC = "ResponseHeadersPolicyConfig";
const _RHPCC = "ResponseHeadersPolicyCorsConfig";
const _RHPCH = "ResponseHeadersPolicyCustomHeader";
const _RHPCHC = "ResponseHeadersPolicyCustomHeadersConfig";
const _RHPCSP = "ResponseHeadersPolicyContentSecurityPolicy";
const _RHPCTO = "ResponseHeadersPolicyContentTypeOptions";
const _RHPFO = "ResponseHeadersPolicyFrameOptions";
const _RHPI = "ResponseHeadersPolicyId";
const _RHPRH = "ResponseHeadersPolicyRemoveHeader";
const _RHPRHC = "ResponseHeadersPolicyRemoveHeadersConfig";
const _RHPRP = "ResponseHeadersPolicyReferrerPolicy";
const _RHPS = "ResponseHeadersPolicySummary";
const _RHPSHC = "ResponseHeadersPolicySecurityHeadersConfig";
const _RHPSTHC = "ResponseHeadersPolicyServerTimingHeadersConfig";
const _RHPSTS = "ResponseHeadersPolicyStrictTransportSecurity";
const _RHPXSSP = "ResponseHeadersPolicyXSSProtection";
const _RI = "ResourceId";
const _RLC = "RealtimeLogConfig";
const _RLCA = "RealtimeLogConfigArn";
const _RLCN = "RealtimeLogConfigName";
const _RMSC = "RealtimeMetricsSubscriptionConfig";
const _RMSS = "RealtimeMetricsSubscriptionStatus";
const _RP = "ReferrerPolicy";
const _RPL = "ReferrerPolicyList";
const _RPP = "ResponsePagePath";
const _RT = "RestrictionType";
const _RTe = "ResourceType";
const _RTed = "RedirectTo";
const _RU = "ReportUri";
const _Re = "Restrictions";
const _Rea = "Reason";
const _Req = "Required";
const _Ru = "Runtime";
const _S = "Staging";
const _SARN = "SourceARN";
const _SARNt = "StreamARN";
const _SB = "SigningBehavior";
const _SC = "SelectionCriteria";
const _SCI = "ServerCertificateId";
const _SCUI = "S3CanonicalUserId";
const _SCt = "StatusCodes";
const _SCta = "StatusCode";
const _SDC = "StreamingDistributionConfig";
const _SDCWT = "StreamingDistributionConfigWithTags";
const _SDDN = "StagingDistributionDnsNames";
const _SDI = "StagingDistributionId";
const _SDS = "StreamingDistributionSummary";
const _SHC = "SecurityHeadersConfig";
const _SHCi = "SingleHeaderConfig";
const _SLC = "StreamingLoggingConfig";
const _SO = "S3Origin";
const _SOC = "S3OriginConfig";
const _SP = "SigningProtocol";
const _SPs = "SslProtocol";
const _SR = "SamplingRate";
const _SS = "SmoothStreaming";
const _SSC = "SessionStickinessConfig";
const _SSCt = "StringSchemaConfig";
const _SSLSM = "SSLSupportMethod";
const _SSt = "StringSchema";
const _ST = "StreamType";
const _STHC = "ServerTimingHeadersConfig";
const _STS = "StrictTransportSecurity";
const _STo = "SourceType";
const _SWC = "SingleWeightConfig";
const _Si = "Signer";
const _St = "Stage";
const _Sta = "Status";
const _T = "Tags";
const _TC = "TrafficConfig";
const _TCe = "TenantConfig";
const _TFR = "TestFunctionRequest";
const _TK = "TagKey";
const _TKG = "TrustedKeyGroups";
const _TKa = "TagKeys";
const _TOI = "TargetOriginId";
const _TR = "TargetResource";
const _TS = "TrustedSigners";
const _TV = "TagValue";
const _Ta = "Tag";
const _Ty = "Type";
const _UCGR = "UpdateConnectionGroupRequest";
const _UDAR = "UpdateDomainAssociationRequest";
const _UDTR = "UpdateDistributionTenantRequest";
const _UFR = "UpdateFunctionRequest";
const _UKVSR = "UpdateKeyValueStoreRequest";
const _URLCR = "UpdateRealtimeLogConfigRequest";
const _V = "Value";
const _VC = "ViewerCertificate";
const _VDCR = "VerifyDnsConfigurationRequest";
const _VOC = "VpcOriginConfig";
const _VOEC = "VpcOriginEndpointConfig";
const _VOI = "VpcOriginId";
const _VOS = "VpcOriginSummary";
const _VPP = "ViewerProtocolPolicy";
const _VTD = "ValidationTokenDetails";
const _VTH = "ValidationTokenHost";
const _W = "Weight";
const _WA = "WebAcl";
const _WAC = "WebAclCustomization";
const _WACLA = "WebACLArn";
const _WACLI = "WebACLId";
const _WN = "WhitelistedNames";
const _WT = "WithTags";
const _XSSP = "XSSProtection";
const _b = "boolean";
const _ct = "content-type";
const _d = "domain";
const _e = "etag";
const _et = "ettag";
const _f = "float";
const _i = "integer";
const _im = "if-match";
const _l = "long";
const _lo = "location";
const _m = "member";
const _s = "string";
const _sST = "sensitiveStringType";
const _st = "staging";
const _ve = '<?xml version="1.0" encoding="UTF-8"?>';

class AssociateAliasCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "AssociateAlias", {})
    .n("CloudFrontClient", "AssociateAliasCommand")
    .f(void 0, void 0)
    .ser(se_AssociateAliasCommand)
    .de(de_AssociateAliasCommand)
    .build() {
}

class AssociateDistributionTenantWebACLCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "AssociateDistributionTenantWebACL", {})
    .n("CloudFrontClient", "AssociateDistributionTenantWebACLCommand")
    .f(void 0, void 0)
    .ser(se_AssociateDistributionTenantWebACLCommand)
    .de(de_AssociateDistributionTenantWebACLCommand)
    .build() {
}

class AssociateDistributionWebACLCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "AssociateDistributionWebACL", {})
    .n("CloudFrontClient", "AssociateDistributionWebACLCommand")
    .f(void 0, void 0)
    .ser(se_AssociateDistributionWebACLCommand)
    .de(de_AssociateDistributionWebACLCommand)
    .build() {
}

class CopyDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CopyDistribution", {})
    .n("CloudFrontClient", "CopyDistributionCommand")
    .f(void 0, CopyDistributionResultFilterSensitiveLog)
    .ser(se_CopyDistributionCommand)
    .de(de_CopyDistributionCommand)
    .build() {
}

class CreateAnycastIpListCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateAnycastIpList", {})
    .n("CloudFrontClient", "CreateAnycastIpListCommand")
    .f(void 0, void 0)
    .ser(se_CreateAnycastIpListCommand)
    .de(de_CreateAnycastIpListCommand)
    .build() {
}

class CreateCachePolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateCachePolicy", {})
    .n("CloudFrontClient", "CreateCachePolicyCommand")
    .f(void 0, void 0)
    .ser(se_CreateCachePolicyCommand)
    .de(de_CreateCachePolicyCommand)
    .build() {
}

class CreateCloudFrontOriginAccessIdentityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateCloudFrontOriginAccessIdentity", {})
    .n("CloudFrontClient", "CreateCloudFrontOriginAccessIdentityCommand")
    .f(void 0, void 0)
    .ser(se_CreateCloudFrontOriginAccessIdentityCommand)
    .de(de_CreateCloudFrontOriginAccessIdentityCommand)
    .build() {
}

class CreateConnectionGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateConnectionGroup", {})
    .n("CloudFrontClient", "CreateConnectionGroupCommand")
    .f(void 0, void 0)
    .ser(se_CreateConnectionGroupCommand)
    .de(de_CreateConnectionGroupCommand)
    .build() {
}

class CreateContinuousDeploymentPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateContinuousDeploymentPolicy", {})
    .n("CloudFrontClient", "CreateContinuousDeploymentPolicyCommand")
    .f(void 0, void 0)
    .ser(se_CreateContinuousDeploymentPolicyCommand)
    .de(de_CreateContinuousDeploymentPolicyCommand)
    .build() {
}

class CreateDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateDistribution", {})
    .n("CloudFrontClient", "CreateDistributionCommand")
    .f(CreateDistributionRequestFilterSensitiveLog, CreateDistributionResultFilterSensitiveLog)
    .ser(se_CreateDistributionCommand)
    .de(de_CreateDistributionCommand)
    .build() {
}

class CreateDistributionTenantCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateDistributionTenant", {})
    .n("CloudFrontClient", "CreateDistributionTenantCommand")
    .f(void 0, void 0)
    .ser(se_CreateDistributionTenantCommand)
    .de(de_CreateDistributionTenantCommand)
    .build() {
}

class CreateDistributionWithTagsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateDistributionWithTags", {})
    .n("CloudFrontClient", "CreateDistributionWithTagsCommand")
    .f(CreateDistributionWithTagsRequestFilterSensitiveLog, CreateDistributionWithTagsResultFilterSensitiveLog)
    .ser(se_CreateDistributionWithTagsCommand)
    .de(de_CreateDistributionWithTagsCommand)
    .build() {
}

class CreateFieldLevelEncryptionConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateFieldLevelEncryptionConfig", {})
    .n("CloudFrontClient", "CreateFieldLevelEncryptionConfigCommand")
    .f(void 0, void 0)
    .ser(se_CreateFieldLevelEncryptionConfigCommand)
    .de(de_CreateFieldLevelEncryptionConfigCommand)
    .build() {
}

class CreateFieldLevelEncryptionProfileCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateFieldLevelEncryptionProfile", {})
    .n("CloudFrontClient", "CreateFieldLevelEncryptionProfileCommand")
    .f(void 0, void 0)
    .ser(se_CreateFieldLevelEncryptionProfileCommand)
    .de(de_CreateFieldLevelEncryptionProfileCommand)
    .build() {
}

class CreateFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateFunction", {})
    .n("CloudFrontClient", "CreateFunctionCommand")
    .f(CreateFunctionRequestFilterSensitiveLog, void 0)
    .ser(se_CreateFunctionCommand)
    .de(de_CreateFunctionCommand)
    .build() {
}

class CreateInvalidationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateInvalidation", {})
    .n("CloudFrontClient", "CreateInvalidationCommand")
    .f(void 0, void 0)
    .ser(se_CreateInvalidationCommand)
    .de(de_CreateInvalidationCommand)
    .build() {
}

class CreateInvalidationForDistributionTenantCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateInvalidationForDistributionTenant", {})
    .n("CloudFrontClient", "CreateInvalidationForDistributionTenantCommand")
    .f(void 0, void 0)
    .ser(se_CreateInvalidationForDistributionTenantCommand)
    .de(de_CreateInvalidationForDistributionTenantCommand)
    .build() {
}

class CreateKeyGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateKeyGroup", {})
    .n("CloudFrontClient", "CreateKeyGroupCommand")
    .f(void 0, void 0)
    .ser(se_CreateKeyGroupCommand)
    .de(de_CreateKeyGroupCommand)
    .build() {
}

class CreateKeyValueStoreCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateKeyValueStore", {})
    .n("CloudFrontClient", "CreateKeyValueStoreCommand")
    .f(void 0, void 0)
    .ser(se_CreateKeyValueStoreCommand)
    .de(de_CreateKeyValueStoreCommand)
    .build() {
}

class CreateMonitoringSubscriptionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateMonitoringSubscription", {})
    .n("CloudFrontClient", "CreateMonitoringSubscriptionCommand")
    .f(void 0, void 0)
    .ser(se_CreateMonitoringSubscriptionCommand)
    .de(de_CreateMonitoringSubscriptionCommand)
    .build() {
}

class CreateOriginAccessControlCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateOriginAccessControl", {})
    .n("CloudFrontClient", "CreateOriginAccessControlCommand")
    .f(void 0, void 0)
    .ser(se_CreateOriginAccessControlCommand)
    .de(de_CreateOriginAccessControlCommand)
    .build() {
}

class CreateOriginRequestPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateOriginRequestPolicy", {})
    .n("CloudFrontClient", "CreateOriginRequestPolicyCommand")
    .f(void 0, void 0)
    .ser(se_CreateOriginRequestPolicyCommand)
    .de(de_CreateOriginRequestPolicyCommand)
    .build() {
}

class CreatePublicKeyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreatePublicKey", {})
    .n("CloudFrontClient", "CreatePublicKeyCommand")
    .f(void 0, void 0)
    .ser(se_CreatePublicKeyCommand)
    .de(de_CreatePublicKeyCommand)
    .build() {
}

class CreateRealtimeLogConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateRealtimeLogConfig", {})
    .n("CloudFrontClient", "CreateRealtimeLogConfigCommand")
    .f(void 0, void 0)
    .ser(se_CreateRealtimeLogConfigCommand)
    .de(de_CreateRealtimeLogConfigCommand)
    .build() {
}

class CreateResponseHeadersPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateResponseHeadersPolicy", {})
    .n("CloudFrontClient", "CreateResponseHeadersPolicyCommand")
    .f(void 0, void 0)
    .ser(se_CreateResponseHeadersPolicyCommand)
    .de(de_CreateResponseHeadersPolicyCommand)
    .build() {
}

class CreateStreamingDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateStreamingDistribution", {})
    .n("CloudFrontClient", "CreateStreamingDistributionCommand")
    .f(void 0, void 0)
    .ser(se_CreateStreamingDistributionCommand)
    .de(de_CreateStreamingDistributionCommand)
    .build() {
}

class CreateStreamingDistributionWithTagsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateStreamingDistributionWithTags", {})
    .n("CloudFrontClient", "CreateStreamingDistributionWithTagsCommand")
    .f(void 0, void 0)
    .ser(se_CreateStreamingDistributionWithTagsCommand)
    .de(de_CreateStreamingDistributionWithTagsCommand)
    .build() {
}

class CreateVpcOriginCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "CreateVpcOrigin", {})
    .n("CloudFrontClient", "CreateVpcOriginCommand")
    .f(void 0, void 0)
    .ser(se_CreateVpcOriginCommand)
    .de(de_CreateVpcOriginCommand)
    .build() {
}

class DeleteAnycastIpListCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteAnycastIpList", {})
    .n("CloudFrontClient", "DeleteAnycastIpListCommand")
    .f(void 0, void 0)
    .ser(se_DeleteAnycastIpListCommand)
    .de(de_DeleteAnycastIpListCommand)
    .build() {
}

class DeleteCachePolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteCachePolicy", {})
    .n("CloudFrontClient", "DeleteCachePolicyCommand")
    .f(void 0, void 0)
    .ser(se_DeleteCachePolicyCommand)
    .de(de_DeleteCachePolicyCommand)
    .build() {
}

class DeleteCloudFrontOriginAccessIdentityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteCloudFrontOriginAccessIdentity", {})
    .n("CloudFrontClient", "DeleteCloudFrontOriginAccessIdentityCommand")
    .f(void 0, void 0)
    .ser(se_DeleteCloudFrontOriginAccessIdentityCommand)
    .de(de_DeleteCloudFrontOriginAccessIdentityCommand)
    .build() {
}

class DeleteConnectionGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteConnectionGroup", {})
    .n("CloudFrontClient", "DeleteConnectionGroupCommand")
    .f(void 0, void 0)
    .ser(se_DeleteConnectionGroupCommand)
    .de(de_DeleteConnectionGroupCommand)
    .build() {
}

class DeleteContinuousDeploymentPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteContinuousDeploymentPolicy", {})
    .n("CloudFrontClient", "DeleteContinuousDeploymentPolicyCommand")
    .f(void 0, void 0)
    .ser(se_DeleteContinuousDeploymentPolicyCommand)
    .de(de_DeleteContinuousDeploymentPolicyCommand)
    .build() {
}

class DeleteDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteDistribution", {})
    .n("CloudFrontClient", "DeleteDistributionCommand")
    .f(void 0, void 0)
    .ser(se_DeleteDistributionCommand)
    .de(de_DeleteDistributionCommand)
    .build() {
}

class DeleteDistributionTenantCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteDistributionTenant", {})
    .n("CloudFrontClient", "DeleteDistributionTenantCommand")
    .f(void 0, void 0)
    .ser(se_DeleteDistributionTenantCommand)
    .de(de_DeleteDistributionTenantCommand)
    .build() {
}

class DeleteFieldLevelEncryptionConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteFieldLevelEncryptionConfig", {})
    .n("CloudFrontClient", "DeleteFieldLevelEncryptionConfigCommand")
    .f(void 0, void 0)
    .ser(se_DeleteFieldLevelEncryptionConfigCommand)
    .de(de_DeleteFieldLevelEncryptionConfigCommand)
    .build() {
}

class DeleteFieldLevelEncryptionProfileCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteFieldLevelEncryptionProfile", {})
    .n("CloudFrontClient", "DeleteFieldLevelEncryptionProfileCommand")
    .f(void 0, void 0)
    .ser(se_DeleteFieldLevelEncryptionProfileCommand)
    .de(de_DeleteFieldLevelEncryptionProfileCommand)
    .build() {
}

class DeleteFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteFunction", {})
    .n("CloudFrontClient", "DeleteFunctionCommand")
    .f(void 0, void 0)
    .ser(se_DeleteFunctionCommand)
    .de(de_DeleteFunctionCommand)
    .build() {
}

class DeleteKeyGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteKeyGroup", {})
    .n("CloudFrontClient", "DeleteKeyGroupCommand")
    .f(void 0, void 0)
    .ser(se_DeleteKeyGroupCommand)
    .de(de_DeleteKeyGroupCommand)
    .build() {
}

class DeleteKeyValueStoreCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteKeyValueStore", {})
    .n("CloudFrontClient", "DeleteKeyValueStoreCommand")
    .f(void 0, void 0)
    .ser(se_DeleteKeyValueStoreCommand)
    .de(de_DeleteKeyValueStoreCommand)
    .build() {
}

class DeleteMonitoringSubscriptionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteMonitoringSubscription", {})
    .n("CloudFrontClient", "DeleteMonitoringSubscriptionCommand")
    .f(void 0, void 0)
    .ser(se_DeleteMonitoringSubscriptionCommand)
    .de(de_DeleteMonitoringSubscriptionCommand)
    .build() {
}

class DeleteOriginAccessControlCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteOriginAccessControl", {})
    .n("CloudFrontClient", "DeleteOriginAccessControlCommand")
    .f(void 0, void 0)
    .ser(se_DeleteOriginAccessControlCommand)
    .de(de_DeleteOriginAccessControlCommand)
    .build() {
}

class DeleteOriginRequestPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteOriginRequestPolicy", {})
    .n("CloudFrontClient", "DeleteOriginRequestPolicyCommand")
    .f(void 0, void 0)
    .ser(se_DeleteOriginRequestPolicyCommand)
    .de(de_DeleteOriginRequestPolicyCommand)
    .build() {
}

class DeletePublicKeyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeletePublicKey", {})
    .n("CloudFrontClient", "DeletePublicKeyCommand")
    .f(void 0, void 0)
    .ser(se_DeletePublicKeyCommand)
    .de(de_DeletePublicKeyCommand)
    .build() {
}

class DeleteRealtimeLogConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteRealtimeLogConfig", {})
    .n("CloudFrontClient", "DeleteRealtimeLogConfigCommand")
    .f(void 0, void 0)
    .ser(se_DeleteRealtimeLogConfigCommand)
    .de(de_DeleteRealtimeLogConfigCommand)
    .build() {
}

class DeleteResponseHeadersPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteResponseHeadersPolicy", {})
    .n("CloudFrontClient", "DeleteResponseHeadersPolicyCommand")
    .f(void 0, void 0)
    .ser(se_DeleteResponseHeadersPolicyCommand)
    .de(de_DeleteResponseHeadersPolicyCommand)
    .build() {
}

class DeleteStreamingDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteStreamingDistribution", {})
    .n("CloudFrontClient", "DeleteStreamingDistributionCommand")
    .f(void 0, void 0)
    .ser(se_DeleteStreamingDistributionCommand)
    .de(de_DeleteStreamingDistributionCommand)
    .build() {
}

class DeleteVpcOriginCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DeleteVpcOrigin", {})
    .n("CloudFrontClient", "DeleteVpcOriginCommand")
    .f(void 0, void 0)
    .ser(se_DeleteVpcOriginCommand)
    .de(de_DeleteVpcOriginCommand)
    .build() {
}

class DescribeFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DescribeFunction", {})
    .n("CloudFrontClient", "DescribeFunctionCommand")
    .f(void 0, void 0)
    .ser(se_DescribeFunctionCommand)
    .de(de_DescribeFunctionCommand)
    .build() {
}

class DescribeKeyValueStoreCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DescribeKeyValueStore", {})
    .n("CloudFrontClient", "DescribeKeyValueStoreCommand")
    .f(void 0, void 0)
    .ser(se_DescribeKeyValueStoreCommand)
    .de(de_DescribeKeyValueStoreCommand)
    .build() {
}

class DisassociateDistributionTenantWebACLCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DisassociateDistributionTenantWebACL", {})
    .n("CloudFrontClient", "DisassociateDistributionTenantWebACLCommand")
    .f(void 0, void 0)
    .ser(se_DisassociateDistributionTenantWebACLCommand)
    .de(de_DisassociateDistributionTenantWebACLCommand)
    .build() {
}

class DisassociateDistributionWebACLCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "DisassociateDistributionWebACL", {})
    .n("CloudFrontClient", "DisassociateDistributionWebACLCommand")
    .f(void 0, void 0)
    .ser(se_DisassociateDistributionWebACLCommand)
    .de(de_DisassociateDistributionWebACLCommand)
    .build() {
}

class GetAnycastIpListCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetAnycastIpList", {})
    .n("CloudFrontClient", "GetAnycastIpListCommand")
    .f(void 0, void 0)
    .ser(se_GetAnycastIpListCommand)
    .de(de_GetAnycastIpListCommand)
    .build() {
}

class GetCachePolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetCachePolicy", {})
    .n("CloudFrontClient", "GetCachePolicyCommand")
    .f(void 0, void 0)
    .ser(se_GetCachePolicyCommand)
    .de(de_GetCachePolicyCommand)
    .build() {
}

class GetCachePolicyConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetCachePolicyConfig", {})
    .n("CloudFrontClient", "GetCachePolicyConfigCommand")
    .f(void 0, void 0)
    .ser(se_GetCachePolicyConfigCommand)
    .de(de_GetCachePolicyConfigCommand)
    .build() {
}

class GetCloudFrontOriginAccessIdentityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetCloudFrontOriginAccessIdentity", {})
    .n("CloudFrontClient", "GetCloudFrontOriginAccessIdentityCommand")
    .f(void 0, void 0)
    .ser(se_GetCloudFrontOriginAccessIdentityCommand)
    .de(de_GetCloudFrontOriginAccessIdentityCommand)
    .build() {
}

class GetCloudFrontOriginAccessIdentityConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetCloudFrontOriginAccessIdentityConfig", {})
    .n("CloudFrontClient", "GetCloudFrontOriginAccessIdentityConfigCommand")
    .f(void 0, void 0)
    .ser(se_GetCloudFrontOriginAccessIdentityConfigCommand)
    .de(de_GetCloudFrontOriginAccessIdentityConfigCommand)
    .build() {
}

class GetConnectionGroupByRoutingEndpointCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetConnectionGroupByRoutingEndpoint", {})
    .n("CloudFrontClient", "GetConnectionGroupByRoutingEndpointCommand")
    .f(void 0, void 0)
    .ser(se_GetConnectionGroupByRoutingEndpointCommand)
    .de(de_GetConnectionGroupByRoutingEndpointCommand)
    .build() {
}

class GetConnectionGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetConnectionGroup", {})
    .n("CloudFrontClient", "GetConnectionGroupCommand")
    .f(void 0, void 0)
    .ser(se_GetConnectionGroupCommand)
    .de(de_GetConnectionGroupCommand)
    .build() {
}

class GetContinuousDeploymentPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetContinuousDeploymentPolicy", {})
    .n("CloudFrontClient", "GetContinuousDeploymentPolicyCommand")
    .f(void 0, void 0)
    .ser(se_GetContinuousDeploymentPolicyCommand)
    .de(de_GetContinuousDeploymentPolicyCommand)
    .build() {
}

class GetContinuousDeploymentPolicyConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetContinuousDeploymentPolicyConfig", {})
    .n("CloudFrontClient", "GetContinuousDeploymentPolicyConfigCommand")
    .f(void 0, void 0)
    .ser(se_GetContinuousDeploymentPolicyConfigCommand)
    .de(de_GetContinuousDeploymentPolicyConfigCommand)
    .build() {
}

class GetDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetDistribution", {})
    .n("CloudFrontClient", "GetDistributionCommand")
    .f(void 0, GetDistributionResultFilterSensitiveLog)
    .ser(se_GetDistributionCommand)
    .de(de_GetDistributionCommand)
    .build() {
}

class GetDistributionConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetDistributionConfig", {})
    .n("CloudFrontClient", "GetDistributionConfigCommand")
    .f(void 0, GetDistributionConfigResultFilterSensitiveLog)
    .ser(se_GetDistributionConfigCommand)
    .de(de_GetDistributionConfigCommand)
    .build() {
}

class GetDistributionTenantByDomainCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetDistributionTenantByDomain", {})
    .n("CloudFrontClient", "GetDistributionTenantByDomainCommand")
    .f(void 0, void 0)
    .ser(se_GetDistributionTenantByDomainCommand)
    .de(de_GetDistributionTenantByDomainCommand)
    .build() {
}

class GetDistributionTenantCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetDistributionTenant", {})
    .n("CloudFrontClient", "GetDistributionTenantCommand")
    .f(void 0, void 0)
    .ser(se_GetDistributionTenantCommand)
    .de(de_GetDistributionTenantCommand)
    .build() {
}

class GetFieldLevelEncryptionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetFieldLevelEncryption", {})
    .n("CloudFrontClient", "GetFieldLevelEncryptionCommand")
    .f(void 0, void 0)
    .ser(se_GetFieldLevelEncryptionCommand)
    .de(de_GetFieldLevelEncryptionCommand)
    .build() {
}

class GetFieldLevelEncryptionConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetFieldLevelEncryptionConfig", {})
    .n("CloudFrontClient", "GetFieldLevelEncryptionConfigCommand")
    .f(void 0, void 0)
    .ser(se_GetFieldLevelEncryptionConfigCommand)
    .de(de_GetFieldLevelEncryptionConfigCommand)
    .build() {
}

class GetFieldLevelEncryptionProfileCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetFieldLevelEncryptionProfile", {})
    .n("CloudFrontClient", "GetFieldLevelEncryptionProfileCommand")
    .f(void 0, void 0)
    .ser(se_GetFieldLevelEncryptionProfileCommand)
    .de(de_GetFieldLevelEncryptionProfileCommand)
    .build() {
}

class GetFieldLevelEncryptionProfileConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetFieldLevelEncryptionProfileConfig", {})
    .n("CloudFrontClient", "GetFieldLevelEncryptionProfileConfigCommand")
    .f(void 0, void 0)
    .ser(se_GetFieldLevelEncryptionProfileConfigCommand)
    .de(de_GetFieldLevelEncryptionProfileConfigCommand)
    .build() {
}

class GetFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetFunction", {})
    .n("CloudFrontClient", "GetFunctionCommand")
    .f(void 0, GetFunctionResultFilterSensitiveLog)
    .ser(se_GetFunctionCommand)
    .de(de_GetFunctionCommand)
    .build() {
}

class GetInvalidationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetInvalidation", {})
    .n("CloudFrontClient", "GetInvalidationCommand")
    .f(void 0, void 0)
    .ser(se_GetInvalidationCommand)
    .de(de_GetInvalidationCommand)
    .build() {
}

class GetInvalidationForDistributionTenantCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetInvalidationForDistributionTenant", {})
    .n("CloudFrontClient", "GetInvalidationForDistributionTenantCommand")
    .f(void 0, void 0)
    .ser(se_GetInvalidationForDistributionTenantCommand)
    .de(de_GetInvalidationForDistributionTenantCommand)
    .build() {
}

class GetKeyGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetKeyGroup", {})
    .n("CloudFrontClient", "GetKeyGroupCommand")
    .f(void 0, void 0)
    .ser(se_GetKeyGroupCommand)
    .de(de_GetKeyGroupCommand)
    .build() {
}

class GetKeyGroupConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetKeyGroupConfig", {})
    .n("CloudFrontClient", "GetKeyGroupConfigCommand")
    .f(void 0, void 0)
    .ser(se_GetKeyGroupConfigCommand)
    .de(de_GetKeyGroupConfigCommand)
    .build() {
}

class GetManagedCertificateDetailsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetManagedCertificateDetails", {})
    .n("CloudFrontClient", "GetManagedCertificateDetailsCommand")
    .f(void 0, void 0)
    .ser(se_GetManagedCertificateDetailsCommand)
    .de(de_GetManagedCertificateDetailsCommand)
    .build() {
}

class GetMonitoringSubscriptionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetMonitoringSubscription", {})
    .n("CloudFrontClient", "GetMonitoringSubscriptionCommand")
    .f(void 0, void 0)
    .ser(se_GetMonitoringSubscriptionCommand)
    .de(de_GetMonitoringSubscriptionCommand)
    .build() {
}

class GetOriginAccessControlCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetOriginAccessControl", {})
    .n("CloudFrontClient", "GetOriginAccessControlCommand")
    .f(void 0, void 0)
    .ser(se_GetOriginAccessControlCommand)
    .de(de_GetOriginAccessControlCommand)
    .build() {
}

class GetOriginAccessControlConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetOriginAccessControlConfig", {})
    .n("CloudFrontClient", "GetOriginAccessControlConfigCommand")
    .f(void 0, void 0)
    .ser(se_GetOriginAccessControlConfigCommand)
    .de(de_GetOriginAccessControlConfigCommand)
    .build() {
}

class GetOriginRequestPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetOriginRequestPolicy", {})
    .n("CloudFrontClient", "GetOriginRequestPolicyCommand")
    .f(void 0, void 0)
    .ser(se_GetOriginRequestPolicyCommand)
    .de(de_GetOriginRequestPolicyCommand)
    .build() {
}

class GetOriginRequestPolicyConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetOriginRequestPolicyConfig", {})
    .n("CloudFrontClient", "GetOriginRequestPolicyConfigCommand")
    .f(void 0, void 0)
    .ser(se_GetOriginRequestPolicyConfigCommand)
    .de(de_GetOriginRequestPolicyConfigCommand)
    .build() {
}

class GetPublicKeyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetPublicKey", {})
    .n("CloudFrontClient", "GetPublicKeyCommand")
    .f(void 0, void 0)
    .ser(se_GetPublicKeyCommand)
    .de(de_GetPublicKeyCommand)
    .build() {
}

class GetPublicKeyConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetPublicKeyConfig", {})
    .n("CloudFrontClient", "GetPublicKeyConfigCommand")
    .f(void 0, void 0)
    .ser(se_GetPublicKeyConfigCommand)
    .de(de_GetPublicKeyConfigCommand)
    .build() {
}

class GetRealtimeLogConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetRealtimeLogConfig", {})
    .n("CloudFrontClient", "GetRealtimeLogConfigCommand")
    .f(void 0, void 0)
    .ser(se_GetRealtimeLogConfigCommand)
    .de(de_GetRealtimeLogConfigCommand)
    .build() {
}

class GetResponseHeadersPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetResponseHeadersPolicy", {})
    .n("CloudFrontClient", "GetResponseHeadersPolicyCommand")
    .f(void 0, void 0)
    .ser(se_GetResponseHeadersPolicyCommand)
    .de(de_GetResponseHeadersPolicyCommand)
    .build() {
}

class GetResponseHeadersPolicyConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetResponseHeadersPolicyConfig", {})
    .n("CloudFrontClient", "GetResponseHeadersPolicyConfigCommand")
    .f(void 0, void 0)
    .ser(se_GetResponseHeadersPolicyConfigCommand)
    .de(de_GetResponseHeadersPolicyConfigCommand)
    .build() {
}

class GetStreamingDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetStreamingDistribution", {})
    .n("CloudFrontClient", "GetStreamingDistributionCommand")
    .f(void 0, void 0)
    .ser(se_GetStreamingDistributionCommand)
    .de(de_GetStreamingDistributionCommand)
    .build() {
}

class GetStreamingDistributionConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetStreamingDistributionConfig", {})
    .n("CloudFrontClient", "GetStreamingDistributionConfigCommand")
    .f(void 0, void 0)
    .ser(se_GetStreamingDistributionConfigCommand)
    .de(de_GetStreamingDistributionConfigCommand)
    .build() {
}

class GetVpcOriginCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "GetVpcOrigin", {})
    .n("CloudFrontClient", "GetVpcOriginCommand")
    .f(void 0, void 0)
    .ser(se_GetVpcOriginCommand)
    .de(de_GetVpcOriginCommand)
    .build() {
}

class ListAnycastIpListsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListAnycastIpLists", {})
    .n("CloudFrontClient", "ListAnycastIpListsCommand")
    .f(void 0, void 0)
    .ser(se_ListAnycastIpListsCommand)
    .de(de_ListAnycastIpListsCommand)
    .build() {
}

class ListCachePoliciesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListCachePolicies", {})
    .n("CloudFrontClient", "ListCachePoliciesCommand")
    .f(void 0, void 0)
    .ser(se_ListCachePoliciesCommand)
    .de(de_ListCachePoliciesCommand)
    .build() {
}

class ListCloudFrontOriginAccessIdentitiesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListCloudFrontOriginAccessIdentities", {})
    .n("CloudFrontClient", "ListCloudFrontOriginAccessIdentitiesCommand")
    .f(void 0, void 0)
    .ser(se_ListCloudFrontOriginAccessIdentitiesCommand)
    .de(de_ListCloudFrontOriginAccessIdentitiesCommand)
    .build() {
}

class ListConflictingAliasesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListConflictingAliases", {})
    .n("CloudFrontClient", "ListConflictingAliasesCommand")
    .f(void 0, void 0)
    .ser(se_ListConflictingAliasesCommand)
    .de(de_ListConflictingAliasesCommand)
    .build() {
}

class ListConnectionGroupsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListConnectionGroups", {})
    .n("CloudFrontClient", "ListConnectionGroupsCommand")
    .f(void 0, void 0)
    .ser(se_ListConnectionGroupsCommand)
    .de(de_ListConnectionGroupsCommand)
    .build() {
}

class ListContinuousDeploymentPoliciesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListContinuousDeploymentPolicies", {})
    .n("CloudFrontClient", "ListContinuousDeploymentPoliciesCommand")
    .f(void 0, void 0)
    .ser(se_ListContinuousDeploymentPoliciesCommand)
    .de(de_ListContinuousDeploymentPoliciesCommand)
    .build() {
}

class ListDistributionsByAnycastIpListIdCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByAnycastIpListId", {})
    .n("CloudFrontClient", "ListDistributionsByAnycastIpListIdCommand")
    .f(void 0, ListDistributionsByAnycastIpListIdResultFilterSensitiveLog)
    .ser(se_ListDistributionsByAnycastIpListIdCommand)
    .de(de_ListDistributionsByAnycastIpListIdCommand)
    .build() {
}

class ListDistributionsByCachePolicyIdCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByCachePolicyId", {})
    .n("CloudFrontClient", "ListDistributionsByCachePolicyIdCommand")
    .f(void 0, void 0)
    .ser(se_ListDistributionsByCachePolicyIdCommand)
    .de(de_ListDistributionsByCachePolicyIdCommand)
    .build() {
}

class ListDistributionsByConnectionModeCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByConnectionMode", {})
    .n("CloudFrontClient", "ListDistributionsByConnectionModeCommand")
    .f(void 0, ListDistributionsByConnectionModeResultFilterSensitiveLog)
    .ser(se_ListDistributionsByConnectionModeCommand)
    .de(de_ListDistributionsByConnectionModeCommand)
    .build() {
}

class ListDistributionsByKeyGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByKeyGroup", {})
    .n("CloudFrontClient", "ListDistributionsByKeyGroupCommand")
    .f(void 0, void 0)
    .ser(se_ListDistributionsByKeyGroupCommand)
    .de(de_ListDistributionsByKeyGroupCommand)
    .build() {
}

class ListDistributionsByOriginRequestPolicyIdCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByOriginRequestPolicyId", {})
    .n("CloudFrontClient", "ListDistributionsByOriginRequestPolicyIdCommand")
    .f(void 0, void 0)
    .ser(se_ListDistributionsByOriginRequestPolicyIdCommand)
    .de(de_ListDistributionsByOriginRequestPolicyIdCommand)
    .build() {
}

class ListDistributionsByRealtimeLogConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByRealtimeLogConfig", {})
    .n("CloudFrontClient", "ListDistributionsByRealtimeLogConfigCommand")
    .f(void 0, ListDistributionsByRealtimeLogConfigResultFilterSensitiveLog)
    .ser(se_ListDistributionsByRealtimeLogConfigCommand)
    .de(de_ListDistributionsByRealtimeLogConfigCommand)
    .build() {
}

class ListDistributionsByResponseHeadersPolicyIdCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByResponseHeadersPolicyId", {})
    .n("CloudFrontClient", "ListDistributionsByResponseHeadersPolicyIdCommand")
    .f(void 0, void 0)
    .ser(se_ListDistributionsByResponseHeadersPolicyIdCommand)
    .de(de_ListDistributionsByResponseHeadersPolicyIdCommand)
    .build() {
}

class ListDistributionsByVpcOriginIdCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByVpcOriginId", {})
    .n("CloudFrontClient", "ListDistributionsByVpcOriginIdCommand")
    .f(void 0, void 0)
    .ser(se_ListDistributionsByVpcOriginIdCommand)
    .de(de_ListDistributionsByVpcOriginIdCommand)
    .build() {
}

class ListDistributionsByWebACLIdCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListDistributionsByWebACLId", {})
    .n("CloudFrontClient", "ListDistributionsByWebACLIdCommand")
    .f(void 0, ListDistributionsByWebACLIdResultFilterSensitiveLog)
    .ser(se_ListDistributionsByWebACLIdCommand)
    .de(de_ListDistributionsByWebACLIdCommand)
    .build() {
}

class ListDistributionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListDistributions", {})
    .n("CloudFrontClient", "ListDistributionsCommand")
    .f(void 0, ListDistributionsResultFilterSensitiveLog)
    .ser(se_ListDistributionsCommand)
    .de(de_ListDistributionsCommand)
    .build() {
}

class ListDistributionTenantsByCustomizationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListDistributionTenantsByCustomization", {})
    .n("CloudFrontClient", "ListDistributionTenantsByCustomizationCommand")
    .f(void 0, void 0)
    .ser(se_ListDistributionTenantsByCustomizationCommand)
    .de(de_ListDistributionTenantsByCustomizationCommand)
    .build() {
}

class ListDistributionTenantsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListDistributionTenants", {})
    .n("CloudFrontClient", "ListDistributionTenantsCommand")
    .f(void 0, void 0)
    .ser(se_ListDistributionTenantsCommand)
    .de(de_ListDistributionTenantsCommand)
    .build() {
}

class ListDomainConflictsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListDomainConflicts", {})
    .n("CloudFrontClient", "ListDomainConflictsCommand")
    .f(void 0, void 0)
    .ser(se_ListDomainConflictsCommand)
    .de(de_ListDomainConflictsCommand)
    .build() {
}

class ListFieldLevelEncryptionConfigsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListFieldLevelEncryptionConfigs", {})
    .n("CloudFrontClient", "ListFieldLevelEncryptionConfigsCommand")
    .f(void 0, void 0)
    .ser(se_ListFieldLevelEncryptionConfigsCommand)
    .de(de_ListFieldLevelEncryptionConfigsCommand)
    .build() {
}

class ListFieldLevelEncryptionProfilesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListFieldLevelEncryptionProfiles", {})
    .n("CloudFrontClient", "ListFieldLevelEncryptionProfilesCommand")
    .f(void 0, void 0)
    .ser(se_ListFieldLevelEncryptionProfilesCommand)
    .de(de_ListFieldLevelEncryptionProfilesCommand)
    .build() {
}

class ListFunctionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListFunctions", {})
    .n("CloudFrontClient", "ListFunctionsCommand")
    .f(void 0, void 0)
    .ser(se_ListFunctionsCommand)
    .de(de_ListFunctionsCommand)
    .build() {
}

class ListInvalidationsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListInvalidations", {})
    .n("CloudFrontClient", "ListInvalidationsCommand")
    .f(void 0, void 0)
    .ser(se_ListInvalidationsCommand)
    .de(de_ListInvalidationsCommand)
    .build() {
}

class ListInvalidationsForDistributionTenantCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListInvalidationsForDistributionTenant", {})
    .n("CloudFrontClient", "ListInvalidationsForDistributionTenantCommand")
    .f(void 0, void 0)
    .ser(se_ListInvalidationsForDistributionTenantCommand)
    .de(de_ListInvalidationsForDistributionTenantCommand)
    .build() {
}

class ListKeyGroupsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListKeyGroups", {})
    .n("CloudFrontClient", "ListKeyGroupsCommand")
    .f(void 0, void 0)
    .ser(se_ListKeyGroupsCommand)
    .de(de_ListKeyGroupsCommand)
    .build() {
}

class ListKeyValueStoresCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListKeyValueStores", {})
    .n("CloudFrontClient", "ListKeyValueStoresCommand")
    .f(void 0, void 0)
    .ser(se_ListKeyValueStoresCommand)
    .de(de_ListKeyValueStoresCommand)
    .build() {
}

class ListOriginAccessControlsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListOriginAccessControls", {})
    .n("CloudFrontClient", "ListOriginAccessControlsCommand")
    .f(void 0, void 0)
    .ser(se_ListOriginAccessControlsCommand)
    .de(de_ListOriginAccessControlsCommand)
    .build() {
}

class ListOriginRequestPoliciesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListOriginRequestPolicies", {})
    .n("CloudFrontClient", "ListOriginRequestPoliciesCommand")
    .f(void 0, void 0)
    .ser(se_ListOriginRequestPoliciesCommand)
    .de(de_ListOriginRequestPoliciesCommand)
    .build() {
}

class ListPublicKeysCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListPublicKeys", {})
    .n("CloudFrontClient", "ListPublicKeysCommand")
    .f(void 0, void 0)
    .ser(se_ListPublicKeysCommand)
    .de(de_ListPublicKeysCommand)
    .build() {
}

class ListRealtimeLogConfigsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListRealtimeLogConfigs", {})
    .n("CloudFrontClient", "ListRealtimeLogConfigsCommand")
    .f(void 0, void 0)
    .ser(se_ListRealtimeLogConfigsCommand)
    .de(de_ListRealtimeLogConfigsCommand)
    .build() {
}

class ListResponseHeadersPoliciesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListResponseHeadersPolicies", {})
    .n("CloudFrontClient", "ListResponseHeadersPoliciesCommand")
    .f(void 0, void 0)
    .ser(se_ListResponseHeadersPoliciesCommand)
    .de(de_ListResponseHeadersPoliciesCommand)
    .build() {
}

class ListStreamingDistributionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListStreamingDistributions", {})
    .n("CloudFrontClient", "ListStreamingDistributionsCommand")
    .f(void 0, void 0)
    .ser(se_ListStreamingDistributionsCommand)
    .de(de_ListStreamingDistributionsCommand)
    .build() {
}

class ListTagsForResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListTagsForResource", {})
    .n("CloudFrontClient", "ListTagsForResourceCommand")
    .f(void 0, void 0)
    .ser(se_ListTagsForResourceCommand)
    .de(de_ListTagsForResourceCommand)
    .build() {
}

class ListVpcOriginsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "ListVpcOrigins", {})
    .n("CloudFrontClient", "ListVpcOriginsCommand")
    .f(void 0, void 0)
    .ser(se_ListVpcOriginsCommand)
    .de(de_ListVpcOriginsCommand)
    .build() {
}

class PublishFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "PublishFunction", {})
    .n("CloudFrontClient", "PublishFunctionCommand")
    .f(void 0, void 0)
    .ser(se_PublishFunctionCommand)
    .de(de_PublishFunctionCommand)
    .build() {
}

class TagResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "TagResource", {})
    .n("CloudFrontClient", "TagResourceCommand")
    .f(void 0, void 0)
    .ser(se_TagResourceCommand)
    .de(de_TagResourceCommand)
    .build() {
}

class TestFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "TestFunction", {})
    .n("CloudFrontClient", "TestFunctionCommand")
    .f(TestFunctionRequestFilterSensitiveLog, TestFunctionResultFilterSensitiveLog)
    .ser(se_TestFunctionCommand)
    .de(de_TestFunctionCommand)
    .build() {
}

class UntagResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UntagResource", {})
    .n("CloudFrontClient", "UntagResourceCommand")
    .f(void 0, void 0)
    .ser(se_UntagResourceCommand)
    .de(de_UntagResourceCommand)
    .build() {
}

class UpdateCachePolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateCachePolicy", {})
    .n("CloudFrontClient", "UpdateCachePolicyCommand")
    .f(void 0, void 0)
    .ser(se_UpdateCachePolicyCommand)
    .de(de_UpdateCachePolicyCommand)
    .build() {
}

class UpdateCloudFrontOriginAccessIdentityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateCloudFrontOriginAccessIdentity", {})
    .n("CloudFrontClient", "UpdateCloudFrontOriginAccessIdentityCommand")
    .f(void 0, void 0)
    .ser(se_UpdateCloudFrontOriginAccessIdentityCommand)
    .de(de_UpdateCloudFrontOriginAccessIdentityCommand)
    .build() {
}

class UpdateConnectionGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateConnectionGroup", {})
    .n("CloudFrontClient", "UpdateConnectionGroupCommand")
    .f(void 0, void 0)
    .ser(se_UpdateConnectionGroupCommand)
    .de(de_UpdateConnectionGroupCommand)
    .build() {
}

class UpdateContinuousDeploymentPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateContinuousDeploymentPolicy", {})
    .n("CloudFrontClient", "UpdateContinuousDeploymentPolicyCommand")
    .f(void 0, void 0)
    .ser(se_UpdateContinuousDeploymentPolicyCommand)
    .de(de_UpdateContinuousDeploymentPolicyCommand)
    .build() {
}

class UpdateDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateDistribution", {})
    .n("CloudFrontClient", "UpdateDistributionCommand")
    .f(UpdateDistributionRequestFilterSensitiveLog, UpdateDistributionResultFilterSensitiveLog)
    .ser(se_UpdateDistributionCommand)
    .de(de_UpdateDistributionCommand)
    .build() {
}

class UpdateDistributionTenantCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateDistributionTenant", {})
    .n("CloudFrontClient", "UpdateDistributionTenantCommand")
    .f(void 0, void 0)
    .ser(se_UpdateDistributionTenantCommand)
    .de(de_UpdateDistributionTenantCommand)
    .build() {
}

class UpdateDistributionWithStagingConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateDistributionWithStagingConfig", {})
    .n("CloudFrontClient", "UpdateDistributionWithStagingConfigCommand")
    .f(void 0, UpdateDistributionWithStagingConfigResultFilterSensitiveLog)
    .ser(se_UpdateDistributionWithStagingConfigCommand)
    .de(de_UpdateDistributionWithStagingConfigCommand)
    .build() {
}

class UpdateDomainAssociationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateDomainAssociation", {})
    .n("CloudFrontClient", "UpdateDomainAssociationCommand")
    .f(void 0, void 0)
    .ser(se_UpdateDomainAssociationCommand)
    .de(de_UpdateDomainAssociationCommand)
    .build() {
}

class UpdateFieldLevelEncryptionConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateFieldLevelEncryptionConfig", {})
    .n("CloudFrontClient", "UpdateFieldLevelEncryptionConfigCommand")
    .f(void 0, void 0)
    .ser(se_UpdateFieldLevelEncryptionConfigCommand)
    .de(de_UpdateFieldLevelEncryptionConfigCommand)
    .build() {
}

class UpdateFieldLevelEncryptionProfileCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateFieldLevelEncryptionProfile", {})
    .n("CloudFrontClient", "UpdateFieldLevelEncryptionProfileCommand")
    .f(void 0, void 0)
    .ser(se_UpdateFieldLevelEncryptionProfileCommand)
    .de(de_UpdateFieldLevelEncryptionProfileCommand)
    .build() {
}

class UpdateFunctionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateFunction", {})
    .n("CloudFrontClient", "UpdateFunctionCommand")
    .f(UpdateFunctionRequestFilterSensitiveLog, void 0)
    .ser(se_UpdateFunctionCommand)
    .de(de_UpdateFunctionCommand)
    .build() {
}

class UpdateKeyGroupCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateKeyGroup", {})
    .n("CloudFrontClient", "UpdateKeyGroupCommand")
    .f(void 0, void 0)
    .ser(se_UpdateKeyGroupCommand)
    .de(de_UpdateKeyGroupCommand)
    .build() {
}

class UpdateKeyValueStoreCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateKeyValueStore", {})
    .n("CloudFrontClient", "UpdateKeyValueStoreCommand")
    .f(void 0, void 0)
    .ser(se_UpdateKeyValueStoreCommand)
    .de(de_UpdateKeyValueStoreCommand)
    .build() {
}

class UpdateOriginAccessControlCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateOriginAccessControl", {})
    .n("CloudFrontClient", "UpdateOriginAccessControlCommand")
    .f(void 0, void 0)
    .ser(se_UpdateOriginAccessControlCommand)
    .de(de_UpdateOriginAccessControlCommand)
    .build() {
}

class UpdateOriginRequestPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateOriginRequestPolicy", {})
    .n("CloudFrontClient", "UpdateOriginRequestPolicyCommand")
    .f(void 0, void 0)
    .ser(se_UpdateOriginRequestPolicyCommand)
    .de(de_UpdateOriginRequestPolicyCommand)
    .build() {
}

class UpdatePublicKeyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdatePublicKey", {})
    .n("CloudFrontClient", "UpdatePublicKeyCommand")
    .f(void 0, void 0)
    .ser(se_UpdatePublicKeyCommand)
    .de(de_UpdatePublicKeyCommand)
    .build() {
}

class UpdateRealtimeLogConfigCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateRealtimeLogConfig", {})
    .n("CloudFrontClient", "UpdateRealtimeLogConfigCommand")
    .f(void 0, void 0)
    .ser(se_UpdateRealtimeLogConfigCommand)
    .de(de_UpdateRealtimeLogConfigCommand)
    .build() {
}

class UpdateResponseHeadersPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateResponseHeadersPolicy", {})
    .n("CloudFrontClient", "UpdateResponseHeadersPolicyCommand")
    .f(void 0, void 0)
    .ser(se_UpdateResponseHeadersPolicyCommand)
    .de(de_UpdateResponseHeadersPolicyCommand)
    .build() {
}

class UpdateStreamingDistributionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateStreamingDistribution", {})
    .n("CloudFrontClient", "UpdateStreamingDistributionCommand")
    .f(void 0, void 0)
    .ser(se_UpdateStreamingDistributionCommand)
    .de(de_UpdateStreamingDistributionCommand)
    .build() {
}

class UpdateVpcOriginCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "UpdateVpcOrigin", {})
    .n("CloudFrontClient", "UpdateVpcOriginCommand")
    .f(void 0, void 0)
    .ser(se_UpdateVpcOriginCommand)
    .de(de_UpdateVpcOriginCommand)
    .build() {
}

class VerifyDnsConfigurationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [
        middlewareSerde.getSerdePlugin(config, this.serialize, this.deserialize),
        middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions()),
    ];
})
    .s("Cloudfront2020_05_31", "VerifyDnsConfiguration", {})
    .n("CloudFrontClient", "VerifyDnsConfigurationCommand")
    .f(void 0, void 0)
    .ser(se_VerifyDnsConfigurationCommand)
    .de(de_VerifyDnsConfigurationCommand)
    .build() {
}

const commands = {
    AssociateAliasCommand,
    AssociateDistributionTenantWebACLCommand,
    AssociateDistributionWebACLCommand,
    CopyDistributionCommand,
    CreateAnycastIpListCommand,
    CreateCachePolicyCommand,
    CreateCloudFrontOriginAccessIdentityCommand,
    CreateConnectionGroupCommand,
    CreateContinuousDeploymentPolicyCommand,
    CreateDistributionCommand,
    CreateDistributionTenantCommand,
    CreateDistributionWithTagsCommand,
    CreateFieldLevelEncryptionConfigCommand,
    CreateFieldLevelEncryptionProfileCommand,
    CreateFunctionCommand,
    CreateInvalidationCommand,
    CreateInvalidationForDistributionTenantCommand,
    CreateKeyGroupCommand,
    CreateKeyValueStoreCommand,
    CreateMonitoringSubscriptionCommand,
    CreateOriginAccessControlCommand,
    CreateOriginRequestPolicyCommand,
    CreatePublicKeyCommand,
    CreateRealtimeLogConfigCommand,
    CreateResponseHeadersPolicyCommand,
    CreateStreamingDistributionCommand,
    CreateStreamingDistributionWithTagsCommand,
    CreateVpcOriginCommand,
    DeleteAnycastIpListCommand,
    DeleteCachePolicyCommand,
    DeleteCloudFrontOriginAccessIdentityCommand,
    DeleteConnectionGroupCommand,
    DeleteContinuousDeploymentPolicyCommand,
    DeleteDistributionCommand,
    DeleteDistributionTenantCommand,
    DeleteFieldLevelEncryptionConfigCommand,
    DeleteFieldLevelEncryptionProfileCommand,
    DeleteFunctionCommand,
    DeleteKeyGroupCommand,
    DeleteKeyValueStoreCommand,
    DeleteMonitoringSubscriptionCommand,
    DeleteOriginAccessControlCommand,
    DeleteOriginRequestPolicyCommand,
    DeletePublicKeyCommand,
    DeleteRealtimeLogConfigCommand,
    DeleteResponseHeadersPolicyCommand,
    DeleteStreamingDistributionCommand,
    DeleteVpcOriginCommand,
    DescribeFunctionCommand,
    DescribeKeyValueStoreCommand,
    DisassociateDistributionTenantWebACLCommand,
    DisassociateDistributionWebACLCommand,
    GetAnycastIpListCommand,
    GetCachePolicyCommand,
    GetCachePolicyConfigCommand,
    GetCloudFrontOriginAccessIdentityCommand,
    GetCloudFrontOriginAccessIdentityConfigCommand,
    GetConnectionGroupCommand,
    GetConnectionGroupByRoutingEndpointCommand,
    GetContinuousDeploymentPolicyCommand,
    GetContinuousDeploymentPolicyConfigCommand,
    GetDistributionCommand,
    GetDistributionConfigCommand,
    GetDistributionTenantCommand,
    GetDistributionTenantByDomainCommand,
    GetFieldLevelEncryptionCommand,
    GetFieldLevelEncryptionConfigCommand,
    GetFieldLevelEncryptionProfileCommand,
    GetFieldLevelEncryptionProfileConfigCommand,
    GetFunctionCommand,
    GetInvalidationCommand,
    GetInvalidationForDistributionTenantCommand,
    GetKeyGroupCommand,
    GetKeyGroupConfigCommand,
    GetManagedCertificateDetailsCommand,
    GetMonitoringSubscriptionCommand,
    GetOriginAccessControlCommand,
    GetOriginAccessControlConfigCommand,
    GetOriginRequestPolicyCommand,
    GetOriginRequestPolicyConfigCommand,
    GetPublicKeyCommand,
    GetPublicKeyConfigCommand,
    GetRealtimeLogConfigCommand,
    GetResponseHeadersPolicyCommand,
    GetResponseHeadersPolicyConfigCommand,
    GetStreamingDistributionCommand,
    GetStreamingDistributionConfigCommand,
    GetVpcOriginCommand,
    ListAnycastIpListsCommand,
    ListCachePoliciesCommand,
    ListCloudFrontOriginAccessIdentitiesCommand,
    ListConflictingAliasesCommand,
    ListConnectionGroupsCommand,
    ListContinuousDeploymentPoliciesCommand,
    ListDistributionsCommand,
    ListDistributionsByAnycastIpListIdCommand,
    ListDistributionsByCachePolicyIdCommand,
    ListDistributionsByConnectionModeCommand,
    ListDistributionsByKeyGroupCommand,
    ListDistributionsByOriginRequestPolicyIdCommand,
    ListDistributionsByRealtimeLogConfigCommand,
    ListDistributionsByResponseHeadersPolicyIdCommand,
    ListDistributionsByVpcOriginIdCommand,
    ListDistributionsByWebACLIdCommand,
    ListDistributionTenantsCommand,
    ListDistributionTenantsByCustomizationCommand,
    ListDomainConflictsCommand,
    ListFieldLevelEncryptionConfigsCommand,
    ListFieldLevelEncryptionProfilesCommand,
    ListFunctionsCommand,
    ListInvalidationsCommand,
    ListInvalidationsForDistributionTenantCommand,
    ListKeyGroupsCommand,
    ListKeyValueStoresCommand,
    ListOriginAccessControlsCommand,
    ListOriginRequestPoliciesCommand,
    ListPublicKeysCommand,
    ListRealtimeLogConfigsCommand,
    ListResponseHeadersPoliciesCommand,
    ListStreamingDistributionsCommand,
    ListTagsForResourceCommand,
    ListVpcOriginsCommand,
    PublishFunctionCommand,
    TagResourceCommand,
    TestFunctionCommand,
    UntagResourceCommand,
    UpdateCachePolicyCommand,
    UpdateCloudFrontOriginAccessIdentityCommand,
    UpdateConnectionGroupCommand,
    UpdateContinuousDeploymentPolicyCommand,
    UpdateDistributionCommand,
    UpdateDistributionTenantCommand,
    UpdateDistributionWithStagingConfigCommand,
    UpdateDomainAssociationCommand,
    UpdateFieldLevelEncryptionConfigCommand,
    UpdateFieldLevelEncryptionProfileCommand,
    UpdateFunctionCommand,
    UpdateKeyGroupCommand,
    UpdateKeyValueStoreCommand,
    UpdateOriginAccessControlCommand,
    UpdateOriginRequestPolicyCommand,
    UpdatePublicKeyCommand,
    UpdateRealtimeLogConfigCommand,
    UpdateResponseHeadersPolicyCommand,
    UpdateStreamingDistributionCommand,
    UpdateVpcOriginCommand,
    VerifyDnsConfigurationCommand,
};
class CloudFront extends CloudFrontClient {
}
smithyClient.createAggregatedClient(commands, CloudFront);

const paginateListCloudFrontOriginAccessIdentities = core.createPaginator(CloudFrontClient, ListCloudFrontOriginAccessIdentitiesCommand, "Marker", "CloudFrontOriginAccessIdentityList.NextMarker", "MaxItems");

const paginateListConnectionGroups = core.createPaginator(CloudFrontClient, ListConnectionGroupsCommand, "Marker", "NextMarker", "MaxItems");

const paginateListDistributionTenantsByCustomization = core.createPaginator(CloudFrontClient, ListDistributionTenantsByCustomizationCommand, "Marker", "NextMarker", "MaxItems");

const paginateListDistributionTenants = core.createPaginator(CloudFrontClient, ListDistributionTenantsCommand, "Marker", "NextMarker", "MaxItems");

const paginateListDistributionsByConnectionMode = core.createPaginator(CloudFrontClient, ListDistributionsByConnectionModeCommand, "Marker", "DistributionList.NextMarker", "MaxItems");

const paginateListDistributions = core.createPaginator(CloudFrontClient, ListDistributionsCommand, "Marker", "DistributionList.NextMarker", "MaxItems");

const paginateListDomainConflicts = core.createPaginator(CloudFrontClient, ListDomainConflictsCommand, "Marker", "NextMarker", "MaxItems");

const paginateListInvalidationsForDistributionTenant = core.createPaginator(CloudFrontClient, ListInvalidationsForDistributionTenantCommand, "Marker", "InvalidationList.NextMarker", "MaxItems");

const paginateListInvalidations = core.createPaginator(CloudFrontClient, ListInvalidationsCommand, "Marker", "InvalidationList.NextMarker", "MaxItems");

const paginateListKeyValueStores = core.createPaginator(CloudFrontClient, ListKeyValueStoresCommand, "Marker", "KeyValueStoreList.NextMarker", "MaxItems");

const paginateListOriginAccessControls = core.createPaginator(CloudFrontClient, ListOriginAccessControlsCommand, "Marker", "OriginAccessControlList.NextMarker", "MaxItems");

const paginateListPublicKeys = core.createPaginator(CloudFrontClient, ListPublicKeysCommand, "Marker", "PublicKeyList.NextMarker", "MaxItems");

const paginateListStreamingDistributions = core.createPaginator(CloudFrontClient, ListStreamingDistributionsCommand, "Marker", "StreamingDistributionList.NextMarker", "MaxItems");

const checkState$3 = async (client, input) => {
    let reason;
    try {
        const result = await client.send(new GetDistributionCommand(input));
        reason = result;
        try {
            const returnComparator = () => {
                return result.Distribution.Status;
            };
            if (returnComparator() === "Deployed") {
                return { state: utilWaiter.WaiterState.SUCCESS, reason };
            }
        }
        catch (e) { }
    }
    catch (exception) {
        reason = exception;
    }
    return { state: utilWaiter.WaiterState.RETRY, reason };
};
const waitForDistributionDeployed = async (params, input) => {
    const serviceDefaults = { minDelay: 60, maxDelay: 2100 };
    return utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState$3);
};
const waitUntilDistributionDeployed = async (params, input) => {
    const serviceDefaults = { minDelay: 60, maxDelay: 2100 };
    const result = await utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState$3);
    return utilWaiter.checkExceptions(result);
};

const checkState$2 = async (client, input) => {
    let reason;
    try {
        const result = await client.send(new GetInvalidationCommand(input));
        reason = result;
        try {
            const returnComparator = () => {
                return result.Invalidation.Status;
            };
            if (returnComparator() === "Completed") {
                return { state: utilWaiter.WaiterState.SUCCESS, reason };
            }
        }
        catch (e) { }
    }
    catch (exception) {
        reason = exception;
    }
    return { state: utilWaiter.WaiterState.RETRY, reason };
};
const waitForInvalidationCompleted = async (params, input) => {
    const serviceDefaults = { minDelay: 20, maxDelay: 600 };
    return utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState$2);
};
const waitUntilInvalidationCompleted = async (params, input) => {
    const serviceDefaults = { minDelay: 20, maxDelay: 600 };
    const result = await utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState$2);
    return utilWaiter.checkExceptions(result);
};

const checkState$1 = async (client, input) => {
    let reason;
    try {
        const result = await client.send(new GetInvalidationForDistributionTenantCommand(input));
        reason = result;
        try {
            const returnComparator = () => {
                return result.Invalidation.Status;
            };
            if (returnComparator() === "Completed") {
                return { state: utilWaiter.WaiterState.SUCCESS, reason };
            }
        }
        catch (e) { }
    }
    catch (exception) {
        reason = exception;
    }
    return { state: utilWaiter.WaiterState.RETRY, reason };
};
const waitForInvalidationForDistributionTenantCompleted = async (params, input) => {
    const serviceDefaults = { minDelay: 20, maxDelay: 600 };
    return utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState$1);
};
const waitUntilInvalidationForDistributionTenantCompleted = async (params, input) => {
    const serviceDefaults = { minDelay: 20, maxDelay: 600 };
    const result = await utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState$1);
    return utilWaiter.checkExceptions(result);
};

const checkState = async (client, input) => {
    let reason;
    try {
        const result = await client.send(new GetStreamingDistributionCommand(input));
        reason = result;
        try {
            const returnComparator = () => {
                return result.StreamingDistribution.Status;
            };
            if (returnComparator() === "Deployed") {
                return { state: utilWaiter.WaiterState.SUCCESS, reason };
            }
        }
        catch (e) { }
    }
    catch (exception) {
        reason = exception;
    }
    return { state: utilWaiter.WaiterState.RETRY, reason };
};
const waitForStreamingDistributionDeployed = async (params, input) => {
    const serviceDefaults = { minDelay: 60, maxDelay: 1500 };
    return utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState);
};
const waitUntilStreamingDistributionDeployed = async (params, input) => {
    const serviceDefaults = { minDelay: 60, maxDelay: 1500 };
    const result = await utilWaiter.createWaiter({ ...serviceDefaults, ...params }, input, checkState);
    return utilWaiter.checkExceptions(result);
};

Object.defineProperty(exports, "$Command", {
    enumerable: true,
    get: function () { return smithyClient.Command; }
});
Object.defineProperty(exports, "__Client", {
    enumerable: true,
    get: function () { return smithyClient.Client; }
});
exports.AccessDenied = AccessDenied;
exports.AssociateAliasCommand = AssociateAliasCommand;
exports.AssociateDistributionTenantWebACLCommand = AssociateDistributionTenantWebACLCommand;
exports.AssociateDistributionWebACLCommand = AssociateDistributionWebACLCommand;
exports.BatchTooLarge = BatchTooLarge;
exports.CNAMEAlreadyExists = CNAMEAlreadyExists;
exports.CachePolicyAlreadyExists = CachePolicyAlreadyExists;
exports.CachePolicyCookieBehavior = CachePolicyCookieBehavior;
exports.CachePolicyHeaderBehavior = CachePolicyHeaderBehavior;
exports.CachePolicyInUse = CachePolicyInUse;
exports.CachePolicyQueryStringBehavior = CachePolicyQueryStringBehavior;
exports.CachePolicyType = CachePolicyType;
exports.CannotChangeImmutablePublicKeyFields = CannotChangeImmutablePublicKeyFields;
exports.CannotDeleteEntityWhileInUse = CannotDeleteEntityWhileInUse;
exports.CannotUpdateEntityWhileInUse = CannotUpdateEntityWhileInUse;
exports.CertificateSource = CertificateSource;
exports.CertificateTransparencyLoggingPreference = CertificateTransparencyLoggingPreference;
exports.CloudFront = CloudFront;
exports.CloudFrontClient = CloudFrontClient;
exports.CloudFrontOriginAccessIdentityAlreadyExists = CloudFrontOriginAccessIdentityAlreadyExists;
exports.CloudFrontOriginAccessIdentityInUse = CloudFrontOriginAccessIdentityInUse;
exports.CloudFrontServiceException = CloudFrontServiceException;
exports.ConnectionMode = ConnectionMode;
exports.ContinuousDeploymentPolicyAlreadyExists = ContinuousDeploymentPolicyAlreadyExists;
exports.ContinuousDeploymentPolicyInUse = ContinuousDeploymentPolicyInUse;
exports.ContinuousDeploymentPolicyType = ContinuousDeploymentPolicyType;
exports.CopyDistributionCommand = CopyDistributionCommand;
exports.CopyDistributionResultFilterSensitiveLog = CopyDistributionResultFilterSensitiveLog;
exports.CreateAnycastIpListCommand = CreateAnycastIpListCommand;
exports.CreateCachePolicyCommand = CreateCachePolicyCommand;
exports.CreateCloudFrontOriginAccessIdentityCommand = CreateCloudFrontOriginAccessIdentityCommand;
exports.CreateConnectionGroupCommand = CreateConnectionGroupCommand;
exports.CreateContinuousDeploymentPolicyCommand = CreateContinuousDeploymentPolicyCommand;
exports.CreateDistributionCommand = CreateDistributionCommand;
exports.CreateDistributionRequestFilterSensitiveLog = CreateDistributionRequestFilterSensitiveLog;
exports.CreateDistributionResultFilterSensitiveLog = CreateDistributionResultFilterSensitiveLog;
exports.CreateDistributionTenantCommand = CreateDistributionTenantCommand;
exports.CreateDistributionWithTagsCommand = CreateDistributionWithTagsCommand;
exports.CreateDistributionWithTagsRequestFilterSensitiveLog = CreateDistributionWithTagsRequestFilterSensitiveLog;
exports.CreateDistributionWithTagsResultFilterSensitiveLog = CreateDistributionWithTagsResultFilterSensitiveLog;
exports.CreateFieldLevelEncryptionConfigCommand = CreateFieldLevelEncryptionConfigCommand;
exports.CreateFieldLevelEncryptionProfileCommand = CreateFieldLevelEncryptionProfileCommand;
exports.CreateFunctionCommand = CreateFunctionCommand;
exports.CreateFunctionRequestFilterSensitiveLog = CreateFunctionRequestFilterSensitiveLog;
exports.CreateInvalidationCommand = CreateInvalidationCommand;
exports.CreateInvalidationForDistributionTenantCommand = CreateInvalidationForDistributionTenantCommand;
exports.CreateKeyGroupCommand = CreateKeyGroupCommand;
exports.CreateKeyValueStoreCommand = CreateKeyValueStoreCommand;
exports.CreateMonitoringSubscriptionCommand = CreateMonitoringSubscriptionCommand;
exports.CreateOriginAccessControlCommand = CreateOriginAccessControlCommand;
exports.CreateOriginRequestPolicyCommand = CreateOriginRequestPolicyCommand;
exports.CreatePublicKeyCommand = CreatePublicKeyCommand;
exports.CreateRealtimeLogConfigCommand = CreateRealtimeLogConfigCommand;
exports.CreateResponseHeadersPolicyCommand = CreateResponseHeadersPolicyCommand;
exports.CreateStreamingDistributionCommand = CreateStreamingDistributionCommand;
exports.CreateStreamingDistributionWithTagsCommand = CreateStreamingDistributionWithTagsCommand;
exports.CreateVpcOriginCommand = CreateVpcOriginCommand;
exports.CustomHeadersFilterSensitiveLog = CustomHeadersFilterSensitiveLog;
exports.CustomizationActionType = CustomizationActionType;
exports.DeleteAnycastIpListCommand = DeleteAnycastIpListCommand;
exports.DeleteCachePolicyCommand = DeleteCachePolicyCommand;
exports.DeleteCloudFrontOriginAccessIdentityCommand = DeleteCloudFrontOriginAccessIdentityCommand;
exports.DeleteConnectionGroupCommand = DeleteConnectionGroupCommand;
exports.DeleteContinuousDeploymentPolicyCommand = DeleteContinuousDeploymentPolicyCommand;
exports.DeleteDistributionCommand = DeleteDistributionCommand;
exports.DeleteDistributionTenantCommand = DeleteDistributionTenantCommand;
exports.DeleteFieldLevelEncryptionConfigCommand = DeleteFieldLevelEncryptionConfigCommand;
exports.DeleteFieldLevelEncryptionProfileCommand = DeleteFieldLevelEncryptionProfileCommand;
exports.DeleteFunctionCommand = DeleteFunctionCommand;
exports.DeleteKeyGroupCommand = DeleteKeyGroupCommand;
exports.DeleteKeyValueStoreCommand = DeleteKeyValueStoreCommand;
exports.DeleteMonitoringSubscriptionCommand = DeleteMonitoringSubscriptionCommand;
exports.DeleteOriginAccessControlCommand = DeleteOriginAccessControlCommand;
exports.DeleteOriginRequestPolicyCommand = DeleteOriginRequestPolicyCommand;
exports.DeletePublicKeyCommand = DeletePublicKeyCommand;
exports.DeleteRealtimeLogConfigCommand = DeleteRealtimeLogConfigCommand;
exports.DeleteResponseHeadersPolicyCommand = DeleteResponseHeadersPolicyCommand;
exports.DeleteStreamingDistributionCommand = DeleteStreamingDistributionCommand;
exports.DeleteVpcOriginCommand = DeleteVpcOriginCommand;
exports.DescribeFunctionCommand = DescribeFunctionCommand;
exports.DescribeKeyValueStoreCommand = DescribeKeyValueStoreCommand;
exports.DisassociateDistributionTenantWebACLCommand = DisassociateDistributionTenantWebACLCommand;
exports.DisassociateDistributionWebACLCommand = DisassociateDistributionWebACLCommand;
exports.DistributionAlreadyExists = DistributionAlreadyExists;
exports.DistributionConfigFilterSensitiveLog = DistributionConfigFilterSensitiveLog;
exports.DistributionConfigWithTagsFilterSensitiveLog = DistributionConfigWithTagsFilterSensitiveLog;
exports.DistributionFilterSensitiveLog = DistributionFilterSensitiveLog;
exports.DistributionListFilterSensitiveLog = DistributionListFilterSensitiveLog;
exports.DistributionNotDisabled = DistributionNotDisabled;
exports.DistributionResourceType = DistributionResourceType;
exports.DistributionSummaryFilterSensitiveLog = DistributionSummaryFilterSensitiveLog;
exports.DnsConfigurationStatus = DnsConfigurationStatus;
exports.DomainStatus = DomainStatus;
exports.EntityAlreadyExists = EntityAlreadyExists;
exports.EntityLimitExceeded = EntityLimitExceeded;
exports.EntityNotFound = EntityNotFound;
exports.EntitySizeLimitExceeded = EntitySizeLimitExceeded;
exports.EventType = EventType;
exports.FieldLevelEncryptionConfigAlreadyExists = FieldLevelEncryptionConfigAlreadyExists;
exports.FieldLevelEncryptionConfigInUse = FieldLevelEncryptionConfigInUse;
exports.FieldLevelEncryptionProfileAlreadyExists = FieldLevelEncryptionProfileAlreadyExists;
exports.FieldLevelEncryptionProfileInUse = FieldLevelEncryptionProfileInUse;
exports.FieldLevelEncryptionProfileSizeExceeded = FieldLevelEncryptionProfileSizeExceeded;
exports.Format = Format;
exports.FrameOptionsList = FrameOptionsList;
exports.FunctionAlreadyExists = FunctionAlreadyExists;
exports.FunctionInUse = FunctionInUse;
exports.FunctionRuntime = FunctionRuntime;
exports.FunctionSizeLimitExceeded = FunctionSizeLimitExceeded;
exports.FunctionStage = FunctionStage;
exports.GeoRestrictionType = GeoRestrictionType;
exports.GetAnycastIpListCommand = GetAnycastIpListCommand;
exports.GetCachePolicyCommand = GetCachePolicyCommand;
exports.GetCachePolicyConfigCommand = GetCachePolicyConfigCommand;
exports.GetCloudFrontOriginAccessIdentityCommand = GetCloudFrontOriginAccessIdentityCommand;
exports.GetCloudFrontOriginAccessIdentityConfigCommand = GetCloudFrontOriginAccessIdentityConfigCommand;
exports.GetConnectionGroupByRoutingEndpointCommand = GetConnectionGroupByRoutingEndpointCommand;
exports.GetConnectionGroupCommand = GetConnectionGroupCommand;
exports.GetContinuousDeploymentPolicyCommand = GetContinuousDeploymentPolicyCommand;
exports.GetContinuousDeploymentPolicyConfigCommand = GetContinuousDeploymentPolicyConfigCommand;
exports.GetDistributionCommand = GetDistributionCommand;
exports.GetDistributionConfigCommand = GetDistributionConfigCommand;
exports.GetDistributionConfigResultFilterSensitiveLog = GetDistributionConfigResultFilterSensitiveLog;
exports.GetDistributionResultFilterSensitiveLog = GetDistributionResultFilterSensitiveLog;
exports.GetDistributionTenantByDomainCommand = GetDistributionTenantByDomainCommand;
exports.GetDistributionTenantCommand = GetDistributionTenantCommand;
exports.GetFieldLevelEncryptionCommand = GetFieldLevelEncryptionCommand;
exports.GetFieldLevelEncryptionConfigCommand = GetFieldLevelEncryptionConfigCommand;
exports.GetFieldLevelEncryptionProfileCommand = GetFieldLevelEncryptionProfileCommand;
exports.GetFieldLevelEncryptionProfileConfigCommand = GetFieldLevelEncryptionProfileConfigCommand;
exports.GetFunctionCommand = GetFunctionCommand;
exports.GetFunctionResultFilterSensitiveLog = GetFunctionResultFilterSensitiveLog;
exports.GetInvalidationCommand = GetInvalidationCommand;
exports.GetInvalidationForDistributionTenantCommand = GetInvalidationForDistributionTenantCommand;
exports.GetKeyGroupCommand = GetKeyGroupCommand;
exports.GetKeyGroupConfigCommand = GetKeyGroupConfigCommand;
exports.GetManagedCertificateDetailsCommand = GetManagedCertificateDetailsCommand;
exports.GetMonitoringSubscriptionCommand = GetMonitoringSubscriptionCommand;
exports.GetOriginAccessControlCommand = GetOriginAccessControlCommand;
exports.GetOriginAccessControlConfigCommand = GetOriginAccessControlConfigCommand;
exports.GetOriginRequestPolicyCommand = GetOriginRequestPolicyCommand;
exports.GetOriginRequestPolicyConfigCommand = GetOriginRequestPolicyConfigCommand;
exports.GetPublicKeyCommand = GetPublicKeyCommand;
exports.GetPublicKeyConfigCommand = GetPublicKeyConfigCommand;
exports.GetRealtimeLogConfigCommand = GetRealtimeLogConfigCommand;
exports.GetResponseHeadersPolicyCommand = GetResponseHeadersPolicyCommand;
exports.GetResponseHeadersPolicyConfigCommand = GetResponseHeadersPolicyConfigCommand;
exports.GetStreamingDistributionCommand = GetStreamingDistributionCommand;
exports.GetStreamingDistributionConfigCommand = GetStreamingDistributionConfigCommand;
exports.GetVpcOriginCommand = GetVpcOriginCommand;
exports.HttpVersion = HttpVersion;
exports.ICPRecordalStatus = ICPRecordalStatus;
exports.IllegalDelete = IllegalDelete;
exports.IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior = IllegalFieldLevelEncryptionConfigAssociationWithCacheBehavior;
exports.IllegalOriginAccessConfiguration = IllegalOriginAccessConfiguration;
exports.IllegalUpdate = IllegalUpdate;
exports.ImportSourceType = ImportSourceType;
exports.InconsistentQuantities = InconsistentQuantities;
exports.InvalidArgument = InvalidArgument;
exports.InvalidAssociation = InvalidAssociation;
exports.InvalidDefaultRootObject = InvalidDefaultRootObject;
exports.InvalidDomainNameForOriginAccessControl = InvalidDomainNameForOriginAccessControl;
exports.InvalidErrorCode = InvalidErrorCode;
exports.InvalidForwardCookies = InvalidForwardCookies;
exports.InvalidFunctionAssociation = InvalidFunctionAssociation;
exports.InvalidGeoRestrictionParameter = InvalidGeoRestrictionParameter;
exports.InvalidHeadersForS3Origin = InvalidHeadersForS3Origin;
exports.InvalidIfMatchVersion = InvalidIfMatchVersion;
exports.InvalidLambdaFunctionAssociation = InvalidLambdaFunctionAssociation;
exports.InvalidLocationCode = InvalidLocationCode;
exports.InvalidMinimumProtocolVersion = InvalidMinimumProtocolVersion;
exports.InvalidOrigin = InvalidOrigin;
exports.InvalidOriginAccessControl = InvalidOriginAccessControl;
exports.InvalidOriginAccessIdentity = InvalidOriginAccessIdentity;
exports.InvalidOriginKeepaliveTimeout = InvalidOriginKeepaliveTimeout;
exports.InvalidOriginReadTimeout = InvalidOriginReadTimeout;
exports.InvalidProtocolSettings = InvalidProtocolSettings;
exports.InvalidQueryStringParameters = InvalidQueryStringParameters;
exports.InvalidRelativePath = InvalidRelativePath;
exports.InvalidRequiredProtocol = InvalidRequiredProtocol;
exports.InvalidResponseCode = InvalidResponseCode;
exports.InvalidTTLOrder = InvalidTTLOrder;
exports.InvalidTagging = InvalidTagging;
exports.InvalidViewerCertificate = InvalidViewerCertificate;
exports.InvalidWebACLId = InvalidWebACLId;
exports.IpAddressType = IpAddressType;
exports.ItemSelection = ItemSelection;
exports.KeyGroupAlreadyExists = KeyGroupAlreadyExists;
exports.ListAnycastIpListsCommand = ListAnycastIpListsCommand;
exports.ListCachePoliciesCommand = ListCachePoliciesCommand;
exports.ListCloudFrontOriginAccessIdentitiesCommand = ListCloudFrontOriginAccessIdentitiesCommand;
exports.ListConflictingAliasesCommand = ListConflictingAliasesCommand;
exports.ListConnectionGroupsCommand = ListConnectionGroupsCommand;
exports.ListContinuousDeploymentPoliciesCommand = ListContinuousDeploymentPoliciesCommand;
exports.ListDistributionTenantsByCustomizationCommand = ListDistributionTenantsByCustomizationCommand;
exports.ListDistributionTenantsCommand = ListDistributionTenantsCommand;
exports.ListDistributionsByAnycastIpListIdCommand = ListDistributionsByAnycastIpListIdCommand;
exports.ListDistributionsByAnycastIpListIdResultFilterSensitiveLog = ListDistributionsByAnycastIpListIdResultFilterSensitiveLog;
exports.ListDistributionsByCachePolicyIdCommand = ListDistributionsByCachePolicyIdCommand;
exports.ListDistributionsByConnectionModeCommand = ListDistributionsByConnectionModeCommand;
exports.ListDistributionsByConnectionModeResultFilterSensitiveLog = ListDistributionsByConnectionModeResultFilterSensitiveLog;
exports.ListDistributionsByKeyGroupCommand = ListDistributionsByKeyGroupCommand;
exports.ListDistributionsByOriginRequestPolicyIdCommand = ListDistributionsByOriginRequestPolicyIdCommand;
exports.ListDistributionsByRealtimeLogConfigCommand = ListDistributionsByRealtimeLogConfigCommand;
exports.ListDistributionsByRealtimeLogConfigResultFilterSensitiveLog = ListDistributionsByRealtimeLogConfigResultFilterSensitiveLog;
exports.ListDistributionsByResponseHeadersPolicyIdCommand = ListDistributionsByResponseHeadersPolicyIdCommand;
exports.ListDistributionsByVpcOriginIdCommand = ListDistributionsByVpcOriginIdCommand;
exports.ListDistributionsByWebACLIdCommand = ListDistributionsByWebACLIdCommand;
exports.ListDistributionsByWebACLIdResultFilterSensitiveLog = ListDistributionsByWebACLIdResultFilterSensitiveLog;
exports.ListDistributionsCommand = ListDistributionsCommand;
exports.ListDistributionsResultFilterSensitiveLog = ListDistributionsResultFilterSensitiveLog;
exports.ListDomainConflictsCommand = ListDomainConflictsCommand;
exports.ListFieldLevelEncryptionConfigsCommand = ListFieldLevelEncryptionConfigsCommand;
exports.ListFieldLevelEncryptionProfilesCommand = ListFieldLevelEncryptionProfilesCommand;
exports.ListFunctionsCommand = ListFunctionsCommand;
exports.ListInvalidationsCommand = ListInvalidationsCommand;
exports.ListInvalidationsForDistributionTenantCommand = ListInvalidationsForDistributionTenantCommand;
exports.ListKeyGroupsCommand = ListKeyGroupsCommand;
exports.ListKeyValueStoresCommand = ListKeyValueStoresCommand;
exports.ListOriginAccessControlsCommand = ListOriginAccessControlsCommand;
exports.ListOriginRequestPoliciesCommand = ListOriginRequestPoliciesCommand;
exports.ListPublicKeysCommand = ListPublicKeysCommand;
exports.ListRealtimeLogConfigsCommand = ListRealtimeLogConfigsCommand;
exports.ListResponseHeadersPoliciesCommand = ListResponseHeadersPoliciesCommand;
exports.ListStreamingDistributionsCommand = ListStreamingDistributionsCommand;
exports.ListTagsForResourceCommand = ListTagsForResourceCommand;
exports.ListVpcOriginsCommand = ListVpcOriginsCommand;
exports.ManagedCertificateStatus = ManagedCertificateStatus;
exports.Method = Method;
exports.MinimumProtocolVersion = MinimumProtocolVersion;
exports.MissingBody = MissingBody;
exports.MonitoringSubscriptionAlreadyExists = MonitoringSubscriptionAlreadyExists;
exports.NoSuchCachePolicy = NoSuchCachePolicy;
exports.NoSuchCloudFrontOriginAccessIdentity = NoSuchCloudFrontOriginAccessIdentity;
exports.NoSuchContinuousDeploymentPolicy = NoSuchContinuousDeploymentPolicy;
exports.NoSuchDistribution = NoSuchDistribution;
exports.NoSuchFieldLevelEncryptionConfig = NoSuchFieldLevelEncryptionConfig;
exports.NoSuchFieldLevelEncryptionProfile = NoSuchFieldLevelEncryptionProfile;
exports.NoSuchFunctionExists = NoSuchFunctionExists;
exports.NoSuchInvalidation = NoSuchInvalidation;
exports.NoSuchMonitoringSubscription = NoSuchMonitoringSubscription;
exports.NoSuchOrigin = NoSuchOrigin;
exports.NoSuchOriginAccessControl = NoSuchOriginAccessControl;
exports.NoSuchOriginRequestPolicy = NoSuchOriginRequestPolicy;
exports.NoSuchPublicKey = NoSuchPublicKey;
exports.NoSuchRealtimeLogConfig = NoSuchRealtimeLogConfig;
exports.NoSuchResource = NoSuchResource;
exports.NoSuchResponseHeadersPolicy = NoSuchResponseHeadersPolicy;
exports.NoSuchStreamingDistribution = NoSuchStreamingDistribution;
exports.OriginAccessControlAlreadyExists = OriginAccessControlAlreadyExists;
exports.OriginAccessControlInUse = OriginAccessControlInUse;
exports.OriginAccessControlOriginTypes = OriginAccessControlOriginTypes;
exports.OriginAccessControlSigningBehaviors = OriginAccessControlSigningBehaviors;
exports.OriginAccessControlSigningProtocols = OriginAccessControlSigningProtocols;
exports.OriginCustomHeaderFilterSensitiveLog = OriginCustomHeaderFilterSensitiveLog;
exports.OriginFilterSensitiveLog = OriginFilterSensitiveLog;
exports.OriginGroupSelectionCriteria = OriginGroupSelectionCriteria;
exports.OriginProtocolPolicy = OriginProtocolPolicy;
exports.OriginRequestPolicyAlreadyExists = OriginRequestPolicyAlreadyExists;
exports.OriginRequestPolicyCookieBehavior = OriginRequestPolicyCookieBehavior;
exports.OriginRequestPolicyHeaderBehavior = OriginRequestPolicyHeaderBehavior;
exports.OriginRequestPolicyInUse = OriginRequestPolicyInUse;
exports.OriginRequestPolicyQueryStringBehavior = OriginRequestPolicyQueryStringBehavior;
exports.OriginRequestPolicyType = OriginRequestPolicyType;
exports.OriginsFilterSensitiveLog = OriginsFilterSensitiveLog;
exports.ParameterDefinitionFilterSensitiveLog = ParameterDefinitionFilterSensitiveLog;
exports.ParameterDefinitionSchemaFilterSensitiveLog = ParameterDefinitionSchemaFilterSensitiveLog;
exports.PreconditionFailed = PreconditionFailed;
exports.PriceClass = PriceClass;
exports.PublicKeyAlreadyExists = PublicKeyAlreadyExists;
exports.PublicKeyInUse = PublicKeyInUse;
exports.PublishFunctionCommand = PublishFunctionCommand;
exports.QueryArgProfileEmpty = QueryArgProfileEmpty;
exports.RealtimeLogConfigAlreadyExists = RealtimeLogConfigAlreadyExists;
exports.RealtimeLogConfigInUse = RealtimeLogConfigInUse;
exports.RealtimeLogConfigOwnerMismatch = RealtimeLogConfigOwnerMismatch;
exports.RealtimeMetricsSubscriptionStatus = RealtimeMetricsSubscriptionStatus;
exports.ReferrerPolicyList = ReferrerPolicyList;
exports.ResourceInUse = ResourceInUse;
exports.ResourceNotDisabled = ResourceNotDisabled;
exports.ResponseHeadersPolicyAccessControlAllowMethodsValues = ResponseHeadersPolicyAccessControlAllowMethodsValues;
exports.ResponseHeadersPolicyAlreadyExists = ResponseHeadersPolicyAlreadyExists;
exports.ResponseHeadersPolicyInUse = ResponseHeadersPolicyInUse;
exports.ResponseHeadersPolicyType = ResponseHeadersPolicyType;
exports.SSLSupportMethod = SSLSupportMethod;
exports.SslProtocol = SslProtocol;
exports.StagingDistributionInUse = StagingDistributionInUse;
exports.StreamingDistributionAlreadyExists = StreamingDistributionAlreadyExists;
exports.StreamingDistributionNotDisabled = StreamingDistributionNotDisabled;
exports.StringSchemaConfigFilterSensitiveLog = StringSchemaConfigFilterSensitiveLog;
exports.TagResourceCommand = TagResourceCommand;
exports.TenantConfigFilterSensitiveLog = TenantConfigFilterSensitiveLog;
exports.TestFunctionCommand = TestFunctionCommand;
exports.TestFunctionFailed = TestFunctionFailed;
exports.TestFunctionRequestFilterSensitiveLog = TestFunctionRequestFilterSensitiveLog;
exports.TestFunctionResultFilterSensitiveLog = TestFunctionResultFilterSensitiveLog;
exports.TestResultFilterSensitiveLog = TestResultFilterSensitiveLog;
exports.TooLongCSPInResponseHeadersPolicy = TooLongCSPInResponseHeadersPolicy;
exports.TooManyCacheBehaviors = TooManyCacheBehaviors;
exports.TooManyCachePolicies = TooManyCachePolicies;
exports.TooManyCertificates = TooManyCertificates;
exports.TooManyCloudFrontOriginAccessIdentities = TooManyCloudFrontOriginAccessIdentities;
exports.TooManyContinuousDeploymentPolicies = TooManyContinuousDeploymentPolicies;
exports.TooManyCookieNamesInWhiteList = TooManyCookieNamesInWhiteList;
exports.TooManyCookiesInCachePolicy = TooManyCookiesInCachePolicy;
exports.TooManyCookiesInOriginRequestPolicy = TooManyCookiesInOriginRequestPolicy;
exports.TooManyCustomHeadersInResponseHeadersPolicy = TooManyCustomHeadersInResponseHeadersPolicy;
exports.TooManyDistributionCNAMEs = TooManyDistributionCNAMEs;
exports.TooManyDistributions = TooManyDistributions;
exports.TooManyDistributionsAssociatedToCachePolicy = TooManyDistributionsAssociatedToCachePolicy;
exports.TooManyDistributionsAssociatedToFieldLevelEncryptionConfig = TooManyDistributionsAssociatedToFieldLevelEncryptionConfig;
exports.TooManyDistributionsAssociatedToKeyGroup = TooManyDistributionsAssociatedToKeyGroup;
exports.TooManyDistributionsAssociatedToOriginAccessControl = TooManyDistributionsAssociatedToOriginAccessControl;
exports.TooManyDistributionsAssociatedToOriginRequestPolicy = TooManyDistributionsAssociatedToOriginRequestPolicy;
exports.TooManyDistributionsAssociatedToResponseHeadersPolicy = TooManyDistributionsAssociatedToResponseHeadersPolicy;
exports.TooManyDistributionsWithFunctionAssociations = TooManyDistributionsWithFunctionAssociations;
exports.TooManyDistributionsWithLambdaAssociations = TooManyDistributionsWithLambdaAssociations;
exports.TooManyDistributionsWithSingleFunctionARN = TooManyDistributionsWithSingleFunctionARN;
exports.TooManyFieldLevelEncryptionConfigs = TooManyFieldLevelEncryptionConfigs;
exports.TooManyFieldLevelEncryptionContentTypeProfiles = TooManyFieldLevelEncryptionContentTypeProfiles;
exports.TooManyFieldLevelEncryptionEncryptionEntities = TooManyFieldLevelEncryptionEncryptionEntities;
exports.TooManyFieldLevelEncryptionFieldPatterns = TooManyFieldLevelEncryptionFieldPatterns;
exports.TooManyFieldLevelEncryptionProfiles = TooManyFieldLevelEncryptionProfiles;
exports.TooManyFieldLevelEncryptionQueryArgProfiles = TooManyFieldLevelEncryptionQueryArgProfiles;
exports.TooManyFunctionAssociations = TooManyFunctionAssociations;
exports.TooManyFunctions = TooManyFunctions;
exports.TooManyHeadersInCachePolicy = TooManyHeadersInCachePolicy;
exports.TooManyHeadersInForwardedValues = TooManyHeadersInForwardedValues;
exports.TooManyHeadersInOriginRequestPolicy = TooManyHeadersInOriginRequestPolicy;
exports.TooManyInvalidationsInProgress = TooManyInvalidationsInProgress;
exports.TooManyKeyGroups = TooManyKeyGroups;
exports.TooManyKeyGroupsAssociatedToDistribution = TooManyKeyGroupsAssociatedToDistribution;
exports.TooManyLambdaFunctionAssociations = TooManyLambdaFunctionAssociations;
exports.TooManyOriginAccessControls = TooManyOriginAccessControls;
exports.TooManyOriginCustomHeaders = TooManyOriginCustomHeaders;
exports.TooManyOriginGroupsPerDistribution = TooManyOriginGroupsPerDistribution;
exports.TooManyOriginRequestPolicies = TooManyOriginRequestPolicies;
exports.TooManyOrigins = TooManyOrigins;
exports.TooManyPublicKeys = TooManyPublicKeys;
exports.TooManyPublicKeysInKeyGroup = TooManyPublicKeysInKeyGroup;
exports.TooManyQueryStringParameters = TooManyQueryStringParameters;
exports.TooManyQueryStringsInCachePolicy = TooManyQueryStringsInCachePolicy;
exports.TooManyQueryStringsInOriginRequestPolicy = TooManyQueryStringsInOriginRequestPolicy;
exports.TooManyRealtimeLogConfigs = TooManyRealtimeLogConfigs;
exports.TooManyRemoveHeadersInResponseHeadersPolicy = TooManyRemoveHeadersInResponseHeadersPolicy;
exports.TooManyResponseHeadersPolicies = TooManyResponseHeadersPolicies;
exports.TooManyStreamingDistributionCNAMEs = TooManyStreamingDistributionCNAMEs;
exports.TooManyStreamingDistributions = TooManyStreamingDistributions;
exports.TooManyTrustedSigners = TooManyTrustedSigners;
exports.TrustedKeyGroupDoesNotExist = TrustedKeyGroupDoesNotExist;
exports.TrustedSignerDoesNotExist = TrustedSignerDoesNotExist;
exports.UnsupportedOperation = UnsupportedOperation;
exports.UntagResourceCommand = UntagResourceCommand;
exports.UpdateCachePolicyCommand = UpdateCachePolicyCommand;
exports.UpdateCloudFrontOriginAccessIdentityCommand = UpdateCloudFrontOriginAccessIdentityCommand;
exports.UpdateConnectionGroupCommand = UpdateConnectionGroupCommand;
exports.UpdateContinuousDeploymentPolicyCommand = UpdateContinuousDeploymentPolicyCommand;
exports.UpdateDistributionCommand = UpdateDistributionCommand;
exports.UpdateDistributionRequestFilterSensitiveLog = UpdateDistributionRequestFilterSensitiveLog;
exports.UpdateDistributionResultFilterSensitiveLog = UpdateDistributionResultFilterSensitiveLog;
exports.UpdateDistributionTenantCommand = UpdateDistributionTenantCommand;
exports.UpdateDistributionWithStagingConfigCommand = UpdateDistributionWithStagingConfigCommand;
exports.UpdateDistributionWithStagingConfigResultFilterSensitiveLog = UpdateDistributionWithStagingConfigResultFilterSensitiveLog;
exports.UpdateDomainAssociationCommand = UpdateDomainAssociationCommand;
exports.UpdateFieldLevelEncryptionConfigCommand = UpdateFieldLevelEncryptionConfigCommand;
exports.UpdateFieldLevelEncryptionProfileCommand = UpdateFieldLevelEncryptionProfileCommand;
exports.UpdateFunctionCommand = UpdateFunctionCommand;
exports.UpdateFunctionRequestFilterSensitiveLog = UpdateFunctionRequestFilterSensitiveLog;
exports.UpdateKeyGroupCommand = UpdateKeyGroupCommand;
exports.UpdateKeyValueStoreCommand = UpdateKeyValueStoreCommand;
exports.UpdateOriginAccessControlCommand = UpdateOriginAccessControlCommand;
exports.UpdateOriginRequestPolicyCommand = UpdateOriginRequestPolicyCommand;
exports.UpdatePublicKeyCommand = UpdatePublicKeyCommand;
exports.UpdateRealtimeLogConfigCommand = UpdateRealtimeLogConfigCommand;
exports.UpdateResponseHeadersPolicyCommand = UpdateResponseHeadersPolicyCommand;
exports.UpdateStreamingDistributionCommand = UpdateStreamingDistributionCommand;
exports.UpdateVpcOriginCommand = UpdateVpcOriginCommand;
exports.ValidationTokenHost = ValidationTokenHost;
exports.VerifyDnsConfigurationCommand = VerifyDnsConfigurationCommand;
exports.ViewerProtocolPolicy = ViewerProtocolPolicy;
exports.paginateListCloudFrontOriginAccessIdentities = paginateListCloudFrontOriginAccessIdentities;
exports.paginateListConnectionGroups = paginateListConnectionGroups;
exports.paginateListDistributionTenants = paginateListDistributionTenants;
exports.paginateListDistributionTenantsByCustomization = paginateListDistributionTenantsByCustomization;
exports.paginateListDistributions = paginateListDistributions;
exports.paginateListDistributionsByConnectionMode = paginateListDistributionsByConnectionMode;
exports.paginateListDomainConflicts = paginateListDomainConflicts;
exports.paginateListInvalidations = paginateListInvalidations;
exports.paginateListInvalidationsForDistributionTenant = paginateListInvalidationsForDistributionTenant;
exports.paginateListKeyValueStores = paginateListKeyValueStores;
exports.paginateListOriginAccessControls = paginateListOriginAccessControls;
exports.paginateListPublicKeys = paginateListPublicKeys;
exports.paginateListStreamingDistributions = paginateListStreamingDistributions;
exports.waitForDistributionDeployed = waitForDistributionDeployed;
exports.waitForInvalidationCompleted = waitForInvalidationCompleted;
exports.waitForInvalidationForDistributionTenantCompleted = waitForInvalidationForDistributionTenantCompleted;
exports.waitForStreamingDistributionDeployed = waitForStreamingDistributionDeployed;
exports.waitUntilDistributionDeployed = waitUntilDistributionDeployed;
exports.waitUntilInvalidationCompleted = waitUntilInvalidationCompleted;
exports.waitUntilInvalidationForDistributionTenantCompleted = waitUntilInvalidationForDistributionTenantCompleted;
exports.waitUntilStreamingDistributionDeployed = waitUntilStreamingDistributionDeployed;
