/**
 * All state for the Draggable in one place.
 *
 * This avoids rerenders (caused by unbatched state updates),
 * but also keeps state logic together.
 */
import type { DraggableLocation, DragStart, DragUpdate, DroppableId, MovementMode } from 'react-beautiful-dnd';
import type { DroppableRegistryEntry } from '../drag-drop-context/droppable-registry';
import type { DraggableDimensions } from '../hooks/use-captured-dimensions';
import type { Action } from '../internal-types';
type DraggableIdleState = {
    type: 'idle';
    draggingOver: null;
};
export type PointerLocation = {
    initial: {
        input: {
            clientX: number;
            clientY: number;
        };
    };
    current: {
        input: {
            clientX: number;
            clientY: number;
        };
    };
};
export type DraggablePreviewOffset = {
    x: number;
    y: number;
};
/**
 * The state of a draggable that is currently being dragged.
 * It does not have a clone.
 */
type DraggableDraggingState = {
    type: 'dragging';
    draggingOver: DroppableId | null;
    start: DraggableLocation;
    location: PointerLocation | null;
    draggableId: string;
    mode: MovementMode;
    previewOffset: DraggablePreviewOffset;
};
/**
 * The state of a draggable that is currently hiding,
 * because it its clone is being rendered instead.
 */
type DraggableHidingState = {
    type: 'hiding';
    draggingOver: null;
    mode: MovementMode;
};
export type DraggableState = DraggableIdleState | DraggableDraggingState | DraggableHidingState;
type UpdateKeyboardPayload = {
    update: DragUpdate;
    draggableDimensions: DraggableDimensions | null;
    droppable: DroppableRegistryEntry | null;
    placeholderRect: DOMRect | null;
    dropIndicatorRect: DOMRect | null;
};
export type DraggableAction = Action<'START_POINTER_DRAG', {
    start: DragStart;
}> | Action<'START_KEYBOARD_DRAG', {
    start: DragStart;
    draggableDimensions: DraggableDimensions;
    droppable: DroppableRegistryEntry;
}>
/**
 * Updates the drag state.
 *
 * This loosely corresponds to an `onDragUpdate` from `react-beautiful-dnd`.
 */
 | Action<'UPDATE_DRAG', {
    update: DragUpdate;
}>
/**
 * Updates the preview based on pointer location
 */
 | Action<'UPDATE_POINTER_PREVIEW', {
    pointerLocation: PointerLocation;
}> | Action<'UPDATE_KEYBOARD_PREVIEW', UpdateKeyboardPayload> | Action<'DROP'>
/**
 * These events are specifically for draggables that `shouldRenderCloneWhileDragging`.
 * They have a distinct hiding state.
 */
 | Action<'START_HIDING', {
    mode: MovementMode;
}> | Action<'STOP_HIDING'>;
export declare const idleState: DraggableIdleState;
export declare function reducer(state: DraggableState, action: DraggableAction): DraggableState;
export {};
