/*
 * Copyright (C) 2008 Martin Willi
 * Hochschule fuer Technik Rapperswil
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * $Id: rng.h 3619 2008-03-19 14:02:52Z martin $
 */
 
/**
 * @defgroup rng rng
 * @{ @ingroup crypto
 */

#ifndef RNG_H_
#define RNG_H_

typedef enum rng_quality_t rng_quality_t;
typedef struct rng_t rng_t;

#include <library.h>

/**
 * Quality of generated random bytes.
 */
enum rng_quality_t {
	/** weak randomness, usable for nonces, IVs */
	RNG_WEAK,
	/** stronger randomness, usable for session keys */
	RNG_STRONG,
	/** real random, key material */
	RNG_REAL,
};

/**
 * enum name for rng_quality_t.
 */
extern enum_name_t *rng_quality_names;

/**
 * Generic interface for random number generators.
 */
struct rng_t {

	/**
	 * Generates random bytes and writes them in the buffer.
	 *
	 * @param len		number of bytes to get
	 * @param buffer	pointer where the generated bytes will be written
	 */
	void (*get_bytes) (rng_t *this, u_int len, u_int8_t *buffer);
	
	/**
	 * Generates random bytes and allocate space for them.
	 * 
	 * @param len		number of bytes to get
	 * @param chunk		chunk which will hold generated bytes
	 */
	void (*allocate_bytes) (rng_t *this, u_int len, chunk_t *chunk);
	
	/**
	 * Destroys a rng object.
	 */
	void (*destroy) (rng_t *this);
};

#endif /*RNG_H_ @} */
