/*************************************************************************/
/* module:          Encoder source file                                  */
/* file:            xltenc.c                                             */
/* target system:   All                                                  */
/* target OS:       All                                                  */   
/*************************************************************************/

/*
 * Copyright Notice
 * Copyright (c) Ericsson, IBM, Lotus, Matsushita Communication 
 * Industrial Co., LTD,Motorola, Nokia, Palm, Inc., Psion, 
 * Starfish Software (2001).
 * All Rights Reserved.
 * Implementation of all or part of any Specification may require 
 * licenses under third party intellectual property rights, 
 * including without limitation, patent rights (such a third party 
 * may or may not be a Supporter). The Sponsors of the Specification 
 * are not responsible and shall not be held responsible in any 
 * manner for identifying or failing to identify any or all such 
 * third party intellectual property rights.
 * 
 * THIS DOCUMENT AND THE INFORMATION CONTAINED HEREIN ARE PROVIDED 
 * ON AN "AS IS" BASIS WITHOUT WARRANTY OF ANY KIND AND ERICSSON, IBM, 
 * LOTUS, MATSUSHITA COMMUNICATION INDUSTRIAL CO. LTD, MOTOROLA, 
 * NOKIA, PALM INC., PSION, STARFISH SOFTWARE AND ALL OTHER SYNCML 
 * SPONSORS DISCLAIM ALL WARRANTIES, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO ANY WARRANTY THAT THE USE OF THE INFORMATION 
 * HEREIN WILL NOT INFRINGE ANY RIGHTS OR ANY IMPLIED WARRANTIES OF 
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT 
 * SHALL ERICSSON, IBM, LOTUS, MATSUSHITA COMMUNICATION INDUSTRIAL CO., 
 * LTD, MOTOROLA, NOKIA, PALM INC., PSION, STARFISH SOFTWARE OR ANY 
 * OTHER SYNCML SPONSOR BE LIABLE TO ANY PARTY FOR ANY LOSS OF 
 * PROFITS, LOSS OF BUSINESS, LOSS OF USE OF DATA, INTERRUPTION OF 
 * BUSINESS, OR FOR DIRECT, INDIRECT, SPECIAL OR EXEMPLARY, INCIDENTAL, 
 * PUNITIVE OR CONSEQUENTIAL DAMAGES OF ANY KIND IN CONNECTION WITH 
 * THIS DOCUMENT OR THE INFORMATION CONTAINED HEREIN, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH LOSS OR DAMAGE.
 * 
 * The above notice and this paragraph must be included on all copies 
 * of this document that are made.
 * 
 */

/*************************************************************************/
/*  Definitions                                                          */
/*************************************************************************/
#include <xltenc.h>
#ifdef __SML_WBXML__
  #include "xltencwbxml.h"
#endif
#ifdef __SML_XML__
  #include "xltencxml.h"
#endif
#include "xlttags.h"
#include <libstr.h>
#include <libmem.h>
#include <smldtd.h>

Ret_t getTNbyPE(SmlProtoElement_t, XltTagID_t*);
Ret_t xltEncBlock(XltTagID_t, XltRO_t, const VoidPtr_t, SmlEncoding_t, BufferMgmtPtr_t, SmlPcdataExtension_t);
Ret_t xltEncList(XltListType_t, XltRO_t, VoidPtr_t, SmlEncoding_t, BufferMgmtPtr_t, SmlPcdataExtension_t);
Ret_t xltGenerateTag(XltTagID_t, XltTagType_t, SmlEncoding_t, BufferMgmtPtr_t, SmlPcdataExtension_t);
#ifdef __META_EXT__
Ret_t xltBuildExtention(SmlPcdataExtension_t, XltRO_t, VoidPtr_t, SmlEncoding_t, BufferMgmtPtr_t);
#endif

typedef struct PEEnc_s
{
    XltTagID_t     tagid;
    SmlProtoElement_t type;
} PEEnc_t, *PEEncPtr_t;

PEEncPtr_t getPEEncTable(void);

PEEncPtr_t getPEEncTable(void)
{ 
  PEEncPtr_t _tmpPEEncPtr;
  PEEnc_t static PE[] =
  {
    { TN_ADD,          SML_PE_ADD            },
    { TN_ALERT,        SML_PE_ALERT          },
#ifndef __SML_LITE__    
    { TN_ATOMIC,       SML_PE_ATOMIC_START   },
    { TN_ATOMIC_END,   SML_PE_ATOMIC_END     },
#endif    
    { TN_COPY,         SML_PE_COPY           },
    { TN_DELETE,       SML_PE_DELETE         },
    { TN_EXEC,         SML_PE_EXEC           },
    { TN_GET,          SML_PE_GET            },
    { TN_MAP,          SML_PE_MAP            },
    { TN_PUT,          SML_PE_PUT            },
    { TN_RESULTS,      SML_PE_RESULTS        },
#ifndef __SML_LITE__        
    { TN_SEARCH,       SML_PE_SEARCH         },    
    { TN_SEQUENCE,     SML_PE_SEQUENCE_START },
    { TN_SEQUENCE_END, SML_PE_SEQUENCE_END },    
#endif    
    { TN_STATUS,       SML_PE_STATUS         },
    { TN_SYNC,         SML_PE_SYNC_START     },
    { TN_SYNC_END,     SML_PE_SYNC_END       },    
    { TN_REPLACE,      SML_PE_REPLACE        },
    { TN_UNDEF,        SML_PE_UNDEF          }
  };
  _tmpPEEncPtr = (PEEncPtr_t)smlLibMalloc(sizeof(PE));
  if (_tmpPEEncPtr == NULL) return NULL;
  smlLibMemcpy(_tmpPEEncPtr, &PE, sizeof(PE));
  return _tmpPEEncPtr;  
}

Ret_t getTNbyPE(SmlProtoElement_t pE, XltTagID_t *tagID)
{    
    int i = 0; 
    PEEncPtr_t pPETbl = getPEEncTable();
    if (pPETbl == NULL)
    {
      smlLibFree(pPETbl);
      return SML_ERR_NOT_ENOUGH_SPACE;
    }
    while (((pPETbl+i)->type) != SML_PE_UNDEF)
    {
      if (((pPETbl+i)->type) == pE)
      {
      	*tagID = (pPETbl+i)->tagid;
      	smlLibFree(pPETbl);
        return SML_ERR_OK;
      }    
      i++;
    }        
   	smlLibFree(pPETbl);
    *tagID = TN_UNDEF;
    return SML_ERR_XLT_INVAL_PROTO_ELEM;
}


/**
 * FUNCTION: smlXltEncInit
 *
 * Initializes an XML buffer; Creates XML code for the SyncHdr
 * and appends it to the buffer.
 * Returns 0 if operation was successful.
 *
 * PRE-Condition:   no memory should be allocated for ppEncoder (should be NULL)
 *                  pHeader has to contain a valid SyncHdr structure
 *                  pBufEnd must point to the end of the (WB)XML buffer
 *                  ppBufPos has to be initialized to the start point of the
 *                  (WB)XML buffer.
 *                  
 *
 * POST-Condition:  After the function call ppBufPos points to the
 *                  first free byte in the buffer behind the (WB)XML document
 *
 * IN:              enc, the encoding constant (SML_WBXML or SML_XML)
 *                  pHeader, the SyncML header structure
 *                  pBufEnd, pointer to the end of the buffer to write on
 * 
 * IN/OUT:          ppBufPos, current position of the bufferpointer
 *                  ppEncoder, the encoder object       
 *
 * RETURN:          shows error codes of function, 
 *                  0, if OK
 *                  Possible Error Codes:
 *                  SML_ERR_XLT_MISSING_CONT            
 *                  SML_ERR_XLT_BUF_ERR                 
 *                  SML_ERR_XLT_INVAL_ELEM_TYPE         
 *                  SML_ERR_XLT_INVAL_LIST_TYPE         
 *                  SML_ERR_XLT_INVAL_TAG_TYPE          
 *                  SML_ERR_XLT_ENC_UNK	               
 *                  SML_ERR_XLT_INVAL_PROTO_ELEM
 */
Ret_t xltEncInit(SmlEncoding_t enc,
                 const SmlSyncHdrPtr_t pHeader,
                 const MemPtr_t pBufEnd,
                 MemPtr_t *ppBufPos,
                 XltEncoderPtr_t *ppEncoder)
{
  // Return variable
  Ret_t _err;

  XltEncoderPtr_t _pEncoder;

  //Structure containing buffer pointers, length and written bytes
  BufferMgmtPtr_t _pBufMgr;

#ifdef __SML_WBXML__
  MemByte_t _stablen =  0x1D; //XLT_STABLEN;
  MemByte_t _wbxmlver = XLT_WBXMLVER;
  MemByte_t _charset = XLT_CHARSET;
  MemByte_t _pubident1 = XLT_PUBIDENT1;
  MemByte_t _pubident2 = XLT_PUBIDENT2;
#endif

#ifdef __SML_XML__
  MemPtr_t _tmpStr;
  MemPtr_t _xmlver = (MemPtr_t)XML_VERSION;
  MemPtr_t _xmlenc = (MemPtr_t)XML_ENCODING;
  MemByte_t _begpar = XML_BEGPAR;
  MemByte_t _endpar = XML_ENDPAR;
#endif

  MemByte_t _tmp = 0x00;

  if ((_pEncoder = (XltEncoderPtr_t)smlLibMalloc(sizeof(XltEncoder_t))) == NULL) return SML_ERR_NOT_ENOUGH_SPACE;

  if ((_pBufMgr = (BufferMgmtPtr_t)smlLibMalloc(sizeof(BufferMgmt_t))) == NULL) return SML_ERR_NOT_ENOUGH_SPACE; 

  //set the encoding
  _pEncoder->enc = enc;

#ifdef __SML_WBXML__
  _pEncoder->cur_codepage = XLT_DEFAULTCODEPAGE;
  _pEncoder->last_codepage = XLT_DEFAULTCODEPAGE;
#endif

  _pBufMgr->smlXltBufferP = *ppBufPos;
  _pBufMgr->smlXltBufferLen = pBufEnd - *ppBufPos;
  _pBufMgr->smlXltStoreBufP = _pBufMgr->smlXltBufferP;
  _pBufMgr->smlXltWrittenBytes = 0;  
  _pBufMgr->smlCurCodepage = _pEncoder->cur_codepage;
  _pBufMgr->smlLastCodepage = _pEncoder->last_codepage;

  switch (enc)
  {

#ifdef __SML_WBXML__
    case SML_WBXML:
    {

      // Set the WBXML Header Values
      // WBXML Version
      if ((_err = wbxmlWriteTypeToBuffer((MemPtr_t)(&_wbxmlver), TAG, 1, _pBufMgr)) != SML_ERR_OK) break;
      // Public Idetifier - default unknown
      if ((_err = wbxmlWriteTypeToBuffer((MemPtr_t)(&_pubident1), TAG, 1, _pBufMgr)) != SML_ERR_OK) break;
      if ((_err = wbxmlWriteTypeToBuffer((MemPtr_t)(&_pubident2), TAG, 1, _pBufMgr)) != SML_ERR_OK) break;
      // Character set - not yet implemented
      if ((_err = wbxmlWriteTypeToBuffer((MemPtr_t)(&_charset), TAG, 1, _pBufMgr)) != SML_ERR_OK) break;
      // Sting table length -  not yet implemented
      if ((_err = wbxmlWriteTypeToBuffer((MemPtr_t)(&_stablen), TAG, 1, _pBufMgr)) != SML_ERR_OK) break;

      if ((_err = xltAddToBuffer((MemPtr_t) "-//SYNCML//DTD SyncML 1.0//EN" , 29, _pBufMgr)) != SML_ERR_OK) break;      
      break;
    }
#endif

#ifdef __SML_XML__
    case SML_XML:
    {    
           
      if ((_err = xltAddToBuffer((MemPtr_t)(&_begpar), 1, _pBufMgr)) != SML_ERR_OK) break;
      _tmpStr = (MemPtr_t)"?xml version=\"";
      if ((_err = xltAddToBuffer(_tmpStr, smlLibStrlen((String_t)_tmpStr), _pBufMgr)) != SML_ERR_OK) break;
      _tmpStr = _xmlver;
      if ((_err = xltAddToBuffer(_tmpStr, smlLibStrlen((String_t)_tmpStr), _pBufMgr)) != SML_ERR_OK) break;
      _tmpStr = (MemPtr_t)"\" encoding=\"";
      if ((_err = xltAddToBuffer(_tmpStr, smlLibStrlen((String_t)_tmpStr), _pBufMgr)) != SML_ERR_OK) break;
      _tmpStr = _xmlenc;
      if ((_err = xltAddToBuffer(_tmpStr, smlLibStrlen((String_t)_tmpStr), _pBufMgr)) != SML_ERR_OK) break;
      _tmpStr = (MemPtr_t)"\"?";
      if ((_err = xltAddToBuffer(_tmpStr, smlLibStrlen((String_t)_tmpStr), _pBufMgr)) != SML_ERR_OK) break;
      if ((_err = xltAddToBuffer((MemPtr_t)(&_endpar), 1, _pBufMgr)) != SML_ERR_OK) break;

      break;
    }
#endif

    default:
    {
      _err = SML_ERR_XLT_ENC_UNK;
    }
  }
  if (_err != SML_ERR_OK)
  {
     smlLibFree(_pBufMgr);
     smlLibFree(_pEncoder);
     return _err;
  }

  // SyncML Tag   
  if ((_err = xltGenerateTag(TN_SYNCML_START, TT_BEG, enc, _pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK)
  {
    smlLibFree(_pBufMgr);
    smlLibFree(_pEncoder);
    return _err;
  }

  // Generate SmlSyncHdr
  if ((_err = xltEncBlock(TN_SYNCHDR, REQUIRED, pHeader, enc, _pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK)
  {
    smlLibFree(_pBufMgr);
    smlLibFree(_pEncoder);
    return _err;
  }

  // SyncBody Tag 
  if ((_err = xltGenerateTag(TN_SYNCBODY, TT_BEG, enc, _pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK)
  {
    smlLibFree(_pBufMgr);
    smlLibFree(_pEncoder);
    return _err;
  }

  _pEncoder->cur_codepage = _pBufMgr->smlCurCodepage;
  _pEncoder->last_codepage = _pBufMgr->smlLastCodepage;

  *ppBufPos = _pBufMgr->smlXltBufferP;

  smlLibFree(_pBufMgr);

  _pEncoder->final = 0;

  *ppEncoder = (XltEncoderPtr_t)_pEncoder;

  return SML_ERR_OK;
}

/**
 * FUNCTION: smlXltEncAppend
 *
 * Generates XML code and appends it to the XML buffer.
 *
 * PRE-Condition:   pEncoder holds the initialized encoder structure.
 *                  the initialization takes place in the xltEncAppend function
 *                  pContent has to contain a valid content structure structure
 *                  pBufEnd must point to the end of the (WB)XML buffer
 *                  ppBufPos has to be initialized to the start point of the
 *                  (WB)XML buffer.
 *                  
 *
 * POST-Condition:  After the function call ppBufPos points to the
 *                  first free byte in the buffer behind the (WB)XML document
 *
 * IN:              pEncoder, the encoder object
 *                  pe, the protocol element (PE_ADD, ...)    
 *                  pBufEnd, pointer to the end of the buffer to write on
 *                  pContent, the content to append to the SyncML document
 * 
 * IN/OUT:          ppBufPos, current position of the bufferpointer
 * 
 * RETURN:          shows error codes of function, 
 *                  0, if OK
 *                  Possible Error Codes:
 *                  SML_ERR_XLT_MISSING_CONT            
 *                  SML_ERR_XLT_BUF_ERR                 
 *                  SML_ERR_XLT_INVAL_ELEM_TYPE         
 *                  SML_ERR_XLT_INVAL_LIST_TYPE         
 *                  SML_ERR_XLT_INVAL_TAG_TYPE          
 *                  SML_ERR_XLT_ENC_UNK	               
 *                  SML_ERR_XLT_INVAL_PROTO_ELEM
 */
Ret_t xltEncAppend(const XltEncoderPtr_t pEncoder, 
                   SmlProtoElement_t pe, 
                   const MemPtr_t pBufEnd,
                   const VoidPtr_t pContent,
                   MemPtr_t *ppBufPos)
{
  // Return variable
  Ret_t _err;

  XltTagID_t tagID = TN_UNDEF;

  // encoding type
  SmlEncoding_t _enc;

  //Structure containing buffer pointers, length and written bytes
  BufferMgmtPtr_t _pBufMgr;

  if ((_pBufMgr = (BufferMgmtPtr_t)smlLibMalloc(sizeof(BufferMgmt_t))) == NULL) return SML_ERR_NOT_ENOUGH_SPACE;

  //get the encoding type
  _enc = pEncoder->enc;
  
  _pBufMgr->smlXltBufferP = *ppBufPos;
  _pBufMgr->smlXltBufferLen = pBufEnd - *ppBufPos;
  _pBufMgr->smlXltStoreBufP = _pBufMgr->smlXltBufferP;
  _pBufMgr->smlXltWrittenBytes = 0;      
  _pBufMgr->smlCurCodepage = pEncoder->cur_codepage;
  _pBufMgr->smlLastCodepage = pEncoder->last_codepage;

  _err = getTNbyPE(pe, &tagID);
  _err = xltEncBlock(tagID, REQUIRED, pContent, _enc, _pBufMgr, SML_EXT_UNDEFINED);
  if (_err != SML_ERR_OK)
  {
     smlLibFree(_pBufMgr);
     return _err;
  }  

  pEncoder->cur_codepage = _pBufMgr->smlCurCodepage;
  pEncoder->last_codepage = _pBufMgr->smlLastCodepage;

  *ppBufPos = _pBufMgr->smlXltBufferP;

  smlLibFree(_pBufMgr);

  return SML_ERR_OK;
}

/**
 * FUNCTION: smlXltEncTerminate
 *
 * Filnalizes the (WB)XML document and returns the size of wrotten bytes to 
 * the workspace module
 *
 * PRE-Condition:   pEncoder holds the initialized encoder structure.
 *                  the initialization takes place in the xltEncAppend function
 *                  pBufEnd must point to the end of the (WB)XML buffer
 *                  ppBufPos has to be initialized to the start point of the
 *                  (WB)XML buffer.
 *                  
 * POST-Condition:  After the function call ppBufPos points to the
 *                  first free byte in the buffer behind the (WB)XML document
 *
 * IN:              pEncoder, the encoder object
 *                  pBufEnd, pointer to the end of the buffer to write on
 * 
 * IN/OUT:          ppBufPos, current position of the bufferpointer
 * 
 * RETURN:          shows error codes of function, 
 *                  0, if OK
 *                  Possible Error Codes:
 *                  SML_ERR_XLT_BUF_ERR      
 *                  SML_ERR_XLT_MISSING_CONT
 *                  SML_ERR_XLT_INVAL_ELEM_TYPE         
 *                  SML_ERR_XLT_INVAL_LIST_TYPE         
 *                  SML_ERR_XLT_INVAL_TAG_TYPE          
 *                  SML_ERR_XLT_ENC_UNK	               
 *                  SML_ERR_XLT_INVAL_PROTO_ELEM
 */
Ret_t xltEncTerminate(const XltEncoderPtr_t pEncoder,
                      const MemPtr_t pBufEnd,
                      MemPtr_t *ppBufPos)
{
  // Return variable      
  Ret_t _err;

  // encoding type
  SmlEncoding_t _enc;

  //Structure containing buffer pointers, length and written bytes
  BufferMgmtPtr_t _pBufMgr;

  //get the encoding type
  _enc = pEncoder->enc;

  //Initialize buffer variables
  if ((_pBufMgr = smlLibMalloc(sizeof(BufferMgmt_t))) == NULL) return SML_ERR_NOT_ENOUGH_SPACE;

  _pBufMgr->smlXltStoreBufP = _pBufMgr->smlXltBufferP;
  _pBufMgr->smlXltWrittenBytes = 0;      
  _pBufMgr->smlXltBufferP = *ppBufPos;
  _pBufMgr->smlXltBufferLen = pBufEnd - *ppBufPos;
  _pBufMgr->smlCurCodepage = pEncoder->cur_codepage;
  _pBufMgr->smlLastCodepage = pEncoder->last_codepage;

  if (pEncoder->final == 1)
  {
    // Final Flag     
    if ((_err = xltGenerateTag(TN_FINAL, TT_ALL, _enc, _pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) 
    {
      smlLibFree(_pBufMgr);
      smlLibFree(pEncoder);
      return _err;
    }
  }

  // SyncBody End Tag     
  if ((_err = xltGenerateTag(TN_SYNCBODY, TT_END, _enc, _pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) 
  {
    smlLibFree(_pBufMgr);
    smlLibFree(pEncoder);
    return _err;
  }

  // SyncML End Tag       
  if ((_err = xltGenerateTag(TN_SYNCML_END, TT_END, _enc, _pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK)
  {
    smlLibFree(_pBufMgr);
    smlLibFree(pEncoder);
    return _err;
  }

  pEncoder->cur_codepage = _pBufMgr->smlCurCodepage;
  pEncoder->last_codepage = _pBufMgr->smlLastCodepage;

  *ppBufPos = _pBufMgr->smlXltBufferP;

  smlLibFree(_pBufMgr);

  smlLibFree(pEncoder);

  return SML_ERR_OK;
}

Ret_t xltEncReset(XltEncoderPtr_t pEncoder)
{
  smlLibFree(pEncoder);
  return SML_ERR_OK;
}

/**
 * FUNCTION: xltEncBlock
 *
 * Generates a (WB)XML Block for a given tag ID and a given content
 *
 * PRE-Condition:   pContent holds a valid content structure
 *                  tagId contains a valid SyncML tag ID                  
 *
 * POST-Condition:  the (WB)XML buffer in the pBufMgr structure contains the
 *                  encoded (WB)XML block
 *
 * IN:              tagId, the ID for the tag to generate (TN_ADD, ...)
 *                  reqOptFlag, flag if the block is required or optional
 *                  pContent, the content structure of the block
 *                  enc, the encoding constant (SML_WBXML or SML_XML)
 *                  attFlag, indicates if the encoded tag contain Attributes in namespace extensions
 * 
 * IN/OUT:          pBufMgr, pointer to a structure containing buffer management elements
 * 
 * RETURN:          shows error codes of function, 
 *                  0, if OK
 */
Ret_t xltEncBlock(XltTagID_t tagId, XltRO_t reqOptFlag, const VoidPtr_t pContent, SmlEncoding_t enc, BufferMgmtPtr_t pBufMgr, SmlPcdataExtension_t attFlag)
{

  //Return variable
  Ret_t _err;

  //Check if pContent of a required field is missing
  if ((reqOptFlag == REQUIRED) && (pContent == NULL))
  {
    switch (tagId)
    {
      case TN_ATOMIC_END:
      case TN_SYNC_END:
      case TN_SEQUENCE_END:      
      {
        break;
      }
      default:  
        return SML_ERR_XLT_MISSING_CONT;
    }    
  }  
  //Check if pContent of a optional fiel is missing
  else if ((pContent == NULL) && (tagId != TN_SYNC_END) && (tagId != TN_ATOMIC_END) && (tagId != TN_SEQUENCE_END))
    return SML_ERR_OK;
  
  //Generate the commands -> see DTD
  switch (tagId)
  {
    case TN_SYNCHDR:
    {
      // SyncHdr Begin Tag
      if ((_err = xltGenerateTag(TN_SYNCHDR, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // Version
      if ((_err = xltEncBlock(TN_VERSION, REQUIRED, ((SmlSyncHdrPtr_t) pContent)->version, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // Proto
      if ((_err = xltEncBlock(TN_PROTO, REQUIRED, ((SmlSyncHdrPtr_t) pContent)->proto, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // SessionID
      if ((_err = xltEncBlock(TN_SESSIONID, REQUIRED, ((SmlSyncHdrPtr_t) pContent)->sessionID, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // MsgID
      if ((_err = xltEncBlock(TN_MSGID, REQUIRED, ((SmlSyncHdrPtr_t) pContent)->msgID, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // Target
      if ((_err = xltEncBlock(TN_TARGET, REQUIRED, ((SmlSyncHdrPtr_t) pContent)->target, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // Source
      if ((_err = xltEncBlock(TN_SOURCE, REQUIRED, ((SmlSyncHdrPtr_t) pContent)->source, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // RespURI?
      if ((_err = xltEncBlock(TN_RESPURI, OPTIONAL, ((SmlSyncHdrPtr_t) pContent)->respURI, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // NoResp?
      if ((_err = xltEncBlock(TN_NORESP, OPTIONAL, &((SmlSyncHdrPtr_t) pContent)->flags, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // Cred?
      if ((_err = xltEncBlock(TN_CRED, OPTIONAL, ((SmlSyncHdrPtr_t) pContent)->cred, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // Meta?
      if ((_err = xltEncBlock(TN_META, OPTIONAL, ((SmlSyncHdrPtr_t) pContent)->meta, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // SyncHdr End Tag
      if ((_err = xltGenerateTag(TN_SYNCHDR, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
    case TN_CRED:
    {
      // Begin tag
      if ((_err = xltGenerateTag(TN_CRED, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // Meta?
      if ((_err = xltEncBlock(TN_META, OPTIONAL, ((SmlCredPtr_t) pContent)->meta, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                           
      // Data
      if ((_err = xltEncBlock(TN_DATA, REQUIRED, ((SmlCredPtr_t) pContent)->data, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                           
      // End tag
      if ((_err = xltGenerateTag(TN_CRED, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
    case TN_SOURCE:
    case TN_TARGET:
    {
      // Begin tag
      if ((_err = xltGenerateTag(tagId, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // LocURI
      if ((_err = xltEncBlock(TN_LOCURI, REQUIRED, ((SmlSourcePtr_t) pContent)->locURI, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                 
      // LocName?
      if ((_err = xltEncBlock(TN_LOCNAME, OPTIONAL, ((SmlSourcePtr_t) pContent)->locName, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                       
      // End tag
      if ((_err = xltGenerateTag(tagId, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
    case TN_ITEM:
    {
      // Begin tag
      if ((_err = xltGenerateTag(TN_ITEM, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // Target?
      if ((_err = xltEncBlock(TN_TARGET, OPTIONAL, ((SmlItemPtr_t) pContent)->target, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                       
      // Source?
      if ((_err = xltEncBlock(TN_SOURCE, OPTIONAL, ((SmlItemPtr_t) pContent)->source, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                       
      // Meta?
      if ((_err = xltEncBlock(TN_META, OPTIONAL, ((SmlItemPtr_t) pContent)->meta, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                           
      // Data?
      if ((_err = xltEncBlock(TN_DATA, OPTIONAL, ((SmlItemPtr_t) pContent)->data, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                           
      // End tag
      if ((_err = xltGenerateTag(TN_ITEM, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
    case TN_ADD:
    case TN_COPY:
    {
      // Begin tag
      if ((_err = xltGenerateTag(tagId, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // CmdID
      if ((_err = xltEncBlock(TN_CMDID, REQUIRED, ((SmlGenericCmdPtr_t) pContent)->cmdID, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                           
      // NoResp?
      if ((_err = xltEncBlock(TN_NORESP, OPTIONAL, &((SmlGenericCmdPtr_t) pContent)->flags, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                           
      // Cred?
      if ((_err = xltEncBlock(TN_CRED, OPTIONAL, ((SmlGenericCmdPtr_t) pContent)->cred, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                     
      // Meta?
      if ((_err = xltEncBlock(TN_META, OPTIONAL, ((SmlGenericCmdPtr_t) pContent)->meta, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                     
      // Item+
      if ((_err = xltEncList(ITEM_LIST, REQUIRED, ((SmlGenericCmdPtr_t) pContent)->itemList, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                   
      // End tag
      if ((_err = xltGenerateTag(tagId, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
    case TN_ALERT:
    {
      // Begin tag
      if ((_err = xltGenerateTag(TN_ALERT, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // CmdID
      if ((_err = xltEncBlock(TN_CMDID, REQUIRED, ((SmlAlertPtr_t) pContent)->cmdID, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                        
      // NoResp?
      if ((_err = xltEncBlock(TN_NORESP, OPTIONAL, &((SmlAlertPtr_t) pContent)->flags, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                        
      // Cred?
      if ((_err = xltEncBlock(TN_CRED, OPTIONAL, ((SmlAlertPtr_t) pContent)->cred, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                          
      // Data?
      if ((_err = xltEncBlock(TN_DATA, OPTIONAL, ((SmlAlertPtr_t) pContent)->data, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                          
      // Item+
      if ((_err = xltEncList(ITEM_LIST, REQUIRED, ((SmlAlertPtr_t) pContent)->itemList, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                   
      // End tag
      if ((_err = xltGenerateTag(TN_ALERT, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
#ifndef __SML_LITE__    
    case TN_ATOMIC:
    case TN_SEQUENCE:
    {
      // Begin tag
      if ((_err = xltGenerateTag(tagId, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // CmdID
      if ((_err = xltEncBlock(TN_CMDID, REQUIRED, ((SmlAtomicPtr_t) pContent)->cmdID, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                       
      // NoResp?
      if ((_err = xltEncBlock(TN_NORESP, OPTIONAL, &((SmlAtomicPtr_t) pContent)->flags, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                       
      // Meta?
      if ((_err = xltEncBlock(TN_META, OPTIONAL, ((SmlAtomicPtr_t) pContent)->meta, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                         
      //End tag in TN_ATOMIC_END

      break;
    }
    case TN_ATOMIC_END:
    {
      // End tag
      if ((_err = xltGenerateTag(TN_ATOMIC, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
    case TN_SEQUENCE_END:
    {
      // End tag
      if ((_err = xltGenerateTag(TN_SEQUENCE, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
#endif    
    case TN_DELETE:
    {
      // Begin tag
      if ((_err = xltGenerateTag(TN_DELETE, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // CmdID
      if ((_err = xltEncBlock(TN_CMDID, REQUIRED, ((SmlDeletePtr_t) pContent)->cmdID, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                       
      // NoResp?
      if ((_err = xltEncBlock(TN_NORESP, OPTIONAL, &((SmlDeletePtr_t) pContent)->flags, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                       
      // Archive?
      if ((_err = xltEncBlock(TN_ARCHIVE, OPTIONAL, &(((SmlDeletePtr_t) pContent)->flags), enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                          
      // SftDel?
      if ((_err = xltEncBlock(TN_SFTDEL, OPTIONAL, &(((SmlDeletePtr_t) pContent)->flags), enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                           
      // Cred?
      if ((_err = xltEncBlock(TN_CRED, OPTIONAL, ((SmlDeletePtr_t) pContent)->cred, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                         
      // Meta?
      if ((_err = xltEncBlock(TN_META, OPTIONAL, ((SmlDeletePtr_t) pContent)->meta, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                         
      // Item+
      if ((_err = xltEncList(ITEM_LIST, REQUIRED, ((SmlDeletePtr_t) pContent)->itemList, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                   
      // End tag
      if ((_err = xltGenerateTag(TN_DELETE, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
#ifndef __SML_LITE__    
    case TN_EXEC:
    {
      // Begin tag
      if ((_err = xltGenerateTag(TN_EXEC, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // CmdID
      if ((_err = xltEncBlock(TN_CMDID, REQUIRED, ((SmlExecPtr_t) pContent)->cmdID, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                         
      // NoResp?
      if ((_err = xltEncBlock(TN_NORESP, OPTIONAL, &((SmlExecPtr_t) pContent)->flags, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                         
      // Cred?
      if ((_err = xltEncBlock(TN_CRED, OPTIONAL, ((SmlExecPtr_t) pContent)->cred, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                         
      // Item
      if ((_err = xltEncBlock(TN_ITEM, REQUIRED, ((SmlExecPtr_t) pContent)->item, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                           
      // End tag
      if ((_err = xltGenerateTag(TN_EXEC, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
#endif
    case TN_GET:
    case TN_PUT:
    {
      // Begin tag
      if ((_err = xltGenerateTag(tagId, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // CmdID
      if ((_err = xltEncBlock(TN_CMDID, REQUIRED, ((SmlGetPtr_t) pContent)->cmdID, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                          
      // NoResp?
      if ((_err = xltEncBlock(TN_NORESP, OPTIONAL, &((SmlGetPtr_t) pContent)->flags, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                          
      // Lang?
      if ((_err = xltEncBlock(TN_LANG, OPTIONAL, ((SmlGetPtr_t) pContent)->lang, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                            
      // Cred?
      if ((_err = xltEncBlock(TN_CRED, OPTIONAL, ((SmlGetPtr_t) pContent)->cred, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                            
      // Meta?
      if ((_err = xltEncBlock(TN_META, OPTIONAL, ((SmlGetPtr_t) pContent)->meta, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                            
      // Item+
      if ((_err = xltEncList(ITEM_LIST, REQUIRED, ((SmlGetPtr_t) pContent)->itemList, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                   
      // End tag
      if ((_err = xltGenerateTag(tagId, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                    
      
      break;
    }
    case TN_MAP:
    {
      // Begin tag
      if ((_err = xltGenerateTag(TN_MAP, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // CmdID
      if ((_err = xltEncBlock(TN_CMDID, REQUIRED, ((SmlMapPtr_t) pContent)->cmdID, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                          
      // Target
      if ((_err = xltEncBlock(TN_TARGET, REQUIRED, ((SmlMapPtr_t) pContent)->target, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                        
      // Source
      if ((_err = xltEncBlock(TN_SOURCE, REQUIRED, ((SmlMapPtr_t) pContent)->source, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                        
      // Cred?
      if ((_err = xltEncBlock(TN_CRED, OPTIONAL, ((SmlMapPtr_t) pContent)->cred, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                            
      // Meta?
      if ((_err = xltEncBlock(TN_META, OPTIONAL, ((SmlMapPtr_t) pContent)->meta, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                            
      // Mapitemlist
      if ((_err = xltEncList(MAPITEM_LIST, REQUIRED, ((SmlMapPtr_t) pContent)->mapItemList, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                         
      // End tag
      if ((_err = xltGenerateTag(TN_MAP, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
    case TN_MAPITEM:
    {
      // Begin tag
      if ((_err = xltGenerateTag(TN_MAPITEM, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // Target
      if ((_err = xltEncBlock(TN_TARGET, REQUIRED, ((SmlMapItemPtr_t) pContent)->target, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                            
      // Source
      if ((_err = xltEncBlock(TN_SOURCE, REQUIRED, ((SmlMapItemPtr_t) pContent)->source, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                            
      // End tag
      if ((_err = xltGenerateTag(TN_MAPITEM, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
    case TN_RESULTS:
    {
      // Begin tag
      if ((_err = xltGenerateTag(TN_RESULTS, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // CmdID
      if ((_err = xltEncBlock(TN_CMDID, REQUIRED, ((SmlResultsPtr_t) pContent)->cmdID, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                       
      // MsgRef?
      if ((_err = xltEncBlock(TN_MSGREF, OPTIONAL, ((SmlResultsPtr_t) pContent)->msgRef, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                            
      // CmdRef
      if ((_err = xltEncBlock(TN_CMDREF, REQUIRED, ((SmlResultsPtr_t) pContent)->cmdRef, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                            
      // Meta?
      if ((_err = xltEncBlock(TN_META, OPTIONAL, ((SmlResultsPtr_t) pContent)->meta, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                        
      // TargetRef?
      if ((_err = xltEncBlock(TN_TARGETREF, OPTIONAL, ((SmlResultsPtr_t) pContent)->targetRef, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                        
      // SourceRef?
      if ((_err = xltEncBlock(TN_SOURCEREF, OPTIONAL, ((SmlResultsPtr_t) pContent)->sourceRef, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                        
      // Item+
      if ((_err = xltEncList(ITEM_LIST, REQUIRED, ((SmlResultsPtr_t) pContent)->itemList, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                   
      // End tag
      if ((_err = xltGenerateTag(TN_RESULTS, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
    case TN_CHAL:
    {
      // Begin tag
      if ((_err = xltGenerateTag(TN_CHAL, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // Meta
      if ((_err = xltEncBlock(TN_META, REQUIRED, ((SmlChalPtr_t) pContent)->meta, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                        
      // End tag
      if ((_err = xltGenerateTag(TN_CHAL, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
#ifndef __SML_LITE__
    case TN_SEARCH:
    {
      // Begin tag
      if ((_err = xltGenerateTag(TN_SEARCH, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // CmdID
      if ((_err = xltEncBlock(TN_CMDID, REQUIRED, ((SmlSearchPtr_t) pContent)->cmdID, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                       
      // NoResp?
      if ((_err = xltEncBlock(TN_NORESP, OPTIONAL, &((SmlSearchPtr_t) pContent)->flags, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                       
      // NoResults?
      if ((_err = xltEncBlock(TN_NORESULTS, OPTIONAL, &((SmlSearchPtr_t) pContent)->flags, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                       
      // Cred?
      if ((_err = xltEncBlock(TN_CRED, OPTIONAL, ((SmlSearchPtr_t) pContent)->cred, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                         
      // Target?
      if ((_err = xltEncBlock(TN_TARGET, OPTIONAL, ((SmlSearchPtr_t) pContent)->target, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                     
      // Source List
      if ((_err = xltEncList(SOURCE_LIST, REQUIRED, ((SmlSearchPtr_t) pContent)->sourceList, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                        
      // Lang?
      if ((_err = xltEncBlock(TN_LANG, OPTIONAL, ((SmlSearchPtr_t) pContent)->lang, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                         
      // Meta
      if ((_err = xltEncBlock(TN_META, REQUIRED, ((SmlSearchPtr_t) pContent)->meta, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                         
      // Dsta
      if ((_err = xltEncBlock(TN_DATA, REQUIRED, ((SmlSearchPtr_t) pContent)->data, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                         
      // End tag
      if ((_err = xltGenerateTag(TN_SEARCH, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
#endif    
    case TN_STATUS:
    {
      // Begin tag
      if ((_err = xltGenerateTag(TN_STATUS, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // CmdID
      if ((_err = xltEncBlock(TN_CMDID, REQUIRED, ((SmlStatusPtr_t) pContent)->cmdID, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                       
      // MsgRef?
      if ((_err = xltEncBlock(TN_MSGREF, OPTIONAL, ((SmlStatusPtr_t) pContent)->msgRef, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                         
      // CmdRef
      if ((_err = xltEncBlock(TN_CMDREF, REQUIRED, ((SmlStatusPtr_t) pContent)->cmdRef, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                         
      // Cmd
      if ((_err = xltEncBlock(TN_CMD, REQUIRED, ((SmlStatusPtr_t) pContent)->cmd, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                       
      // TargetRefList?
      if ((_err = xltEncList(TARGETREF_LIST, OPTIONAL, ((SmlStatusPtr_t) pContent)->targetRefList, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                       
      // SourceRefList?
      if ((_err = xltEncList(SOURCEREF_LIST, OPTIONAL, ((SmlStatusPtr_t) pContent)->sourceRefList, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                       
      // Cred?
      if ((_err = xltEncBlock(TN_CRED, OPTIONAL, ((SmlStatusPtr_t) pContent)->cred, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                         
      // Chal?
      if ((_err = xltEncBlock(TN_CHAL, OPTIONAL, ((SmlStatusPtr_t) pContent)->chal, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                         
      // Data
      if ((_err = xltEncBlock(TN_DATA, REQUIRED, ((SmlStatusPtr_t) pContent)->data, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                          
      // Item*
      if ((_err = xltEncList(ITEM_LIST, OPTIONAL, ((SmlStatusPtr_t) pContent)->itemList, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                   
      // End tag
      if ((_err = xltGenerateTag(TN_STATUS, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
    case TN_SYNC:
    {
      // Begin tag
      if ((_err = xltGenerateTag(TN_SYNC, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // CmdID
      if ((_err = xltEncBlock(TN_CMDID, REQUIRED, ((SmlSyncPtr_t) pContent)->cmdID, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                         
      // NoResp?
      if ((_err = xltEncBlock(TN_NORESP, OPTIONAL, &((SmlSyncPtr_t) pContent)->flags, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                         
      // Cred?
      if ((_err = xltEncBlock(TN_CRED, OPTIONAL, ((SmlSyncPtr_t) pContent)->cred, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                           
      // Target?
      if ((_err = xltEncBlock(TN_TARGET, OPTIONAL, ((SmlSyncPtr_t) pContent)->target, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                       
      // Source?
      if ((_err = xltEncBlock(TN_SOURCE, OPTIONAL, ((SmlSyncPtr_t) pContent)->source, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                       
      // Meta?
      if ((_err = xltEncBlock(TN_META, OPTIONAL, ((SmlSyncPtr_t) pContent)->meta, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                           

      // End tag in TN_SYNC_END

      break;
    }
    case TN_SYNC_END:
    {
      //End tag
      if ((_err = xltGenerateTag(TN_SYNC, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
    case TN_REPLACE:
    {
      // Begin tag
      if ((_err = xltGenerateTag(tagId, TT_BEG, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      // CmdID
      if ((_err = xltEncBlock(TN_CMDID, REQUIRED, ((SmlGenericCmdPtr_t) pContent)->cmdID, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                           
      // NoResp?
      if ((_err = xltEncBlock(TN_NORESP, OPTIONAL, &((SmlGenericCmdPtr_t) pContent)->flags, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                           
      // Cred?
      if ((_err = xltEncBlock(TN_CRED, OPTIONAL, ((SmlGenericCmdPtr_t) pContent)->cred, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                     
      // Meta?
      if ((_err = xltEncBlock(TN_META, OPTIONAL, ((SmlGenericCmdPtr_t) pContent)->meta, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                     
      // Item+     
      if ((_err = xltEncList(ITEM_LIST, REQUIRED, ((SmlGenericCmdPtr_t) pContent)->itemList, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;                                   
      // End tag
      if ((_err = xltGenerateTag(tagId, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;

      break;
    }
    case TN_ARCHIVE:                        
    {
      //set the flag in the (WB)XML document if the flag is in the pContent
      if ((*((Flag_t*)pContent)) & (SmlArchive_f))
      {
        if ((_err = xltGenerateTag(tagId, TT_ALL, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      }                       
      break;
    }
    case TN_SFTDEL:                 
    {
      //set the flag in the (WB)XML document if the flag is in the pContent
      if ((*((Flag_t *) pContent)) & (SmlSftDel_f))
      {
        if ((_err = xltGenerateTag(tagId, TT_ALL, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      }
      break;
    }
#ifndef __SML_LITE__    
    case TN_NORESULTS:
    {
      //set the flag in the (WB)XML document if the flag is in the pContent
      if ((*((Flag_t *) pContent)) & (SmlNoResults_f))
      {
        if ((_err = xltGenerateTag(tagId, TT_ALL, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      }
      break;
    }
#endif    
    case TN_NORESP:                                 
    {
      //set the flag in the (WB)XML document if the flag is in the pContent
      if ((*((Flag_t *) pContent)) & (SmlNoResp_f))
      {
        if ((_err = xltGenerateTag(tagId, TT_ALL, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      }
      break;
    }
    case TN_FINAL:                  
    {
      //set the flag in the (WB)XML document if the flag is in the pContent
      if ((*((Flag_t *) pContent)) & (SmlFinal_f))
      {
        if ((_err = xltGenerateTag(tagId, TT_ALL, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      }
      break;
    }
#ifdef __META_EXT__
    case TN_META_SHAREDMEM:                  
    {
      //set the flag in the (WB)XML document if the flag is in the pContent
      if ((*((Flag_t *) pContent)) & (SmlSharedMem_f))
      {
        if ((_err = xltGenerateTag(tagId, TT_ALL, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
      }
      break;
    }
    case TN_META_ANCHOR:
    {
      // Begin tag
      if ((_err = xltGenerateTag(tagId, TT_BEG, enc, pBufMgr, SML_EXT_META)) != SML_ERR_OK) return _err;
      // Last?
      if ((_err = xltEncBlock(TN_META_LAST, OPTIONAL, ((SmlAnchorPtr_t) pContent)->last, enc, pBufMgr, SML_EXT_META)) != SML_ERR_OK) return _err;                                           
      // Next
      if ((_err = xltEncBlock(TN_META_NEXT, REQUIRED, ((SmlAnchorPtr_t) pContent)->next, enc, pBufMgr, SML_EXT_META)) != SML_ERR_OK) return _err;                                     
      // End tag
      if ((_err = xltGenerateTag(tagId, TT_END, enc, pBufMgr, SML_EXT_META)) != SML_ERR_OK) return _err;

      break;
    }
    case TN_META_MEM:
    {
      // Begin tag
      if ((_err = xltGenerateTag(tagId, TT_BEG, enc, pBufMgr, SML_EXT_META)) != SML_ERR_OK) return _err;
      // SharedMem?
      if ((_err = xltEncBlock(TN_META_SHAREDMEM, OPTIONAL, &((SmlMemPtr_t) pContent)->sharedmem, enc, pBufMgr, SML_EXT_META)) != SML_ERR_OK) return _err;                                           
      // FreeMem
      if ((_err = xltEncBlock(TN_META_FREEMEM, REQUIRED, ((SmlMemPtr_t) pContent)->freemem, enc, pBufMgr, SML_EXT_META)) != SML_ERR_OK) return _err;                                     
      // FreeID
      if ((_err = xltEncBlock(TN_META_FREEID, REQUIRED, ((SmlMemPtr_t) pContent)->freeid, enc, pBufMgr, SML_EXT_META)) != SML_ERR_OK) return _err;                                     
      // End tag
      if ((_err = xltGenerateTag(tagId, TT_END, enc, pBufMgr, SML_EXT_META)) != SML_ERR_OK) return _err;

      break;
    }
#endif
    default:  // all leaf nodes (PCDATA#)
    {
      //generate PCDATA begin tag
      if ((_err = xltGenerateTag(tagId, TT_BEG, enc, pBufMgr, attFlag)) != SML_ERR_OK) return _err;

      //write the pContent to the buffer according the encoding type
      switch (enc)
      {

#ifdef __SML_WBXML__
        case SML_WBXML:
        { 
          switch (((SmlPcdataPtr_t)pContent)->contentType)
          {
            //STR_I
            case SML_PCDATA_STRING:
            {                                       
              if ((_err = wbxmlWriteTypeToBuffer(((SmlPcdataPtr_t)pContent)->content, STR_I, ((SmlPcdataPtr_t)pContent)->length, pBufMgr)) != SML_ERR_OK) return _err;
              break;
            }
            //OPAQUE
            case SML_PCDATA_OPAQUE:
            {
              if ((_err = wbxmlWriteTypeToBuffer(((SmlPcdataPtr_t)pContent)->content, OPAQUE, ((SmlPcdataPtr_t)pContent)->length, pBufMgr)) != SML_ERR_OK) return _err;
              break;
            }
#ifdef __META_EXT__
            case SML_PCDATA_EXTENSION:
            {
              if ((_err = xltBuildExtention(((SmlPcdataPtr_t)pContent)->extension, reqOptFlag, ((SmlPcdataPtr_t)pContent)->content, enc, pBufMgr)) != SML_ERR_OK) return _err;
              break;
            }
#endif            
            default:
              return SML_ERR_XLT_INVAL_PCDATA_TYPE;
          }
          break;
        }
#endif

#ifdef __SML_XML__
        case SML_XML:
        {
          MemPtr_t _tmpStr;        
          switch (((SmlPcdataPtr_t)pContent)->contentType)
          {
            //CDATA
            case SML_PCDATA_CDATA:
            {      
              _tmpStr = (MemPtr_t) "<![CDATA[";
              if ((_err = xltAddToBuffer(_tmpStr, smlLibStrlen((String_t)_tmpStr), pBufMgr)) != SML_ERR_OK) return _err;
              if ((_err = xltAddToBuffer(((SmlPcdataPtr_t)pContent)->content, ((SmlPcdataPtr_t)pContent)->length, pBufMgr)) != SML_ERR_OK) return _err;
              _tmpStr = (MemPtr_t) "]]";
              if ((_err = xltAddToBuffer(_tmpStr, smlLibStrlen((String_t)_tmpStr), pBufMgr)) != SML_ERR_OK) return _err;
              break;
            }
            //STR_I
            case SML_PCDATA_STRING:
            {                    
              if ((_err = xltAddToBuffer(((SmlPcdataPtr_t)pContent)->content, ((SmlPcdataPtr_t)pContent)->length, pBufMgr)) != SML_ERR_OK) return _err;
              break;
            }
#ifdef __META_EXT__
            case SML_PCDATA_EXTENSION:
            {
              if ((_err = xltBuildExtention(((SmlPcdataPtr_t)pContent)->extension, reqOptFlag, ((SmlPcdataPtr_t)pContent)->content, enc, pBufMgr)) != SML_ERR_OK) return _err;
              break;
            }
#endif
            default:
              return SML_ERR_XLT_INVAL_PCDATA_TYPE;
          }  
          break;          
        }
#endif

        default:
        {
          return SML_ERR_XLT_ENC_UNK;
        }  
      }

      //generate PCDATA END tag
      if ((_err = xltGenerateTag(tagId, TT_END, enc, pBufMgr, SML_EXT_UNDEFINED)) != SML_ERR_OK) return _err;
    }
  }

  return SML_ERR_OK;
}

/**
 * FUNCTION: xltEncList
 *
 * Generates a list element which is not directly related to a tag
 *
 * PRE-Condition:   pList holds a valid list structure
 *                  listId contains a valid SyncML list ID                  
 *
 * POST-Condition:  the (WB)XML buffer in the pBufMgr structure contains the
 *                  encoded (WB)XML list
 *
 * IN:              listId, the ID of the list to generate (e.g. TARGET_LIST, ...)
 *                  pList, reference to the list to process
 *                  enc, the encoding constant (SML_WBXML or SML_XML)
 *                  attFlag, indicates if the encoded tag contain Attributes in namespace extensions
 * 
 * IN/OUT:          pBufMgr, pointer to a structure containing buffer management elements
 * 
 * RETURN:          shows error codes of function, 
 *                  0, if OK
 */
Ret_t xltEncList(XltListType_t listId, XltRO_t reqOptFlag, VoidPtr_t pList, SmlEncoding_t enc, BufferMgmtPtr_t pBufMgr, SmlPcdataExtension_t attFlag)
{
  //Return variable
  Ret_t _err;

  //check if list is required or not
  if ((reqOptFlag == REQUIRED) && (pList == NULL))
    return SML_ERR_XLT_MISSING_CONT;
  else if (pList == NULL)
    return SML_ERR_OK;
  
  //encode the different list types
  switch (listId)
  {
    case ITEM_LIST:
    {
      do
      {        
        if ((_err = xltEncBlock(TN_ITEM, OPTIONAL, ((SmlItemListPtr_t)pList)->item, enc, pBufMgr, attFlag)) != SML_ERR_OK) return _err;
        pList = ((SmlItemListPtr_t)pList)->next;
      } while ((SmlItemListPtr_t)pList != NULL);

      break;
    }
    case SOURCE_LIST:
    {
      do
      {        
        if ((_err = xltEncBlock(TN_SOURCE, OPTIONAL, ((SmlSourceListPtr_t)pList)->source, enc, pBufMgr, attFlag)) != SML_ERR_OK) return _err;
        pList = ((SmlSourceListPtr_t)pList)->next;
      } while ((SmlSourceListPtr_t)pList != NULL);

      break;
    }
#ifdef __META_EXT__
    case META_LIST:
    {
      do
      {        
        if ((_err = xltEncBlock(TN_META, OPTIONAL, ((SmlMetaListPtr_t)pList)->meta, enc, pBufMgr, attFlag)) != SML_ERR_OK) return _err;
        pList = ((SmlMetaListPtr_t)pList)->next;
      } while ((SmlMetaListPtr_t)pList != NULL);

      break;
    }
#endif
    case TARGETREF_LIST:
    {
      do
      {        
        if ((_err = xltEncBlock(TN_TARGETREF, OPTIONAL, ((SmlTargetRefListPtr_t)pList)->targetRef, enc, pBufMgr, attFlag)) != SML_ERR_OK) return _err;
        pList = ((SmlTargetRefListPtr_t)pList)->next;
      } while ((SmlTargetRefListPtr_t)pList != NULL);

      break;
    }
    case SOURCEREF_LIST:
    {
      do
      {        
        if ((_err = xltEncBlock(TN_SOURCEREF, OPTIONAL, ((SmlSourceRefListPtr_t)pList)->sourceRef, enc, pBufMgr, attFlag)) != SML_ERR_OK) return _err;
        pList = ((SmlSourceRefListPtr_t)pList)->next;
      } while ((SmlSourceRefListPtr_t)pList != NULL);

      break;
    }
    case MAPITEM_LIST:
    {
      do
      {        
        if ((_err = xltEncBlock(TN_MAPITEM, OPTIONAL, ((SmlMapItemListPtr_t)pList)->mapItem, enc, pBufMgr, attFlag)) != SML_ERR_OK) return _err;
        pList = ((SmlMapItemListPtr_t)pList)->next;
      } while ((SmlMapItemListPtr_t)pList != NULL);

      break;
    }
#ifdef __META_EXT__
    case EMI_LIST:
    {
      do
      {        
        if ((_err = xltEncBlock(TN_META_EMI, OPTIONAL, ((SmlEmiListPtr_t)pList)->emi, enc, pBufMgr, attFlag)) != SML_ERR_OK) return _err;
        pList = ((SmlEmiListPtr_t)pList)->next;
      } while ((SmlEmiListPtr_t)pList != NULL);

      break;
    }
#endif
    return SML_ERR_XLT_INVAL_LIST_TYPE;
    }

  return SML_ERR_OK;
}

/**
 * FUNCTION: xltGenerateTag
 *
 * Generates a (WB)XML tag
 *
 * PRE-Condition:   valis parameters
 *
 * POST-Condition:  the buffer contains a new tag
 *
 * IN:              tagId, the tag ID
 *                  TagType, the tag type (begin tag, end tag, ...)
 *                  enc, the encoding constant (SML_WBXML or SML_XML)                  
 *                  attFlag, indicates if the encoded tag contain Attributes in namespace extensions
 * 
 * IN/OUT:          pBufMgr, pointer to a structure containing buffer management elements
 * 
 * RETURN:          shows error codes of function, 
 *                  0, if OK
 */
Ret_t xltGenerateTag(XltTagID_t tagId, XltTagType_t TagType, SmlEncoding_t enc, BufferMgmtPtr_t pBufMgr, SmlPcdataExtension_t attFlag)
{

#ifdef __SML_WBXML__
  Ret_t _err;
  MemByte_t _newcodepage = 0x00;
  MemByte_t _switchpage = XLT_SWITCHPAGE;
#endif

  switch (enc)
  {

#ifdef __SML_WBXML__
    case SML_WBXML:
    {
      if ((TagType == TT_ALL) || (TagType == TT_BEG))
      {
        _err = (MemByte_t)getCodePageById(tagId, &_newcodepage);
        if ((_newcodepage == 255) || (_err != SML_ERR_OK)) return _err;

        //codepage switching with wbxml instead of namespaces
        if (_newcodepage != pBufMgr->smlCurCodepage)
        {
          //start codepage switching
          if ((_err = wbxmlWriteTypeToBuffer((MemPtr_t)(&_switchpage), TAG, 1, pBufMgr)) != SML_ERR_OK) return _err;
          if ((_err = wbxmlWriteTypeToBuffer((MemPtr_t)(&_newcodepage), TAG, 1, pBufMgr)) != SML_ERR_OK) return _err;
          pBufMgr->smlLastCodepage = pBufMgr->smlCurCodepage;
          pBufMgr->smlCurCodepage = _newcodepage;
        }
      }
      return wbxmlGenerateTag(tagId, TagType, pBufMgr);
    }
#endif

#ifdef __SML_XML__
    case SML_XML:
    {
      return xmlGenerateTag(tagId, TagType, pBufMgr, attFlag);
    }
#endif

    default:
    {
      return SML_ERR_XLT_ENC_UNK;
    }
  }

  return SML_ERR_XLT_ENC_UNK;
}

#ifdef __META_EXT__
Ret_t xltBuildExtention(SmlPcdataExtension_t extId, XltRO_t reqOptFlag, VoidPtr_t pContent, SmlEncoding_t enc, BufferMgmtPtr_t pBufMgr)
{
  Ret_t _err;

  switch (extId)
  {
    case (SML_EXT_META):
    {
      SmlMetaPtr_t _tmp;
      SmlPcdataExtension_t _attFlag = SML_EXT_UNDEFINED;
      int _tmpFlag = 0;
      _tmp = (SmlMetaPtr_t) pContent;
      // in future versions we should call external functions here     
      // Format?
      if ((((SmlMetaPtr_t) pContent)->format != NULL) && (_tmpFlag == 0))
      {
        _attFlag = SML_EXT_META;
        _tmpFlag = 1;
      }
      if ((_err = xltEncBlock(TN_META_FORMAT, OPTIONAL, ((SmlMetaPtr_t) pContent)->format, enc, pBufMgr, _attFlag)) != SML_ERR_OK) return _err;                                       
      // Type?
      if (((SmlMetaPtr_t) pContent)->type != NULL) 
      {
        _attFlag = SML_EXT_META;
        _tmpFlag = 1;
      }
      if ((_err = xltEncBlock(TN_META_TYPE, OPTIONAL, ((SmlMetaPtr_t) pContent)->type, enc, pBufMgr, _attFlag)) != SML_ERR_OK) return _err;                                           
      // Mark?
      if ((((SmlMetaPtr_t) pContent)->mark != NULL) && (_tmpFlag == 0))
      {
        _attFlag = SML_EXT_META;
        _tmpFlag = 1;
      }
      if ((_err = xltEncBlock(TN_META_MARK, OPTIONAL, ((SmlMetaPtr_t) pContent)->mark, enc, pBufMgr, _attFlag)) != SML_ERR_OK) return _err;                                           
      // Size?
      if ((((SmlMetaPtr_t) pContent)->size != NULL) && (_tmpFlag == 0))
      {
        _attFlag = SML_EXT_META;
      }
      if ((_err = xltEncBlock(TN_META_SIZE, OPTIONAL, ((SmlMetaPtr_t) pContent)->size, enc, pBufMgr, _attFlag)) != SML_ERR_OK) return _err;                                           
      // Anchor?
      if ((((SmlMetaPtr_t) pContent)->anchor != NULL) && (_tmpFlag == 0))
      {
        _attFlag = SML_EXT_META;
        _tmpFlag = 1;
      }
      if ((_err = xltEncBlock(TN_META_ANCHOR, OPTIONAL, ((SmlMetaPtr_t) pContent)->anchor, enc, pBufMgr, _attFlag)) != SML_ERR_OK) return _err;                                     
      // Version?
      if ((((SmlMetaPtr_t) pContent)->version != NULL) && (_tmpFlag == 0))
      {
        _attFlag = SML_EXT_META;
        _tmpFlag = 1;
      }
      if ((_err = xltEncBlock(TN_META_VERSION, OPTIONAL, ((SmlMetaPtr_t) pContent)->version, enc, pBufMgr, _attFlag)) != SML_ERR_OK) return _err;                                     
      // NextOnce?
      if ((((SmlMetaPtr_t) pContent)->nextnonce != NULL) && (_tmpFlag == 0))
      {
        _attFlag = SML_EXT_META;
        _tmpFlag = 1;
      }
      if ((_err = xltEncBlock(TN_META_NEXTNONCE, OPTIONAL, ((SmlMetaPtr_t) pContent)->nextnonce, enc, pBufMgr, _attFlag)) != SML_ERR_OK) return _err;                                     
      // MaxMsgSize?
      if ((((SmlMetaPtr_t) pContent)->maxmsgsize != NULL) && (_tmpFlag == 0))
      {
        _attFlag = SML_EXT_META;
        _tmpFlag = 1;
      }
      if ((_err = xltEncBlock(TN_META_MAXMSGSIZE, OPTIONAL, ((SmlMetaPtr_t) pContent)->maxmsgsize, enc, pBufMgr, _attFlag)) != SML_ERR_OK) return _err;                                     
      // EMI List?
      if ((((SmlMetaPtr_t) pContent)->emilist != NULL) && (_tmpFlag == 0))
      {
        _attFlag = SML_EXT_META;
        _tmpFlag = 1;
      }
      if ((_err = xltEncList(EMI_LIST, OPTIONAL, ((SmlMetaPtr_t) pContent)->emilist, enc, pBufMgr, _attFlag)) != SML_ERR_OK) return _err;                                     
      // Mem?
      if ((((SmlMetaPtr_t) pContent)->mem != NULL) && (_tmpFlag == 0))
      {
        _attFlag = SML_EXT_META;
        _tmpFlag = 1;
      }
      if ((_err = xltEncBlock(TN_META_MEM, OPTIONAL, ((SmlMetaPtr_t) pContent)->mem, enc, pBufMgr, _attFlag)) != SML_ERR_OK) return _err;                                     


      break;
    }
    default:
      return SML_ERR_XLT_INVAL_EXT;
  }
  return SML_ERR_OK;
}
#endif
