/* @(#) $Id: //bas/46D/src/include/nixx.h#6 $ SAP*/
/*####################################################################*/
/*
 *     SAP AG Walldorf
 *     Systeme, Anwendungen und Produkte in der Datenverarbeitung
 *
 *     (C) Copyright (c) 1991-2004 SAP AG - 1997
 */
/*####################################################################*/



/*include-------------------------------------------------------------*/
/*                                                                    */
/* Beschreibung:                                                      */
/* ============                                                       */
/*                                                                    */
/* nixx.h:   ni = Projekt Network-Interface                           */
/*           xx = OS-unabhaengiges Coding                             */
/*                                                                    */
/* nixx.h enthaelt die Deklarationen fuer das Network-Interface.      */
/* Das Network-Interface bietet Dienste fuer die Interprozess-        */
/* Kommunikation ueber Rechnergrenzen hinweg an.                      */
/*                                                                    */
/* Ein einfaches Beispielprogramm fr Client und Server ist niexample */
/* (Modul niexampl.c). Hier findet man auch einfuehrende Worte in die */
/* Programmierung mit NI.                                             */
/*                                                                    */
/* Die ni-Bibliothek (nilib.o, Makefile ni.mak) kann mit              */
/* dem Programm 'niping' getestet werden (Modul nixxtst.c, Makefile   */
/* ni.mak)                                                            */
/*                                                                    */
/*-NiQue--------------------------------------------------------------*/
/*                                                                    */
/* Verbergen der Fragmentierung mit interner Queue (NiBuf):           */
/* ========================================================           */
/*                                                                    */
/* Die alte NI-Schicht verbirgt die Fragmentierung der                */
/* Pakete durch die Netzwerkschicht nicht.                            */
/* Dies fuehrt zu immer gleichem Coding in den Anwendungen.           */
/* Dies kann jetzt umgangen werden. Dafuer gibt es                    */
/* verschiedene Moeglichkeiten:                                       */
/*                                                                    */
/* 1) Man ersetzt alle Aufrufe von NiInit, NiConnect und              */
/*    NiListen durch die entsprechenden NiBuf*                        */
/*    Aufrufe. Ansonsten braucht man nichts zu aendern.               */
/* 2) In einer zweiten Stufe kann man die NiRead und                  */
/*    NiWrite-Aufrufe durch andere Calls ersetzen, die                */
/*    verhindern, dass die Daten kopiert werden muessen.              */
/*    Siehe unten "Neue Puffer-API"                                   */
/*                                                                    */
/* Es wird dann sichergestellt, dass entweder das ganze Paket oder    */
/* gar nichts verarbeitet wird.                                       */
/* Durch setzen der Parameter NIBUFP_MAXQUEUE und NIBUFP_MAXHEAP kann */
/* man die interne Queue aktivieren. Hierbei bedeutet NIBUFP_MAXQUEUE */
/* die maximale Queuelaenge pro handle und NIBUFP_MAXHEAP die         */
/* maximale Groesse aller Queues.                                     */
/* Falls die Queue voll ist wird zunaechst NIEQUE_FULL                */
/* zurueckgegeben, das Paket aber verarbeitet. Erst beim naechsten    */
/* Versuch wird mit NIETIMEOUT signalisiert, dass das Paket nicht     */
/* verarbeitet wurde. Nach NIEQUE_FULL sollte man mit NiWriteOK       */
/* sicherstellen, dass die Queue abgearbeitet wurde.                  */
/* Hinweis: Bei Ueberschreiten von NIBUFP_MAXHEAP wird immer nur      */
/* NIEQUE_FULL zurueckgegeben, aber die Queues trotzdem               */
/* erweitert. Dies ist nur ein Softlimit.                             */
/* NIEPING wird nur noch signalisiert, wenn als Antwort auf ein       */
/* NICheck direkt NI_PONG empfangen wird. D.h., wer NiCheck nicht     */
/* aufruft, bekommt NIEPING nie als Fehler zurueck                    */
/*                                                                    */
/* NiBufInit           Initialisierung der nilib und initialisierung  */
/*                     der internen Pufferung. Diese Funktion muss vor*/
/*                     den anderen NiBuf-Aufrufen gerufen werden      */
/* NiBufListen         Vorbereitungen fuer Connect-Requests von       */
/*                     Clients treffen (Server). Die per NiAccept auf */
/*                     diesem Handle erzeugten NI-Handles verbergen   */
/*                     die Fragmentierung                             */
/* NiBufConnect        Verbindung zu Server aufbauen. Der             */
/*                     resultierende NI-Handle verbirgt die           */
/*                     Fragmentierung                                 */
/*                                                                    */
/* Nonblocking Connects:                                              */
/* ====================                                               */
/* Ab Version 31 unterstuetzt NiBuf nonblocking Connects auf          */
/* Plattformen, die das bereitstellen:                                */
/* Nach NiBuf*Connect kann NIECONN_PENDING ignoriert werden und die   */
/* Pakete werden dann in die Queue gestellt. (Achtung: Wer            */
/* NIECONN_PENDING _nicht_ ignoriert, muss den Handle selber          */
/* schliessen!)                                                       */
/* Es ist damit mglich Connects mit timeout 0 (NI_ASYNC) aufzurufen. */
/*                                                                    */
/*-NiBuf--------------------------------------------------------NiQue-*/
/*                                                                    */
/* Neue Puffer-API:                                                   */
/* ===============                                                    */
/*                                                                    */
/* Diese API basiert auf der NiBuf-ERweiterung und bietet ein paar    */
/* neue Funktionen, die NiRead/Write ersetzen, bzw. Hilfsfunktionen   */
/* dazu.                                                              */
/* Diese Funktionen arbeiten nicht mit beliebigen Datenbereichen, in  */
/* die und aus denen dann kopiert werden muss, sondern es werden      */
/* Pointer auf Strukturen vom Typ NIBUFFER uebergeben. Dies fuehrt zu */
/* deutlich kuerzeren Laufzeiten als bei NiRead/Write                 */
/*                                                                    */
/* Die Structur NIBUFFER hat zwei oeffentliche Variablen:             */
/* struct NIBUFFER                                                    */
/* {                                                                  */
/*    SAP_RAW *data;                                                  */
/*    SAP_INT  len;                                                   */
/*    ...                                                             */
/* }                                                                  */
/* data ist ein Zeiger auf den Anfang des Datenbereichs und len ist   */
/* die Laenge desselben.                                              */
/*                                                                    */
/* Man erhaelt solche Puffer implizit ueber NiBufReceive und kann sie */
/* explizit ueber NiBufAlloc anfordern                                */
/* Diese Puffer muessen dann aber auch wieder freigegeben werden.     */
/* Dies geschieht wieder implizit ueber NiBufSend oder explizit ueber */
/* NiBufFree.                                                         */
/* Genauso,wie bei NiBufRead/Write wird garantiert, dass ein NiBufSend*/
/* nach einem NiBufReceive nicht mit NIETIMEOUT zurueckkehrt.         */
/* Diese Funktionen koennen auch mit NiRead/Write vermischt werden.   */
/*                                                                    */
/* Funktionen:                                                        */
/*                                                                    */
/* NiBufReceive    Empfange einen NIBUFFER                            */
/* NiBufSend       Sende einen NIBUFFER                               */
/*                                                                    */
/* NiBufFree       Gebe einen NIBUFFER wieder frei                    */
/* NiBufAlloc      Allokiere einen NIBUFFER                           */
/* NiBufWrap       'Verpacke' einen Datenbereich in einem NI-Puffer   */
/* NiBufUnWrap     'Entpacke' einen mit NiBufWrap                     */
/*                            erzeugten NI-Puffer                     */
/* NiBufDup        Dupliziere einen NIBUFFER                          */
/*                                                                    */
/* NiBufPush       Lege einen Puffer auf einen Stack                  */
/* NiBufPop        Hole einen Puffer vom Stack                        */
/*                                                                    */
/* Ueber den Handle-Parameter NIHP_ALLOC_FUNC kann man eine Funktion  */
/* registrieren, die beim Empfangen eines Pakets gerufen wird, um     */
/* einen Puffer zu allokieren.                                        */
/* Dies ermglicht zusammen mit NiBufWrap, auch z.B. direkt aus       */
/* shared Memory heraus zu senden, ohne kopieren zu mssen            */
/*                                                                    */
/*--------------------------------------------------------------NiBuf-*/
/*-NiSel--------------------------------------------------------------*/
/*                                                                    */
/* Um die folgenden  Funktionen nutzen zu koennen, muss zusaetzlich   */
/* zur nilib die nisellib und die filib gebunden werden.              */
/*                                                                    */
/*--------------------------------------------------------------------*/
/*                                                                    */
/* Neue Select-API:                                                   */
/* ===============                                                    */
/*                                                                    */
/* Das alte NiSelect ist dadurch sehr langsam, dass es einen          */
/* eigenen Typ fuer die selectmasken definiert. Diese muessen         */
/* bei jedem Aufruf in die socket-masken umgesetzt                    */
/* werden.                                                            */
/*                                                                    */
/* Die neue API verwaltet jetzt nur noch intern                       */
/* socket-Masken. Ausserdem kann der Aufrufer eigene Daten            */
/* hinterlegen, die den Zugriff auf seine Daten beschleunigt.         */
/*                                                                    */
/* Dafuer muessen die Selectsets jetzt angefordert und                */
/* explizit wieder freigegeben werden.                                */
/*                                                                    */
/* Ein typischer Ablauf waere:                                        */
/* {                                                                  */
/*   NISEL_HDL set;                                                   */
/*   SAP_BOOL  read;                                                  */
/*                                                                    */
/*   NiSelNewSet (&set);                                              */
/*                                                                    */
/*   NiSelSet (set, nihdl, NI_CONNECT, NULL);                         */
/*   [...]                                                            */
/*   while (something_to_do)                                          */
/*   {                                                                */
/*     NiSelSelect (set, timeout);                                    */
/*     while (NiSelNext (set, &read, NULL, &conn, &myclient)          */
/*            != NI_INVALID_HDL)                                      */
/*     {                                                              */
/*       if (conn)                                                    */
/*       {                                                            */
/*         MyConnect(); // hier wird irgendwo                         */
/*            // NiSelSet (set, newnihdl, NI_READ, myclient)          */
/*            // gerufen                                              */
/*       }                                                            */
/*       if (read)                                                    */
/*         Myread (myclient);// hier wird evtl.                       */
/*            // NiSelSet (set, myclient->nihdl, NI_WRITE,            */
/*            //           myclient)                                  */
/*            // gerufen                                              */
/*       if (write)                                                   */
/*         MyWrite (myclient);                                        */
/*     }                                                              */
/*   }                                                                */
/*   NiSelDestroySet (&set);                                          */
/* }                                                                  */
/*                                                                    */
/*-NiSnc--------------------------------------------------------NiSel-*/
/*                                                                    */
/* Transparente Integration von SNC (Secure Network Communication):   */
/* ===============================================================    */
/*                                                                    */
/* Es ist moeglich, NI-vebindungen verschluesseln zu lassen. Hierzu   */
/* wurde NI so erweitert, dass die SNC-Library transparent gerufen    */
/* wird.                                                              */
/* Die Loesung basiert auf NiBuf (s.o.). Es muessen NiInit, NiConnect */
/* und NiListen durch NiSncInit, NiSncConnect und NiBufListen ersetzt */
/* werden.                                                            */
/* NiSncInit erhaelt als Parameter den eigenen SNC-Namen.             */
/* NiConnect erhaelt den SNC-Namen des Partners. Falls Dieser leer    */
/* ist, verhaelt sich NiSncConnect wie NiBufConnect                   */
/* Nach dem NiAccept kann man, nachdem der Handle Schreib-oder        */
/* Lesebereit ist, per NiSncGetPeer erfragen, ob die Verbindung       */
/* verschluesselt ist und wie der Partner heisst                      */
/* Um diese Funktionen nutzen zu koennen, muss zusaetzlich zur nilib  */
/* die nisnclib gebunden werden. Diese braucht zusaetzlich die filib, */
/* snclib und sappar0/sapparam.                                       */
/*                                                                    */
/* NiSncInit          Ersetzt NiInit (muss gerufen werden)            */
/* NiSncConnect       Ersetzt NiConnect                               */
/* NiSnc2Connect      Ersetzt Ni2onnect                               */
/* NiSncGetPeer       Gibt den evtuellen SNC-Account des Partners     */
/*                    zurueck                                         */
/* NiSncRequest       Fordert eine SNC-Verschluesselung zu            */
/*                    bestehender Verbindung an                       */
/* NiSncAclStr        Gibt einen lesbaren String fuer                 */
/*                    einen SNC-Account zurueck                       */
/* NiSncNameToAcl     Macht aus einem lesbaren SNC-Namen einen        */
/*                    internen                                        */
/* NiSncAclCmp        Vergleicht zwei interne SNC-Namen               */
/* NiSncFinHS         Versucht den Austausch von SNC-Overhead         */
/*                    anzustossen                                     */
/*                                                                    */
/*--------------------------------------------------------------NiSnc-*/
/*-NiSendHandle-------------------------------------------------------*/
/*                                                                    */
/* Mit der Funktion NiSendHandle kann ein Ende einer socket-Verbindung*/
/* ueber eine zweite lokale socket-Verbindung verschickt werden.      */
/* Diese Funktionalitaet ist fuer Unix und Windows NT (mit WinSock2)  */
/* implementiert.                                                     */
/* Die Uebertragung erfolgt in den mehreren Einzelschritten. Der      */
/* client ruft NiSendHandle, um den socket zu uebertragen.            */
/* Die empfangende Seite muss den Socket, ueber den empfangen werden  */
/* soll, in der read-Maske haben. Die Uebertragung wird in der select */
/* loop des servers mit dem status 'connect' angezeigt. Der server    */
/* kann die Verbindung mit NiAccept annehmen und anschliesend ueber   */
/* die empfangene Verbindung kommunizieren.                           */
/*                                                                    */
/*-------------------------------------------------------NiSendHandle-*/
/*--------------------------------------------------------------------*/
/*                                                                    */
/* allgemeine Dienste                                                 */
/* ==================                                                 */
/* NiInit              Initialisierung der nilib                      */
/*                     Sollte immer vor Benutzen von Ni-Funktionen    */
/*                     gerufen werden.                                */
/* NiThrInit           Thread specific initialization                 */
/* NiExit              Aufrumen der Resourcen                        */
/* NiThrExit           Thread specific cleanup                        */
/* NiCloseHandle       Handle freigeben, evtl. Verbindung abbauen     */
/* NiCloseHandleKP     wie oben, UDS Pfad nicht loeschen              */
/* NiShutdownHandle    Verbindung einseitig schliessen                */
/* NiCloseAll          alle offenen Handles freigeben                 */
/* NiSelect            Auf Daten warten                               */
/* Ni2Select           Auf Daten und freie Sendeleitungen warten      */
/* NiPeek              Testen, ob Daten fuer Handle angekommen        */
/* NiWriteOK           Testen, ob Write auf Handle blockieren wuerde  */
/* NiRouteToTable      Zerlege einen Routestring                      */
/* NiAddrToHost        Adresse in Hostnamen umwandeln                 */
/* NiAddrToHostCanon   Adresse in canonischen Hostnamen umwandeln     */
/* NiServToNo          Service-Namen in (Port-)Nummer umwandeln       */
/* NiNoToServ          (Port-)Nummer in Service-Namen umwandeln       */
/* NiMyHostName        eigenen Hostnamen ermitteln                    */
/* NiMyAddrVerify      eigene Host-Adresse verifizieren               */
/* NiReadPending       noch zu lesende Bytes (letztes Paket)          */
/* NiWritePending      noch zu schreibende Bytes (letztes Paket)      */
/* NiWait              n Millisekunden warten                         */
/* NiGetParam          allg. NI-Parameter lesen                       */
/* NiSetParam          allg. NI-Parameter setzen                      */
/* NiGetParamEx        allg. NI-Parameter lesen (beliebige Datentypen)*/
/* NiSetParamEx        allg. NI-Parameter setzen(beliebige Datentypen)*/
/* NiGetSapDllRelease  gibt SAP Release der NIDLL zurueck             */
/* NiErrSet            Sap-Error-Info zu NI-Returncode bereitstellen  */
/* NiErrStr            Namens-String  zu Ni-Error-Konstante liefern   */
/* NiSecStr            SapRouter-Passwort aus String ausblenden       */
/* NiHdlToSock         Gebe den Socket zu einem NI-Handle zurueck     */
/* NiSockToHdl         Gebe den NiHandle zu einem Socket zurueck      */
/* NiHdlGetOutMsgs     Anzahl der tatsaechlich geschr. NI-Pakete      */
/*                                                                    */
/*                                                                    */
/* Dienste f. verbindungsorientierte Kommunikation (Stream, "Telefon")*/
/* ===================================================================*/
/* NiListen            Vorbereitungen fuer Connect-Requests von       */
/*                     Clients treffen (Server)                       */
/* NiConnect           Verbindung zu Server aufbauen                  */
/* NiAccept            Verbindung zu Client aufbauen                  */
/* NiWrite             Daten an Partner senden                        */
/* NiRead              Daten vom Partner empfangen                    */
/* NiCheck             Verbindung zum Partner ueberpruefen            */
/* Ni2Listen           wie NiListen, aber Service-Nummer als Input    */
/* Ni2Connect          wie NiConnect, Host-Adr + Serv-Nr als Input    */
/*                                                                    */
/* Alle Funktionen fuer Native TCP/IP Verbindungen nutzen direkt      */
/* die Systemfunktionen und benutzen NICHT die Pufferung (NiBuf):     */
/* NiRawConnect        wie NiConnect, Talkmode nach Verbindungsaufbau */
/*                     Native statt NI (fuer SapTelnet ..)            */
/* NiRawWrite          Daten an "Native-TCP-Partner" senden           */
/* NiRawRead           Daten von "Native-TCP-Partner" empfangen       */
/* NiRawPeek           Testen, ob Daten fuer Handle angekommen        */
/* NiRawWriteOK        Testen, ob Write auf Handle blockieren wuerde  */
/*                                                                    */
/*                                                                    */
/* Dienste fuer verbindungslose Kommunikation (Datagramm, "Brief")    */
/* ===============================================================    */
/* NiDgBind            Vorbereitungen fuer Empfang von Datagrammen    */
/*                     treffen                                        */
/* NiDgReceive         Datagramm empfangen                            */
/* NiDgSend            Datagramm senden                               */
/* NiDgGetHdl          intern benutzten Handle abfragen               */
/* NiDgFreeResrc       belegte "Datagramm-Resourcen" wieder freigeben */
/* NiDg2Send           wie NiDgSend, aber Host-Adresse als Input      */
/*                                                                    */
/*end-----------------------------------------------------------------*/

#ifndef NIXX_H
    #define NIXX_H

#ifdef __cplusplus
extern "C"
{
#endif


/*---------------------------------------------------------------------*/
/* Version der NI-Schicht                                              */
/*                                                                     */
/* NI_VERSION wird bei allen Aenderungen der NI-Schicht (inklusive     */
/* SapRouter) inkrementiert und bei NI-internen Messages uebers Netz   */
/* mitgesendet                                                         */
/*                                                                     */
/* 13  02.12.94 hq   Interface Erweiterung NiSetParam(), NiGetParam()  */
/* 14  21.12.94 hq   Host/Serv-Buf VALID/UNKNOWN/alloc/flush/dump      */
/*                   NiSetParam() BUF_OPCOD, HSBUF_MAX, LIB_VERS       */
/*                   NiAdrToStr(), NiSrvToStr() speichern 10 Strings   */
/*                   SapRouter Queue Pre-Alloc                         */
/* 15  03.01.95 hq   Interface Erweiterung NiMyAddrVerify ()           */
/* 16  16.03.95 hq   Interface Erweiterung NiRawConnect()              */
/*                   fragmentierte Header jetzt erlaubt                */
/* 17  25.04.95 hq   niping client delay wg. keepalive Tests           */
/*                   SapRouter Def-Werte fuer -B /-Q von 100KB / 10MB  */
/*                   auf 500 KB / 20 MB                                */
/*     22.05.95 hq   Def. fuer Routtab-Laenge von 1000 auf 2000        */
/* 18  06.09.95 hq   Maske von 21 auf 32 Int's erweitert               */
/*     18.09.95 hq   NiErrStr () aufgenommen                           */
/*     05.10.95 hq   NiSecStr () aufgenommen                           */
/*     16.10.95 hq   NiGetSapDllRelease () aufgenommen                 */
/* 19  17.10.95 hq   NiPBind () und NiBind () erweitert (log-Param)    */
/*                   SapRouter Portrange-Feature                       */
/* 20  30.10.95 hq   SapRouter Bug unter NT gefixt (close in NiRAdmin) */
/* 21  13.11.95 hq   ErrInfo-Makros ueberall eingebaut                 */
/*                   NiPing SNC-Schicht integriert                     */
/*                   SAProuter RT_ROUTTAB_MAX_NO von 2000 auf 3000     */
/*                   SAProuter Permission Field Gross / Klein          */
/*     07.12.95 hq   Patch wg. ErrInfo in NiICloseHandle ...           */
/* 22  11.12.95 hq   Vorbereitung fur Umstellung auf Handshake-Protok. */
/*                   NiIConnect als Zustandsautomat                    */
/* 23  20.12.95 hq   Bug in Connect-Fktn behoben (INVALID_HDL setzen)  */
/* 24  27.02.96 hq   ErrInfo analog errno-Konzept, ni_compname global  */
/*                   API's setzen bei Fehlern immer ErrInfo, machen    */
/*                   nie ErrClear, ErrSet ueberschreibt immer          */
/*                   SAProuter bug (Routtab vorm Neueinlesen loeschen) */
/* 25  01.04.96 hq   SAProuter kommt runter, wenn keine Routtab da     */
/*                   oder wenn kein Log geschrieben werden kann        */
/* 26  06.05.96 cr   NiTcpConnect realisiert                           */
/*                   Passwordverfahren fr SAPSERVX                    */
/*                   Saprouter von sapparam unabhngig                 */
/*     15.10.96	cr   NiRouteToTable                                    */
/*     29.10.96 cr   Saprouter laesst sich auch bei maximaler Anzahl   */
/*                   von clients administrieren		               */
/*                   Default fuer die maximale Anzahl erhoeht auf 800  */
/*     05.11.96 cr   Mehr Sicherheit bei der maximalen Anzahl	       */
/*		     Dynamische Anpassung der Routtab		       */
/*		     NI_MAX_HDLS auf 2048			       */
/*     02.12.96 cr   Saprouter gibt Statusmeldung in Admin-requests    */
/*                   zurueck					       */
/*     04.02.97 cr   Saprouter ersetzt in der iroute seinen Hostname   */
/*                   durch die IP-Addresse des Vorgaengers	       */
/* 27  12.11.96 rw   NiPRead, NiPWrite bekommen jetzt Flag fuer	       */
/*                   nonblocking				       */
/*     02.01.97 cr   Threadsafe: NiRouteAdd, NiRouteConnect gestrichen */
/*     31.01.97 hq   fragmentierte Header werden sauber behandelt      */
/*	             Fragment-Test-Modus gebaut (siehe NiSetParam (),  */
/*	             niping -Q)                                        */
/*     12.02.97 cr   Threadsafe als Compileroption		       */
/* 28  21.03.97 hq   SNC in SAProuter eingebaut (Alpha Version !)      */
/*     25.04.97 cr   SNC in Saprouter erweitert                        */
/*                   NiSel-API realisiert       (Alpha Version !)      */
/*     20.06.97 cr   Administration des Saprouter von Remote im Server */
/*                   verhindern                                        */
/* 29  11.07.97 cr   NiBuf und NiSnc realisiert und im Saprouter       */
/*                   integriert                                        */
/* 30  23.07.97 cr   Soft Shutdown im Router                           */
/*                   Administration ueber neue Requests anfordern      */
/*                   Workingdirectory und saprouttab im INFO-request   */
/*                   ausgeben                                          */
/*     29.07.97 cr   NiSnc, Raw-Modus und Test von NiBuf in niping     */
/*     13.08.97 cr   NiRawRead gab zu frueh NIECONN_BROKEN zurueck     */
/*     15.08.97 cr   Mehrfache FD_SET's vermeiden fuer NT              */
/*                   NiBuf mit in die nilib                            */
/*     03.09.97 cr   NI_FINDSERV_OS implementiert fuer sapftp          */
/*                   REUSEADDR unter NT abgeschaltet                   */
/*     21.10.97 cr   Parsing der Source-Addresse im Router repariert   */
/* 31  31.10.97 cr   NiQueue implementiert                             */
/*     27.11.97 cr   Nonblocking Connect                               */
/*     03.12.97 cr   Server koennen jetzt auch Verschluesselung        */
/*                   nachfordern.                                      */
/*                   Hostnamenpuffer kann das Retry fuer revers        */
/*                   lookups abschalten                                */
/*                   Local bind ist jetzt moeglich.                    */
/*                   (diese drei Features werden im Saprouter jetzt    */
/*                   genutzt)                                          */
/*     11.12.97 cr   NiBufWrap/NiBufDup                                */
/*                   Nonblocking connect mit extra Funktion            */
/*     12.01.98 cr   EWOULDBLOCK auch pruefen                          */
/* 32  17.09.98 cr   Local port selection fixed                        */
/*                   Saprouter uses NT-Eventlog for fatal errors       */
/* 33  19.02.98 cr   NiDgSend optimiert                                */
/*                   AF_UNIX fuer localhost-Kommunikation              */
/*     11.05.98 cr   NiDgSend war nicht optimal                        */
/*                   Ni{Host,Net}INT in nixx.h                         */
/*     16.07.98 cr   Interne Handles auf Pointer umgestellt            */
/*                   Queuelimit jetzt per Packet und nicht per Byte    */
/*     18.02.99 ol   NiGetHostAddrList eingebaut                       */
/*      7.06.99 ol   NiCheck2 implementiert                            */
/*      8.07.99 ol   Konfigurierbarer Pfad fuer Unix Domain Sockets    */
/* 34  30.08.99 ol   SNC Fehler mit SAProuter (NIVERSION 34)           */
/*      9.08.99 ol   NiRawPeek und NiRawWriteOK implementiert. NiPeek  */
/*                   und NiWriteOK fuellen intern die Puffer, die dann */
/*                   fuer NiRawRead verloren gehen.                    */
/*     11.10.99 ol   NiShutdownHandle implementiert                    */
/*     30.03.00 cps  NiSncFinHS implementiert                          */
/*                                                                     */
/*---------------------------------------------------------------------*/

#define NI_COMPNAME		cU("NI (network interface)")
#define NI_VERSION		34
#define NI_LAST_MODIFIED	cU("August 30 1999")

/*--------------------------------------------------------------------*/
/* constants                                                          */
/*--------------------------------------------------------------------*/

/* Fehler-Konstanten--------------------------------------------------*/
#define NIEINTERN            -1    /* NI-interner Fehler              */
#define NIEHOST_UNKNOWN      -2    /* Hostname unbekannt              */
#define NIESERV_UNKNOWN      -3    /* Service unbekannt               */
#define NIESERV_USED         -4    /* Service already used            */
#define NIETIMEOUT           -5    /* Zeitlimit ueberschritten        */
#define NIECONN_BROKEN       -6    /* Verbindung zum Partner abgebr.  */
#define NIETOO_SMALL         -7    /* Daten-bereich zu klein          */
#define NIEINVAL             -8    /* ungueltiger Parameter           */
#define NIEWAKEUP            -9    /* Wake-Up (ohne Daten)            */
#define NIECONN_REFUSED      -10   /* Verbindungsaufbau misslungen    */
#define NIEPING              -11   /* PING/PONG-Signal erhalten       */
#define NIECONN_PENDING      -12   /* Verbindung zum Partner ueber    */
				   /* NiRouter noch nicht aufgebaut   */
#define NIEVERSION           -13   /* ungueltige Version              */
#define NIEMYHOSTNAME        -14   /* eigener Hostname nicht ermittelb*/
#define NIENOFREEPORT        -15   /* kein freier Port innerhalb Range*/
#define NIEMYHOST_VERIFY     -16   /* eigener Hostname ungueltig      */
#define NIESNC_FAILURE       -17   /* Fehler in der SNC Schicht       */
				   /* im saprouter ==>                */
#define NIEOPCODE            -18   /* Opcode received                 */
#define NIEQUE_FULL          -19   /* Queue limit erreicht, naechstes */
                                   /* Paket wird nicht akzeptiert     */ 
#define NIETOO_BIG           -20   /* Angefordertes Paket zu gross    */
#define NIEACCEPT            -21   /* Accepted handle pending         */
#define NIEPASSED            -22   /* Handle passed to other process  */
#define NIENO_ANSWER         -23   /* Partner hat nicht geantwortet   */

#define NIEROUT_HOST_UNKNOWN -90   /* Hostname unbekannt              */
#define NIEROUT_SERV_UNKNOWN -91   /* Service unbekannt               */
#define NIEROUT_CONN_REFUSED -92   /* Verbindungsaufbau misslungen    */
#define NIEROUT_INTERN       -93   /* NI-interner Fehler              */
#define NIEROUT_PERM_DENIED  -94   /* Connect von Source an           */
				   /* Destination verboten            */
#define NIEROUT_CONN_BROKEN  -95   /* Verbindung abgebrochen          */
#define NIEROUT_VERSION      -96   /* ungueltige Client-Version       */
#define NIEROUT_CANCELED     -97   /* Verbindung von Administrator    */
				   /* ge"canceled"                    */
#define NIEROUT_SHUTDOWN     -98   /* saprouter shutdown              */
#define NIEROUT_INFO_DENIED  -99   /* Informations Request abgelehnt  */
#define NIEROUT_OVERFLOW    -100   /* max. Anzahl Clients erreicht    */
#define NIEROUT_MODE_DENIED -101   /* talkmode not allowed            */
#define NIEROUT_NOCLIENT    -102   /* client nicht vorhanden          */
#define NIEROUT_EXTERN      -103   /* error in external library       */
#define NIEROUT_SNC_FAILURE -104   /* Fehler in der SNC Schicht       */

#define NI_ISROUTERR(nirc)   ((nirc) <= NIEROUT_HOST_UNKNOWN)
#define NI_ISLIBERR(nirc)    ((nirc) == NIEROUT_SNC_FAILURE || \
			      (nirc) == NIESNC_FAILURE)
#define NI_ISNONFATAL(nirc)  (NI_ISLIBERR(nirc) ||    \
                              (nirc) == NIEPING ||    \
                              (nirc) == NIEVERSION || \
                              (nirc) == NIEROUT_VERSION)

/* Konstante fuer Timeout-Parameter-----------------------------------*/
#define NI_BLOCK             -1
#define NI_ASYNC              0

/* Konstanten fuer servno-Parameter-----------------------------------*/
#define NI_FINDSERV         (SAP_USHORT) -1 /* Hier suchen wir selber */
#define NI_FINDSERV_OS      (SAP_USHORT) 0  /* Hier sucht das OS      */

/* Konstanten fuer servno-Parameter-----------------------------------*/
#define NI_INVALID_HDL       (NI_HDL) -1
#define NI_INVALID_PTR       (nitab+NI_MAX_HDLS)
#define NI_INVALID_SOCK     -1

/* Default fuer SapRouter-Service-------------------------------------*/
#define NI_ROUTER_SERVICE    "sapdp99"  
#define NI_ROUTER_SERVNO     cU("3299") 

/* Separatoren fuer Routen im Hostnamen-------------------------------*/
#define   NI_HOSTSEP   cU("/H/")  /* Separator fuer Hostname in Route */
#define   NI_SERVSEP   cU("/S/")  /* Separator fuer Service in Route  */
#define   NI_PASSSEP   cU("/P/")  /* Separator fuer Passwort in Route */
#define   NI_PAS2SEP   cU("/W/")  /* Separator fuer Passwort in Route */
#define   NI_KRPTSEP   "/K/"      /* Separator fuer SNC-Name          */
/* reservierte Separatoren fuer zukuenftige Entwicklung, momentan 'extern'
   in commandline parser NT sapgui, JAVAgui implementiert -> M.Tewes  */
#define   NI_SYSTSEP   "/R/"      /* Separator fuer Systemname R/3 ('MLP') */
#define   NI_GRUPSEP   "/G/"      /*                Gruppenname ('PUBLIC') */
#define   NI_ALIASEP   "/A/"      /*                Aliasname f. Saprouter */
#define   NI_MSSVSEP   "/M/"      /*                Messageserver name     */

#define   NI_SEPSIZ    3

/* Parameter fuer NiSel/NiPeek */
#define NI_READ             (SAP_RAW) 1
#define NI_WRITE            (SAP_RAW) 2
#define NI_CONNECT          (SAP_RAW) 4

/* Parameter fuer NiShutdownHandlek */
#define NI_SHUT_RD          0
#define NI_SHUT_WR          1
#define NI_SHUT_RDWR        2


/* Parameter fuer Ni?etHdlParam					      */
/* Beschreibung: siehe Ni?GetParameterEx                              */
typedef enum {
    NIHP_MODE,                  /* I/O Modus (RAW)                    */
    NIHP_SINGLE_READ,           /* NT only: Einzelschritt (INT)       */
    NIHP_CHECK_WAIT,            /* PING/PONG Synchronisation (INT)    */
    NIHP_CHECK_QUE,             /* check queue flag (RAW)             */
    NIHP_ALLOC_FUNC,            /* NIBUF_ALLOC_FUNC                   */
    NIHP_MAXQUEUE,              /* maximale Anzahl Pakete  (UINT)     */
    NIHP_WARNQUEUE,             /* minimale Anzahl verf. Pakete (UINT)*/
    NIHP_BUFSTAT                /* Bufferstatus: ON / OFF             */
} NIHDL_PARAM;

typedef enum {
    NIBUFP_MAXHEAP = 100        /* maximaler HEAP-Bedarf (UINT)       */
} NIBUF_PARAM;

/* Parameter fuer NIHP_MODE in Ni?etHdlParam                          */
#define NI_MESG_IO          (SAP_RAW) 0  /* message-orientierte Komm. */
#define NI_RAW_IO           (SAP_RAW) 1  /* stream-orient. Kommunik.  */
#define NI_ROUT_IO          (SAP_RAW) 2  /* spez. Mode fuer Saprouter */

/* Parameter fuer NIHP_BUFSTAT in Ni?etHdlParam                       */
#define NI_BUFSTAT_OFF      (SAP_RAW) 0  /* Pufferung explizit aus    */
#define NI_BUFSTAT_ON       (SAP_RAW) 1  /* Pufferung an              */

/* maximale Laenge des Pfades fuer Unix Domain Sockets                */
#define NI_UDS_PATH_LEN     108

/* Datentypen fuer die Parameter                                      */
/* In C++ waeren dies einfach ueberladene Funktionen...               */
#define NI_DT_UNKNOWN           0
#define NI_DT_CHAR              1
#define NI_DT_INT               2
#define NI_DT_UINT              3
#define NI_DT_SHORT             4
#define NI_DT_USHORT            5
#define NI_DT_RAW               6
#define NI_DT_BOOL              7
#define NI_DT_A7                8
#define NI_DT_RAWPTR            9
#define NI_DT_UCPTR             10
#define NI_DT_VOIDPTR           11
#define NI_DT_ALLOCFUNC         12


/*--------------------------------------------------------------------*/
/* typedefs                                                           */
/*--------------------------------------------------------------------*/

/*------------------------------------------------------------*/
/* Neue Puffer-API:                                           */
/*                                                            */
/*------------------------------------------------------------*/

typedef struct NIBUFFER * NIBUF_P;
typedef void NIBUF_FREE_FUNC (NIBUF_P *buf);

enum NIBUFFER_TYPE
{
    NIBUF_NORMAL,
    NIBUF_REFERENZ,
    NIBUF_WRAPPED
};

typedef struct NIBUFFER
{
    /* public data */
    void       *data;           /* Zeiger auf Userdaten               */
    SAP_INT     len;            /* Laenge der Userdaten               */

    /* private data */
#if defined(NI_PRIVATE_DATA)
    void       *space;          /* Zeiger auf Datenbereich            */
    SAP_INT     spacelen;       /* Laenge des Datenbereichs           */
# ifdef LST_NODE
    enum NIBUFFER_TYPE type;
    SAP_USHORT  refcount;
    union {
        struct NIBUFFER *ref;
        NIBUF_FREE_FUNC *freef;
    } u;
    void *uptr;
    LST_NODE (struct NIBUFFER, queue); /* Queueliste                  */
# endif /* LST_NODE */
#endif /* NI_INTERNAL */
} NIBUFFER;

typedef SAPRETURN NIBUF_ALLOC_FUNC (NIBUFFER **buf,
                                    SAP_INT    len,
                                    SAP_UC    *text);
                                  
typedef struct NISEL_SET *NISEL_HDL;

typedef SAP_INT         NI_HDL;

/*--------------------------------------------------------*/
/* Struktur fuer Host-Adresse (nur um bei den Funktions-  */
/* Parametern Typ-Pruefungen zu ermoeglichen)             */
/*--------------------------------------------------------*/

#if !defined  (NI_IADR_LN) && !defined (__NTUTIL_H_)
#define NI_IADR_LN           4

typedef struct /* NI_HOSTADR */
{
    SAP_RAW       addr[NI_IADR_LN];
}   NI_HOSTADR;

#define NI_HOSTADR_SIZ       sizeofR (NI_HOSTADR)
#endif

/*---------------------------------------------------------------------*/
/* Struktur fuer Routenanalyse                                         */
/*---------------------------------------------------------------------*/

#define NI_HOSTLN (size_t) 60
#define NI_SERVLN (size_t) 30
#define NI_PASSLN (size_t) 20

typedef struct /* NI_ROUTE_TABLE */
{
    SAP_UC hostname[NI_HOSTLN];
    SAP_UC service [NI_SERVLN];
} NI_ROUTE_TABLE;


#define NI_SET_NO_ADR(adr)       (memsetR (&(adr), 0, NI_HOSTADR_SIZ))
#define NI_IS_NO_ADR(adr)        (((adr).addr[0] == 0) && \
				  ((adr).addr[1] == 0) && \
				  ((adr).addr[2] == 0) && \
				  ((adr).addr[3] == 0))


/*--------------------------------------------------------*/
/* Struktur fuer Select-Masken                            */
/*--------------------------------------------------------*/

#define NI_IMASK_LN          64      /* number of SAP_INT's in a mask  */

#define BITS_PER_INT         (sizeofR (SAP_INT) * 8)

#define NI_MAX_HDLS          (NI_IMASK_LN * BITS_PER_INT)

typedef struct /* NI_MASK */
{
    SAP_INT       hdl_count;
    SAP_INT       mask[NI_IMASK_LN];
}   NI_MASK;

#define NI_MASK_SIZ          sizeofR (NI_MASK)

/*--------------------------------------------------------*/
/* Parameter fuer NiP?etEx                                */
/*--------------------------------------------------------*/
typedef struct /* NI_PARAM_DATA */
{
    SAP_RAW                     param;  /* betroffener Parameter      */
    SAP_RAW                     datatype;/* Datentyp des Parameters   */
    union {                             /* angehaengte Daten          */
        SAP_CHAR                char_val;
        SAP_INT                 int_val;
        SAP_UINT                uint_val;
        SAP_SHORT               short_val;
        SAP_USHORT              ushort_val;
        SAP_RAW                 raw_val;
        SAP_BOOL                bool_val;
        SAP_A7                  a7_val;
        SAP_RAW                 *rawptr_val;
        SAP_UC                  *ucptr_val;
        void                    *voidptr_val;
        NIBUF_ALLOC_FUNC        *allocfunc_val; /* SAPUNICODEOK       */
    } value;
}   NI_PARAM_DATA;

#define NI_PARAM_DATA_LN  sizeofR (NI_PARAM_DATA)


/*--------------------------------------------------------*/
/* Groessere Packete kann NiBuf nicht verarbeiten         */
/*--------------------------------------------------------*/

#define NIBUF_MAXALLOC (8*1024*1024)

/*--------------------------------------------------------*/
/* Route = Host/Service-Liste (NiRouteConnect)            */
/*--------------------------------------------------------*/

typedef struct ni_iroute *NI_ROUTE;

/*--------------------------------------------------------*/
/* Struktur fuer interne SNC-Namen                        */
/*--------------------------------------------------------*/

#define NISNC_ACLKEY_MAXLEN 1024

typedef struct NISNC_ACLKEY {
    SAP_INT  len;
    SAP_RAW  key[NISNC_ACLKEY_MAXLEN];
} NISNC_ACLKEY;


/*--------------------------------------------------------------------*/
/* macros                                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------*/
/* Macros zum Manipulieren von Select-Masken              */
/*--------------------------------------------------------*/

/*                                                        */
/* Inhalt einer Maske auf HEX 0 setzen                    */
/*                                                        */
#define NI_CLEAR_MASK(M) \
    (memsetR (&(M), 0, sizeofR(M)))

/*                                                        */
/* Anzahl der gesetzten Bits in Maske liefern             */
/*                                                        */
#define NI_COUNT_MASK(M)      ((M).hdl_count)

/*                                                        */
/* in Maske 'mask' Bit fuer Handle 'hdl' setzen           */
/*                                                        */
#define NI_ADDTO_MASK(M, hdl) \
    { if (NI_HDL_OK (hdl) && (! NI_ISIN_MASK (M, hdl))) \
      { (M).mask [hdl / BITS_PER_INT] |= (1 << (hdl % BITS_PER_INT)); \
	(M).hdl_count++; } }

/*                                                        */
/* in Maske 'mask' Bit fuer Handle 'hdl' loeschen         */
/*                                                        */
#define NI_DELFROM_MASK(M, hdl) \
    { if (NI_ISIN_MASK (M, hdl)) \
      { (M).mask [hdl / BITS_PER_INT] &= ~(1 << (hdl % BITS_PER_INT)); \
	(M).hdl_count--; } }

/*                                                        */
/* Ist in Maske 'mask' Bit fuer Handle 'hdl' gesetzt ?    */
/*                                                        */
#define NI_ISIN_MASK(M, hdl) \
    (NI_HDL_OK (hdl) ? \
       ((M).mask [hdl / BITS_PER_INT] & (1 << (hdl % BITS_PER_INT))) \
       : FALSE )

/*                                                        */
/* Pruefen, ob Handle zu gross fuer Maske                 */
/*                                                        */
#define NI_HDL_OK(hdl) \
    (((hdl >= 0) && (hdl < NI_MAX_HDLS)) ? TRUE :FALSE)

#define NI_PTR_OK(hdl) \
    (((hdl >= nitab) && (hdl < nitab + NI_MAX_HDLS)) ? TRUE :FALSE)

#define NI_CHK_HDL(handle, func) \
    if (! NI_HDL_OK (handle)) TRCRET ((tf, cU("%s: invalid handle (%d)"),\
                                 cU(#func), (int) handle), NIERR(NIEINVAL))

/* aus Kompatibilitaetsgruenden: */
#ifdef SAPonNT
#  define NI_SOCK_OK(x) ((x) != NI_INVALID_SOCK)
#else
#  define NI_SOCK_OK     NI_HDL_OK
#endif


/*--------------------------------------------------------------------*/
/* function declarations                                              */
/*--------------------------------------------------------------------*/


/*--------------------------------------------------------*/
/* NiInit                                                 */
/* NiBufInit                                              */
/*                                                        */
/* Initialisierung der nilib                              */
/*                                                        */
/* INPUT                                                  */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar (Daten da)            */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiInit	   (void);
#define NiBufInit() NiInit()
#define NiBufReInit() SAP_O_K
SAPRETURN API_DF NiSncInit (SAP_UC * my_name);

/*--------------------------------------------------------*/
/* NiThrInit                                              */
/*                                                        */
/* Thread specific initialization                         */
/*                                                        */
/* INPUT                                                  */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar (Daten da)            */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiThrInit (void);

/*--------------------------------------------------------*/
/* NiExit                                                 */
/*                                                        */
/* Verlassen der nilib                                    */
/*                                                        */
/* INPUT                                                  */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar (Daten da)            */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiExit	 (void);

/*--------------------------------------------------------*/
/* NiThrExit                                              */
/*                                                        */
/* Thread specific cleanup                                */
/*                                                        */
/* INPUT                                                  */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/*--------------------------------------------------------*/

void API_DF NiThrExit (void * data);

/*--------------------------------------------------------*/
/* NiListen ()                                            */
/* Ni2Listen ()                                           */
/* Ni3Listen ()                                           */
/* NiBufListen ()                                         */
/* NiBuf2Listen ()                                        */
/* NiBuf3Listen ()                                        */
/*                                                        */
/* Vorbereitungen fuer Connect-Requests von Clients       */
/* treffen (Server)                                       */
/*                                                        */
/* INPUT   servname      log. Name des Service (NiListen) */
/*                       (null-terminiert)                */
/*                                                        */
/* OUTPUT  handle        Handle fuer NiAccept- und        */
/*                       NiSelect- Aufrufe                */
/*         servno        Nummer des Service (Ni2Listen)   */
/*                       Wird automatisch vergeben, wenn  */
/*                       *servno == NI_FINDSERV           */
/*                       (Host-Byte-Order)                */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIESERV_UNKNOWN       log. Servicename unbekannt       */
/* NIESERV_USED          Service schon benutzt            */
/* NIEMYHOST_VERIFY      locadr ist keine locale Addresse */
/* NIEINVAL              Handle zu gross fuer NI-Maske    */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiListen      (SAP_UC   *  servname,
                                NI_HDL   *  handle);

SAPRETURN API_DF Ni2Listen     (SAP_USHORT *  servno,
                                NI_HDL     *  handle);

SAPRETURN API_DF Ni3Listen     (SAP_USHORT *servno,
                                NI_HOSTADR *locadr,
                                NI_HDL     *handle);

SAPRETURN API_DF NiBufListen   (SAP_UC   *  servname,
                                NI_HDL   *  handle);

SAPRETURN API_DF NiBuf2Listen  (SAP_USHORT *  servno,
                                NI_HDL     *  handle);

SAPRETURN API_DF NiBuf3Listen  (SAP_USHORT *servno,
                                NI_HOSTADR *locadr,
                                NI_HDL     *handle);


/*--------------------------------------------------------*/
/* NiAccept ()                                            */
/*                                                        */
/* Connect-Request von Client entgegennehmen und          */
/* Verbindung aufbauen (Server)                           */
/*                                                        */
/* INPUT   inhandle      Handle von NiListen              */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* OUTPUT  outhandle     Handle fuer Kommunikation mit    */
/*                       Client                           */
/*         hostaddr      Adresse des Client-Host          */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINVAL              inhandle ungueltig oder          */
/*                       outhandle zu gross fuer NI-Maske */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiAccept       	 (NI_HDL        inhandle,
					  SAP_INT       timeout,
					  NI_HDL     *  outhandle,
					  NI_HOSTADR *  hostaddr);


/*--------------------------------------------------------*/
/* NiConnect                                              */
/* Ni2Connect    Host-Adr + Serv-Nr als Input             */
/* NiRawConnect  Talkmode nach Verbindungsaufbau          */
/*               Native statt NI (fuer SapTelnet..)       */
/* NiBufConnect                                           */
/* NiBuf2Connect    Host-Adr + Serv-Nr als Input          */
/*                                                        */
/* Verbindung zum Server aufbauen (Connect-Request senden)*/
/*                                                        */
/* INPUT   hostname      Name des Server-Host (NiConnect) */
/*                       (null-terminiert)                */
/*         hostaddr      Adr des Server-Host (Ni2Connect) */
/*         servname      Name des Service (NiConnect)     */
/*                       (null-terminiert)                */
/*         servno        Nummer des Service (Ni2Connect)  */
/*                       (Host-Byte-Order)                */
/*         timeout       in MilliSekunden > 0             */
/*                       oder NI_BLOCK                    */
/*                                                        */
/* OUTPUT  handle        Handle fuer Kommunikation mit    */
/*                       Server                           */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEHOST_UNKNOWN       log. Hostname unbekannt          */
/* NIESERV_UNKNOWN       log. Servicename unbekannt       */
/* NIECONN_REFUSED       Verbindungsaufbau misslungen     */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINVAL              Handle zu gross fuer NI-Maske    */
/*                       oder timeout == 0                */
/* NIENOFREEPORT         kein freier Port innerhalb Range */
/*                       (siehe NiLocPortRange)           */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*                                                        */
/* Im Unix muss der timeout in diesem Fall ueber alarm () */
/* implementiert werden, der nur in Sekunden angegeben    */
/* werden kann. Bei timeout-Werten bis 1000 wird intern   */
/* 1 Sekunde genommen, ab 1000 wird abgerundet.           */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiConnect   (SAP_UC   *  hostname,
			       SAP_UC   *  servname,
			       SAP_INT     timeout,
			       NI_HDL   *  handle);
SAPRETURN API_DF  Ni2Connect   (NI_HOSTADR *  hostaddr,
				SAP_USHORT    servno,
				SAP_INT       timeout,
				NI_HDL     *  handle);
SAPRETURN API_DF  NiRawConnect (SAP_UC   *  hostname,
				SAP_UC   *  servname,
				SAP_INT     timeout,
				NI_HDL   *  handle);

SAPRETURN API_DF NiBufConnect  (SAP_UC     *hostname,
                                SAP_UC     *servname,
                                SAP_INT     timeout,
                                NI_HDL     *hdl);
SAPRETURN API_DF NiBuf2Connect (NI_HOSTADR *dest_adr,
                                SAP_USHORT  nservno,
                                SAP_INT     timeout,
                                NI_HDL     *hdl);
SAPRETURN API_DF NiBufRawConnect (SAP_UC   *  hostname,
                                  SAP_UC   *  servname,
                                  SAP_INT     timeout,
                                  NI_HDL   *  handle);

SAPRETURN API_DF NiSncConnect  (SAP_UC     *hostname,
                                SAP_UC     *servname,
                                SAP_UC     *peername,
                                SAP_INT     timeout,
                                NI_HDL     *hdl);
SAPRETURN API_DF NiSnc2Connect (NI_HOSTADR  *dest_adr,
                                SAP_USHORT   nservno,
                                NISNC_ACLKEY *peer,
                                SAP_INT      timeout,
                                NI_HDL      *hdl);

/*--------------------------------------------------------*/
/* NiSelect ()                                            */
/*                                                        */
/* Auf ankommende Daten warten                            */
/*                                                        */
/* INPUT   timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* IN/OUT  readmask      Maske der zu ueberwachenden      */
/*                       Handles (Masken-Macros verwenden)*/
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

#define NiSelect(mask, timeout) Ni2Select (mask, NULL, timeout)

/*--------------------------------------------------------*/
/* Ni2Select ()                                           */
/*                                                        */
/* Auf ankommende Daten warten und freie Sendeleitungen   */
/* warten                                                 */
/*                                                        */
/* INPUT   timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* IN/OUT  readmask      Maske der zu ueberwachenden      */
/*                       Read-Handles                     */
/*         writemask     Maske der zu ueberwachenden      */
/*                       Write-Handles                    */
/*                       (Masken-Macros verwenden)        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  Ni2Select      	 (NI_MASK *  readmask,
					  NI_MASK *  writemask,
					  SAP_INT    timeout);


/*--------------------------------------------------------*/
/* NiPeek ()                                              */
/*                                                        */
/* Testen, ob Daten auf einer best. Leitung angekommen    */
/*                                                        */
/* INPUT   handle        zu ueberpruefender Handle        */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar (Daten da)            */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiPeek         	 (NI_HDL   handle,
					  SAP_INT  timeout);


/*--------------------------------------------------------*/
/* NiWriteOK ()                                           */
/*                                                        */
/* Testen, ob Write auf Handle blockieren wuerde          */
/*                                                        */
/* INPUT   handle        zu ueberpruefender Handle        */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar (Daten da)            */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiWriteOK      	 (NI_HDL   handle,
					  SAP_INT  timeout);


/*--------------------------------------------------------*/
/* NiWrite ()                                             */
/*                                                        */
/* Daten an Partner senden                                */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         buf           Daten-Puffer                     */
/*         buflen        Laenge der zu versendenden Daten */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                       Bezieht sich auf das Senden der  */
/*                       internen Datenpakete, aus denen  */
/*                       sich die Nachricht zusammensetzt)*/
/*                                                        */
/* OUTPUT  writelen      Laenge der gesendeten Daten      */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINVAL              handle ungueltig                 */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiWrite        	 (NI_HDL      handle,
					  void     *  buf,
					  SAP_INT     buflen,
					  SAP_INT     timeout,
					  SAP_INT  *  writelen);


/*--------------------------------------------------------*/
/* NiRead ()                                              */
/*                                                        */
/* Daten vom Partner empfangen                            */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         buf           Daten-Puffer                     */
/*         maxlen        Groesse des Daten-Puffers        */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                       Bezieht sich auf den Empfang der */
/*                       internen Datenpakete, aus denen  */
/*                       sich die Nachricht zusammensetzt)*/
/*                                                        */
/* OUTPUT  readlen       Laenge der empfangenen Daten     */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETOO_SMALL          Puffer zu klein fuer Nachricht   */
/*                       (nachfolgende NiRead-Calls       */
/*                       liefern Rest des NI-Pakets)      */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/*                       (nachfolgende NiRead-Calls       */
/*                       liefern Rest des NI-Pakets)      */
/* NIEPING               Partner hat Ping-Signal gesendet */
/*                       (keine Daten, readlen = 0)       */
/* NIEINVAL              handle ungueltig                 */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/* NIEROUT_HOST_UNKNOWN  im nirouter: log. Hostname unbek.*/
/* NIEROUT_SERV_UNKNOWN  im nirouter: log. Service unbek. */
/* NIEROUT_CONN_REFUSED  im nirouter: Verb.aufbau missl.  */
/* NIEROUT_CONN_BROKEN   im nirouter: Verbindung abgebr.  */
/* NIEROUT_PERM_DENIED   im nirouter: route nicht erlaubt */
/* NIEROUT_VERSION       im nirouter: unguelt. Clnt-Vers. */
/* NIEROUT_CANCELED      im nirouter: Route von Administr.*/
/*                                    ge"canceled"        */
/* NIEROUT_SHUTDOWN      im nirouter: Router Shutdown     */
/* NIEROUT_INTERN        im nirouter: interner Fehler     */
/*                                                        */
/*                                                        */
/* Auf Betriebssystemen, bei denen der Abbruch einer      */
/* Verbindung zu einem Signal (SIGPIPE) fuehrt, muss der  */
/* Aufrufer selbst fuer die Signalbehandlung sorgen (z.B. */
/* auf SIG_IGN setzen)                                    */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiRead         	 (NI_HDL      handle,
					  void     *  buf,
					  SAP_INT     maxlen,
					  SAP_INT     timeout,
					  SAP_INT  *  readlen);


/*--------------------------------------------------------*/
/* NiCheck ()                                             */
/*                                                        */
/* Verbindung zum Partner ueberpruefen.                   */
/* NiCheck sendet ein "Ping"-Signal zum Partner. Ist die  */
/* Verbindung o.k., bekommen beide Partner bei einem der  */
/* naechsten NiRead-Calls den Returncode NIEPING (ohne    */
/* Daten).                                                */
/* (NiRead sendet automatisch Antwort auf Ping)           */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiCheck        	 (NI_HDL      handle,
					  SAP_INT     timeout);

/*--------------------------------------------------------*/
/* NiCheck1 ()						  */
/*                                                        */
/* Wie Nicheck, nur, dass keine Antwort zurueckkommt      */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiCheck1       	 (NI_HDL      handle,
					  SAP_INT     timeout);

/*--------------------------------------------------------*/
/* NiCheck2 ()						  */
/*                                                        */
/* Wie Nicheck, nur wird intern der Status gespeichert    */
/* und nicht NIEPING zurueckgegeben. Beim naechsten Aufruf*/
/* wird der Status zurueckgeliefert. Funktioniert nur     */
/* mit Verwendung der gepufferten Schnittstelle!          */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* RETURN                                                 */
/*         count         Anzahl Tests ohne Antwort        */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIENO_ANSWER          bisher keine Antwort vom Partner */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINVAL              illegale Parameter               */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiCheck2       	 (NI_HDL      handle,
					  SAP_INT     timeout,
                                          SAP_INT     *count);


/*--------------------------------------------------------*/
/* NiCheck2Cancel ()                                      */
/*                                                        */
/* Diese Funktion setzt den Status von NiCheck2 zurueck.  */
/* Damit werden bereits eingegangene Antworten vom        */
/* Partner verworfen. Diese Funktion kann verwendet       */
/* werden, wenn nach einem NiCheck2 richtige Daten ueber  */
/* das Netz kommen.                                       */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiCheck2Cancel       	 (NI_HDL      handle);


/*--------------------------------------------------------*/
/* NiCloseHandle ()                                       */
/*                                                        */
/* Mit NiListen oder NiConnect erzeugten Handle freigeben */
/* und evtl. bestehende Verbindung abbauen                */
/*                                                        */
/* INPUT   handle                                         */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              ungueltiger Handle               */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiCloseHandle  	 (NI_HDL  handle);

/*--------------------------------------------------------*/
/* NiCloseHandleKP ()                                     */
/*                                                        */
/* Mit NiListen oder NiConnect erzeugten Handle freigeben */
/* und evtl. bestehende Verbindung abbauen. Evtl.         */
/* vorhandener UDS Pfad wird nicht geloescht.             */
/*                                                        */
/* INPUT   handle                                         */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              ungueltiger Handle               */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiCloseHandleKP	 (NI_HDL  handle);

/*--------------------------------------------------------*/
/* NiShutdownHandle ()                                    */
/*                                                        */
/* Schliessen von Lese- und/oder Schreibseite einer Ver-  */
/* bindung. Der Handle muss mit NiCloseHandle noch        */
/* ansckliessend geschlossen werden.                      */
/*                                                        */
/* INPUT   handle        NI handle                        */
/*         howto         Verbindungsseite, die geschlossen*/
/*                       werden soll: NI_SHUT_RD,         */
/*                       NI_SHUT_WR oder NI_SHUTRDWR      */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              ungueltiger Handle               */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiShutdownHandle  	 (NI_HDL  handle,
                                          SAP_INT howto);


/*--------------------------------------------------------*/
/* NiCloseAll ()                                          */
/*                                                        */
/* Alle offenen Handles freigeben                         */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiCloseAll     	(void);


/*--------------------------------------------------------*/
/* NiRouteToTable ()                                      */
/*                                                        */
/* Routingstring in eine Tabelle von NI_ROUTE-Eintraegen  */
/* konvertieren                                           */
/*                                                        */
/* INPUT   route	Routestring (null-terminiert)     */
/*         size         Groesse des NI_ROUTE-Arrays       */
/*                                                        */
/* IN/OUT  table        Tabelle mit Routingeintraegen     */
/*                                                        */
/* OUTPUT  cnt          Anzahl der gefundenen Eintaege    */
/*                                                        */
/* RETURN  SAP_O_K	alles klar                        */
/* 	   NIEINTERN	Interner Fehler                   */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiRouteToTable (int            * cnt,
				 const SAP_UC	* routestring,
				 NI_ROUTE_TABLE * table,
				 int              size);


/*--------------------------------------------------------*/
/* NiGetHostAddrList ()                                   */
/*                                                        */
/* gibt die aktiven Interfaces der Maschine als IP-Adresse*/
/* zur"uck.                                               */
/*                                                        */
/* INPUT   hostaddr      array von bufsiz NI_HOSTADR      */
/*         bufsiz        maximale Anzahl an Eintraegen    */
/*                                                        */
/* OUTPUT  if_found      Anzahl der wirklich gefundenen   */
/*                       Interfaces                       */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/*                       sonst interner Fehler            */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiGetHostAddrList  (NI_HOSTADR  *hostaddr ,
                      SAP_INT bufsiz, SAP_INT* if_found);



/*--------------------------------------------------------*/
/* NiHostToAddr ()                                        */
/*                                                        */
/* log. Hostname in interne Adresse umwandeln             */
/* Achtung: Hostnamen und IP-Addr. werden beim 1. Zugriff */
/*          gepuffert. Dump/Flush siehe NiSetParam ()     */
/*                                                        */
/* INPUT   hostname      Hostname (null-terminiert)       */
/*                       oder Adress-String, z.B.         */
/*                       "155.56.2.3"                     */
/*                                                        */
/* OUTPUT  hostaddr      interne Host-Adresse             */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEHOST_UNKNOWN       log. Hostname unbekannt          */
/* NIEINVAL              Hostname == (SAP_UC   *) 0       */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiHostToAddr   	 (SAP_UC     *  hostname ,
					  NI_HOSTADR *  hostaddr );


/*--------------------------------------------------------*/
/* NiAddrToHost ()                                        */
/*                                                        */
/* interne Adresse in log. Hostnamen umwandeln            */
/* Achtung: Hostnamen und IP-Addr. werden beim 1. Zugriff */
/*          gepuffert. Dump/Flush siehe NiSetParam ()     */
/*                                                        */
/* INPUT   hostaddr      interne Host-Adresse             */
/*                                                        */
/* RETURN                Hostname (null-terminiert)       */
/*                       Host-Adr-String, falls unbekannt */
/*                                                        */
/*--------------------------------------------------------*/

SAP_UC  * API_DF  NiAddrToHost   	 (NI_HOSTADR *  hostaddr);

/*--------------------------------------------------------*/
/* NiAddrToHostCanon ()                                   */
/*                                                        */
/* interne Adresse in log. Hostnamen umwandeln            */
/* Achtung: Es wird immer der kanonische Name des Rechners*/
/*          zurueckgegeben. Ist dieser nicht voll         */
/*          qualifiziert, wird auch nur der Rechnernamen  */
/*          zurueckgegeben. Die Eintraege werden nicht    */
/*          gepuffert.                                    */
/*                                                        */
/* INPUT   hostaddr      interne Host-Adresse             */
/*                                                        */
/* RETURN                Hostname (null-terminiert)       */
/*                       Host-Adr-String, falls unbekannt */
/*                                                        */
/*--------------------------------------------------------*/

SAP_UC  * API_DF  NiAddrToHostCanon   	 (NI_HOSTADR *  hostaddr);


/*--------------------------------------------------------*/
/* NiServToNo ()                                          */
/*                                                        */
/* Service-Namen in (Port-)Nummer umwandeln               */
/* Achtung: Servicenamen und Ports werden beim 1. Zugriff */
/*          gepuffert. Dump/Flush siehe NiSetParam ()     */
/*                                                        */
/* INPUT   servname      Servicename (null-terminiert)    */
/*                       oder Service-Nummer-String, z.B. */
/*                       "3200"                           */
/*                                                        */
/* OUTPUT  servno        Service-Nummer (Host-Byte-Order) */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIESERV_UNKNOWN       log. Servicename unbekannt       */
/* NIEINVAL              servname == (SAP_UC   *) 0       */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiServToNo     	 (SAP_UC     *  servname,
					  SAP_USHORT *  servno);


/*--------------------------------------------------------*/
/* NiNoToServ ()                                          */
/*                                                        */
/* (Port-)Nummer in Service-Namen umwandeln               */
/* Achtung: Servicenamen und Ports werden beim 1. Zugriff */
/*          gepuffert. Dump/Flush siehe NiSetParam ()     */
/*                                                        */
/* INPUT   servno        Service-Nummer (Host-Byte-Order) */
/*                                                        */
/* RETURN                Service-Name (null-terminiert)   */
/*                       Service-Nummer-String, falls     */
/*                       unbekannt                        */
/*                                                        */
/*--------------------------------------------------------*/

SAP_UC  * API_DF  NiNoToServ     	 (SAP_USHORT  servno);


/*--------------------------------------------------------*/
/* NiMyHostName ()                                        */
/*                                                        */
/* eigenen Hostnamen ermitteln                            */
/*                                                        */
/* OUTPUT  hostname      eigener Hostname (null-termin.)  */
/*                                                        */
/* INPUT   maxlen        maximale Laenge fuer Hostnamen   */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIETOO_SMALL          Hostname wurde abgeschnitten     */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiMyHostName   	 (SAP_UC     *  hostname,
					  SAP_INT       maxlen);


/*--------------------------------------------------------*/
/* NiMyAddrVerify ()                                      */
/* NiMyAddrCheck ()                                       */
/*                                                        */
/* eigene Host-Adresse verifizieren (mittels NiIlisten,   */
/* NiConnect () und NiAccept ())                          */
/* NiMyAddrVerify laesst 127.0.0.1 nicht zu               */
/*                                                        */
/* INPUT   myhostname    eigener Hostname (null-termin.)  */
/*                       oder Adress-String, z.B.         */
/*                       "155.56.2.3"                     */
/*                       oder (SAP_UC   *) 0 (Funktion    */
/*                       ermittelt eig. Hostadr. selbst)  */
/*                                                        */
/* OUTPUT  myhostaddr    eigene Host-Adresse              */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEMYHOST_VERIFY      Fehler beim Verifizieren der     */
/*                       eigenen Host-Adresse             */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiMyAddrVerify	 (SAP_UC     *  myhostname,
                                  NI_HOSTADR *  myhostaddr);

SAPRETURN API_DF NiMyAddrCheck   (SAP_UC     *  myhostname,
				  NI_HOSTADR *  myhostaddr);

/*---------------------------------------------------------*/
/* NiHdlGetOutMsgs ()                                      */
/*                                                         */
/* Anzahl der geschriebenen NI-Pakete zurueckliefern       */
/*                                                         */
/* INPUT   handle       NI-Handle                          */
/*                                                         */
/* OUTPUT  cnt          Pointer auf den Rueckgabewert      */
/*                                                         */
/* RETURN                                                  */
/*                                                         */
/* SAP_O_K              alles klar                         */
/* NIEINVAL             handle ist kein NI-Handle          */
/*                      Pointer ist NULL                   */
/*                                                         */
/*---------------------------------------------------------*/

SAPRETURN API_DF NiHdlGetOutMsgs (NI_HDL    handle,
				  SAP_INT   *cnt);

/*---------------------------------------------------------*/
/* NiHdlToSock ()                                          */
/*                                                         */
/* Gebe den Socket zu einem NI-Handle zurueck              */
/*                                                         */
/* INPUT   handle       NI-Handle                          */
/*                                                         */
/* OUTPUT  sock         korrespondierender Socket          */
/*                                                         */
/* RETURN                                                  */
/*                                                         */
/* SAP_O_K              alles klar                         */
/* NIEINVAL             handle ist kein NI-Handle          */
/*                                                         */
/*---------------------------------------------------------*/

SAPRETURN API_DF NiHdlToSock 	 (NI_HDL    handle,
				  int    *  sock);


/*---------------------------------------------------------*/
/* NiSockToHdl ()                                          */
/*                                                         */
/* Gebe den NiHandle zu einem Socket zurueck               */
/*                                                         */
/* INPUT   sock         Socket                             */
/* OUTPUT  handle       korrespondierender NI-Handle       */
/*                                                         */
/* RETURN                                                  */
/*                                                         */
/* SAP_O_K              alles klar                         */
/* NIEINVAL             sock hat keinen korrespondierenden */
/*                      NI-Handle                          */
/*                                                         */
/*---------------------------------------------------------*/

SAPRETURN API_DF NiSockToHdl 	 (int       sock,
				  NI_HDL *  handle);


/*--------------------------------------------------------*/
/* NiDgBind ()                                            */
/*                                                        */
/* Vorbereitungen fuer Empfang von Datagrammen treffen    */
/*                                                        */
/* INPUT   servname      Name des Service                 */
/*                       Falls (SAP_UC   *) 0, wird       */
/*                       automatisch ein freier Port      */
/*                       gesucht                          */
/*                                                        */
/* OUTPUT  servno        Nummer des Service               */
/*                       (Host-Byte-Order)                */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIESERV_UNKNOWN       log. Servicename unbekannt       */
/* NIESERV_USED          Service schon benutzt            */
/* NIEINVAL              Handle zu gross fuer NI-Maske    */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiDgBind       	 (SAP_UC     *  servname,
					  SAP_USHORT *  servno);


/*--------------------------------------------------------*/
/* NiDgReceive ()                                         */
/*                                                        */
/* Datagramm empfangen                                    */
/*                                                        */
/* OUTPUT  buf_p_p       Zeiger auf Daten-Puffer          */
/*                       Laenge der Datagramme ist auf    */
/*                       max. 1024 begrenzt               */
/*         len_p         Laenge des Datagramms            */
/*                                                        */
/* INPUT   timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              NiDgBind nicht vorher aufgerufen */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEWAKEUP             Weck-Datagramm empfangen         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/* Datagramme der Laenge <= 1 werden als "Weck-Datagramme"*/
/* behandelt. Mehrere Weck-Datagramme werden beim Receive */
/* zu einem zusammengefasst (Wer wach ist, braucht nicht  */
/* geweckt zu werden).                                    */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiDgReceive    	 (void     **  buf_p_p,
					  SAP_INT   *  len_p,
					  SAP_INT      timeout);


/*--------------------------------------------------------*/
/* NiDgSend ()                                            */
/* NiDg2Send ()                                           */
/*                                                        */
/* Datagramm senden                                       */
/*                                                        */
/* INPUT   buf_p         Daten-Puffer                     */
/*         len           Laenge des Datagramms            */
/*         hostname      Name des Ziel-Host (NiDgSend)    */
/*                       (null-terminiert)                */
/*         hostaddr      Adr des Server-Host (NiDg2Send)  */
/*         servname      log. Name des Service            */
/*                       (null-terminiert)                */
/*         servno        Service-Nummer, falls servname   */
/*                       == (SAP_UC   *) 0                */
/*                       (Host-Byte-Order)                */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              buf_p oder len ungueltig         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiDgSend       	 (void       *  buf_p,
					  SAP_INT       len,
					  SAP_UC     *  hostname,
					  SAP_UC     *  servname,
					  SAP_USHORT    servno);

SAPRETURN API_DF  NiDg2Send      	 (void       *  buf_p,
					  SAP_INT       len,
					  NI_HOSTADR *  hostaddr,
					  SAP_UC     *  servname,
					  SAP_USHORT    servno);


/*--------------------------------------------------------*/
/* NiDgGetHdl ()                                          */
/*                                                        */
/* intern benutzten Handle abfragen (fuer NiSelect-       */
/* Aufrufe)                                               */
/*                                                        */
/* OUTPUT  handle                                         */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              Handle wurde noch nicht angelegt */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiDgGetHdl     	 (NI_HDL *  handle);


/*--------------------------------------------------------*/
/* NiDgFreeResrc ()                                       */
/*                                                        */
/* alle belegten "Datagramm-Resourcen" wieder freigeben   */
/* Aufrufe)                                               */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiDgFreeResrc  	(void);


/*--------------------------------------------------------*/
/* NiRawWrite ()                                          */
/*                                                        */
/* Daten an "Native-TCP-Partner" senden                   */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         buf           Daten-Puffer                     */
/*         buflen        Laenge der zu versendenden Daten */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                       Bezieht sich auf das Senden der  */
/*                       internen Datenpakete, aus denen  */
/*                       sich die Nachricht zusammensetzt)*/
/*                                                        */
/* OUTPUT  writelen      Laenge der gesendeten Daten      */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINVAL              timeout == 0 oder handle unguelt.*/
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiRawWrite     	 (NI_HDL      handle,
					  void     *  buf,
					  SAP_INT     buflen,
					  SAP_INT     timeout,
					  SAP_INT  *  writelen);


/*--------------------------------------------------------*/
/* NiRawRead ()                                           */
/*                                                        */
/* Daten von "Native-TCP-Partner" empfangen               */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         buf           Daten-Puffer                     */
/*         maxlen        Groesse des Daten-Puffers        */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                       Bezieht sich auf den Empfang der */
/*                       internen Datenpakete, aus denen  */
/*                       sich die Nachricht zusammensetzt)*/
/*                                                        */
/* OUTPUT  readlen       Laenge der empfangenen Daten     */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETOO_SMALL          Puffer zu klein fuer Nachricht   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/*                       (nachfolgende NiRead-Calls       */
/*                       liefern restliche Daten des      */
/*                       NI-Pakets)                       */
/* NIEPING               Partner hat Ping-Signal gesendet */
/*                       (keine Daten, readlen = 0)       */
/* NIEINVAL              handle ungueltig                 */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/* NIEROUT_HOST_UNKNOWN  im nirouter: log. Hostname unbek.*/
/* NIEROUT_SERV_UNKNOWN  im nirouter: log. Service unbek. */
/* NIEROUT_CONN_REFUSED  im nirouter: Verb.aufbau missl.  */
/* NIEROUT_CONN_BROKEN   im nirouter: Verbindung abgebr.  */
/* NIEROUT_PERM_DENIED   im nirouter: route nicht erlaubt */
/* NIEROUT_VERSION       im nirouter: unguelt. Clnt-Vers. */
/* NIEROUT_CANCELED      im nirouter: Route von Administr.*/
/*                                    ge"canceled"        */
/* NIEROUT_SHUTDOWN      im nirouter: Router Shutdown     */
/* NIEROUT_INTERN        im nirouter: interner Fehler     */
/*                                                        */
/*                                                        */
/* Auf Betriebssystemen, bei denen der Abbruch einer      */
/* Verbindung zu einem Signal (SIGPIPE) fuehrt, muss der  */
/* Aufrufer selbst fuer die Signalbehandlung sorgen (z.B. */
/* auf SIG_IGN setzen)                                    */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiRawRead      	 (NI_HDL      handle,
					  void     *  buf,
					  SAP_INT     maxlen,
					  SAP_INT     timeout,
					  SAP_INT  *  readlen);


/*--------------------------------------------------------*/
/* NiRawPeek ()                                           */
/*                                                        */
/* Testen, ob Daten auf einer best. Leitung angekommen    */
/* sind. Dabei wird nicht die Pufferung benutzt. Diese    */
/* wuerde interne Puffer fuellen, die fuer NiRawRead      */
/* nicht nutzbar sind.                                    */
/*                                                        */
/* INPUT   handle        zu ueberpruefender Handle        */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar (Daten da)            */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiRawPeek         	 (NI_HDL   handle,
					  SAP_INT  timeout);

/*--------------------------------------------------------*/
/* NiRawWriteOK ()                                        */
/*                                                        */
/* Testen, ob Write auf Handle blockieren wuerde.         */
/* Dabei wird nicht die Pufferung benutzt. Diese          */
/* wuerde interne Puffer fuellen, die fuer NiRawRead      */
/* nicht nutzbar sind.                                    */
/*                                                        */
/* INPUT   handle        zu ueberpruefender Handle        */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar (Daten da)            */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiRawWriteOK      	 (NI_HDL   handle,
					  SAP_INT  timeout);


/*--------------------------------------------------------*/
/* NiReadPending ()                                       */
/* NiWritePending ()                                      */
/*                                                        */
/* Info ueber noch zu lesende / schreibende Bytes liefern */
/* (fuer das letzte NI-Paket)                             */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* < 0                   handle ungueltig                 */
/* sonst                 noch zu lesende / schreibende    */
/*                       Bytes                            */
/*                                                        */
/*--------------------------------------------------------*/

SAP_INT API_DF    NiReadPending  	 (NI_HDL  handle);
SAP_INT API_DF    NiWritePending 	 (NI_HDL  handle);


/*--------------------------------------------------------*/
/* NiWait ()                                              */
/*                                                        */
/* n Millisekunden warten                                 */
/*                                                        */
/* INPUT   millisec      Zeitspanne in Millisekunden      */
/*                                                        */
/*--------------------------------------------------------*/

void API_DF       NiWait         	 (SAP_INT  millisec);


/*--------------------------------------------------------*/
/* NiSetParam ()                                          */
/* NiSetParamEx ()                                        */
/*                                                        */
/* Allgemeinen NI-Parameter setzen                        */
/* NiSetParamEx mit beliebigen Datentypen                 */
/* Fuer komplexe Datentypen muss NiSetParamEx benutzt     */
/* werden, da sonst beim cast Daten verloren gehen        */
/* koennen.                                               */
/*                                                        */
/* INPUT   param         NI-Parameter (s.u.)              */
/*                                                        */
/* IN/OUT  value         Wert                             */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              unzulaessiger Parameter / Wert   */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiSetParam     	 (SAP_RAW       param,
					  SAP_INT       value);
SAPRETURN API_DF  NiSetParamEx     	 (NI_PARAM_DATA *pdata);

/*--------------------------------------------------------*/
/* NiGetParam ()                                          */
/*                                                        */
/* Allgemeinen NI-Parameter lesen (s.u.)                  */
/*                                                        */
/* INPUT   param         NI-Parameter                     */
/*                                                        */
/* RETURN                Wert des Parameters              */
/*                                                        */
/*--------------------------------------------------------*/
/*--------------------------------------------------------*/
/* NiGetParamEx ()                                        */
/*                                                        */
/* Allgemeinen NI-Parameter lesen (s.u.)                  */
/* NiGetParamEx kann mit beliebigen Datentypen augerufen  */
/* werden. Der Speicher fuer die Daten muss vom Aufrufer  */
/* zur Verfuegung gestellt werden (bei Zeigerwerten).     */
/* Vor dem Aufruf muss der Datentyp in der Struktur auf   */
/* den richtigen Wert gesetzt werden (NI_DT_???).         */
/* Der Wert des Parameters wird im Erfolgsfall in der     */
/* in der Struktur zurueckgeliefert.                      */
/*                                                        */
/* INPUT/OUTPUT                                           */
/*         pdata         Zeiger auf NI-Parameterstruktur  */
/*                                                        */
/* RETURN                Wert                             */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              unzulaessiger Parameter / Wert   */
/*                                                        */
/*--------------------------------------------------------*/

SAP_INT   API_DF  NiGetParam     	 (SAP_RAW       param);
SAPRETURN API_DF  NiGetParamEx     	 (NI_PARAM_DATA *pdata);


#define NIP_PORTRANGE       (SAP_RAW) 1
#define NIP_MAXPORT         (SAP_RAW) 2
#define NIP_MINPORT         (SAP_RAW) 3
#define NIP_BUF_OPCOD       (SAP_RAW) 4
#define NIP_HSBUF_MAX       (SAP_RAW) 5  
#define NIP_LIB_VERS        (SAP_RAW) 6  
#define NIP_HSBUF_RETRY     (SAP_RAW) 7
#define NIP_HSBUF_REV_RETRY (SAP_RAW) 8  
#define NIP_FRAGTEST        (SAP_RAW) 9
#define NIP_CONNLOCAL       (SAP_RAW) 10
#define NIP_UDS_PATH        (SAP_RAW) 11
#define NIP_QUEUE_LEN       (SAP_RAW) 12

/*--------------------------------------------------------------------*/
/* allgemeine NI-Parameter NIP_...                                    */
/*                                                                    */
/* PORTRANGE   Bereich fuer lokale Connect-Port Auswahl einschraenken */
/*             0 = nein, sonst ja, Default 0                          */
/*                                                                    */
/* MAXPORT     Obergrenze Port-Range (s. NIP_PORTRANGE, Default 1023) */
/*                                                                    */
/* MINPORT     Untergrenze Port-Range (s. NIP_PORTRANGE, Default 0)   */
/*                                                                    */
/* BUF_OPCOD   NI-Buffer Opcodes (nur NiSetParam)                     */
/*             NI-Buffer ins Trace-File dumpen (0), flushen (1)       */
/*                                                                    */
/* HSBUF_MAX   Anzahl der max. pufferbaren Host/Serv-Eintraege setzen */
/*             (Default 100)                                          */
/*                                                                    */
/* LIB_VERS    NI_VERSION der NI-Library (nur NiGetParam)             */
/*                                                                    */
/* HSBUF_RETRY Anzahl in Sekunden, nach denen ein "Invalid"-Eintrag   */
/*             im Host/Serv-Buffer verfaellt (0 = nie)                */
/*                                                                    */
/* HSBUF_REV_RETRY Schalter, ob "Invalid"-Eintraege fuer              */
/*             IP->host-Aufloesungen verfallen sollen                 */
/*                                                                    */
/* FRAGTEST    Schaltet Fragment-Test-Modus ein (es wird jeweils nur  */
/*             1 Byte gelesen / geschrieben und dann ein Timeout      */
/*             zurueckgegeben (falls nicht NI_BLOCK))                 */
/*             (0 = aus, sonst ein, Default 0)                        */
/*                                                                    */
/* CONNLOCAL   Steuert, ob Connect ueber lokale Domain (AF_UNIX)      */
/*             erfolgen soll                                          */
/*                                                                    */
/* UDS_PATH    Pfadname fuer UNIX DOMAIN SOCKETS. Der Pfadname muss   */
/*             immer absolut angegeben werden!!!                      */
/*             Der Pfad muss ein %d enthalten. Dieser Platzhalter     */
/*             wird durch die socketnummer ersetzt.                   */
/*                                                                    */
/* QUEUE_LEN   Laenge der backlog queue beim listen call.             */
/*             Default Wert: 20                                       */
/*                                                                    */
/*--------------------------------------------------------------------*/
/*--------------------------------------------------------------------*/
/* Pufferspezifische NI-Parameter NIHP_...                            */
/*                                                                    */
/* MODE        Sprachmodus der Verbindung: MESG, ROUT, RAW            */
/*                                                                    */
/* CHECK_QUE   Statusabfrage/Setzen einer Verbindung bei der          */
/*             Pufferverwaltung.                                      */
/*             Mit SetParam wird das Check-flag gesetzt               */
/*             Mit GetParam wird der Status abgefragt                 */
/*             (0: Outqueue ist leer, 1: queue voll und check flag 0  */
/*              2: outqueue voll und check flag 1)                    */
/*                                                                    */
/* ALLOC_FUNC  Funktion zur Erfuellung von Speicheranfragen           */
/*                                                                    */
/* MAXQUEUE    Maximale Anzahl von Paketen in der outqueue. Der       */
/*             Sender wird abgebremst, bis wieder Pakete in die       */
/*             queue passen.                                          */
/*                                                                    */
/* WARNQUEUE   Schwellenwert fuer die Anzahl an Paketen, die noch     */
/*             in die Queue passen sollen. Wird dieser Wert unter-    */
/*             schritten, wird der Fehler NIEQUEFULL zurueckgegeben.  */
/*                                                                    */
/* MAXHEAP     Maximale Speichernutzung durch die Queues. Wird dieser */
/*             Schwellenwert ueberschritten, wird der Fehler          */
/*             NIEQUEFULL zurueckgegeben.                             */
/*                                                                    */
/*--------------------------------------------------------------------*/


/*--------------------------------------------------------*/
/* NiSetHdlParam ()					  */
/* NiGetHdlParam ()					  */
/*                                                        */
/* Handlespezifischen NI-Parameter setzen/lesen		  */
/*                                                        */
/* INPUT   hdl           NI-Handle			  */
/*         param         NI-Parameter (s.u.)              */
/*                                                        */
/* IN/OUT  value         Wert                             */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              unzulaessiger Parameter / Wert   */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiSetHdlParam (NI_HDL    hdl,
				SAP_RAW   param,
				void     *value);
SAPRETURN API_DF NiGetHdlParam (NI_HDL    hdl,
				SAP_RAW   param,
				void     *value);

/*--------------------------------------------------------*/
/* NiGetSapDllRelease()                                   */
/*                                                        */
/* SAP Release zurueckgeben.                              */
/* Die Funktion kann verwendet werden um das SAP Release  */
/* der Bibliothek dem Aufrufer bekannt zu geben           */
/*                                                        */
/* INPUT   len           maximale Laenge Release string   */
/*                                                        */
/* OUTPUT  release       Release string                   */
/*                                                        */
/*--------------------------------------------------------*/

void API_DF       NiGetSapDllRelease (SAP_UC * release,
				      int    len);


/*--------------------------------------------------------*/
/* NiErrSet ()                                            */
/*                                                        */
/* Sap-Error-Info ueber Ni-Returncode bereitstellen.      */
/* Die Sap-Error-Info kann anschliessend ueber die        */
/* Funktionen der Sap-Error-Schnittstelle abgeholt werden */
/* (siehe err.h)                                          */
/*                                                        */
/* INPUT   nirc          Ni-Returncode                    */
/*                                                        */
/*--------------------------------------------------------*/

void API_DF       NiErrSet       	 (SAPRETURN  nirc);


/*--------------------------------------------------------*/
/* NiErrStr ()                                            */
/*                                                        */
/* Namens-String  zu Ni-Error-Konstante liefern           */
/*                                                        */
/* INPUT   nirc          Ni-Returncode                    */
/*                                                        */
/* RETURN                Namensstring                     */
/*                                                        */
/*--------------------------------------------------------*/

SAP_UC   * API_DF NiErrStr       (SAPRETURN nirc);


/*--------------------------------------------------------*/
/* NiSecStr ()                                            */
/*                                                        */
/* SapRouter-Passwort aus String ausblenden               */
/*                                                        */
/* INPUT   istr          Inputstring                      */
/*                                                        */
/* RETURN                String mit ausgeblendetem        */
/*                       Passwort                         */
/*                                                        */
/*--------------------------------------------------------*/

SAP_UC   * API_DF NiSecStr       (const SAP_UC * istr);


/*------------------------------------------------------- */
/* Ni???ToStr ()					  */
/*                                                        */
/*                                                        */
/* INPUT    Hostadresse bzw. Servicenummer		  */
/*                                                        */
/* RETURN   formatierter String                           */
/*                                                        */
/*--------------------------------------------------------*/

SAP_UC   * API_DF NiAdrToStr	(NI_HOSTADR *  hostaddr);

SAP_UC   * API_DF NiSrvToStr	(SAP_USHORT  nservno);

SAP_UC   * API_DF NiSrvToStrL	(SAP_USHORT  nservno);


/*--------------------------------------------------------*/
/* NiNetINT ()                                            */
/* NiHostINT ()                                           */
/*                                                        */
/* Konvertiere SAP_INTs von, bzw. nach Netzbyteorder      */
/*                                                        */
/* INPUT     Integer                                      */
/*                                                        */
/* RETURN    konvertierte Nummer                          */
/*                                                        */
/*--------------------------------------------------------*/

/* SAPUNICODEOK_MIXEDLEN */
SAP_INT API_DF    NiNetINT     	 (SAP_INT  hostint);

/* SAPUNICODEOK_MIXEDLEN */
SAP_INT API_DF    NiHostINT    	 (SAP_INT  netint);

/*--------------------------------------------------------*/
/* NiNetServno ()                                         */
/* NiHostServno ()                                        */
/*                                                        */
/* Konvertiere Servicenummern von, bzw. nach              */
/* Netzbyteorder                                          */
/*                                                        */
/* INPUT     servicenummer                                */
/*                                                        */
/* RETURN    konvertierte Nummer                          */
/*                                                        */
/*--------------------------------------------------------*/

SAP_USHORT API_DF NiNetServno 	 (SAP_USHORT  hostservno);

SAP_USHORT API_DF NiHostServno 	 (SAP_USHORT  netservno);


/*--------------------------------------------------------*/
/* NiSetSingleRead ()                                     */
/*                                                        */
/* Setze Socket auf 'Single Read'                         */
/* `Single Read' heisst, dass in jedem NiRead nur einmal  */
/* versucht wird zu lesen. Normalerweise wird nach        */
/* erfolgreichem Lesen, nocheinmal versucht zu lesen,     */
/* falls das NI-Paket nicht komplett ist. Dies wird       */
/* hiermit verhindert. (Ist unter Windows notwendig.)     */
/*                                                        */
/* INPUT  handle    NI-Handle                             */
/*        mode      TRUE/FALSE                            */
/*                                                        */
/* RETURN SAP_O_K   alles klar                            */
/*        NIEINVAL  kein gueltiger NI-Handle              */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiSetSingleRead (SAP_INT  handle,
				  SAP_BOOL  mode);

/*--------------------------------------------------------*/
/* NiGetSingleRead ()                                     */
/*                                                        */
/* Abfrage auf 'Single Read'                              */
/*                                                        */
/* INPUT  handle    NI-Handle                             */
/*                                                        */
/* RETURN modus                                           */
/*                                                        */
/*--------------------------------------------------------*/

SAP_BOOL  API_DF NiGetSingleRead (SAP_INT  handle);

/*--------------------------------------------------------*/
/* NiGetRawSocket                                         */
/*                                                        */
/* Gebe den socket an den Aufrufer zurueck. Nur RAW       */
/* sockets koennen (sinnvoll) genutzt werden. Socket      */
/* darf nicht von extern veraendert (z.B. geschlossen)    */
/* werden!                                                */
/*                                                        */
/* IN                                                     */
/*      handle           NI-Handle                        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINTERN             interner Fehler aufgetreten      */
/* NIEINVAL              kein valider Parameter           */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiGetRawSocket    (NI_HDL      handle,
                                    int         *socket);

/*--------------------------------------------------------*/
/* NiSelNewSet ()                                         */
/*                                                        */
/* Fordert eine neues SelectSet an                        */
/*                                                        */
/* IN/OUT  set           selectset                        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINTERN             interner Fehler aufgetreten      */
/*                                                        */
/*--------------------------------------------------------*/

/*-NiSel--------------------------------------------------*/

SAPRETURN NiSelNewSet     (NISEL_HDL *set);

/*--------------------------------------------------------*/
/* NiSelDestroySet ()                                     */
/*                                                        */
/* Gibt eine altes SelectSet wieder frei                  */
/*                                                        */
/* IN/OUT  set           selectset                        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              kein valider selectset           */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN NiSelDestroySet (NISEL_HDL *set);

/*--------------------------------------------------------*/
/* NiSelClearSet ()                                       */
/*                                                        */
/* Lscht alle Eintaege in einem SelectSet                */
/*                                                        */
/* INPUT   set           selectset                        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              kein valider selectset           */
/* NIEINTERN             interner Fehler                  */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN NiSelClearSet   (NISEL_HDL set);

/*--------------------------------------------------------*/
/* NiSelSet/Clear ()                                      */
/*                                                        */
/* Setzt/Lscht die flags fr diesen NI-Handle flags kann */
/* hier aus einer Bit-Kombination von NI_WRITE, NI_READ   */
/* und NI_CONNECT bestehen. NiSelSet setzt auerdem die   */
/* data zu diesem Handle.                                 */
/*                                                        */
/* INPUT   set           selectset                        */
/*         hdl           NI-Handle                        */
/*         flags         Bitkombination aus NI_CONNECT,   */
/*                       NI_READ und NI_WRITE             */
/*         void          beliebige Applikationsdaten      */
/*                                                        */
/* IN/OUT  value         Wert                             */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINTERN             interner Fehler                  */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN NiSelSet        (NISEL_HDL set,
                           NI_HDL hdl,
                           SAP_RAW flags,
                           void *data);

SAPRETURN NiSelClear      (NISEL_HDL set,
                           NI_HDL hdl,
                           SAP_RAW flags);

/*--------------------------------------------------------*/
/* NiSelSelect ()                                         */
/*                                                        */
/* Ersatz fr NiSelect                                    */
/* Setzt den Iterator fuer NiSelNext implizit zurueck     */
/*                                                        */
/* INPUT   set           selectset                        */
/*         timeout       timeout fuers select             */
/*                                                        */
/* RETURN                                                 */
/*  siehe NiSelect                                        */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN NiSelSelect     (NISEL_HDL set, SAP_INT timeout);

/*--------------------------------------------------------*/
/* NiSelNext  ()                                          */
/*                                                        */
/* Geben ersten/nchsten NI-Handle, der laut mode bereit  */
/* ist, zurck. Direkt nach NiSelect mit diesem Set sind  */
/* beide quivalent                                       */
/*                                                        */
/* INPUT   set           selectset                        */
/* IN/OUT  read/                                          */
/*         write/                                         */
/*         connect       Wenn NULL, modus interessiert    */
/*                                  nicht                 */
/*                       sonst Pointer auf Variable       */
/*                                                        */
/* OUT     data          Userdata                         */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* Ni_INVALID_HDL        kein Handle mehr verfuegbar      */
/*                                                        */
/*--------------------------------------------------------*/

NI_HDL    NiSelNext       (NISEL_HDL set,
                           SAP_BOOL *read,
                           SAP_BOOL *write,
                           SAP_BOOL *connect,
                           void **data);

/*--------------------------------------------------------*/
/* NiSelStartLoop ()                                      */
/*                                                        */
/* Setzt den Iterator fuer NiSelNext wieder auf den ersten*/
/* Eintrag                                                */
/*                                                        */
/* INPUT   set           selectset                        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              kein valider selectset           */
/* NIEINTERN             interner Fehler                  */
/*                                                        */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN NiSelStartLoop  (NISEL_HDL set);

/*--------------------------------------------------------*/
/* NiSelMode ()                                           */
/* NiSelStat ()                                           */
/*                                                        */
/* Gib den Status/Modus zu hdl zurck                     */
/* Status ist dabei der Status, den das letzte Select     */
/* zurueckgeliefert hat, waehrend modus der in diesem     */
/* Set angeforderte Modus fuer das naechste Select ist.   */
/*                                                        */
/* INPUT   set           selectset                        */
/*         hdl           NI-Handle                        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* modus des Handles in der maske                         */
/*                                                        */
/*--------------------------------------------------------*/

SAP_RAW API_DF NiSelMode     (NISEL_HDL set, NI_HDL hdl);
SAP_RAW API_DF NiSelStat     (NISEL_HDL set, NI_HDL hdl);

/*-NiBuf--------------------------------------------NiSel-*/

/*--------------------------------------------------------*/
/* NiBufReceive                                           */
/*                                                        */
/* Empfange einen NI-Puffer                               */
/*                                                        */
/* INPUT   hdl           Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         timeout       Zeitlimit                        */
/*                                                        */
/* OUTPUT  buffer        Pointer auf Pufferpointer        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINVAL              Puffer oder handle unguelt.      */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiBufReceive  (NI_HDL      hdl,
                                NIBUFFER  **buffer,
                                SAP_INT     timeout);

/*--------------------------------------------------------*/
/* NiBufSend                                              */
/*                                                        */
/* Sende einen NI-Puffer                                  */
/*                                                        */
/* INPUT   hdl           Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         timeout       Zeitlimit                        */
/* IN/OUT  buffer        NI-Puffer zum senden. Wird bei   */
/*                       Erfolg freigegeben.              */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINVAL              Puffer oder handle unguelt.      */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiBufSend     (NI_HDL      hdl,
                                NIBUFFER  **buffer,
                                SAP_INT     timeout);

/*--------------------------------------------------------*/
/* NiBufSend                                              */
/*                                                        */
/* Sende einen NI-Handle ueber einen anderen              */
/*                                                        */
/* INPUT   hdl           Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         sendhdl       Handle, der gesendet werden soll */
/*         timeout       Zeitlimit                        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIEINVAL              Puffer oder handle unguelt.      */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN NiSendHandle (NI_HDL  hdl,
                        NI_HDL  sendhdl,
                        SAP_INT timeout);

/*--------------------------------------------------------*/
/* NiBufFree                                              */
/*                                                        */
/* Dies ist die Funktion, ueber die beliebige NI-Puffer   */
/* freigegeben werden. Sie wird intern dazu aufgerufen    */
/* und kann von der Applikation gerufen werden.           */
/*                                                        */
/* IN/OUT  buffer        Alter Puffer                     */
/*                                                        */
/*--------------------------------------------------------*/

void     API_DF NiBufFree     (NIBUFFER  **buffer);

/*--------------------------------------------------------*/
/* NiBufAlloc                                             */
/*                                                        */
/* Allokiere einen neuen NI-Puffer                        */
/*                                                        */
/* INPUT   len           Laenge des Datenbereichs         */
/*         txt           Erkennungstext fuer den Puffer   */
/*                                                        */
/* OUTPUT  buffer        Neuer Puffer                     */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              Puffer oder handle unguelt.      */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiBufAlloc    (NIBUFFER  **buffer,
                                SAP_INT     len,
                                SAP_UC     *txt);

/*--------------------------------------------------------*/
/* NiBufAlloc2                                            */
/*                                                        */
/* Allokiere einen neuen NI-Puffer, bei dem eine externe  */
/* Free-Funktion vorhanden ist, die von NiBufFree         */
/* aufgerufen wird, anstatt den Puffer sofort wieder      */
/* freizugeben. Diese Freefunktion kann dann NiBufFree2   */
/* aufrufen, um den Puffer endgltig freizugeben          */
/*                                                        */
/* INPUT  len           Laenge des Datenbereichs          */
/*        txt           Erkennungstext fuer den Puffer    */
/* OUTPUT buffer        Neuer Puffer                      */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              Puffer oder handle unguelt.      */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/
/* NiBufFree2                                             */
/*                                                        */
/* Gibt einen NI-Puffer frei, ohne evtuelle externe       */
/* freefunktion zu rufen (siehe NiBufAlloc2)              */
/*                                                        */
/* IN/OUT  buffer        Alter Puffer                     */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiBufAlloc2   (NIBUFFER  **buffer,
                                SAP_INT     len,
                                SAP_UC     *text,
                                NIBUF_FREE_FUNC *func);

void      API_DF NiBufFree2    (NIBUFFER  **buffer);

/*--------------------------------------------------------*/
/* NiBufDup                                               */
/*                                                        */
/* Dupliziere einen Puffer, um ihn an mehrere Clients zu  */
/* senden. Nach dem Senden einer Referenz darf der Inhalt */
/* des Puffers nicht mehr veraendert werden.              */
/*                                                        */
/* INPUT   from          Bestehender Puffer               */
/*                                                        */
/* OUTPUT  to            Neue Referenz auf Puffer from    */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              Puffer oder handle unguelt.      */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiBufDup      (NIBUFFER  **to,
                                NIBUFFER   *from);

/*--------------------------------------------------------*/
/* NiBufWrap                                              */
/*                                                        */
/* Erstelle einen Puffer zu einem bestehenden             */
/* Datenbereich:                                          */
/* Wenn der Puffer NiBufFree ruft dann die den Parameter  */
/* 'freef', um den Datenbereich freizugeben. Diese        */
/* Funktion muss dann NiBufUnwrap aufrufen.               */
/*                                                        */
/* INPUT   data          Anfang auf den Datenbereich      */
/*         len           Laenge des Datenbereichs         */
/*         spacelen      Laenge des gesammten Puffers     */
/*         freef         Freigabefunktion                 */
/*                                                        */
/* OUTPUT  to            Neue Referenz auf Puffer from    */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              Puffer oder handle unguelt.      */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/
/* NiBufUnwrap                                            */
/*                                                        */
/* Entpacke einen mit NiBufWrap erzeugten NI-Puffer.      */
/* Diese Funktion sollte nur aus der in NiBufWrap         */
/* angegebenen Freefunktion aufgerufen werden             */
/*                                                        */
/* IN/OUT  buffer        NI-Puffer                        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* NULL                  Puffer war nicht mit NiBufWrap   */
/*                       erzeugt                          */
/* sonst                 Datenbereich                     */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiBufWrap     (NIBUFFER  **to,
                                SAP_RAW    *data,
                                SAP_INT     len,
                                SAP_INT     spacelen,
                                NIBUF_FREE_FUNC *freef,
                                void       *uptr);

void *    API_DF NiBufUnWrap   (NIBUFFER  **buffer);

/*--------------------------------------------------------*/
/* NiBufGetPtr                                            */
/*                                                        */
/* Gebe den Userptr zurck                                */
/*                                                        */
/* IN      buf           Puffer fuer den stack            */
/*                                                        */
/* RETURN  userpointer                                    */
/*                                                        */
/*--------------------------------------------------------*/

void * API_DF NiBufGetPtr      (NIBUFFER  *buf);


/*--------------------------------------------------------*/
/* NiBufPush                                              */
/*                                                        */
/* Baue einen Stack aus NI-Puffern                        */
/*                                                        */
/* IN/OUT  stack         stack-pointer                    */
/*                                                        */
/* IN      buf           Puffer fuer den stack            */
/*                                                        */
/*--------------------------------------------------------*/
/*                                                        */
/* NiBufPop                                               */
/*                                                        */
/* Hole einen Puffer vom stack                            */
/*                                                        */
/* IN/OUT  stack         stack-pointer                    */
/*                                                        */
/* RETURN  NULL          Stack leer                       */
/*         sonst         Puffer                           */
/*                                                        */
/*--------------------------------------------------------*/

void      API_DF NiBufPush     (NIBUFFER **stack,
                                NIBUFFER *buf);
NIBUFFER *API_DF NiBufPop      (NIBUFFER **stack);

/*-NiSnc--------------------------------------------NiBuf-*/

/*--------------------------------------------------------*/
/* NiSncAclStr                                            */
/*                                                        */
/* erzeuge einen lesbaren String aus einem SNC-Account    */
/*                                                        */
/* INPUT   acl           SNC-Account                      */
/*                                                        */
/* RETURN  string                                         */
/*                                                        */
/*--------------------------------------------------------*/

SAP_UC  * API_DF NiSncAclStr   (const NISNC_ACLKEY *acl);

/*--------------------------------------------------------*/
/* NiSncGetPeer                                           */
/*                                                        */
/*                                                        */
/*                                                        */
/* INPUT   hdl           Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*                                                        */
/* OUTPUT  peer          SNC-Account                      */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIETIMEOUT            Der Handle ist noch nicht lese-  */
/*                       oder schreibbereit               */
/* NIESNC_FAILURE        Verbindung ist nicht             */
/*                       SNC-gesichert                    */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiSncGetPeer  (NI_HDL        hdl,
                                NISNC_ACLKEY *peer);

/*--------------------------------------------------------*/
/* NiSncRequest                                           */
/*                                                        */
/* Fordere eine SNC-Verschlsselung zum Client mit        */
/* SNC-Name peer:                                         */
/* Hiermit kann ein Server Verschluesselung nach dem      */
/* Accept einer normalen Verbindung anfordern.            */
/* SNC-Fehler werden aber erst waehrend der normalen      */
/* Communication gemeldet                                 */
/*                                                        */
/* INPUT   hdl          NI_HDL                            */
/*         timeout      timeout                           */
/*         peer         Client-SNC-Name                   */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiSncRequest (NI_HDL        hdl,
                               SAP_INT       timeout,
                               NISNC_ACLKEY *peer);

/*--------------------------------------------------------*/
/* NiSncNameToAcl                                         */
/*                                                        */
/*                                                        */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*                                                        */
/* OUTPUT                                                 */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINVAL              Puffer oder handle unguelt.      */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiSncNameToAcl (SAP_UC       *name,
                                 NISNC_ACLKEY *acl);

/*--------------------------------------------------------*/
/* NiSncAclCmp                                            */
/*                                                        */
/*                                                        */
/*                                                        */
/* INPUT   acl1, acl2    SNC-Accounts                     */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* TRUE                  SNC-Namen sind gleich            */
/* FALSE                 sonst                            */
/*                                                        */
/*--------------------------------------------------------*/

SAP_BOOL  API_DF NiSncAclCmp   (const NISNC_ACLKEY *acl1,
                                const NISNC_ACLKEY *acl2);

/*--------------------------------------------------------*/
/* NiSncFinHS                                             */
/*                                                        */
/*                                                        */
/*                                                        */
/* INPUT                 handle fuer Verb zum Partner     */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               Austausch von Overhead war noetig*/
/* NIESNC_FAILURE        kein Overhead  zu verarbeiten    */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN  API_DF NiSncFinHS(NI_HDL handle);

#ifdef __cplusplus
}
#endif

#endif /* NIXX_H */
