/*!
    @file     SharedSQL_ICommandCache.hpp
    @ingroup  SharedSQL
    @author   DirkT
    @brief    Cache for SQLCommands, shared
    @see            

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/

#ifndef SHAREDSQL_ICOMMANDCACHE_HPP
#define SHAREDSQL_ICOMMANDCACHE_HPP

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SQLManager/SharedSQL/SharedSQL_CommandCacheStatistics.hpp"
#include "SQLManager/SharedSQL/SharedSQL_CommandCacheIterator.hpp"
#include "SQLManager/SharedSQL/SharedSQL_Types.hpp"

#include "SAPDBCommon/SAPDB_Types.hpp"

class SharedSQL_IPrepareHandle;
class SharedSQL_ICachedCommand;
class SAPDBMem_IRawAllocator;
class SAPDBMem_RawAllocator;

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

/*---------------------------------------------------------------------------*/
/*!
    @interface      SharedSQL_ICommandCache
    @brief          Interface to SharedSQL_CommandCache
 */
class SharedSQL_ICommandCache : public SAPDBMem_IRawAllocator
{
public:
    // Methods derived from SAPDBMem_IRawAllocator

    /*!
        @copydoc        SAPDBMem_IRawAllocator
    */
    virtual void* Allocate( SAPDB_ULong ByteCount ) = 0;
    /*!
        @copydoc        SAPDBMem_IRawAllocator
    */
    virtual void* Allocate( SAPDB_ULong ByteCount, const void* Hint ) = 0;
    /*!
        @copydoc        SAPDBMem_IRawAllocator
    */
    virtual void  Deallocate( void* p ) = 0;
    /*!
        @copydoc        SAPDBMem_IRawAllocator
    */
    virtual SAPDBMem_RawAllocator* GetBaseAllocator( void ) = 0;
    /*!
        @copydoc        SAPDBMem_IRawAllocator
    */
    virtual void GetBaseAllocatorCallStatistics( SAPDB_ULong &CountAlloc, SAPDB_ULong &CountDealloc ) const = 0;
    /*!
        @copydoc        SAPDBMem_IRawAllocator
    */
    virtual void GetCallStatistics( SAPDB_ULong &CountAlloc, SAPDB_ULong &CountDealloc ) const = 0;
    /*!
        @copydoc        SAPDBMem_IRawAllocator
    */
    virtual int  GetErrorCount( void ) const = 0;
    //

    /*!
        @brief  Free as much memory as possible by destroying unnecessary elements
    */
    virtual void CleanUpAll( void ) = 0;

    /*!
        @brief  Invalidate all commands within the cache
    */
    virtual void InvalidateAll( void ) = 0;
    //

    /*!
        @brief  Get an interface to a SharedSQL_PrepareHandle for some SharedSQL_CachedCommand
        @param  Alloc    [in]
        @param  Context  [in]
        @param  Stmt     [in]
        @param  Internal [in]
        @return SharedSQL_IPrepareHandle*
    */
	virtual SharedSQL_IPrepareHandle* NewPrepareHandle( SAPDBMem_IRawAllocator& Alloc, SharedSQL_SQLContext& Context, SharedSQL_SQLStmt& Stmt, SAPDB_Bool Internal ) = 0;

    /*!
        @brief  Destroy the Handle created with NewPrepareHandle
        @param  PH [in]
    */
    virtual void DestroyPrepareHandle( SharedSQL_IPrepareHandle* PH ) = 0;
    //

    /*!
        @brief  ReloadSQLStmt
        @param  TransContext [in]
        @param  Command [in]
        @return ...
    */
	virtual SAPDB_Bool ReloadSQLStmt( tgg00_TransContext& TransContext, SharedSQL_ICachedCommand* Command ) = 0;

    /*!
        @brief  DeleteUnloadedSQLStmt
        @param  TransContext [in]
        @param  Command [in]
        @return ...
    */
	virtual SAPDB_Bool DeleteUnloadedSQLStmt( tgg00_TransContext& TransContext, SharedSQL_ICachedCommand* Command ) = 0;
    //

    /*!
        @brief  LRUListMoveCommandToTop
        @param  Command [in]
    */
    virtual void LRUListMoveCommandToTop( SharedSQL_ICachedCommand* Command ) = 0;

    /*!
        @brief  GetHashTableEntry
        @param  i [in]
        @return ...
    */
    virtual SharedSQL_ICachedCommand* GetHashTableEntry(SharedSQL_HashValue i) = 0; 
    //

    /*!
        @brief  Get information about the CommandCache
        @param  S [out]
    */
    virtual void GetCacheInfo( SharedSQL_CommandCacheInfo& S ) = 0;

    /*!
        @brief  Get information about the CommandCache (reset version)
        @param  S [out]
    */
    virtual void GetCacheInfoReset( SharedSQL_CommandCacheInfo& S ) = 0;

    /*!
        @brief  Reset statistics information of the cache
    */
    virtual void ResetCacheInfo( void ) = 0;

    /*!
        @brief  Reset statistics information of the commands stored in the cache
    */
    virtual void ResetCommandInfos( void ) = 0;


    /*!
        @brief  Calculate the slot to store the command in
        @param  Hash [in]
        @return SharedSQL_HashValue
    */
    virtual SharedSQL_HashValue CalcSlot( const SharedSQL_HashValue Hash ) = 0;
	//

    /*!
        @brief  Begin
        @return SharedSQL_CommandCacheIterator
    */
    virtual SharedSQL_CommandCacheIterator Begin( void ) = 0;

    /*!
        @brief  End
        @return SharedSQL_CommandCacheIterator
    */
    virtual SharedSQL_CommandCacheIterator End( void ) = 0;
    // 

    /*!
        @brief  Lock some hash table slot NONEXCLUSIVE
        @param  i [in]
    */
    virtual void NonExclusiveLockSlot(SharedSQL_HashValue i) = 0;

    /*!
        @brief  Leave the lock
        @param  i [in]
    */
    virtual void LeaveLockSlot(SharedSQL_HashValue i) = 0;

    /// ...
    virtual void IncAccessCountExecute( void ) = 0;
    /// ...
    virtual void IncSuccessfulAccessCountExecute( void ) = 0;
    /// ...
    virtual void IncUnSuccessfulAccessCountExecute( void ) = 0;

    //
};  // SharedSQL_ICommandCache

#endif  /* SHAREDSQL_ICOMMANDCACHE_HPP */
