/*!
    @file     SharedSQL_Handles.hpp
    @ingroup  SharedSQL
    @author   DirkT
    @brief    Handles for SharedSQL_Commands
    @see            

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/

#ifndef SHAREDSQL_HANDLES_HPP
#define SHAREDSQL_HANDLES_HPP

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SQLManager/SharedSQL/SharedSQL_IHandles.hpp"
#include "SQLManager/SharedSQL/SharedSQL_ICommandCache.hpp"
#include "SQLManager/SharedSQL/SharedSQL_Types.hpp"

#include "SAPDBCommon/SAPDB_Types.hpp"

class SharedSQL_ICachedCommand;

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/


/*---------------------------------------------------------------------------*/
/*!
    @class          SharedSQL_PrepareHandle
    @brief          Handle for the SharedSQL_CachedCommand used during parsing
 */
class SharedSQL_PrepareHandle : public SharedSQL_IPrepareHandle
{
public:
	/*!
        @brief  Constructor
        @param  Alloc [in] - Pointer to the allocator that was used to create the handle
        @param  R     [in] - Pointer to the command the handle refers to
    */
	SharedSQL_PrepareHandle( SAPDBMem_IRawAllocator* Alloc, SharedSQL_ICachedCommand* R );
    /*!
        @brief  Destructor
    */
    ~SharedSQL_PrepareHandle( void );

    /*!
        @copydoc        SharedSQL_IPrepareHandle
    */
    virtual SAPDB_Bool MustPrepare( SharedSQL_CommandStatus& Status ) const;
    /*!
        @copydoc        SharedSQL_IPrepareHandle
    */
    virtual const SharedSQL_ParseID* GetMasterParseID( void ) const;
    /*!
        @copydoc        SharedSQL_IPrepareHandle
    */
	virtual SharedSQL_Error	PutPlan( SharedSQL_ParseID MPID, void *const P, SAPDB_ULong Size ) const;
    /*!
        @copydoc        SharedSQL_IPrepareHandle
    */
	virtual const void*	GetPlan( void* Key ) const;
    /*!
        @copydoc        SharedSQL_IPrepareHandle
    */
	virtual void ErrorWhilePreparing( void ) const;
    /*!
        @copydoc        SharedSQL_IPrepareHandle
    */
    virtual void ReleaseHandle( SAPDB_UInt8 ParseTimeSec, SAPDB_UInt8 ParseTimeMicroSec );


    /*!
        @brief  Operaror=
        @param  H [in]
    */
    SharedSQL_PrepareHandle& operator=( const SharedSQL_PrepareHandle& H );

    /*!
        @brief  Sets mFlgMustPrepare to indicate wheather or not the session that gets the handle must parse
        @param  must [in]
    */
    void SetMustPrepare(SAPDB_Bool must);

    /*!
        @brief  Sets mOldStatus to store the status of the command before the handle was created
        @param  Status [in]
    */
    void SetOldStatus(SharedSQL_CommandStatus Status);

    /*!
        @brief  Returns the allocator that was used to create the handle
        @return SAPDBMem_IRawAllocator&
    */
    SAPDBMem_IRawAllocator& GetAllocator(void);
	//
	friend class SharedSQL_ParseIDHandle;

private:
	/// Pointer to the allocator that was used to create the handle
    SAPDBMem_IRawAllocator*     mAllocator;
	/// Pointer to the command the PrepareHandle refers to
	SharedSQL_ICachedCommand*	mRef;
	/// Flag to indicate that the session that holds this handle has to prepare the command
	SAPDB_Bool		            mFlgMustPrepare;
	/// The commands status before this handle was created by the command cache. The actual status will be PREPARING if mFlgMustPrepare is TRUE or PREPARED if its FALSE. The old status may be NEW, INVALID or DROPPED if FlgPrepare is TRUE or PREPARED if its FALSE.
    SharedSQL_CommandStatus     mOldStatus;

};  // SharedSQL_PrepareHandle

/*---------------------------------------------------------------------------*/
/*!
    @class          SharedSQL_ParseIDHandle
    @brief          Handle for the SharedSQL_CachedCommand used within the SharedSQL_ParseIDCache
    @copydoc        SharedSQL_ParseIDHandle
 */
class SharedSQL_ParseIDHandle
{
public:
	/*!
        @brief  Constructor
    */
    SharedSQL_ParseIDHandle( void );

    /*!
        @brief  Constructor
        @param  R [in]
    */
    SharedSQL_ParseIDHandle( SharedSQL_ICachedCommand* R );

    /*!
        @brief  Constructor
        @param  Alloc [in] - Pointer to the allocator that was used to create the handle
        @param  H     [in] - PrepareHandle to create the ParseIDHandle from
        @param  M9    [in] - Flag to indicate wheather -8 or -9 should be returned to the application
    */
    SharedSQL_ParseIDHandle( SAPDBMem_IRawAllocator* Alloc, const SharedSQL_PrepareHandle& H, const SAPDB_Bool M9 );
    
    /*!
        @brief  Operaror =
        @param  H [in]
    */
    SharedSQL_ParseIDHandle& operator=( const SharedSQL_ParseIDHandle& H ); 

    /*!
        @brief  Destructor
    */
    ~SharedSQL_ParseIDHandle( void );

    /*!
        @brief  Returns the allocator that was used to create the handle
        @return SAPDBMem_IRawAllocator&
    */
    SAPDBMem_IRawAllocator& GetAllocator(void);

    /*!
        @brief  SetExecuteStatisticData
        @param  ExecuteTimeSec      [in]
        @param  ExecuteTimeMicroSec [in]
        @param  ReadRowCount        [in]
        @param  QualifiedRowCount   [in]
        @param  VirtualReadCount    [in]
        @param  PhysicalReadCount   [in]
        @param  FetchRowCount       [in]
        @param  SuspendCount        [in]
        @param  WaitCount           [in]
        @return TRUE if ok, else FALSE
    */
    SAPDB_Bool SetExecuteStatisticData(SAPDB_UInt8 ExecuteTimeSec, SAPDB_UInt8 ExecuteTimeMicroSec,
                                        SAPDB_Int8 ReadRowCount, SAPDB_Int8 QualifiedRowCount,
                                        SAPDB_Int8 VirtualReadCount, SAPDB_Int8 PhysicalReadCount,
                                        SAPDB_Int8 FetchRowCount, 
                                        SAPDB_Int8 SuspendCount, SAPDB_Int8 WaitCount ) const;

    void GetCommandID( SharedSQL_CommandID& ID ) const;

	friend class SharedSQL_ExecuteHandle;

private:
	/// Pointer to the allocator that was used to create the handle
    SAPDBMem_IRawAllocator*     mAllocator;
	/// Pointer to the command the ParseIDHandle refers to
	SharedSQL_ICachedCommand*	mRef;
	/// The master parse id that was valid when this handle was created
    SharedSQL_ParseID           mMPID;
	/// Flag to indicate wheather -8 or -9 should be returned to the application
    SAPDB_Bool                  mMinus9;

};  // SharedSQL_ParseIDHandle

/*---------------------------------------------------------------------------*/
/*!
    @class          SharedSQL_ExecuteHandle
    @brief          Handle for the SharedSQL_CachedCommand used during execution
 */
class SharedSQL_ExecuteHandle : public SharedSQL_IExecuteHandle
{
public:
    /*!
        @brief  Constructor
        @param  Alloc          [in] - Pointer to the allocator that was used to create the handle
        @param  H              [in] - ParseIDHandle to create the ExecuteHandle from
        @param  CountExecuting [in] - Flag to indicate if the Execution counter shall be increased
    */
	SharedSQL_ExecuteHandle( SAPDBMem_IRawAllocator* Alloc, const SharedSQL_ParseIDHandle& H, bool CountExecuting );

    /*!
        @brief  Destructor
    */
	~SharedSQL_ExecuteHandle( void ); 

    /*!
        @copydoc        SharedSQL_IExecuteHandle
    */
    virtual const SharedSQL_ParseID* GetMasterParseID( void ) const;
    /*!
        @copydoc        SharedSQL_IExecuteHandle
    */
	virtual const void*	GetPlan( void* Key ) const;
    /*!
        @copydoc        SharedSQL_IExecuteHandle
    */
	virtual void SetPlanInvalid( void ) const;
    /*!
        @copydoc        SharedSQL_IExecuteHandle
    */
	virtual SAPDB_Bool IsPlanValid( SAPDB_Bool& M9 ) const;
    /*!
        @copydoc        SharedSQL_IExecuteHandle
    */
	virtual const SharedSQL_SQLStmt GetSQLStmt( const SharedSQL_SQLContext* &Context ) const;
    /*!
        @copydoc        SharedSQL_IExecuteHandle
    */
    virtual void ReleaseHandle( void );

    /*!
        @brief  Operaror =
        @param  H [in]
    */
    SharedSQL_ExecuteHandle& operator=( const SharedSQL_ExecuteHandle& H );

    /*!
        @brief  Returns the allocator that was used to create the handle
        @return SAPDBMem_IRawAllocator&
    */
    SAPDBMem_IRawAllocator& GetAllocator(void);

    /// ...
	friend class SharedSQL_ParseIDCache;

private:
	/// Pointer to the allocator that was used to create the handle
    SAPDBMem_IRawAllocator*     mAllocator;
	/// Pointer to the command the ExecuteHandle refers to
	SharedSQL_ICachedCommand*	mRef;
	/// Flag to indicate that the command was reparsed
    SAPDB_Bool                  mReparsed;
	/// Flag to indicate wheather -8 or -9 should be returned to the application
    SAPDB_Bool                  mMinus9;
    //
}; // SharedSQL_ExecuteHandle

#endif  /* SHAREDSQL_HANDLES_HPP */
