/*!
    @file     SharedSQL_Command.cpp
    @ingroup  SharedSQL
    @author   DirkT
    @brief    SQLCommand cached by the SharedSQL_CommandCache
    @see            

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SQLManager/SharedSQL/SharedSQL_Command.hpp"
#include "SQLManager/SharedSQL/SharedSQL_CommandStatistics.hpp"
#include "SQLManager/SharedSQL/SharedSQL_ICommandCache.hpp"

#include "SQLManager/SharedSQL/SharedSQL_Plan.hpp"
#include "SQLManager/SharedSQL/SharedSQL_Types.hpp"

#include "SQLManager/SQLMan_Context.hpp"

#include "SAPDBCommon/SAPDB_Types.hpp"
#include "SAPDBCommon/Tracing/SAPDBTrace_Usage.hpp"

#include "RunTime/System/RTESys_AtomicOperation.hpp"
#include "RunTime/Synchronisation/RTESync_RWRegion.hpp"

#include "heo58.h"

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

SharedSQL_CachedCommand::SharedSQL_CachedCommand( SharedSQL_CommandID ID, SharedSQL_HashValue Hash, SharedSQL_HashValue Slot, 
							SharedSQL_ICommandCache& Cache, SAPDBMem_IRawAllocator& Allocator,
                            SharedSQL_SQLContext& Context, SharedSQL_SQLStmt& Stmt, SAPDB_Bool Internal,
                            RTESync_BaseSpinlockPool& SpinLockPool 
                            ) :
        mCommandCache(Cache),
        mpCommandLock ( RTESync_CreateRWRegion(ID, SpinLockPool, Allocator) ),
        mNextHashCommand(0),
        mPrevHashCommand(0),
	    mNextLRUCommand(0),
	    mPrevLRUCommand(0),
        mLRUOldFlg(false),
        mSQLStmt(Stmt),
        mPlan(ID, SpinLockPool, Allocator),

	mCommandID(ID),
	mHashValue(Hash),
    mSlot(Slot),
	mStatus(New),
	mFlgExtern(SAPDB_FALSE),
	mSQLContext(Context),
    mPrepareCount(0),
    mParseIDCount(0),	
    mExecuteCount(0),
    mStatistics(),
    mStatisticsReset(),
    mInternal(Internal)
{
};

SharedSQL_CachedCommand::~SharedSQL_CachedCommand( void ) 
{
    if (mpCommandLock)
        RTESync_DestroyRWRegion(mpCommandLock, *(mPlan.GetAllocator()));
};

SAPDB_Bool SharedSQL_CachedCommand::CompletelyConstructed( void )
{
    return ((mpCommandLock != NULL) && mPlan.CompletelyConstructed());
}

void SharedSQL_CachedCommand::FillCommandInfo( SharedSQL_CommandInfo& I )
{
  	I.mCommandID    = mCommandID;
    I.mInternal     = mInternal;
	I.mHashValue    = mHashValue;
    I.mSlot         = mSlot;
    //
    I.mSQLContext   = mSQLContext;
    //
    I.mStatus       = mStatus;
	I.mFlgExtern    = mFlgExtern;
    I.mPrepareCount = GetPrepareCount();
    I.mParseIDCount = GetParseIDCount();
    I.mExecuteCount = GetExecuteCount();
    //
    I.mStmtSize     = mSQLStmt.mStmtSize;
    I.mStmt         = mSQLStmt.mStmt;   // DDT: LongDescriptor
    I.mPlanSize     = mPlan.GetSize();
}

void SharedSQL_CachedCommand::GetCommandInfo( SharedSQL_CommandInfo& I )
{
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_CachedCommand::GetStatistics", SharedSQL_Trace, 5);

    I.mStatistics   = mStatistics;
    I.mStatistics  += mStatisticsReset;
    FillCommandInfo(I);
}

void SharedSQL_CachedCommand::GetCommandInfoReset( SharedSQL_CommandInfo& I )
{
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_CachedCommand::GetStatisticsReset", SharedSQL_Trace, 5);

    I.mStatistics   = mStatisticsReset;
    FillCommandInfo(I);
}

void SharedSQL_CachedCommand::ResetStatisticsInfo( void )
{
    SAPDBTRACE_METHOD_DEBUG ("SharedSQL_CachedCommand::ResetStatistics", SharedSQL_Trace, 5);

    mStatistics += mStatisticsReset;
    mStatisticsReset = SharedSQL_CommandStatistics();
}

SAPDB_Int4 SharedSQL_CachedCommand::GetPrepareCount() { return RTESys_AtomicRead( mPrepareCount ); }
void SharedSQL_CachedCommand::IncPrepareCount() { RTESys_AtomicModify( mPrepareCount,  1 ); }
void SharedSQL_CachedCommand::DecPrepareCount() { RTESys_AtomicModify( mPrepareCount, -1 ); }

SAPDB_Int4 SharedSQL_CachedCommand::GetParseIDCount() { return RTESys_AtomicRead( mParseIDCount ); }
void SharedSQL_CachedCommand::IncParseIDCount() { RTESys_AtomicModify( mParseIDCount,  1 ); }
void SharedSQL_CachedCommand::DecParseIDCount() { RTESys_AtomicModify( mParseIDCount, -1 ); }

SAPDB_Int4 SharedSQL_CachedCommand::GetExecuteCount() { return RTESys_AtomicRead( mExecuteCount ); }
void SharedSQL_CachedCommand::IncExecuteCount() { RTESys_AtomicModify( mExecuteCount,  1 ); }
void SharedSQL_CachedCommand::DecExecuteCount() { RTESys_AtomicModify( mExecuteCount, -1 ); }

SharedSQL_CommandStatus SharedSQL_CachedCommand::GetStatus( void ) { return mStatus; }
void SharedSQL_CachedCommand::SetStatus( SharedSQL_CommandStatus Status ) { mStatus = Status; }

SAPDB_Byte SharedSQL_CachedCommand::GetFlgExtern( void ) { return mFlgExtern; }
void SharedSQL_CachedCommand::SetFlgExtern( SAPDB_Byte FlgExtern ) { mFlgExtern = FlgExtern; }

SAPDB_Bool SharedSQL_CachedCommand::GetFlgLRUListOld( void ) { return mLRUOldFlg; }
void SharedSQL_CachedCommand::SetFlgLRUListOld( SAPDB_Bool Flg ) { mLRUOldFlg = Flg; }

SharedSQL_CommandStatistics&     SharedSQL_CachedCommand::Statistics( void ) { return mStatistics; }

SharedSQL_SQLStmt&               SharedSQL_CachedCommand::SQLStmt( void ) { return mSQLStmt; }
SharedSQL_SQLContext&            SharedSQL_CachedCommand::SQLContext( void ) { return mSQLContext; }
SharedSQL_IPlan& SharedSQL_CachedCommand::Plan( void ) { return mPlan; }

SharedSQL_ICommandCache& SharedSQL_CachedCommand::CommandCache( void ) { return mCommandCache; }

SharedSQL_CommandID SharedSQL_CachedCommand::GetCommandID( void ) { return mCommandID; }
SharedSQL_HashValue SharedSQL_CachedCommand::GetHashValue( void ) { return mHashValue; }
SAPDB_Bool          SharedSQL_CachedCommand::IsInternal( void ) { return mInternal; }

void SharedSQL_CachedCommand::SetInternal(SAPDB_Bool Internal)
{
    if (!Internal)
        mInternal = Internal;
}

const SharedSQL_SQLStmt SharedSQL_CachedCommand::GetSQLStmt( void )
{
    EnterLock(1);
    if (mFlgExtern) 
    {
        SQLMan_Context* pContext = (SQLMan_Context*) vGetAcvPtrFromCurrentTask();
        tgg00_TransContext& TransContext = pContext->TransContext();
		CommandCache().ReloadSQLStmt(TransContext, this);
    }
    LeaveLock(1);
   	return (mSQLStmt);
}

void SharedSQL_CachedCommand::EnterLock( bool exclusive ) { mpCommandLock->enter(exclusive); }
bool SharedSQL_CachedCommand::TryEnterLock( bool exclusive ) { return mpCommandLock->tryEnter(exclusive); }
void SharedSQL_CachedCommand::LeaveLock( bool exclusive ) { mpCommandLock->leave(exclusive); }

void SharedSQL_CachedCommand::IncCacheAccessCountExecute( void ) { mCommandCache.IncAccessCountExecute(); }
void SharedSQL_CachedCommand::IncCacheSuccessfulAccessCountExecute( void ) { mCommandCache.IncSuccessfulAccessCountExecute(); }
void SharedSQL_CachedCommand::IncCacheUnSuccessfulAccessCountExecute( void ) { mCommandCache.IncUnSuccessfulAccessCountExecute(); }

SharedSQL_ICachedCommand*        SharedSQL_CachedCommand::NextHashCommand( void ) const { return mNextHashCommand; }
SharedSQL_CachedCommand*&        SharedSQL_CachedCommand::NextHashCommand( void ) { return mNextHashCommand; }
SharedSQL_CachedCommand*&        SharedSQL_CachedCommand::PrevHashCommand( void ) { return mPrevHashCommand; }
SharedSQL_CachedCommand*&        SharedSQL_CachedCommand::NextLRUCommand( void )  { return mNextLRUCommand; }
SharedSQL_CachedCommand*&        SharedSQL_CachedCommand::PrevLRUCommand( void )  { return mPrevLRUCommand; }


