/*!
   @file           Log_ActionRefreshSurrogSeq.hpp
   @author         JuergenA
   @author         UweH
   @ingroup        ActionHandling
   @brief          defines a class to handle log entries for surrogates and sequences

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
 */



#ifndef LOG_ACTION_REFRESH_SURROG_SEQ_H
#define LOG_ACTION_REFRESH_SURROG_SEQ_H

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SAPDBCommon/SAPDB_Types.hpp"
#include "SAPDBCommon/MemoryManagement/SAPDBMem_IRawAllocator.hpp"
#include "Logging/Log_Types.hpp"
#include "Logging/Log_ReadWriteActionImage.hpp"

#include "gsp00.h"
#include "ggg00.h"

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/


//===================================================================================
/*!
   @brief          defines a class to handle the after image of refresh surrogate
   @copydoc        Log_ReadWriteActionImage
 */

class Log_ActionRefreshSurrogate: public Log_ReadWriteActionImage
{

private:
    
    /// physical representation in the Redo/Undo-File
    typedef struct
    {
        tgg00_Surrogate Surrogate;      /// refreshed surrogate 
        tgg00_Surrogate SysKey;         /// Syskey
    } RefreshSurrogateEntry;
    
    /// in memory representation
    RefreshSurrogateEntry m_Entry;

public:
    
    /// this constructor is designed for a further call of ReadImagePersistent
    inline Log_ActionRefreshSurrogate ();

    /// this constructor is designed for a further call of WriteImagePersistent
    Log_ActionRefreshSurrogate (
        const tgg00_Surrogate &Surrogate,
        const tgg00_Surrogate &SysKey);

    /// redo only: refreshes the last surrogate and the last system key
    void Execute (tgg00_TransContext &TransContext) const;
    
    /// the savepoint depends on writing this entry. must not be suspended.
    virtual bool ForceLogWriting () const
    {
        return true;
    }

    virtual SAPDB_UInt GetPersistentLength () const
    {
        return ( Log_AlignedImageLen (sizeof(m_Entry)) );
    }

    virtual void ReadImagePersistent (
        Log_ActionImageReader &ImageReader,
        bool                  &IsOk);
    
    virtual void WriteImagePersistent (
        Log_ActionImageWriter &ImageWriter,
        bool                  &IsOk)       const
    {
        ImageWriter.PersistentWrite (&m_Entry, sizeof (m_Entry), IsOk);
    }
    
};



//===================================================================================
/*!
   @brief          defines a class to handle the after image of refresh sequence
   @copydoc        Log_ReadWriteActionImage
 */

class Log_ActionRefreshSequence: public Log_ReadWriteActionImage
{
public:
    
    typedef struct
    {
        tgg00_Surrogate Surrogate;
        tsp00_Number    Number;
    } SequenceItem;
    
    
private:
    
    SAPDB_Int4              m_SequenceCount;
    const SequenceItem     *m_pSequenceList;
    
    SequenceItem           *m_pSequenceListForRead;
    SAPDBMem_IRawAllocator *m_pSequenceListAllocator;
    
public:
    
    /// this constructor is designed for a further call of ReadImagePersistent
    inline Log_ActionRefreshSequence ();

    /// this constructor is designed for a further call of WriteImagePersistent
    inline Log_ActionRefreshSequence (
        SAPDB_Int4  SequenceCount,
        const void *pSequenceList);

    /// deallocation if necessary
    ~Log_ActionRefreshSequence ();

    /// redo only: refreshes the sequences
    void Execute (tgg00_TransContext &TransContext) const;
    

	// PTS 1117320 UH 2002-10-28 added
    virtual bool ForceLogWriting () const
    {
        return true;
    }

    virtual SAPDB_UInt GetPersistentLength () const
    {
        return ( Log_AlignedImageLen (sizeof(m_SequenceCount))
            +    Log_AlignedImageLen (m_SequenceCount * sizeof(SequenceItem)) );
    }

    virtual void ReadImagePersistent (
        Log_ActionImageReader &ImageReader,
        bool                  &IsOk);
    
    virtual void WriteImagePersistent (
        Log_ActionImageWriter &ImageWriter,
        bool                  &IsOk)       const;
};



/*==========================================================================*
*  DEFINITION OF INLINE METHODS                                             *
*==========================================================================*/

inline Log_ActionRefreshSurrogate::Log_ActionRefreshSurrogate ():

Log_ReadWriteActionImage (Log_RefreshSurrogate)
{
}

//---------------------------------------------------------------------------

inline Log_ActionRefreshSequence::Log_ActionRefreshSequence ():

Log_ReadWriteActionImage (Log_RefreshSequence),
m_SequenceCount          (0),
m_pSequenceList          (NULL),
m_pSequenceListForRead   (NULL),
m_pSequenceListAllocator (NULL)
{
}

//---------------------------------------------------------------------------

inline Log_ActionRefreshSequence::Log_ActionRefreshSequence (SAPDB_Int4  SequenceCount,
                                                             const void *pSequenceList):
Log_ReadWriteActionImage (Log_RefreshSequence),
m_SequenceCount          (SequenceCount),
m_pSequenceList          (REINTERPRET_CAST (const SequenceItem*, pSequenceList)),
m_pSequenceListForRead   (NULL),
m_pSequenceListAllocator (NULL)
{
}

#endif  /* LOG_ACTION_REFRESH_SURROG_SEQ_H */
