/*!
  -----------------------------------------------------------------------------
  module: database.cpp
  -----------------------------------------------------------------------------
  responsible:  MartinR
  special area: DBanalyzer - The performance analyse tool for SAPDB
  description:  odbc database connection 
  version:      7.4 and higher 
  -----------------------------------------------------------------------------

  Copyright (c) 2002-2004 SAP AG


    ========== licence begin  GPL
    Copyright (c) 2002-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



  -----------------------------------------------------------------------------
*/

#include "SAPDB/SQLStudio/OAL/StudioOAL_WCommon.h"
#include "SAPDB/SQLStudio/OAL/StudioOAL_WResult.hpp"
#include "SAPDB/SQLStudio/OAL/StudioOAL_WError.hpp"
#include "SAPDB/DBanalyzer/dbanalyzer.hpp"
#include "SAPDB/DBanalyzer/database.hpp"
#include "SAPDB/DBanalyzer/error.hpp"

extern DBA_error gError;

#define DBA_SQL_PARAM_INSTANCETYPE  "SELECT VALUE FROM DBPARAMETERS WHERE DESCRIPTION = 'INSTANCE_TYPE'"
#define DBA_SQL_INFO_KERNELVERSION  "SELECT KERNEL FROM VERSIONS"
#define DBA_SQL_PARAM_RUNDIRECTORY  "SELECT VALUE FROM DBPARAMETERS WHERE DESCRIPTION = 'RUNDIRECTORY'"
#define DBA_SQL_INFO_SESSION        "SELECT SESSION FROM CONNECTPARAMETERS"
#define DBA_SQL_INFO_PID            "SELECT T.APPLPROCESS FROM TRANSACTIONS T,CONNECTPARAMETERS C WHERE T.SESSION = C.SESSION"
#define DBA_SQL_CHECK_ANALYZER      "SELECT T.APPLPROCESS FROM TRANSACTIONS T,CONNECTPARAMETERS C WHERE T.SESSION = %u AND T.APPLPROCESS = %u"
#define DBA_SQL_INFO_DATETIME       "SELECT YEAR(NOW()),MONTH(NOW()),DAY(NOW()),HOUR(NOW()),MINUTE(NOW()),SECOND(NOW()) FROM DUAL"

class DBA_handles {
private:
  DBA_handles()
    : hEnv(NULL),
      hDbc(NULL)
  {}
  SQLHANDLE  hEnv;
  SQLHANDLE  hDbc;
  friend class DBA_database; 
}; // end class DBA_handles

// ========================================================
// DBA_database::DBA_database
// ========================================================
DBA_database::DBA_database (const DBA_string & sServer,
                            const DBA_string & sDatabase,
                            const DBA_string & sUser,
                            const DBA_string & sPassword) 
             : m_sServer(sServer),
               m_sDatabase(sDatabase),
               m_sUser(sUser),
               m_sPassword(sPassword)
{
  m_bReconnectEnabled = false;
  m_bReconnect = false;

  m_pError = NULL;

  m_pHandles = new DBA_handles();

  m_nInstanceType  = DBA_INSTANCETYPE_OLTP;
  m_sKernelVersion = "";
  m_sRundirectory  = "";
  m_nSid           = 0;
  m_nPid           = 0;
} // end DBA_database::DBA_database

// ========================================================
// DBA_database::~DBA_database
// ========================================================
DBA_database::~DBA_database () 
{
  delete m_pHandles;
} // end DBA_database::~DBA_database

// ========================================================
// DBA_database::Connect
// ========================================================
DBA_bool DBA_database::Connect()
{
  gError.Clear(); 

  if (SQLAllocHandle(SQL_HANDLE_ENV, SQL_NULL_HANDLE, &m_pHandles->hEnv) != SQL_SUCCESS) {
    return false;
  }
  
  if (SQLSetEnvAttr(m_pHandles->hEnv, SQL_ATTR_ODBC_VERSION, (SQLPOINTER) SQL_OV_ODBC3, 0) != SQL_SUCCESS) {
    SQLFreeHandle(SQL_HANDLE_ENV, m_pHandles->hEnv);
    m_pHandles->hEnv = NULL;
    return false;
  }

  if (SQLAllocHandle(SQL_HANDLE_DBC, m_pHandles->hEnv, &m_pHandles->hDbc) != SQL_SUCCESS) {
    SQLFreeHandle(SQL_HANDLE_ENV, m_pHandles->hEnv);
    m_pHandles->hEnv = NULL;
    return false;
  }

  if (SQLSetConnectAttr( m_pHandles->hDbc, SQL_ATTR_SESSION_TIMEOUT, (SQLPOINTER) 0, 0 ) != SQL_SUCCESS) {
    SQLFreeHandle(SQL_HANDLE_DBC, m_pHandles->hDbc);
    SQLFreeHandle(SQL_HANDLE_ENV, m_pHandles->hEnv);
    m_pHandles->hEnv = NULL;
    m_pHandles->hDbc = NULL;
    return false;
  } // end if

  if (SQLSetConnectAttr( m_pHandles->hDbc, SQL_TXN_ISOLATION, (SQLPOINTER) SQL_TXN_READ_UNCOMMITTED, 0 ) != SQL_SUCCESS) {
    SQLFreeHandle(SQL_HANDLE_DBC, m_pHandles->hDbc);
    SQLFreeHandle(SQL_HANDLE_ENV, m_pHandles->hEnv);
    m_pHandles->hEnv = NULL;
    m_pHandles->hDbc = NULL;
    return false;
  } // end if
  
  if (SQLSetConnectAttr( m_pHandles->hDbc, SQL_PRODUCER, (SQLPOINTER) sp1pr_internal_cmd, 0 ) != SQL_SUCCESS) {
    SQLFreeHandle(SQL_HANDLE_DBC, m_pHandles->hDbc);
    SQLFreeHandle(SQL_HANDLE_ENV, m_pHandles->hEnv);
    m_pHandles->hEnv = NULL;
    m_pHandles->hDbc = NULL;
    return false;
  } // end if

  // create the error object for odbc
  m_pError = new StudioOAL_WError(m_pHandles->hDbc, m_pHandles->hEnv);
  if (!m_pError) {
    SQLFreeHandle(SQL_HANDLE_DBC, m_pHandles->hDbc);
    SQLFreeHandle(SQL_HANDLE_ENV, m_pHandles->hEnv);
    m_pHandles->hEnv = NULL;
    m_pHandles->hDbc = NULL;
    gError.SetError(DBA_ERR_NOMEM);
    return false;
  }

  SQLCHAR   sConnect[1024];
  SQLCHAR   sOut[1024];
  DBA_short nOut;
  RETCODE   nRC;

  if (m_sUser[0] != '"') {
    m_sUser.ToUpper();
  } // end if
  if (m_sPassword[0] != '"') {
    m_sPassword.ToUpper();
  } // end if

  m_sDriver = "SAP DB <MAJOR_VERSION>.<MINOR_VERSION>";
  sprintf((char *)sConnect, "DRIVER=%s;SERVERDB=%s;SERVERNODE=%s;UID=%s;PWD=%s",
          m_sDriver.CharPtr(),
          m_sDatabase.CharPtr(),
          m_sServer.CharPtr(),
          m_sUser.CharPtr(),
          m_sPassword.CharPtr());
  

  nRC = SQLDriverConnect(m_pHandles->hDbc, (SQLHWND)NULL, sConnect, SQL_NTS, sOut, 
                         sizeof(sOut), &nOut, SQL_DRIVER_NOPROMPT);

  bool bError =  (!m_pError->checkSQLReturnCode(nRC, NULL));

  if (bError && (m_pError->getSQLStateErrorNumber() == "IM002")) {
    m_sDriver = "SAP DB";
    sprintf((char *)sConnect, "DRIVER=%s;SERVERDB=%s;SERVERNODE=%s;UID=%s;PWD=%s",
            m_sDriver.CharPtr(),
            m_sDatabase.CharPtr(),
            m_sServer.CharPtr(),
            m_sUser.CharPtr(),
            m_sPassword.CharPtr());
    nRC = SQLDriverConnect(m_pHandles->hDbc, (SQLHWND)NULL, sConnect, SQL_NTS, sOut, 
                          sizeof(sOut), &nOut, SQL_DRIVER_NOPROMPT);
    bError = (!m_pError->checkSQLReturnCode(nRC, NULL));
  } // end if

  if (bError) {
    CheckReconnectError(m_pError->getNativeError());
    if (!m_bReconnectEnabled || !m_bReconnect) {
      SQLFreeHandle(SQL_HANDLE_DBC, m_pHandles->hDbc);
      SQLFreeHandle(SQL_HANDLE_ENV, m_pHandles->hEnv);
      m_pHandles->hEnv = NULL;
      m_pHandles->hDbc = NULL;
      gError.SetError(DBA_ERR_CONNECT);
      gError.SetExtText(GetErrorMsg());
      return false;
    } else {
      gError.SetReconnect(true);
      return true;
    }
  }

  if(!GetInfos()) {
    return false;
  }

  return true;
} // DBA_database::Connect

// ========================================================
// DBA_database::Execute
// ========================================================
DBA_bool DBA_database::Execute(const DBA_string & sSQL, DBA_snapshot * pSnapshot) 
{
  StudioOAL_WResult   oResult(m_pHandles->hDbc, m_pHandles->hEnv);
  StudioOAL_WResult * pResult = &oResult;
  if (!PrepareQuery(sSQL, pResult)) {
    return false;
  } // end if
  return Execute(sSQL, pSnapshot, pResult);
} // end DBA_database::Execute

// ========================================================
// DBA_database::PrepareQuery
// ========================================================
DBA_bool DBA_database::PrepareQuery (const DBA_string & sSQL, StudioOAL_WResult * & pPreparedQuery)
{
  gError.Clear(); 

  if (pPreparedQuery == NULL) {
    pPreparedQuery = new StudioOAL_WResult(m_pHandles->hDbc, m_pHandles->hEnv);
    if (pPreparedQuery == NULL) {
      gError.SetError(DBA_ERR_NOMEM);
      return false;
     } // end if
  } // end if

  if (!pPreparedQuery->openResultPrepare (sSQL)) {
    gError.SetError(DBA_ERR_PREPARE);
    DBA_string sError = pPreparedQuery->getErrorText();
    gError.SetExtText(sError.Append("\n").Append(sSQL));
    return false;
  } // end if

  return true;
} // end DBA_database::PrepareQuery

// ========================================================
// DBA_database::Execute
// ========================================================
DBA_bool DBA_database::Execute (const DBA_string & sSQL, DBA_snapshot * pSnapshot, StudioOAL_WResult * & pPreparedQuery)
{
  gError.Clear(); 

  if (m_bReconnectEnabled && m_bReconnect) {
    Disconnect();
    if (!Connect()) {
      return false;
    } else {
      if (gError.GetReconnect()) { 
        return true;
      } else {
        m_bReconnect = false;
        if (!PrepareQuery(sSQL, pPreparedQuery)) {
          return false;
        } // end if
      } // end if
    } // end if
  } // end if

  gError.SetIgnore(false);

  if (pPreparedQuery == NULL) {
    if (!PrepareQuery(sSQL, pPreparedQuery)) {
      return false;
    } // end if
  } // end if

  if (!pPreparedQuery->executePreparedStmt()) {
    CheckReconnectError(pPreparedQuery->getNativeError());
    if (!m_bReconnectEnabled || !m_bReconnect) {
      if (CheckIgnoreError(pPreparedQuery->getNativeError())) {
        gError.SetError(DBA_ERR_EXECUTE);
        DBA_string sError = pPreparedQuery->getErrorText();
        gError.SetExtText(sError.Append("\n").Append(sSQL));
        gError.SetIgnore(true);
        return true;
      } else {
        gError.SetError(DBA_ERR_EXECUTE);
        DBA_string sError = pPreparedQuery->getErrorText();
        gError.SetExtText(sError.Append("\n").Append(sSQL));
        // prepare again
        if (!PrepareQuery(sSQL, pPreparedQuery)) {
          return false;
        } // end if
        // excute again
        if (!pPreparedQuery->executePreparedStmt()) {
          gError.SetError(DBA_ERR_EXECUTE);
          DBA_string sError = pPreparedQuery->getErrorText();
          gError.SetExtText(sError.Append("\n").Append(sSQL));
          return false;
        } // end if
      } // end if
    } else {
      gError.SetReconnect(true);
      return true;
    }
  }

  if (pSnapshot) {
    pSnapshot->Clear();  

    DBA_int nCols = pPreparedQuery->getColCount();
    DBA_int nCol;
    
    for (nCol = 0; nCol < nCols; nCol++) {
      DBA_string sName;
      pPreparedQuery->getColumnName(nCol+1, sName);
      if (!pSnapshot->AddColumn(sName)) {
        return false;
      }
    }

    DBA_bool bFetch = true;
  
    while (bFetch) {
      bFetch = pPreparedQuery->fetch();
      if(!bFetch) {
        CheckReconnectError(pPreparedQuery->getNativeError());
        if (!m_bReconnectEnabled || !m_bReconnect) {
          if (CheckIgnoreError(pPreparedQuery->getNativeError())) {
            gError.SetError(DBA_ERR_EXECUTE);
            DBA_string sError = pPreparedQuery->getErrorText();
            gError.SetExtText(sError.Append("\n").Append(sSQL));
            gError.SetIgnore(true);
            return true;
          } else {
            gError.SetError(DBA_ERR_EXECUTE);
            DBA_string sError = pPreparedQuery->getErrorText();
            gError.SetExtText(sError.Append("\n").Append(sSQL));
            return false;
          } // end if
        } else {
          gError.SetReconnect(true);
          pPreparedQuery->closeCursor ();
          return true;
        }
      } else {
        if (pPreparedQuery->noRows()) {
          pPreparedQuery->closeCursor ();
          return true;
        } else {
          if (pSnapshot->AddRow()) {
            for (nCol = 0; nCol < nCols; nCol++) {
              DBA_string sValue;
              pPreparedQuery->getColumnDataAsString(nCol+1, sValue);
              if (!pSnapshot->AddField(sValue)) {
                return false;
              }
            }
          }
        }
      }
    }

  } 

  pPreparedQuery->closeCursor ();
  return true;
} // end DBA_database::Execute

// ========================================================
// DBA_database::Disconnect
// ========================================================
DBA_bool DBA_database::Disconnect ()
{
  SQLDisconnect(m_pHandles->hDbc);
  SQLFreeHandle(SQL_HANDLE_DBC, m_pHandles->hDbc);
  SQLFreeHandle(SQL_HANDLE_ENV, m_pHandles->hEnv);

  m_pHandles->hEnv = NULL;
  m_pHandles->hDbc = NULL;

  if(m_pError) {
    delete m_pError;
    m_pError = NULL;
  }

  return true;
} // end DBA_database::Disconnect

// ========================================================
// DBA_database::GetTimestamp
// ========================================================
DBA_bool DBA_database::GetTimestamp(time_t & tTime)
{
  StudioOAL_WResult oResult(m_pHandles->hDbc, m_pHandles->hEnv);
  gError.Clear(); 

  // select date and time
  if (!oResult.openResult(DBA_SQL_INFO_DATETIME)) {
    gError.SetError(DBA_ERR_EXECUTE);
    DBA_string sError = oResult.getErrorText();
    gError.SetExtText(sError.Append("\n").Append(DBA_SQL_INFO_DATETIME));
    return false;
  } // end if

  if (!oResult.fetch()) {
    gError.SetError(DBA_ERR_EXECUTE);
    gError.SetExtText(oResult.getErrorText());
    return false;
  } // end if
  
  if (!oResult.noRows()) {
    struct tm oTime;
    memset(&oTime, 0, sizeof(oTime));
    DBA_string sColumn = "";
    oResult.getColumnDataAsString(1, sColumn);
    oTime.tm_year = atoi(sColumn.CharPtr()) - 1900;
    oResult.getColumnDataAsString(2, sColumn);
    oTime.tm_mon  = atoi(sColumn.CharPtr()) - 1;
    oResult.getColumnDataAsString(3, sColumn);
    oTime.tm_mday = atoi(sColumn.CharPtr());
    oResult.getColumnDataAsString(4, sColumn);
    oTime.tm_hour = atoi(sColumn.CharPtr());
    oResult.getColumnDataAsString(5, sColumn);
    oTime.tm_min  = atoi(sColumn.CharPtr());
    oResult.getColumnDataAsString(6, sColumn);
    oTime.tm_sec  = atoi(sColumn.CharPtr());
    oTime.tm_isdst = -1;
    tTime = mktime(&oTime);
  } else {
    gError.SetError(DBA_ERR_EXECUTE);
    gError.SetExtText(DBA_SQL_INFO_DATETIME);
    tTime = 0;
    return false;
  } // end if

  return true;
} // end DBA_database::GetTimestamp

// ========================================================
// DBA_database::GetInfos
// ========================================================
// get some infos from the database
//
// INSTANCE_TYPE
// KERNELVERSION
// RUNDIRECTORY
// SESSION
// APPLPROCESS
//
DBA_bool DBA_database::GetInfos()
{
  StudioOAL_WResult oResult(m_pHandles->hDbc, m_pHandles->hEnv);

  gError.Clear(); 

  // get INSTANCE_TYPE from table DBPARAMETERS
  if (!oResult.openResult(DBA_SQL_PARAM_INSTANCETYPE)) {
    gError.SetError(DBA_ERR_EXECUTE);
    DBA_string sError = oResult.getErrorText();
    gError.SetExtText(sError.Append("\n").Append(DBA_SQL_PARAM_INSTANCETYPE));
    return false;
  }

  if (!oResult.fetch()) {
    gError.SetError(DBA_ERR_EXECUTE);
    gError.SetExtText(oResult.getErrorText());
    return false;
  }
  
  if (!oResult.noRows()) {
    DBA_string sInstanceType;
    oResult.getColumnDataAsString(1, sInstanceType);
    if (sInstanceType.Find(DBA_INSTANCETYPE_TXT_OLTP) != DBA_string::NPos) {
      m_nInstanceType = DBA_INSTANCETYPE_OLTP;
    } else if (sInstanceType.Find(DBA_INSTANCETYPE_TXT_LVC) != DBA_string::NPos) {
      m_nInstanceType = DBA_INSTANCETYPE_LVC;
    }
  }

  oResult.closeResult();

  // get KERNEL from table VERSIONS
  if (!oResult.openResult(DBA_SQL_INFO_KERNELVERSION)) {
    gError.SetError(DBA_ERR_EXECUTE);
    DBA_string sError = oResult.getErrorText();
    gError.SetExtText(sError.Append("\n").Append(DBA_SQL_INFO_KERNELVERSION));
    return false;
  }

  if (!oResult.fetch()) {
    gError.SetError(DBA_ERR_EXECUTE);
    gError.SetExtText(oResult.getErrorText());
    return false;
  }
  
  if (!oResult.noRows()) {
    oResult.getColumnDataAsString(1, m_sKernelVersion);
//    sscanf(m_sKernelVersion.CharPtr(), "Kernel    %d.%d.%d    Build %d-000-000-000", &m_nMajor, &m_nMinor, &m_nLevel, &m_nBuild);
    sscanf(m_sKernelVersion.CharPtr(), "%*s %d.%d.%d Build %d-000-000-000", &m_nMajor, &m_nMinor, &m_nLevel, &m_nBuild);
  }

  oResult.closeResult();

  // get RUNDIRECTORY from table DBPARAMETERS
  if (!oResult.openResult(DBA_SQL_PARAM_RUNDIRECTORY)) {
    gError.SetError(DBA_ERR_EXECUTE);
    DBA_string sError = oResult.getErrorText();
    gError.SetExtText(sError.Append("\n").Append(DBA_SQL_PARAM_RUNDIRECTORY));
    return false;
  }

  if (!oResult.fetch()) {
    gError.SetError(DBA_ERR_EXECUTE);
    gError.SetExtText(oResult.getErrorText());
    return false;
  }
  
  if (!oResult.noRows()) {
    oResult.getColumnDataAsString(1, m_sRundirectory);
  }

  oResult.closeResult();

  // get SESSION from table CONNECTPARAMETERS
  if (!oResult.openResult(DBA_SQL_INFO_SESSION)) {
    gError.SetError(DBA_ERR_EXECUTE);
    DBA_string sError = oResult.getErrorText();
    gError.SetExtText(sError.Append("\n").Append(DBA_SQL_INFO_SESSION));
    return false;
  }

  if (!oResult.fetch()) {
    gError.SetError(DBA_ERR_EXECUTE);
    gError.SetExtText(oResult.getErrorText());
    return false;
  }
  
  if (!oResult.noRows()) {
    DBA_string sSession;
    oResult.getColumnDataAsString(1, sSession);
    sscanf(sSession.CharPtr(), "%u", &m_nSid);
  }

  oResult.closeResult();

  // get APPLPROCESS  from table TRANSACTIONS
  if (!oResult.openResult(DBA_SQL_INFO_PID)) {
    gError.SetError(DBA_ERR_EXECUTE);
    DBA_string sError = oResult.getErrorText();
    gError.SetExtText(sError.Append("\n").Append(DBA_SQL_INFO_PID));
    return false;
  }

  if (!oResult.fetch()) {
    gError.SetError(DBA_ERR_EXECUTE);
    gError.SetExtText(oResult.getErrorText());
    return false;
  }
  
  if (!oResult.noRows()) {
    DBA_string sSession;
    oResult.getColumnDataAsString(1, sSession);
    sscanf(sSession.CharPtr(), "%u", &m_nPid);
  }

  oResult.closeResult();

//
  
  return true;
} // DBA_database::GetInfos

//
// SetReconnect
//
DBA_void DBA_database::ReconnectEnabled(const DBA_bool bEnabled) 
{
  m_bReconnectEnabled = bEnabled;
}

//
// IsReconnectError
//
DBA_bool DBA_database::CheckReconnectError(DBA_long nError)
{
  m_bReconnect = false;

  switch (nError) {
    case DBA_ODBC_RECONNECT_ERR1:
    case DBA_ODBC_RECONNECT_ERR2:
    case DBA_ODBC_RECONNECT_ERR3:
    case DBA_ODBC_RECONNECT_ERR4:
    case DBA_ODBC_RECONNECT_ERR5:
    case DBA_ODBC_RECONNECT_ERR6:
    case DBA_ODBC_RECONNECT_ERR7:
      m_bReconnect = true;
      return true;
    default:
      return false;
  }
}

//
// DBA_database::SetIgnoreErrors
//
void DBA_database::SetIgnoreErrors(DBA_string sIgnoreErrors)
{
  DBA_string::BasisElementIndex nKomma;
  DBA_string::BasisElementIndex nCurrent = 0;
  DBA_long                      nError;

  nKomma = sIgnoreErrors.Find(',', nCurrent);
  while (nKomma != DBA_string::NPos) {
    nError = atol(sIgnoreErrors.SubStr(nCurrent, nKomma - nCurrent).CharPtr());
    m_oIgnoreErrors.push_back(nError);
    nCurrent = nKomma + 1;
    nKomma = sIgnoreErrors.Find(',', nCurrent);
  } // end while
  // last Element
  if (sIgnoreErrors.SubStr(nCurrent).Length() > 0) {
    nError = atol(sIgnoreErrors.SubStr(nCurrent).CharPtr());
    m_oIgnoreErrors.push_back(nError);
  } // end if

} // end DBA_database::SetIgnoreErrors

//
// DBA_database::CheckIgnoreError
//
DBA_bool DBA_database::CheckIgnoreError(DBA_long nError)
{
  ErrorList::iterator iter = m_oIgnoreErrors.begin();
  while (iter != m_oIgnoreErrors.end()) {
    if ((*iter) == nError) {
      return true;;
    } // end if
    ++iter;
  } // end while

  return false;
} // end DBA_database::CheckIgnoreError

//
// GetErrorMsg
//
DBA_string DBA_database::GetErrorMsg()
{
  DBA_string sMsg = "";

  if (m_pError->getNativeError() != 0) {
    sMsg.Append(m_pError->getNativeErrorAsString());
    sMsg.Append(" ");
    sMsg.Append(m_pError->getErrorText());
  } else {
    sMsg.Append(m_pError->getErrorText());
  }
  return sMsg;
}

//
// CheckSid
// 
DBA_bool DBA_database::CheckAnalyzer(DBA_uint nSid, DBA_uint nPid, const DBA_string & sDir)
{
  char              szStatement[200];
  StudioOAL_WResult oResult(m_pHandles->hDbc, m_pHandles->hEnv);
  
  gError.Clear(); 

  sprintf(szStatement, DBA_SQL_CHECK_ANALYZER, nSid, nPid);

  // select a record
  if (!oResult.openResult(szStatement)) {
    gError.SetError(DBA_ERR_EXECUTE);
    DBA_string sError = oResult.getErrorText();
    gError.SetExtText(sError.Append("\n").Append(szStatement));
    return false;
  } // end if

  if (!oResult.fetch()) {
    gError.SetError(DBA_ERR_EXECUTE);
    gError.SetExtText(oResult.getErrorText());
    return false;
  } // end if
  
  if (!oResult.noRows()) {
    // active session found
    gError.SetError(DBA_ERR_ACTIVE, sDir);
    return false;
  } else {
    gError.SetError(DBA_ERR_NOTACTIVE, sDir);
  } // end if

  oResult.closeResult();

  return true;
} // end DBA_database::CheckSid
