/*!
  @file           Converter_BackUpPageNoContainer.hpp
  @author         TorstenS
  @author         AlexanderK
  @ingroup        Converter
  @brief          Container which holds all page numbers that belong
                  to a back up
*/

/* ========== licence begin  GPL
  Copyright (c) 2001-2004 SAP AG

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  ========== licence end */


#ifndef CONVERTER_BACKUPPAGENOCONTAINER_HPP
#define CONVERTER_BACKUPPAGENOCONTAINER_HPP

/*===========================================================================*
 *  INCLUDES                                                                 *
 **/

#include "DataAccess/Data_Types.hpp"
#include "Converter/Converter_Types.hpp"
#include "Container/Container_Vector.hpp"
#include "Converter/Converter_FlagInfo.hpp"
#include "RunTime/Synchronisation/RTESync_InterlockedCounter.hpp"

/*!
  @class Converter_BackUpPageNoContainer
  @brief          Container which holds all page numbers that belong
                  to a back up
 */


class Converter_BackUpPageNoContainer
{

public:

    /*!
    @brief          Deletes container structures; deallocates memory
    @return         none
    */
    void Delete()
    {
        m_FlagInfoPtrVector.Delete();
        m_FlagInfoVector.Delete();
        m_NumUsedFlagInfoBlocks = 0;
        m_NumFlagInfoBlocks     = 0;
        m_PageNoCount           = 0;
    }

    /*!
       @brief          Constructor
       @param          Allocator [in] allocator for container resources
       @return         none
    */
    Converter_BackUpPageNoContainer  (SAPDBMem_IRawAllocator&  Allocator)
    : 
     m_FlagInfoPtrVector(Allocator),
     m_FlagInfoVector(Allocator)
    {
        Delete();
    }


    /*!
       @brief          Initializes container
       @param          MaxPageNo [in] maximum page number to be stored in the container
       @param          NumFlagInfoBlocks [in] 
       @param          NumPageNosPerFlagInfoBlock [in] 
       @return         (Converter_ReturnCode) if the memory for the container cannot be allocated 
                       the value Converter_MemoryAllocationFailed is returned 
                       otherwise Converter_Ok
     */
    Converter_ReturnCode  Initialize (SAPDB_UInt  MaxPageNo,
                            SAPDB_UInt  NumFlagInfoBlocks,
                            SAPDB_UInt  NumPageNosPerFlagInfoBlock);

    /*!
    @brief          inserts a page number into the container
    @param          PageNo [in] page number to insert
    @return         (Converter_ReturnCode)  in case the page number is already included
                    the value Converter_DuplicatePageNo is returned otherwise Converter_Ok.

    */
    Converter_ReturnCode Insert(Data_PageNo  PageNo);

    /*!
    @brief          Checks whether container is empty
    @return         (SAPDB_Bool) true if empty
    */
    SAPDB_Bool IsEmpty() const
    {
        return m_PageNoCount == 0;
    }
    
    /*!
    @brief          Returns total number of pagenos stored in container
    @return         (SAPDB_UInt) number of pagenos stored in container
    */
    SAPDB_UInt GetPageNoCount() const
    {
        return m_PageNoCount;
    }
    
    /*!
    @brief          Removes a page number from the container
    @param          PageNo [in] page number to be removed
    @return         (SAPDB_Bool) true if pageno found in container and removed
    */

    SAPDB_Bool Remove (Data_PageNo  PageNo);

private: // methods

    /*!
    @brief  Checks whether container is initialized
    @return (SAPDB_Bool) true if initialized
    */
    SAPDB_Bool IsInitialized() const
    {
        return (m_FlagInfoPtrVector.GetSize() > 0) && (m_FlagInfoVector.GetSize() > 0) ;
    }

private: // members

    /// ...
    Container_Vector<Converter_FlagInfo::FlagInfoData *>    m_FlagInfoPtrVector;
    /// ...
    Container_Vector<Converter_FlagInfo::FlagInfoData>      m_FlagInfoVector;
    /// ...
    RTESync_InterlockedCounter<SAPDB_UInt>                  m_PageNoCount;
    /// ...
    SAPDB_UInt                                              m_NumUsedFlagInfoBlocks;
    /// ...
    SAPDB_UInt                                              m_NumFlagInfoBlocks;
    /// ...
    SAPDB_UInt                                              m_NumPageNosPerFlagInfoBlock;
    /// ...
    SAPDB_UInt                                              m_SizeOfFlagInfoBlock;
};





#endif  /* CONVERTER_BACKUPPAGENOCONTAINER_HPP */

