/*
 *  Copyright (C) 2000-2001 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "eel-gconf-extensions.h"
#include "glade.h"
#include "misc_gui.h"
#include "prefs_utils.h"
#include "mozilla.h"
#include "dialog.h"
#include "window.h"
#include "embed.h"

#include <libgnomeui/gnome-dialog-util.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkspinbutton.h>

typedef struct
{
	GtkWidget *print_dialog;
	GtkWidget *print_entry;
	GtkWidget *file_entry;
	GtkWidget *print_rb;
	GtkWidget *letter_rb;
	GtkWidget *top_entry;
	GtkWidget *bottom_entry;
	GtkWidget *left_entry;
	GtkWidget *right_entry;

	/* the 'gtk notebook' tabbed thingy. */
	GtkWidget *print_notebook;

	/* headers and footers stuff */
	GtkWidget *print_page_title_checkbutton;
	GtkWidget *print_page_url_checkbutton;
	GtkWidget *print_page_numbers_checkbutton;
	GtkWidget *print_date_checkbutton;

	GtkWidget *all_pages_rb;
	GtkWidget *from_spinbutton;
	GtkWidget *to_spinbutton;
	GtkWidget *first_rb;
	GtkWidget *print_color_rb;
	GtkWidget *orient_p_rb;
	GtkWidget *frame_no_rb;
	GaleonEmbed *embed;
}
PrintWidgets;

typedef struct
{
	GtkWidget *dialog;
	GtkWidget *pageSpinner;
	GtkWidget *pageTotal;
	gint numPages;

	gboolean menuBarState;
	gboolean toolBarState;
	gboolean bookmarkBarState;
	gboolean statusBarState;
	gboolean notebookState;
	gboolean dockState;

	GaleonEmbed *embed;
}
PrintPreviewData;

typedef enum
{
	PRINTPREVIEW_GOTO_PAGENUM = 0,
	PRINTPREVIEW_PREV_PAGE = 1,
	PRINTPREVIEW_NEXT_PAGE = 2,
	PRINTPREVIEW_HOME = 3,
	PRINTPREVIEW_END = 4
} printPreviewNavType;


static PrintInfo * print_get_info (PrintWidgets *pw);
static void print_info_free (PrintInfo *info);
static void print_options_set (PrintWidgets *widgets);
static void print_lookup_widgets (GladeXML *gxml, PrintWidgets *pw);
static float get_float_from_entry (GtkWidget *entry);
static int get_int_from_entry (GtkWidget *entry);
static void print_options_get (PrintWidgets *widgets);

void print_ok_button_cb (GtkWidget *button, PrintWidgets *widgets);
void print_preview_button_cb (GtkWidget *button, PrintWidgets *widgets);
void print_cancel_button_cb (GtkWidget *button, PrintWidgets *widgets);
void print_destroy_cb (GtkWidget *dialog, PrintWidgets *widgets);

void print_preview_page_first_cb (GtkWidget *button, PrintPreviewData *ppd);
void print_preview_page_last_cb (GtkWidget *button, PrintPreviewData *ppd);
void print_preview_page_spinner_cb (GtkWidget *spinner, PrintPreviewData *ppd);
void print_preview_close_cb (GtkWidget *button, PrintPreviewData *ppd);
void print_preview_destroy_cb (GtkWidget *dialog, PrintPreviewData *ppd);

/* returns a newly allocated PrintWidgets struct */
static PrintWidgets * print_widgets_new()
{
	PrintWidgets * pw;
	pw = g_new0 (PrintWidgets, 1);
	return pw;
}

void print_show_dialog (GaleonEmbed *embed)
{	
	GladeXML *gxml;
	PrintWidgets *pw;
	GaleonWindow *window;
	gboolean selected;
	GSList *rb_group;

	if (embed->print_dialog) return;

	pw = print_widgets_new();

	gxml = glade_widget_new ("print.glade", "print_dialog", &pw->print_dialog, pw);
	
	print_lookup_widgets (gxml, pw);

	gtk_window_set_modal (GTK_WINDOW(pw->print_dialog), TRUE);
	
	window = embed->parent_window;
	if (window)
	{
		dialog_set_parent (pw->print_dialog, window->wmain);
	}

	embed->print_dialog = pw->print_dialog;
	pw->embed = embed;
	print_options_get (pw);

	selected = mozilla_can_cut_selection (embed) || 
		   mozilla_can_copy_selection (embed);
	rb_group = gtk_radio_button_group (GTK_RADIO_BUTTON(pw->all_pages_rb));
	if (!selected && eel_gconf_get_integer (CONF_PRINT_ALL_PAGES) == 2) {
		eel_gconf_set_integer (CONF_PRINT_ALL_PAGES, 0);
		gtk_toggle_button_set_active 
			(GTK_TOGGLE_BUTTON (g_slist_nth (rb_group, 2)->data),
			 TRUE);
	}
	gtk_widget_set_sensitive (GTK_WIDGET (g_slist_nth (rb_group, 0)->data), 
				  selected);
	
	/* show the print dialog */
	gtk_widget_show (embed->print_dialog);

	gtk_object_unref (GTK_OBJECT (gxml));	
}

static void
print_lookup_widgets (GladeXML *gxml, PrintWidgets *pw)
{
	pw->print_entry = glade_xml_get_widget(gxml, "printer_entry");
	pw->file_entry = glade_xml_get_widget(gxml, "file_entry");
	pw->print_rb = glade_xml_get_widget(gxml, "printer_radiobutton");
	pw->letter_rb = glade_xml_get_widget(gxml, "letter_radiobutton");
	pw->top_entry = glade_xml_get_widget(gxml, "top_entry");
	pw->bottom_entry = glade_xml_get_widget(gxml, "bottom_entry");
	pw->left_entry = glade_xml_get_widget(gxml, "left_entry");
	pw->right_entry = glade_xml_get_widget(gxml, "right_entry");

	pw->print_notebook = glade_xml_get_widget(gxml, "print_notebook");

	/* header/footer stuff */
	pw->print_page_title_checkbutton = glade_xml_get_widget(gxml, "print_page_title_checkbutton");
	pw->print_page_url_checkbutton = glade_xml_get_widget(gxml, "print_page_url_checkbutton");
	pw->print_page_numbers_checkbutton = glade_xml_get_widget(gxml, "print_page_numbers_checkbutton");
	pw->print_date_checkbutton = glade_xml_get_widget(gxml, "print_date_checkbutton");

	pw->all_pages_rb = glade_xml_get_widget(gxml, "all_pages_radiobutton");
	pw->first_rb = glade_xml_get_widget(gxml, "first_radiobutton");
	pw->to_spinbutton = glade_xml_get_widget(gxml, "to_spinbutton");
	pw->from_spinbutton = glade_xml_get_widget(gxml, "from_spinbutton");
	pw->print_color_rb = glade_xml_get_widget(gxml, "print_color_radiobutton");
	pw->orient_p_rb = glade_xml_get_widget(gxml, "orient_p_radiobutton");
	pw->frame_no_rb = glade_xml_get_widget(gxml, "printframe_asis_radiobutton");
}

PrintInfo *
print_get_info (PrintWidgets *pw)
{
	PrintInfo *info;

	info = g_new0 (PrintInfo, 1);

      	info->print_to_file = 
		misc_gui_gtk_radio_button_get (GTK_RADIO_BUTTON (pw->print_rb));
	info->printer = gtk_editable_get_chars (
		GTK_EDITABLE (pw->print_entry), 0, -1);
	info->file = gtk_editable_get_chars (
		GTK_EDITABLE (pw->file_entry), 0, -1);

	info->top_margin = get_float_from_entry (pw->top_entry);
	info->bottom_margin = get_float_from_entry (pw->bottom_entry);
	info->left_margin = get_float_from_entry (pw->left_entry);
	info->right_margin = get_float_from_entry (pw->right_entry);

	info->from_page = get_int_from_entry (pw->from_spinbutton);
	info->to_page = get_int_from_entry (pw->to_spinbutton);

	info->paper = misc_gui_gtk_radio_button_get (
				GTK_RADIO_BUTTON (pw->letter_rb));
	info->pages = misc_gui_gtk_radio_button_get (
				GTK_RADIO_BUTTON (pw->all_pages_rb));
	info->reversed = misc_gui_gtk_radio_button_get (
				GTK_RADIO_BUTTON (pw->first_rb));
	info->print_color = misc_gui_gtk_radio_button_get (
				GTK_RADIO_BUTTON (pw->print_color_rb)) == 0;
	info->orientation = misc_gui_gtk_radio_button_get (
				GTK_RADIO_BUTTON (pw->orient_p_rb));
	info->frame_type = misc_gui_gtk_radio_button_get (
				GTK_RADIO_BUTTON (pw->frame_no_rb));

	/* header/footer stuff */
	/* this works basically by inserting 'special codes' into the
	 * "header strings" and "footer strings" that are eventually
	 * passed on to mozilla in mozilla.cpp/mozilla_print. 
	 * 
	 * Basically, the galeon UI presents an extremely simplified version
	 * of header/footer options that are available from within mozilla,
	 * because we are too lazy to figure out a better UI. You can
	 * choose whether or not to have the date, url, page number, and
	 * page title, but not where they will go on the page, nor whether
	 * they will be in the header or footer. see also ui/print.glade */
	/* the 'special codes' ("&T", etc) for headers & footers are
	 * documented in print.h (nov2001) */
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pw->print_page_title_checkbutton))) {
		info->header_left_string = g_strdup ("&T");
	} else {
		info->header_left_string = g_strdup ("");
	}
	info->header_center_string = g_strdup("");
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pw->print_page_url_checkbutton))) {
		info->header_right_string = g_strdup ("&U");
	} else {
		info->header_right_string = g_strdup ("");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pw->print_page_numbers_checkbutton))) {
		info->footer_left_string = g_strdup ("&PT");
	} else {
		info->footer_left_string = g_strdup ("");
	}
	info->footer_center_string = g_strdup("");
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pw->print_date_checkbutton))) {
		info->footer_right_string = g_strdup ("&D");
	} else {
		info->footer_right_string = g_strdup ("");
	}
	return info;
}

void
print_info_free (PrintInfo *info)
{
	g_free (info->printer);
	g_free (info->file);
	g_free (info->header_left_string);
	g_free (info->header_center_string);
	g_free (info->header_right_string);
	g_free (info->footer_left_string);
	g_free (info->footer_center_string);
	g_free (info->footer_right_string);

	g_free (info);
}

static
void print_options_get (PrintWidgets *widgets)
{
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (widgets->top_entry),
			    eel_gconf_get_float (CONF_PRINT_TOP_MARGIN));
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (widgets->bottom_entry),
			    eel_gconf_get_float (CONF_PRINT_BOTTOM_MARGIN));
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (widgets->left_entry),
			    eel_gconf_get_float (CONF_PRINT_LEFT_MARGIN));
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (widgets->right_entry),
			    eel_gconf_get_float (CONF_PRINT_RIGHT_MARGIN));

	/* headers/footers stuff */
	pu_set_togglebutton_from_config (widgets->print_page_title_checkbutton,
					 CONF_PRINT_PAGE_TITLE);
	pu_set_togglebutton_from_config (widgets->print_page_url_checkbutton,
					 CONF_PRINT_PAGE_URL);
	pu_set_togglebutton_from_config (widgets->print_date_checkbutton,
					 CONF_PRINT_DATE);
	pu_set_togglebutton_from_config (widgets->print_page_numbers_checkbutton,
					 CONF_PRINT_PAGE_NUMBERS);

	pu_set_editable_from_config (widgets->print_entry, 
				     CONF_PRINT_PRINTER);
	pu_set_editable_from_config (widgets->file_entry, 
				     CONF_PRINT_FILE);

	pu_set_radiobuttongroup_from_config (widgets->all_pages_rb,
					     CONF_PRINT_ALL_PAGES);
	pu_set_radiobuttongroup_from_config (widgets->first_rb,
					     CONF_PRINT_FIRST);
	pu_set_radiobuttongroup_from_config (widgets->print_rb, 
					     CONF_PRINT_PRINTON); 
	pu_set_radiobuttongroup_from_config (widgets->letter_rb, 
					     CONF_PRINT_PAPER); 
	pu_set_radiobuttongroup_from_config (widgets->print_color_rb,
					     CONF_PRINT_COLOR);
	pu_set_radiobuttongroup_from_config (widgets->orient_p_rb,
					     CONF_PRINT_ORIENTATION);
	pu_set_radiobuttongroup_from_config (widgets->frame_no_rb,
					     CONF_PRINT_FRAME_TYPE);
}

void print_options_set (PrintWidgets *widgets)
{
	eel_gconf_set_float (CONF_PRINT_TOP_MARGIN,
			     get_float_from_entry (widgets->top_entry));
	eel_gconf_set_float (CONF_PRINT_BOTTOM_MARGIN,
			     get_float_from_entry (widgets->bottom_entry));
	eel_gconf_set_float (CONF_PRINT_LEFT_MARGIN,
			     get_float_from_entry (widgets->left_entry));
	eel_gconf_set_float (CONF_PRINT_RIGHT_MARGIN,
			     get_float_from_entry (widgets->right_entry));

	pu_set_config_from_editable (widgets->print_entry, 
				     CONF_PRINT_PRINTER);
	pu_set_config_from_editable (widgets->file_entry, 
				     CONF_PRINT_FILE);

	/* headers/footers stuff */
	pu_set_config_from_togglebutton (widgets->print_page_title_checkbutton,
					 CONF_PRINT_PAGE_TITLE);
	pu_set_config_from_togglebutton (widgets->print_page_url_checkbutton,
					 CONF_PRINT_PAGE_URL);
	pu_set_config_from_togglebutton (widgets->print_date_checkbutton,
					 CONF_PRINT_DATE);
	pu_set_config_from_togglebutton (widgets->print_page_numbers_checkbutton,
					 CONF_PRINT_PAGE_NUMBERS);

	pu_set_config_from_radiobuttongroup (widgets->all_pages_rb,
					     CONF_PRINT_ALL_PAGES);
	pu_set_config_from_radiobuttongroup (widgets->first_rb,
					     CONF_PRINT_FIRST);
	pu_set_config_from_radiobuttongroup (widgets->print_rb, 
					     CONF_PRINT_PRINTON); 
	pu_set_config_from_radiobuttongroup (widgets->letter_rb, 
					     CONF_PRINT_PAPER);        
	pu_set_config_from_radiobuttongroup (widgets->print_color_rb, 
					     CONF_PRINT_COLOR);
	pu_set_config_from_radiobuttongroup (widgets->orient_p_rb,
					     CONF_PRINT_ORIENTATION);
	pu_set_config_from_radiobuttongroup (widgets->frame_no_rb,
					     CONF_PRINT_FRAME_TYPE);
}

static float
get_float_from_entry (GtkWidget *entry)
{
	double f;
	char *temp;

	temp = gtk_editable_get_chars (
		GTK_EDITABLE (entry), 0, -1);

	f = strtod(temp, (char**)NULL);

	g_free (temp);

	return f;
}

static int
get_int_from_entry (GtkWidget *entry)
{
	int i;
	char *temp;

	temp = gtk_editable_get_chars (
		GTK_EDITABLE (entry), 0, -1);
	i = strtol(temp, (char **)NULL, 10);
	g_free (temp);

	return i;
}

void
print_ok_button_cb (GtkWidget *button, PrintWidgets *widgets)
{
	PrintInfo *info;

	info = print_get_info (widgets);

	if (!mozilla_print (widgets->embed, info, FALSE))
	{
		gnome_error_dialog ("Not yet implemented.");
	}

	print_options_set (widgets);
	
	print_info_free (info);

	gtk_widget_destroy (widgets->print_dialog);
}

void
print_preview_button_cb (GtkWidget *button, PrintWidgets *widgets)
{
	PrintInfo *info;
//	GaleonEmbed *pp_embed;
//	DisplayType type = DISPLAY_NORMAL;

	info = print_get_info (widgets);
/*
	pp_embed = embed_create_after_embed (widgets->embed, TRUE,
					     widgets->embed->location,
					     EMBED_CREATE_FORCE_JUMP);
	mozilla_copy_page (pp_embed, widgets->embed, type);
	mozilla_print (pp_embed, info, TRUE);
*/
	mozilla_print (widgets->embed, info, TRUE);

	print_info_free (info);

	gtk_widget_destroy (widgets->print_dialog);
}

void
print_cancel_button_cb (GtkWidget *button, PrintWidgets *widgets)
{
	gtk_widget_destroy (widgets->print_dialog);
}

void
print_destroy_cb (GtkWidget *dialog, PrintWidgets *widgets)
{
	widgets->embed->print_dialog = NULL;
	g_free (widgets);
}

void print_show_preview_controls (GaleonEmbed *embed)
{	
	GladeXML *gxml;
	PrintPreviewData *ppd;
	GaleonWindow *window;
	gchar *numPagesStr = 0;
	GtkAdjustment *adj;

	ppd = g_new0(PrintPreviewData, 1);

	gxml = glade_widget_new ("print-preview.glade", "print_preview_panel",
				 &ppd->dialog, ppd);
	
	window = embed->parent_window;
	if (window)
	{
		dialog_set_parent (ppd->dialog, window->wmain);

		ppd->menuBarState = GTK_CHECK_MENU_ITEM(window->view_menubar)->active;
		ppd->toolBarState = GTK_CHECK_MENU_ITEM(window->view_toolbar)->active;
		ppd->bookmarkBarState = GTK_CHECK_MENU_ITEM(window->view_bookmarks)->active;
		ppd->statusBarState = GTK_CHECK_MENU_ITEM(window->view_statusbar)->active;

		window_menubar_hide(window);
		window_toolbar_hide(window);
		window_statusbar_hide(window);
		bookmarks_toolbars_set_visibility(window, FALSE);

		if(window->notebook)
		{
			ppd->notebookState = GTK_NOTEBOOK(window->notebook)->show_tabs;
			gtk_notebook_set_show_tabs(GTK_NOTEBOOK(window->notebook), FALSE);
		}

		if(window->docked)
		{
			ppd->dockState = GTK_WIDGET_VISIBLE(window->docked);
			gtk_widget_hide(window->hpaned);
			gtk_widget_hide(window->notebook);
			gtk_widget_reparent(window->notebook, window->hbox);
			gtk_widget_show(window->notebook);
		}

	}

	ppd->numPages = mozilla_print_preview_num_pages (embed);

	ppd->embed = embed;

	ppd->pageSpinner = glade_xml_get_widget(gxml,
						"print_preview_page_spinner");

	ppd->pageTotal = glade_xml_get_widget(gxml,
					      "print_preview_page_total");

	adj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(ppd->pageSpinner));
	adj->upper = ppd->numPages;
	gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON(ppd->pageSpinner), adj);

	numPagesStr = g_strdup_printf("%d", ppd->numPages);
	gtk_label_set_text (GTK_LABEL(ppd->pageTotal), numPagesStr);
	g_free(numPagesStr);

	/* show the print dialog */
	gtk_widget_show (ppd->dialog);

	gtk_object_unref (GTK_OBJECT (gxml));	
}

void print_preview_page_first_cb (GtkWidget *button, PrintPreviewData *ppd)
{
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(ppd->pageSpinner), 1);
}

void print_preview_page_last_cb (GtkWidget *button, PrintPreviewData *ppd)
{
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(ppd->pageSpinner),
				  ppd->numPages);
}

void print_preview_page_spinner_cb (GtkWidget *spinner, PrintPreviewData *ppd)
{
	printPreviewNavType navType = PRINTPREVIEW_GOTO_PAGENUM;
	mozilla_print_preview_navigate(ppd->embed, navType,
			gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spinner)));
}

void print_preview_close_cb (GtkWidget *button, PrintPreviewData *ppd)
{
	gtk_widget_destroy (ppd->dialog);
}

void print_preview_destroy_cb (GtkWidget *dialog, PrintPreviewData *ppd)
{
	GaleonWindow *window;

	mozilla_print_preview_close(ppd->embed);

	window = ppd->embed->parent_window;
	if(window)
	{
		if(ppd->menuBarState) window_menubar_show(window);
		if(ppd->toolBarState) window_toolbar_show(window);
		if(ppd->statusBarState) window_statusbar_show(window);
		bookmarks_toolbars_set_visibility(window, ppd->bookmarkBarState);
		if(ppd->notebookState) gtk_notebook_set_show_tabs(GTK_NOTEBOOK(window->notebook), TRUE);
		if(ppd->dockState)
		{
			gtk_widget_hide(window->notebook);
			gtk_widget_reparent(window->notebook, window->hpaned);
			gtk_widget_show(window->notebook);
			gtk_widget_show(window->hpaned);
		}
	}

	g_free (ppd);
}

