#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <tiffio.h>

#include "loader.h"
#ifdef USE_X11
# include "viewer.h"
#endif

struct tiff_state {
    TIFF*          tif;
    char           emsg[1024];
    uint32         width,height;
    uint16         config,nsamples;
    uint32*        row;
    uint16         resunit;
    float          xres,yres;
};

static void*
tiff_init(FILE *fp, char *filename, struct ida_image_info *i)
{
    struct tiff_state *h;

    fclose(fp);
    h = malloc(sizeof(*h));
    memset(h,0,sizeof(*h));

    h->tif = TIFFOpen(filename,"r");
    if (NULL == h->tif)
	goto oops;

    TIFFGetField(h->tif, TIFFTAG_IMAGEWIDTH,      &h->width);
    TIFFGetField(h->tif, TIFFTAG_IMAGELENGTH,     &h->height);
    TIFFGetField(h->tif, TIFFTAG_PLANARCONFIG,    &h->config);
    TIFFGetField(h->tif, TIFFTAG_SAMPLESPERPIXEL, &h->nsamples);
    h->row = malloc(TIFFScanlineSize(h->tif));
    if (debug)
	fprintf(stderr,"tiff: %ldx%ld, planar=%d, nsamples=%d, scanline=%ld\n",
		h->width,h->height,h->config,h->nsamples,
		TIFFScanlineSize(h->tif));

    i->width  = h->width;
    i->height = h->height;

    if (TIFFGetField(h->tif, TIFFTAG_RESOLUTIONUNIT,  &h->resunit) &&
	TIFFGetField(h->tif, TIFFTAG_XRESOLUTION,     &h->xres)    &&
	TIFFGetField(h->tif, TIFFTAG_YRESOLUTION,     &h->yres)) {
	switch (h->resunit) {
	case RESUNIT_NONE:
	    break;
	case RESUNIT_INCH:
	    i->dpi = h->xres;
	    break;
	case RESUNIT_CENTIMETER:
	    i->dpi = res_cm_to_inch(h->xres);
	    break;
	}
    }

    return h;

 oops:
    if (h->tif)
	TIFFClose(h->tif);
    free(h);
    return NULL;
}

static void
tiff_read(unsigned char *dst, int line, void *data)
{
    struct tiff_state *h = data;
    int s;

    if (h->config == PLANARCONFIG_CONTIG) {
	TIFFReadScanline(h->tif, h->row, line, 0);
    } else if (h->config == PLANARCONFIG_SEPARATE) {
	for (s = 0; s < h->nsamples; s++)
	    TIFFReadScanline(h->tif, h->row, line, s);
    }

    switch (h->nsamples) {
    case 1:
	/* gray */
	load_gray(dst,(unsigned char*)(h->row),h->width);
	break;
    case 3:
	/* rgb */
	memcpy(dst,h->row,3*h->width);
	break;
    case 4:
	/* rgb+alpha */
	load_rgba(dst,(unsigned char*)(h->row),h->width);
	break;
    }
}

static void
tiff_done(void *data)
{
    struct tiff_state *h = data;

    TIFFClose(h->tif);
    free(h->row);
    free(h);
}

struct ida_loader tiff1_loader = {
    magic: "MM\x00\x2a",
    moff:  0,
    mlen:  4,
    name:  "libtiff",
    init:  tiff_init,
    read:  tiff_read,
    done:  tiff_done,
};
struct ida_loader tiff2_loader = {
    magic: "II\x2a\x00",
    moff:  0,
    mlen:  4,
    name:  "libtiff",
    init:  tiff_init,
    read:  tiff_read,
    done:  tiff_done,
};

#ifdef USE_X11
/* ---------------------------------------------------------------------- */
/* save                                                                   */

static int
tiff_write(FILE *fp, struct ida_image *img)
{
    TIFF     *TiffHndl;
    tdata_t  buf;
    int      y;

    TiffHndl = TIFFFdOpen(fileno(fp),NULL,"w");
    if (TiffHndl == NULL)
	return -1;
    TIFFSetField(TiffHndl, TIFFTAG_IMAGEWIDTH, img->i.width);
    TIFFSetField(TiffHndl, TIFFTAG_IMAGELENGTH, img->i.height);
    TIFFSetField(TiffHndl, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
    TIFFSetField(TiffHndl, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB);
    TIFFSetField(TiffHndl, TIFFTAG_BITSPERSAMPLE, 8);
    TIFFSetField(TiffHndl, TIFFTAG_SAMPLESPERPIXEL, 3);
    TIFFSetField(TiffHndl, TIFFTAG_ROWSPERSTRIP, 2);
    TIFFSetField(TiffHndl, TIFFTAG_ORIENTATION, ORIENTATION_TOPLEFT);
#if 0 /* fixme: make this configureable */
    TIFFSetField(TiffHndl, TIFFTAG_COMPRESSION, COMPRESSION_LZW);
    TIFFSetField(TiffHndl, TIFFTAG_PREDICTOR, 2);
#endif
    if (img->i.dpi) {
	float dpi = img->i.dpi;
	TIFFSetField(TiffHndl, TIFFTAG_RESOLUTIONUNIT, RESUNIT_INCH);
	TIFFSetField(TiffHndl, TIFFTAG_XRESOLUTION,    dpi);
	TIFFSetField(TiffHndl, TIFFTAG_YRESOLUTION,    dpi);
    }

    for (y = 0; y < img->i.height; y++) {
	buf = img->data + 3*img->i.width*y;
	TIFFWriteScanline(TiffHndl, buf, y, 0);
    }
    TIFFClose(TiffHndl);
    return 0;
}

struct ida_writer tiff_writer = {
    label:  "TIFF",
    ext:    { "tif", "tiff", NULL},
    write:  tiff_write,
};
#endif
