import unittest
import shutil
import os
import re
import time

import eoc

DOTDIR = "dot-dir-for-testing"
eoc.DOTDIR = DOTDIR
eoc.quiet = 1

def no_op(*args):
    pass

class AddressParsingTestCases(unittest.TestCase):

    def setUp(self):
    	self.lists = ["foo@example.com", 
	    	      "foo-announce@example.com"]
	self.mlm = eoc.MailingListManager(DOTDIR, lists=self.lists)

    def testEmpty(self):
    	self.failUnlessRaises(eoc.BadCommandAddress,
	    	    	      self.mlm.parse_command_address, "", "")

    def try_address(self, address, should_be):
	local_part, domain = address.split("@")
    	self.failUnlessEqual(self.mlm.parse_command_address(local_part, 
							    domain), 
			     should_be)

    def testPost(self):
	self.try_address("foo@example.com",
			     {
				"name": "foo@example.com",
				"command": "post",
			     })
	self.try_address("foo-announce@example.com",
			     {
				"name": "foo-announce@example.com",
				"command": "post",
			     })

    def testSimpleCommands(self):
    	for command in ["help", "list", "owner"]:
	    self.try_address("foo-%s@example.com" % command,
				 {
				    "name": "foo@example.com",
				    "command": command,
				 })
	    self.try_address("foo-announce-%s@example.com" % command,
				 {
				    "name": "foo-announce@example.com",
				    "command": command,
				 })

    def testSubscribe(self):
    	for command in ["subscribe", "unsubscribe"]:
	    self.try_address("foo-%s@example.com" % command,
				 {
				    "name": "foo@example.com",
				    "command": command,
				    "sender": "",
				 })
	    self.try_address("foo-announce-%s@example.com" % command,
				 {
				    "name": "foo-announce@example.com",
				    "command": command,
				    "sender": "",
				 })

    def testSubscribeWithAddressAddress(self):
    	for command in ["subscribe", "unsubscribe"]:
	    self.try_address("foo-%s-foo=foo-bar.com@example.com" % 
	    	    	    	    command,
				 {
				    "name": "foo@example.com",
				    "command": command,
				    "sender": "foo@foo-bar.com",
				 })
	    self.try_address("foo-announce-%s-foo=foo-bar.com@example.com" % 
	    	    	    	    command,
				 {
				    "name": "foo-announce@example.com",
				    "command": command,
				    "sender": "foo@foo-bar.com",
				 })

    def try_bad_signature(self, command_part):
    	self.failUnlessRaises(eoc.BadSignature,
	    	    	      self.mlm.parse_command_address, 
			      "foo-" + command_part + "-123-badhash",
			      "example.com")
    	self.failUnlessRaises(eoc.BadSignature,
	    	    	      self.mlm.parse_command_address, 
			      "foo-announce-" + command_part + "-123-badhash",
			      "example.com")

    def testBadlySignedCommands(self):
    	self.try_bad_signature("subyes")
    	self.try_bad_signature("subapprove")
    	self.try_bad_signature("subreject")
    	self.try_bad_signature("unsubyes")
    	self.try_bad_signature("bounce")
    	self.try_bad_signature("approve")
    	self.try_bad_signature("reject")
    	self.try_bad_signature("probe")

    def try_good_signature(self, command):
    	s = "foo-announce-%s-1" % command
	hash = self.mlm.compute_hash("%s@%s" % (s, "example.com"))
	local_part = "%s-%s" % (s, hash)
    	self.failUnlessEqual(self.mlm.parse_command_address(local_part,
							    "example.com"),
			     {
				"name": "foo-announce@example.com",
				"command": command,
				"id": "1",
			     })

    def testProperlySignedCommands(self):
    	self.try_good_signature("subyes")
    	self.try_good_signature("subapprove")
    	self.try_good_signature("subreject")
    	self.try_good_signature("unsubyes")
    	self.try_good_signature("bounce")
    	self.try_good_signature("approve")
    	self.try_good_signature("reject")
    	self.try_good_signature("probe")


class ListTestCases(unittest.TestCase):

    def setUp(self):
    	if os.path.exists(DOTDIR):
	    shutil.rmtree(DOTDIR)
    	self.mlm = eoc.MailingListManager(DOTDIR)

    def tearDown(self):
    	self.mlm = None
	shutil.rmtree(DOTDIR)

    def create_list(self, options=None):
    	if not options:
	    options = {
		"owners": "listmaster@example.com",
		"subscription": "free",
		"posting": "free",
		"archived": "no",
	    }
    	dirname = os.path.join(DOTDIR, "foo@example.com")
	os.mkdir(dirname)
	f = open(os.path.join(dirname, "config"), "w")
	f.write("[list]\n")
	for key in options.keys():
	    f.write("%s = %s\n" % (key, options[key]))
	f.close()
	f = open(os.path.join(dirname, "subscribers"), "w")
	f.write("0 ok 12345 0 . user1@example.com user2@example.com\n")
	f.close()
	self.mlm = eoc.MailingListManager(DOTDIR)

    def testDotDirExistsAndIsEmpty(self):
    	self.failUnless(os.path.isdir(DOTDIR))
	self.failUnlessEqual(os.listdir(DOTDIR), ["secret"])

    def testCreateExisting(self):
    	list = self.mlm.create_list("foo@example.com")
    	self.failUnlessRaises(eoc.ListExists,
	    	    	      self.mlm.create_list, "foo@example.com")

    def check_options(self, list, correct):
    	self.failUnlessEqual(list.cp.sections(), ["list"])
	list_keys = list.cp.options("list")
	list_keys.sort()
	correct_keys = correct.keys()
	correct_keys.sort()
	self.failUnlessEqual(list_keys, correct_keys)

    def testCreateList(self):
    	list = self.mlm.create_list("foo@example.com")
	self.failUnlessEqual(list.__class__, eoc.MailingList)
    	self.failUnlessEqual(list.dirname, os.path.join(DOTDIR, 
							"foo@example.com"))
    	self.failUnless(os.path.isdir(list.dirname))
	for filename in ["config", "subscribers"]:
	    full_name = os.path.join(list.dirname, filename)
	    self.failUnless(os.path.isfile(full_name))
	for dirname in ["bounce-box"]:
	    full_name = os.path.join(list.dirname, dirname)
	    self.failUnless(os.path.isdir(full_name))
    	self.failIfEqual(list.cp, None)
	self.check_options(list, {
	    "owners": "",
	    "subscription": "free",
	    "posting": "free",
	    "archived": "no",
	    "mail-on-subscription-changes": "no",
	    "mail-on-forced-unsubscribe": "no",
	    "language": "",
	})

    def testOpenExisting(self):
	options = {
	    "owners": "listmaster@example.com",
	    "subscription": "moderated",
	    "posting": "moderated",
	    "archived": "yes",
	    "mail-on-subscription-changes": "no",
	    "mail-on-forced-unsubscribe": "no",
	    "language": "",
	}
    	self.create_list(options)
    	list = self.mlm.open_list("foo@example.com")
	self.check_options(list, options)

    def testChangeListOptions(self):
    	list = self.mlm.create_list("foo@example.com")
	self.failUnlessEqual(list.cp.get("list", "posting"), "free")
	list.cp.set("list", "posting", "moderated")
	list.save_config()
	
	list = self.mlm.open_list("foo@example.com")
	self.failUnlessEqual(list.cp.get("list", "posting"), "moderated")

    def testAddSubscriber(self):
    	list = self.mlm.create_list("foo@example.com")
	self.failUnlessEqual(list.subscribers.get_all(), [])
	
    	self.failUnless(list.subscribers.lock())
	list.subscribers.add("user@example.com")
	self.failUnlessEqual(list.subscribers.get_all(), ["user@example.com"])

    	list.subscribers.save()
	self.failIf(list.subscribers.locked)
    	list = self.mlm.open_list("foo@example.com")
	self.failUnlessEqual(list.subscribers.get_all(), ["user@example.com"])

    def testSubscribers(self):
    	self.create_list()
    	list = self.mlm.open_list("foo@example.com")
	self.failUnlessEqual(list.subscribers.groups(), ["0"])
	self.failUnlessEqual(list.subscribers.in_group("0"),
	    	    	     ["user1@example.com", "user2@example.com"])

    def testUnsubscribe(self):
    	self.create_list()
	list = self.mlm.open_list("foo@example.com")
	self.failUnless(list.subscribers.lock())
	list.subscribers.remove("user1@example.com")
	self.failUnlessEqual(list.subscribers.get_all(), ["user2@example.com"])

    def testModerationBox(self):
    	list = self.mlm.create_list("foo@example.com")
	listdir = os.path.join(DOTDIR, "foo@example.com")
	boxdir = os.path.join(listdir, "moderation-box")

    	self.failUnlessEqual(boxdir, list.moderation_box.boxdir)
	self.failUnless(os.path.isdir(boxdir))

    	id = list.moderation_box.add("foo", "From: foo\nTo: bar\n\nhello\n")
	filename = os.path.join(boxdir, id)
	self.failUnless(os.path.isfile(filename))
	self.failUnless(os.path.isfile(filename + ".address"))
	
	list.moderation_box.remove(id)
	self.failUnless(not os.path.exists(filename))


class IncomingMessageTestCases(unittest.TestCase):

    def setUp(self):
    	if os.path.isdir(DOTDIR):
	    shutil.rmtree(DOTDIR)
    	self.mlm = eoc.MailingListManager(DOTDIR)
	list = self.mlm.create_list("foo@example.com")
	list.cp.set("list", "owners", "listmaster@example.com")
	list.save_config()
	list.subscribers.lock()
	list.subscribers.add("user1@example.com")
	list.subscribers.add("user2@example.com")
	list.subscribers.save()
	self.sent_mail = []

    def tearDown(self):
    	shutil.rmtree(DOTDIR)
	self.sent_mail = []

    def environ(self, sender, recipient):
	eoc.set_environ({
	    "SENDER": sender,
	    "RECIPIENT": recipient,
	})

    def catch_sendmail(self, sender, recipients, text):
    	self.sent_mail.append({
	    "sender": sender,
	    "recipients": recipients,
	    "text": text,
	})

    def obey(self, sender, recipient, text, force_moderation=0, force_posting=0):
    	self.environ(sender, recipient)
	dict = self.mlm.parse_recipient_address(None, None)
	dict["force-moderation"] = force_moderation
	dict["force-posting"] = force_posting
	self.list = self.mlm.open_list(dict["name"])
	self.list.read_stdin = lambda t=text: t
	self.list.send_mail = self.catch_sendmail
	self.sent_mail = []
	self.list.obey(dict)

    def send(self, sender, recipient, text=""):
    	self.obey(sender, recipient, text)
	
    def sender_matches(self, mail, sender):
	pat = "(?P<address>" + sender + ")"
	m = re.match(pat, mail["sender"])
	if m:
	    return m.group("address")
	else:
	    return None
	
    def replyto_matches(self, mail, replyto):
	pat = "(.|\n)*(?P<address>" + replyto + ")"
	m = re.match(pat, mail["text"])
	if m:
	    return m.group("address")
	else:
	    return None

    def receiver_matches(self, mail, recipient):
	return mail["recipients"] == [recipient]

    def body_matches(self, mail, body):
    	if body:
	    pat = re.compile("(.|\n)*" + body + "(.|\n)*")
	    m = re.match(pat, mail["text"])
	    return m
	else:
	    return 1

    def match(self, sender, replyto, receiver, body=None):
    	ret = None
    	for mail in self.sent_mail:
	    if replyto is None:
		m1 = self.sender_matches(mail, sender)
		m3 = self.receiver_matches(mail, receiver)
		m4 = self.body_matches(mail, body)
		if m1 != None and m3 and m4:
		    ret = m1
		    self.sent_mail.remove(mail)
		    break
    	    else:
		m1 = self.sender_matches(mail, sender)
		m2 = self.replyto_matches(mail, replyto)
		m3 = self.receiver_matches(mail, receiver)
		m4 = self.body_matches(mail, body)
		if m1 != None and m2 != None and m3 and m4:
		    ret = m2
		    self.sent_mail.remove(mail)
		    break
    	self.failUnless(ret != None)
	return ret

    def no_more_mail(self):
    	self.failUnlessEqual(self.sent_mail, [])

    def testDeduceList(self):
    	self.environ("user@example.com", "foo-help@example.com")
	dict = self.mlm.parse_recipient_address(None, None)
    	self.failUnlessEqual(dict["name"], "foo@example.com")

    def testHelp(self):
    	self.send("outsider@example.com", "foo-help@example.com")
	self.match("foo-ignore@example.com", None, "outsider@example.com", 
	    	   "Subject: Help for")
	self.no_more_mail()

    def testList(self):
    	self.send("listmaster@example.com", "foo-list@example.com")
	self.match("foo-ignore@example.com", None, "listmaster@example.com",
		   "user[12]@example.com\nuser[12]@example.com")
	self.no_more_mail()

    def testListDenied(self):
    	self.send("outsider@example.com", "foo-list@example.com")
	self.match("foo-ignore@example.com", None, "outsider@example.com", 
	    	   "Subject: Subscriber list denied")
	self.no_more_mail()

    def testSetlist(self):
    	self.send("listmaster@example.com", "foo-setlist@example.com",
	    	  "From: foo\n\nnew1@example.com\nuser1@example.com\n")
    	a = self.match("foo-ignore@example.com", 
	    	       "foo-setlistyes-[^@]*@example.com", 
		       "listmaster@example.com", 
		       "Subject: Please moderate subscriber list")
    	self.no_more_mail()
	
	self.send("listmaster@example.com", a)
	self.match("foo-ignore@example.com", None, "listmaster@example.com",
	    	   "Subject: Subscriber list has been changed")
	self.match("foo-ignore@example.com", None, "new1@example.com",
	    	   "Subject: Welcome to")
	self.match("foo-ignore@example.com", None, "user2@example.com",
	    	   "Subject: Goodbye from")
    	self.no_more_mail()

    def testSetlistSilently(self):
    	self.send("listmaster@example.com", "foo-setlistsilently@example.com",
	    	  "From: foo\n\nnew1@example.com\nuser1@example.com\n")
    	a = self.match("foo-ignore@example.com", 
	    	       "foo-setlistsilentyes-[^@]*@example.com", 
		       "listmaster@example.com", 
		       "Subject: Please moderate subscriber list")
    	self.no_more_mail()
	
	self.send("listmaster@example.com", a)
	self.match("foo-ignore@example.com", None, "listmaster@example.com",
	    	   "Subject: Subscriber list has been changed")
    	self.no_more_mail()

    def testSetlistDenied(self):
    	self.send("outsider@example.com", "foo-setlist@example.com",
	    	  "From: foo\n\nnew1@example.com\nnew2@example.com\n")
    	self.match("foo-ignore@example.com", 
	    	   None,
		   "outsider@example.com", 
		   "Subject: You can't set the subscriber list")
    	self.no_more_mail()

    def testSetlistBadlist(self):
    	self.send("listmaster@example.com", "foo-setlist@example.com",
	    	  "From: foo\n\nBlah blah blah.\n")
    	self.match("foo-ignore@example.com", 
	    	   None,
		   "listmaster@example.com", 
		   "Subject: Bad address list")
    	self.no_more_mail()

    def testOwner(self):
	self.send("outsider@example.com", "foo-owner@example.com", "abcde")
	self.match("outsider@example.com", None, "listmaster@example.com",
		   "abcde")
	self.no_more_mail()

    def configure_list(self, subscription, posting):
    	list = self.mlm.open_list("foo@example.com")
    	list.cp.set("list", "subscription", subscription)
    	list.cp.set("list", "posting", posting)
	list.save_config()

    def subscribe_common(self, user_address, request_address):
    	self.configure_list("free", "free")
	
	# Send subscription request. List sends confirmation request.
	self.send(user_address, request_address)
	a = self.match("foo-ignore@example.com", 
	    	       "foo-subyes-[^@]*@example.com", 
		       "outsider@example.com",
		       "Please confirm subscription")
	self.no_more_mail()
	
	# Confirm sub. req. List sends welcome.
	self.send(user_address, a)
	self.match("foo-ignore@example.com", 
	    	   None, 
		   "outsider@example.com", 
		   "Welcome to the")
	self.no_more_mail()

    def testSubscribe(self):
    	self.subscribe_common("outsider@example.com",
	    	    	      "foo-subscribe@example.com")

    def testSubscribeWithAddress(self):
    	self.subscribe_common("somebody.else@example.com",
	    	  "foo-subscribe-outsider=example.com@example.com")

    def testOwnerSubscribesSomeoneElse(self):
    	self.configure_list("free", "free")
	
	# Send subscription request. List sends confirmation request.
	self.send("listmaster@example.com",
	    	  "foo-subscribe-outsider=example.com@example.com")
	a = self.match("foo-ignore@example.com", 
	    	       "foo-subyes-[^@]*@example.com", 
		       "listmaster@example.com",
		       "Please confirm subscription")
	self.no_more_mail()
	
	# Confirm sub. req. List sends welcome.
	self.send("listmaster@example.com", a)
	self.match("foo-ignore@example.com", 
	    	   None, 
		   "outsider@example.com", 
		   "Welcome to the")
	self.no_more_mail()

    def testOwnerUnubscribesSomeoneElse(self):
    	self.configure_list("free", "free")
	
	# Send unsubscription request. List sends confirmation request.
	self.send("listmaster@example.com",
	    	  "foo-unsubscribe-outsider=example.com@example.com")
	a = self.match("foo-ignore@example.com", 
	    	       "foo-unsubyes-[^@]*@example.com", 
		       "listmaster@example.com",
		       "Subject: Please confirm UNsubscription")
	self.no_more_mail()
	
	# Confirm sub. req. List sends welcome.
	self.send("listmaster@example.com", a)
	self.match("foo-ignore@example.com", None, "outsider@example.com", 
	    	   "Goodbye")
	self.no_more_mail()

    def subscribe_to_moderated_common(self, owner_reply_pattern):
    	self.configure_list("moderated", "free")
	
    	# Send subscription request, list sends confirmation req.
	self.send("outsider@example.com", "foo-subscribe@example.com")
	a = self.match("foo-ignore@example.com", 
	    	       "foo-subyes-[^@]*@example.com", 
	    	       "outsider@example.com")
	self.no_more_mail()
	
	# Reply to confirmation req., list sends notice to us, req. to owner
	self.send("outsider@example.com", a)
	self.match("foo-ignore@example.com", None, "outsider@example.com")
	a = self.match("foo-ignore@example.com", owner_reply_pattern,
	    	       "listmaster@example.com")
	self.no_more_mail()
	
	# Owner confirms or rejects, list notifies subscriber
	self.send("listmaster@example.com", a)
	self.match("foo-ignore@example.com", None, "outsider@example.com")
	self.no_more_mail()

    def testSubscribeToModerated(self):
    	self.subscribe_to_moderated_common("foo-subapprove-[^@]*@example.com")

    def testSubscribeToModeratedDenied(self):
    	self.subscribe_to_moderated_common("foo-subreject-[^@]*@example.com")

    def unsubscribe_common(self, user_address, request_address):
    	# Send unsubscription request. List sends confirmation request.
    	self.send(user_address, request_address)
	a = self.match("foo-ignore@example.com", 
	    	       "foo-unsubyes-[^@]*@example.com", 
	    	       "outsider@example.com",
		       "Subject: Please confirm UNsubscription")
    	self.no_more_mail()

    	# Confirm. List sends goodbye.
    	self.send(user_address, a)
	self.match("foo-ignore@example.com", None, "outsider@example.com",
	    	   "Subject: Goodbye from")

    def testUnsubscribe(self):
    	self.unsubscribe_common("outsider@example.com",
	    	    	    	"foo-unsubscribe@example.com")

    def testUnsubscribeWithAddress(self):
    	self.unsubscribe_common("somebody.else@example.com",
		     "foo-unsubscribe-outsider=example.com@example.com")

    def check_mail_to_list(self):
	self.match("foo-bounce-.*@example.com", None, "user1@example.com",
	    	   "hello, world")
	self.match("foo-bounce-.*@example.com", None, "user2@example.com",
	    	   "hello, world")
    	self.no_more_mail()

    def testPostToUnmoderated(self):
	self.send("user1@example.com", "foo@example.com", "hello, world")
	self.check_mail_to_list()

    def testPostWithRequestToBeModerated(self):
    	self.configure_list("free", "free")

    	self.obey("user1@example.com", "foo@example.com", "hello, world",
	    	  force_moderation=1)
    	self.match("foo-ignore@example.com", 
	    	   None, 
		   "user1@example.com", 
		   "Subject: Please wait")
    	a = self.match("foo-ignore@example.com", 
	    	       "foo-approve-[^@]*@example.com", 
		       "listmaster@example.com")
    	self.no_more_mail()

	self.send("listmaster@example.com", a)
	self.check_mail_to_list()

    def testPostWithModerationOverride(self):
    	self.configure_list("moderated", "moderated")
    	self.obey("user1@example.com", "foo@example.com", "hello, world",
	    	  force_posting=1)
	self.check_mail_to_list()

    def testPostToModeratedApproved(self):
    	self.configure_list("free", "moderated")

    	self.obey("user1@example.com", "foo@example.com", "hello, world")
    	self.match("foo-ignore@example.com", None, "user1@example.com", 
	    	   "Subject: Please wait")
    	a = self.match("foo-ignore@example.com", 
	    	       "foo-approve-[^@]*@example.com", 
		       "listmaster@example.com")
    	self.no_more_mail()

	self.send("listmaster@example.com", a)
	self.check_mail_to_list()

    def has_message_in_moderation_box(self):
    	list = self.mlm.open_list("foo@example.com")
	return os.listdir(list.moderation_box.boxdir)

    def testPostToModeratedRejected(self):
    	self.configure_list("free", "moderated")

    	self.obey("user1@example.com", "foo@example.com", "hello, world")
    	self.match("foo-ignore@example.com", None, "user1@example.com", 
	    	   "Subject: Please wait")
    	a = self.match("foo-ignore@example.com", 
	    	       "foo-reject-[^@]*@example.com", 
		       "listmaster@example.com")
    	self.no_more_mail()
	self.failUnless(self.has_message_in_moderation_box())

	self.send("listmaster@example.com", a)
	self.no_more_mail()
	self.failUnless(not self.has_message_in_moderation_box())

    def check_subscriber_status(self, must_be):
    	list = self.mlm.open_list("foo@example.com")
	for id in list.subscribers.groups():
	    self.failUnlessEqual(list.subscribers.get(id, "status"), must_be)

    def bounce_sent_mail(self):
	for m in self.sent_mail[:]:
	    self.send("something@example.com", m["sender"], "eek")
	    self.failUnlessEqual(len(self.sent_mail), 0)

    def send_mail_to_list_then_bounce_everything(self):
	self.obey("user@example.com", "foo@example.com", "hello")
	for m in self.sent_mail[:]:
	    self.obey("foo@example.com", m["sender"], "eek")
	    self.failUnlessEqual(len(self.sent_mail), 0)

    def testBounceOnceThenRecover(self):
    	self.check_subscriber_status("ok")
	self.send_mail_to_list_then_bounce_everything()

    	self.check_subscriber_status("bounced")
	
	list = self.mlm.open_list("foo@example.com")
	for id in list.subscribers.groups():
	    bounce_id = list.subscribers.get(id, "bounce-id")
    	    self.failUnless(bounce_id)
    	    self.failUnless(list.bounce_box.has(bounce_id))

    	bounce_ids = []
	now = time.time()
	list = self.mlm.open_list("foo@example.com")
	list.subscribers.lock()
	for id in list.subscribers.groups():
	    timestamp = float(list.subscribers.get(id, "timestamp-bounced"))
	    self.failUnless(abs(timestamp - now) < 10.0)
	    list.subscribers.set(id, "timestamp-bounced", "69.0")
	    bounce_ids.append(list.subscribers.get(id, "bounce-id"))
	list.subscribers.save()

	self.mlm.cleaning_woman(no_op)
    	self.check_subscriber_status("probed")

    	for bounce_id in bounce_ids:
	    self.failUnless(list.bounce_box.has(bounce_id))

	self.mlm.cleaning_woman(no_op)
    	list = self.mlm.open_list("foo@example.com")
	self.failUnlessEqual(len(list.subscribers.groups()), 2)
    	self.check_subscriber_status("ok")
    	for bounce_id in bounce_ids:
	    self.failUnless(not list.bounce_box.has(bounce_id))

    def testBounceProbeAlso(self):
    	self.check_subscriber_status("ok")
	self.send_mail_to_list_then_bounce_everything()
    	self.check_subscriber_status("bounced")
	
	list = self.mlm.open_list("foo@example.com")
	for id in list.subscribers.groups():
	    bounce_id = list.subscribers.get(id, "bounce-id")
    	    self.failUnless(bounce_id)
    	    self.failUnless(list.bounce_box.has(bounce_id))

    	bounce_ids = []
	now = time.time()
	list = self.mlm.open_list("foo@example.com")
	list.subscribers.lock()
	for id in list.subscribers.groups():
	    timestamp = float(list.subscribers.get(id, "timestamp-bounced"))
	    self.failUnless(abs(timestamp - now) < 10.0)
	    list.subscribers.set(id, "timestamp-bounced", "69.0")
	    bounce_ids.append(list.subscribers.get(id, "bounce-id"))
	list.subscribers.save()

    	self.sent_mail = []
	self.mlm.cleaning_woman(self.catch_sendmail)
    	self.check_subscriber_status("probed")
    	for bounce_id in bounce_ids:
	    self.failUnless(list.bounce_box.has(bounce_id))
    	self.bounce_sent_mail()
    	self.check_subscriber_status("probebounced")

	self.mlm.cleaning_woman(no_op)
    	list = self.mlm.open_list("foo@example.com")
	self.failUnlessEqual(len(list.subscribers.groups()), 0)
    	for bounce_id in bounce_ids:
	    self.failUnless(not list.bounce_box.has(bounce_id))

    def testCleaningWomanJoinsAndBounceSplitsGroups(self):
    	# Check that each group contains one address and set the creation
	# timestamp to an ancient time.
    	list = self.mlm.open_list("foo@example.com")
	bouncedir = os.path.join(list.dirname, "bounce-box")
	list.subscribers.lock()
	for id in list.subscribers.groups():
	    addrs = list.subscribers.in_group(id)
	    self.failUnlessEqual(len(addrs), 1)
	    bounce_id = list.subscribers.get(id, "bounce-id")
	    self.failUnlessEqual(bounce_id, "..notexist..")
	    bounce_id = "bounce-" + id
	    list.subscribers.set(id, "bounce-id", bounce_id)
	    bounce_path = os.path.join(bouncedir, bounce_id)
	    self.failUnless(not os.path.isfile(bounce_path))
    	    f = open(bounce_path, "w")
	    f.close()
    	    f = open(bounce_path + ".address", "w")
	    f.close()
	    self.failUnless(os.path.isfile(bounce_path))
	    list.subscribers.set(id, "timestamp-created", "1")
    	list.subscribers.save()

    	# Check that --cleaning-woman joins the two groups into one.
    	self.failUnlessEqual(len(list.subscribers.groups()), 2)
    	self.mlm.cleaning_woman(no_op)
	list = self.mlm.open_list("foo@example.com")
    	self.failUnlessEqual(len(list.subscribers.groups()), 1)
	self.failUnlessEqual(os.listdir(bouncedir), [])
	
    	# Check that a bounce splits the single group.
	self.send_mail_to_list_then_bounce_everything()
	list = self.mlm.open_list("foo@example.com")
    	self.failUnlessEqual(len(list.subscribers.groups()), 2)
	
    	# Check that a --cleaning-woman immediately after doesn't join.
	# (The groups are new, thus shouldn't be joined for a week.)
    	self.failUnlessEqual(len(list.subscribers.groups()), 2)
    	self.mlm.cleaning_woman(no_op)
	list = self.mlm.open_list("foo@example.com")
    	self.failUnlessEqual(len(list.subscribers.groups()), 2)
	

class SubscriberDbTestCases(unittest.TestCase):

    def setUp(self):
    	self.mlm = eoc.MailingListManager(DOTDIR)
	self.list = self.mlm.create_list("foo@example.com")
	self.list.subscribers.lock()
	self.list.subscribers.add("user@example.com")
	self.list.subscribers.save()

    def tearDown(self):
    	shutil.rmtree(DOTDIR)

    def testGroupStatus(self):
    	subs = self.list.subscribers
    	id = subs.groups()[0]
    	self.failUnlessEqual(subs.get(id, "status"), "ok")
	subs.lock()
	subs.set(id, "status", "bounced")
	subs.save()
    	self.failUnlessEqual(subs.get(id, "status"), "bounced")

    def testSubscribeTwice(self):
    	list = self.mlm.create_list("foo2@example.com")
    	self.failUnlessEqual(list.subscribers.get_all(), [])
	list.subscribers.lock()
	list.subscribers.add("user@example.com")
	list.subscribers.add("user@example.com")
    	subs = list.subscribers.get_all()
	subs.sort()
	self.failUnlessEqual(subs, ["user@example.com"])

    def testSubscribeMany(self):
    	list = self.mlm.create_list("foo2@example.com")
    	self.failUnlessEqual(list.subscribers.get_all(), [])
	ok = []
	list.subscribers.lock()
	for i in range(50):
	    addr = "user%d@example.com" % i
	    ok.append(addr)
	    list.subscribers.add(addr)
    	subs = list.subscribers.get_all()
	subs.sort()
	ok.sort()
	self.failUnlessEqual(subs, ok)
