/*
 *  apcsmart.c -- The decoding of the chatty little beasts.
 *		  THE LOOK-A-LIKE ( UPSlink(tm) Language )
 *
 *  apcupsd.c  -- Simple Daemon to catch power failure signals from a
 *		  BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *	       -- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick <andre@suse.com>
 *  All rights reserved.
 *
 */

/*
 *		       GNU GENERAL PUBLIC LICENSE
 *			  Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *			     675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

/*
 * Parts of the information below was taken from apcd.c & apcd.h
 *
 * Definitons file for APC SmartUPS daemon
 *
 *  Copyright (c) 1995 Pavel Korensky
 *  All rights reserved
 *
 *  IN NO EVENT SHALL PAVEL KORENSKY BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 *  OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF PAVEL KORENSKY
 *  HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  PAVEL KORENSKY SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"
 *  BASIS, AND PAVEL KORENSKY HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 *  UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  Pavel Korensky	    pavelk@dator3.anet.cz
 *
 *  8.11.1995
 *
 *  P.S. I have absolutely no connection with company APC. I didn't sign any
 *  non-disclosure agreement and I didn't got the protocol description anywhere.
 *  The whole protocol decoding was made with a small program for capturing
 *  serial data on the line. So, I think that everybody can use this software
 *  without any problem.
 *
 * A few words regarding the APC serial protocol
 *
 * Firstly, the cable:
 * You will need a simple 3 wires cable connected as follows:
 *
 * PC (9 pin)	  APC
 *  2 RxD	   2
 *  3 TxD	   1
 *  5 GND	   9
 *
 * For a description of programming the UPS, see docs/manual/upsbible.html
 *
 * One supposedly illegal command remains to be deciphered:
 *
 * Code  Value Returned
 * ?	 unknown 33AB or 3C65
 *
 */

#include "apc.h"
extern UPSCOMMANDS cmd[];
extern UPSCMDMSG cmd_msg[];

/*
 * Helper functions for smart_poll. These are generic functions but when
 * there are too specialized tasks, like malloc the buffer or don't check
 * the ups link, call directly the smart_poll as this way you will be
 * doing something out of the ordinary tasks.
 */

/*
 * apc_write: write a command to UPS but don't read the answer.
 */
int apc_write(char cmd, UPSINFO *ups)
{
    if (write(ups->fd, &cmd, 1) != 1) {
        log_event(ups, LOG_ERR, "Error writing to UPS. ERR=%s",
		  strerror(errno));
	return FAILURE;
    }
    return SUCCESS;
}

/*
 * apc_read: read an answer from UPS without writing a command.
 * This means that a command has been previously sent and now we
 * are reading the answer.
 */
char *apc_read(UPSINFO *ups)
{
    static char line[1000];
    getline(line, sizeof(line), ups);
    return line;
}

/*
 * apc_chat_static: full chat with the ups, write a command and
 * read the answer. Return a ptr to static buffer.
 */
char *apc_chat(char cmd, UPSINFO *ups)
{
    return smart_poll(cmd, ups);
}

/********************************************************************* 
 *
 * Send a charcter to the UPS and get
 * its response. Returns a pointer to the response string.
 *
 */
char *smart_poll(char cmd, UPSINFO *ups)
{
    static char answer[2000];
    int stat;

    *answer=0;
    if (ups->mode.type <= SHAREBASIC)
	return answer;	
    write(ups->fd, &cmd, 1);
    stat = getline(answer, sizeof answer, ups);
    /* If nothing returned, the link is probably down */
    if (*answer == 0 && stat == FAILURE) {
	UPSlinkCheck(ups);    /* wait for link to come up */
	*answer = 0;
    } 

    return answer;
}

/*
 * If s == NULL we are just waiting on FD for status changes.
 * If s != NULL we are asking the UPS to tell us the value of something.
 *
 * If s == NULL there is a much more fine-grained locking.
 */
int getline(char *s, int len, UPSINFO *ups)
{
    int i = 0;
    int ending = 0;
    char c;
    int retval;

    while (!ending) {
#ifndef HAVE_CYGWIN
	fd_set rfds;
	struct timeval tv;
 
	FD_ZERO(&rfds);
	FD_SET(ups->fd, &rfds);
	if (s != NULL || ups->FastPoll) {
	    tv.tv_sec = 1;     /* expect fast response */
	} else {
	    tv.tv_sec =  TIMER_SERIAL;
	}
	tv.tv_usec = 0;
	   
	errno = 0;
	retval = select((ups->fd)+1, &rfds, NULL, NULL, &tv);

	switch (retval) {
	case 0: /* No chars available in TIMER_SERIAL seconds. */
	    return FAILURE;
	case -1:
	    if (errno == EINTR || errno == EAGAIN) /* assume SIGCHLD */
		    continue;
            Error_abort1("Select error on UPS FD. %s\n",
		   strerror(errno));
	    break;
	default:
	    break;
	}

#endif

	do {
	   retval =  read(ups->fd, &c, 1);
	} while (retval == -1 && (errno == EAGAIN || errno == EINTR));
	if (retval == 0) {
	   return FAILURE;
	}

	switch(c) {
	/*
	 * Here we can be called in two ways:
	 * 
	 * s == NULL
	 *     The shm lock is not held so we must hold it here.
	 *
	 * s != NULL
	 *     We are called from a routine that have 
	 *     already held the shm lock so no need to hold it
	 *     another time. Simply update the UPS structure
	 *     fields and the shm will be updated when
	 *     write_andunlock_shmarea is called by the calling
	 *     routine.
	 */
        case UPS_ON_BATT:        /* UPS_ON_BATT = '!'   */
	    if (s == NULL)
		read_andlock_shmarea(ups);
	    ups->OnBatt = 1;
	    if (s == NULL)
		write_andunlock_shmarea(ups);
	    break;
        case UPS_REPLACE_BATTERY: /* UPS_REPLACE_BATTERY = '#'   */
	    if (s == NULL)
		read_andlock_shmarea(ups);
	    if (!ups->ChangeBatt) {   /* set if not already set */
	       ups->ChangeBatt = 1;
	    }
	    if (s == NULL)
		write_andunlock_shmarea(ups);
	    break;
        case UPS_ON_LINE:        /* UPS_ON_LINE = '$'   */
	    if (s == NULL)
		read_andlock_shmarea(ups);
	    ups->OnBatt = 0;
	    if (s == NULL)
		write_andunlock_shmarea(ups);
	    break;
        case BATT_LOW:           /* BATT_LOW    = '%'   */
	    if (s == NULL)
		read_andlock_shmarea(ups);
	    ups->BattLow = 1;
	    if (s == NULL)
		write_andunlock_shmarea(ups);
	    break;
        case BATT_OK:            /* BATT_OK     = '+'   */
	    if (s == NULL)
		read_andlock_shmarea(ups);
	    ups->BattLow = 0;
	    if (s == NULL)
		write_andunlock_shmarea(ups);
	    break;
        case UPS_EPROM_CHANGE:   /* UPS_EPROM_CHANGE = '|'   */
	    break;
        case UPS_TRAILOR:        /* UPS_TRAILOR = ':'   */
	    break;

	/* NOTE: The UPS terminates what it sends to us
         * with a \r\n. Thus the line feed signals the
	 * end of what we are to receive.
	 */
        case UPS_LF:             /* UPS_LF      = '\n'  */
	    if (s != NULL)
		ending = 1;	 /* This what we waited for */
	    break;
        case UPS_CR:             /* UPS_CR      = '\r'  */
	    break;	 
	default:
	    if (s != NULL) {
		if (i+1 < len)
		    s[i++] = c;
		else
		    ending = 1;  /* no more room in buffer */
	    }
	    break;
	}
    }

    if (s != NULL) {
        s[i] = '\0';
    }
    return SUCCESS;
}

static int linkcheck = FALSE;

/*********************************************************************/
void UPSlinkCheck (UPSINFO *ups)
{
    char *a;
    int comm_err = FALSE;
    int tlog;

    if (linkcheck)
	return;
    linkcheck = TRUE;		  /* prevent recursion */

    tcflush(ups->fd, TCIOFLUSH);
    if (strcmp((a=smart_poll('Y', ups)), "SM") == 0) {
	linkcheck = FALSE;
	ups->CommLost = FALSE;
	return;
    }
    tcflush(ups->fd, TCIOFLUSH);

    for (tlog=0; strcmp((a=smart_poll('Y', ups)), "SM") != 0; tlog -= (1+TIMER_SERIAL)) {
	if (tlog <= 0) {
	    tlog = 10 * 60; /* notify every 10 minutes */
	    log_event(ups, cmd_msg[CMDCOMMFAILURE].level,
		      cmd_msg[CMDCOMMFAILURE].msg);
	    if (!comm_err)  /* execute script once */
		execute_command(ups, cmd[CMDCOMMFAILURE]);

	}
	/* This sleep should not be necessary since the smart_poll() 
	 * routine normally waits TIMER_SERIAL (5) seconds. However,
	 * in case the serial port is broken and generating spurious
	 * characters, we sleep to reduce CPU consumption. 
	 */
	sleep(1);
	comm_err = TRUE;
	ups->CommLost = TRUE;
	tcflush(ups->fd, TCIOFLUSH);
    }

    if (comm_err) {
	tcflush(ups->fd, TCIOFLUSH);
	generate_event(ups, CMDCOMMOK);
    }
    ups->CommLost = FALSE;
    linkcheck = FALSE;
}

/********************************************************************* 
 *
 *  This subroutine is called to load our shared memory with
 *  information that is changing inside the UPS depending
 *  on the state of the UPS and the mains power.
 */
static void read_volatile_ups_data(UPSINFO *ups)
{
    UPSlinkCheck(ups);		  /* make sure serial port is working */

    time(&ups->poll_time);	  /* save time stamp */

    /* UPS_STATUS */
    if (ups->UPS_Cap[CI_STATUS]) {
	ups->Status = strtoul(smart_poll(ups->UPS_Cmd[CI_STATUS], ups),NULL,16);
	/* Use the info in the status bits */
	test_status_bits(ups);
    }

    /* ONBATT_STATUS_FLAG -- line quality */ 
    if (ups->UPS_Cap[CI_LQUAL])
	strncpy(ups->linequal, smart_poll(ups->UPS_Cmd[CI_LQUAL], ups),
	       sizeof(ups->linequal));

    /* Reason for last transfer to batteries */
    if (ups->UPS_Cap[CI_WHY_BATT])
	strncpy(ups->G, smart_poll(ups->UPS_Cmd[CI_WHY_BATT], ups), 
	       sizeof(ups->G));

    /* Results of last self test */
    if (ups->UPS_Cap[CI_ST_STAT])
	strncpy(ups->X,smart_poll(ups->UPS_Cmd[CI_ST_STAT], ups), 
	       sizeof(ups->X));

    /* LINE_VOLTAGE */
    if (ups->UPS_Cap[CI_VLINE])
	ups->LineVoltage = atof(smart_poll(ups->UPS_Cmd[CI_VLINE], ups));

    /* UPS_LINE_MAX */
    if (ups->UPS_Cap[CI_VMAX])
	ups->LineMax = atof(smart_poll(ups->UPS_Cmd[CI_VMAX], ups));

    /* UPS_LINE_MIN */
    if (ups->UPS_Cap[CI_VMIN])
	ups->LineMin = atof(smart_poll(ups->UPS_Cmd[CI_VMIN], ups));

    /* OUTPUT_VOLTAGE */
    if (ups->UPS_Cap[CI_VOUT])
	ups->OutputVoltage = atof(smart_poll(ups->UPS_Cmd[CI_VOUT], ups));

    /* BATT_FULL Battery level percentage */
    if (ups->UPS_Cap[CI_BATTLEV])
	ups->BattChg = atof(smart_poll(ups->UPS_Cmd[CI_BATTLEV], ups));

    /* BATT_VOLTAGE */
    if (ups->UPS_Cap[CI_VBATT])
	ups->BattVoltage = atof(smart_poll(ups->UPS_Cmd[CI_VBATT], ups));

    /* UPS_LOAD */
    if (ups->UPS_Cap[CI_LOAD])
	ups->UPSLoad = atof(smart_poll(ups->UPS_Cmd[CI_LOAD], ups));

    /* LINE_FREQ */
    if (ups->UPS_Cap[CI_FREQ])
	ups->LineFreq = atof(smart_poll(ups->UPS_Cmd[CI_FREQ], ups));

    /* UPS_RUNTIME_LEFT */
    if (ups->UPS_Cap[CI_RUNTIM])
	ups->TimeLeft = atof(smart_poll(ups->UPS_Cmd[CI_RUNTIM], ups));

    /* UPS_TEMP */
    if (ups->UPS_Cap[CI_ITEMP])
	ups->UPSTemp = atof(smart_poll(ups->UPS_Cmd[CI_ITEMP], ups));

    /* DIP_SWITCH_SETTINGS */
    if (ups->UPS_Cap[CI_DIPSW])
	ups->dipsw = strtoul(smart_poll(ups->UPS_Cmd[CI_DIPSW], ups), NULL, 16);

    /* Register 1 */
    if (ups->UPS_Cap[CI_REG1])
	ups->reg1 = strtoul(smart_poll(ups->UPS_Cmd[CI_REG1], ups), NULL, 16);

    /* Register 2 */
    if (ups->UPS_Cap[CI_REG2])
	ups->reg2 = strtoul(smart_poll(ups->UPS_Cmd[CI_REG2], ups), NULL, 16);

    /* Register 3 */
    if (ups->UPS_Cap[CI_REG3])
	ups->reg3 = strtoul(smart_poll(ups->UPS_Cmd[CI_REG3], ups), NULL, 16);

    /*	Humidity percentage */ 
    if (ups->UPS_Cap[CI_HUMID])
	ups->humidity = atof(smart_poll(ups->UPS_Cmd[CI_HUMID], ups));

    /*	Ambient temperature */ 
    if (ups->UPS_Cap[CI_ATEMP])
	ups->ambtemp = atof(smart_poll(ups->UPS_Cmd[CI_ATEMP], ups));

    /*	Hours since self test */
    if (ups->UPS_Cap[CI_ST_TIME])
	ups->LastSTTime = atof(smart_poll(ups->UPS_Cmd[CI_ST_TIME], ups));

}

/********************************************************************* 
 *
 *  This subroutine is called to load our shared memory with
 *  information that is static inside the UPS.	Hence it
 *  normally would only be called once when starting up the
 *  UPS.
 */
void read_static_ups_data(UPSINFO *ups)
{
    /* Everything from here on down is non-volitile, that is
     * we do not expect it to change while the UPS is running
     * unless we explicitly change it.
     */

    /* SENSITIVITY */
    if (ups->UPS_Cap[CI_SENS])
	strncpy(ups->sensitivity, smart_poll(ups->UPS_Cmd[CI_SENS], ups),
	       sizeof(ups->sensitivity));

    /* WAKEUP_DELAY */
    if (ups->UPS_Cap[CI_DWAKE])
	ups->dwake = atof(smart_poll(ups->UPS_Cmd[CI_DWAKE], ups));

    /* SLEEP_DELAY */
    if (ups->UPS_Cap[CI_DSHUTD])
	ups->dshutd = atof(smart_poll(ups->UPS_Cmd[CI_DSHUTD], ups));

    /* LOW_TRANSFER_LEVEL */
    if (ups->UPS_Cap[CI_LTRANS])
	ups->lotrans = atof(smart_poll(ups->UPS_Cmd[CI_LTRANS], ups));

    /* HIGH_TRANSFER_LEVEL */
    if (ups->UPS_Cap[CI_HTRANS])
	ups->hitrans = atof(smart_poll(ups->UPS_Cmd[CI_HTRANS], ups));

    /* UPS_BATT_CAP_RETURN */
    if (ups->UPS_Cap[CI_RETPCT])
	ups->rtnpct = atof(smart_poll(ups->UPS_Cmd[CI_RETPCT], ups));

    /* ALARM_STATUS */
    if (ups->UPS_Cap[CI_DALARM])
	strncpy(ups->beepstate, smart_poll(ups->UPS_Cmd[CI_DALARM], ups), 
	       sizeof(ups->beepstate));

    /* LOWBATT_SHUTDOWN_LEVEL */
    if (ups->UPS_Cap[CI_DLBATT])
	ups->dlowbatt = atof(smart_poll(ups->UPS_Cmd[CI_DLBATT], ups));

    /* UPS_NAME */
    if (ups->upsname[0] == 0 && ups->UPS_Cap[CI_IDEN])
	strncpy(ups->upsname, smart_poll(ups->UPS_Cmd[CI_IDEN], ups), 
	       sizeof(ups->upsname));

    /* UPS_SELFTEST */
    if (ups->UPS_Cap[CI_STESTI])
	strncpy(ups->selftest, smart_poll(ups->UPS_Cmd[CI_STESTI], ups), 
	       sizeof(ups->selftest));

    /* UPS_MANUFACTURE_DATE */
    if (ups->UPS_Cap[CI_MANDAT])
	strncpy(ups->birth, smart_poll(ups->UPS_Cmd[CI_MANDAT], ups), 
	       sizeof(ups->birth));

    /* UPS_SERIAL_NUMBER */
    if (ups->UPS_Cap[CI_SERNO])
	strncpy(ups->serial, smart_poll(ups->UPS_Cmd[CI_SERNO], ups), 
	       sizeof(ups->serial));

    /* UPS_BATTERY_REPLACE */
    if (ups->UPS_Cap[CI_BATTDAT])
	strncpy(ups->battdat, smart_poll(ups->UPS_Cmd[CI_BATTDAT], ups), 
	       sizeof(ups->battdat));

    /* Nominal output voltage when on batteries */
    if (ups->UPS_Cap[CI_NOMOUTV])
	ups->NomOutputVoltage = (int)atof(smart_poll(ups->UPS_Cmd[CI_NOMOUTV], ups));

    /* Nominal battery voltage */
    if (ups->UPS_Cap[CI_NOMBATTV]) 
	ups->nombattv = atof(smart_poll(ups->UPS_Cmd[CI_NOMBATTV], ups));

    /*	Firmware revision */ 
    if (ups->UPS_Cap[CI_REVNO])
	strncpy(ups->firmrev, smart_poll(ups->UPS_Cmd[CI_REVNO], ups), 
	       sizeof(ups->firmrev));

    /*	Number of external batteries installed */
    if (ups->UPS_Cap[CI_EXTBATTS])
	ups->extbatts = (int) atof(smart_poll(ups->UPS_Cmd[CI_EXTBATTS], ups));
    
    /*	Number of bad batteries installed */
    if (ups->UPS_Cap[CI_BADBATTS])
	ups->badbatts = (int) atof(smart_poll(ups->UPS_Cmd[CI_BADBATTS], ups));

    /*	Old firmware revision */
    if (ups->UPS_Cap[CI_UPSMODEL])
	strncpy(ups->upsmodel, smart_poll(ups->UPS_Cmd[CI_UPSMODEL], ups),
	       sizeof(ups->upsmodel));


    /*	EPROM Capabilities */
    if (ups->UPS_Cap[CI_EPROM])
	strncpy(ups->eprom, smart_poll(ups->UPS_Cmd[CI_EPROM], ups),
	       sizeof(ups->eprom));

}


/********************************************************************* 
 *
 *  Called from apcserial.c once per second to read all UPS
 *  info
 */
int fillUPS (UPSINFO *ups)
{
    if (ups->mode.type <= SHAREBASIC)
	return 0;	      /* dumb UPS */
    read_andlock_shmarea(ups);

    read_volatile_ups_data(ups);

    /* UPS_ENABLE */
    smart_poll('Y', ups);
    smart_poll('Y', ups);

    write_andunlock_shmarea(ups);
    return(0);
}


/********************************************************************* 
 * Read all available information about UPS by asking it questions
 *
 *  Called only from apcupsd during configuration (-c option)
 */
void ReadUPS (UPSINFO *ups)
{
    if (ups->mode.type <= SHAREBASIC)
	return; 	      /* dumb UPS */
    read_volatile_ups_data(ups);
    read_static_ups_data(ups);
}

/**********************************************************************
 * Is this a special case for smart UPSes since,
 * getline(int getfd, int len, char *s) tests the entire 8-bit serial read
 * regardless of the polling command character. 
 *********************************************************************/
void test_status_bits(UPSINFO *ups)
{
    if (ups->Status & UPS_ONBATT)
	ups->OnBatt = 1;	      /* On battery power */
    else
	ups->OnBatt = 0;
    if (ups->Status & UPS_BATTLOW)
	ups->BattLow = 1;	      /* battery low */ 
    else
	ups->BattLow = 0;
    if (ups->Status & UPS_SMARTBOOST)
	ups->LineLevel = -1;	      /* LineVoltage Low */
    else if (ups->Status & UPS_SMARTTRIM)
	ups->LineLevel = 1;	      /* LineVoltage High */
    else
	ups->LineLevel = 0;	      /* LineVoltage Normal */

    if (ups->Status & UPS_REPLACEBATT) { /* Replace Battery */
	if (!ups->ChangeBatt) {       /* This is a counter, so check before */
	   ups->ChangeBatt = 1;       /* setting it */
	}
    }
}
