#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <machine/sysarch.h>
#include <machine/pio.h>
#include <unistd.h>

#if 0
#define DPR(a) printf a
#else
#define DPR(a)
#endif

struct iomap {
#define IOMAP_NUMBITS	1024
#define IOMAP_EACHBITS	32
	u_long iom_map[IOMAP_NUMBITS/IOMAP_EACHBITS];
};

int
iomap_init(struct iomap *iom)
{
	return i386_get_ioperm(iom->iom_map);
}

int
iomap_setbit(struct iomap *iom, int addr)
{
	if (addr<0 || addr>=IOMAP_NUMBITS)
		return -1;
	iom->iom_map[addr/IOMAP_EACHBITS] |= 1 << (addr%IOMAP_EACHBITS);
	return 0;
}

int
iomap_clearbit(struct iomap *iom, int addr)
{
	if (addr<0 || addr>=IOMAP_NUMBITS)
		return -1;
	iom->iom_map[addr/IOMAP_EACHBITS] &= ~(1 << (addr%IOMAP_EACHBITS));
	return 0;
}

int
iomap_set(struct iomap *iom)
{
	return i386_set_ioperm(iom->iom_map);
}

/* ---------------------------------------------------------------------- */

#define EC_DATAREG	0x62
#define EC_CSREG	0x66

#define ECCMD_READ	0x80
#define ECCMD_WRITE	0x81
#define ECCMD_ENBURST	0x82
#define ECCMD_DIBURST	0x83
#define ECCMD_QUERY	0x84

#define ECST_OBF	0x01
#define ECST_IBF	0x02

int
enable_ec()
{
	struct iomap iom;
	if (iomap_init(&iom)) {
		DPR(("iomap_init failed\n"));
		return -1;
	}
	iomap_clearbit(&iom, EC_DATAREG);
	iomap_clearbit(&iom, EC_CSREG);
	if (iomap_set(&iom)) {
		DPR(("iomap_set failed\n"));
		return -1;
	}
}

#define EC_RETRY 1000
int
wait_inbuf()
{
	int i;
	for (i=0; i<EC_RETRY; i++) {
		if (!(inb(EC_CSREG) & ECST_IBF))
			return 0;
		usleep(1000);
	}
	DPR(("wait_inbuf: timeout\n"));
	return -1;
}

int
wait_outbuf()
{
	int i;
	for (i=0; i<EC_RETRY; i++) {
		if ((inb(EC_CSREG) & ECST_OBF))
			return 0;
		usleep(1000);
	}
	DPR(("wait_outbuf: timeout\n"));
	return -1;
}

u_int8_t
read_ec(u_int8_t addr)
{
	outb(EC_CSREG, ECCMD_READ);
	wait_inbuf();
	outb(EC_DATAREG, addr);
	wait_outbuf();
	return inb(EC_DATAREG);
}

void
write_ec(u_int8_t addr, u_int8_t data)
{
	outb(EC_CSREG, ECCMD_WRITE);
	wait_inbuf();
	outb(EC_DATAREG, addr);
	wait_inbuf();
	outb(EC_DATAREG, data);
	wait_inbuf();
}

/* ---------------------------------------------------------------------- */

struct acpi_pbif {
	u_int32_t pbif_power_unit;
	u_int32_t pbif_design_capacity;
	u_int32_t pbif_last_full_charge_capacity;
	u_int32_t pbif_battery_technology;
	u_int32_t pbif_design_voltage;
	u_int32_t pbif_design_capacity_of_warning;
	u_int32_t pbif_design_capacity_of_low;
	u_int32_t pbif_battery_capacity_granulatity1;
	u_int32_t pbif_battery_capacity_granulatity2;
	char *pbif_model_number;
	char *pbif_serial_number;
	char *pbif_battery_type;
	char *pbif_oem_information;
};

struct acpi_pbst {
	u_int32_t pbst_state;
	u_int32_t pbst_present_rate;
	u_int32_t pbst_remaining_capacity;
	u_int32_t pbst_present_voltage;
};

u_int32_t bp__ = 0;

u_int8_t
get_ec_bank()
{
	return (read_ec(0x53)>>4)&0xF;
}

void
set_ec_bank(u_int8_t v)
{
	write_ec(0x53, ((v&0xF)<<4) | (read_ec(0x53)&0xF));
}

u_int32_t
get_ec_dbat()
{
	u_int32_t v;
	v = read_ec(0xfd);
	v |= (u_int32_t)read_ec(0xfe) << 8;
	return v;
}

void
get_bp__()
{
	u_int8_t tmp;
	tmp = get_ec_bank(0x53);
	set_ec_bank(4);
	bp__ = get_ec_dbat() & 1;
	DPR(("bp__=0x%02X\n", (unsigned)bp__));
}

u_int32_t
get_ec_mbct()
{
	u_int32_t v;
	v = read_ec(0x9a);
	v |= (u_int32_t)read_ec(0x9b) << 8;
	return v;
}

u_int32_t
get_ec_mbvt()
{
	u_int32_t v;
	v = read_ec(0x96);
	v |= (u_int32_t)read_ec(0x97) << 8;
	return v;
}

u_int32_t
get_ec_z00t()
{
	u_int32_t v;
	v = read_ec(0xf6);
	v |= (u_int32_t)read_ec(0xf7) << 8;
	return v;
}

void
call_upbs(struct acpi_pbst *pbst)
{
	u_int32_t local0, local1, local2, local3, local4, local5;
	u_int32_t tmpb, tbat;
	local1 = get_ec_bank();
	set_ec_bank(6);
	local2 = get_ec_mbct();
	if (local2 & 0x8000) {
		local2 &= 0xffff0000;
		local2 = ~local2 + 1;
	}
	local3 = get_ec_mbvt();
	set_ec_bank(4);
	local5 = get_ec_z00t();
	set_ec_bank(local1);
	local3 *= 0x14;
	local4 = local3/0x40; local3 %= 0x40;
	pbst->pbst_present_rate = local2;
	pbst->pbst_present_voltage = local3;
	pbst->pbst_remaining_capacity = local5;

	local4 = 0;
	tmpb = get_ec_bank();
	set_ec_bank(4);
	tbat = get_ec_dbat();
	set_ec_bank(tmpb);
	if (tbat & 0x0c00) {
		local4 = 2;
	} else {
		local4 = 1;
	}
	pbst->pbst_state = local4;
}

void
call__bst(struct acpi_pbst *pbst)
{
	if (bp__) {
		/* UPBS */
		call_upbs(pbst);
	} else {
		/* IVBS */
		pbst->pbst_state = 0;
		pbst->pbst_present_rate = 5;
		pbst->pbst_remaining_capacity = 5;
		pbst->pbst_present_voltage = 0x2710;
	}
}

#define SMPR 0x18
#define SMST 0x19
#define SMAD 0x1a
#define SMCM 0x1b
#define SMW0 0x1c
#define SMB0 0x1c
#define SMD0 0x1c

struct ecref {
	enum { ECREF_BYTE, ECREF_WORD, ECREF_DWORD, ECREF_DESC } type;
	int bits;
	union {
		uint8_t byte;
		uint16_t word;
		uint32_t dword;
		char *desc;
	} u;
};

uint32_t
call_smrd(uint32_t a0, uint32_t a1, uint32_t a2, struct ecref *a3)
{
	uint32_t l0, l1;
	int i;
	char *p;

	if (a0!=7 && a0!=9 && a0!=11)
		return 0x19;
	l0 = 4;
	while (l0>1) {
		write_ec(SMST, read_ec(SMST) & 0x40);
		write_ec(SMCM, a2);
		write_ec(SMAD, a1);
		write_ec(SMPR, a0);
		while (!(l1=(read_ec(SMST) & 0xbf))) {
			usleep(2000);
		}
		if (l1==0x80)
			l0 = 0;
		else
			l0--;
	}
	if (l0)
		l0 = l1&0x1f;
	else {
		switch (a0) {
		case 7:
			a3->type = ECREF_BYTE;
			a3->bits = 8;
			a3->u.byte = read_ec(SMB0);
			break;
		case 9:
			a3->type = ECREF_WORD;
			a3->bits = 16;
			a3->u.word = read_ec(SMW0);
			a3->u.word |= (uint16_t)read_ec(SMW0+1) << 8;
			break;
		case 11:
			a3->type = ECREF_DESC;
			a3->bits = 264;
			a3->u.desc = p = malloc(264/8);
			for (i=0; i<264/8; i++) {
				p[i] = read_ec(SMD0+i);
			}
			break;
		}
	}
	return l0;
}

void
call_upbi(struct acpi_pbif *pbif)
{
	struct ecref ref;

	/* 0 */
	pbif->pbif_power_unit = 1;
	/* 1 */
	pbif->pbif_design_capacity = 5;
	if (!call_smrd(9, 0x16, 0x18, &ref)) {
		pbif->pbif_design_capacity = ref.u.word;
	}
	/* 2 */
	pbif->pbif_last_full_charge_capacity = 5;
	if (!call_smrd(9, 0x16, 0x10, &ref)) {
		pbif->pbif_last_full_charge_capacity = ref.u.word;
	}
	/* 3 */
	pbif->pbif_battery_technology = 1;
	/* 4 */
	pbif->pbif_design_voltage = 5;
	if (!call_smrd(9, 0x19, 0x10, &ref)) {
		pbif->pbif_design_voltage = ref.u.word;
	}
	/* 5 */
	pbif->pbif_design_capacity_of_warning = 0x190;
	/* 6 */
	pbif->pbif_design_capacity_of_low = 0x32;
	/* 7 */
	pbif->pbif_battery_capacity_granulatity1 = 0x40;
	/* 8 */
	pbif->pbif_battery_capacity_granulatity2 = 0x40;
	/* 9 */
	pbif->pbif_model_number = strdup("BAT1");
	if (!call_smrd(11, 0x16, 0x21, &ref)) {
		free(pbif->pbif_model_number);
		pbif->pbif_model_number = ref.u.desc;
	}
	/* A */
	pbif->pbif_serial_number = strdup(" ");
	if (!call_smrd(9, 0x16, 0x1c, &ref)) {
		free(pbif->pbif_serial_number);
		asprintf(&pbif->pbif_serial_number, "%d", (int)ref.u.word);
	}
	/* B */
	pbif->pbif_battery_type = strdup(" ");
	if (!call_smrd(11, 0x16, 0x22, &ref)) {
		free(pbif->pbif_battery_type);
		pbif->pbif_battery_type = ref.u.desc;
	}
	/* C */
	pbif->pbif_oem_information = strdup(" ");
	if (!call_smrd(11, 0x16, 0x22, &ref)) {
		free(pbif->pbif_oem_information);
		pbif->pbif_oem_information = ref.u.desc;
	}
}

void
call_ivbi(struct acpi_pbif *pbif)
{
	/* 0 */
	pbif->pbif_power_unit = 1;
	/* 1 */
	pbif->pbif_design_capacity = 5;
	/* 2 */
	pbif->pbif_last_full_charge_capacity = 5;
	/* 3 */
	pbif->pbif_battery_technology = 1;
	/* 4 */
	pbif->pbif_design_voltage = 5;
	/* 5 */
	pbif->pbif_design_capacity_of_warning = 0x190;
	/* 6 */
	pbif->pbif_design_capacity_of_low = 0x32;
	/* 7 */
	pbif->pbif_battery_capacity_granulatity1 = 0x40;
	/* 8 */
	pbif->pbif_battery_capacity_granulatity2 = 0x40;
	/* 9 */
	pbif->pbif_model_number = strdup("Bad");
	/* A */
	pbif->pbif_model_number = strdup("Bad");
	/* B */
	pbif->pbif_battery_type = strdup("Bad");
	/* C */
	pbif->pbif_oem_information = strdup("Bad");
}

void
call__bif(struct acpi_pbif *pbif)
{
	if (bp__) {
		/* UPBI */
		call_upbi(pbif);
	} else {
		/* IVBI */
		call_ivbi(pbif);
	}
}



/* ---------------------------------------------------------------------- */

char *
skip_space(char *p)
{
	while (*p && isspace(*p))
		p++;
	return p;
}

int
main()
{
	int addr, data;
	char buf[256], *p, *q;
	struct acpi_pbst pbst;
	struct acpi_pbif pbif;

	if (enable_ec()) {
		printf("error: cannot enable embedded controller.\n");
		return 1;
	}
	get_bp__();
	call__bif(&pbif);
	call__bst(&pbst);

	if (!bp__) {
		printf("no battery.\n");
		return 0;
	}
	printf("Battery Information:\n");
	printf("\tPower Unit: %s\n", pbif.pbif_power_unit?"mAh":"mWh");
	printf("\tDesign Capacity: %d\n", pbif.pbif_design_capacity);
	printf("\tLast Full Charge Capacity: %d\n",
	       pbif.pbif_last_full_charge_capacity);
	printf("\tBattery Technology: %s\n",
	       pbif.pbif_battery_technology?"secondary":"primary");
	printf("\tDesign Voltage: %d\n", pbif.pbif_design_voltage);
	printf("\tDesign Capacity of Warning: %d\n",
	       pbif.pbif_design_capacity_of_warning);
	printf("\tDesign Capacity of Low: %d\n",
	       pbif.pbif_design_capacity_of_low);
	printf("\tBattery Capacity Granulatity1: %d\n",
	       pbif.pbif_battery_capacity_granulatity1);
	printf("\tBattery Capacity Granulatity2: %d\n",
	       pbif.pbif_battery_capacity_granulatity2);
	printf("\tModel Number: \"%s\"\n", pbif.pbif_model_number);
	printf("\tSerial Number: \"%s\"\n", pbif.pbif_serial_number);
	printf("\tBattery Type: \"%s\"\n", pbif.pbif_battery_type);
	printf("\tOEM Information: \"%s\"\n", pbif.pbif_oem_information);

	printf("Battery Status:\n");
	printf("\tState: discharging=%s, charging=%s, critical=%s\n",
	       pbst.pbst_state&1?"YES":"NO",
	       pbst.pbst_state&2?"YES":"NO",
	       pbst.pbst_state&4?"YES":"NO");
	printf("\tPresent Rate: %d\n", pbst.pbst_present_rate);
	printf("\tRemaining Capacity: %d\n",
	       pbst.pbst_remaining_capacity);
	printf("\tPresent Voltage: %d\n",
	       pbst.pbst_present_voltage);

	printf("\n");

	printf("battery remain=%d%%\n",
	       (int)((pbst.pbst_remaining_capacity/(double)
		      pbif.pbif_last_full_charge_capacity+0.005)*100));
	printf("battery life=%dm\n",
	       (int)((pbst.pbst_remaining_capacity/(double)
		      pbst.pbst_present_rate)*60));
	       
	
	return 0;
}
