/*
 	Copyright (C) 2001  Sony Computer Entertainment Inc.
 
  This file is subject to the terms and conditions of the GNU Library
  General Public License Version 2. See the file "COPYING.LIB" in the 
  main directory of this archive for more details.
*/

//
// "main.c"
//
//
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <sys/fcntl.h>
#include "libvu0.h"
#include "ps2gs.h"
#include "ps2vpu.h"
#include "ps2dma.h"
#include "ps2vpufile.h"
#include "vu0.h"
#include "sjoy.h"

//----------------------------------------------------------------------
#define IMAGE_SIZE   256

//----------------------------------------------------------------------
IMPORT_VPU_SYMBOL(__u128 *, My_texture1, 0);

//----------------------------------------------------------------------
extern __u32 SampleCubeDataHead[];
extern __u32 SampleTorusDataHead[];
extern __u32 SampleTorus1DataHead[];

//----------------------------------------------------------------------
typedef struct {
	int size;
	ps2_giftag giftag;
	ps2_gs_texenv gs_tex;
} TexEnv;

typedef union {
	__u128 ul128;
	__u64 ul64[2];
	__u32 ui32[4];
	ps2_vu0_fvector vect;
} QWdata;

typedef struct{
	QWdata *buf;
	__u32 size;
	__u32 *pBase;
}GifPacket;

typedef struct{
    __u32 *pDataHead;
    __u32 magic;
    __u32 blockNum;
    __u32 prim;
    __u32 dataNum;
    __u32 *vertexNum;
    float **pData;
    GifPacket *pack;
}ObjData;

//----------------------------------------------------------------------
char *g_program_name;

int g_inter;
int g_out_mode;
int g_ff_mode;
int g_resolution;
int g_refresh_rate;
int g_psm;
int g_zpsm;
int g_zbits;

ps2_vu0_fvector camera_p = { 0, 0, -25, 0 };
ps2_vu0_fvector camera_zd = { 0, 0, 1, 1 };
ps2_vu0_fvector camera_yd = { 0, 1, 0, 1 };
ps2_vu0_fvector camera_rot = { 0, 0, 0, 0 };

ps2_vu0_fvector light0 = { 0.0, 1.5, 0.5, 0 };
ps2_vu0_fvector light1 = { 1.5, -0.5, 0.5, 0 };
ps2_vu0_fvector light2 = { -1.5, -0.5, 0.5, 0 };

ps2_vu0_fvector color0  = { 0.8, 0.3, 0.3, 1 };
ps2_vu0_fvector color1  = { 0.3, 0.8, 0.3, 1 };
ps2_vu0_fvector color2  = { 0.3, 0.3, 0.8, 1 };

ps2_vu0_fvector ambient = { 0.2, 0.2, 0.2, 0 };

ps2_vu0_fvector obj_trans = { 0, 0, 0, 0 };
ps2_vu0_fvector obj_rot = { 0, 0, 0, 0 };

ps2_vu0_fmatrix local_world;
ps2_vu0_fmatrix world_view;
ps2_vu0_fmatrix view_screen;
ps2_vu0_fmatrix local_screen;

ps2_vu0_fmatrix normal_light;
ps2_vu0_fmatrix light_color;
ps2_vu0_fmatrix local_light;
ps2_vu0_fmatrix local_color;

ps2_vu0_fmatrix work;

int paddata;

ps2_gs_dbuff g_db;
ps2_gs_finish g_finish;
struct ps2_image g_img;
int g_textop64;
TexEnv texenv;

int g_fd_gs;
ps2_vpu *g_vpu0;

//----------------------------------------------------------------------
void load_teximages(void)
{
	ps2_gs_set_image(&g_img, g_textop64, IMAGE_SIZE / 64, PS2_GS_PSMCT32,
					 0, 0, IMAGE_SIZE, IMAGE_SIZE, My_texture1);
	ps2_gs_load_image(&g_img);
}

//----------------------------------------------------------------------
int release(void)
{
	if (g_fd_gs >= 0) {
		ps2_gs_close();
	}
	
	if (g_vpu0) {
		ps2_vpu_close(g_vpu0);
	}
	
	sjoy_close();
	
	return PS2_GS_VC_REL_SUCCESS;
}

//----------------------------------------------------------------------
int acquire(void)
{
	g_fd_gs = ps2_gs_open(-1);
	g_vpu0 = ps2_vpu_open(0); // must be opened for vpu1...
	
	if (g_fd_gs < 0 || g_vpu0 == NULL) {
	    release();
	    return PS2_GS_VC_ACQ_FAILURE;
	}
	
	ps2_vpu_reset(g_vpu0);
	
	ps2_gs_reset(0, g_inter, g_out_mode, g_ff_mode, g_resolution,
				 g_refresh_rate);
	
	load_teximages();
	
	sjoy_open();
	
	return PS2_GS_VC_ACQ_SUCCESS;
}

//----------------------------------------------------------------------
void LoadObj(ObjData *o, __u32 *Head)
{
    int i;
	
    o->pDataHead = Head;
    o->magic = *o->pDataHead;
    o->blockNum = *(o->pDataHead + 2) >> 16;
    o->prim = *(o->pDataHead + 2) & 0x0000ffff;
    o->dataNum = *(o->pDataHead + 3);
	
    o->pData = malloc(sizeof(float *) * o->blockNum);
    o->vertexNum = malloc(sizeof(__u32) * o->blockNum);
    o->pack = malloc(sizeof(GifPacket) * o->blockNum);
	
    o->pData[0] = (float *)(o->pDataHead + 8);
    o->vertexNum[0] = *(o->pDataHead + 4);
    for (i = 0; i < o->blockNum - 1; i++) {
		int nv = o->vertexNum[i] * 4 * 4;
		o->vertexNum[i + 1] = *(__u32 *)(o->pData[i] + nv);
		o->pData[i + 1] = o->pData[i] + nv + 4;
    }
}

//----------------------------------------------------------------------
void ReleaseObj(ObjData *o)
{
    free(o->pData);
    free(o->vertexNum);
    free(o->pack);
}

//----------------------------------------------------------------------
void MakePacket(ObjData *obj, int num)
{
	static __u8 s_spr[16 * 1024]; // scratch pad emu
	ps2_giftag *giftag;
	ps2_dmatag *dmatag;
    GifPacket *pack;
    ps2_vu0_fvector *vertex, *normal, *texUV, *color;
	
	pack = &obj->pack[num];
    
	vertex = (ps2_vu0_fvector *)obj->pData[num];
    normal = (ps2_vu0_fvector *)(obj->pData[num] +obj->vertexNum[num] * 4);
    texUV = (ps2_vu0_fvector *)(obj->pData[num] +obj->vertexNum[num] * 4 * 2);
    color = (ps2_vu0_fvector *)(obj->pData[num] +obj->vertexNum[num] * 4 * 3);
	
	// make packet(PACKED mode)
    pack->size = 0;
    pack->buf = (QWdata *)s_spr;
	
	// add DMAtag
    pack->buf[pack->size].ul128 = 0;
	dmatag = (ps2_dmatag *)&pack->buf[pack->size].ul128;
	dmatag->ID = PS2_DMATAG_END;
	dmatag->QWC = obj->vertexNum[num] * 3 + 1;
	pack->size++;
	
	// add vertex info(GIFtag, STQ & RGBA & XYZ)
	giftag = (ps2_giftag *)&pack->buf[pack->size++];
	giftag->NLOOP = obj->vertexNum[num];
	giftag->EOP = 1;
	giftag->PRE = 1;
	giftag->PRIM = obj->prim;
	giftag->FLG = PS2_GIFTAG_FLG_PACKED;
	giftag->NREG = 3;
	giftag->REGS0 = PS2_GIFTAG_REGS_ST;
	giftag->REGS1 = PS2_GIFTAG_REGS_RGBAQ;
	giftag->REGS2 = PS2_GIFTAG_REGS_XYZ2;
	
    vu0_rot_trans_pers_n_clip_col(&pack->buf[pack->size].ul128, local_screen,
								  vertex, normal, texUV,color, local_light,
								  light_color, obj->vertexNum[num]);
	
    pack->size += obj->vertexNum[num] * 3;
}

//----------------------------------------------------------------------
void usage(const char *msg1, const char *msg2)
{
	if (msg1) {
		fprintf(stderr, "%s: %s %s\n", g_program_name, msg1, msg2 ? msg2 : "");
	}
	
	fprintf(stderr,
			"Usage: %s [<ii>] [<mode>] [<ff>] [<reso>] [<rate>] [<bpp>] [<z-bpp>]\n",
			g_program_name);
	fprintf(stderr,
			"    ii: interlace or noninterlace\n"
			"        -inter, -nointer\n"
			"    mode: video mode\n"
			"        -ntsc, -pal, -dtv, -vesa\n"
			"    ff: field/frame mode\n"
			"        -frame, -field\n"
			"    reso: resolution\n"
			"        -640x480, -800x600, -1024x768\n"
			"    rate: refresh rate\n"
			"        -60, -75\n"
			"    bpp: color bits per pixel\n"
			"        -16, -24, -32\n"
			"    z-bpp: depth bits per pixel\n"
			"        -z0, -z16, -z24\n"
			);
	
	exit(1);
}

//----------------------------------------------------------------------
void options(int argc, char *argv[])
{
	int i;
	
	g_inter = PS2_GS_NOINTERLACE;
	g_out_mode = PS2_GS_VESA;
	g_ff_mode = PS2_GS_FRAME;
	g_resolution = PS2_GS_640x480;
	g_refresh_rate = PS2_GS_60Hz;
	g_psm = PS2_GS_PSMCT32;
	g_zpsm = PS2_GS_PSMZ24;
	g_zbits = 24;
	
	for (i = 1; i < argc; i++) {
		char *p = argv[i];
		
		if (strcmp(p, "-inter") == 0) {
			g_inter = PS2_GS_INTERLACE;
		} else if (strcmp(p, "-nointer") == 0) {
			g_inter = PS2_GS_NOINTERLACE;
			
		} else if (strcmp(p, "-ntsc") == 0) {
			g_out_mode = PS2_GS_NTSC;
		} else if (strcmp(p, "-pal") == 0) {
			g_out_mode = PS2_GS_PAL;
		} else if (strcmp(p, "-dtv") == 0) {
			g_out_mode = PS2_GS_DTV;
		} else if (strcmp(p, "-vesa") == 0) {
			g_out_mode = PS2_GS_VESA;
			
		} else if (strcmp(p, "-frame") == 0) {
			g_ff_mode = PS2_GS_FRAME;
		} else if (strcmp(p, "-field") == 0) {
			g_ff_mode = PS2_GS_FIELD;
			
		} else if (strcmp(p, "-640x480") == 0) {
			g_resolution = PS2_GS_640x480;
		} else if (strcmp(p, "-800x600") == 0) {
			g_resolution = PS2_GS_800x600;
		} else if (strcmp(p, "-1024x768") == 0) {
			g_resolution = PS2_GS_1024x768;
		} else if (strcmp(p, "-1280x1024") == 0) {
			g_resolution = PS2_GS_1280x1024;
			
		} else if (strcmp(p, "-60") == 0) {
			g_refresh_rate = PS2_GS_60Hz;
		} else if (strcmp(p, "-75") == 0) {
			g_refresh_rate = PS2_GS_75Hz;
			
		} else if (strcmp(p, "-32") == 0) {
			g_psm = PS2_GS_PSMCT32;
		} else if (strcmp(p, "-24") == 0) {
			g_psm = PS2_GS_PSMCT24;
		} else if (strcmp(p, "-16") == 0) {
			g_psm = PS2_GS_PSMCT16S;
			
		} else if (strcmp(p, "-z24") == 0) {
			g_zpsm = PS2_GS_PSMZ24;
			g_zbits = 24;
		} else if (strcmp(p, "-z16") == 0) {
			g_zpsm = PS2_GS_PSMZ16S;
			g_zbits = 16;
		} else if (strcmp(p, "-z0") == 0) {
			g_zpsm = PS2_GS_PSMZ24;
			g_zbits = 0;
			
		} else {
			usage("illegal optopn", p);
		}
	}
	
	if (g_psm == PS2_GS_PSMCT16S && g_zpsm == PS2_GS_PSMZ16) {
		g_psm = PS2_GS_PSMCT16;
		g_zpsm = PS2_GS_PSMZ16;
	}
}

//----------------------------------------------------------------------
int main(int argc, char *argv[])
{
	int i, frame, odev, obj_switch, next2k;
    ObjData obj[2];
    float delta;
    int toggle = 0;
	VPUFILE *vfd;
	ps2_gs_gparam *gp = ps2_gs_get_gparam();
	
	g_program_name = argv[0];
	
	options(argc, argv);
	
	g_fd_gs = ps2_gs_open(-1);
	
	ps2_gs_vc_graphicsmode();
	
	g_vpu0 = ps2_vpu_open(0); // for VU0 macro mode
		 
	vfd = vpuobj_open("flower.elf", O_DATA_PS2MEM);
	if (vfd == NULL) {
		perror("vpu_open");
		exit(1);
	}
	
	obj_switch = 0;
	delta = 1.0f * (float)M_PI / 180.0f;
	
    LoadObj(&obj[0], SampleCubeDataHead);
    LoadObj(&obj[1], SampleTorus1DataHead);
	
	ps2_gs_reset(0, g_inter, g_out_mode, g_ff_mode, g_resolution,
				 g_refresh_rate);
	
	next2k = ps2_gs_set_dbuff(&g_db, g_psm, gp->width, gp->height,
							  (g_zbits == 0) ? 0 : PS2_GS_ZGREATER,
							  g_zpsm, 1);
    *(__u64 *)&g_db.clear0.rgbaq = PS2_GS_SETREG_RGBAQ(0x10, 0x10, 0x10, 0x80,
													 0x3f800000);
    *(__u64 *)&g_db.clear1.rgbaq = PS2_GS_SETREG_RGBAQ(0x10, 0x10, 0x10, 0x80,
													 0x3f800000);
	
	g_textop64 = next2k * 2048 / 64;
	
	load_teximages();
	
	// view -> screen matrix
    ps2_vu0_view_screen_matrix(view_screen, 512.0f, 1.0f, gp->pixel_ratio,
							   gp->center_x, gp->center_y,
							   1.0f, (g_zbits == 0) ? 2 : ((1 << g_zbits) - 1),
							   1.0f, 65536.0f);
	
    sjoy_open();
 	
    frame = 0;
	
	// clear back buffer
	ps2_gs_put_drawenv(&g_db.giftag1);
	
	// wait clear done
	ps2_gs_set_finish(&g_finish);
	ps2_gs_wait_finish(&g_finish);
	
	odev = !ps2_gs_sync_v(0);
	
	ps2_gs_start_display(1);
	
	ps2_gs_vc_enablevcswitch(acquire, release);
	
    for (; ;) {
		const float pi = (float)M_PI;
		const float pi2 = pi * 2;
		
		ps2_gs_vc_lock();
		
		ps2_gs_set_half_offset((frame & 1) ? &g_db.draw1 : &g_db.draw0, odev);
		ps2_gs_swap_dbuff(&g_db, frame);
		
		// --- read pad ---
		sjoy_poll();
		paddata = sjoy_get_ps2_button(0);
		
        // --- object rotate & change view point ---
        if (paddata & SJOY_PS2_L_DOWN) {
			obj_rot[0] += delta;
			if (obj_rot[0] > pi) {
				obj_rot[0] -= pi2;
			}
		}
		
        if (paddata & SJOY_PS2_L_UP) {
			obj_rot[0] -= delta;
			if (obj_rot[0] < -pi) {
				obj_rot[0] += pi2;
			}
		}
		
        if (paddata & SJOY_PS2_L_RIGHT) {
			obj_rot[1] += delta;
			if (obj_rot[1] > pi) {
				obj_rot[1] -= pi2;
			}
		}
		
        if (paddata & SJOY_PS2_L_LEFT) {
			obj_rot[1] -= delta;
			if (obj_rot[1] < -pi) {
				obj_rot[1] += pi2;
			}
		}
		
        if (paddata & SJOY_PS2_L1) {
			obj_rot[2] += delta;
			if (obj_rot[2] > pi) {
				obj_rot[2] -= pi2;
			}
		}
		
        if (paddata & SJOY_PS2_L2) {
			obj_rot[2] -= delta;
			if (obj_rot[2] < -pi) {
				obj_rot[2] += pi2;
			}
		}
		
        if (paddata & SJOY_PS2_R_DOWN) {
			camera_rot[0] += delta;
			if (camera_rot[0] > 0.4f * pi) {
				camera_rot[0] = 0.4f * pi;
			}
		}
		
        if (paddata & SJOY_PS2_R_UP) {
			camera_rot[0] -= delta;
			if (camera_rot[0] < -0.4f * pi) {
				camera_rot[0] = -0.4f * pi;
			}
		}
		
        if (paddata & SJOY_PS2_R_RIGHT) {
			camera_rot[1] += delta;
			if (camera_rot[1] > pi) {
				camera_rot[1] -= pi2;
			}
		}
		
        if (paddata & SJOY_PS2_R_LEFT) {
			camera_rot[1] -= delta;
			if (camera_rot[1] < -pi) {
				camera_rot[1] += pi2;
			}
		}
		
        if (paddata & SJOY_PS2_R1) {
			camera_p[2] -= delta * 5;
			if (camera_p[2] < -100) {
				camera_p[2] = -100;
			}
		}
		
        if (paddata & SJOY_PS2_R2) {
			camera_p[2] += delta * 5;
			if (camera_p[2] > -10) {
				camera_p[2] = -10;
			}
		}
		
        if (!toggle && (paddata & SJOY_PS2_SELECT)) {
			obj_switch ^= 1;
			toggle = 1;
        } else if (!(paddata & SJOY_PS2_SELECT)) {
			toggle = 0;
		}
		
		// set texture env
		texenv.size =
			 ps2_gs_set_texenv(&texenv.gs_tex, 0, &g_img, 8, 8, 0, 0, 0, 0, 1);
	
		PS2_GIFTAG_CLEAR_TAG(&texenv.giftag);
		texenv.giftag.NLOOP = texenv.size;
		texenv.giftag.EOP = 1;
		texenv.giftag.NREG = 1;
		texenv.giftag.REGS0 = PS2_GIFTAG_REGS_AD;
		
		*(__u64 *)&texenv.gs_tex.clamp1 =
			PS2_GS_SETREG_CLAMP(0, 0, 0, 0, 0, 0);
		
		ps2_dma_start_n(g_fd_gs, &texenv.giftag, texenv.size + 1);
		
        // local -> world (rotate)matrix
		ps2_vu0_unit_matrix(work);
		ps2_vu0_rot_matrix(local_world, work, obj_rot);
		
		// color&normal matrix setting
		ps2_vu0_normal_light_matrix(normal_light, light0, light1, light2);
		ps2_vu0_light_color_matrix(light_color, color0, color1, color2, ambient);
		
		// light(normal) -> local_light matrix
		ps2_vu0_mul_matrix(local_light, normal_light, local_world);
		
        // local -> world (rotate&translate)matrix
		ps2_vu0_trans_matrix(local_world, local_world, obj_trans);
		
        // world -> view matrix
        vu0_rot_camera_matrix(world_view, camera_p, camera_zd, camera_yd,
							  camera_rot);
		
		// local -> screen matrix
        ps2_vu0_mul_matrix(work, world_view, local_world);
        ps2_vu0_mul_matrix(local_screen, view_screen, work);
		
		for (i = 0; i < obj[obj_switch].blockNum; i++) {
			MakePacket(&obj[obj_switch], i);
			ps2_dma_start(g_fd_gs, vfd, (ps2_dmatag *)obj[obj_switch].pack[i].buf);
		}
		
		ps2_gs_vc_unlock();
		
        frame++;
		odev = !ps2_gs_sync_v(0);
    }
	
    ReleaseObj(&obj[0]);
    ReleaseObj(&obj[1]);
	
    return 0;
}
