/*-
 * Copyright (c) 1997, 1998, 1999 Nicolas Souchu
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD: src/sys/dev/ppbus/ppbconf.c,v 1.17.2.1 2000/05/24 00:20:57 n_hibma Exp $
 *
 */
#include "opt_ppbus.h"

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/device.h>
#include <sys/proc.h>

#include <dev/ppbus/ppbus_1284.h>
#include <dev/ppbus/ppbus_base.h>
#include <dev/ppbus/ppbus_conf.h>
#include <dev/ppbus/ppbus_device.h>
#include <dev/ppbus/ppbus_var.h>

/* Probe, attach, and detach functions for ppbus. */
static int ppbus_probe __P((struct device *, struct cfdata *, void *));
static void ppbus_attach __P((struct device *, struct device *, void *));
static int ppbus_detach __P((struct device *, int));

/* Utility function prototypes */
int ppbus_print_child(void *, const char *); 


CFATTACH_DECL(ppbus, sizeof(struct ppbus_softc), ppbus_probe, ppbus_attach,
	ppbus_detach, NULL);

/* Probe function for ppbus. */
static int
ppbus_probe(struct device * parent, struct cfdata * cf, void * aux)
{
	struct parport_adapter * sc_link = aux;

	/* Check adapter for consistency */ 
	if(
		sc_link->parport_io == NULL || 
		sc_link->parport_exec_microseq == NULL ||
		sc_link->parport_reset_epp_timeout == NULL ||
		sc_link->parport_setmode == NULL ||
		sc_link->parport_getmode == NULL ||
		sc_link->parport_ecp_sync == NULL ||
		sc_link->parport_read == NULL ||
		sc_link->parport_write == NULL ||
		sc_link->parport_read_ivar == NULL ||
		sc_link->parport_write_ivar == NULL ||
		((sc_link->capabilities & PPBUS_HAS_DMA) && 
		(sc_link->parport_dma_malloc == NULL ||
		sc_link->parport_dma_free == NULL))
		) {
		
#ifdef PPBUS_DEBUG
		printf("%s(%s): parport_adaptor is incomplete. Child device "
			"probe failed.\n", __func__, parent->dv_xname);
#endif
		return 0;
	}
	else {
		return 1;
	}
}

/* Attach function for ppbus. */
static void
ppbus_attach(struct device * parent, struct device * self, void * aux)
{
	struct ppbus_softc * ppbus = (struct ppbus_softc *) self;
	struct parport_adapter * sc_link = aux;
	struct ppbus_attach_args args;
	int val;

	printf("\n");

	/* Initialize config data from adapter (bus + device methods) */
	args.capabilities = sc_link->capabilities; 
	ppbus->ppbus_io = sc_link->parport_io;
	ppbus->ppbus_exec_microseq = sc_link->parport_exec_microseq;
	ppbus->ppbus_reset_epp_timeout = sc_link->
		parport_reset_epp_timeout;
	ppbus->ppbus_setmode = sc_link->parport_setmode;
	ppbus->ppbus_getmode = sc_link->parport_getmode;
	ppbus->ppbus_ecp_sync = sc_link->parport_ecp_sync;
	ppbus->ppbus_read = sc_link->parport_read;
	ppbus->ppbus_write = sc_link->parport_write;
        ppbus->ppbus_read_ivar = sc_link->parport_read_ivar;
	ppbus->ppbus_write_ivar = sc_link->parport_write_ivar;
	ppbus->ppbus_dma_malloc = sc_link->parport_dma_malloc;
	ppbus->ppbus_dma_free = sc_link->parport_dma_free;

	/* Initially there is no device owning the bus */
	ppbus->ppbus_owner = NULL;

	/* Initialize locking structures */
	lockinit(&(ppbus->sc_lock), PPBUSPRI | PCATCH, "ppbuslock", (3 * hz), 
		LK_SLEEPFAIL);

	/* Set up bus mode, ieee state, and dma usage */
	ppbus->sc_mode = ppbus->ppbus_getmode(self->dv_parent);
	ppbus->sc_use_ieee = 1;
	if(args.capabilities & PPBUS_HAS_DMA) {
		val = 1;
		if(ppbus_write_ivar(self, PPBUS_IVAR_DMA, &val) != 0) {
			printf(" <problem initializing dma usage state>");
		}
	}
	ppbus->sc_1284_state = PPBUS_FORWARD_IDLE;
	ppbus->sc_1284_error = PPBUS_NO_ERROR;

	/* Record device's sucessful attachment */
	ppbus->sc_dev_ok = PPBUS_OK;

#ifndef DONTPROBE_1284
	/* detect IEEE1284 compliant devices */
	if(ppbus_scan_bus(self)) {
		printf("%s: No IEEE1284 device found.\n", self->dv_xname);
	}
	else {
		printf("%s: IEEE1284 device found.\n", self->dv_xname);

		/* Detect device ID when interrupts are enabled */
		config_interrupts(self, ppbus_pnp_detect);
	}
#endif /* !DONTPROBE_1284 */

	/* Configure child devices */
	config_found_sm(self, &args, ppbus_print_child, NULL);

	return;
}

/* Detach function for ppbus. */
static int
ppbus_detach(struct device * self, int flag)
{
	struct ppbus_softc * ppbus = (struct ppbus_softc *) self;

	if(ppbus->sc_dev_ok != PPBUS_OK) {
		if(!(flag & DETACH_QUIET))
			printf("%s: detach called on unattached device.\n", 
				ppbus->sc_dev.dv_xname);
		if(!(flag & DETACH_FORCE))
			return 0;
		if(!(flag & DETACH_QUIET))
			printf("%s: continuing detach (DETACH_FORCE).\n",
				ppbus->sc_dev.dv_xname);
	}

	if(lockmgr(&(ppbus->sc_lock), LK_DRAIN, NULL)) {
		if(!(flag & DETACH_QUIET))
			printf("%s: error while waiting for lock activity to "
				"end.\n", ppbus->sc_dev.dv_xname);
		if(!(flag & DETACH_FORCE))
			return 0;
		if(!(flag & DETACH_QUIET))
			printf("%s: continuing detach (DETACH_FORCE).\n",
				ppbus->sc_dev.dv_xname);
	}

	/* XXX Detach children devices */
	
	if(!(flag & DETACH_QUIET))
		printf("%s: detached\n", ppbus->sc_dev.dv_xname);
	
	return 1;
}


/* Utility functions */

/* Used by config_found_sm() to print out result of child probe */
int 
ppbus_print_child(void * aux, const char * name)
{
	if(name != NULL) {
		printf("%s: child devices", name);
		return UNCONF;
	}
	else {
		return QUIET;
	}
}

