﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kinesisanalytics/KinesisAnalyticsRequest.h>
#include <aws/kinesisanalytics/KinesisAnalytics_EXPORTS.h>
#include <aws/kinesisanalytics/model/CloudWatchLoggingOption.h>

#include <utility>

namespace Aws {
namespace KinesisAnalytics {
namespace Model {

/**
 */
class AddApplicationCloudWatchLoggingOptionRequest : public KinesisAnalyticsRequest {
 public:
  AWS_KINESISANALYTICS_API AddApplicationCloudWatchLoggingOptionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AddApplicationCloudWatchLoggingOption"; }

  AWS_KINESISANALYTICS_API Aws::String SerializePayload() const override;

  AWS_KINESISANALYTICS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Kinesis Analytics application name.</p>
   */
  inline const Aws::String& GetApplicationName() const { return m_applicationName; }
  inline bool ApplicationNameHasBeenSet() const { return m_applicationNameHasBeenSet; }
  template <typename ApplicationNameT = Aws::String>
  void SetApplicationName(ApplicationNameT&& value) {
    m_applicationNameHasBeenSet = true;
    m_applicationName = std::forward<ApplicationNameT>(value);
  }
  template <typename ApplicationNameT = Aws::String>
  AddApplicationCloudWatchLoggingOptionRequest& WithApplicationName(ApplicationNameT&& value) {
    SetApplicationName(std::forward<ApplicationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version ID of the Kinesis Analytics application.</p>
   */
  inline long long GetCurrentApplicationVersionId() const { return m_currentApplicationVersionId; }
  inline bool CurrentApplicationVersionIdHasBeenSet() const { return m_currentApplicationVersionIdHasBeenSet; }
  inline void SetCurrentApplicationVersionId(long long value) {
    m_currentApplicationVersionIdHasBeenSet = true;
    m_currentApplicationVersionId = value;
  }
  inline AddApplicationCloudWatchLoggingOptionRequest& WithCurrentApplicationVersionId(long long value) {
    SetCurrentApplicationVersionId(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Provides the CloudWatch log stream Amazon Resource Name (ARN) and the IAM
   * role ARN. Note: To write application messages to CloudWatch, the IAM role that
   * is used must have the <code>PutLogEvents</code> policy action enabled.</p>
   */
  inline const CloudWatchLoggingOption& GetCloudWatchLoggingOption() const { return m_cloudWatchLoggingOption; }
  inline bool CloudWatchLoggingOptionHasBeenSet() const { return m_cloudWatchLoggingOptionHasBeenSet; }
  template <typename CloudWatchLoggingOptionT = CloudWatchLoggingOption>
  void SetCloudWatchLoggingOption(CloudWatchLoggingOptionT&& value) {
    m_cloudWatchLoggingOptionHasBeenSet = true;
    m_cloudWatchLoggingOption = std::forward<CloudWatchLoggingOptionT>(value);
  }
  template <typename CloudWatchLoggingOptionT = CloudWatchLoggingOption>
  AddApplicationCloudWatchLoggingOptionRequest& WithCloudWatchLoggingOption(CloudWatchLoggingOptionT&& value) {
    SetCloudWatchLoggingOption(std::forward<CloudWatchLoggingOptionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationName;

  long long m_currentApplicationVersionId{0};

  CloudWatchLoggingOption m_cloudWatchLoggingOption;
  bool m_applicationNameHasBeenSet = false;
  bool m_currentApplicationVersionIdHasBeenSet = false;
  bool m_cloudWatchLoggingOptionHasBeenSet = false;
};

}  // namespace Model
}  // namespace KinesisAnalytics
}  // namespace Aws
