/* Copyright (C) 1994 
            Olav Woelfelschneider (wosch@rbg.informatik.th-darmstadt.de)

 This library is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <X11/Xlib.h>
#include <stdio.h>
#include <string.h>
#include "McAlloc.h"
#include "McApp.h"
#include "McGadget.h"
#include "McSlider.h"

#define SLIDER ((McSlider *)(gadget->specialInfo))

static void McSliderUpdate(McGadget *gadget, int busy, int all);
static int McSliderEvent(McGadget *gadget, XEvent *event);
static void McSliderCleanup(McGadget *gadget);

McSpecialInfo *McCreateSlider(unsigned int flags, unsigned int maxValue) {
  McSlider *slider = (McSlider *) McAlloc(sizeof(McSlider));
  slider->flags = flags;
  slider->leftValue = slider->rightValue = maxValue>>1;
  slider->maxValue = maxValue;
  slider->specialInfo.updateProc = McSliderUpdate;
  slider->specialInfo.eventProc = McSliderEvent;
  slider->specialInfo.cleanupProc = McSliderCleanup;
  slider->step = 0;
  slider->width = slider->pixelWidth = 0;
  return (McSpecialInfo *)slider;
}

static void McSliderCleanup(McGadget *gadget) {
  McFree(gadget->specialInfo);
}

static void ReCalcProps(McGadget *gadget) {
  if (SLIDER->width) {
    int max=SLIDER->maxValue;

    if (SLIDER->flags & SLIDER_VERTICAL) {
      if (SLIDER->flags & SLIDER_VOLUME) {
	SLIDER->pixelWidth = 24;
      } else {
	if (SLIDER->width!=max) {
	  SLIDER->pixelWidth=
	    (SLIDER->width-max)*(gadget->height-(BW<<2))/SLIDER->width;
	} else {
	  SLIDER->pixelWidth = (gadget->height-(BW<<2)) / max;
	}
      }
    } else {
      if (SLIDER->width!=max) {
	SLIDER->pixelWidth=
	  (SLIDER->width-max)*(gadget->width-(BW<<2))/SLIDER->width;
      } else {
	SLIDER->pixelWidth = (gadget->width-(BW<<2)) / max;
      }
    }

    if (SLIDER->pixelWidth<5)
      SLIDER->pixelWidth=5;
  } else {
    SLIDER->pixelWidth = 0;
  }
}

void McSliderSetProps(McGadget *gadget, int max, int width) {
  if (max>1) { if (width<max) width=max; } else { width=1; }
  SLIDER->maxValue = max;
  SLIDER->width = width;
  ReCalcProps(gadget);
}

static void DrawVertical(McWindow *mcw, McGadget *gadget,
			 int pixels, int offs, int gwi) {
  int width = SLIDER->pixelWidth;
  McApp *app=mcw->app;

  if ((pixels-BW)>0) {
    XFillRectangle(app->display, mcw->window, app->gc[GC_CLEAR],
		   gadget->x+BW+offs, gadget->y-BW+gadget->height-pixels,
		   gwi - BW - BW, width?width:(pixels-BW));
    
    McAppDrawbox(mcw, 0,
		 gadget->x+BW+offs, gadget->y-BW+gadget->height-pixels,
		 gwi - BW - BW, width?width:(pixels-BW), _3D_OUT);
  }
  XFillRectangle(app->display, mcw->window, app->gc[GC_SELECTED],
		 gadget->x+offs, gadget->y+BW,
		 gwi, gadget->height-pixels-BW-BW-BW);
  
  if (width) {
    XFillRectangle(app->display, mcw->window, app->gc[GC_SELECTED],
		   gadget->x+offs, gadget->y+gadget->height-pixels+width,
		   gwi, pixels-width);
  }

  if (SLIDER->flags & SLIDER_VOLUME) {
    int i, y;
    XSegment segments[5];

    y=gadget->y-BW+gadget->height-pixels+(width>>1)-1;

    segments[0].x1=gadget->x+offs+BW+1;
    segments[0].x2=gadget->x+offs+gwi-BW-2;
    segments[0].y1=segments[0].y2=y;

    for (i=1; i<5; i++) {
      segments[i].x1=segments[0].x1+1;
      segments[i].x2=segments[0].x2-1;
    }

    segments[1].y1=segments[1].y2=segments[0].y1-4;
    segments[2].y1=segments[2].y2=segments[0].y1-8;
    segments[3].y1=segments[3].y2=segments[0].y1+4;
    segments[4].y1=segments[4].y2=segments[0].y1+8;

    XDrawSegments(app->display, mcw->window, app->gc[GC_DARK],
		  segments, 5);

    for (i=0; i<5; i++) { segments[i].y1++; segments[i].y2++; }
    XDrawSegments(app->display, mcw->window, app->gc[GC_BRIGHT],
		  segments, 5);



    segments[0].x1=segments[0].x2=segments[1].x1=segments[1].x2=
      gadget->x+offs+(gwi>>1)-1;
    segments[0].y1=gadget->y+1;
    segments[0].y2=gadget->y-BW-BW-1+gadget->height-pixels;
    segments[1].y1=gadget->y+gadget->height-pixels+width;
    segments[1].y2=gadget->y+gadget->height-2;

    XDrawSegments(app->display, mcw->window, app->gc[GC_DARK],
		  segments, 2);
    for (i=0; i<2; i++) { segments[i].x1++; segments[i].x2++; }
    XDrawSegments(app->display, mcw->window, app->gc[GC_BRIGHT],
		  segments, 2);
  }
}


static void DrawHorizontal(McWindow *mcw, McGadget *gadget,
			   int pixels, int offs, int gwi) {
  int width = SLIDER->pixelWidth;
  McApp *app=mcw->app;
  
  if ((pixels-BW)>0) {
    XClearArea(app->display, mcw->window,
	       gadget->x+BW+BW+(width?(pixels-width-BW):0),
	       gadget->y+BW+offs,
	       width?width:(pixels - BW), gwi-BW-BW, False);

    McAppDrawbox(mcw, mcw->window,
		 gadget->x+BW+BW+(width?(pixels-width-BW):0),
		 gadget->y+BW+offs,
		 width?width:(pixels - BW),
		 gwi-BW-BW, _3D_OUT);
  }

  XFillRectangle(app->display, mcw->window, app->gc[GC_SELECTED],
		 gadget->x+BW+BW+pixels,
		 gadget->y+offs,
		 gadget->width-pixels-BW-BW-BW,
		 gwi);
  if (width)
    XFillRectangle(app->display, mcw->window, app->gc[GC_SELECTED],
		   gadget->x + BW, gadget->y + offs,
		   pixels-width-BW,gwi);

}

static void McSliderUpdate(McGadget *gadget, int busy,int all) {
  int leftPixels;
  int rightPixels;
  int width, sub;
  int max;
  McWindow *mcw=gadget->mcw;

  if (all)  ReCalcProps(gadget);
  if ((width = SLIDER->pixelWidth)) width+=BW;
  if ((max=SLIDER->maxValue)<1) max=1;
  sub = BW + BW + BW + width;

  leftPixels=SLIDER->leftValue;
  if (leftPixels>SLIDER->maxValue)
    leftPixels=SLIDER->leftValue = SLIDER->maxValue;
  if (leftPixels<0) leftPixels=SLIDER->leftValue = 0;

  rightPixels=SLIDER->rightValue;
  if (rightPixels>SLIDER->maxValue)
    rightPixels=SLIDER->rightValue = SLIDER->maxValue;
  if (rightPixels<0) rightPixels=SLIDER->rightValue = 0;


  if (SLIDER->flags & SLIDER_VERTICAL) {
    /************************/
    /* Update it vertically */
    /************************/
    leftPixels  = leftPixels  * (gadget->height-sub) / max + width;

    if (SLIDER->flags & SLIDER_STEREO) {
      rightPixels = rightPixels * (gadget->height-sub) / max + width;
      DrawVertical(mcw, gadget,  leftPixels, BW,
		   (gadget->width>>1) - BW);
      DrawVertical(mcw, gadget, rightPixels, 1 + gadget->width/2,
		   (gadget->width>>1) - BW);
    } else {
      DrawVertical(mcw, gadget,  leftPixels, BW, gadget->width - BW - BW);
    }
  } else {
    /**************************/
    /* Update it horizontally */
    /**************************/
    leftPixels  = leftPixels  * (gadget->width-sub) / max + width;

    if (SLIDER->flags & SLIDER_STEREO) {
      rightPixels = rightPixels * (gadget->width-sub) / max + width;
      DrawHorizontal(mcw, gadget,  leftPixels, BW,
		   (gadget->height>>1) - BW);
      DrawHorizontal(mcw, gadget, rightPixels, 1 + gadget->height/2,
		   (gadget->height>>1) - BW);
    } else {
      DrawHorizontal(mcw, gadget,  leftPixels, BW, gadget->height - BW - BW);
    }
  }
}

static int CalcValue(McGadget *gadget, int mv, int x, int y) {
  int tl, tr, tx, changed;
  int width = SLIDER->pixelWidth;
  changed = 0;

  if (SLIDER->flags & SLIDER_VERTICAL) {
    tl = tr = (gadget->y + gadget->height - y - (width>>1)) *
      (SLIDER->maxValue) / (gadget->height - width - (BW*3));
  } else {
    tl = tr = (x - gadget->x - BW - (width>>1)) *
      (SLIDER->maxValue) / (gadget->width - width - (BW*3));
  }

  if (!mv) {
    tx=tl;
    if (!(tr=SLIDER->step)) tr=1;
    if (tx>SLIDER->leftValue) tl=SLIDER->leftValue+tr;
    else tl=SLIDER->leftValue-tr;
    if (tx>SLIDER->rightValue) tr=SLIDER->rightValue+tr;
    else tr=SLIDER->rightValue-tr;
  }

  if (tl < 0) tl = 0;
  else if (tl > SLIDER->maxValue) tl = SLIDER->maxValue;

  if (tr < 0) tr = 0;
  else if (tr > SLIDER->maxValue) tr = SLIDER->maxValue;

  if (SLIDER->flags & SLIDER_SET_LEFT) {
    if (tl!=SLIDER->leftValue) {
      SLIDER->leftValue=tl;
      changed=1;
    }
  }
  if (SLIDER->flags & SLIDER_SET_RIGHT) {
    if (tr!=SLIDER->rightValue) {
      SLIDER->rightValue=tr;
      changed=1;
    }
  }
  return changed;
}

static int McSliderEvent(McGadget *gadget, XEvent *event) {
  int i,j,x,y;
  McWindow *mcw;

  if (event->type == Expose) return 0;

  mcw=gadget->mcw;

  if (gadget->flags&GAD_PRESSED) {
    switch (event->type) {
    case ButtonRelease:
      gadget->flags&=~GAD_PRESSED;
      SLIDER->flags&=~SLIDER_SET_MASK;
      if (McInRectangle(event->xbutton.x, event->xbutton.y,
			gadget->x, gadget->y, gadget->width, gadget->height)) {
	if (gadget->callbackUp) {
	  (*gadget->callbackUp)(gadget);
	}
      }
      return 1;

    case MotionNotify:
      if ((x=event->xmotion.x)<0) x=0;
      if ((y=event->xmotion.y)<0) y=0;
      if (CalcValue(gadget, 1, x, y)) {
	McSliderUpdate(gadget, 0, 0);
	if (gadget->callbackDown) {
#ifdef FLUSH
	  XFlush(mcw->app->display);
#endif
	  (*gadget->callbackDown)(gadget);
	}
      }
      return 1;
    case ButtonPress:
      return 1;
    default:
      return 0;
    }
  } else { /* not currently pressed */
    switch (event->type) {
    case ButtonPress:
      if (McInRectangle(event->xbutton.x, event->xbutton.y,
			gadget->x, gadget->y, gadget->width, gadget->height)) {
	if (SLIDER->flags & SLIDER_STEREO) {
	  SLIDER->flags&=~SLIDER_SET_MASK;
	  if (SLIDER->flags & SLIDER_VERTICAL) {
	    i = (event->xbutton.x - gadget->x);
	    j = gadget->width;
	  } else {
	    i = (event->xbutton.y - gadget->y);
	    j = gadget->height;
	  }
	  if (i < (j/3))
	    SLIDER->flags|=SLIDER_SET_LEFT;
	  else
	    if (i > (j/3*2))
	      SLIDER->flags|=SLIDER_SET_RIGHT;
	    else
	      SLIDER->flags|=SLIDER_SET_LEFT | SLIDER_SET_RIGHT;
	} else {
	  SLIDER->flags|=SLIDER_SET_LEFT | SLIDER_SET_RIGHT;
	}
	switch (event->xbutton.button) {
	case 1:
	  CalcValue(gadget, 0, event->xbutton.x, event->xbutton.y);
	  McSliderUpdate(gadget, 0, 0);
	  if (gadget->callbackDown) {
#ifdef FLUSH
	    XFlush(mcw->app->display);
#endif
	    (*gadget->callbackDown)(gadget);
	  }
	  if (gadget->callbackUp) {
#ifdef FLUSH
	    XFlush(mcw->app->display);
#endif
	    (*gadget->callbackUp)(gadget);
	  }
	  gadget->flags&=~GAD_PRESSED;
	  McMoveGadgetToStart(gadget);
	  return 1;
	case 2:
	  CalcValue(gadget, 1, event->xbutton.x, event->xbutton.y);
	  McSliderUpdate(gadget, 0, 0);
	  if (gadget->callbackDown) {
#ifdef FLUSH
	    XFlush(app->display);
#endif
	    (*gadget->callbackDown)(gadget);
	  }
	  gadget->flags|=GAD_PRESSED;
	  McMoveGadgetToStart(gadget);
	  return 1;
	}
      } else {
	return 0;
      }
    default:
      return 0;
    }
  }
}

/***************************************************************************/

McGadget *MakeProp(McWindow *mcw, int x, int y, int w, int h, int id,
		   void (*callback)(struct McGadget *)) {
  McGadget *gadget;

  if (x<0) x=mcw->w+x-w;
  if (y<0) x=mcw->h+y-h;

  gadget = McCreateGadget(mcw, GAD_H3D|GAD_3D|GAD_SELECTED|GAD_ACTIVE,
			  SLIDERGADGET, x, y, w, h);
  gadget->specialInfo = McCreateSlider(SLIDER_MONO|SLIDER_VERTICAL, 1);
  McSliderSetProps(gadget, 1, 0);
  gadget->callbackUp = gadget->callbackDown = callback;
  gadget->id = id;
  return gadget;
}

