/* AX.25 Utilities: Connect two file descriptors to each other.
 * Bruce Perens, November 1994
 *
 * Copyright 1994 Bruce Perens.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/socket.h>

#include "global.h"

/*
 * Data structure used to bridge two files together.
 */
struct _BridgeFile {
	int			fd;
	void			(*function)(int fd);
	int			readyForWrite:1;
	int			inputWaiting:1;
	struct _BridgeFile *	peer;
	int			length;
	char			buffer[192];
};
typedef struct _BridgeFile	BridgeFile;

static BridgeFile *	files[OPEN_MAX] = { 0 };
static int		highestFileIndex = -1;

static void	inputWaiting(int fd);

static int
addFile(int fd)
{
	BridgeFile *	f = (BridgeFile *)malloc(sizeof(BridgeFile));

	if ( f == 0 )
		return 0;

	if ( files[fd] != 0 )
		abort();

	memset((void *)f, 0, sizeof(*f));

	f->fd = fd;
	files[fd] = f;

	if ( fd > highestFileIndex )
		highestFileIndex = fd;

	fcntl(fd, F_SETFL, O_NONBLOCK);

	return 1;
}

int
bridge_add_special_file(int fd, void (*function)(int fd))
{
	if ( !addFile(fd) )
		return 0;
	files[fd]->function = function;
	return 1;
}

void
bridge_remove_file(int fd)
{
	BridgeFile *	f;
	BridgeFile *	peer;

	if ( (f = files[fd]) == 0 )
		return;
		
	files[fd] = 0;

	peer = f->peer;
	f->peer = 0;
	if ( peer != 0 ) {
		peer->peer = 0;
		bridge_remove_file(peer->fd);
	}
	shutdown(fd, 2);
	close(fd);

	if ( fd == highestFileIndex ) {
		int i = OPEN_MAX;
		highestFileIndex = -1;
		do {
			if ( files[--i] != 0 ) {
				highestFileIndex = i;
				break;
			}
		} while ( i > 0 );
	}
	free(f);
}

static void
readyForWrite(int fd)
{
	BridgeFile *	f = files[fd];
	BridgeFile *	peer;

	if ( f == 0 )
		return;

	f->readyForWrite = 0;

	if ( (peer = f->peer) != 0 && peer->inputWaiting )
		inputWaiting(peer->fd);

	if ( f->length > 0 ) {
		int	status = write(f->fd, f->buffer, f->length);

		if ( status > 0 ) {
			f->length -= status;
			if ( f->length > 0 )
				memmove(
				 f->buffer
				,&f->buffer[status]
				,f->length);
		}
		else {
			if ( errno != EAGAIN ) {
				/*
				 * If there's been a write error, assume
				 * I won't be able to write again. However,
				 * I want to keep this file around because
				 * there may still be something to read from
				 * it. I'll flush anything that had been
				 * queued for write (to prevent deadlock),
				 * and tell the other side I'm not
				 * able to take any more input.
				 */
				f->length = 0;
				if ( f->peer != 0 ) {
					shutdown(f->fd, 1);
					shutdown(peer->fd, 0);
					f->peer->peer = 0;
				}
			}
		}
	}
	else
		f->readyForWrite = 1;

	/*
	 * If the peer's gone and I've finished writing,
	 * it's time to close.
	 */
	if ( f->length == 0 && f->peer == 0 )
		bridge_remove_file(f->fd);
}

static void
inputWaiting(int fd)
{
	BridgeFile *	f = files[fd];
	BridgeFile *	peer;
	int		space;

	if ( f == 0 )
		return;

	if ( f->function != 0 )
		(*f->function)(fd);

	f->inputWaiting = 0;

	if ( (peer = f->peer) == 0 )
		return;

	space = sizeof(f->buffer) - f->length;

	if ( space > 0 ) {
		int	status = read(
			 f->fd
			,&peer->buffer[peer->length]
			,space);

		if ( status > 0 ) {
			peer->length += status;
			if ( peer->readyForWrite )
				readyForWrite(peer->fd);
		}
		else if ( errno != EAGAIN ) {
			if ( peer->length != 0 ) {
				/*
				 * I'm about to delete this file object
				 * because I detected a close, but I want
				 * the peer object to still have a chance
				 * to write. Thus, I divorce from the peer
				 * before deleting.
				 */
				peer->peer = 0;
				f->peer = 0;
			}
			bridge_remove_file(f->fd);
			return;
		}
	}
	else
		f->inputWaiting = 1;
}

int
bridge_add_file_pair(int a, int b)
{
	if ( !addFile(a) )
		return 0;
	if ( !addFile(b) ) {
		bridge_remove_file(a);
		return 0;
	}

	files[a]->peer = files[b];
	files[b]->peer = files[a];

	return 1;
}

int
bridge_run()
{
	int	i;
	int	status;

	fd_set	readFDs;
	fd_set	writeFDs;
	fd_set	exceptionFDs;
	struct timeval	tv;

	if ( highestFileIndex < 0 )
		return 0;

	timerclear(&tv);
	tv.tv_sec = 60 * 60;

	FD_ZERO(&readFDs);
	FD_ZERO(&writeFDs);
	FD_ZERO(&exceptionFDs);

	for ( i = 0; i <= highestFileIndex; i++ ) {
		BridgeFile *	f = files[i];

		if ( f == 0 )
			continue;

		FD_SET(i, &exceptionFDs);

		if ( f->function != 0 ) {
			FD_SET(i, &readFDs);
		}
		else {
			/*
			 * Only once a file is ready for write will I check
			 * if its peer is ready for read.
			 */
			if ( !f->readyForWrite ) {
				FD_SET(i, &writeFDs);
			}
			else if ( f->length != sizeof(f->buffer)
			 && f->peer != 0 && !f->peer->inputWaiting ) {
				FD_SET(i, &readFDs);
			}
		}
	}

	status = select(
	 highestFileIndex + 1
	,&readFDs
	,&writeFDs
	,&exceptionFDs
	,&tv);
	
	if ( status > 0 ) {
		for ( i = 0; i <= highestFileIndex; i++ ) {
			if ( FD_ISSET(i, &writeFDs) )
				readyForWrite(i);
			if ( FD_ISSET(i, &readFDs) || FD_ISSET(i, &exceptionFDs) )
				inputWaiting(i);
		}
	}

	return 1;
}
