#include "wily.h"
#include "view.h"

static void
rfill(Rectangle r, Fcode f) {
	r = inset(r,3);
	bitblt(&screen, r.min, &screen, r, f);
}

static Rectangle
resizebox(Rectangle r) {
	r.max.x = r.min.x + SCROLLWIDTH;
	r.max.y = r.min.y + SCROLLWIDTH;
	return r;
}

void
view_fillbutton(View*v, Fcode f){
	assert(v);
	rfill(resizebox(v->r), f);
}

static void
button_set(View *v)
{
	Rectangle	r;

	assert(ISTAG(v));	/* we're a tag */
	r = resizebox(v->r);
	border(&screen, r, 2, F);

	if( tile_isdirty(v->tile))
		rfill(r, F);
}

/*
PRE: v->visible.p0 is correct.  The text which is currently
displayed in the frame is correct, but there might not be
enough.

POST:  The frame is displaying everything it should,
and v->visible.p1 is set.
*/
void
fill(View *v)
{
	Frame	*f = &v->f;
	ulong	p; 	/* text position of last visible rune */
	Rune	buf[CHUNK];
	int		n;
	Text		*t = v->t;
	int		bw;	/* borderwidth */

	/* view_invariants may not hold at this point */

	if(!f->b)
		return;

	/* Stuff runes on the end until we exhaust the text or fill the frame */
	p = v->visible.p0 + f->nchars;
	while (!frame_isfull(f) && (n = text_ncopy(t, buf, p, CHUNK)) ) {
		frinsert(f, buf, buf+n, f->nchars);
		p += n;
	}

	v->visible.p1 = v->visible.p0 + f->nchars;

	if(v->scroll)
		scroll_set(v->scroll,  v->visible.p0,
				f->nchars, text_length(v->t)
				);
	else
		button_set(v);

	bw = (v==last_selection)? SELECTEDBORDER : 1;
	border(&screen, v->r, bw, F);

	/* Ensure that if v->sel is in the visible area, it is selected */
	frselectp(f, F&~D);
	f->p0 = clip(v->sel.p0 - v->visible.p0, 0, f->nchars);
	f->p1 = clip(v->sel.p1 - v->visible.p0, 0, f->nchars);
	frselectp(f, F&~D);

	assert(view_invariants(v));
}

/* Set the rectangles for 'v', v's frame and v's scrollbar.
 * Assumes that 'r' is already correct.
 * If we can be displayed, set up the frame and scrollbar.
 */
static void
setrects(View*v, Rectangle r)
{
	Frame	*f = &v->f;
	Font		*ft = f->font;
	Rectangle	scrollr, framer;
	Bitmap	*b;

	assert(Dx(r) >= MINWIDTH);	/* Or our tile is bizarre */

	scrollr = framer = r;
	framer.min.x += SCROLLWIDTH;
	scrollr.max.x = framer.min.x;
	framer = inset(framer, INSET);
	scrollr.min.y += INSET;

	/* If 'r' is too small, we're hidden: use null bitmap */ 
	b = Dy(r) < tagheight ? 0 : &screen;

	v->r = r;
	scroll_setrects(v->scroll, b, scrollr);
	frclear(f);
	frinit(f, framer, ft, b);
}

int
snapheight(View*v, int h)
{
	int	lines;
	int	fh = v->f.font->height;
	int	border = 2*INSET;
	
	lines = v->scroll? (h - border)/ fh : 1;
	return lines ? border + lines * fh : 0;
}

static Rectangle
snap(View*v, Rectangle r)
{
	r.max.y = r.min.y + snapheight(v, Dy(r));
	return  r;
}

/* Try to redraw 'v' inside 'r' */
void
view_reshaped(View*v, Rectangle r)
{
	assert(view_invariants(v));

	r = snap(v,r);
	setrects(v, r);
	if (ISVISIBLE(v)) {
		if(text_refreshdir(v->t)){
			v->visible.p0 = v->sel.p0 = v->sel.p1 = 0;
			frdelete(&v->f, 0, v->f.nchars);
		}
		fill(v);
	}
}

/* Return point just after the last line in 'v' */
int
view_lastlinepos(View*v)
{
	Frame	*f = &v->f;
	Point	p = frptofchar(f, f->nchars);
	int	y =  p.y + f->font->height;

	return MIN(y, f->r.max.y);
}

/* Return the amount by which 'v' could be squeezed */
int
view_stripwhitespace(View*v)
{
	Frame	*f;
	int		blanklines;

	if(v && ISVISIBLE(v)) {
		f = &v->f;
		assert(Dy(v->r) >= f->maxlines * f->font->height);
		blanklines = f->maxlines - f->nlines;
		if(blanklines > 0) {
			return blanklines * f->font->height;
		}
	}
	return 0;
}

