#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <pwd.h>
#include "wily.h"
#include "view.h"

static const char *	nextstr (const char *p, const char *c, int *n);
static char *	pathfind (const char *, const char *, const char *);
static int		getkey(char * dest, char *orig);

/* Expand '~' or '$var' at the start of 'dest', store in 'orig' */
void
envexpand(char*dest, char*orig)
{
	char c = orig[0];
	
	if (c == '$' || c == '~') {
		int	n;
		Path	key;
		char *val = 0;
		
		n = getkey(key, orig+1);
		if(c == '$') {
			val = getenv(key); 
		} else {
			struct passwd *pw;
			
			pw = n? getpwnam(key) : getpwuid(getuid());
			val = pw? pw->pw_dir : 0;
		}
		if(val){
			sprintf(dest, "%s%s", val, orig +n + 1);
			return;
		}
	}
	strcpy(dest, orig);
}

/*
 * Start with 'orig'.
 *
 * Resolve environment variables, ~ expansion, context,
 * store in dest
 */
void
pathexpand(char*orig, View*v, char *dest)
{
	Path	tmp,context;

	switch(orig[0]) {
	case '$':
	case '~':
		envexpand(tmp, orig);
		break;
	case '/':
		strcpy(tmp, orig);
		break;
	default:
		data_context(view_data(v), context);
		sprintf(tmp, "%s%s", context, orig);
		break;
	}
	realpath ( tmp, dest);
}

/* If 'path' is the name of a file, fill 'v' and 'r' with the view of
 * that file and the selection, and return true.
 * If 'path' is not the name of a file, return false, and leave
 * 'v' and 'r' alone.
 */
static Bool
foundpath(char *path, View **v, Range *r) {
	View *found;
	
	if( (found=data_find(path)) || (found=data_open(path,false)) ) {
		*v = found;
		*r = found->sel;
	}
	return found != 0;
}

/* More general, use $INCLUDES, only on files in "" or <> */
static Bool
realfind(View **v, Range *r, char *cbuf) {
	int	last;
	char	*s;
	Path	pbuf;
	
	last = strlen(cbuf) -1;
	
	if (! ((cbuf[0]=='"' && cbuf[last]=='"') || (cbuf[0]=='<' && cbuf[last]=='>') )) 
		return false;

	cbuf[last]='\0';
	s = pathfind(getenv("INCLUDES"),  cbuf+1, "rf");
	if(!s) {
		sprintf(pbuf, "/usr/include/%s", cbuf+1);
		s = pbuf;
	}
	return s? foundpath(s, v, r): false;
}

/*
 * The user has selected '*r' in '*v', which has been expanded to 's'
 * If possible, open an appropriate include file, and fill in *v and *r
 * and return true.
 *
 * If no include file is appropriate, return false.
 */
Bool
language_specific_expand(View **v, Range *r)
{
	Range	expanded;
	int		len;
	char		*cbuf;
	Bool	found;
	
	expanded = text_expand((*v)->t, *r, notinclude);
	len = RLEN(expanded);
	if( len > MAXPATH || len < 2)
		return false;
	cbuf = text_duputf((*v)->t, expanded);

	found = realfind(v,r,cbuf);
	free(cbuf);
	return found;
	
}

/* mode ignored */

static char *
pathfind (const char *paths, const char *file, const char *mode)
{
	const char *p,*ptmp;
	int flen;
	int plen;

	if (!paths || !file)
		return 0;

	flen = strlen(file);
	p = paths;
	while((ptmp = nextstr(p, ":", &plen))!= 0) {
		int fd;
		char *tmp = malloc(plen+1+flen+1);

		if (tmp) {
			sprintf(tmp, "%.*s/%s", plen, p, file);
			if ((fd = open(tmp, 0)) < 0) {
				free(tmp);
			} else {
				close(fd);
				return tmp;
			}
		}
		p = ptmp;
	}
	return 0;
}

static const char *
nextstr (const char *p, const char *c, int *n)
{
	int i;

	if (!p || !*p)
		return 0;

	*n = i = strcspn (p, c);	/* XXX - utf */
	if (p[i])
		i += 1;					/* strspn (p+i, c); ? */
	return p+i;
}

/*
 * Copy characters from 'orig' to 'dest' until reaching one which isn't
 * an alnum.  Return the number of characters copied.
 */
static int
getkey(char * dest, char *orig)
{
	int n;
	n = strspn(orig, alnums);
	strncpy(dest, orig, n);
	dest[n] = 0;
	return n;
}

