/**
 *
 * $Id: Label.c,v 1.40 1996/05/02 07:10:59 u27113 Exp $
 *
 * Copyright (C) 1995 Free Software Foundation, Inc.
 *
 * This file is part of the GNU LessTif Library.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **/

static char rcsid[] = "$Id: Label.c,v 1.40 1996/05/02 07:10:59 u27113 Exp $";

#include <LTconfig.h>
#include <Xm/XmP.h>
#include <Xm/DebugUtil.h>
#include <Xm/AtomMgr.h>
#include <Xm/DragC.h>
#include <Xm/LabelP.h>
#include <Xm/LabelGP.h>
#include <Xm/MenuShell.h>
#include <Xm/MenuUtilP.h>
#include <Xm/RowColumnP.h>
#include <Xm/TransltnsP.h>
#include <Xm/XmosP.h>	
#include <stdio.h>
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

/* Forward Declarations */

static void class_initialize();
static void class_part_initialize(WidgetClass class);
static void initialize(Widget request, Widget new, ArgList args, Cardinal *num_args);
static void destroy(Widget w);
static void resize(Widget w);
static void expose(Widget w, XEvent *event, Region region);
static XtGeometryResult query_geometry(Widget w, XtWidgetGeometry *proposed, XtWidgetGeometry *answer);
static Boolean set_values(Widget current, Widget request, Widget new, ArgList args, Cardinal *num_args);
static void initialize_prehook(Widget request, Widget new, ArgList args, Cardinal *num_args);
static void initialize_posthook(Widget request, Widget new, ArgList args, Cardinal *num_args);

void _XmLabelRecomputeSize(Widget w);
void _XmLabelGetPixmapSize(Widget w,Pixmap Pix, Dimension *width, Dimension *height);
void _XmLabelDimensions(Widget w, Dimension *wd, Dimension *ht);

void _XmLabelRecalcDimensions(Widget request, Widget new);

static void label_menu_procs(int function, Widget menushell_parent, ...);
Boolean _XmLabelShowsAccelerators(Widget w);
Boolean _XmLabelShowsMnemonic(Widget w);

/* prototypes for drag-drop */

static Boolean drag_convert_proc(Widget w, Atom *selection, Atom *target, Atom *type_return, XtPointer *value_return, unsigned long *length_return, int *format_return);
static void drag_drop_finish(Widget w, XtPointer client_data, XtPointer call_data);

/*
 * Resources for the label class
 */
#define Offset(field) XtOffsetOf(XmLabelRec, label.field)
static XtResource resources[] = {
    {
	XmNshadowThickness, XmCShadowThickness, XmRHorizontalDimension,
	sizeof(Dimension), XtOffsetOf(XmLabelRec, primitive.shadow_thickness),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNalignment, XmCAlignment, XmRAlignment,
	sizeof(unsigned char), Offset(alignment),
	XmRImmediate, (XtPointer)XmALIGNMENT_CENTER
    },
    {
	XmNlabelType, XmCLabelType, XmRLabelType,
	sizeof(unsigned char), Offset(label_type),
	XmRImmediate, (XtPointer)XmSTRING
    },
    {
	XmNmarginWidth, XmCMarginWidth, XmRHorizontalDimension,
	sizeof(Dimension), Offset(margin_width),
	XmRImmediate, (XtPointer)2
    },
    {
	XmNmarginHeight, XmCMarginHeight, XmRVerticalDimension,
	sizeof(Dimension), Offset(margin_height),
	XmRImmediate, (XtPointer)2
    },
    {
	XmNmarginLeft, XmCMarginLeft, XmRHorizontalDimension,
	sizeof(Dimension), Offset(margin_left),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNmarginRight, XmCMarginRight, XmRHorizontalDimension,
	sizeof(Dimension), Offset(margin_right),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNmarginTop, XmCMarginTop, XmRVerticalDimension,
	sizeof(Dimension), Offset(margin_top),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNmarginBottom, XmCMarginBottom, XmRVerticalDimension,
	sizeof(Dimension), Offset(margin_bottom),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNfontList, XmCFontList, XmRFontList,
	sizeof(XmFontList), Offset(font),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNlabelPixmap, XmCLabelPixmap, XmRPrimForegroundPixmap,
	sizeof(Pixmap), Offset(pixmap),
	XmRImmediate, (XtPointer)XmUNSPECIFIED_PIXMAP
    },
    {
	XmNlabelInsensitivePixmap, XmCLabelInsensitivePixmap, XmRPrimForegroundPixmap,
	sizeof(Pixmap), Offset(pixmap_insen),
	XmRImmediate, (XtPointer)XmUNSPECIFIED_PIXMAP
    },
    {
	XmNlabelString, XmCXmString, XmRXmString,
	sizeof(XmString), Offset(_label),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNmnemonic, XmCMnemonic, XmRKeySym,
	sizeof(KeySym), Offset(mnemonic),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNmnemonicCharSet, XmCMnemonicCharSet, XmRString,
	sizeof(String), Offset(mnemonicCharset),
	XmRImmediate, (XtPointer)XmFONTLIST_DEFAULT_TAG
    },
    {
	XmNaccelerator, XmCAccelerator, XmRString,
	sizeof(String), Offset(accelerator),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNacceleratorText, XmCAcceleratorText, XmRXmString,
	sizeof(XmString), Offset(_acc_text),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNrecomputeSize, XmCRecomputeSize, XmRBoolean,
	sizeof(Boolean), Offset(recompute_size),
	XmRImmediate, (XtPointer)True
    },
    {
	XmNstringDirection, XmCStringDirection, XmRStringDirection,
	sizeof(XmStringDirection), Offset(string_direction),
	XmRImmediate, (XtPointer)((XmStringDirection)XmUNSPECIFIED)
    },
    /* resources we override from XmPrimitive */
    {
	XmNtraversalOn, XmCTraversalOn, XmRBoolean,
	sizeof(Boolean), XtOffsetOf(XmLabelRec, primitive.traversal_on),
	XmRImmediate, (XtPointer)False
    },
    {
	XmNhighlightThickness, XmCHighlightThickness, XmRHorizontalDimension,
	sizeof(Dimension), XtOffsetOf(XmLabelRec, primitive.highlight_thickness),
	XmRImmediate, (XtPointer)0
    }
};

/* Synthetic Resources for the Label Widget */

static XmSyntheticResource syn_resources[] = {
    {
	XmNmarginWidth,
	sizeof(Dimension), Offset(margin_width),
	_XmFromHorizontalPixels, _XmToHorizontalPixels
    },
    {
	XmNmarginHeight,
	sizeof(Dimension), Offset(margin_height),
	_XmFromVerticalPixels, _XmToVerticalPixels
    },
    {
	XmNmarginLeft,
	sizeof(Dimension), Offset(margin_left),
	_XmFromHorizontalPixels, _XmToHorizontalPixels
    },
    {
	XmNmarginRight,
	sizeof(Dimension), Offset(margin_right),
	_XmFromHorizontalPixels, _XmToHorizontalPixels
    },
    {
	XmNmarginTop,
	sizeof(Dimension), Offset(margin_top),
	_XmFromVerticalPixels, _XmToVerticalPixels
    },
    {
	XmNmarginBottom,
	sizeof(Dimension), Offset(margin_bottom),
	_XmFromVerticalPixels, _XmToVerticalPixels
    },
    {
	XmNlabelString,
	sizeof(XmString), Offset(_label),
	NULL /* FIXME */, NULL
    },
    {
	XmNaccelerator,
	sizeof(XmString), Offset(_acc_text),
	NULL /* FIXME */, NULL
    },
    {
	XmNacceleratorText,
	sizeof(String), Offset(accelerator),
	NULL /* FIXME */, NULL
    },
    {
	XmNmnemonicCharSet,
	sizeof(String), Offset(mnemonicCharset),
	NULL /* FIXME */, NULL
    }
};

static void ProcessDrag(Widget w, XEvent *event, String *params, Cardinal *num_params);
static void Help(Widget w, XEvent *event, String *params, Cardinal *num_params);
static void LabelFocusIn(Widget w, XEvent *event, String *params, Cardinal *num_params);
static void LabelFocusOut(Widget w, XEvent *event, String *params, Cardinal *num_params);
static void LabelUnmap(Widget w, XEvent *event, String *params, Cardinal *num_params);

char _XmLabel_defaultTranslations[] = 
   "<EnterWindow>:        Enter()\n\
    <LeaveWindow>:        Leave()\n\
    <Btn2Down>:           ProcessDrag()\n\
    <Key>osfActivate:     PrimitiveParentActivate()\n\
    <Key>osfCancel:       PrimitiveParentCancel()\n\
    <Key>osfHelp:         Help()\n\
    ~s ~m ~a <Key>Return: PrimitiveParentActivate()";

/*
 * MLM: FIXME -- these need to go somewhere; without them, misc/test1.c won't
 * compile.  Perhaps they are installed by default for Labels in menus, and
 * overridden for subclasses, like PushB and ToggleB?
 */
char _XmLabel_menuTranslations[] =
    "<EnterWindow>: Enter() \n\
     <LeaveWindow>: Leave() \n\
     :<Key>osfHelp: Help()";

char _XmLabel_menu_traversal_events[] =  /* fix me -- add these translations to something -- chris*/
   "<Unmap>:              Unmap()\n\
    <FocusOut>:           FocusOut()\n\
    <FocusIn>:            FocusIn()\n\
    <Key>osfCancel:       MenuEscape()\n\
    <Key>osfLeft:         MenuTraverseLeft()\n\
    <Key>osfRight:        MenuTraverseRight()\n\
    <Key>osfUp:           MenuTraverseUp()\n\
    <Key>osfDown:         MenuTraverseDown()";

static XtActionsRec actions[] = {
    {"Enter", _XmPrimitiveEnter}, /* are these two right ? */
    {"Leave", _XmPrimitiveLeave}, 
    {"ProcessDrag", ProcessDrag},
    {"Help", Help},
    {"MenuTraverseLeft", _XmMenuTraverseLeft},
    {"MenuTraverseRight", _XmMenuTraverseRight},
    {"MenuTraverseUp", _XmMenuTraverseUp},
    {"MenuTraverseDown", _XmMenuTraverseDown},
    {"MenuEscape", _XmMenuEscape},
    {"FocusIn", LabelFocusIn},
    {"FocusOut", LabelFocusOut},
    {"Unmap", LabelUnmap},
};

static XmBaseClassExtRec _XmLabelCoreClassExtRec = {
    /* next_extension            */ NULL,
    /* record_type               */ NULLQUARK,                             
    /* version                   */ XmBaseClassExtVersion,
    /* size                      */ sizeof(XmBaseClassExtRec),
    /* initialize_prehook        */ initialize_prehook,
    /* set_values_prehook        */ NULL, /* FIXME */
    /* initialize_posthook       */ initialize_posthook,
    /* set_values_posthook       */ NULL, /* FIXME */
    /* secondary_object_class    */ NULL, /* FIXME */
    /* secondary_object_create   */ NULL, /* FIXME */
    /* get_secondary_resources   */ NULL, /* FIXME */
    /* fast_subclass             */ { 0 }, /* FIXME */
    /* get_values_prehook        */ NULL, /* FIXME */
    /* get_values_posthook       */ NULL, /* FIXME */
    /* class_part_init_prehook   */ NULL,
    /* class_part_init_posthook  */ NULL,
    /* ext_resources             */ NULL,
    /* compiled_ext_resources    */ NULL,
    /* num_ext_resources         */ 0,
    /* use_sub_resources         */ FALSE,
    /* widget_navigable          */ XmInheritWidgetNavigable,
    /* focus_change              */ XmInheritFocusChange,
    /* wrapper_data              */ NULL
};

XmPrimitiveClassExtRec _XmLabelPrimClassExtRec = {
    /* next_extension      */ NULL,
    /* record_type         */ NULLQUARK,
    /* version             */ XmPrimitiveClassExtVersion,
    /* record_size         */ sizeof(XmPrimitiveClassExtRec),
    /* widget_baseline     */ NULL, /* FIX ME */
    /* widget_display_rect */ NULL, /* FIX ME */
    /* widget_margins      */ NULL  /* FIX ME */
};

XmLabelClassRec xmLabelClassRec = {
    /* Core class part */
    {
	/* superclass            */ (WidgetClass) &xmPrimitiveClassRec,
        /* class_name            */ "XmLabel",
	/* widget_size           */ sizeof(XmLabelRec),
	/* class_initialize      */ class_initialize,
	/* class_part_initialize */ class_part_initialize,
	/* class_inited          */ FALSE,
	/* initialize            */ initialize,
	/* initialize_hook       */ NULL,
	/* realize               */ XtInheritRealize,
	/* actions               */ actions,
	/* num_actions           */ XtNumber(actions),
	/* resources             */ resources,
	/* num_resources         */ XtNumber(resources),
	/* xrm_class             */ NULLQUARK,
	/* compress_motion       */ TRUE,
	/* compress_exposure     */ XtExposeCompressMultiple,
	/* compress_enterleave   */ TRUE,
	/* visible_interest      */ FALSE,
	/* destroy               */ destroy,
	/* resize                */ resize, 
	/* expose                */ expose,
	/* set_values            */ set_values,
	/* set_values_hook       */ NULL,
	/* set_values_almost     */ XtInheritSetValuesAlmost,
	/* get_values_hook       */ NULL,
	/* accept_focus          */ NULL,
	/* version               */ XtVersion,
	/* callback offsets      */ NULL,
	/* tm_table              */ _XmLabel_defaultTranslations,
	/* query_geometry        */ query_geometry,
	/* display_accelerator   */ XtInheritDisplayAccelerator,
	/* extension             */ (XtPointer)&_XmLabelCoreClassExtRec
    },
    /* Primitive Class part */
    {
        /* border_highlight      */ XmInheritBorderHighlight,
        /* border_unhighlight    */ XmInheritBorderUnhighlight,
        /* translations          */ XtInheritTranslations,
        /* arm_and_activate_proc */ XmInheritArmAndActivate,
        /* Synthetic Resources   */ syn_resources,
        /* num syn res           */ XtNumber(syn_resources),
	/* extension             */ (XtPointer)&_XmLabelPrimClassExtRec
    },
    /* Label Class part */
    {
        /* setOverrideCallback */ NULL,
        /* menuProcs           */ label_menu_procs,
        /* translations        */ _XmLabel_menu_traversal_events,
	/* extension           */ NULL
    }
};

WidgetClass xmLabelWidgetClass = (WidgetClass)&xmLabelClassRec;


extern XmFontList _XmFontListCreateDefault(Display *);

static void
class_initialize()
{
    _XmLabelCoreClassExtRec.record_type = XmQmotif;
}

static void
class_part_initialize(WidgetClass widget_class)
{
    XmLabelWidgetClass lwc = (XmLabelWidgetClass)widget_class;

    /* Handle label class part inheritance */

    if (lwc->label_class.menuProcs == XmInheritMenuProc)
	lwc->label_class.menuProcs = label_menu_procs;
    if (lwc->label_class.setOverrideCallback == XmInheritSetOverrideCallback)
	lwc->label_class.setOverrideCallback = False; /* FIX ME */
    if (lwc->label_class.translations == XtInheritTranslations)
	lwc->label_class.translations = _XmLabel_menu_traversal_events;

    _XmFastSubclassInit(widget_class, XmLABEL_BIT);

}

static void
CreateNormalGC(Widget w)
{
    XGCValues values;

    values.foreground = Prim_Foreground(w);
    values.background = XtBackground(w);

    Lab_NormalGC(w) = XtGetGC(w, GCForeground | GCBackground, &values);
}

static void 
CreateInsensitiveGC(Widget w)
{
    XGCValues values;

    values.foreground = Prim_Foreground(w);
    values.background = XtBackground(w);
    values.fill_style = FillStippled;

    Lab_InsensitiveGC(w) = XtGetGC(w, GCForeground | GCBackground | GCFillStyle, &values);
}

static void
initialize_prehook(Widget request,
		   Widget new,
		   ArgList args,
		   Cardinal *num_args)
{
    XdbDebug(__FILE__, new, "Initialize Prehook\n");
}

static void
initialize_posthook(Widget request,
		    Widget new,
		    ArgList args,
		    Cardinal *num_args)
{
    XdbDebug(__FILE__, new, "Initialize Posthook\n");
}

static void
initialize(Widget request,
	   Widget new,
	   ArgList args,
	   Cardinal *num_args)
{
    XdbDebug(__FILE__, new, "Initialize\n");

    /* If the label was not initialized with the resource labelString set,
       use its name -- the following _XmString code comes from MegaButton */
    if (Lab_Label(new) == (XmString)XmUNSPECIFIED 
	|| Lab_Label(new) == (XmString)0) {	/* Shouldn't be necessary but is */
	_XmString xmstring;

	xmstring = _XmOSGetLocalizedString((char *) NULL,
					   (Widget)new,
					   XmNlabelString,
					   XtName(new));

	Lab_Label(new) = _XmStringCreate(xmstring);
    }

    /* get the default fontlist if the label was created without one. */
    if (Lab_Font(new) == (XmFontList)XmUNSPECIFIED || Lab_Font(new) == NULL)
	Lab_Font(new) = _XmGetDefaultFontList(new, XmLABEL_FONTLIST);

    /* have to check request since new may have been polluted by a
     * superclass 
     */
    if (XtWidth(request) == (Dimension)0 || XtHeight(request) == (Dimension)0) {
	/* call this to determine the real dimensions of the label 
	 * since we don't want to use 0
	 */
	_XmLabelRecalcDimensions(request, new);
    }
    /* compute the internal sizes */
    _XmCalcLabelDimensions(new);
    /*_XmLabelRecomputeSize(new); */

    /* allocate the normal and insensitive GC's */
    CreateNormalGC(new);
    CreateInsensitiveGC(new);

    /* if the parent is a row column, set the menu_type to
       it's type.  Otherwise, XmNONE  (Is this right?) FIX ME */

    if (XmIsRowColumn(XtParent(new)))
	Lab_MenuType(new) = RC_Type(XtParent(new));
    else
	Lab_MenuType(new) = XmNONE;

    /* Force the traversal and highlight on enter resources if
       in an popup, pulldown, and option menus. */

    if (Lab_MenuType(new) == XmMENU_POPUP 
	|| Lab_MenuType(new) == XmMENU_PULLDOWN
	|| Lab_MenuType(new) == XmMENU_OPTION)
    {
	Prim_TraversalOn(new) = False;
	Prim_HighlightOnEnter(new) = False;
    }

    /* Put the traversal translations for menus in place */
    if (Lab_MenuType(new) == XmMENU_BAR
	|| Lab_MenuType(new) == XmMENU_PULLDOWN
	|| Lab_MenuType(new) == XmMENU_POPUP) 
    {
	XmLabelWidgetClass lwc = (XmLabelWidgetClass)XtClass(new);

	XtOverrideTranslations(new, XtParseTranslationTable(lwc->label_class.translations));
    }
}

static void
destroy(Widget w)
{
    XtReleaseGC(w, Lab_NormalGC(w));
    XtReleaseGC(w, Lab_InsensitiveGC(w));
}


/* 
 * it is bad to call _XmLabelRecomputeSize inside SetValues
 * since it is possible the parent will reject the size
 * changes and _XmLabelRecomputeSize will have changed widget 
 * internals based on the size it thinks it will get.
 * If the changes are rejected, we are left with the changed
 * internals and the unchanged Width and Height.
 *
 * We could change the label widget to use set_values_almost, but
 * I don't see the need to do so.  In fact, I can't think of any
 * simple widgets that use set_values_almost anyway.  The internal 
 * positions are always calcuated in resize.
 *
 * Lab_RecomputeSize seems to be checked in OSF/Motif when either
 *	o the pixmap changes
 *	o the labelString changes
 * it is possible to change into the other...
 *
 * What happens if we are changing the label (possibly
 * to a pixmap) and the size change is rejected or the new size evaluates 
 * to the same value as the old size?  Resize won't be called.  We need to 
 * do something so expose doesn't use the old internal values.  
 *
 * Just recompute the label internals in expose. :)
 *
 * Notes:
 *
 *    if Lab_RecomputeSize is set, the widget wants to resize to fit
 *    changes to the labelString or pixmap
 */

static Boolean
set_values(Widget old,
	   Widget request,
	   Widget new,
	   ArgList args,
	   Cardinal *num_args)
{
    Boolean refresh_needed = False;

    XdbDebug(__FILE__, new, "set_values()\n");

/* This is a Primitive resource but we have the GC's for it... */
    if (Prim_Foreground(new) != Prim_Foreground(old)) {
	XtReleaseGC(new, Lab_NormalGC(new));
	CreateNormalGC(new);
	XtReleaseGC(new, Lab_InsensitiveGC(new));
	CreateInsensitiveGC(new);
	refresh_needed = True;
    }

    if (Lab_IsText(new))
    {
	if (Lab_AcceleratorText(new) != Lab_AcceleratorText(old)
	    || Lab_Alignment(new) != Lab_Alignment(old)
	    || Lab_Font(new) != Lab_Font(old)
	    || Lab_MarginTop(new) != Lab_MarginTop(old)
	    || Lab_MarginBottom(new) != Lab_MarginBottom(old)
	    || Lab_MarginLeft(new) != Lab_MarginLeft(old)
	    || Lab_MarginRight(new) != Lab_MarginRight(old)
	    || Lab_MarginWidth(new) != Lab_MarginWidth(old)
	    || Lab_MarginHeight(new) != Lab_MarginHeight(old)
	    || Lab_Mnemonic(new) != Lab_Mnemonic(old)
	    || Lab_MnemonicCharset(new) != Lab_MnemonicCharset(old)
	    || Lab_StringDirection(new) != Lab_StringDirection(old))

	    refresh_needed = True;
    }

    /* check for change in insensitive pixmap */
    if ((Lab_PixmapInsensitive(new) != Lab_PixmapInsensitive(old)) 
	&& !XtSensitive(new) && Lab_IsPixmap(new))
    {
	if (Lab_RecomputeSize(new))
	{
	    XtWidth(new) = 0;
	    XtHeight(new) = 0;
	}
	refresh_needed = True;    
    }

    /* check for change in pixmap */
    if (Lab_Pixmap(new) != Lab_Pixmap(old)) 
    {
	/* if changed pixmap to UNSPECIFIED, automatically configure to a
	 * string
	 */
	if (Lab_IsPixmap(new)
	    && Lab_Pixmap(new) == (Pixmap)XmUNSPECIFIED_PIXMAP)
	{
	    Lab_LabelType(new) = XmSTRING;
	}

	if (Lab_RecomputeSize(new))
	{
	    XtWidth(new) = 0;
	    XtHeight(new) = 0;
	}
	refresh_needed = True;
    }

    /* if strings changed, free old string, set new string, and
     * ask for refresh 
     */
    if (! XmStringByteCompare(Lab_Label(new), Lab_Label(old)))
    {
	XmStringFree(Lab_Label(old));
	if (Lab_Label(new))
	    Lab_Label(new) = XmStringCopy(Lab_Label(new));
	else
	    Lab_Label(new) = XmStringCreateSimple(XtName(new));
	
	if (Lab_RecomputeSize(new)) 
	{
	    XtWidth(new) = 0;
	    XtHeight(new) = 0;
 	}
	refresh_needed = True;
    }

    /* did the label change types? */
    if (Lab_LabelType(new) != Lab_LabelType(old)) 
    {
	if (Lab_RecomputeSize(new)) 
	{
	    XtWidth(new) = 0;
	    XtHeight(new) = 0;
	}
	refresh_needed = True;
    }

    /* if labelString is still NULL, it was set that way by the
     * user: NULL labelStrings get the widget name
     */
    if (Lab_Label(new) == NULL) 
    {
	Lab_Label(new) = XmStringCreateSimple(XtName(new));
    }

    /* are we supposed to resize? */
    if (XtWidth(new) == 0 || XtHeight(new) == 0) 
    {
	_XmLabelRecalcDimensions(new, new);
	/* don't recompute the size of the internals, this is done
	 * in resize IF the new width/height are ACCEPTED by the parent
	 */
    }

    return refresh_needed;
}

static void
expose(Widget w,
       XEvent *event,
       Region region)
{
    XRectangle cliprect;
    int width, height;
    GC myGC;

    /* 
     * recompute the labels internals... they could be off if resize
     * hadn't been called after a change (see set_values)
     */
    _XmCalcLabelDimensions(w);

    /* call our superclass's expose method */

#define superclass (&xmPrimitiveClassRec)
    if (Lab_MenuType(w) == XmWORK_AREA)
	(*superclass->core_class.expose)(w, event, region);
#undef superclass

    /* use the right GC */
    if (XtSensitive(w))
        myGC = Lab_NormalGC(w);
    else
        myGC = Lab_InsensitiveGC(w);

    /* Set a clip rectangle for the GC - ensure we don't overwrite shadows */
    cliprect.x = Lab_TextRect_x(w);
    cliprect.y = Lab_TextRect_y(w);
#if	1
/*
 * Had to modify this to get the accelerator in correctly :
 * Danny 17/4/1996
 */
    width = (XtWidth(w) 
	     - 2 * (Lab_Highlight(w) + Lab_MarginWidth(w) + Lab_Shadow(w)) 
	     - Lab_MarginLeft(w) );
#else
/*
 * Guys, this case is valid only if you refuse to show accelerator texts.
 * Unfortunately, the LessTif style guide (a clone of an OSF document)
 * includes accelerator texts.
 */
    width = Lab_TextRect_width(w);
#endif
    height = Lab_TextRect_height(w);

#if	1
    XdbDebug(__FILE__, w, "Label Expose: Lab_TextRect_width = %d, XtWidth = %d, my stuff %d\n",
	Lab_TextRect_width(w), XtWidth(w),
	(XtWidth(w) - 2 * (+ Lab_Highlight(w) + Lab_MarginWidth(w) + Lab_Shadow(w))
             - Lab_MarginLeft(w)));
#endif


    if(width <= 0)
	width = 1;
    if(height <= 0)
	height = 1;
    cliprect.width = width;
    cliprect.height = height;
    XSetClipRectangles(XtDisplay(w),myGC,0,0,&cliprect,1,Unsorted);

    if (Lab_IsText(w)) {
/* LabelString */
	if (_XmLabelShowsMnemonic(w) && Lab_Mnemonic(w)) {
	    char	m[2];

	    m[0] = Lab_Mnemonic(w);
	    m[1] = '\0';

	    _XmStringDrawMnemonic(XtDisplay(w), XtWindow(w),
		Lab_Font(w), Lab_Label(w), myGC,
		Lab_TextRect_x(w), Lab_TextRect_y(w), Lab_TextRect_width(w),
		Lab_Alignment(w),
		0,
		NULL,
		m, Lab_MnemonicCharset(w));
	} else {
	    XmStringDraw(XtDisplay(w),
		     XtWindow(w),
		     Lab_Font(w),
		     Lab_Label(w),
		     myGC,
		     Lab_TextRect_x(w),
		     Lab_TextRect_y(w),
		     Lab_TextRect_width(w),
		     Lab_Alignment(w),
		     0,
		     NULL);
	}
/* AcceleratorText */
	if (_XmLabelShowsAccelerators(w)) {
		XdbDebug(__FILE__, w, "Drawing acceleratorString, widget width is %d, txtr width is %d, Acc at x %d wid %d\n",
			XtWidth(w), Lab_TextRect_width(w), Lab_AccTextRect(w).x, Lab_AccTextRect(w).width);

		XmStringDraw(XtDisplay(w),	/* Do I need a gadget version of this call ? */
		     XtWindow(w),
		     Lab_Font(w),
		     Lab_AcceleratorText(w),
		     myGC,
		     Lab_AccTextRect(w).x,
		     Lab_AccTextRect(w).y,
		     Lab_AccTextRect(w).width,
		     XmALIGNMENT_BEGINNING,
		     0,
		     NULL);
	}
    } 
    else if (Lab_Pixmap(w) != (Pixmap)XmUNSPECIFIED_PIXMAP)
    {
	XCopyArea(XtDisplay(w),
		  Lab_Pixmap(w),
		  XtWindow(w),
		  myGC,
		  0,
		  0,
		  Lab_TextRect_width(w),
		  Lab_TextRect_height(w), 
		  Lab_TextRect_x(w), Lab_TextRect_y(w));
    }

    /* Clear the clip rect */
    XSetClipMask(XtDisplay(w), myGC, None);
}

static void
resize(Widget w)
{
    _XmCalcLabelDimensions(w);

    if (XtIsRealized(w))
	expose(w, NULL, NULL);
}

/*
 * ask how we want to look
 */
#define	Wants(x)	(proposed->request_mode & x)

static XtGeometryResult 
query_geometry(Widget w, XtWidgetGeometry *proposed, XtWidgetGeometry *answer)
{
    XtWidgetGeometry	a;	/* Standin for answer if NULL parameter */

    XdbDebug(__FILE__, w, "query_geometry\n");

    /*
     * NULL case should not yet end here !
     */
    if (proposed->request_mode != 0)
    {
	if ((!(Wants(CWWidth))) && (!Wants(CWHeight))) {
	    /*
	     * If they don't ask width/height, let them have whatever they like
	     */
	    if (answer)
		*answer = *proposed;
	    return XtGeometryYes;
	}
    }

#if 0
    if (Lab_RecomputeSize(w))
	_XmLabelDimensions(w, &a.width, &a.height);
    else
#endif
    {
	a.width = XtWidth(w);
	a.height = XtHeight(w);
    }

    a.request_mode = CWWidth | CWHeight;

    if (answer)
	*answer = a;

    if (proposed->request_mode == 0) /* NULL proposed -> return Width+Height */
	return XtGeometryAlmost;

    if (proposed->width >= answer->width && proposed->height >= answer->height) 
	return XtGeometryYes;
    else if (answer->width == XtWidth(w) && answer->height == XtHeight(w)) {
	if (answer)
	    answer->request_mode = 0;
	return XtGeometryNo;
    }
    else 
	return XtGeometryAlmost;
}

static void
LabelFocusIn(Widget w,
	     XEvent *event,
	     String *params,
	     Cardinal *num_params) 
{
    XdbDebug(__FILE__, w, "LabelFocusIn()\n");

    if (Lab_MenuType(w) == XmMENU_BAR
	|| Lab_MenuType(w) == XmMENU_PULLDOWN
	|| Lab_MenuType(w) == XmMENU_POPUP) {

	XtCallActionProc(w, "MenuFocusIn", event, params, *num_params);
    }
}

static void 
LabelFocusOut(Widget w,
	      XEvent *event,
	      String *params,
	      Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "LabelFocusOut()\n");
}

static void 
LabelUnmap(Widget w,
	      XEvent *event,
	      String *params,
	      Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "LabelUnmap()\n");
}

static Boolean
drag_convert_proc(Widget w, 
		  Atom *selection, 
		  Atom *target, 
		  Atom *type_return, 
		  XtPointer *value_return, 
		  unsigned long *length_return, 
		  int *format_return)
{
    Atom COMPOUND_TEXT;
    Atom MOTIF_DROP;

    COMPOUND_TEXT = XmInternAtom(XtDisplay(w), "COMPOUND_TEXT", False);
    MOTIF_DROP = XmInternAtom(XtDisplay(w), "_MOTIF_DROP", False);

    if (*selection != MOTIF_DROP)
	return False;

    XdbDebug(__FILE__, w, "We're dealing with a motif drop\n");
    return False;
}

static void 
drag_drop_finish(Widget w, 
		 XtPointer client_data, 
		 XtPointer call_data)
{
    Widget source_icon = NULL;

    XtVaGetValues(w, XmNsourceCursorIcon, &source_icon, NULL);

    if (source_icon)
	XtDestroyWidget(source_icon);
}

static void 
ProcessDrag(Widget w, 
	    XEvent *event, 
	    String *params, 
	    Cardinal *num_params)
{
    Atom export_target;
    Arg args[10];
    int n = 0;
    Widget dc;

    XdbDebug(__FILE__, w, "Processing a drag-drop request\n");

    if (Lab_IsPixmap(w))
        export_target = XmInternAtom(XtDisplay(w),
                                     "PIXMAP",
                                     False);
    else 
        export_target = XmInternAtom(XtDisplay(w),
                                     "COMPOUND_TEXT", 
                                     False);

    XtSetArg(args[n], XmNexportTargets, &export_target); n++;
    XtSetArg(args[n], XmNnumExportTargets, 1); n++;
    XtSetArg(args[n], XmNdragOperations, XmDROP_COPY); n++;
    XtSetArg(args[n], XmNconvertProc, drag_convert_proc); n++;
    XtSetArg(args[n], XmNclientData, w); n++;

    dc = XmDragStart(w, event, args, n);
    XtAddCallback (dc, XmNdragDropFinishCallback, drag_drop_finish, NULL);
}

static void 
Help(Widget w, 
     XEvent *event, 
     String *params, 
     Cardinal *num_params)
{
}

void 
_XmLabelGetPixmapSize(Widget w,
		      Pixmap Pix,
		      Dimension *width,
		      Dimension *height)
{
    unsigned int tmp, tmpw, tmph;
    int tmpx, tmpy;
    unsigned Depth;
    Window tmpwin;

    if (Pix == XmUNSPECIFIED_PIXMAP) {
       *width = *height = 10;
       return;
    }

    XGetGeometry(XtDisplayOfObject(w),
		 Pix,
		 &tmpwin,
                 &tmpx, &tmpy,
                 &tmpw, &tmph,
		 &tmp,&Depth);

    *width = (Dimension)tmpw;
    *height = (Dimension)tmph;
}

Boolean
_XmLabelShowsMnemonic(Widget w)
{
	if (! XtIsSubclass(XtParent(w), xmRowColumnWidgetClass))
		return False;
	if (RC_Type(XtParent(w)) != XmMENU_PULLDOWN && RC_Type(XtParent(w)) != XmMENU_BAR)
		return False;
	return True;
}

Boolean
_XmLabelShowsAccelerators(Widget w)
{
	if (! XtIsSubclass(XtParent(w), xmRowColumnWidgetClass))
		return False;
	if (RC_Type(XtParent(w)) != XmMENU_PULLDOWN)
		return False;
	if (XmIsLabel(w)) {
		if (Lab_IsPixmap(w) || Lab_AcceleratorText(w) == NULL)
			return False;
	} else if (XmIsLabelGadget(w)) {
		if (LabG_IsPixmap(w) || LabG_AcceleratorText(w) == NULL)
			return False;
	} else
		return False;
	return True;
}

void
_XmLabelAccTextSize(Widget w)
{
	Dimension	width, height, width2;
	XmString	xms;

	if (XmIsLabel(w)) {
		Lab_AccTextRect(w).width = Lab_AccTextRect(w).height = 0;

		if (! _XmLabelShowsAccelerators(w))
			return;

		xms = XmStringCreateSimple("w");
		XmStringExtent(Lab_Font(w), xms, &width, &height);
		XmStringFree(xms);

		XmStringExtent(Lab_Font(w), Lab_AcceleratorText(w), &width2, &height);

		Lab_AccTextRect(w).height = height;
		Lab_AccTextRect(w).width = 2 * width + width2;

		if (Lab_MarginRight(w) < Lab_AccTextRect(w).width) {
			Lab_MarginRight(w) = Lab_AccTextRect(w).width;
			XdbDebug(__FILE__, w, "_XmLabelAccTextSize: set right margin to %d\n",
				Lab_MarginRight(w));
		}
	} else if (XmIsLabelGadget(w)) {
		LabG_AccTextRect(w).width = LabG_AccTextRect(w).height = 0;

		if (! _XmLabelShowsAccelerators(w))
			return;

		xms = XmStringCreateSimple("w");
		XmStringExtent(LabG_Font(w), xms, &width, &height);
		XmStringFree(xms);
		XmStringExtent(LabG_Font(w), LabG_AcceleratorText(w), &width2, &height);

		LabG_AccTextRect(w).height = height;
		LabG_AccTextRect(w).width = 2 * width + width2;

		if (LabG_MarginRight(w) < LabG_AccTextRect(w).width) {
			LabG_MarginRight(w) = LabG_AccTextRect(w).width;
			XdbDebug(__FILE__, w, "_XmLabelAccTextSize: set right margin to %d\n",
				LabG_MarginRight(w));
		}
	}
	return;
}

/*
 * private label functions 
 * converted to work for both widgets and gadgets -- CT
 *
 * --- JRA
 * this should NEVER be called with XtWidth or XtHeight == 0
 * 
 * this touches the internal dimensions of the widget, it should
 * NOT be called from inside a SetValues call as a result of a 
 * size change since the size may be REJECTED by the PARENT
 *
 * MLM: Hmmm.  The Motif version of this function appears to be
 * always called with XtWidth/XtHeight = 0.  Also, I don't think
 * the statement about the size rejection is entirely true (I've
 * run into a situation with XmPushB where it is essential that
 * the size be changed in the set_values method).  I believe if
 * the parent doesn't like the change, it will reject it -- AND
 * call the child's resize procedure -- FIXME.
 *
 * FIX ME
 *
 * This thing is to be changed for PushButtons (including gadgets)
 * when they're in a menu, in the case that XmNacceleratorText
 * is non-NULL.
 */
void 
_XmCalcLabelDimensions(Widget w)
{
    Dimension width, height;
    Dimension pix_width, pix_height;
    Boolean	showAcc;

    showAcc = _XmLabelShowsAccelerators(w);

    if (XmIsLabel(w))
    {
	/* set the label's size so the pixmap/string fits */
	if (Lab_IsText(w))
	    XmStringExtent(Lab_Font(w),
			   Lab_Label(w),
			   &width,
			   &height);
	else /* Lab_IsPixmap(w) */
	{
	    _XmLabelGetPixmapSize(w,Lab_Pixmap(w), &pix_width, &pix_height);	
	    width = pix_width;
	    height = pix_height;

	    XdbDebug(__FILE__, w, "_XmLabelGetPixmapSize: w = %d, h = %d\n", width, height);
	}
	
	Lab_TextRect_width(w) = width;
	Lab_TextRect_height(w) = height;

	    
	/* The alignments only modify x values if a size was set */

	_XmLabelAccTextSize(w);

	switch(Lab_Alignment(w))
	{
	case XmALIGNMENT_END:
	    Lab_TextRect_x(w) = (XtWidth(w)  
				 - Lab_Highlight(w)
				 - Lab_Shadow(w)
				 - Lab_MarginWidth(w)
				 - Lab_MarginRight(w)
				 - Lab_TextRect_width(w));
	    break;
	case XmALIGNMENT_BEGINNING:
	    /* Default X Position of TextRect */	
	    Lab_TextRect_x(w) = (Lab_Highlight(w)
				 + Lab_Shadow(w)
				 + Lab_MarginWidth(w)
				 + Lab_MarginLeft(w));
	    break;
	case XmALIGNMENT_CENTER:
	default:
	    Lab_TextRect_x(w) = Lab_Highlight(w)
				 + Lab_Shadow(w)
				 + Lab_MarginWidth(w)
				 + Lab_MarginLeft(w)
				 + (XtWidth(w)
				     - Lab_TextRect_width(w)
				     - 2 * Lab_Highlight(w)
				     - 2 * Lab_Shadow(w)
				     - 2 * Lab_MarginWidth(w)
				     - Lab_MarginLeft(w)
				     - Lab_MarginRight(w))/2;
	    break;
	}
	Lab_TextRect_y(w) = (Lab_Highlight(w)
			     + Lab_Shadow(w)
			     + Lab_MarginHeight(w)
			     + Lab_MarginTop(w)
			     + (XtHeight(w) 
				- 2*(Lab_Highlight(w)
				     + Lab_Shadow(w)
				     + Lab_MarginHeight(w))
				- Lab_MarginTop(w)
				- Lab_MarginBottom(w)
				- Lab_TextRect_height(w))/2);
	if (showAcc) {
		Lab_AccTextRect(w).x = XtWidth(w)
				- Lab_Shadow(w)
				- Lab_Highlight(w)
				- Lab_AccTextRect(w).width;
		Lab_AccTextRect(w).y = Lab_TextRect_y(w);

		XdbDebug(__FILE__, w, "Accelerator @ %d %d (x = %d - %d - %d - %d)\n",
			Lab_AccTextRect(w).x, Lab_AccTextRect(w).y,
			XtWidth(w), Lab_Shadow(w), Lab_Highlight(w),
			Lab_AccTextRect(w).width);
	}
    }
    else 
    {
	/* NB. For gadgets, the TextRect is relative to the XtX and XtY
	   positions, as the Xt[X|Y] and border_widths are added onto
	   them before the text is drawn. */
	
	_XmLabelAccTextSize(w);

	/* set the label's size so the pixmap/string fits */
	if (LabG_IsText(w))
	    XmStringExtent(LabG_Font(w),
			   LabG_Label(w),
			   &width,
			   &height);
	else /* LabG_IsPixmap(w) */
	{
	    _XmLabelGetPixmapSize(w,LabG_Pixmap(w), &pix_width, &pix_height);	
	    width = pix_width;
	    height = pix_height;
	}
	
	LabG_TextRect_width(w) = width;
	LabG_TextRect_height(w) = height;
	
	switch (LabG_Alignment(w))
	{
	case XmALIGNMENT_END:
	    LabG_TextRect_x(w) = (XtWidth(w)  
				  - LabG_Highlight(w)
				  - LabG_Shadow(w)
				  - LabG_MarginWidth(w)
				  - LabG_MarginRight(w)
				  - LabG_TextRect_width(w));
	    break;
	case XmALIGNMENT_BEGINNING:
	    LabG_TextRect_x(w) = (LabG_Highlight(w)
				  + LabG_Shadow(w)
				  + LabG_MarginWidth(w)
				  + LabG_MarginLeft(w));
	    break;
	case XmALIGNMENT_CENTER:
	default:
	    LabG_TextRect_x(w) = LabG_Highlight(w)
				 + LabG_Shadow(w)
				 + LabG_MarginWidth(w)
				 + LabG_MarginLeft(w)
				 + (XtWidth(w)
				     - LabG_TextRect_width(w)
				     - 2 * LabG_Highlight(w)
				     - 2 * LabG_Shadow(w)
				     - 2 * LabG_MarginWidth(w)
				     - LabG_MarginLeft(w)
				     - LabG_MarginRight(w))/2;
	    break;
	}
	
	LabG_TextRect_y(w) = (LabG_Highlight(w)
			      + LabG_Shadow(w)
			      + LabG_MarginHeight(w)
			      + LabG_MarginTop(w)
			      + (XtHeight(w)
				 - 2*(LabG_Highlight(w)
				      + LabG_Shadow(w)
				      + LabG_MarginHeight(w))
				 - LabG_MarginTop(w)
				 - LabG_MarginBottom(w)
				 - LabG_TextRect_height(w))/2);
	if (showAcc) {
		LabG_AccTextRect(w).x = XtWidth(w) - LabG_MarginRight(w);	/* FIX ME */
		LabG_AccTextRect(w).y = LabG_TextRect_y(w);
	}
    }
}

void 
_XmLabelDimensions(Widget w, 
		   Dimension *wd, 
		   Dimension *ht)
{
    if (XmIsLabel(w))
    {
	_XmCalcLabelDimensions(w);

	*wd = (Lab_Highlight(w)
	       + Lab_Shadow(w)
	       + Lab_MarginLeft(w)
	       + Lab_MarginWidth(w)
	       + Lab_TextRect_width(w)
	       + Lab_MarginWidth(w)
	       + Lab_MarginRight(w)
	       + Lab_Shadow(w)
	       + Lab_Highlight(w));
	
	*ht = (Lab_Highlight(w)
	       + Lab_Shadow(w)
	       + Lab_MarginTop(w)
	       + Lab_MarginHeight(w)
	       + Lab_TextRect_height(w)
	       + Lab_MarginHeight(w)
	       + Lab_MarginBottom(w)
	       + Lab_Shadow(w)
	       + Lab_Highlight(w));
    }
    else if (XmIsLabelGadget(w))
    {
	_XmCalcLabelDimensions(w);

	*wd = (LabG_Highlight(w)
	       + LabG_Shadow(w)
	       + LabG_MarginLeft(w)
	       + LabG_MarginWidth(w)
	       + LabG_TextRect_width(w)
	       + LabG_MarginWidth(w)
	       + LabG_MarginRight(w)
	       + LabG_Shadow(w)
	       + LabG_Highlight(w));
	
	*ht = (LabG_Highlight(w)
	       + LabG_Shadow(w)
	       + LabG_MarginTop(w)
	       + LabG_MarginHeight(w)
	       + LabG_TextRect_height(w)
	       + LabG_MarginHeight(w)
	       + LabG_MarginBottom(w)
	       + LabG_Shadow(w)
	       + LabG_Highlight(w));
    }
    else
    {
	Cardinal np = 2;
	String	pp[2];

	pp[0] = XtName(w);
	pp[1] = w->core.widget_class->core_class.class_name;

	XtAppWarningMsg(XtWidgetToApplicationContext(w),
		"widgetType", "widgetType",
		"LessTifError",
		"Widget %s is not a XmLabel(Gadget) but a %s",
		pp, &np);
    }
}


/* special request to recompute length/width values
 * the width/height wasn't specified or was set to 0, make it the
 * width/height of the string + margins
 *
 * I broke this out of _XmCalcLabelDimensions since it may depend
 * on the REQUEST widget's values and it's cleaner.
 */
void
_XmLabelRecalcDimensions(Widget request,
			 Widget new)
{
    Dimension width, height;

    if (!XmIsLabelGadget(new))
    {
	if (Lab_IsText(new))
	{
	    XmStringExtent(Lab_Font(new),
			   Lab_Label(new),
			   &width,
			   &height);
	}
	else	/* pixmap */
	{
	    _XmLabelGetPixmapSize(new, Lab_Pixmap(new), &width, &height);
	}

	if (XtWidth(request) == 0)
	    XtWidth(new) = (width
			    + Lab_Highlight(new) * 2
			    + Lab_Shadow(new) * 2
			    + Lab_MarginWidth(new) * 2
			    + Lab_MarginRight(new)
			    + Lab_MarginLeft(new));
	if (XtHeight(request) == 0)
	    XtHeight(new) = (height
			     + Lab_Highlight(new) * 2
			     + Lab_Shadow(new) * 2
			     + Lab_MarginHeight(new) * 2
			     + Lab_MarginTop(new)
			     + Lab_MarginBottom(new));
    }
    else /* this is a gadget */
    {
	if (LabG_IsText(new))
	{
	    XmStringExtent(LabG_Font(new),
			   LabG_Label(new),
			   &width,
			   &height);
	}
	else	/* pixmap */
	{
	    _XmLabelGetPixmapSize(new, LabG_Pixmap(new), &width, &height);
	}

	if (XtWidth(request) == 0) {
	    XtWidth(new) = (width
			    + LabG_Highlight(new) * 2
			    + LabG_Shadow(new) * 2
			    + LabG_MarginWidth(new) * 2
			    + LabG_MarginRight(new)
			    + LabG_MarginLeft(new));
	}
	if (XtHeight(request) == 0) {
	    XtHeight(new) = (height
			     + LabG_Highlight(new) * 2
			     + LabG_Shadow(new) * 2
			     + LabG_MarginHeight(new) * 2
			     + LabG_MarginTop(new)
			     + LabG_MarginBottom(new));
	}
    }
}

void 
_XmLabelRecomputeSize(Widget w)
{
    _XmLabelDimensions(w, 
		       &XtWidth(w),
		       &XtHeight(w));
    XdbDebug(__FILE__, w, "_XmLabelRecomputeSize => %d %d\n",
	     XtWidth(w), XtHeight(w));
}

static void 
#ifdef __STDC__
label_menu_procs(int function, Widget widget, ...)
{
    va_list arg_list;
    XtPointer foo;
    XEvent *event; 
    XtPointer returnData;

    va_start(arg_list, widget);

#else
label_menu_procs(function, widget, va_alist)
    int function;
    Widget widget;
    va_dcl
{
    va_list arglist;
    XtPointer foo;
    XEvent *event; 
    XtPointer returnData;

    va_start(arg_list);

#endif
    foo = va_arg(arg_list, XtPointer);
    event = va_arg(arg_list, XEvent *);
    returnData = va_arg(arg_list, XtPointer);

    switch (function)
    {
    case XmMENU_BUTTON: 
	XdbDebug(__FILE__, widget, "label_menu_procs(XmMENU_BUTTON)\n");
	{
	    /* There was a button press in the menu.  
               Check to see if it was the valid button
	       for that type of menu */
	    XButtonEvent *xbe = (XButtonEvent*)event;

	    if (!XmIsRowColumn(widget)) {
		va_end(arg_list);
		return;
	    }

	    switch (RC_Type(widget))
	    {
	    case XmMENU_BAR:
	    case XmMENU_PULLDOWN:
	    case XmMENU_OPTION:
		*(Boolean*)returnData = (xbe->button == 1);
		break;
	    case XmMENU_POPUP:
		/* the third mouse button also works in popups */
		*(Boolean*)returnData = ((xbe->button == 1) || (xbe->button == 3));		
		break;
	    default:
		*(Boolean*)returnData = False; /* ? */
		break;
	    }
	    break;
	}
    case XmMENU_POPDOWN:
	XdbDebug(__FILE__, widget, "label_menu_procs(XmMENU_POPDOWN)\n");
	break;
    case XmMENU_BUTTON_POPDOWN:
	XdbDebug(__FILE__, widget, "label_menu_procs(XmMENU_BUTTON_POPDOWN)\n");
	{
	    Widget shell = XtParent(widget);

	    while (!XtIsShell(shell))
		shell = XtParent(shell);

	    if (Lab_MenuType(widget) == XmMENU_POPUP)
		*(Boolean*)returnData = True;
	    else
		*(Boolean*)returnData = False;

	    XUngrabKeyboard(XtDisplay(widget), CurrentTime);

	    XtCallActionProc(shell, "MenuShellPopdownDone", event, NULL, 0);

	    XUngrabPointer(XtDisplay(widget), CurrentTime);

	    break;
	}
    case XmMENU_SHELL_POPDOWN:
	XdbDebug(__FILE__, widget, "label_menu_procs(XmMENU_SHELL_POPDOWN)\n");
	{
	    XtCallActionProc(widget, "MenuShellPopdownOne", event, NULL, 0);

	    break;
	}
	break;
/*
 * Uncaught cases
 */
    default:
	XdbDebug(__FILE__, widget, "label_menu_procs(%d) : uncaught case in switch\n", function);
	break;
    }
    va_end(arg_list);
}

Widget
XmCreateLabel(Widget parent,
	      char *name,
	      Arg *arglist,
	      Cardinal argcount)
{
    return XtCreateWidget(name,
			  xmLabelWidgetClass,
			  parent,
			  arglist,
			  argcount);
}
