/****************************************************************************
 * This module is based on Twm, but has been siginificantly modified 
 * by Rob Nation 
 ****************************************************************************/
/*****************************************************************************/
/**       Copyright 1988 by Evans & Sutherland Computer Corporation,        **/
/**                          Salt Lake City, Utah                           **/
/**  Portions Copyright 1989 by the Massachusetts Institute of Technology   **/
/**                        Cambridge, Massachusetts                         **/
/**                                                                         **/
/**                           All Rights Reserved                           **/
/**                                                                         **/
/**    Permission to use, copy, modify, and distribute this software and    **/
/**    its documentation  for  any  purpose  and  without  fee is hereby    **/
/**    granted, provided that the above copyright notice appear  in  all    **/
/**    copies and that both  that  copyright  notice  and  this  permis-    **/
/**    sion  notice appear in supporting  documentation,  and  that  the    **/
/**    names of Evans & Sutherland and M.I.T. not be used in advertising    **/
/**    in publicity pertaining to distribution of the  software  without    **/
/**    specific, written prior permission.                                  **/
/**                                                                         **/
/**    EVANS & SUTHERLAND AND M.I.T. DISCLAIM ALL WARRANTIES WITH REGARD    **/
/**    TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES  OF  MERCHANT-    **/
/**    ABILITY  AND  FITNESS,  IN  NO  EVENT SHALL EVANS & SUTHERLAND OR    **/
/**    M.I.T. BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL  DAM-    **/
/**    AGES OR  ANY DAMAGES WHATSOEVER  RESULTING FROM LOSS OF USE, DATA    **/
/**    OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER    **/
/**    TORTIOUS ACTION, ARISING OUT OF OR IN  CONNECTION  WITH  THE  USE    **/
/**    OR PERFORMANCE OF THIS SOFTWARE.                                     **/
/*****************************************************************************/


/***********************************************************************
 * mwm - "LessTif Window Manager"
 ***********************************************************************/
#include "mwm.h"

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#ifdef TIME_WITH_SYS_TIME
#include <time.h>
#endif
#endif
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif

#ifdef HAVE_SYS_SYSTEMINFO_H
#include <sys/systeminfo.h>
#endif

#include <X11/Shell.h>
#include <Xm/DisplayP.h>
#include <Xm/ScreenP.h>

/*
 * application globals
 */
#define MAXHOSTNAME 255

MwmInternalInfo Mwm;
ScreenInfo      Scr;		/* structures for the screen */
Display        *dpy;		/* which display are we talking to */
Widget          toplevel;
Widget          quit_mb = NULL;
Widget          restart_mb = NULL;
Widget          xmDisplay;
Widget         *xmScreens;
Boolean         multiscreen = False;
char           *mwm_name = "mwm";

XErrorHandler   CatchRedirectError(Display *, XErrorEvent *);
XErrorHandler   MwmErrorHandler(Display *, XErrorEvent *);
void            newhandler(int sig);
void            CreateCursors(void);
void            NoisyExit(int);
void            ChildDied(int nonsense);
void            SaveDesktopState(void);

XContext        MwmContext;	/* context for mwm windows */
XContext        MenuContext;	/* context for mwm menus */

XClassHint      NoClass;	/* for applications with no class */

int             JunkX = 0, JunkY = 0;
Window          JunkRoot, JunkChild;	/* junk window */
unsigned int    JunkWidth, JunkHeight, JunkBW, JunkDepth, JunkMask;

Bool            debugging = False, PPosOverride;
char          **g_argv;

int             ShapeEventBase, ShapeErrorBase;

long            isIconicState = 0;
extern XEvent   Event;
Bool            Restarting = False;
int             fd_width, x_fd;

/* function prototype for yyerror.  Why is this used? */
extern void yyerror(char *fmt, ...);

/***********************************************************************
 *
 *  Procedure:
 *	main - start of mwm
 *
 ***********************************************************************
 */
void
main(int argc, char **argv)
{
    unsigned long   valuemask;	/* mask for create windows */
    XSetWindowAttributes attributes;	/* attributes for create windows */
    void            InternUsefulAtoms(void);
    void            InitVariables(void);
    void            enterAlarm(int);
    int             i;
    extern int      x_fd;
    int             len;
    char           *display_string;
    char            message[255];
    char            num[10];
    Bool            option_error = FALSE;
    XtAppContext    app;
    extern void     MwmInit(void);
    Atom            mwm_info;
    PropMotifWmInfo info;



    XtSetLanguageProc(NULL, NULL, NULL);

    toplevel = XtVaAppInitialize(&app, "Mwm", NULL, 0, &argc, argv, NULL, NULL);

    for (i = 1; i < argc; i++) {
	if (strncasecmp(argv[i], "-debug", 6) == 0)
	    debugging = True;
	else if (strncasecmp(argv[i], "-multiscreen", 12) == 0) {
	    multiscreen = True;
	}
	else if (strncasecmp(argv[i], "-name", 5) == 0) {
	    if (++i >= argc)
		usage();
	    mwm_name = argv[i];
	}
	else if (strncasecmp(argv[i], "-version", 8) == 0) {
	    fprintf(stderr, "Mwm Version %s\n", VERSION);
	}
	else {
	    fprintf(stderr, "mwm:  Unknown option:  `%s'\n", argv[i]);
	    option_error = TRUE;
	}
    }

    if (option_error) {
	usage();
    }

    g_argv = argv;

    newhandler(SIGINT);
    newhandler(SIGHUP);
    newhandler(SIGQUIT);
    newhandler(SIGTERM);
    signal(SIGUSR1, Restart);
    signal(SIGALRM, enterAlarm);

    dpy = XtDisplay(toplevel);
    xmDisplay = XmGetXmDisplay(dpy);

    x_fd = XConnectionNumber(dpy);

    /*
     * this is enormously dangerous, but _is_ the original code. MLM
     */
    if (fcntl(x_fd, F_SETFD, 1) == -1) {
	yyerror("close-on-exec failed");
	exit(1);
    }
    Scr.screen = DefaultScreen(dpy);
    Mwm.number_of_screens = ScreenCount(dpy);

    if (multiscreen) {
	xmScreens = (WidgetList) XtMalloc(Mwm.number_of_screens * sizeof(Widget));
	for (i = 0; i < Mwm.number_of_screens; i++) {
	    xmScreens[i] = XmGetXmScreen(ScreenOfDisplay(dpy, i));
	}
    }


    /*  Add a DISPLAY entry to the environment, incase we were started
     * with mwm -display term:0.0
     */
    len = strlen(XDisplayString(dpy));
    display_string = XtMalloc(len + 10);
    sprintf(display_string, "DISPLAY=%s", XDisplayString(dpy));
    putenv(display_string);
    /* Add a HOSTDISPLAY environment variable, which is the same as
     * DISPLAY, unless display = :0.0 or unix:0.0, in which case the full
     * host name will be used for ease in networking . */
    if (strncmp(display_string, "DISPLAY=:", 9) == 0) {
	char            client[MAXHOSTNAME], *rdisplay_string;

#ifdef HAVE_SYS_SYSTEMINFO_H
	sysinfo(SI_HOSTNAME, client, MAXHOSTNAME);
#else
#ifdef HAVE_GETHOSTNAME
	gethostname(client, MAXHOSTNAME);
#else
#error You lose.
#endif
#endif
	rdisplay_string = XtMalloc(len + 14 + strlen(client));
	sprintf(rdisplay_string, "HOSTDISPLAY=%s:%s", client, &display_string[9]);
	putenv(rdisplay_string);
    }
    else if (strncmp(display_string, "DISPLAY=unix:", 13) == 0) {
	char            client[MAXHOSTNAME], *rdisplay_string;

#ifdef HAVE_SYS_SYSTEMINFO_H
	sysinfo(SI_HOSTNAME, client, MAXHOSTNAME);
#else
#ifdef HAVE_GETHOSTNAME
	gethostname(client, MAXHOSTNAME);
#else
#error You lose.
#endif
#endif
	rdisplay_string = XtMalloc(len + 14 + strlen(client));
	sprintf(rdisplay_string, "HOSTDISPLAY=%s:%s", client,
		&display_string[13]);
	putenv(rdisplay_string);
    }
    else {
	char           *rdisplay_string;

	rdisplay_string = XtMalloc(len + 14);
	sprintf(rdisplay_string, "HOSTDISPLAY=%s", XDisplayString(dpy));
	putenv(rdisplay_string);
    }

    Scr.root_win = RootWindow(dpy, Scr.screen);
    if (Scr.root_win == None) {
	yyerror("Screen %d is not a valid screen", (char *) Scr.screen);
	exit(1);
    }

    XShapeQueryExtension(dpy, &ShapeEventBase, &ShapeErrorBase);

    InternUsefulAtoms();

    /* set the MWM_INFO property on the Root */
    if (is_custom)
	info.flags = MWM_INFO_STARTUP_CUSTOM;
    else
	info.flags = MWM_INFO_STARTUP_STANDARD;

    info.wmWindow = Scr.root_win;

    mwm_info = XInternAtom(dpy, _XA_MOTIF_WM_INFO, False);

    XChangeProperty(dpy, Scr.root_win, mwm_info, mwm_info,
		    32, PropModeReplace,
		    (unsigned char *)&info, PROP_MWM_INFO_ELEMENTS);

    /* Make sure property priority colors is empty */
    XChangeProperty(dpy, Scr.root_win, _XA_MIT_PRIORITY_COLORS,
		    XA_CARDINAL, 32, PropModeReplace, NULL, 0);

    XSetErrorHandler((XErrorHandler) CatchRedirectError);
    XSelectInput(dpy, Scr.root_win,
		 LeaveWindowMask | EnterWindowMask | PropertyChangeMask |
		 SubstructureRedirectMask | KeyPressMask |
		 ButtonPressMask | ButtonReleaseMask);
    XSync(dpy, 0);

    XSetErrorHandler((XErrorHandler) MwmErrorHandler);
    CreateCursors();
    InitVariables();

    XGrabServer(dpy);

    /* initialize the Mwm resources */
    Scr.d_depth = DefaultDepth(dpy, Scr.screen);
    MwmInit();
    Scr.flags = BackingStore|AppsBackingStore;

    /* create a window which will accept the keyboard focus when no other 
       windows have it */
    attributes.event_mask = KeyPressMask | FocusChangeMask;
    attributes.override_redirect = True;
    Scr.no_focus_win = XCreateWindow(dpy, Scr.root_win, -10, -10, 10, 10, 0, 0,
				   InputOnly, CopyFromParent,
				   CWEventMask | CWOverrideRedirect,
				   &attributes);
    XMapWindow(dpy, Scr.no_focus_win);

    XSetInputFocus(dpy, Scr.no_focus_win, RevertToParent, CurrentTime);
    Scr.TitleHeight = Scr.WindowFont.font->ascent +
	Scr.WindowFont.font->descent + 3;

    XSync(dpy, 0);
    if (debugging)
	XSynchronize(dpy, 1);

    Scr.SizeStringWidth = XTextWidth(Scr.StdFont.font,
				     " +8888 x +8888 ", 15);
    attributes.border_pixel = Mwm.components[MWM_FEEDBACK].foreground;
    attributes.background_pixel = Mwm.components[MWM_FEEDBACK].background;
    attributes.bit_gravity = NorthWestGravity;
    valuemask = (CWBorderPixel | CWBackPixel | CWBitGravity);
    Scr.size_win = XCreateWindow(dpy, Scr.root_win,
				   0, 0,
				   (unsigned int) (Scr.SizeStringWidth +
						   SIZE_HINDENT * 2),
				   (unsigned int) (Scr.StdFont.height +
						   SIZE_VINDENT * 2),
				   (unsigned int) 0, 0,
				   (unsigned int) CopyFromParent,
				   (Visual *) CopyFromParent,
				   valuemask, &attributes);
    initPanFrames();
    CaptureAllWindows();
    checkPanFrames();
    XUngrabServer(dpy);
    MoveResizeViewPortIndicator();
#ifdef HAVE_SYSCONF
    fd_width = sysconf(_SC_OPEN_MAX);
#else
#ifdef HAVE_GETDTABLESIZE
    fd_width = getdtablesize();
#else
#error You lose
#endif
#endif


    if (Restarting) {
	if (Scr.RestartFunction != NULL)
	    ExecuteFunction(F_FUNCTION, NULL, None, NULL, &Event, C_ROOT, 0, 0,
			    0, 0, Scr.RestartFunction);
    }
    else {
	if (Scr.InitFunction != NULL)
	    ExecuteFunction(F_FUNCTION, NULL, None, NULL, &Event, C_ROOT, 0, 0,
			    0, 0, Scr.InitFunction);
    }

    HandleEvents();
    return;
}

/***********************************************************************
 *
 *  Procedure:
 *      CaptureAllWindows
 *
 *   Decorates all windows at start-up
 *
 ***********************************************************************/

void
CaptureAllWindows(void)
{
    int             i, j;
    unsigned int    nchildren;
    Window          root, parent, *children;

    PPosOverride = TRUE;

    if (!XQueryTree(dpy, Scr.root_win, &root, &parent, &children, &nchildren))
	return;

    /*
     * weed out icon windows
     */
    for (i = 0; i < nchildren; i++) {
	if (children[i]) {
	    XWMHints       *wmhintsp = XGetWMHints(dpy, children[i]);

	    if (wmhintsp) {
		if (wmhintsp->flags & IconWindowHint) {
		    for (j = 0; j < nchildren; j++) {
			if (children[j] == wmhintsp->icon_window) {
			    children[j] = None;
			    break;
			}
		    }
		}
		XFree((char *) wmhintsp);
	    }
	}
    }
    /*
     * map all of the non-override windows
     */

    for (i = 0; i < nchildren; i++) {
	if (children[i] && MappedNotOverride(children[i])) {
	    XUnmapWindow(dpy, children[i]);
	    Event.xmaprequest.window = children[i];
	    HandleMapRequest();
	}
    }

    isIconicState = DontCareState;

    if (nchildren > 0)
	XFree((char *) children);

    /* after the windows already on the screen are in place,
     * don't use PPosition */
    PPosOverride = FALSE;
}

/***********************************************************************
 *
 *  Procedure:
 *	MappedNotOverride - checks to see if we should really
 *		put a mwm frame on the window
 *
 *  Returned Value:
 *	TRUE	- go ahead and frame the window
 *	FALSE	- don't frame the window
 *
 *  Inputs:
 *	w	- the window to check
 *
 ***********************************************************************/

int
MappedNotOverride(Window w)
{
    XWindowAttributes wa;
    Atom            atype;
    int             aformat;
    unsigned long   nitems, bytes_remain;
    unsigned char  *prop;

    isIconicState = DontCareState;

    if (!XGetWindowAttributes(dpy, w, &wa))
	return False;

    if (XGetWindowProperty(dpy, w, _XA_WM_STATE, 0L, 3L, False, _XA_WM_STATE,
	      &atype, &aformat, &nitems, &bytes_remain, &prop) == Success) {
	if (prop != NULL) {
	    isIconicState = *(long *) prop;
	    XFree(prop);
	}
    }
    if (w == Scr.Pager_w)
	return True;
    return (((isIconicState == IconicState) || (wa.map_state != IsUnmapped)) &&
	    (wa.override_redirect != True));
}


/***********************************************************************
 *
 *  Procedure:
 *	InternUsefulAtoms:
 *            Dont really know what it does
 *
 ***********************************************************************
 */
Atom            _XA_MIT_PRIORITY_COLORS;
Atom            _XA_WM_CHANGE_STATE;
Atom            _XA_WM_STATE;
Atom            _XA_WM_COLORMAP_WINDOWS;
Atom            _XA_WM_PROTOCOLS;
Atom            _XA_WM_TAKE_FOCUS;
Atom            _XA_WM_DELETE_WINDOW;
Atom            _XA_WM_DESKTOP;
Atom            _XA_MwmAtom;

void
InternUsefulAtoms(void)
{
    /* 
     * Create priority colors if necessary.
     */
    _XA_MIT_PRIORITY_COLORS = XInternAtom(dpy, "_MIT_PRIORITY_COLORS", False);
    _XA_WM_CHANGE_STATE = XInternAtom(dpy, "WM_CHANGE_STATE", False);
    _XA_WM_STATE = XInternAtom(dpy, "WM_STATE", False);
    _XA_WM_COLORMAP_WINDOWS = XInternAtom(dpy, "WM_COLORMAP_WINDOWS", False);
    _XA_WM_PROTOCOLS = XInternAtom(dpy, "WM_PROTOCOLS", False);
    _XA_WM_TAKE_FOCUS = XInternAtom(dpy, "WM_TAKE_FOCUS", False);
    _XA_WM_DELETE_WINDOW = XInternAtom(dpy, "WM_DELETE_WINDOW", False);
    _XA_WM_DESKTOP = XInternAtom(dpy, "WM_DESKTOP", False);
    _XA_MwmAtom = XInternAtom(dpy, "_MOTIF_WM_HINTS", False);
    return;
}

/***********************************************************************
 *
 *  Procedure:
 *	newhandler: Installs new signal handler
 *
 ************************************************************************/
void
newhandler(int sig)
{
    if (signal(sig, SIG_IGN) != SIG_IGN)
	signal(sig, SigDone);
}


/*************************************************************************
 * Restart on a signal
 ************************************************************************/
RETSIGTYPE
Restart(int nonsense)
{
    Done(1, *g_argv); 
#ifndef _HPUX_SOURCE
    return (RETSIGTYPE) 0;
#endif
}

/***********************************************************************
 *
 *  Procedure:
 *	CreateCursors - Loads mwm cursors
 *
 ***********************************************************************
 */
void
CreateCursors(void)
{
    /* define cursors */
    Scr.MwmCursors[POSITION_CURS] = XCreateFontCursor(dpy, XC_top_left_corner);
    Scr.MwmCursors[DEFAULT_CURS] = XCreateFontCursor(dpy, XC_top_left_arrow);
    Scr.MwmCursors[SYS_CURS] = XCreateFontCursor(dpy, XC_hand2);
    Scr.MwmCursors[TITLE_CURS] = XCreateFontCursor(dpy, XC_top_left_arrow);
    Scr.MwmCursors[MOVE_CURS] = XCreateFontCursor(dpy, XC_fleur);
    Scr.MwmCursors[MENU_CURS] = XCreateFontCursor(dpy, XC_sb_left_arrow);
    Scr.MwmCursors[WAIT_CURS] = XCreateFontCursor(dpy, XC_watch);
    Scr.MwmCursors[SELECT_CURS] = XCreateFontCursor(dpy, XC_dot);
    Scr.MwmCursors[DESTROY_CURS] = XCreateFontCursor(dpy, XC_pirate);
    Scr.MwmCursors[LEFT_CURS] = XCreateFontCursor(dpy, XC_left_side);
    Scr.MwmCursors[RIGHT_CURS] = XCreateFontCursor(dpy, XC_right_side);
    Scr.MwmCursors[TOP_CURS] = XCreateFontCursor(dpy, XC_top_side);
    Scr.MwmCursors[BOTTOM_CURS] = XCreateFontCursor(dpy, XC_bottom_side);
    Scr.MwmCursors[TOP_LEFT_CURS] = XCreateFontCursor(dpy, XC_top_left_corner);
    Scr.MwmCursors[TOP_RIGHT_CURS] = XCreateFontCursor(dpy, XC_top_right_corner);
    Scr.MwmCursors[BOT_LEFT_CURS] = XCreateFontCursor(dpy, XC_bottom_left_corner);
    Scr.MwmCursors[BOT_RIGHT_CURS] = XCreateFontCursor(dpy, XC_bottom_right_corner);
}

/***********************************************************************
 *
 *  Procedure:
 *	InitVariables - initialize mwm variables
 *
 ************************************************************************/
void
InitVariables(void)
{
    MwmContext = XUniqueContext();
    MenuContext = XUniqueContext();
    NoClass.res_name = NoName;
    NoClass.res_class = NoName;

    Scr.d_depth = DefaultDepth(dpy, Scr.screen);
    Scr.MwmRoot.w = Scr.root_win;
    Scr.MwmRoot.next = 0;
    XGetWindowAttributes(dpy, Scr.root_win, &(Scr.MwmRoot.attr));
    Scr.root_pushes = 0;
    Scr.pushed_window = &Scr.MwmRoot;
    Scr.MwmRoot.number_cmap_windows = 0;


    Scr.d_width = DisplayWidth(dpy, Scr.screen);
    Scr.d_height = DisplayHeight(dpy, Scr.screen);

    Scr.NoBoundaryWidth = 1;
    Scr.BoundaryWidth = BOUNDARY_WIDTH;
    Scr.CornerWidth = CORNER_WIDTH;
    Scr.Hilite = NULL;
    Scr.Focus = NULL;
    Scr.Ungrabbed = NULL;

    Scr.StdFont.font = NULL;
    Scr.StdFont.name = "fixed";
    Scr.WindowFont.name = "fixed";

    Scr.VScale = 32;
    Scr.VxMax = 3;
    Scr.VyMax = 3;
    Scr.Vx = Scr.Vy = 0;

    /* Sets the current desktop number to zero */
    /* Multiple desks are available even in non-virtual
     * compilations */
    {
	Atom            atype;
	int             aformat;
	unsigned long   nitems, bytes_remain;
	unsigned char  *prop;

	Scr.CurrentDesk = 0;
	if ((XGetWindowProperty(dpy, Scr.root_win, _XA_WM_DESKTOP, 0L, 1L, True,
				_XA_WM_DESKTOP, &atype, &aformat, &nitems,
				&bytes_remain, &prop)) == Success) {
	    if (prop != NULL) {
		Restarting = True;
		Scr.CurrentDesk = *(unsigned long *) prop;
	    }
	}
    }

    Scr.EdgeScrollX = Scr.EdgeScrollY = -100000;
    Scr.ScrollResistance = Scr.MoveResistance = 0;
    Scr.OpaqueSize = 5;
    Scr.ClickTime = 150;

    /* set major operating modes */
    Scr.flags = 0;
    Scr.NumBoxes = 0;

    Scr.randomx = Scr.randomy = 0;
    Scr.buttons2grab = 7;

    Scr.PagerFont.name = NULL;
    Scr.PagerFont.height = 0;
    Scr.PagerFont.y = 0;
    Scr.MwmPager = (MwmWindow *) 0;
    Scr.Pager_w = None;
    Scr.CPagerWin = None;
    Scr.InitFunction = NULL;
    Scr.RestartFunction = NULL;
    return;
}



/***********************************************************************
 *
 *  Procedure:
 *	Reborder - Removes mwm border windows
 *
 ************************************************************************/
void
Reborder(void)
{
    MwmWindow      *tmp;	/* temp mwm window structure */
    int             i;
    extern unsigned PopupCount;
    extern MenuRoot *PopupTable[MAXPOPUPS];

    /* put a border back around all windows */
    XGrabServer(dpy);

    if (Scr.Pager_w != None)
	XDestroyWindow(dpy, Scr.Pager_w);

    InstallWindowColormaps(&Scr.MwmRoot);	/* force reinstall */
    for (tmp = Scr.MwmRoot.next; tmp != NULL; tmp = tmp->next) {
	XUnmapWindow(dpy, tmp->frame);
	RestoreWithdrawnLocation(tmp, True);
	XDestroyWindow(dpy, tmp->frame);
    }

    for (i = 0; i < PopupCount; i++)
	if (PopupTable[i]->w != None)
	    XDestroyWindow(dpy, PopupTable[i]->w);

    XUngrabServer(dpy);
    XSetInputFocus(dpy, PointerRoot, RevertToPointerRoot, CurrentTime);
    XSync(dpy, 0);

}

/***********************************************************************
 *
 *  Procedure: NoisyExit
 *	Print error messages and die. (segmentation violation)
 *
 **********************************************************************/
void
NoisyExit(int nonsense)
{
    XErrorEvent     event;

    yyerror("Seg Fault");
    event.error_code = 0;
    event.request_code = 0;
    MwmErrorHandler(dpy, &event);

    /* Attempt to do a re-start of mwm */
    Done(0, NULL);
}





/***********************************************************************
 *
 *  Procedure:
 *	Done - cleanup and exit mwm
 *
 ***********************************************************************
 */
RETSIGTYPE
SigDone(int nonsense)
{
    Done(0, NULL);
#ifndef _HPUX_SOURCE
    return (RETSIGTYPE) 0;
#endif
}

void
Done(int restart, char *command)
{
    MoveViewport(0, 0, False);

    Reborder();

    if (restart) {
	SaveDesktopState();	/* I wonder why ... */

	/* Really make sure that the connection is closed and cleared! */
	XSelectInput(dpy, Scr.root_win, 0);
	XSync(dpy, 0);
	XCloseDisplay(dpy);

	{
	    char           *my_argv[10];
	    int             i, done, j;

	    i = 0;
	    j = 0;
	    done = 0;
	    while ((g_argv[j] != NULL) && (i < 8)) {
		if (strcmp(g_argv[j], "-multiscreen") != 0) {
		    my_argv[i] = g_argv[j];
		    i++;
		    j++;
		}
		else
		    j++;
	    }
	    while (i < 10)
		my_argv[i++] = NULL;

	    /* really need to destroy all windows, explicitly,
	     * not sleep, but this is adequate for now */
	    sleep(1);
	    ReapChildren();
	    execvp(command, my_argv);
	}
	fprintf(stderr, "MWM: Call of '%s' failed!!!!\n", command);
	execvp(g_argv[0], g_argv);	/* that _should_ work */
	fprintf(stderr, "MWM: Call of '%s' failed!!!!\n", g_argv[0]);
    }
    else {
	XCloseDisplay(dpy);
	exit(0);
    }
}

/***********************************************************************
 *
 *  Procedure:
 *	CatchRedirectError - Figures out if there's another WM running
 *
 ************************************************************************/
XErrorHandler
CatchRedirectError(Display * dpy, XErrorEvent * event)
{
    yyerror("another WM is running");
    exit(1);
}


/***********************************************************************
 *
 *  Procedure:
 *	MwmErrorHandler - displays info on internal errors
 *
 ************************************************************************/
XErrorHandler
MwmErrorHandler(Display * dpy, XErrorEvent * event)
{
    extern int      last_event_type;

    /* some errors are acceptable, mostly they're caused by 
     * trying to update a lost  window */
    if ((event->error_code == BadWindow) || (event->request_code == X_GetGeometry) ||
	(event->error_code == BadDrawable) || (event->request_code == X_SetInputFocus) ||
	(event->request_code == X_GrabButton) ||
	(event->request_code == X_ChangeWindowAttributes) ||
	(event->request_code == X_InstallColormap))
	return 0;


    yyerror("internal error");
    fprintf(stderr, "      Request %d, Error %d\n", event->request_code,
	    event->error_code);
    fprintf(stderr, "      EventType: %d", last_event_type);
    fprintf(stderr, "\n");
    return 0;
}

void
usage(void)
{
#define USAGE "Mwm Ver %s\n\nusage: mwm [-display dpy] [-debug] [-xrm resourcestring] [-multiscreen] [-name name] [-screens name [name [...]]]\n"

    fprintf(stderr, USAGE, VERSION);

}




/* the root window is surrounded by four window slices, which are InputOnly.
 * So you can see 'through' them, but they eat the input. An EnterEvent in
 * one of these windows causes a Paging. The windows have the according cursor
 * pointing in the pan direction or are hidden if there is no more panning
 * in that direction. This is mostly intended to get a panning even atop
 * of Motif applictions, which does not work yet. It seems Motif windows
 * eat all mouse events.
 *
 * Hermann Dunkel, HEDU, dunkel@cul-ipn.uni-kiel.de 1/94
 */

/***************************************************************************
 * checkPanFrames hides PanFrames if they are on the very border of the
 * VIRTUELL screen and EdgeWrap for that direction is off. 
 * (A special cursor for the EdgeWrap border could be nice) HEDU
 ****************************************************************************/
void
checkPanFrames(void)
{
    extern Bool     DoHandlePageing;
    int             wrapX = (Scr.flags & EdgeWrapX);
    int             wrapY = (Scr.flags & EdgeWrapY);

    /* Remove Pan frames if paging by edge-scroll is permanently or
     * temporarily disabled */
    if ((Scr.EdgeScrollY == 0) || (!DoHandlePageing)) {
	XUnmapWindow(dpy, Scr.PanFrameTop.win);
	Scr.PanFrameTop.isMapped = False;
	XUnmapWindow(dpy, Scr.PanFrameBottom.win);
	Scr.PanFrameBottom.isMapped = False;
    }
    if ((Scr.EdgeScrollX == 0) || (!DoHandlePageing)) {
	XUnmapWindow(dpy, Scr.PanFrameLeft.win);
	Scr.PanFrameLeft.isMapped = False;
	XUnmapWindow(dpy, Scr.PanFrameRight.win);
	Scr.PanFrameRight.isMapped = False;
    }
    if (((Scr.EdgeScrollX == 0) && (Scr.EdgeScrollY == 0)) || (!DoHandlePageing))
	return;

    /* LEFT, hide only if EdgeWrap is off */
    if (Scr.Vx == 0 && Scr.PanFrameLeft.isMapped && (!wrapX)) {
	XUnmapWindow(dpy, Scr.PanFrameLeft.win);
	Scr.PanFrameLeft.isMapped = False;
    }
    else if (Scr.Vx > 0 && Scr.PanFrameLeft.isMapped == False) {
	XMapRaised(dpy, Scr.PanFrameLeft.win);
	Scr.PanFrameLeft.isMapped = True;
    }
    /* RIGHT, hide only if EdgeWrap is off */
    if (Scr.Vx == Scr.VxMax && Scr.PanFrameRight.isMapped && (!wrapX)) {
	XUnmapWindow(dpy, Scr.PanFrameRight.win);
	Scr.PanFrameRight.isMapped = False;
    }
    else if (Scr.Vx < Scr.VxMax && Scr.PanFrameRight.isMapped == False) {
	XMapRaised(dpy, Scr.PanFrameRight.win);
	Scr.PanFrameRight.isMapped = True;
    }
    /* TOP, hide only if EdgeWrap is off */
    if (Scr.Vy == 0 && Scr.PanFrameTop.isMapped && (!wrapY)) {
	XUnmapWindow(dpy, Scr.PanFrameTop.win);
	Scr.PanFrameTop.isMapped = False;
    }
    else if (Scr.Vy > 0 && Scr.PanFrameTop.isMapped == False) {
	XMapRaised(dpy, Scr.PanFrameTop.win);
	Scr.PanFrameTop.isMapped = True;
    }
    /* BOTTOM, hide only if EdgeWrap is off */
    if (Scr.Vy == Scr.VyMax && Scr.PanFrameBottom.isMapped && (!wrapY)) {
	XUnmapWindow(dpy, Scr.PanFrameBottom.win);
	Scr.PanFrameBottom.isMapped = False;
    }
    else if (Scr.Vy < Scr.VyMax && Scr.PanFrameBottom.isMapped == False) {
	XMapRaised(dpy, Scr.PanFrameBottom.win);
	Scr.PanFrameBottom.isMapped = True;
    }
}

/****************************************************************************
 *
 * Gotta make sure these things are on top of everything else, or they
 * don't work!
 *
 ***************************************************************************/
void
raisePanFrames(void)
{
    if (Scr.PanFrameTop.isMapped)
	XRaiseWindow(dpy, Scr.PanFrameTop.win);
    if (Scr.PanFrameLeft.isMapped)
	XRaiseWindow(dpy, Scr.PanFrameLeft.win);
    if (Scr.PanFrameRight.isMapped)
	XRaiseWindow(dpy, Scr.PanFrameRight.win);
    if (Scr.PanFrameBottom.isMapped)
	XRaiseWindow(dpy, Scr.PanFrameBottom.win);
}

/****************************************************************************
 *
 * Creates the windows for edge-scrolling 
 *
 ****************************************************************************/
void
initPanFrames()
{
    XSetWindowAttributes attributes;	/* attributes for create */
    unsigned long   valuemask;

    attributes.event_mask = (EnterWindowMask | LeaveWindowMask |
			     VisibilityChangeMask);
    valuemask = (CWEventMask | CWCursor);

    attributes.cursor = Scr.MwmCursors[TOP_CURS];
    Scr.PanFrameTop.win =
	XCreateWindow(dpy, Scr.root_win,
		      0, 0,
		      Scr.d_width, PAN_FRAME_THICKNESS,
		      0,	/* no border */
		      CopyFromParent, InputOnly,
		      CopyFromParent,
		      valuemask, &attributes);
    attributes.cursor = Scr.MwmCursors[LEFT_CURS];
    Scr.PanFrameLeft.win =
	XCreateWindow(dpy, Scr.root_win,
		      0, PAN_FRAME_THICKNESS,
		      PAN_FRAME_THICKNESS,
		      Scr.d_height - 2 * PAN_FRAME_THICKNESS,
		      0,	/* no border */
		      CopyFromParent, InputOnly, CopyFromParent,
		      valuemask, &attributes);
    attributes.cursor = Scr.MwmCursors[RIGHT_CURS];
    Scr.PanFrameRight.win =
	XCreateWindow(dpy, Scr.root_win,
	      Scr.d_width - PAN_FRAME_THICKNESS, PAN_FRAME_THICKNESS,
		      PAN_FRAME_THICKNESS,
		      Scr.d_height - 2 * PAN_FRAME_THICKNESS,
		      0,	/* no border */
		      CopyFromParent, InputOnly, CopyFromParent,
		      valuemask, &attributes);
    attributes.cursor = Scr.MwmCursors[BOTTOM_CURS];
    Scr.PanFrameBottom.win =
	XCreateWindow(dpy, Scr.root_win,
		      0, Scr.d_height - PAN_FRAME_THICKNESS,
		      Scr.d_width, PAN_FRAME_THICKNESS,
		      0,	/* no border */
		      CopyFromParent, InputOnly, CopyFromParent,
		      valuemask, &attributes);
    Scr.PanFrameTop.isMapped = Scr.PanFrameLeft.isMapped =
	Scr.PanFrameRight.isMapped = Scr.PanFrameBottom.isMapped = False;

    Scr.usePanFrames = True;
}

/****************************************************************************
 *
 * Save Desktop State
 *
 ****************************************************************************/
void
SaveDesktopState()
{
    MwmWindow      *t;
    unsigned long   data[1];

    for (t = Scr.MwmRoot.next; t != NULL; t = t->next) {
	data[0] = (unsigned long) t->Desk;
	XChangeProperty(dpy, t->w, _XA_WM_DESKTOP, _XA_WM_DESKTOP, 32,
			PropModeReplace, (unsigned char *) data, 1);
    }

    data[0] = (unsigned long) Scr.CurrentDesk;
    XChangeProperty(dpy, Scr.root_win, _XA_WM_DESKTOP, _XA_WM_DESKTOP, 32,
		    PropModeReplace, (unsigned char *) data, 1);

    XSync(dpy, 0);
}

/**************************************************************************
 * 
 * Sleep for n microseconds
 *
 *************************************************************************/
void
sleep_a_little(int n)
{
    struct timeval  value;

    if (n <= 0)
	return;

    value.tv_usec = n % 1000000;
    value.tv_sec = n / 1000000;

    (void) select(1, 0, 0, 0, &value);
}
