/*
Copyright (c) 1991-1995 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: transprt.c,v 1.54 1996/02/24 04:26:55 janssen Exp $ */
/* Last edited by Mike Spreitzer December 7, 1995 9:26 am PST */

#define _POSIX_SOURCE


#include "iluntrnl.h"
#include "transprt.h"

struct transports_s {
  /* L1, L2, Main unconstrained */

  ilu_string      name;
  ilu_TransportInstantiator instantiator;
};

/* L1_sup < trmu; L2 unconstrained */

#ifdef TCPIP_TRANSPORT
extern ilu_TransportCreator 
_ilu_tcp_TransportCreator(ilu_TransportInfo tinfo,
			  ILU_ERRS((no_memory, inv_objref)) * err);
#endif /* TCPIP_TRANSPORT */

#ifdef UDPSOCKET_TRANSPORT
extern ilu_TransportCreator 
_ilu_udp_TransportCreator(ilu_TransportInfo tinfo,
			  ILU_ERRS((no_memory, inv_objref)) * err);
#endif /* UDPSOCKET_TRANSPORT */

#ifdef SUNRPCRM_TRANSPORT
extern ilu_TransportCreator 
_ilu_sunrpcrm_TransportCreator(ilu_TransportInfo tinfo,
			  ILU_ERRS((no_memory, inv_objref)) * err);
#endif /* def SUNRPCRM_TRANSPORT */

#ifdef SECURE_TRANSPORT
extern ilu_TransportCreator 
_ilu_security_TransportCreator(ilu_TransportInfo tinfo,
			       ILU_ERRS((no_memory, inv_objref)) * err);
#endif /* def SECURE_TRANSPORT */

#ifdef W3MUX_TRANSPORT
extern ilu_TransportCreator 
_ilu_w3mux_TransportCreator(ilu_TransportInfo tinfo,
			    ILU_ERRS((no_memory, inv_objref)) * err);
#endif /* W3MUX_TRANSPORT */

extern ilu_TransportCreator 
_ilu_inmem_TransportCreator(ilu_TransportInfo tinfo,
			  ILU_ERRS((no_memory, inv_objref)) * err);

/*L1, L2, Main unconstrained*/

static struct transports_s transports[MAX_TRANSPORTS] = {
#ifdef UDPSOCKET_TRANSPORT
  { "udp", _ilu_udp_TransportCreator },
#endif /* UDPSOCKET_TRANSPORT */
#ifdef TCPIP_TRANSPORT
  { "tcp", _ilu_tcp_TransportCreator },
#endif /* TCPIP_TRANSPORT */
#ifdef SUNRPCRM_TRANSPORT
  { "sunrpcrm", _ilu_sunrpcrm_TransportCreator },
#endif /* SUNRPCRM_TRANSPORT */
#ifdef W3MUX_TRANSPORT
  { "w3mux", _ilu_w3mux_TransportCreator },
#endif /* W3MUX_TRANSPORT */
#ifdef SECURE_TRANSPORT
  { "security", _ilu_security_TransportCreator },
#endif /* SECURE_TRANSPORT */
  { "inmem", _ilu_inmem_TransportCreator },
  { NIL, NULLFN } };

ILU_ERRS((TransportAlreadyRegistered, MaxCountExceeded))
ilu_RegisterTransport(char *name,
		      ilu_TransportInstantiator new_transport,
		      ilu_boolean override)
{
  int             i;
  ilu_Error       e;

  DEBUG(EXPORT_DEBUG, (stderr, "ilu_RegisterTransport (%s)\n", name));
  for (i = 0; i < MAX_TRANSPORTS && transports[i].name != NIL; i++) {
    if (strcmp(transports[i].name, name) == 0) {
      if (override)
	transports[i].instantiator = new_transport;
      else
	{
	  DEBUG(EXPORT_DEBUG,
		    (stderr, "ilu_RegisterTransport:  \"%s\" already registered.\n",
		     name));
	  return ILU_ERR_CONS3(TransportAlreadyRegistered, &e,
			       name, name,
			       old_transport, transports[i].instantiator,
			       new_transport, new_transport, e);
	}
    }
  }
  if (i < MAX_TRANSPORTS && transports[i].name == NIL) {
    transports[i].name = name;
    transports[i].instantiator = new_transport;
    if ((i + 1) < MAX_TRANSPORTS)
      transports[i + 1].name = NIL;
    return ILU_NO_ERR;
  } else
    {
      DEBUG(EXPORT_DEBUG,
		(stderr, "ilu_RegisterTransport:  too many transports"
		 "(%d) to register \"%s\"\n", MAX_TRANSPORTS, name));
      return ILU_ERR_CONS1(MaxCountExceeded, &e,
			   max_count, MAX_TRANSPORTS, e);
    }
}

static struct transports_s *
  FindTransport(ilu_string tinfo)
{
  ilu_integer     i, l1;
  char           *p;
  if (tinfo == NIL)
    return NIL;
  p = strchr(tinfo, '_');
  if (p == NIL)
    l1 = strlen(tinfo);
  else
    l1 = p - tinfo;
  for (i = 0; transports[i].name != NIL; i += 1)
    if (_ilu_casefree_ncmp(tinfo, transports[i].name, l1) == 0
	&& transports[i].name[l1] == 0)
      {
	DEBUG(CONNECTION_DEBUG,
		  (stderr, "(FindTransport)(%s) => %p\n",
		   tinfo, &transports[i]));
	return (&transports[i]);
      }
  DEBUG(CONNECTION_DEBUG,
	    (stderr, "(FindTransport)(%s) => NIL\n", tinfo));
  return (NIL);
}

/*L1_sup < trmu*/
ilu_TransportCreator
_ilu_GetTransportCreator(ilu_TransportInfo tinfo,
			 ILU_ERRS((no_memory, inv_objref)) * err)
{
  struct transports_s *p;
  ilu_TransportCreator tcr;

  if ((p = FindTransport(tinfo[0])) == NIL) {
    DEBUG(CONNECTION_DEBUG,
	  (stderr,
	   "transprt.c: Unable to find registered transport creator for info \"%s\".\n",
	   tinfo[0]));
    return ILU_ERR_CONS1(inv_objref, err, minor, ilu_iom_tc, NIL);
  }
  tcr = (*(p->instantiator)) (tinfo, err);
  DEBUG(CONNECTION_DEBUG, (stderr, "_ilu_GetTransportCreator (%s...) => %p\n",
			   tinfo[0], tcr));
  return tcr;
}

/*Main Invariant holds; L2 disjoint {conn's iomu, callmu}*/
ilu_boolean 
ilu_SetTransportInputHandler(ilu_Transport trans,
			     ilu_TransportInputHandler tih,
			     ilu_refany tih_rock,
			     ILU_ERRS((no_memory, internal,
				       no_resources)) * err)
{
  if (tih != NULLFN && trans->tr_inBuff != NIL &&
      trans->tr_inNext < trans->tr_inLimit) {
    (*tih) (tih_rock);
    return FALSE;
  } else
    return ((*trans->tr_class->tc_set_input_handler)
	    (trans, tih, tih_rock, err));
}

/*Main Invariant holds; L2 >= {conn's I/O mutex}*/
ilu_boolean
_ilu_transportWriteBytes(ilu_Transport self,
			 ilu_bytes buf,
			 ilu_cardinal bufLen,
			 ILU_ERRS((IoErrs)) * err)
{
  ilu_TransportClass tc = self->tr_class;
  return ((*tc->tc_write_bytes) (self, buf, bufLen, FALSE, err));
}

/*Main Invariant holds; L2 >= {conn's I/O mutex}*/
ilu_bytes
_ilu_transportGetOutputBuffer(ilu_Transport self,
			      ilu_cardinal len,
			      ILU_ERRS((IoErrs)) * err)
{
  if (len > 16)
    return ILU_ERR_CONS1(internal, err, minor, ilu_im_trBufSize, NIL);
  if (!(*self->tr_class->tc_write_bytes) (self, NIL, 0, FALSE, err))
    return FALSE;
  if (self->tr_outBuff == NIL || self->tr_outNext >= self->tr_outLimit
      || 16 > self->tr_outLimit - self->tr_outNext)
    return ILU_ERR_CONS1(internal, err, minor, ilu_im_tcBug, NIL);
  return (self->tr_outBuff + (self->tr_outNext += len) - len);
}

/*Main Invariant holds; L2 >= {conn's I/O mutex}*/
ilu_boolean
_ilu_transportReadBytes(ilu_Transport self,
			ilu_bytes buf,
			ilu_cardinal len,
			ILU_ERRS((IoErrs)) * err)
{
  ilu_cardinal    l1 = 0, l2;
  ilu_TransportReport rpt;
  ILU_CLER(*err);
  if (self->tr_inBuff != NIL) {
    l1 = MIN(len, self->tr_inLimit - self->tr_inNext);
    memcpy((void *) buf, (void *) (self->tr_inBuff + self->tr_inNext),
	   l1);
    self->tr_inNext += l1;
    buf += l1;
  }
  if (l1 == len)
    return TRUE;
  while (1) {
    l2 = (*self->tr_class->tc_read_bytes) (self, buf, len - l1, &rpt,
					   err);
    if (ILU_ERRNOK(*err))
      return FALSE;
    buf += l2;
    l1 += l2;
    if (l1 == len)
      return TRUE;
    if (rpt.tr_eof)
      return ILU_ERR_CONS1(comm_failure, err, minor, ilu_cfm_eof,
			   FALSE);
    else if (rpt.tr_eom)
      return ILU_ERR_CONS1(marshal, err, minor, ilu_mm_eom, FALSE);
    if (!(*self->tr_class->tc_wait_for_input) (self, NIL, err))
      return FALSE;
  }
}

/*Main Invariant holds; L2 >= {conn's I/O mutex}*/
ilu_bytes
_ilu_transportGetInputBuffer(ilu_Transport self,
			     ilu_cardinal len,
			     ILU_ERRS((IoErrs)) * err)
{
  if (len > 16)
    return ILU_ERR_CONS1(internal, err, minor, ilu_im_trBufSize, NIL);
  if (!transport_read_bytes(self, self->tr_tinBuff, len, err))
    return NIL;
  ILU_CLER(*err);
  return self->tr_tinBuff;
}

/*Main Invariant holds; L2 >= {conn's I/O mutex}*/
ilu_cardinal
_ilu_transportReadUpToBytes(ilu_Transport self,
			    ilu_bytes buf,
			    ilu_cardinal len,
			    ilu_TransportReport * rpt,
			    ILU_ERRS((IoErrs)) * err)
{
  ilu_cardinal    l1 = 0, l2;
  ILU_CLER(*err);
  rpt->tr_eom = rpt->tr_eof = FALSE;
  if (self->tr_inBuff != NIL) {
    l1 = MIN(len, self->tr_inLimit - self->tr_inNext);
    memcpy((void *) buf, (void *) (self->tr_inBuff + self->tr_inNext),
	   l1);
    self->tr_inNext += l1;
    buf += l1;
  }
  if (l1 < len) {
    ilu_cardinal    lreq = len - l1;
    l2 = (*self->tr_class->tc_read_bytes) (self, buf, lreq, rpt, err);
    l1 += l2;
    buf += l2;
    if (ILU_ERRNOK(*err))
      return l1;
  }
  return l1;
}

/*Main Invariant holds; L2 >= {conn's I/O mutex}*/
extern          ilu_boolean
_ilu_transportReadMessage(ilu_Transport self,
			  ilu_bytes * msg,
			  ilu_cardinal * msgLen,
			  ilu_TransportReport * rpt,
			  ILU_ERRS((IoErrs)) * err)
{
  ilu_cardinal    nread, pos = 0, len = 4096;
  ilu_bytes       buf = NIL;
  rpt->tr_eom = rpt->tr_eof = FALSE;
  DEBUG(CONNECTION_DEBUG, (stderr, "_ilu_transportReadMessage:  reading message from transport <%x>\n",
			   self));
  if (transport_begin_message (self, TRUE, err), ILU_ERRNOK(*err))
    return FALSE;
  while (!(rpt->tr_eom || rpt->tr_eof)) {
    if (buf) {
      len *= 2;
      buf = ilu_realloc(buf, len);
      if ((*self->tr_class->tc_wait_for_input) (self, NIL, err), ILU_ERRNOK(*err))
	return FALSE;
    } else
      buf = ilu_malloc(len);
    if (buf == NIL)
      return ILU_ERR_CONS1(no_memory, err, nbytes, len, FALSE);
    nread = transport_read_upto_bytes(self, buf + pos, len - pos,
				      rpt, err);
    DEBUG(CONNECTION_DEBUG, (stderr, "_ilu_transportReadMessage:  read %u bytes, eof=%s, eom=%s\n",
			     nread, rpt->tr_eof ? "true" : "false", rpt->tr_eom ? "true" : "false"));
    if (ILU_ERRNOK(*err))
      {
	ilu_free(buf);
	return FALSE;
      }
    pos += nread;
  }
  if (transport_end_message(self, FALSE, NIL, err), ILU_ERRNOK(*err))
    {
      ilu_free(buf);
      return FALSE;
    }
  DEBUG(CONNECTION_DEBUG, (stderr, "_ilu_transportReadMessage:  returning buf %p of %u bytes\n",
			   buf, pos));
  *msg = buf;
  *msgLen = pos;
  return TRUE;
}

ilu_boolean
  _ilu_CompareTinfo (ilu_TransportInfo t1, ilu_TransportInfo t2)
{
  int i;
  for (i = 0;  t1[i] != NIL && t2[i] != NIL;  i++)
    if (strcmp(t1[i], t2[i]) != 0)
      return ilu_FALSE;
  if (t1[i] != NIL || t2[i] != NIL)
    return ilu_FALSE;
  return ilu_TRUE;
}

ilu_TransportInfo
_ilu_CopyTinfo(ilu_TransportInfo tinfo, ILU_ERRS((no_memory)) * err)
{
  int             count;
  int             i, sz = 0;
  ilu_string      s;
  ilu_TransportInfo ans;

  for (s = tinfo[count = 0]; s != NIL; s = tinfo[++count])
    sz += strlen(s) + 1;
  sz += (sizeof(ilu_string) * (count + 1));
  if ((ans = (ilu_TransportInfo) ilu_MallocE(sz, err)) == NIL)
    return NIL;
  s = (char *) (ans + count + 1);
  for (i = 0; i < count; i++) {
    ans[i] = s;
    strcpy(ans[i], tinfo[i]);
    s += strlen(tinfo[i]) + 1;
  }
  ans[count] = NIL;
  ILU_CLER(*err);
  return (ans);
}

ilu_TransportInfo
_ilu_ConcatTinfo(ilu_string info, ilu_TransportInfo tinfo, ILU_ERRS((no_memory)) * err)
{
  ilu_cardinal    len, count;
  ilu_TransportInfo ans;
  ilu_string      p;

  for (len = strlen(info) + 1, count = 0; tinfo[count] != NIL; count++)
    len += (strlen(tinfo[count]) + 1);
  len += (sizeof(ilu_string) * (count + 2));
  if ((ans = (ilu_TransportInfo) ilu_MallocE(len, err)) == NIL)
    return NIL;
  ans[0] = p = (char *) (ans + count + 2);
  strcpy(ans[0], info);
  p += strlen(info) + 1;
  for (len = 0; tinfo[len] != NIL; len++) {
    ans[len + 1] = p;
    strcpy(p, tinfo[len]);
    p += strlen(tinfo[len]) + 1;
  }
  ans[len + 1] = NIL;
  ILU_CLER(*err);
  return ans;
}

ilu_string 
_ilu_StringifyTinfo(ilu_TransportInfo tinfo, ILU_ERRS((no_memory)) * err)
{
  int             i, len = 0;
  ilu_string      ans, p;

  for (i = 0; tinfo[i] != NIL; i++)
    len += strlen(tinfo[i]) + 1;
  if ((ans = (char *) ilu_MallocE(len, err)) == NIL)
    return NIL;
  for (i = 0, p = ans; tinfo[i] != NIL; i++) {
    strcpy(p, tinfo[i]);
    p += strlen(p);
    if (tinfo[i + 1] != NIL)
      *p++ = ILU_TINFO_DIVIDER;
  }
  *p = 0;
  return (ans);
}

#ifdef ENABLE_DEBUGGING
void
  _ilu_PrintTinfo (ilu_TransportInfo tinfo)
{
  int i;

  for (i = 0;  tinfo[i] != NIL;  i++)
    {
      ilu_DebugPrintf ("<%s>", tinfo[i]);
    }
}
#endif
