/*
Copyright (c) 1991-1996 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: identity.c,v 1.8 1996/02/23 03:45:29 janssen Exp $ */
/* Last edited by Mike Spreitzer November 17, 1995 7:23 am PST */

#include "iluntrnl.h"

#include "connect.h"
#include "vector.h"
#include "call.h"

struct _ilu_Passport_s {
  struct ilu_vector_s v;
};

/*Main Invariant holds; L2 >= {call's conn's callmu}*/

void _ilu_AddConnIdentities (ilu_Call call, ilu_Error *err)
{
  ilu_Error lerr;
  ilu_IdentityInfo i;
  ilu_Passport ppn = call->ca_caller;

  if (ppn == NIL)
    {
      ppn = ilu_CreatePassport(NIL, err);
      if (ILU_ERRNOK(*err)) return;
    }

  if (call_connection(call)->co_auth_info != NIL)
    {
      ilu_Passport pp = (ilu_Passport) (call_connection(call)->co_auth_info);
      ilu_IdentityInfo ident;

      if ((ident = ilu_FindIdentity(pp, ilu_ConnectionIdentity)) != NIL)
	{
	  ilu_AddIdentity (ppn, ident, err);
	  if (ILU_ERRNOK(*err))
	    {
	      ilu_DestroyPassport (ppn, &lerr);
	      ILU_HANDLED(lerr);
	      return;
	    }
	}

#ifdef SECURE_TRANSPORT
      if ((ident = ilu_FindIdentity(pp, ilu_GSSIdentity)) != NIL)
	{
	  ilu_AddIdentity (ppn, ident, err);
	  if (ILU_ERRNOK(*err))
	    {
	      ilu_DestroyPassport(ppn, &lerr);
	      ILU_HANDLED(lerr);
	      return;
	    }
	}
#endif

    }
  if (call->ca_caller == NIL)
    call->ca_caller = ppn;
  ILU_CLER(*err);
}

/*L1, L2, Main unconstrained*/

extern ilu_Passport
  ilu_CreatePassport (ilu_IdentityInfo info,
		      ILU_ERRS((no_memory)) *err)
/* creates and returns a passport, optionally containing the specified identity */
{
  ilu_Passport p = (ilu_Passport) _ilu_vector_new(2);
  if (p == NIL)
    return ILU_ERR_CONS1(no_memory, err, nbytes, sizeof(*p), NIL);
  else
    ILU_CLER(*err);
  if (info != NIL)
    {
      ilu_AddIdentity (p, info, err);
      if (ILU_ERRNOK(*err))
	{
	  ilu_free(p);
	  p = NIL;
	}
    }
  return p;
}

extern ilu_IdentityInfo
  ilu_CopyIdentity (const ilu_IdentityInfo info,
		    ILU_ERRS((no_memory, bad_param)) *err)
/* allocates and returns a copy of the ilu_IdentityInfo parameter */
{
  ilu_IdentityInfo i;

  i = (ilu_IdentityInfo) ilu_MallocE(sizeof(*i), err);
  if (ILU_ERRNOK(*err))
    return NIL;
  memset ((void *) i, 0, sizeof(*i));
  if (info->ii_type == ilu_NoIdentity)
    {
      i->ii_type = ilu_NoIdentity;
    }
#ifdef SECURE_TRANSPORT
  else if (info->ii_type == ilu_GSSIdentity)
    {
      i->ii_info.ii_GSSInfo = info->ii_info.ii_GSSInfo;
    }
#endif /* SECURE_TRANSPORT */
#ifdef SUNRPC_PROTOCOL
  else if (info->ii_type == ilu_SunRPCAuthUnixIdentity)
    {
      int n;

      i->ii_info.ii_SunRPCAuthUnixInfo.ii_UID = info->ii_info.ii_SunRPCAuthUnixInfo.ii_UID;
      i->ii_info.ii_SunRPCAuthUnixInfo.ii_GID = info->ii_info.ii_SunRPCAuthUnixInfo.ii_GID;
      i->ii_info.ii_SunRPCAuthUnixInfo.ii_hostname = ilu_StrdupE(info->ii_info.ii_SunRPCAuthUnixInfo.ii_hostname, err);
      if (ILU_ERRNOK(*err))
	{
	  ilu_free(i);
	  return NIL;
	}
      i->ii_info.ii_SunRPCAuthUnixInfo.ii_ngids = info->ii_info.ii_SunRPCAuthUnixInfo.ii_ngids;
      i->ii_info.ii_SunRPCAuthUnixInfo.ii_gids = ilu_MallocE(info->ii_info.ii_SunRPCAuthUnixInfo.ii_ngids * sizeof(ilu_shortcardinal), err);
      if (ILU_ERRNOK(*err))
	{
	  ilu_free(i);
	  return NIL;
	}
      for (n = 0;  n < (int) i->ii_info.ii_SunRPCAuthUnixInfo.ii_ngids;  n++)
	i->ii_info.ii_SunRPCAuthUnixInfo.ii_gids[n] = info->ii_info.ii_SunRPCAuthUnixInfo.ii_gids[n];
    }
#endif
  else if (info->ii_type == ilu_ConnectionIdentity)
    {
      i->ii_info.ii_ConnectionInfo = ilu_StrdupE(info->ii_info.ii_ConnectionInfo, err);
      if (ILU_ERRNOK(*err))
	{
	  ilu_free(i);
	  return NIL;
	}
    }
  else
    return ILU_ERR_CONS1(bad_param, err, minor, ilu_bpm_duh, NIL);

  return i;  
}

extern ilu_boolean
  ilu_AddIdentity (ilu_Passport p,
		   const ilu_IdentityInfo info,
		   ILU_ERRS((no_memory)) *err)
/* added identity to Passport.  Only one identity of each type is allowed.
   Returns ILU_ERROK() of the error parameter. */
{
  _ilu_vector_add((ilu_Vector) p, (ilu_refany) info);
  ILU_CLER(*err);
  return ILU_ERROK(*err);
}

extern ilu_IdentityInfo
  ilu_FindIdentity (ilu_Passport p,
		    ilu_IdentityType infotype)
/* return identity of specified type, if present.  Returns NIL if not present. */
{
  ilu_Vector v = (ilu_Vector) p;
  int i;

  for (i = 0;  i < v->ve_size;  i++)
    if (((ilu_IdentityInfo)(v->ve_elements[i]))->ii_type == infotype)
      return ((ilu_IdentityInfo)(v->ve_elements[i]));
  return NIL;
}

extern ilu_boolean
  ilu_DestroyPassport (ilu_Passport passport,
		       ilu_Error * err)
/* destroys the passport and frees any associated identities */
{
  ilu_IdentityInfo info;
  ilu_Vector vec = (ilu_Vector) passport;
  int n;

  ILU_CLER(*err);
  for (n = 0;  n < vec->ve_size;  n++)
    {
      info = (ilu_IdentityInfo)(vec->ve_elements[n]);
      if (!info->ii_owned_by_passport)
	continue;
      if (info->ii_type == ilu_NoIdentity)
	;
#ifdef SECURE_TRANSPORT
      else if (info->ii_type == ilu_GSSIdentity)
	;
#endif /* SECURE_TRANSPORT */
#ifdef SUNRPC_PROTOCOL
      else if (info->ii_type == ilu_SunRPCAuthUnixIdentity)
	{
	  ilu_free(info->ii_info.ii_SunRPCAuthUnixInfo.ii_hostname);
	  ilu_free(info->ii_info.ii_SunRPCAuthUnixInfo.ii_gids);
	}
#endif /* SUNRPC_PROTOCOL */
      else if (info->ii_type == ilu_ConnectionIdentity)
	{
	  ilu_free(info->ii_info.ii_ConnectionInfo);
	}
      else
	return ILU_ERR_CONS1(internal,err,minor,ilu_im_broken,ilu_FALSE);

      ilu_free(info);
    }
  if (vec->ve_elements != NIL)
    ilu_free(vec->ve_elements);
  ilu_free(passport);
  return ILU_ERROK(*err);
}
