;;;; 	copyright (C) 1995 Free Software Foundation, Inc.
;;;; 
;;;; This program is free software; you can redistribute it and/or modify
;;;; it under the terms of the GNU General Public License as published by
;;;; the Free Software Foundation; either version 2, or (at your option)
;;;; any later version.
;;;; 
;;;; This program is distributed in the hope that it will be useful,
;;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;;; GNU General Public License for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License
;;;; along with this software; see the file COPYING.  If not, write to
;;;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;;;; 


;;; 			{BASIC DEFINITIONS}
;;;
;;; These are definitions that are basic parts of libguile, but that
;;; are written in scheme instead of C.
;;;



;;; Parts of this file derived from
;;; 	"Init.scm", Scheme initialization code for SCM.
;;; 	Copyright (C) 1991, 1992, 1993, 1994 Aubrey Jaffer.
;;;


;;; {apply and call-with-current-continuation}
;;;
;;; These turn syntax, @apply and @call-with-current-continuation,
;;; into procedures.   But the compiler can't hack the syntactic
;;; forms, so post-pone their creation until load-time.
;;;
(eval '(set! apply
	     (lambda (fun . args)
	       (@apply fun (apply:nconc2last args)))))

(eval '(define (call-with-current-continuation proc)
	 (@call-with-current-continuation proc)))



;;; {Some Revised^2 Names That Changed}
;;;

(define < <?)
(define <= <=?)
(define = =?)
(define > >?)
(define >= >=?)

(define 1+
  (let ((+ +))
    (lambda (n) (+ n 1))))

(define -1+
  (let ((+ +))
    (lambda (n) (+ n -1))))


(define 1- -1+)
(define t #t)
(define nil #f)
(define sequence begin)


;;; {Integer Math}
;;;

(define (ipow-by-squaring x k acc proc)
  (cond ((zero? k) acc)
	((= 1 k) (proc acc x))
	(else (logical:ipow-by-squaring (proc x x)
					(quotient k 2)
					(if (even? k) acc (proc acc x))
					proc))))


;;; {Basic Port Code}
;;; 
;;; Specificly, the parts of the low-level port code that are written in 
;;; Scheme rather than C.
;;;

;; VMS does something strange when output is sent to both
;; CURRENT-OUTPUT-PORT and CURRENT-ERROR-PORT.
(case (software-type)
  ((VMS) (set-current-error-port (current-output-port))))

;; OPEN_READ, OPEN_WRITE, and OPEN_BOTH are used to request the proper
;; mode to open files in.  MSDOS does carraige return - newline
;; translation if not opened in `b' mode.
;;
(define OPEN_READ (case (software-type)
		    ((MS-DOS WINDOWS ATARIST) "rb")
		    (else "r")))
(define OPEN_WRITE (case (software-type)
		     ((MS-DOS WINDOWS ATARIST) "wb")
		     (else "w")))
(define OPEN_BOTH (case (software-type)
		    ((MS-DOS WINDOWS ATARIST) "r+b")
		    (else "r+")))


(define (open-input-file str)
  (or (open-file str OPEN_READ)
      (error "OPEN-INPUT-FILE couldn't find file " str)))

(define (open-output-file str)
  (or (open-file str OPEN_WRITE)
      (error "OPEN-OUTPUT-FILE couldn't find file " str)))

(define (open-io-file str) (open-file str OPEN_BOTH))
(define close-input-port close-port)
(define close-output-port close-port)
(define close-io-port close-port)

(define (call-with-input-file str proc)
  (let* ((file (open-input-file str))
	 (ans (proc file)))
    (close-input-port file)
    ans))

(define (call-with-output-file str proc)
  (let* ((file (open-output-file str))
	 (ans (proc file)))
    (close-output-port file)
    ans))

(define (with-input-from-port port thunk)
  (let* ((swaports (lambda () (set! port (set-current-input-port port)))))
    (dynamic-wind swaports thunk swaports)))

(define (with-output-to-port port thunk)
  (let* ((swaports (lambda () (set! port (set-current-output-port port)))))
    (dynamic-wind swaports thunk swaports)))

(define (with-error-to-port port thunk)
  (let* ((swaports (lambda () (set! port (set-current-error-port port)))))
    (dynamic-wind swaports thunk swaports)))

(define (with-input-from-file file thunk)
  (let* ((nport (open-input-file file))
	 (ans (with-input-from-port nport thunk)))
    (close-port nport)
    ans))

(define (with-output-to-file file thunk)
  (let* ((nport (open-output-file file))
	 (ans (with-output-to-port nport thunk)))
    (close-port nport)
    ans))

(define (with-error-to-file file thunk)
  (let* ((nport (open-output-file file))
	 (ans (with-error-to-port nport thunk)))
    (close-port nport)
    ans))


(define (output-port-width . arg) 80)

(define (output-port-height . arg) 24)

(define (with-input-from-string string thunk)
  (call-with-input-string string
   (lambda (p) (with-input-from-port p thunk))))

(define (with-output-to-string thunk)
  (call-with-input-string
   (lambda (p) (with-output-to-port p thunk))))

(define (with-error-to-string thunk)
  (call-with-input-string
   (lambda (p) (with-error-to-port p thunk))))



;;; {Symbol Properties}
;;;

(define (symbol-property sym prop)
  (let ((pair (assoc prop (symbol-pref sym))))
    (and pair (cdr pair))))

(define (set-symbol-property! sym prop val)
  (let ((pair (assoc prop (symbol-pref sym))))
    (if pair
	(set-cdr! pair val)
	(symbol-pset! sym (acons prop val (symbol-pref sym))))))

(define (set-symbol-property! sym prop val)
  (let ((pair (assoc prop (symbol-pref sym))))
    (if pair
	(set-cdr! pair val)
	(symbol-pset! sym (acons prop val (symbol-pref sym))))))

(define (symbol-property-remove! sym prop)
  (let ((pair (assoc prop (symbol-pref sym))))
    (if pair
	(symbol-pset! sym (delq! pair (symbol-pref sym))))))


;;; {Assoc List Functions}
;;;

(define (predicate->asso pred)
  (cond ((eq? eq? pred) assq)
	((eq? = pred) assv)
	((eq? eqv? pred) assv)
	((eq? char=? pred) assv)
	((eq? equal? pred) assoc)
	((eq? string=? pred) assoc)
	(else (lambda (key alist)
		(let l ((al alist))
		  (cond ((null? al) #f)
			((pred key (caar al)) (car al))
			(else (l (cdr al)))))))))

(define (alist-inquirer pred)
  (let ((assofun (predicate->asso pred)))
    (lambda (alist key)
      (let ((pair (assofun key alist)))
	(and pair (cdr pair))))))

(define (alist-associator pred)
  (let ((assofun (predicate->asso pred)))
    (lambda (alist key val)
      (let* ((pair (assofun key alist)))
	(cond (pair (set-cdr! pair val)
		    alist)
	      (else (cons (cons key val) alist)))))))

(define (alist-remover pred)
  (lambda (alist key)
    (cond ((null? alist) alist)
	  ((pred key (caar alist)) (cdr alist))
	  ((null? (cdr alist)) alist)
	  ((pred key (caadr alist))
	   (set-cdr! alist (cddr alist)) alist)
	  (else
	   (let l ((al (cdr alist)))
	     (cond ((null? (cdr al)) alist)
		   ((pred key (caadr al))
		    (set-cdr! al (cddr al)) alist)
		   (else (l (cdr al)))))))))

(define (alist-map proc alist)
  (map (lambda (pair) (cons (car pair) (proc (car pair) (cdr pair))))
       alist))

(define (alist-for-each proc alist)
  (for-each (lambda (pair) (proc (car pair) (cdr pair))) alist))


;;; {Hash Table Functions}
;;;

(define (predicate->hash pred)
  (cond ((eq? pred eq?) hashq)
	((eq? pred eqv?) hashv)
	((eq? pred equal?) hash)
	((eq? pred =) hashv)
	((eq? pred char=?) hashv)
	((eq? pred char-ci=?) hashv)
	((eq? pred string=?) hash)
	((eq? pred string-ci=?) hash)
	(else (slib:error "unknown predicate for hash" pred))))

(define (make-hash-table k) (make-vector k '()))

(define (predicate->hash-asso pred)
  (let ((hashfun (predicate->hash pred))
	(asso (predicate->asso pred)))
    (lambda (key hashtab)
      (asso key
	    (vector-ref hashtab (hashfun key (vector-length hashtab)))))))

(define (hash-inquirer pred)
  (let ((hashfun (predicate->hash pred))
	(ainq (alist-inquirer pred)))
    (lambda (hashtab key)
      (ainq (vector-ref hashtab (hashfun key (vector-length hashtab)))
	    key))))

(define (hash-associator pred)
  (let ((hashfun (predicate->hash pred))
	(asso (alist-associator pred)))
    (lambda (hashtab key val)
      (let* ((num (hashfun key (vector-length hashtab))))
	(vector-set! hashtab num
		     (asso (vector-ref hashtab num) key val)))
      hashtab)))

(define (hash-remover pred)
  (let ((hashfun (predicate->hash pred))
	(arem (alist-remover pred)))
    (lambda (hashtab key)
      (let* ((num (hashfun key (vector-length hashtab))))
	(vector-set! hashtab num
		     (arem (vector-ref hashtab num) key)))
      hashtab)))

(define (hash-map proc ht)
  (define nht (make-vector (vector-length ht)))
  (do ((i (+ -1 (vector-length ht)) (+ -1 i)))
      ((negative? i) nht)
    (vector-set!
     nht i
     (alist-map proc (vector-ref ht i)))))

(define (hash-for-each proc ht)
  (do ((i (+ -1 (vector-length ht)) (+ -1 i)))
      ((negative? i))
    (alist-for-each proc (vector-ref ht i))))


;;;  {Shorthand for Medium-Sized equal?-based Hash Tables}
;;;
(define aref (hash-inquirer equal?))
(define aremove (hash-remover equal?))
(define aset! (hash-associator equal?))
(define (make-table) (make-hash-table 64))



;;; {Arrays}
;;;

(begin
  (define uniform-vector? array?)
  (define make-uniform-vector dimensions->uniform-array)
  ;      (define uniform-vector-ref array-ref)
  (define (uniform-vector-set! u i o)
    (uniform-vector-set1! u o i))
  (define uniform-vector-fill! array-fill!)
  (define uniform-vector-read! uniform-array-read!)
  (define uniform-vector-write uniform-array-write)

  (define (make-array fill . args)
    (dimensions->uniform-array args () fill))
  (define (make-uniform-array prot . args)
    (dimensions->uniform-array args prot))
  (define (list->array ndim lst)
    (list->uniform-array ndim '() lst))
  (define (list->uniform-vector prot lst)
    (list->uniform-array 1 prot lst))
  (define (array-shape a)
    (map (lambda (ind) (if (number? ind) (list 0 (+ -1 ind)) ind))
	 (array-dimensions a))))


;;; {Lvectors}
;;;

;; These are the offsets of hook functions within a type lvector.
;; They must agree with declarations in scm.h (bleah)
;;
(define lvector-hook-user-data 1)
(define lvector-hook-ref-fn 2)
(define lvector-hook-set-fn 3)
(define lvector-hook-print-fn 4)
(define lvector-hook-equal-fn 5)
(define lvector-hook-isa-fn 6)

(define lvector-hook-list '(lvector-hook-user-data
			    lvector-hook-ref-fn
			    lvector-hook-set-fn
			    lvector-hook-print-fn
			    lvector-hook-equal-fn
			    lvector-hook-isa-fn))

			    

(define lvector-hook-slots 7)
(define lvector-ref lvector-refq)


;;; {Keywords}
;;;

(define make-kw make-keyword)

(define (symbol->keyword symbol)
  (make-keyword (symbol-append '- symbol)))
(define (keyword->symbol kw)
  (let ((sym (keyword-symbol kw)))
    (string->symbol (substring sym 1 (length sym)))))

(define (kw-arg-ref args kw)
  (let ((rem (member kw args)))
    (and rem (pair? (cdr rem)) (cadr rem))))


;;; {Records}
;;;


(define record? #f)
(define record-type? #f)
(define make-record-type #f)
(define record-constructor #f)
(define record-predicate #f)
(define record-accessor #f)
(define record-modifier #f)
(define record-type-descriptor #f)
(define record-type-name #f)
(define record-type-field-names #f)
(define record-type-print-fn #f)

(let* ((hook-fields (map (lambda (f)
			    `(,f :lvector-key ,lvector-ref))
			 lvector-hook-list))
       (additional-fields (map list '(name
				      fields
				      keyvec
				      print-fn)))
       (record-type-fields (append! hook-fields additional-fields))
       (print-record (lambda (rec port wr?)
			(display "#<record ")
			(display (record-type-name (record-type-descriptor rec)))
			(display " ")
			(display (string->number (object-address rec) 16))
			(display ">")))
       (print-record-lvector (lambda (lvec port wr?)
				((or (record-type-print-fn (record-type-descriptor lvec)) print-record)
				 lvec port wr?)))
       (field-spec->key (lambda (fs)
			   (cond
			    ((symbol? fs)  (string->obarray-symbol #f fs))
			    ((list? fs)	   (or (kw-arg-ref fs :lvector-key )
					       (string->obarray-symbol #f (car fs))))
			    (else (throw 'bad-record-field fs)))))
       (field-spec->name (lambda (fs)
			    (cond
			     ((symbol? fs) fs)
			     ((list? fs) (car fs)))))
       (field-pos (lambda (fields name)
		     (let loop ((n 1)
				(fields fields))
		       (and (not (null? fields))
			    (if (eq? name (field-spec->name (car fields)))
				n
				(loop (+ n 1) (cdr fields)))))))
       (record-type-keys (map field-spec->key record-type-fields))
       (record-type-keyvector (apply vector #f record-type-keys))
       (instantiate-record (lambda (keys . inits)
			      (lock-vector!
			       (make-vector (length keys) (cons keys inits) #t))))
       (getter (lambda (fields keyvector name)
		  (let ((pos (field-pos fields name)))
		    (if (not pos)
			(throw 'no-such-record-field name))
		    (lvector-accessor keyvector pos))))
       (setter (lambda (fields keyvector name)
		  (let ((pos (field-pos fields name)))
		    (if (not pos)
			(throw 'no-such-record-field name))
		    (lvector-modifier keyvector pos))))
       (-type-fields (getter record-type-fields record-type-keyvector 'fields))
       (-type-name (getter record-type-fields record-type-keyvector 'name))
       (-type-keyvector (getter record-type-fields record-type-keyvector 'keyvec))
       (-set-type-keyvector! (setter record-type-fields record-type-keyvector 'keyvec))
       (-type-print-fn (getter record-type-fields record-type-keyvector 'print-fn))
       (-type-type (lock-vector!
		    (make-vector (length record-type-keys)
				 (list record-type-keyvector
				       #f #f #f print-record-lvector #f #f
				       '<record>
				       record-type-fields
				       record-type-keyvector
				       #f)
				 #t)))
       (-side-effect-1 (vector-set! record-type-keyvector 0 -type-type))
       (fields->keyvector (lambda (rtd fields)
			     (apply vector rtd
				    (map field-spec->key fields))))
       (-make-record-type (lambda (name fields print-fn)
			     (let ((rtd
				    (instantiate-record record-type-keyvector
							#f #f #f print-record-lvector #f #f
							name
							fields
							#f
							print-fn)))
			       (-set-type-keyvector! rtd (fields->keyvector rtd fields))
			       rtd)))
       (-make-record-type-opt (lambda (name fields . opts)
				 (let ((print-fn (and opts (car opts))))
				   (-make-record-type name fields print-fn))))
       (-record-type? (lambda (obj)
			 (and (lvector? obj)
			      (eq? (vector-ref (lvector-peek obj 0) 0)
				   -type-type))))
       (-record? (lambda (obj)
		    (and (lvector? obj)
			 (-record-type? (vector-ref (lvector-peek obj 0) 0)))))
       (-constructor (lambda (rtd field-names)
			(if (not (record-type? rtd))
			    (throw 'bad-record-type rtd))
			(let ((initters
			       (map (lambda (n) (setter (-type-fields rtd)
						   (-type-keyvector rtd)
						   n))
				    field-names))
			      (n-initters (length field-names)))
			  (lambda args
			   (if (not (= (length args) n-initters))
			       (throw 'wrong-number-of-record-inits rtd args))
			   (let ((obj (instantiate-record (-type-keyvector rtd))))
			     (for-each (lambda (f a) (f obj a))
				       initters args)
			     obj)))))
       (-constructor-opt (lambda (rtd . opt)
			    (let ((field-names (if opt
						   (car opt)
						   (-type-fields rtd))))
			      (-constructor rtd field-names))))
       (-predicate (lambda (rtd)
		      (lambda (obj)
			 (and (lvector? obj)
			      (eq? rtd (vector-ref (lvector-peek obj 0) 0))))))
       (-accessor (lambda (rtd name)
		     (getter (-type-fields rtd)
			     (-type-keyvector rtd)
			     name)))
       (-modifier (lambda (rtd name)
		     (setter (-type-fields rtd)
			     (-type-keyvector rtd)
			     name)))
       (-rtd (lambda (obj) (if (-record? obj)
			  (vector-ref (lvector-peek obj 0) 0)
			  (throw 'not-a-record obj))))
       (-type-field-names (lambda (rtd)
			     (map field-spec->name
				  (-type-fields rtd)))))
  (set! record? -record?)
  (set! record-type? -record-type?)
  (set! make-record-type -make-record-type-opt)
  (set! record-constructor -constructor-opt)
  (set! record-predicate -predicate)
  (set! record-accessor -accessor)
  (set! record-modifier -modifier)
  (set! record-type-descriptor -rtd)
  (set! record-type-name -type-name)
  (set! record-type-field-names -type-field-names)
  (set! record-type-print-fn -type-print-fn))




;;; {Symbols}
;;;

(define (symbol-append . args)
  (string->symbol (apply string-append args)))

(define (obarray-symbol-append ob . args)
  (string->obarray-symbol (apply string-append args)))


;;; {Lists}
;;;

(define (list-index l k)
  (let loop ((n 0)
	     (l l))
    (and (not (null? l))
	 (if (eq? (car l) k)
	     n
	     (loop (+ n 1) (cdr l))))))

(define (make-list n init)
  (let loop ((answer '())
	     (n n))
    (if (<= n 0)
	answer
	(loop (cons init (answer)) (- n 1)))))



;;; {and-map, or-map, and map-in-order}
;;;
;;; (and-map fn lst) is like (and (fn (car lst)) (fn (cadr lst)) (fn...) ...)
;;; (or-map fn lst) is like (or (fn (car lst)) (fn (cadr lst)) (fn...) ...)
;;; (map-in-order fn lst) is like (map fn lst) but definately in order of lst.
;;;

;; and-map f l
;;
;; Apply f to successive elements of l until exhaustion or f returns #f.
;; If returning early, return #f.  Otherwise, return the last value returned
;; by f.  If f has never been called because l is empty, return #t.
;; 
(define (and-map f lst)
  (let loop ((result #t)
	     (l lst))
    (and result
	 (or (and (null? l)
		  result)
	     (loop (f (car l)) (cdr l))))))

;; or-map f l
;;
;; Apply f to successive elements of l until exhaustion or while f returns #f.
;; If returning early, return the return value of f.
;;
(define (or-map f lst)
  (let loop ((result #f)
	     (l lst))
    (or result
	(and (not (null? l))
	     (loop (f (car l)) (cdr l))))))

;; map-in-order
;;
;; Like map, but guaranteed to process the list in order.
;;
(define (map-in-order fn l)
  (if (null? l)
      '()
      (cons (fn (car l))
	    (map-in-order fn (cdr l)))))


;;; {Sequences}
;;;


(define (nth obj n)
  (cond
   ((vector? obj) (vector-ref obj n))
   ((list? obj) (list-index obj n))
   ((or (string? obj) (symbol? obj)) (string-ref obj n))))



;;; {Generalized Eval From File (for slib)}
;;;

(define (eval-load <filename> evl)
  (if (not (file-exists? <filename>))
      (set! <filename> (string-append <filename> (scheme-file-suffix))))
  (call-with-input-file <filename>
    (lambda (port)
      (let ((old-load-pathname *load-pathname*))
	(set! *load-pathname* <filename>)
	(do ((o (read port) (read port)))
	    ((eof-object? o))
	  (evl o))
	(set! *load-pathname* old-load-pathname)))))


;;; {Time}
;;;

(define difftime -)
(define offset-time +)


;;; {Files}
;;;

(define (file-exists? str)
  (let ((port (open-file str OPEN_READ)))
    (if port (begin (close-port port) #t)
	#f)))

(define (has-suffix? str suffix)
  (let ((sufl (string-length suffix))
	(sl (string-length str)))
    (and (> sl sufl)
	 (string=? (substring str (- sl sufl) sl) suffix))))




;;; {Error Handling}
;;;


;; (error . args) is short for (throw (quote error) . args)
;;
(define (error . args)
  (apply throw 'error args))




;; Error handling a la SCM.
;;
(define (%%default-error-handler ignored . args)
  (define cep (current-error-port))
  (perror "ERROR")
  (errno 0)
  (display "ERROR: " cep)
  (if (not (null? args))
      (begin (display (car args) cep)
	     (for-each (lambda (x) (display #\  cep) (write x cep))
		       (cdr args))))
  (newline cep)
  (force-output cep)
  (abort))



;; Install SCM error handling as the default.
;;
(set-symbol-property! 'error
		      'throw-handler-default
		      %%default-error-handler)





;; %%bad-throw is the hook that is called upon a throw to a an unhandled
;; key.  If the key has a default handler (a throw-handler-default property),
;; it is applied to the throw.
;;
(define (%%bad-throw key . args)
  (let ((default (symbol-property key 'throw-handler-default)))
    (and default (apply default key args))))





;; A number of internally defined error types are represented
;; as integers.  Here is the mapping to symbolic names
;; and error messages.
;;
(define %%system-errors
  '((-1 UNKNOWN "Unknown error")
    (0 ARGn  "Wrong type argument to ")
    (1 ARG1  "Wrong type argument in position 1 to ")
    (2 ARG2  "Wrong type argument in position 2 to ")
    (3 ARG3  "Wrong type argument in position 3 to ")
    (4 ARG4  "Wrong type argument in position 4 to ")
    (5 ARG5  "Wrong type argument in position 5 to ")
    (6 ARG5  "Wrong type argument in position 5 to ")
    (7 ARG5  "Wrong type argument in position 5 to ")
    (8 WNA "Wrong number of arguments to ")
    (9 OVFLOW "Numerical overflow to ")
    (10 OUTOFRANGE "Argument out of range to ")
    (11 NALLOC "Could not allocate to ")
    (12 EXIT "Exit (internal error?).")
    (13 HUP_SIGNAL "hang-up")
    (14 INT_SIGNAL "user interrupt")
    (15 FPE_SIGNAL "arithmetic error")
    (16 BUS_SIGNAL "bus error")
    (17 SEGV_SIGNAL "segmentation violation")
    (18 ALRM_SIGNAL "alarm")))



;; The default handler for built-in error types when
;; thrown by their symbolic name.  The action is to 
;; convert the error into a generic error, building
;; a descriptive message for the error.
;;
(define (%%handle-system-error ignored desc proc . args)
  (let* ((b (assoc desc %%system-errors))
	 (msghead (cond
		   (b (caddr b))
		   ((or (symbol? desc) (string? desc))
		    (string-append desc " "))
		   (#t "Unknown error")))
	 (msg (if (symbol? proc)
		  (string-append msghead proc ":")
		  msghead))
	 (rest (if (and proc (not (symbol? proc)))
		   (cons proc args)
		   args))
	 (fixed-args (cons msg rest)))
    (apply error fixed-args)))


(set-symbol-property! '%%system-error
		      'throw-handler-default
		      %%handle-system-error)


;; Install default handlers for built-in errors.
;;
(map (lambda (err)
       (set-symbol-property! (cadr err)
			     'throw-handler-default
			     %%handle-system-error))
     (cdr %%system-errors))



;;; {Load Paths}
;;;

;; library-vicinity should return the pathname of the
;; directory where files of Scheme library functions reside.
;;
(define library-vicinity
  (let ((library-path
	 (or (getenv "SCHEME_LIBRARY_PATH")
	     (case (software-type)
	       ((UNIX COHERENT) (compiled-library-path))
	       ((VMS) "lib$scheme:")
	       ((MS-DOS WINDOWS ATARIST) "C:\\SCM\\SLIB\\")
	       ((OS/2) "\\languages\\scm\\slib\\")
	       ((MACOS THINKC) "camus Napoleon:Think C4.0:scm3.0:")
	       ((AMIGA) "Scheme:libs/")
	       (else "")))))

    (lambda () library-path)))

;;; Here for backward compatability
;;
(define scheme-file-suffix
  (case (software-type)
    ((NOSVE) (lambda () "_scm"))
    (else (lambda () ".scm"))))

(define in-vicinity string-append)

(define (parse-path env_path)
  (cond ((string? env_path)
	 (let loop ((curdir "") (env env_path) (path '()))
	   (cond ((= (string-length env) 0) 
		  (if (> (string-length curdir) 0)
		      (append path (list curdir))
		      path))
		 ((char=? (string-ref env 0) #\:)
		  (loop "" 
			(substring env 1 (string-length env))
			(append path (list curdir))))
		 (#t
		  (loop (string-append curdir (substring env 0 1))
			(substring env 1 (string-length env))
			path)))))
	(#t '())))

(define %load-path (append (parse-path (getenv "SCHEME_LOAD_PATH"))
			   (list ""
				 (library-vicinity)
				 (in-vicinity (library-vicinity) "gls/")
				 (in-vicinity (library-vicinity) "slib/"))))

;;; {try-load}
;;;


(define (try-load-with-path file-name path)
  (or-map (lambda (d)
	    (%try-load (in-vicinity d file-name)))
	  path))

(define (try-load name)
  (if (eval '(defined? %load-path))
      (try-load-with-path name (eval '%load-path))
      (%try-load name)))

;;; {Load}
;;;

(define %load-verbosely #t)
(define (assert-load-verbosity v) (set! %load-verbosely v))
(define %load-indent -2)

(define (%load f)
  (or (%try-load f)
      (and (not (has-suffix? f (scheme-file-suffix)))
	   (%try-load (string-append f (scheme-file-suffix))))))

(define (%load-announce file)
  (if %load-verbosely
      (begin
	(display ";;; " (current-error-port))
	(display (make-string %load-indent #\ ))
	(display "loading ")
	(display file)
	(display "...")
	(newline)
	(force-output))))

(define (%load-announce-win file)
  (if %load-verbosely
      (begin
	(display ";;; " (current-error-port))
	(display (make-string %load-indent #\ ))
	(display "...loaded ")
	(display file)
	(display ".")
	(newline)
	(force-output))))

(define (%load-announce-lossage file path)
  (if %load-verbosely
      (begin
	(display ";;; " (current-error-port))
	(display (make-string %load-indent #\ ))
	(display "...COULD NOT LOAD ")
	(display file)
	(display " from ")
	(write path)
	(newline)
	(force-output)))
  
  (throw 'could-not-load file path))


(define (load-with-path name path)
  (define (do-load)
    (%load-announce name)
    (if (not (or-map (lambda (d)
		       (if (%load (in-vicinity d name))
			   (begin
			     (%load-announce-win (in-vicinity d name))
			     #t)
			   #f))
		     path))
	(%load-announce-lossage name path)))

  (let ((indent %load-indent))
    (dynamic-wind
     (lambda () (set! %load-indent (modulo (+ indent 2) 16)))
     do-load
     (lambda () (set! %load-indent indent)))))


(define (load name)
  (if (eval '(defined? %load-path))
      (load-with-path name (eval '%load-path))
      (load-with-path name '("."))))



;;; {Transcendental Functions}
;;;
;;; Derived from "Transcen.scm", Complex trancendental functions for SCM.
;;; Copyright (C) 1992, 1993 Jerry D. Hedden.
;;; See the file `COPYING' for terms applying to this program.
;;;

(define (exp z)
  (if (real? z) ($exp z)
      (make-polar ($exp (real-part z)) (imag-part z))))

(define (log z)
  (if (and (real? z) (>= z 0))
      ($log z)
      (make-rectangular ($log (magnitude z)) (angle z))))

(define (sqrt z)
  (if (real? z)
      (if (negative? z) (make-rectangular 0 ($sqrt (- z)))
	  ($sqrt z))
      (make-polar ($sqrt (magnitude z)) (/ (angle z) 2))))

(define expt
  (let ((integer-expt integer-expt))
    (lambda (z1 z2)
      (cond ((exact? z2)
	     (integer-expt z1 z2))
	    ((and (real? z2) (real? z1) (>= z1 0))
	     ($expt z1 z2))
	    (else
	     (exp (* z2 (log z1))))))))

(define (sinh z)
  (if (real? z) ($sinh z)
      (let ((x (real-part z)) (y (imag-part z)))
	(make-rectangular (* ($sinh x) ($cos y))
			  (* ($cosh x) ($sin y))))))
(define (cosh z)
  (if (real? z) ($cosh z)
      (let ((x (real-part z)) (y (imag-part z)))
	(make-rectangular (* ($cosh x) ($cos y))
			  (* ($sinh x) ($sin y))))))
(define (tanh z)
  (if (real? z) ($tanh z)
      (let* ((x (* 2 (real-part z)))
	     (y (* 2 (imag-part z)))
	     (w (+ ($cosh x) ($cos y))))
	(make-rectangular (/ ($sinh x) w) (/ ($sin y) w)))))

(define (asinh z)
  (if (real? z) ($asinh z)
      (log (+ z (sqrt (+ (* z z) 1))))))

(define (acosh z)
  (if (and (real? z) (>= z 1))
      ($acosh z)
      (log (+ z (sqrt (- (* z z) 1))))))

(define (atanh z)
  (if (and (real? z) (> z -1) (< z 1))
      ($atanh z)
      (/ (log (/ (+ 1 z) (- 1 z))) 2)))

(define (sin z)
  (if (real? z) ($sin z)
      (let ((x (real-part z)) (y (imag-part z)))
	(make-rectangular (* ($sin x) ($cosh y))
			  (* ($cos x) ($sinh y))))))
(define (cos z)
  (if (real? z) ($cos z)
      (let ((x (real-part z)) (y (imag-part z)))
	(make-rectangular (* ($cos x) ($cosh y))
			  (- (* ($sin x) ($sinh y)))))))
(define (tan z)
  (if (real? z) ($tan z)
      (let* ((x (* 2 (real-part z)))
	     (y (* 2 (imag-part z)))
	     (w (+ ($cos x) ($cosh y))))
	(make-rectangular (/ ($sin x) w) (/ ($sinh y) w)))))

(define (asin z)
  (if (and (real? z) (>= z -1) (<= z 1))
      ($asin z)
      (* -i (asinh (* +i z)))))

(define (acos z)
  (if (and (real? z) (>= z -1) (<= z 1))
      ($acos z)
      (+ (/ (angle -1) 2) (* +i (asinh (* +i z))))))

(define (atan z . y)
  (if (null? y)
      (if (real? z) ($atan z)
	  (/ (log (/ (- +i z) (+ +i z))) +2i))
      ($atan2 z (car y))))

(set! abs magnitude)


;;; {User Settable Hooks}
;;;
;;; Parts of the C code check the bindings of these variables.
;;; 

(define ticks-interrupt #f)
(define user-interrupt #f)
(define alarm-interrupt #f)
(define out-of-storage #f)
(define could-not-open #f)
(define end-of-program #f)
(define hang-up #f)
(define arithmetic-error #f)
(define read:sharp #f)



;;; {Reader Extensions}
;;;

;;; Reader code for various "#c" forms.
;;;

(define (%read-sharp c port)
  (define (barf)
    (error "unknown # object" c))

  (case c
    ((#\') (read port))
    ((#\.) (eval (read port)))
    ((#\b) (read:uniform-vector #t port))
    ((#\a) (read:uniform-vector #\a port))
    ((#\u) (read:uniform-vector 1 port))
    ((#\e) (read:uniform-vector -1 port))
    ((#\s) (read:uniform-vector 1.0 port))
    ((#\i) (read:uniform-vector 1/3 port))
    ((#\c) (read:uniform-vector 0+i port))
    ((#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)
     (read:array c port))
    ((#\!) (if (= 1 (line-number))
	       (let skip () (if (eq? #\newline (peek-char port))
				(read port)
				(begin (read-char port) (skip))))
	       (barf)))
    (else (barf))))

(define (read:array digit port)
  (define chr0 (char->integer #\0))
  (let ((rank (let readnum ((val (- (char->integer digit) chr0)))
		(if (char-numeric? (peek-char port))
		    (readnum (+ (* 10 val)
				(- (char->integer (read-char port)) chr0)))
		    val)))
	(prot (if (eq? #\( (peek-char port))
		  '()
		  (let ((c (read-char port)))
		    (case c ((#\b) #t)
			  ((#\a) #\a)
			  ((#\u) 1)
			  ((#\e) -1)
			  ((#\s) 1.0)
			  ((#\i) 1/3)
			  ((#\c) 0+i)
			  (else (error "read:array unknown option " c)))))))
    (if (eq? (peek-char port) #\()
	(list->uniform-array rank prot (read port))
	(error "read:array list not found"))))

(define (read:uniform-vector proto port)
  (if (eq? #\( (peek-char port))
      (list->uniform-array 1 proto (read port))
      (error "read:uniform-vector list not found")))


(define read:sharp %read-sharp)



;;; {Dynamic Roots}
;;;

; mystery integers passed dynamic root error handlers
(define repl-quit -1)
(define repl-abort -2)

(define (with-dynamic-root-handlers start-off handle-quit handle-abort)
  (with-dynamic-root
   start-off
   (lambda (errcode)
     (cond
       ((eq? repl-quit errcode) (handle-quit))
       ((eq? repl-abort errcode) (handle-abort))))))



;;; {Command Line Options}
;;;

(define (get-option argv kw-opts kw-args return)
  (cond
   ((null? argv)
    (return #f #f argv))

   ((or (not (eq? #\- (string-ref (car argv) 0)))
	(eq? (string-length (car argv)) 1))
    (return 'normal-arg (car argv) (cdr argv)))

   ((eq? #\- (string-ref (car argv) 1))
    (let* ((kw-arg-pos (or (string-index (car argv) #\=)
			   (string-length (car argv))))
	   (kw (symbol->keyword (substring (car argv) 2 kw-arg-pos)))
	   (kw-opt? (member kw kw-opts))
	   (kw-arg? (member kw kw-args))
	   (arg (or (and (not (eq? kw-arg-pos (string-length (car argv))))
			 (substring (car argv)
				    (+ kw-arg-pos 1)
				    (string-length (car argv))))
		    (and kw-arg?
			 (begin (set! argv (cdr argv)) (car argv))))))
      (if (or kw-opt? kw-arg?)
	  (return kw arg (cdr argv))
	  (return 'usage-error kw (cdr argv)))))

   (else
    (let* ((char (substring (car argv) 1 2))
	   (kw (symbol->keyword char)))
      (cond

       ((member kw kw-opts)
	(let* ((rest-car (substring (car argv) 2 (string-length (car argv))))
	       (new-argv (if (= 0 (string-length rest-car))
			     (cdr argv)
			     (cons (string-append "-" rest-car) (cdr argv)))))
	  (return kw #f new-argv)))

       ((member kw kw-args)
	(let* ((rest-car (substring (car argv) 2 (string-length (car argv))))
	       (arg (if (= 0 (string-length rest-car))
			(cadr argv)
			rest-car))
	       (new-argv (if (= 0 (string-length rest-car))
			     (cddr argv)
			     (cdr argv))))
	  (return kw arg new-argv)))

       (else (return 'usage-error kw argv)))))))

(define (for-next-option proc argv kw-opts kw-args)
  (let loop ((argv argv))
    (get-option argv kw-opts kw-args
		(lambda (opt opt-arg argv)
		  (and opt (proc opt opt-arg argv loop))))))



;;; {Low Level Modules}
;;;
;;; These are the low level data structures for modules.
;;;
;;; (make-module size use-list lazy-binding-proc) => module
;;; module-{obarray,uses,binder}[|-set!]
;;; (module? obj) => [#t|#f]
;;; (module-locally-bound? module symbol) => [#t|#f]
;;; (module-bound? module symbol) => [#t|#f]
;;; (module-symbol-locally-interned? module symbol) => [#t|#f]
;;; (module-symbol-interned? module symbol) => [#t|#f]
;;; (module-local-variable module symbol) => [#<variable ...> | #f]
;;; (module-variable module symbol) => [#<variable ...> | #f]
;;; (module-symbol-binding module symbol opt-value)
;;;		=> [ <obj> | opt-value | an error occurs ]
;;; (module-make-local-var! module symbol) => #<variable...>
;;; (module-add! module symbol var) => unspecified
;;; (module-remove! module symbol) =>  unspecified
;;; (module-for-each proc module) => unspecified
;;; (make-scm-module) => module ; a lazy copy of the symhash module
;;; (set-current-module module) => unspecified
;;; (current-module) => #<module...>
;;;
;;;


;; This is how modules are printed.
;; You can re-define it.
;;
(define (%print-module mod port wr?)
  (display "#<" port)
  (display (or (module-kind mod) "module") port)
  (let ((name (module-name mod)))
    (if name
	(begin
	  (display " " port)
	  (display name port))))
  (display " " port)
  (display (number->string (object-address mod) 16) port)
  (display ">" port))

;; module-type
;;
;; A module is characterized by an obarray in which local symbols
;; are interned, a list of modules, "uses", from which non-local
;; bindings can be inherited, and an optional lazy-binder which
;; is a (THUNK module symbol) which, as a last resort, can provide
;; bindings that would otherwise not be found locally in the module.
;;
(define module-type
  (make-record-type 'module '(obarray uses binder eval-thunk name kind)
		    (lambda (mod port wr?)
		      (%print-module mod port wr?))))

;; make-module &opt size uses
;;
;; Create a new module, perhaps with a particular size of obarray
;; or initial uses list.
;;
(define module-constructor (record-constructor module-type))

(define make-module
    (lambda args
      (let* ((size 1021)
	     (uses '())
	     (binder #f)
	     (answer #f)
	     (eval-thunk
	      (lambda (symbol define?)
		(if define?
		    (module-make-local-var! answer symbol)
		    (module-variable answer symbol)))))

	(if (> (length args) 0)
	    (begin
	      (set! size (or (car args) size))
	      (set! args (cdr args))))

	(if (> (length args) 0)
	    (begin
	      (set! uses (or (car args) uses))
	      (set! args (cdr args))))

	(if (> (length args) 0)
	    (begin
	      (set! binder (or (car args) binder))
	      (set! args (cdr args))))

	(if (not (null? args))
	    (error "Too many args to make-module." args))

	(if (not (integer? size))
	    (error "Illegal size to make-module." size))

	(and (list? uses)
	     (or (and-map module? uses)
		 (error "Incorrect use list." uses)))

	(if (and binder (not (procedure? binder)))
	    (error
	     "Lazy-binder expected to be a procedure or #f." binder))

	(set! answer
	      (module-constructor (make-vector size '())
				  uses
				  binder
				  eval-thunk
				  #f
				  #f))
	answer)))

(define module-obarray  (record-accessor module-type 'obarray))
(define module-obarray-set! (record-modifier module-type 'obarray))
(define module-uses  (record-accessor module-type 'uses))
(define module-uses-set! (record-modifier module-type 'uses))
(define module-binder (record-accessor module-type 'binder))
(define module-binder-set! (record-modifier module-type 'binder))
(define module-eval-thunk (record-accessor module-type 'eval-thunk))
(define module-eval-thunk-set! (record-modifier module-type 'eval-thunk))
(define module-name (record-accessor module-type 'name))
(define module-name-set! (record-modifier module-type 'name))
(define module-kind (record-accessor module-type 'kind))
(define module-kind-set! (record-modifier module-type 'kind))
(define module? (record-predicate module-type))


;;; {Module Searching in General}
;;;
;;; We sometimes want to look for properties of a symbol
;;; just within the obarray of one module.  If the property
;;; holds, then it is said to hold ``locally'' as in, ``The symbol
;;; DISPLAY is locally rebound in the module `safe-guile'.''
;;;
;;;
;;; Other times, we want to test for a symbol property in the obarray
;;; of M and, if it is not found there, try each of the modules in the
;;; uses list of M.  This is the normal way of testing for some
;;; property, so we state these properties without qualification as
;;; in: ``The symbol 'fnord is interned in module M because it is
;;; interned locally in module M2 which is a member of the uses list
;;; of M.''
;;;

;; module-search fn m
;; 
;; return the first non-#f result of FN applied to M and then to
;; the modules in the uses of m, and so on recursively.  If all applications
;; return #f, then so does this function.
;;
(define (module-search fn m v)
  (define (loop pos)
    (and (pair? pos)
	 (or (module-search fn (car pos) v)
	     (loop (cdr pos)))))
  (or (fn m v)
      (loop (module-uses m))))


;;; {Is a symbol bound in a module?}
;;;
;;; Symbol S in Module M is bound if S is interned in M and if the binding
;;; of S in M has been set to some well-defined value.
;;;

;; module-locally-bound? module symbol
;;
;; Is a symbol bound (interned and defined) locally in a given module?
;;
(define (module-locally-bound? m v)
  (let ((var (module-local-variable m v)))
    (and var
	 (variable-bound? var))))

;; module-bound? module symbol
;;
;; Is a symbol bound (interned and defined) anywhere in a given module
;; or its uses?
;;
(define (module-bound? m v)
  (module-search module-locally-bound? m v))

;;; {Is a symbol interned in a module?}
;;;
;;; Symbol S in Module M is interned if S occurs in 
;;; of S in M has been set to some well-defined value.
;;;
;;; It is possible to intern a symbol in a module without providing
;;; an initial binding for the corresponding variable.  This is done
;;; with:
;;;       (module-add! module symbol (make-undefined-variable))
;;;
;;; In that case, the symbol is interned in the module, but not
;;; bound there.  The unbound symbol shadows any binding for that
;;; symbol that might otherwise be inherited from a member of the uses list.
;;;

;; module-symbol-locally-interned? module symbol
;; 
;; is a symbol interned (not neccessarily defined) locally in a given module
;; or its uses?  Interned symbols shadow inherited bindings even if
;; they are not themselves bound to a defined value.
;;
(define (module-symbol-locally-interned? m v)
  (symbol-interned? (module-obarray m) v))


;; module-symbol-interned? module symbol
;; 
;; is a symbol interned (not neccessarily defined) anywhere in a given module
;; or its uses?  Interned symbols shadow inherited bindings even if
;; they are not themselves bound to a defined value.
;;
(define (module-symbol-interned? m v)
  (module-search module-symbol-locally-interned? m v))


;;; {Mapping modules x symbols --> variables}
;;;

;; module-local-variable module symbol
;; return the local variable associated with a MODULE and SYMBOL.
;;
;;; This function is very important. It is the only function that can
;;; return a variable from a module other than the mutators that store
;;; new variables in modules.  Therefore, this function is the location
;;; of the "lazy binder" hack.
;;;
;;; If symbol is defined in MODULE, and if the definition binds symbol
;;; to a variable, return that variable object.
;;;
;;; If the symbols is not found at first, but the module has a lazy binder,
;;; then try the binder.
;;;
;;; If the symbol is not found at all, return #f.
;;;
(define (module-local-variable m v)
  (or (and (module-symbol-locally-interned? m v)
	   (let ((b (symbol-binding (module-obarray m) v)))
	     (and (variable? b) b)))
      (and (module-binder m)
	   ((module-binder m) m v #f))))

;; module-variable module symbol
;; 
;; like module-local-variable, except search the uses in the 
;; case V is not found in M.
;;
(define (module-variable m v)
  (module-search module-local-variable m v))


;;; {Mapping modules x symbols --> bindings}
;;;
;;; These are similar to the mapping to variables, except that the
;;; variable is dereferenced.
;;;

;; module-symbol-binding module symbol opt-value
;; 
;; return the binding of a variable specified by name within
;; a given module, signalling an error if the variable is unbound.
;; If the OPT-VALUE is passed, then instead of signalling an error,
;; return OPT-VALUE.
;;
(define (module-symbol-local-binding m v . opt-val)
  (let ((var (module-local-variable m v)))
    (if var
	(variable-ref var)
	(if (not (null? opt-val))
	    (car opt-val)
	    (error "Locally unbound variable." v)))))

;; module-symbol-binding module symbol opt-value
;; 
;; return the binding of a variable specified by name within
;; a given module, signalling an error if the variable is unbound.
;; If the OPT-VALUE is passed, then instead of signalling an error,
;; return OPT-VALUE.
;;
(define (module-symbol-binding m v . opt-val)
  (let ((var (module-variable m v)))
    (if var
	(variable-ref var)
	(if (not (null? opt-val))
	    (car opt-val)
	    (error "Unbound variable." v)))))



;;; {Adding Variables to Modules}
;;;
;;;


;; module-make-local-var! module symbol
;; 
;; ensure a variable for V in the local namespace of M.
;; If no variable was already there, then create a new and uninitialzied
;; variable.
;;
(define (module-make-local-var! m v)
  (or (and (module-symbol-locally-interned? m v)
	   (let ((b (symbol-binding (module-obarray m) v)))
	     (and (variable? b) b)))
      (and (module-binder m)
	   ((module-binder m) m v #t))
      (begin
	(intern-symbol (module-obarray m) v)
	(let ((answer (make-undefined-variable v)))
	  (symbol-set! (module-obarray m) v answer)
	  answer))))

;; module-add! module symbol var
;; 
;; ensure a particular variable for V in the local namespace of M.
;;
(define (module-add! m v var)
  (if (not (variable? var))
      (error "Bad variable to module-add!" var))
  (intern-symbol (module-obarray m) v)
  (symbol-set! (module-obarray m) v var))


;; module-remove! 
;; 
;; make sure that a symbol is undefined in the local namespace of M.
;;
(define (module-remove! m v)
  (unintern-symbol (module-obarray m) v))

;; MODULE-FOR-EACH -- exported
;; 
;; Call PROC on each symbol in MODULE, with arguments of (SYMBOL VARIABLE).
;;
(define (module-for-each proc module)
  (let ((obarray (module-obarray module)))
    (do ((index 0 (+ index 1))
	 (end (vector-length obarray)))
	((= index end))
      (for-each
       (lambda (bucket)
	 (proc (car bucket) (cdr bucket)))
       (vector-ref obarray index)))))


;;; {Low Level Bootstrapping}
;;;

;; make-root-module 

:; A root module uses the symhash table (the system's privileged 
;; obarray).  Being inside a root module is like using SCM without
;; any module system.
;;


(define (make-root-module)
  (make-module 1019 #f
	       (lambda (m s define?)
		 (let ((bi (and (symbol-interned? #f s)
				(builtin-variable s))))
		   (and bi
			(or define? (variable-bound? bi))
			bi)))))


;; make-scm-module 

;; An scm module is a module into which the lazy binder copies
;; variable bindings from the system symhash table.  The mapping is
;; one way only; newly introduced bindings in an scm module are not
;; copied back into the system symhash table (and can be used to override
;; bindings from the symhash table).
;;

(define (make-scm-module)
  (make-module 1019 #f
	       (lambda (m s define?)
		 (let ((bi (and (symbol-interned? #f s)
				(builtin-variable s))))
		   (and bi
			(variable-bound? bi)
			bi)))))




;; the-module
;; 

(define the-module (make-root-module))

;; set-current-module module
;;
;; set the current module as viewed by the normalizer.
;;
(define (set-current-module m)
  (set! the-module m)
  (if m
      (set! *top-level-lookup-thunk* (module-eval-thunk the-module))
      (set! *top-level-lookup-thunk* #f)))


;; current-module
;;
;; return the current module as viewed by the normalizer.
;;
(define (current-module) the-module)




;;; {Macros}
;;;

(define (defmacro? m)
  (object-property m '%%defmacro?))
(define (assert-defmacro?! m)
  (set-object-property! m '%%defmacro? #t))

(define (defmacro-transformer m)
  (object-property m '%%defmacro-transformer))
(define (set-defmacro-transformer! m t)
  (set-object-property! m '%%defmacro-transformer t))

(define defmacro:transformer
  (lambda (f)
    (let ((a (procedure->memoizing-macro
	      (lambda (exp env)
		(copy-tree (apply f (cdr exp)))))))
      (set-defmacro-transformer! a t)
      (assert-defmacro?! a)
      a)))


(define defmacro
  (let ((defmacro-transformer
	  (lambda (name parms . body)
	    (let ((transformer `(lambda ,parms ,@body)))
	      `(define ,name
		 (,(lambda (transformer)
		     (defmacro:transformer transformer))
		  ,transformer))))))
    (defmacro:transformer defmacro-transformer)))

(define defmacro:syntax-transformer
  (lambda (f)
    (procedure->syntax
	      (lambda (exp env)
		(copy-tree (apply f (cdr exp)))))))

(define (macroexpand-1 e)
  (cond
   ((pair? e) (let ((a (car e))
		    (val (and (symbol? a) (defined? a) (eval a))))
		(if (defmacro? val)
		    (apply (defmacro-transformer val) (cdr e))
		    e)))
   (#t e)))

(define (macroexpand e)
  (cond
   ((pair? e) (let ((a (car e))
		    (val (and (symbol? a) (defined? a) (eval a))))
		(if (defmacro? val)
		    (macroexpand (apply (defmacro-transformer val) (cdr e)))
		    e)))
   (#t e)))

(define gentemp
  (let ((*gensym-counter* -1))
    (lambda ()
      (set! *gensym-counter* (+ *gensym-counter* 1))
      (string->symbol
       (string-append "scm:G" (number->string *gensym-counter*))))))





;;; {Running Repls}
;;;

(define (repl read eval print)
  (let loop ((source (read)))
    (print (eval source))
    (loop (read))))

(define (error-catching-repl read eval print quit abort)
  (with-dynamic-root-handlers
   (lambda () (repl read eval print))
   quit abort))

;; A provisional repl that acts like the SCM repl:
;;
(define scm-repl-silent #f)
(define (assert-repl-silence v) (set! scm-repl-silent v))

(define scm-repl-verbose #t)
(define (assert-repl-verbosity v) (set! scm-repl-verbose v))

(define scm-repl-prompt #t)
(define (assert-repl-prompt v) (set! scm-repl-prompt v))

(define (scm-style-repl)
  (letrec ((-read (lambda ()
		    (if scm-repl-prompt
			(begin
			  (display "guile> ")
			  (force-output)
			  (repl-report-reset)))
		    (let ((val (read)))
		      (if (eof-object? val)
			  (begin
			    (if scm-repl-verbose
				(begin
				  (newline)
				  (display ";;; EOF -- quitting")
				  (newline)))
			    (quit 0)))
		      val)))

	   (-eval (lambda (sourc)
		    (repl-report-start-timing)
		    (eval sourc)))

	   (-print (lambda (result)
		     (if (not scm-repl-silent)
			 (begin
			   (write result)
			   (newline)
			   (if scm-repl-verbose
			       (repl-report))
			   (force-output)))))

	   (-quit (lambda ()
		    (if scm-repl-verbose
			(begin
			  (display ";;; QUIT executed, repl exitting")
			  (newline)
			  (repl-report)))
		    #t))

	   (-abort (lambda ()
		     (if scm-repl-verbose
			 (begin
			   (display ";;; ABORT executed.")
			   (newline)
			   (repl-report)))
		     (repl -read -eval -print))))

    (error-catching-repl -read
			 -eval
			 -print
			 -quit
			 -abort)))

(define (stand-alone-repl)
  (let ((oport (current-input-port)))
    (set-current-input-port *stdin*)
    (scm-style-repl)
    (set-current-input-port oport)))



(define (peek . stuff)
  (newline)
  (display ";;; ")
  (write stuff)
  (newline)
  (car (last-pair stuff)))

(define pk peek)
