/*
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/
/*
** source.c
**
** Innterface to source file abstraction
**
**	NOTE:	    This module is almost identical to the one for LCL.  The
**		    only difference is that a couple of source lines have been
**		    commented out.
**
**		    This module has too many dependencies to be in the common
**		    source area.  Any of the solutions that would allow this
**		    module to be common had its own set of compromises.  It
**		    seemed best and most straightforward to just keep separte
**		    copies for LSL and LCL.  We should examine this again if we
**		    ever reorganize the module structure.
**
**  AUTHORS:
**
**     Steve Garland, Massachusetts Institute of Technology
**     Joe Wild, Technical Languages and Environments, DECspec project
*/

# include "lclintMacros.nf"
# include "llbasic.h"
# include "osd.h"

extern bool
tsource_close (/*@only@*/ tsource *s)
{
  if (s->file == 0)
    {
      tsource_free (s);
      return FALSE;
    }
  else
    {
      DPRINTF (("close: %s", tsource_fileName (s)));
      check (fclose (s->file) == 0);
      /* could be file static! sfree (s->file); */
      s->file = 0;
      tsource_free (s);
      return TRUE;
    }
}

extern void
tsource_free (/*@null@*/ /*@only@*/ tsource *s)
{
  if (s != NULL)
    {
      sfree (s->name);
      sfree (s->stringSource);
      sfree (s);
    }
}

extern /*@only@*/ tsource *
  tsource_create (char *name, char *suffix, bool echo)
{
  char *ps;
  tsource *s = (tsource *) dmalloc (sizeof (*s));
  
  s->name = (char *) dmalloc (strlen (name) + strlen (suffix) + 1);
  s->file = 0;
  strcpy (s->name, name);

  ps = strrchr (s->name, '/');

  if (ps == (char *) 0)
    {
      ps = s->name;
    }

  if (strchr (ps, '.') == NULL)
    {
      strcat (s->name, suffix);
    }

  

  s->lineNo = 0;
  s->echo = echo;
  s->fromString = FALSE;
  s->stringSource = NULL;
  s->stringSourceTail = NULL;
  

  return s;
}

extern /*@only@*/ tsource *
tsource_fromString (char *name, char *str)
{
  tsource *s = (tsource *) dmalloc (sizeof (*s));

  s->name = mstring_copy (name);
  s->stringSource = mstring_copy (str);
  s->stringSourceTail = s->stringSource;
  s->file = 0;
  s->echo = FALSE;
  s->fromString = TRUE;
  s->lineNo = 0;

  DPRINTF (("source: %s", str));
  return s;
}

extern /*@dependent@*/ /*@null@*/ 
char *tsource_nextLine (tsource *s)
{
  char *currentLine;
  int len;

  if (s->fromString)
    {
      if (s->stringSourceTail == NULL || (strlen (s->stringSourceTail) == 0))
	{
	  currentLine = 0;
	}
      else
	{
	  char *c = strchr (s->stringSourceTail, '\n');
	  
	  DPRINTF (("tail: %s", s->stringSourceTail));

	  /* in case line is terminated not by newline */ 
	  if (c == 0)
	    {
	      c = strchr (s->stringSourceTail, '\0');
	    }

	  len = c - s->stringSourceTail + 1;

	  if (len > STUBMAXRECORDSIZE - 2)
	    {
	      len = (STUBMAXRECORDSIZE - 2);
	    }

	  currentLine = &(s->buffer)[0];
	  strncpy (currentLine, s->stringSourceTail, size_fromInt (len));
	  currentLine[len] = '\0';
	  s->stringSourceTail += len;
	}
      
    }
  else
    {
      llassert (s->file != NULL);
      currentLine = fgets (&(s->buffer)[0], STUBMAXRECORDSIZE, s->file);
    }
  if (currentLine == 0)
    {
      strcpy (s->buffer, "*** End of File ***");
    }
  else
    {
      s->lineNo++;
      len = strlen (currentLine) - 1;
      if (s->buffer[len] == '\n')
	{
	  s->buffer[len] = '\0';
	}
      else 
	{
	  if (len >= STUBMAXRECORDSIZE - 2)
	    {
	      llmsg (message ("Input line too long: %s",
			      cstring_fromChars (currentLine)));
	    }
	}
    }
  /* if (s->echo) slo_echoLine (currentLine);		only needed in LCL */
  DPRINTF (("nextline %s: %s", s->name, currentLine));
  return currentLine;
}

extern bool
tsource_open (tsource *s)
{
  if (s->fromString)
    {
      s->stringSourceTail = s->stringSource; /* not an error: tail is dependent */
      return TRUE;
    }

  /* sfree (s->file);  */
  s->file = fopen (s->name, "r");
  return (s->file != 0 || s->fromString);
}

/*
** requires
**  path != NULL \and
**  s != NULL \and
**  *s.name == filename (*s.name) || filetype (*s.name)
**      // *s.name consists of a file name and type only ("<filename>.<type>)
**	// No path name is included
**
** ensures
**  if filefound (*path, *s) then
**	result = true \and *s.name = filespec_where_file_found (*path, *s)
**  else
**	result = false
*/

extern bool
tsource_getPath (char *path, tsource *s, ltoken t)
{
  char *returnPath;
  filestatus status;		/* return status of osd_getEnvPath.*/
  bool rVal;			/* return value of this procedure. */

 /* Check if requires met. */
  if (path == NULL || s == NULL || s->name == NULL)
    {
      llbuglit ("tsource_getPath: invalid parameter");
    }

  status = osd_getPath (path, s->name, &returnPath);

  if (status == OSD_FILEFOUND)
    {				/* Should be majority of cases. */
      rVal = TRUE;
      
      sfree (s->name);
      s->name = returnPath;
    }
  else if (status == OSD_FILENOTFOUND)
    {
      rVal = FALSE;
    }
  else if (status == OSD_PATHTOOLONG)
    {
      rVal = FALSE;
     /* Directory and filename are too long.  Report error. */
      if (ltoken_isUndefined (t))
	{
	  lclplainerror 
	    (cstring_makeLiteral ("soure_getPath: Filename plus directory from "
				  "search path too long"));
	}
      else
	{
	  lclerror 
	    (t, 
	     cstring_makeLiteral ("Filename plus directory from search path too long"));
	}
    }
  else
    {
      rVal = FALSE;
      llbuglit ("tsource_getPath: invalid return status");
    }
  return rVal;
}

char *specFullName (char *specfile, /*@out@*/ char **inpath)
{
  /* extract the path and the specname associated with the given file */
  char *specname = (char *) dmalloc (sizeof (*specname) 
				     * (strlen (specfile) + 9));
  char *path = (char *) dmalloc (sizeof (*path) * (strlen (specfile)));
  size_t size;
  long int i, j;
  
  /* initialized path to empty string or may have accidental garbage */
  *path = '\0';
  strcpy (specname, specfile);
  /* trim off pathnames in specfile */
  size = strlen (specname);

  for (i = size_toInt (size) - 1; i >= 0; i--)
    {
      if (specname[i] == '/')
	{
	  /*      strcpy (specname, (char *)specname+i+1); */
	  for (j = 0; j <= i; j++)	/* include '/'  */
	    {
	      path[j] = specname[j];
	    }

	  path[i + 1] = '\0';
	  specname += i + 1;
	  break;
	}
    }
  
  /* 
  ** also remove .lcl file extension, assume it's the last extension
  ** of the file name 
  */

  size = strlen (specname);

  for (i = size_toInt (size) - 1; i >= 0; i--)
    {
      if (specname[i] == '.')
	{
	  specname[i] = '\0';
	  break;
	}
    }
  
  *inpath = path;
  return specname;
}
