/*
   window.c - xslideshow main routine.

   Author:   Susumu Shiohara (shiohara@tpp.epson.co.jp)

          Copyright 1993-1997 by Susumu Shiohara

                   All Rights Reserved

*/

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

/* Athena Widget
    toplevel
        vPanedWidgetClass

            boxWidgetClass
                formWidgetClass
                    commandWidgetClass - control buttons (pause head prev ... quit)

            boxWidgetClass
                commandWidgetClass - animation buttons

            boxWidgetClass
                commandWidgetClass - Set File Path Directory button
                commandWidgetClass - Set Target Display Name button
                commandWidgetClass - Set Interval Time (sec) button
                commandWidgetClass - Set Command Line Options button

            boxWidgetClass
                formWidgetClass
                    commandWidgetClass - Get the file list from the xslideshow button
                    viewportWidgetClass
                        listWidgetClass - display the file list
                    commandWidgetClass - Set selected file as the xslideshow's current file button
            labelWidgetClass - show message for display name
            labelWidgetClass - show message for some actions
*/

/* Motif Widget
    toplevel
        xmPanedWindowWidgetClass

            xmBulletinBoardWidgetClass
                xmFormWidgetClass
                    xmPushButtonWidgetClass - control buttons (pause head prev ... quit)

            xmRowColumnWidgetClass
                xmPushButtonWidgetClass - animation buttons

            xmRowColumnWidgetClass
                xmPushButtonWidgetClass - Set File Path Directory button
                xmPushButtonWidgetClass - Set Target Display Name button
                xmPushButtonWidgetClass - Set Interval Time (sec) button
                xmPushButtonWidgetClass - Set Command Line Options button

            xmBulletinBoardWidgetClass
                xmFormWidgetClass
                    xmPushButtonWidgetClass - Get the file list from the xslideshow button
                    xmListWidgetClass - display the file list
                    xmPushButtonWidgetClass - Set selected file as the xslideshow's current file button
            xmLabelWidgetClass - show message for display name
            xmLabelWidgetClass - show message for some actions
*/

#if defined (_MOTIF)

# include <Xm/PanedW.h>
# include <Xm/BulletinB.h>
# include <Xm/RowColumn.h>
# include <Xm/Form.h>
# include <Xm/PushB.h>
# include <Xm/ToggleB.h>
# include <Xm/List.h>
# include <Xm/Text.h>
# include <Xm/Label.h>

# define vPanedWidgetClass xmPanedWindowWidgetClass
# define boxWidgetClass xmRowColumnWidgetClass
# define formWidgetClass xmFormWidgetClass
# define commandWidgetClass xmPushButtonWidgetClass
# define labelWidgetClass xmLabelWidgetClass
# define listWidgetClass xmListWidgetClass
# define toggleWidgetClass xmToggleButtonWidgetClass

# define XtNhorizDistance XmNleftOffset
# define XtNfromHoriz XmNleftWidget
# define XtNfromVert XmNtopWidget
# define XtNresizable XmNresizable
# define XtNdefaultColumns XmNvisibleItemCount
# define XtJustifyLeft XmNalignment

#else  /* Athena widget */

# ifdef _X11R4
# include <X11/VPaned.h>
# include <X11/Form.h>
# include <X11/Dialog.h>
# include <X11/Command.h>
# include <X11/AsciiText.h>
# include <X11/Box.h>
# include <X11/Viewport.h>
# include <X11/List.h>
# include <X11/Label.h>
# include <X11/Toggle.h>
# include <X11/Scroll.h>
# else
# include <X11/Xaw/Paned.h>
# include <X11/Xaw/Form.h>
# include <X11/Xaw/Dialog.h>
# include <X11/Xaw/Command.h>
# include <X11/Xaw/AsciiText.h>
# include <X11/Xaw/Box.h>
# include <X11/Xaw/Viewport.h>
# include <X11/Xaw/List.h>
# include <X11/Xaw/Label.h>
# include <X11/Xaw/Toggle.h>
# include <X11/Xaw/Scrollbar.h>

# define asciiStringWidgetClass asciiTextWidgetClass
# define XtDialogGetValueString XawDialogGetValueString
# define vPanedWidgetClass panedWidgetClass
# define XtListReturnStruct XawListReturnStruct

# endif
#endif

#include "comm-control.h"

#include "button_pause.bm"
#include "button_head.bm"
#include "button_prev.bm"
#include "button_stop.bm"
#include "button_next.bm"
#include "button_tail.bm"

#define DEFAULT_BUF_SIZ 1024
#define MAX_TEXT_LEN 512
#define BUTTON_WIDTH 75
#define BUTTON_DISTANCE 10
#define MAX_ARGS 32

#define CMDOPT_CLOCK     0
#define CMDOPT_BIFF      1
#define CMDOPT_SHOWFNAME 2
#define CMDOPT_ROOT      3
#define CMDOPT_FULL      4
#define CMDOPT_OVERRIDE  5
#define CMDOPT_RESIZE    6
#define CMDOPT_FIT       7
#define CMDOPT_MAXFIT    8
#define CMDOPT_PRIVATE   9
#define CMDOPT_PERFECT   10

static Visual *theVisual;

static Atom XA_XSLIDESHOW;
static Atom XA_XSLIDESHOWCONTROL;
static Atom XA_XSLIDESHOWCONTROL_NFILES;
static Atom XA_XSLIDESHOWCONTROL_FLIST;
static Atom XA_XSLIDESHOWCONTROL_FLIST_END;
static Atom wm_delete_window;

Widget	toplevel, vpaned, command, anim_command;
Widget	messageW, dpyNameW;
Widget	quit;

Widget textDirW, textDirPopup, textDirDialog;
Widget textDpyW, textDpyPopup, textDpyDialog;
Widget textIntW, textIntPopup, textIntDialog;
Widget textOptW, textOptPopup, textOptDialog;

Widget listFormW, listFileView, fileListW;
char currentFileName[MAX_TEXT_LEN];

static char *pProgramName;
static String *pFileListTable;
static char *pRcvBuf;
static int nFiles, nIndex, prevNFiles, bufLen, rcvLen;

static char remoteDisplay[MAX_TEXT_LEN];
static char intData[MAX_TEXT_LEN];
static char msgdata[MAX_TEXT_LEN];
static char animation[MAX_TEXT_LEN];

static char cmdLineOptClock[16];
static char cmdLineOptBiff[16];
static char cmdLineOptShowFileName[16];
static char cmdLineOptRoot[16];
static char cmdLineOptFull[16];
static char cmdLineOptOverride[16];
static char cmdLineOptResize[16];
static char cmdLineOptPrivate[16];
static char cmdLineOptPerfect[16];

typedef struct _CB_BUTTON {
	char	*name;
	char	*color;
	CBActionType cb;
} CB_BUTTON;

static CB_BUTTON control_buttons[] = {
	{(char *)pause_bits, 	"cyan",		CB_PAUSE},
	{(char *)head_bits, 	"yellow",	CB_HEAD},
	{(char *)prev_bits, 	"green",	CB_PREV},
	{(char *)stop_bits, 	"red",		CB_STOP},
	{(char *)next_bits, 	"green",	CB_NEXT},
	{(char *)tail_bits, 	"yellow",	CB_TAIL},
	{NULL,			NULL,		0}
};

static CB_BUTTON anim_buttons[] = {
	{"slide", 		"yellow",	CB_SLIDE},
	{"fade", 		"yellow",	CB_FADE},
	{"reverse", 	"yellow",	CB_REVERSE},
	{"monotone", 	"yellow",	CB_MONOTONE},
	{"flush", 		"yellow",	CB_FLUSH},
	{"wipe", 		"yellow",	CB_WIPE},
	{"wipe24", 		"yellow",	CB_WIPE24},
	{"block", 		"yellow",	CB_BLOCK},
	{"block24", 	"yellow",	CB_BLOCK24},
	{"mosaic", 		"yellow",	CB_MOSAIC},
	{"coloranim", 	"yellow",	CB_COLORANIM},
	{"coloranim2", 	"yellow",	CB_COLORANIM2},
	{"shuffle", 	"yellow",	CB_SHUFFLE},
	{"puzzle", 		"yellow",	CB_PUZZLE},
	{"slit", 	    "yellow",	CB_SLIT},
	{"transparent",	"yellow",	CB_TRANSPARENT},
	{"move", 		"yellow",	CB_MOVE},
	{"float24", 	"yellow",	CB_FLOAT24},
	{"default", 	"yellow",	CB_DEFAULT},
	{NULL,			NULL,	0}
};

typedef struct _OPT_BUTTON {
	char	*opt;
	int		cb;
} OPT_BUTTON;

static OPT_BUTTON opt_buttons[] = {
	{"clock",	CMDOPT_CLOCK},
	{"biff",	CMDOPT_BIFF},
	{"fname",	CMDOPT_SHOWFNAME},
	{"root",	CMDOPT_ROOT},
	{"full",	CMDOPT_FULL},
	{"override",CMDOPT_OVERRIDE},
	{"resize",	CMDOPT_RESIZE},
	{"fit",		CMDOPT_FIT},
	{"maxfit",	CMDOPT_MAXFIT},
	{"private",	CMDOPT_PRIVATE},
	{"perfect",	CMDOPT_PERFECT},
	{NULL,			0}
};

static void getDisplayName() ;
static XtCallbackProc sendIntMessage() ;
static void wm_quit();
XtActionsRec actionTable[] = {
	{"intOk",		(XtActionProc)sendIntMessage},
	{"dpyOk",		(XtActionProc)getDisplayName},
	{"Quit",		(XtActionProc)wm_quit}
};

typedef struct {
	char	*intValueTranslations;
	char	*dpyValueTranslations;
#if !defined (_MOTIF)
	char	*translations;
#endif
}ApplicationData, *ApplicationDataPtr;

static XtResource resources[]={
#if defined (_MOTIF)
	{"intValueTranslations", "ValueTranslations", XtRString, sizeof(String),
		XtOffset(ApplicationDataPtr, intValueTranslations),
		XtRString,
		"*textIntDialog*translations: #override <Key>Return: intOk()",
	},
	{"dpyValueTranslations", "ValueTranslations", XtRString, sizeof(String),
		XtOffset(ApplicationDataPtr, dpyValueTranslations),
		XtRString,
		"*textDpyDialog*translations: #override <Key>Return: dpyOk()",
	},
#else
	{"intValueTranslations", "ValueTranslations", XtRString, sizeof(String),
		XtOffset(ApplicationDataPtr, intValueTranslations),
		XtRString,
		"*textIntDialog*value.translations: #override <Key>Return: intOk()",
	},
	{"dpyValueTranslations", "ValueTranslations", XtRString, sizeof(String),
		XtOffset(ApplicationDataPtr, dpyValueTranslations),
		XtRString,
		"*textDpyDialog*value.translations: #override <Key>Return: dpyOk()",
	},
	{"translations", "Translations", XtRString, sizeof(String),
		XtOffset(ApplicationDataPtr, translations),
		XtRString,
		"*translations: #override <Message>WM_PROTOCOLS: Quit()",
	},
#endif
};
		
static String fallback_resources[] = {
	"*font: fixed",
	"*input: True",
	"*foreground: white",
	"*background: black",
	"*borderColor: gray",
	"*internalBorderColor: white",
	"*quit.foreground: red",
	"*quit.background: green",
	"*viewPort.borderColor: cyan",
	"*viewPort.horizontal.foreground: cyan",
	"*viewPort.horizontal.borderColor: cyan",
	"*viewPort.vertical.foreground: cyan",
	"*viewPort.vertical.borderColor: cyan",
	"*textdir.foreground: green",
	"*textint.foreground: green",
	"*textdpy.foreground: green",
	"*textopt.foreground: green",
	"*getflist.foreground: yellow",
	"*flist.foreground: cyan",
	"*setfile.foreground: yellow",
	"*msg1.foreground: white",
	"*msg2.foreground: white",
	"*pause.label:",
	"*head.label:",
	"*prev.label:",
	"*stop.label:",
	"*next.label:",
	"*tail.label:",
#if defined (_MOTIF)
	"*textIntDialog*translations: #override <Key>Return: intOk()",
	"*textDpyDialog*translations: #override <Key>Return: dpyOk()",
#else
	"*textIntDialog*value.translations: #override <Key>Return: intOk()",
	"*textDpyDialog*value.translations: #override <Key>Return: dpyOk()",
	"*translations: #override <Message>WM_PROTOCOLS: Quit()",
#endif
	NULL,
};

static char *dummy[]={
	" ",
	NULL
};

/* Display message label */
#if defined(__STDC__) || defined(__cplusplus)
static void print_message(char *msg)
#else
static void print_message(msg)
char *msg;
#endif
{
Arg args[MAX_ARGS];
int ac;
#if defined (_MOTIF)
XmString xms;
#endif

	ac = 0;
#if defined (_MOTIF)
	xms = XmStringCreate(msg, XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, (XtArgVal)xms); ac++;
	XtSetArg(args[ac], XmNalignment, (XtArgVal)XmALIGNMENT_BEGINNING); ac++;
#else
	XtSetArg(args[ac], XtNlabel, (XtArgVal)msg); ac++;
#endif

	XtSetValues(messageW, args, ac);

#if defined (_MOTIF)
	XmStringFree(xms);
#endif

}

/* Find a window which has XA_XSLIDESHOW property on the remote display */
#if defined(__STDC__) || defined(__cplusplus)
static Window find_xslideshow_window(Display *display, Window top_window)
#else
static Window find_xslideshow_window(display, top_window)
Display *display;
Window	top_window;
#endif
{
int i;
Atom actual_type_return;
int actual_format_return;
unsigned long nitems_return, bytes_after_return;
unsigned char *prop_return;
unsigned int nchildren_return;
Window found_window = 0, dummy_window, *children_return;

	XA_XSLIDESHOW = XInternAtom(display, AtomXslideshow, False);

	if ((XGetWindowProperty(
							display,
							top_window, 
							XA_XSLIDESHOW,	/* property */
							0,		/* long_offset */
							1,		/* long_length */
							False,		/* remove */
							XA_WINDOW,	/* req_type */
							&actual_type_return,
							&actual_format_return,
							&nitems_return,
							&bytes_after_return,
							&prop_return)) == Success
				&& actual_type_return == XA_WINDOW)
		return(top_window);

	if (! XQueryTree(display, top_window,
		&dummy_window, &dummy_window,
		&children_return, &nchildren_return))
		return (0);

	for (i = 0; i < (int)nchildren_return; i++)
		if ((found_window = find_xslideshow_window(display, children_return[i])))
			break;

	if (children_return)
		XFree((char *)children_return);

	return(found_window);
}


/* Send client_message (interval time) to the xslideshow */
#if defined(__STDC__) || defined(__cplusplus)
static XtCallbackProc sendIntMessage(Widget widget, CBActionType client_data, caddr_t call_data)
#else
static XtCallbackProc sendIntMessage( widget, client_data, call_data)
Widget widget;
CBActionType client_data;
caddr_t call_data;
#endif
{
String string;
Display *display;
Window found_window = 0;
Window dummy_window, *children_return;
unsigned int nchildren_return;
XClientMessageEvent event;
int i;

	/* Get input string from the text dialog */
#if defined (_MOTIF)
	string = XmTextGetString(textIntDialog);
#else
	string = XtDialogGetValueString(textIntDialog);
#endif
	strcpy(intData, string);

	/* Open remote display */
	if(remoteDisplay[0] != 0){
		display = XOpenDisplay(remoteDisplay);
		if(display == NULL) {
			sprintf(msgdata,"Connection Failed %s",remoteDisplay);
			print_message(msgdata);
			return 0;
		}
	}
	else{
		display = XtDisplay(widget);
	}

	/* Find xslideshow window */
	if( !XQueryTree(display,
		RootWindowOfScreen(DefaultScreenOfDisplay(display)),
		&dummy_window, &dummy_window,
		&children_return, &nchildren_return) )
		return 0;

	for (i = 0; i < (int)nchildren_return; i++)
		if((found_window = find_xslideshow_window( display, children_return[i] )))
			break;

	if ( ! found_window ) {
		sprintf(msgdata, "No xslideshow is running on display %s",
			DisplayString(display));
		print_message(msgdata);
		return 0;
	}

	/* Send client_message */
	event.display      = display;
	event.window       = found_window;
	event.type         = ClientMessage;
	event.format       = 32;
	event.message_type = XInternAtom(display, AtomXslideshowInt, False);
	event.data.l[0]    = (long)atoi(intData);

	if ( !XSendEvent ( display, found_window, True, 0L, (XEvent *)&event) ) {
		fprintf (stderr,
			"xslideshow-control: XSendEvent(display, 0x%x ...) failed.\n",
			(unsigned int) found_window);
		exit(1);
	}
	XSync (display, False);

	sprintf(msgdata,"Sent message : interval time = %d",(int)event.data.l[0]);
	print_message(msgdata);


	/* Close remote display */
	if(remoteDisplay[0] != 0)
		XCloseDisplay(display);

	XtPopdown(textIntPopup);

	return 0;
}


/* Send client_message (strings) to the xslideshow */
#if defined(__STDC__) || defined(__cplusplus)
XtCallbackProc sendStringMessage(Widget widget, CBActionType client_data, caddr_t call_data)
#else
XtCallbackProc sendStringMessage( widget, client_data, call_data)
Widget widget;
CBActionType client_data;
caddr_t call_data;
#endif
{
String string;
Display *display;
Window found_window = 0;
Window dummy_window, *children_return;
unsigned int nchildren_return;
XClientMessageEvent event;
int i, sendBufLen;
char *sendBufp;
char sendBuf[MAX_TEXT_LEN];

	if(client_data == CB_DIR_PATH){
		/* Get input string from the text dialog */
#if defined (_MOTIF)
		string = XmTextGetString(textDirDialog);
#else
		string = XtDialogGetValueString(textDirDialog);
#endif
		strcpy(sendBuf, string);
	}
	else if(client_data == CB_SEL_FILE){
		strcpy(sendBuf, currentFileName);
	}
	else if(client_data == CB_REQ_FLIST){
		/* Set xslideshow-control display name */
		strcpy(sendBuf, DisplayString(XtDisplay(toplevel)));
	}

	/* Open remote display */
	if(remoteDisplay[0] != 0){
		display = XOpenDisplay(remoteDisplay);
		if(display == NULL) {
			sprintf(msgdata,"Connection Failed %s",remoteDisplay);
			print_message(msgdata);
			return 0;
		}
	}
	else{
		display = XtDisplay(widget);
	}

	/* Find xslideshow window */
	if( !XQueryTree(display,
		RootWindowOfScreen(DefaultScreenOfDisplay(display)),
		&dummy_window, &dummy_window,
		&children_return, &nchildren_return) )
		return 0;

	for (i = 0; i < (int)nchildren_return; i++)
		if((found_window = find_xslideshow_window( display, children_return[i] )))
			break;

	if ( ! found_window ) {
		sprintf(msgdata, "No xslideshow is running on display %s",
			DisplayString(display));
		print_message(msgdata);
		return 0;
	}

	/* Send starting message */
	event.display      = display;
	event.window       = found_window;
	event.type         = ClientMessage;
	event.format       = 32;
	event.message_type = XInternAtom(display, AtomXslideshowTextStart, False);
	if ( !XSendEvent ( display, found_window, True, 0L, (XEvent *)&event) ) {
		fprintf (stderr,
			"xslideshow-control: XSendEvent(display, 0x%x ...) failed.\n",
			(unsigned int) found_window);
		exit(1);
	}
	XSync (display, False);

	/* Send message strings */
	event.display      = display;
	event.window       = found_window;
	event.type         = ClientMessage;
	event.format       = 8;
	event.message_type = XInternAtom(display, AtomXslideshowText, False);
	sendBufLen         = strlen(sendBuf);
	sendBufp           = sendBuf;
	while(sendBufLen > 0){
		strncpy(event.data.b, sendBufp, 19);
		event.data.b[19] = '\0';
		if ( !XSendEvent ( display, found_window, True, 0L, (XEvent *)&event) ) {
			fprintf (stderr,
				"xslideshow-control: XSendEvent(display, 0x%x ...) failed.\n",
				(unsigned int) found_window);
			exit(1);
		}
		XSync (display, False);
		sendBufLen -= 19;
		sendBufp += 19;
	}

	/* Send ending message */
	event.display      = display;
	event.window       = found_window;
	event.type         = ClientMessage;
	event.format       = 32;
	event.message_type = XInternAtom(display, AtomXslideshowTextEnd, False);
	if(client_data == CB_DIR_PATH){
		event.data.l[0] = (long)MSG_SETPATHDIR;	/* This value means "File Path Directory" */
	}
	else if(client_data == CB_SEL_FILE){
		event.data.l[0] = (long)MSG_SETCURFILE;	/* This value means "Selected File Name" */
	}
	else if(client_data == CB_REQ_FLIST){
		event.data.l[0] = (long)MSG_REQFLIST;	/* This value means "Request File List" */
	}
	if ( !XSendEvent ( display, found_window, True, 0L, (XEvent *)&event) ) {
		fprintf (stderr,
			"xslideshow-control: XSendEvent(display, 0x%x ...) failed.\n",
			(unsigned int) found_window);
		exit(1);
	}
	XSync (display, False);

	if(client_data == CB_DIR_PATH){
		sprintf(msgdata,"Sent message : file path dir = %s",sendBuf);
	}
	else if(client_data == CB_SEL_FILE){
		sprintf(msgdata,"Sent message : select file = %s",sendBuf);
	}
	else if(client_data == CB_REQ_FLIST){
		sprintf(msgdata,"Sent message : request file list");
	}
	print_message(msgdata);


	/* Close remote display */
	if(remoteDisplay[0] != 0)
		XCloseDisplay(display);

	if(client_data == CB_DIR_PATH)
		XtPopdown(textDirPopup);

	return 0;
}

/*
* Popup the input display name text dialog
*/
#if defined(__STDC__) || defined(__cplusplus)
XtCallbackProc popupDpyText(Widget widget, caddr_t client_data, caddr_t call_data)
#else
XtCallbackProc popupDpyText( widget, client_data, call_data)
Widget widget;
caddr_t client_data;
caddr_t call_data;
#endif
{
Dimension width, height;
Position x, y;
Arg args[MAX_ARGS];
int ac;

	ac = 0;
	XtSetArg(args[ac], XtNwidth, (XtArgVal)&width); ac++;
	XtSetArg(args[ac], XtNheight, (XtArgVal)&height); ac++;
	XtGetValues(textDpyW, args, ac);

	XtTranslateCoords(textDpyW, width / 2, height / 2, &x, &y);

	ac = 0;
	XtSetArg(args[ac], XtNvalue, (XtArgVal)remoteDisplay); ac++;
	XtSetValues(textDpyDialog, args, ac);

	ac = 0;
	XtSetArg(args[ac], XtNx, (XtArgVal)x); ac++;
	XtSetArg(args[ac], XtNy, (XtArgVal)y); ac++;
	XtSetArg(args[ac], XtNwidth, (XtArgVal)200); ac++;
	XtSetValues(textDpyPopup, args, ac);
	
	XtPopup(textDpyPopup, XtGrabNone);

	return 0;
}

/*
* Get input string from the text dialog
*/
#if defined(__STDC__) || defined(__cplusplus)
static void getDisplayName(Widget widget, CBActionType client_data, caddr_t call_data)
#else
static void getDisplayName( widget, client_data, call_data)
Widget widget;
CBActionType client_data;
caddr_t call_data;
#endif
{
String string;
Arg args[MAX_ARGS];
int ac;
#if defined (_MOTIF)
XmString xms;
#endif

#if defined (_MOTIF)
	string = XmTextGetString(textDpyDialog);
#else
	string = XtDialogGetValueString(textDpyDialog);
#endif
	strcpy(remoteDisplay, string);

	XtPopdown(textDpyPopup);

	sprintf(msgdata,"Target Display = %s",remoteDisplay);
	ac = 0;
#if defined (_MOTIF)
	xms = XmStringCreate(msgdata, XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, (XtArgVal)xms); ac++;
	XtSetArg(args[ac], XmNalignment, (XtArgVal)XmALIGNMENT_BEGINNING); ac++;
#else
	XtSetArg(args[ac], XtNlabel, (XtArgVal)msgdata); ac++;
#endif

	XtSetValues(dpyNameW, args, ac);

#if defined (_MOTIF)
	XmStringFree(xms);
#endif

}

#if defined(__STDC__) || defined(__cplusplus)
XtCallbackProc cancelDisplayName(Widget widget, caddr_t client_data, caddr_t call_data)
#else
XtCallbackProc cancelDisplayName( widget, client_data, call_data)
Widget widget;
caddr_t client_data;
caddr_t call_data;
#endif
{
	XtPopdown(textDpyPopup);
	return 0;
}


/*
* Popup the Command line option input text dialog
*/
#if defined(__STDC__) || defined(__cplusplus)
XtCallbackProc popupOptText(Widget widget, caddr_t client_data, caddr_t call_data)
#else
XtCallbackProc popupOptText( widget, client_data, call_data)
Widget widget;
caddr_t client_data;
caddr_t call_data;
#endif
{
Dimension width, height;
Position x, y;
Arg args[MAX_ARGS];
int ac;

	ac = 0;
	XtSetArg(args[ac], XtNwidth, (XtArgVal)&width); ac++;
	XtSetArg(args[ac], XtNheight, (XtArgVal)&height); ac++;
	XtGetValues(textOptW, args, ac);

	XtTranslateCoords(textOptW, width / 2, height / 2, &x, &y);

	ac = 0;
	XtSetArg(args[ac], XtNx, (XtArgVal)x); ac++;
	XtSetArg(args[ac], XtNy, (XtArgVal)y); ac++;
	XtSetValues(textOptPopup, args, ac);
	
	XtPopup(textOptPopup, XtGrabNone);
	return 0;
}

/*
* Get input string from the text dialog
*/
#if defined(__STDC__) || defined(__cplusplus)
static void popDownCmdLineOptions(Widget widget, CBActionType client_data, caddr_t call_data)
#else
static void popDownCmdLineOptions( widget, client_data, call_data)
Widget widget;
CBActionType client_data;
caddr_t call_data;
#endif
{
	XtPopdown(textOptPopup);
}

#if defined(__STDC__) || defined(__cplusplus)
XtCallbackProc cmdLineOptCB(Widget widget, caddr_t client_data, caddr_t call_data)
#else
XtCallbackProc cmdLineOptCB( widget, client_data, call_data)
Widget widget;
caddr_t client_data;
caddr_t call_data;
#endif
{
int set;

#if defined (_MOTIF)
	set = !(int)((XmToggleButtonCallbackStruct *)call_data)->set;
#else
	set = (int)call_data;
#endif

#if defined (_DEBUG)
	fprintf(stderr,"button %d = %d\n",(int)client_data, set);
#endif

	switch((int)client_data){
	case CMDOPT_CLOCK:
		if((int)set == True)
			strcat(cmdLineOptClock,"-clock");
		else
			cmdLineOptClock[0] = 0;
		break;
	case CMDOPT_BIFF:
		if((int)set == True)
			strcat(cmdLineOptBiff,"-biff");
		else
			cmdLineOptBiff[0] = 0;
		break;
	case CMDOPT_SHOWFNAME:
		if((int)set == True)
			strcat(cmdLineOptShowFileName,"-fname");
		else
			cmdLineOptShowFileName[0] = 0;
		break;
	case CMDOPT_ROOT:
		if((int)set == True)
			strcat(cmdLineOptRoot,"-root");
		else
			cmdLineOptRoot[0] = 0;
		break;
	case CMDOPT_FULL:
		if((int)set == True)
			strcat(cmdLineOptFull,"-full");
		else
			cmdLineOptFull[0] = 0;
		break;
	case CMDOPT_OVERRIDE:
		if((int)set == True)
			strcat(cmdLineOptOverride,"-override");
		else
			cmdLineOptOverride[0] = 0;
		break;
	case CMDOPT_RESIZE:
		if((int)set == True)
			strcat(cmdLineOptResize,"-resize");
		else
			cmdLineOptResize[0] = 0;
		break;
	case CMDOPT_PRIVATE:
		if((int)set == True)
			strcat(cmdLineOptPrivate,"-private");
		else
			cmdLineOptPrivate[0] = 0;
		break;
	case CMDOPT_PERFECT:
		if((int)set == True)
			strcat(cmdLineOptPerfect,"-perfect");
		else
			cmdLineOptPerfect[0] = 0;
		break;
	default:
		break;
	}

	return 0;
}


/*
* Popup the interval time input text dialog
*/
#if defined(__STDC__) || defined(__cplusplus)
XtCallbackProc popupIntText(Widget widget, caddr_t client_data, caddr_t call_data)
#else
XtCallbackProc popupIntText( widget, client_data, call_data)
Widget widget;
caddr_t client_data;
caddr_t call_data;
#endif
{
Dimension width, height;
Position x, y;
Arg args[MAX_ARGS];
int ac;

	ac = 0;
	XtSetArg(args[ac], XtNwidth, (XtArgVal)&width); ac++;
	XtSetArg(args[ac], XtNheight, (XtArgVal)&height); ac++;
	XtGetValues(textIntW, args, ac);

	XtTranslateCoords(textIntW, width / 2, height / 2, &x, &y);

	ac = 0;
	XtSetArg(args[ac], XtNvalue, (XtArgVal)intData); ac++;
	XtSetValues(textIntDialog, args, ac);

	ac = 0;
	XtSetArg(args[ac], XtNx, (XtArgVal)x); ac++;
	XtSetArg(args[ac], XtNy, (XtArgVal)y); ac++;
	XtSetArg(args[ac], XtNwidth, (XtArgVal)200); ac++;
	XtSetValues(textIntPopup, args, ac);
	
	XtPopup(textIntPopup, XtGrabNone);
	return 0;
}

#if defined(__STDC__) || defined(__cplusplus)
XtCallbackProc cancelIntText(Widget widget, caddr_t client_data, caddr_t call_data)
#else
XtCallbackProc cancelIntText( widget, client_data, call_data)
Widget widget;
caddr_t client_data;
caddr_t call_data;
#endif
{
	XtPopdown(textIntPopup);
	return 0;
}


/*
* Popup the file path directory input text dialog
*/
#if defined(__STDC__) || defined(__cplusplus)
XtCallbackProc popupDirText(Widget widget, caddr_t client_data, caddr_t call_data)
#else
XtCallbackProc popupDirText( widget, client_data, call_data)
Widget widget;
caddr_t client_data;
caddr_t call_data;
#endif
{
Dimension width, height;
Position x, y;
Arg args[MAX_ARGS];
register int ac;

	ac = 0;
	XtSetArg(args[ac], XtNwidth, (XtArgVal)&width); ac++;
	XtSetArg(args[ac], XtNheight, (XtArgVal)&height); ac++;
	XtGetValues(textDirW, args, ac);

	XtTranslateCoords(textDirW, width / 2, height / 2, &x, &y);

	ac = 0;
	XtSetArg(args[ac], XtNx, (XtArgVal)x); ac++;
	XtSetArg(args[ac], XtNy, (XtArgVal)y); ac++;
	XtSetArg(args[ac], XtNwidth, (XtArgVal)400); ac++;
	XtSetValues(textDirPopup, args, ac);
	
	XtPopup(textDirPopup, XtGrabNone);
	return 0;
}


#if defined(__STDC__) || defined(__cplusplus)
XtCallbackProc cancelDirText(Widget widget, caddr_t client_data, caddr_t call_data)
#else
XtCallbackProc cancelDirText( widget, client_data, call_data)
Widget widget;
caddr_t client_data;
caddr_t call_data;
#endif
{
	XtPopdown(textDirPopup);
	return 0;
}

/*		Function Name: ConvertColor
 *		Description: This converts a string into a color.
 *		Arguments: color_name - name of the color.
 *		Returns: a pixel value for that color.
 */

#if defined(__STDC__) || defined(__cplusplus)
static int ConvertColor(Widget w, char *color_name)
#else
static int ConvertColor(w, color_name)
Widget w;
char * color_name;
#endif
{
XrmValue from, to;

	from.size = strlen(color_name) + 1;	
	from.addr = color_name;

	/*
	 * This conversion accepts a colorname from rgb.txt, or a #rrrgggbbb 
	 * rgb color definition, or the special toolkit strings "XtDefaultForeground" 
	 * and "XtDefaultBackground".
	 */

	XtConvert(w, XtRString, (XrmValuePtr) &from, XtRPixel, (XrmValuePtr) &to);
	if (to.addr == NULL) {
		return(-1);
	}

	return( (int) *((Pixel *) to.addr) );
}

/*
* Setup text dialog
*/
void initTextPopup()
{
Widget tmpw;
Arg args[MAX_ARGS];
int ac, i;
#if defined (_MOTIF)
Widget dialog,form;
XmString xms;
#endif

#if defined (_MOTIF)
	/* Create Input File Path Directory Dialog */
	textDirPopup = XtCreatePopupShell(
			NULL, transientShellWidgetClass, vpaned, NULL, 0);

	ac = 0;
	dialog = XtCreateManagedWidget(
					NULL,
					boxWidgetClass,
					textDirPopup,
					args, ac
					);

	/* Text field */
	ac = 0;
	xms = XmStringCreate("Set File Path Directory", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, xms); ac++;
	XtSetArg(args[ac], XmNalignment, XmALIGNMENT_BEGINNING); ac++;
	tmpw = XtCreateManagedWidget( NULL, labelWidgetClass, dialog, args, ac);

	ac = 0;
	XtSetArg(args[ac], XmNeditable, True); ac++;
	XtSetArg(args[ac], XmNeditMode, XmMULTI_LINE_EDIT); ac++;
	XtSetArg(args[ac], XmNtopPosition, 0); ac++;
	XtSetArg(args[ac], XmNautoShowCursorPosition, True); ac++;
	XtSetArg(args[ac], XmNrows, 3); ac++;
	textDirDialog = XtCreateManagedWidget(
			NULL, xmTextWidgetClass, dialog, args, ac);
	XmStringFree(xms);

	/* button field */
	ac = 0;
	form = XtCreateManagedWidget(
			NULL, xmFormWidgetClass, dialog, args, ac);

	/* Ok button */
	ac = 0;
	xms = XmStringCreate("Ok", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, xms); ac++;
	XtSetArg(args[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	tmpw = XtCreateManagedWidget(
			NULL, commandWidgetClass, form, args, ac);
	XtAddCallback(tmpw, XmNactivateCallback, (XtPointer)sendStringMessage, (caddr_t)CB_DIR_PATH);
	XmStringFree(xms);

	/* Cancel button */
	ac = 0;
	xms = XmStringCreate("Cancel", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, xms); ac++;
	XtSetArg(args[ac], XmNleftAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(args[ac], XmNleftWidget, tmpw); ac++;
	tmpw = XtCreateManagedWidget(
			NULL, commandWidgetClass, form, args, ac);
	XtAddCallback(tmpw, XmNactivateCallback, (XtPointer)cancelDirText, NULL);
	XmStringFree(xms);
	
	/* Create Input Display Name Dialog */
	textDpyPopup = XtCreatePopupShell(
			NULL, transientShellWidgetClass, vpaned, NULL, 0);

	ac = 0;
	dialog = XtCreateManagedWidget(
					NULL,
					boxWidgetClass,
					textDpyPopup,
					args, ac
					);

	/* Text field */
	ac = 0;
	xms = XmStringCreate("Set Target Display Name", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, xms); ac++;
	XtSetArg(args[ac], XmNalignment, XmALIGNMENT_BEGINNING); ac++;
	tmpw = XtCreateManagedWidget( NULL, labelWidgetClass, dialog, args, ac);

	ac = 0;
	textDpyDialog = XtCreateManagedWidget(
			"textDpyDialog", xmTextWidgetClass, dialog, args, ac);
	XmStringFree(xms);

	/* button field */
	ac = 0;
	form = XtCreateManagedWidget(
			NULL, xmFormWidgetClass, dialog, args, ac);

	/* Ok button */
	ac = 0;
	xms = XmStringCreate("Ok", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, xms); ac++;
	XtSetArg(args[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	tmpw = XtCreateManagedWidget(
			NULL, commandWidgetClass, form, args, ac);
	XtAddCallback(tmpw, XmNactivateCallback, (XtPointer)getDisplayName, NULL);
	XmStringFree(xms);

	/* Cancel button */
	ac = 0;
	xms = XmStringCreate("Cancel", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, xms); ac++;
	XtSetArg(args[ac], XmNleftAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(args[ac], XmNleftWidget, tmpw); ac++;
	tmpw = XtCreateManagedWidget(
			NULL, commandWidgetClass, form, args, ac);
	XtAddCallback(tmpw, XmNactivateCallback, (XtPointer)cancelDisplayName, NULL);
	XmStringFree(xms);
	
	/* Create Input Interval Time Dialog */
	textIntPopup = XtCreatePopupShell(
			NULL, transientShellWidgetClass, vpaned, NULL, 0);

	ac = 0;
	dialog = XtCreateManagedWidget(
					NULL,
					boxWidgetClass,
					textIntPopup,
					args, ac
					);

	/* Text field */
	ac = 0;
	xms = XmStringCreate("Set Interval Time (sec)", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, xms); ac++;
	XtSetArg(args[ac], XmNalignment, XmALIGNMENT_BEGINNING); ac++;
	tmpw = XtCreateManagedWidget( NULL, labelWidgetClass, dialog, args, ac);

	ac = 0;
	textIntDialog = XtCreateManagedWidget(
			"textIntDialog", xmTextWidgetClass, dialog, args, ac);
	XmStringFree(xms);

	/* button field */
	ac = 0;
	form = XtCreateManagedWidget(
			NULL, xmFormWidgetClass, dialog, args, ac);

	/* Ok button */
	ac = 0;
	xms = XmStringCreate("Ok", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, xms); ac++;
	XtSetArg(args[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	tmpw = XtCreateManagedWidget(
			NULL, commandWidgetClass, form, args, ac);
	XtAddCallback(tmpw, XmNactivateCallback, (XtPointer)sendIntMessage, (caddr_t)0);
	XmStringFree(xms);

	/* Cancel button */
	ac = 0;
	xms = XmStringCreate("Cancel", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, xms); ac++;
	XtSetArg(args[ac], XmNleftAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(args[ac], XmNleftWidget, tmpw); ac++;
	tmpw = XtCreateManagedWidget(
			NULL, commandWidgetClass, form, args, ac);
	XtAddCallback(tmpw, XmNactivateCallback, (XtPointer)cancelIntText, NULL);
	XmStringFree(xms);
	

	/* Create Input Command Line Option Dialog */
	textOptPopup = XtCreatePopupShell(
			NULL, transientShellWidgetClass, vpaned, NULL, 0);

	ac = 0;
	textOptDialog = XtCreateManagedWidget(
			"textOptDialog", formWidgetClass, textOptPopup, args, ac);

	ac = 0;
	XtSetArg(args[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	for(i=0; opt_buttons[i].opt != NULL; i++){
		xms = XmStringCreate(opt_buttons[i].opt, XmSTRING_DEFAULT_CHARSET);
		XtSetArg(args[ac], XmNlabelString, xms); ac++;
		XtSetArg(args[ac], XmNindicatorOn, True); ac++;
		XtSetArg(args[ac], XmNselectColor, (XtArgVal)ConvertColor(toplevel, "yellow")); ac++;
		tmpw = XtCreateManagedWidget(
						opt_buttons[i].opt,
						toggleWidgetClass,
						textOptDialog,
						args, ac);
		XtAddCallback(tmpw, XmNarmCallback, (XtPointer)cmdLineOptCB, (caddr_t)opt_buttons[i].cb);
		XmStringFree(xms);

		ac = 0;
		XtSetArg(args[ac], XmNleftAttachment, XmATTACH_WIDGET); ac++;
		XtSetArg(args[ac], XtNfromHoriz, (XtArgVal)tmpw); ac++;
	}

	/* Ok button */
	ac = 0;
	xms = XmStringCreate("Ok", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, xms); ac++;
	XtSetArg(args[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
	XtSetArg(args[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(args[ac], XtNfromVert, (XtArgVal)tmpw); ac++;
	tmpw = XtCreateManagedWidget(
			NULL, commandWidgetClass, textOptDialog, args, ac);
	XtAddCallback(tmpw, XmNactivateCallback, (XtPointer)popDownCmdLineOptions, (caddr_t)0);
	XmStringFree(xms);

	
	
#else

	/* Create Input File Path Directory Dialog */
	textDirPopup = XtCreatePopupShell(
			NULL, transientShellWidgetClass, vpaned, NULL, 0);

	ac = 0;
	XtSetArg(args[ac], XtNlabel, (XtArgVal)"Set File Path Directory"); ac++;
	XtSetArg(args[ac], XtNvalue, (XtArgVal)""); ac++;
	textDirDialog = XtCreateManagedWidget(
			NULL, dialogWidgetClass, textDirPopup, args, ac);

	ac = 0;
	XtSetArg(args[ac], XtNlabel, (XtArgVal)"Ok"); ac++;
	tmpw = XtCreateManagedWidget(
			NULL, commandWidgetClass, textDirDialog, args, ac);
	XtAddCallback(tmpw, XtNcallback, (XtPointer)sendStringMessage, (caddr_t)CB_DIR_PATH);

	ac = 0;
	XtSetArg(args[ac], XtNlabel, (XtArgVal)"Cancel"); ac++;
	tmpw = XtCreateManagedWidget(
			NULL, commandWidgetClass, textDirDialog, args, ac);
	XtAddCallback(tmpw, XtNcallback, (XtPointer)cancelDirText, NULL);


	/* Create Input Display Name Dialog */
	textDpyPopup = XtCreatePopupShell(
			NULL, transientShellWidgetClass, vpaned, NULL, 0);

	ac = 0;
	XtSetArg(args[ac], XtNlabel, (XtArgVal)"Set Target Display Name"); ac++;
	XtSetArg(args[ac], XtNvalue, (XtArgVal)""); ac++;
	textDpyDialog = XtCreateManagedWidget(
			"textDpyDialog", dialogWidgetClass, textDpyPopup, args, ac);

	ac = 0;
	XtSetArg(args[ac], XtNlabel, (XtArgVal)"Ok"); ac++;
	tmpw = XtCreateManagedWidget(
			NULL, commandWidgetClass, textDpyDialog, args, ac);
	XtAddCallback(tmpw, XtNcallback, (XtPointer)getDisplayName, (caddr_t)0);

	ac = 0;
	XtSetArg(args[ac], XtNlabel, (XtArgVal)"Cancel"); ac++;
	tmpw = XtCreateManagedWidget(
			NULL, commandWidgetClass, textDpyDialog, args, ac);
	XtAddCallback(tmpw, XtNcallback, (XtPointer)cancelDisplayName, NULL);

	/* Create Input Interval Time Dialog */
	textIntPopup = XtCreatePopupShell(
			NULL, transientShellWidgetClass, vpaned, NULL, 0);

	ac = 0;
	XtSetArg(args[ac], XtNlabel, (XtArgVal)"Set Interval Time (sec)"); ac++;
	XtSetArg(args[ac], XtNvalue, (XtArgVal)""); ac++;
	textIntDialog = XtCreateManagedWidget(
			"textIntDialog", dialogWidgetClass, textIntPopup, args, ac);

	ac = 0;
	XtSetArg(args[ac], XtNlabel, (XtArgVal)"Ok"); ac++;
	tmpw = XtCreateManagedWidget(
			NULL, commandWidgetClass, textIntDialog, args, ac);
	XtAddCallback(tmpw, XtNcallback, (XtPointer)sendIntMessage, (caddr_t)0);

	ac = 0;
	XtSetArg(args[ac], XtNlabel, (XtArgVal)"Cancel"); ac++;
	tmpw = XtCreateManagedWidget(
			NULL, commandWidgetClass, textIntDialog, args, ac);
	XtAddCallback(tmpw, XtNcallback, (XtPointer)cancelIntText, NULL);

	/* Create Input Command Line Option Dialog */
	textOptPopup = XtCreatePopupShell(
			NULL, transientShellWidgetClass, vpaned, NULL, 0);

	ac = 0;
	textOptDialog = XtCreateManagedWidget(
			"textOptDialog", formWidgetClass, textOptPopup, args, ac);

	ac = 0;
	for(i=0; opt_buttons[i].opt != NULL; i++){
		tmpw = XtCreateManagedWidget(
						opt_buttons[i].opt,
						toggleWidgetClass,
						textOptDialog,
						args, ac);
		XtAddCallback(tmpw, XtNcallback, (XtPointer)cmdLineOptCB, (caddr_t)opt_buttons[i].cb);
		ac = 0;
		XtSetArg(args[ac], XtNfromHoriz, (XtArgVal)tmpw); ac++;
	}

	ac = 0;
	XtSetArg(args[ac], XtNfromVert, (XtArgVal)tmpw); ac++;
	XtSetArg(args[ac], XtNlabel, (XtArgVal)"Ok"); ac++;
	tmpw = XtCreateManagedWidget(
			NULL, commandWidgetClass, textOptDialog, args, ac);
	XtAddCallback(tmpw, XtNcallback, (XtPointer)popDownCmdLineOptions, (caddr_t)0);

#endif
}


/*
* Button Callback
*/
#if defined(__STDC__) || defined(__cplusplus)
XtCallbackProc ButtonCB(Widget w, CBActionType client_data, caddr_t call_data)
#else
XtCallbackProc ButtonCB( w, client_data, call_data)
Widget w;
CBActionType client_data;
caddr_t call_data;
#endif
{
Display *display;
Window found_window = 0;
XClientMessageEvent event;
Window dummy_window, *children_return;
unsigned int nchildren_return;
int i;

	sprintf(animation,"slide");

	switch(client_data){

	case CB_HEAD:
		sprintf(msgdata,"action = head");
		if(nFiles > 0) {
#if defined (_MOTIF)
			XmListSelectPos(fileListW, 0, True);
#else
			XawListHighlight(fileListW, 0);
#endif
		}
	break;

	case CB_TAIL:
		sprintf(msgdata,"action = tail");
		if(nFiles > 0){
#if defined (_MOTIF)
			XmListSelectPos(fileListW, nFiles-1, True);
#else
			XawListHighlight(fileListW, nFiles-1);
#endif
		}
	break;

	case CB_PAUSE:
		sprintf(msgdata,"action = pause");
		break;
	case CB_PREV:
		sprintf(msgdata,"action = prev");
		break;
	case CB_STOP:
		sprintf(msgdata,"action = stop");
		break;
	case CB_NEXT:
		sprintf(msgdata,"action = next");
		break;

	case CB_SLIDE:
		sprintf(msgdata,"animate = slide");
		sprintf(animation,"slide");
		break;
	case CB_FADE:
		sprintf(msgdata,"animate = fade");
		sprintf(animation,"fade");
		break;
	case CB_REVERSE:
		sprintf(msgdata,"animate = reverse");
		sprintf(animation,"reverse");
		break;
	case CB_MONOTONE:
		sprintf(msgdata,"animate = monotone");
		sprintf(animation,"monotone");
		break;
	case CB_FLUSH:
		sprintf(msgdata,"animate = flush");
		sprintf(animation,"flush");
		break;
	case CB_WIPE:
		sprintf(msgdata,"animate = wipe");
		break;
	case CB_WIPE24:
		sprintf(msgdata,"animate = wipe24");
		sprintf(animation,"wipe24");
		break;
	case CB_BLOCK:
		sprintf(msgdata,"animate = block");
		sprintf(animation,"block");
		break;
	case CB_BLOCK24:
		sprintf(msgdata,"animate = block24");
		sprintf(animation,"block24");
		break;
	case CB_MOSAIC:
		sprintf(msgdata,"animate = mosaic");
		sprintf(animation,"mosaic");
		break;
	case CB_COLORANIM:
		sprintf(msgdata,"animate = coloranim");
		sprintf(animation,"coloranim");
		break;
	case CB_COLORANIM2:
		sprintf(msgdata,"animate = coloranim2");
		sprintf(animation,"coloranim2");
		break;
	case CB_SHUFFLE:
		sprintf(msgdata,"animate = shuffle");
		sprintf(animation,"shuffle");
		break;
	case CB_PUZZLE:
		sprintf(msgdata,"animate = puzzle");
		sprintf(animation,"puzzle");
		break;
	case CB_SLIT:
		sprintf(msgdata,"animate = slit");
		sprintf(animation,"slit");
		break;
	case CB_TRANSPARENT:
		sprintf(msgdata,"animate = transparent");
		sprintf(animation,"transparent");
		break;
	case CB_MOVE:
		sprintf(msgdata,"animate = move");
		sprintf(animation,"move");
		break;
	case CB_FLOAT24:
		sprintf(msgdata,"animate = float24");
		sprintf(animation,"float24");
		break;

	case CB_DEFAULT:   sprintf(msgdata,"Set animation mode to default"); break;

	case CB_CLOSE:     exit(0);

	default: return 0;
		break;
	}
	print_message(msgdata);

	/* Open remote display */
	if(remoteDisplay[0] != 0){
		display = XOpenDisplay(remoteDisplay);
		if(display == NULL) {
			sprintf(msgdata,"Connection Failed %s",remoteDisplay);
			print_message(msgdata);
			return 0;
		}
	}
	else{
		display = XtDisplay(w);
	}

	/* Find xslideshow window */
	if( !XQueryTree(display,
		RootWindowOfScreen(DefaultScreenOfDisplay(display)),
		&dummy_window, &dummy_window,
		&children_return, &nchildren_return) )
		return 0;

	for (i = 0; i < (int)nchildren_return; i++)
		if((found_window = find_xslideshow_window( display, children_return[i] )))
			break;

	/* If no xslideshow is running */
	if ( ! found_window ) {

		if(client_data == CB_PAUSE
			|| client_data == CB_HEAD
			|| client_data == CB_PREV
			|| client_data == CB_NEXT
			|| client_data == CB_TAIL
			|| client_data == CB_SLIDE
			|| client_data == CB_FADE
			|| client_data == CB_REVERSE
			|| client_data == CB_MONOTONE
			|| client_data == CB_FLUSH
			|| client_data == CB_WIPE
			|| client_data == CB_WIPE24
			|| client_data == CB_BLOCK
			|| client_data == CB_BLOCK24
			|| client_data == CB_MOSAIC
			|| client_data == CB_COLORANIM
			|| client_data == CB_COLORANIM2
			|| client_data == CB_SHUFFLE
			|| client_data == CB_PUZZLE
			|| client_data == CB_SLIT
			|| client_data == CB_TRANSPARENT
			|| client_data == CB_MOVE
			|| client_data == CB_FLOAT24
			|| client_data == CB_DEFAULT
			){

			if(remoteDisplay[0] == 0)
				strcpy(remoteDisplay,DisplayString(display));

			/* exec xslideshow */
			if(fork() == 0){

				execlp("xslideshow",
						"xslideshow",
						"-display",remoteDisplay,
						cmdLineOptClock,
						cmdLineOptBiff,
						cmdLineOptShowFileName,
						cmdLineOptResize,
						cmdLineOptPrivate,
						cmdLineOptPerfect,
						cmdLineOptRoot,
						cmdLineOptFull,
						cmdLineOptOverride,
						"-animate",animation,
						0);
			}
			else {
				sprintf(msgdata,
						"exec xslideshow -display %s %s %s %s %s %s %s %s %s %s -animate %s",
						remoteDisplay,
						cmdLineOptClock,
						cmdLineOptBiff,
						cmdLineOptShowFileName,
						cmdLineOptResize,
						cmdLineOptPrivate,
						cmdLineOptPerfect,
						cmdLineOptRoot,
						cmdLineOptFull,
						cmdLineOptOverride,
						animation
						);

				print_message(msgdata);
			}

		}
		else {
			sprintf(msgdata,
					"No xslideshow is running on display %s",
					DisplayString(display));
			print_message(msgdata);
			return 0;
		}
	}

	/* Send button action message to the xslideshow */
	event.display      = display;
	event.window       = found_window;
	event.type         = ClientMessage;
	event.format       = 32;
	event.message_type = XInternAtom(display, AtomXslideshowCommControl, False);
	event.data.l[0]    = (long)client_data;

	if ( !XSendEvent ( display, found_window, True, 0L, (XEvent *)&event) ) {
		fprintf (stderr,
				"xslideshow-control: XSendEvent(display, 0x%x ...) failed.\n",
				(unsigned int) found_window);
		exit(1);
	}
	XSync (display, False);

	/* Close remote display */
	if(remoteDisplay[0] != 0)
		XCloseDisplay(display);

	return 0;
}

/*
* wm_quit
*/
#if defined(__STDC__) || defined(__cplusplus)
static void wm_quit(Widget w,XEvent *event,String *params,Cardinal *num_params)
#else
static void wm_quit(w,event,params,num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
#endif
{
	if (event->type == ClientMessage &&
		(Atom)event->xclient.data.l[0] != wm_delete_window) {
		XBell (XtDisplay(w), 0);
		return;
	}
	XCloseDisplay(XtDisplay(w));
	exit (0);
}

/*
* List Proc Callback
*/
#if defined(__STDC__) || defined(__cplusplus)
XtCallbackProc ListProcCB(Widget w, caddr_t client_data, caddr_t call_data)
#else
XtCallbackProc ListProcCB(w, client_data, call_data)
Widget w;
caddr_t client_data;
caddr_t call_data;
#endif
{
#if defined (_MOTIF)
char *text;
XmListCallbackStruct *list_ret = (XmListCallbackStruct *)call_data;
	XmStringGetLtoR(list_ret->item, XmSTRING_DEFAULT_CHARSET, &text);
	strcpy(currentFileName, text);
#else
XtListReturnStruct *list_ret = (XtListReturnStruct *)call_data;
	strcpy(currentFileName, list_ret->string);
#endif
	return 0;
}

/* Receive message from xslideshow */
#if defined(__STDC__) || defined(__cplusplus)
void MessageHandler(Widget w, caddr_t client_data, XEvent *event)
#else
void MessageHandler(w, client_data, event)
Widget w;
caddr_t client_data;
XEvent *event;
#endif
{
int i;
#if defined (_MOTIF)
XmString xms;
#endif
char *tmpBufp, *bufp, buf[MAX_TEXT_LEN];

	if(event->type == ClientMessage){

		/* Get the number of files */
		if(event->xclient.message_type == XA_XSLIDESHOWCONTROL_NFILES){
			nFiles = event->xclient.data.l[0];
			if(nFiles == 0) /* No files */
				return;

			if(pFileListTable != NULL){ /* Free previous list */
				for(i=0; i<prevNFiles; i++){
					if(pFileListTable[i] != NULL)
						XtFree((char *)pFileListTable[i]);
				}
				XtFree((char *)pFileListTable);
			}
			pFileListTable = (char **)XtMalloc(sizeof(char *) * nFiles);

			if(pRcvBuf != NULL) /* Free previous receive buffer */
				XtFree((char *)pRcvBuf);
			pRcvBuf = (char *)XtMalloc(sizeof(char) * DEFAULT_BUF_SIZ);

			bufLen = DEFAULT_BUF_SIZ;
			prevNFiles = nFiles;
			rcvLen = 0;
#if defined (_DEBUG)
			fprintf(stderr,"xslideshow-control: nFiles = %d\n",nFiles);
#endif
		}

		/* Get file list strings */
		else if(event->xclient.message_type == XA_XSLIDESHOWCONTROL_FLIST){
			{
			int i;
				for(i=0; i<20; i++){
					if(event->xclient.data.b[i] == '\0') break;
					if(rcvLen >= bufLen){	/* Resize receive buffer */
						pRcvBuf = XtRealloc(pRcvBuf, bufLen + sizeof(char) * DEFAULT_BUF_SIZ);
						bufLen += DEFAULT_BUF_SIZ;
					}
					pRcvBuf[rcvLen++] = event->xclient.data.b[i];
				}
			}
		}

		/* Get the end of file list */
		else if(event->xclient.message_type == XA_XSLIDESHOWCONTROL_FLIST_END){
			nIndex = event->xclient.data.l[0];
			if(rcvLen >= bufLen){
				pRcvBuf = XtRealloc(pRcvBuf, bufLen + sizeof(char) * DEFAULT_BUF_SIZ);
			}
			pRcvBuf[rcvLen] = 0;
			tmpBufp = pRcvBuf;

			for(i=0; i<nFiles; i++){
				sscanf(tmpBufp,"%s",buf);
				tmpBufp += (strlen(buf)+1);
				bufp = (char *)XtCalloc(strlen(buf)+1,sizeof(char));	
				strcpy(bufp,buf);
				pFileListTable[i] = bufp;

#if defined (_DEBUG)
				fprintf(stderr,"xslideshow-control: %s\n",bufp);
#endif
			}
	
			/* Update the list widget */
#if defined (_MOTIF)
			XmListDeleteAllItems(fileListW);
			for(i=1; i <= nFiles; i++){
				xms = XmStringCreate(pFileListTable[nFiles - i], XmSTRING_DEFAULT_CHARSET);
				XmListAddItem(fileListW, xms, 1, 0);
				XmStringFree(xms);
			}
			XmListSelectPos(fileListW, nIndex, True);
#else
			XawListChange(fileListW, pFileListTable, nFiles, 0, True);
			XawListHighlight(fileListW, nIndex);
#endif
		}
	}

}

/*
*
*/
static void createControlButtons()
{
Arg	args[20];
Cardinal ac;
Widget tmpw, tmpw2;
Pixmap tmpx;
int i, pixel;
#if defined (_MOTIF)
XmString xms;
#endif

	/* Pause, Head, Prev, Stop, Next, Tail,      Quit */
	ac = 0;
	command = XtCreateManagedWidget(
					NULL,
					boxWidgetClass,
					vpaned,
					args, ac
					);

	ac = 0;
	tmpw = XtCreateManagedWidget(
					NULL,
					formWidgetClass,
					command,
					args, ac
					);

	ac = 0;
	XtSetArg(args[ac], XtNhorizDistance, (XtArgVal)24*4); ac++;
	for(i=0; control_buttons[i].name != NULL; i++){
		pixel = ConvertColor(toplevel, control_buttons[i].color);
		tmpx = XCreatePixmapFromBitmapData(
					XtDisplay(toplevel),
					DefaultRootWindow(XtDisplay(toplevel)),
					(char *)control_buttons[i].name,
					24, 24,
					pixel,
					BlackPixel(XtDisplay(toplevel), DefaultScreen(XtDisplay(toplevel))),
					DefaultDepth(XtDisplay(toplevel), DefaultScreen(XtDisplay(toplevel)))
					);

#if defined (_MOTIF)
		XtSetArg(args[ac], XmNleftAttachment, (XtArgVal)XmATTACH_WIDGET); ac++;
		XtSetArg(args[ac], XmNlabelType, (XtArgVal)XmPIXMAP); ac++;
		XtSetArg(args[ac], XmNlabelPixmap, (XtArgVal)tmpx); ac++;
#else
		XtSetArg(args[ac], XtNbackgroundPixmap, (XtArgVal)tmpx); ac++;
#endif
		XtSetArg(args[ac], XtNwidth, (XtArgVal)24); ac++;
		XtSetArg(args[ac], XtNheight, (XtArgVal)24); ac++;
		XtSetArg(args[ac], XtNforeground, (XtArgVal)ConvertColor(toplevel, control_buttons[i].color)); ac++;
		tmpw2 = XtCreateManagedWidget(
					NULL,
					commandWidgetClass,
					tmpw,
					args, ac
					);
#if defined (_MOTIF)
		XtAddCallback(tmpw2, XmNactivateCallback, (XtPointer)ButtonCB, (caddr_t)control_buttons[i].cb);
#else
		XtAddCallback(tmpw2, XtNcallback, (XtPointer)ButtonCB, (caddr_t)control_buttons[i].cb);
#endif
		ac = 0;
		XtSetArg(args[ac], XtNfromHoriz, (XtArgVal)tmpw2); ac++;
		XtSetArg(args[ac], XtNhorizDistance, (XtArgVal)BUTTON_DISTANCE); ac++;
	}

	/* Quit button */
	ac = 0;
#if defined (_MOTIF)
	XtSetArg(args[ac], XmNleftAttachment, (XtArgVal)XmATTACH_WIDGET); ac++;
#endif
	XtSetArg(args[ac], XtNheight, (XtArgVal)24); ac++;
	XtSetArg(args[ac], XtNfromHoriz, (XtArgVal)tmpw2); ac++;
	XtSetArg(args[ac], XtNhorizDistance, (XtArgVal)24*4); ac++;
#if defined (_MOTIF)
	xms = XmStringCreate("Exit", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, (XtArgVal)xms); ac++;
#else
	XtSetArg(args[ac], XtNlabel, (XtArgVal)"Exit"); ac++;
#endif
	quit = XtCreateManagedWidget(
					"quit",
					commandWidgetClass,
					tmpw,
					args, ac
					);
#if defined (_MOTIF)
	XtAddCallback(quit, XmNactivateCallback, (XtPointer)ButtonCB, (caddr_t)CB_CLOSE);
#else
	XtAddCallback(quit, XtNcallback, (XtPointer)ButtonCB, (caddr_t)CB_CLOSE);
#endif
}


/*
* main
*/
#if defined(__STDC__) || defined(__cplusplus)
void main( int argc, char **argv)
#else
void main( argc, argv)
int argc;
char **argv;
#endif
{
XtAppContext app_con;
ApplicationData app_data;
Arg	args[20];
Cardinal ac;
Widget tmpw;
int i;
#if defined (_MOTIF)
XmString xms;
#endif

	currentFileName[0] = 0;
	cmdLineOptClock[0] = 0;
	cmdLineOptBiff[0] = 0;
	cmdLineOptShowFileName[0] = 0;
	cmdLineOptRoot[0] = 0;
	cmdLineOptFull[0] = 0;
	cmdLineOptOverride[0] = 0;
	cmdLineOptResize[0] = 0;
	cmdLineOptPrivate[0] = 0;
	cmdLineOptPerfect[0] = 0;

	pProgramName = argv[0];

	toplevel = XtAppInitialize(
					&app_con,
					"XSlideshowControl",
					NULL, 0,
					&argc, argv,
					fallback_resources,
					NULL, 0
					);

	XtGetApplicationResources(toplevel,
					&app_data,
					resources,
					XtNumber(resources),
					NULL, 0
					);

#if !defined (_MOTIF)
	if(app_data.translations != NULL)
		XtOverrideTranslations(toplevel,
					XtParseTranslationTable(app_data.translations));
#endif

	XtAppAddActions(
					(XtAppContext)app_con,
					actionTable,
					XtNumber(actionTable)
					);

	theVisual = DefaultVisual(XtDisplay(toplevel),DefaultScreen(XtDisplay(toplevel)));

	ac = 0;
	vpaned = XtCreateManagedWidget(
					NULL,
					vPanedWidgetClass,
					toplevel,
					args, ac
					);

	createControlButtons();

	/* Create Animation buttons */
	ac = 0;
#if defined (_MOTIF)
	XtSetArg(args[ac], XmNpacking, (XtArgVal)XmPACK_COLUMN); ac++;
	XtSetArg(args[ac], XmNnumColumns, (XtArgVal)5); ac++;
	XtSetArg(args[ac], XmNadjustLast, (XtArgVal)False); ac++;
	XtSetArg(args[ac], XmNorientation, (XtArgVal)XmVERTICAL); ac++;
	XtSetArg(args[ac], XmNisAligned, (XtArgVal)True); ac++;
	XtSetArg(args[ac], XmNentryAlignment, (XtArgVal)XmALIGNMENT_CENTER); ac++;
	XtSetArg(args[ac], XmNspacing, (XtArgVal)3); ac++;
#endif
	anim_command = XtCreateManagedWidget(
					NULL,
					boxWidgetClass,
					vpaned,
					args, ac
					);

	for(i=0; anim_buttons[i].name != NULL; i++) {
		ac = 0;
		XtSetArg(args[ac], XtNwidth, (XtArgVal)BUTTON_WIDTH); ac++;
		XtSetArg(args[ac], XtNforeground, (XtArgVal)ConvertColor(toplevel, anim_buttons[i].color)); ac++;
		tmpw = XtCreateManagedWidget(
						anim_buttons[i].name,
						commandWidgetClass,
						anim_command,
						args, ac
						);
#if defined (_MOTIF)
		XtAddCallback(tmpw, XmNactivateCallback, (XtPointer)ButtonCB, (caddr_t)anim_buttons[i].cb);
#else
		XtAddCallback(tmpw, XtNcallback, (XtPointer)ButtonCB, (caddr_t)anim_buttons[i].cb);
#endif
	}


	/* Create Input Text Buttons */
	ac = 0;
#if defined (_MOTIF)
	XtSetArg(args[ac], XmNpacking, (XtArgVal)XmPACK_COLUMN); ac++;
	XtSetArg(args[ac], XmNadjustLast, (XtArgVal)False); ac++;
	XtSetArg(args[ac], XmNnumColumns, (XtArgVal)2); ac++;
	XtSetArg(args[ac], XmNorientation, (XtArgVal)XmVERTICAL); ac++;
	XtSetArg(args[ac], XmNisAligned, (XtArgVal)True); ac++;
	XtSetArg(args[ac], XmNentryAlignment, (XtArgVal)XmALIGNMENT_CENTER); ac++;
	XtSetArg(args[ac], XmNspacing, (XtArgVal)3); ac++;
#endif
	command = XtCreateManagedWidget(
					NULL,
					boxWidgetClass,
					vpaned,
					args, ac
					);

	/* Create the button for set File path directory */
	ac = 0;
	XtSetArg(args[ac], XtNwidth, (XtArgVal)200); ac++;
#if defined (_MOTIF)
	xms = XmStringCreate("   Set File Path Directory   ", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, (XtArgVal)xms); ac++;
	XtSetArg(args[ac], XmNalignment, (XtArgVal)XmALIGNMENT_CENTER); ac++;
#else
	XtSetArg(args[ac], XtNlabel, (XtArgVal)"Set File Path Directory"); ac++;
#endif
	textDirW = XtCreateManagedWidget(
					"textdir",
					commandWidgetClass,
					command,
					args, ac
					);
#if defined (_MOTIF)
	XmStringFree(xms);
	XtAddCallback(textDirW, XmNactivateCallback, (XtPointer)popupDirText, NULL);
#else
	XtAddCallback(textDirW, XtNcallback, (XtPointer)popupDirText, NULL);
#endif

	/* Create the button for set Display Name */
	ac = 0;
	XtSetArg(args[ac], XtNwidth, (XtArgVal)200); ac++;
#if defined (_MOTIF)
	xms = XmStringCreate("   Set Target Display Name   ", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, (XtArgVal)xms); ac++;
	XtSetArg(args[ac], XmNalignment, (XtArgVal)XmALIGNMENT_CENTER); ac++;
#else
	XtSetArg(args[ac], XtNlabel, (XtArgVal)"Set Target Display Name"); ac++;
#endif
	textDpyW = XtCreateManagedWidget(
					"textdpy",
					commandWidgetClass,
					command,
					args, ac
					);
#if defined (_MOTIF)
	XmStringFree(xms);
	XtAddCallback(textDpyW, XmNactivateCallback, (XtPointer)popupDpyText, NULL);
#else
	XtAddCallback(textDpyW, XtNcallback, (XtPointer)popupDpyText, NULL);
#endif

	/* Create the button for set Interval Time */
	ac = 0;
	XtSetArg(args[ac], XtNwidth, (XtArgVal)200); ac++;
#if defined (_MOTIF)
	xms = XmStringCreate("   Set Interval Time (sec)   ", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, (XtArgVal)xms); ac++;
	XtSetArg(args[ac], XmNalignment, (XtArgVal)XmALIGNMENT_CENTER); ac++;
#else
	XtSetArg(args[ac], XtNlabel, (XtArgVal)"Set Interval Time (sec)"); ac++;
#endif
	textIntW = XtCreateManagedWidget(
					"textint",
					commandWidgetClass,
					command,
					args, ac
					);
#if defined (_MOTIF)
	XmStringFree(xms);
	XtAddCallback(textIntW, XmNactivateCallback, (XtPointer)popupIntText, NULL);
#else
	XtAddCallback(textIntW, XtNcallback, (XtPointer)popupIntText, NULL);
#endif

	/* Create the button for set Command line Options */
	ac = 0;
	XtSetArg(args[ac], XtNwidth, (XtArgVal)200); ac++;
#if defined (_MOTIF)
	xms = XmStringCreate("   Set Command Line Options   ", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, (XtArgVal)xms); ac++;
	XtSetArg(args[ac], XmNalignment, (XtArgVal)XmALIGNMENT_CENTER); ac++;
#else
	XtSetArg(args[ac], XtNlabel, (XtArgVal)"Set Command Line Options"); ac++;
#endif
	textOptW = XtCreateManagedWidget(
					"textopt",
					commandWidgetClass,
					command,
					args, ac
					);
#if defined (_MOTIF)
	XmStringFree(xms);
	XtAddCallback(textOptW, XmNactivateCallback, (XtPointer)popupOptText, NULL);
#else
	XtAddCallback(textOptW, XtNcallback, (XtPointer)popupOptText, NULL);
#endif

	initTextPopup();

	/* Create File List Window */
	pFileListTable = NULL;
	pRcvBuf        = NULL;
	nFiles         = 0;
	prevNFiles     = 0;
	rcvLen         = 0;

	command = XtCreateManagedWidget(
					NULL,
					boxWidgetClass,
					vpaned,
					NULL, 0
					);

	ac = 0;
	XtSetArg(args[ac], XtNresizable, (XtArgVal)TRUE); ac++;
	listFormW = XtCreateManagedWidget(
					NULL,
					formWidgetClass,
					command,
					args, ac
					);

	/* Create the button for getting the File List */
	ac = 0;
	XtSetArg(args[ac], XtNwidth, (XtArgVal)400); ac++;
#if defined (_MOTIF)
	xms = XmStringCreate("Get the file list from the xslideshow", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, (XtArgVal)xms); ac++;
#else
	XtSetArg(args[ac], XtNlabel,
		(XtArgVal)"Get the file list from the xslideshow"); ac++;
#endif
	tmpw = XtCreateManagedWidget(
					"getflist",
					commandWidgetClass,
					listFormW,
					args, ac);
#if defined (_MOTIF)
	XmStringFree(xms);
	XtAddCallback(tmpw, XmNactivateCallback, (XtPointer)sendStringMessage, (caddr_t)CB_REQ_FLIST);
#else
	XtAddCallback(tmpw, XtNcallback, (XtPointer)sendStringMessage, (caddr_t)CB_REQ_FLIST);
#endif

#if defined (_MOTIF)
	listFileView = listFormW;
#else
	/* Viewport */
	ac = 0;
	XtSetArg(args[ac], XtNallowHoriz, (XtArgVal)TRUE); ac++;
	XtSetArg(args[ac], XtNallowVert, (XtArgVal)TRUE); ac++;
	XtSetArg(args[ac], XtNuseBottom, (XtArgVal)TRUE); ac++;
	XtSetArg(args[ac], XtNuseRight, (XtArgVal)TRUE); ac++;
	XtSetArg(args[ac], XtNfromVert, (XtArgVal)tmpw); ac++;
	XtSetArg(args[ac], XtNheight, (XtArgVal)100); ac++;
	XtSetArg(args[ac], XtNwidth, (XtArgVal)400); ac++;
	listFileView = XtCreateManagedWidget(
					"viewPort",
					viewportWidgetClass,
					listFormW,
					args, ac);
#endif

	/* List */
	ac = 0;
#if defined (_MOTIF)
	XtSetArg(args[ac], XmNtopAttachment, (XtArgVal)XmATTACH_WIDGET); ac++;
	XtSetArg(args[ac], XtNfromVert, (XtArgVal)tmpw); ac++;
	XtSetArg(args[ac], XtNdefaultColumns, (XtArgVal)1); ac++;
	XtSetArg(args[ac], XmNscrollBarDisplayPolicy, (XtArgVal)XmAS_NEEDED); ac++;
	XtSetArg(args[ac], XmNscrollingPolicy, (XtArgVal)XmAUTOMATIC); ac++;
	XtSetArg(args[ac], XmNscrollBarPlacement, (XtArgVal)XmBOTTOM_RIGHT); ac++;
	XtSetArg(args[ac], XmNlistSizePolicy, (XtArgVal)XmCONSTANT); ac++;
	XtSetArg(args[ac], XmNvisibleItemCount, (XtArgVal)6); ac++;
	XtSetArg(args[ac], XmNheight, (XtArgVal)100); ac++;
	XtSetArg(args[ac], XmNwidth, (XtArgVal)400); ac++;
	fileListW = XmCreateScrolledList(
					listFileView,
					"flist",
					args, ac);
	XtManageChild(fileListW);
#else
	XtSetArg(args[ac], XtNverticalList, (XtArgVal)TRUE); ac++;
	XtSetArg(args[ac], XtNdefaultColumns, (XtArgVal)1); ac++;
	fileListW = XtCreateManagedWidget(
					"flist",
					listWidgetClass,
					listFileView,
					args, ac);
#endif


#if defined (_MOTIF)
	XtAddCallback(fileListW, XmNbrowseSelectionCallback, (XtPointer)ListProcCB, (caddr_t)0);
#else
	XtAddCallback(fileListW, XtNcallback, (XtPointer)ListProcCB, (caddr_t)0);
#endif

	/* Create the button for selection a File */
	ac = 0;
	XtSetArg(args[ac], XtNfromVert, (XtArgVal)listFileView); ac++;
	XtSetArg(args[ac], XtNwidth, (XtArgVal)400); ac++;
#if defined (_MOTIF)
	xms = XmStringCreate("Set selected file as the xslideshow's current file", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, (XtArgVal)xms); ac++;
	XtSetArg(args[ac], XmNtopAttachment, (XtArgVal)XmATTACH_WIDGET); ac++;
	XtSetArg(args[ac], XtNfromVert, (XtArgVal)fileListW); ac++;
#else
	XtSetArg(args[ac], XtNlabel,
			(XtArgVal)"Set selected file as the xslideshow's current file"); ac++;
#endif
	tmpw = XtCreateManagedWidget(
					"setfile",
					commandWidgetClass,
					listFormW,
					args, ac);
#if defined (_MOTIF)
	XmStringFree(xms);
	XtAddCallback(tmpw, XmNactivateCallback, (XtPointer)sendStringMessage, (caddr_t)CB_SEL_FILE);
#else
	XtAddCallback(tmpw, XtNcallback, (XtPointer)sendStringMessage, (caddr_t)CB_SEL_FILE);
#endif



	/* Create Display Name window */
	ac = 0;
	XtSetArg(args[ac], XtNjustify, XtJustifyLeft); ac++;
	dpyNameW = XtCreateManagedWidget(
				"msg1",
				labelWidgetClass,
				vpaned,
				args, ac
				);
	/* Create Message window */
	messageW = XtCreateManagedWidget(
				"msg2",
				labelWidgetClass,
				vpaned,
				args, ac
				);


	/* Realize the window */
	XtRealizeWidget(toplevel);

	/* Check if Pseaudo or True Color visual class */
	if(theVisual->class == TrueColor) {
		XtSetArg(args[0], XtNsensitive, (XtArgVal)FALSE);
		XtSetValues(XtNameToWidget(anim_command,"fade"), args, 1);
		XtSetValues(XtNameToWidget(anim_command,"reverse"), args, 1);
		XtSetValues(XtNameToWidget(anim_command,"monotone"), args, 1);
		XtSetValues(XtNameToWidget(anim_command,"flush"), args, 1);
		XtSetValues(XtNameToWidget(anim_command,"coloranim"), args, 1);
		XtSetValues(XtNameToWidget(anim_command,"coloranim2"), args, 1);
	}

	wm_delete_window = XInternAtom(XtDisplay(toplevel),"WM_DELETE_WINDOW",False);
	(void)XSetWMProtocols(
			XtDisplay(toplevel),XtWindow(toplevel),&wm_delete_window, 1);

	XtAddEventHandler(toplevel, NoEventMask, TRUE,
				(XtEventHandler)MessageHandler, (XtPointer)NULL);

	XA_XSLIDESHOWCONTROL           = XInternAtom(
												XtDisplay(toplevel),
												AtomXslideshowControl,
												False);
	XA_XSLIDESHOWCONTROL_NFILES    = XInternAtom(
												XtDisplay(toplevel),
												AtomXslideshowControlNfiles,
												False);
	XA_XSLIDESHOWCONTROL_FLIST     = XInternAtom(
												XtDisplay(toplevel),
												AtomXslideshowControlFlist,
												False);
	XA_XSLIDESHOWCONTROL_FLIST_END = XInternAtom(
												XtDisplay(toplevel),
												AtomXslideshowControlFlistEnd,
												False);

	XChangeProperty(
					XtDisplay(toplevel),
					XtWindow(toplevel),
					XA_XSLIDESHOWCONTROL,
					XA_WINDOW,
					32,
					PropModeReplace,
					NULL, 0);


	/* Clear file list */
#if defined (_MOTIF)
	for(i=0; dummy[i] != NULL; i++){
		xms = XmStringCreate(dummy[i], XmSTRING_DEFAULT_CHARSET);
		XmListAddItem(fileListW, xms, 1, 0);
		XmStringFree(xms);
	}
#else
	XawListChange(fileListW, dummy, 0, 0, True);
#endif

	/* show default display name */
	strcpy(remoteDisplay, DisplayString(XtDisplay(toplevel)));
	sprintf(msgdata,"Target Display = %s",remoteDisplay);
	ac = 0;
#if defined (_MOTIF)
	xms = XmStringCreate(msgdata, XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, (XtArgVal)xms); ac++;
	XtSetArg(args[ac], XmNalignment, (XtArgVal)XmALIGNMENT_BEGINNING); ac++;
#else
	XtSetArg(args[ac], XtNlabel, (XtArgVal)msgdata); ac++;
#endif
	XtSetValues(dpyNameW, args, ac);
#if defined (_MOTIF)
	XmStringFree(xms);
#endif

	/* Clear message window */
	ac = 0;
#if defined (_MOTIF)
	xms = XmStringCreate("", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args[ac], XmNlabelString, (XtArgVal)xms); ac++;
	XtSetArg(args[ac], XmNalignment, (XtArgVal)XmALIGNMENT_BEGINNING); ac++;
#else
	XtSetArg(args[ac], XtNlabel, (XtArgVal)""); ac++;
#endif
	XtSetValues(messageW, args, ac);
#if defined (_MOTIF)
	XmStringFree(xms);
#endif


	XtAppMainLoop(app_con);
}

