/*
	Dear Mr. Anthony Dekker.

	I tried to contact with you. But I could not send mail to <tdekker@iscs.nus.sg>.
	I think you are not with National University of Singapore at present.

	This xslideshow is really free software. I hope you give me a permission
	to use your NeuQuant source in xslideshow.

    Thank you and best regards.

	Susumu Shiohara <shiohara@tpp.epson.co.jp> 9th Dec '96 
*/

/*----------------------------------------------------------------------*/
/*																		*/
/*			 					NeuQuant								*/
/*								--------								*/
/*																		*/
/* 					Copyright: Anthony Dekker, June 1994				*/
/*																		*/
/* This program performs colour quantization of graphics images (SUN	*/
/* raster files).  It uses a Kohonen Neural Network.  It produces		*/
/* better results than existing methods and runs faster, using minimal	*/
/* space (8kB plus the image itself).  The algorithm is described in	*/
/* the paper "Kohonen Neural Networks for Optimal Colour Quantization"	*/
/* to appear in the journal "Network: Computation in Neural Systems".	*/
/* It is a significant improvement of an earlier algorithm.				*/
/*																		*/
/* This program is distributed free for academic use or for evaluation	*/
/* by commercial organizations.											*/
/*																		*/
/* 	Usage:	NeuQuant -n inputfile > outputfile							*/
/* 																		*/
/* where n is a sampling factor for neural learning.					*/
/*																		*/
/* Program performance compared with other methods is as follows:		*/
/*																		*/
/* 	Algorithm				|  Av. CPU Time	|  Quantization Error		*/
/* 	-------------------------------------------------------------		*/
/* 	NeuQuant -3				|  314 			|  5.55						*/
/* 	NeuQuant -10			|  119 			|  5.97						*/
/* 	NeuQuant -30			|  65 			|  6.53						*/
/* 	Oct-Trees 				|  141 			|  8.96						*/
/* 	Median Cut (XV -best) 	|  420 			|  9.28						*/
/* 	Median Cut (XV -slow) 	|  72 			|  12.15					*/
/*																		*/
/* Author's address:	Dept of ISCS, National University of Singapore	*/
/*			Kent Ridge, Singapore 0511									*/
/* Email:	tdekker@iscs.nus.sg											*/
/*----------------------------------------------------------------------*/

#include "xslideshow.h"

#if defined(__STDC__) || defined(__cplusplus)
# define P_(s) s
#else
# define P_(s) ()
#endif
 
extern void myevent P_(());
extern void goodbyekiss P_(());

#define RED		0
#define GREEN	1
#define BLUE	2
#define COLOR	3

#define initrad			32
#define radiusdec		30
static int alphadec;
static int samplefac;
static int lengthcount;
#define defaultsamplefac	1

/* defs for freq and bias */
#define gammashift  	10
#define betashift  	gammashift
#define intbiasshift    16
#define intbias		(1<<intbiasshift)
#define gamma   	(1<<gammashift)
#define beta		(intbias>>betashift)
#define betagamma	(intbias<<(gammashift-betashift))
#define gammaphi	(intbias<<(gammashift-8))

/* defs for rad and alpha */
#define maxrad	 	(initrad+1)
#define radiusbiasshift	6
#define radiusbias	(1<<radiusbiasshift)
#define initradius	((int) (initrad*radiusbias))
#define alphabiasshift	10
#define initalpha	(1<<alphabiasshift)
#define radbiasshift	8
#define radbias		(1<<radbiasshift)
#define alpharadbshift  (alphabiasshift+radbiasshift)
#define alpharadbias    (1<<alpharadbshift)

/* other defs */
#define MAXCOLORNUM	256
#define netbiasshift	4
#define funnyshift	(intbiasshift-netbiasshift)
#define maxnetval	((MAXCOLORNUM<<netbiasshift)-1)
#define ncycles		100
#define jump1		499	/* prime */
#define jump2		491	/* prime */
#define jump3		487	/* any pic whose size was divisible by all */
#define jump4		503	/* four primes would be simply enormous */

typedef int pixel[4];  /* RGBc */

static byte *thepicture;
static pixel network[MAXCOLORNUM];
static int netindex[MAXCOLORNUM];
static int bias [MAXCOLORNUM];
static int freq [MAXCOLORNUM];
static int radpower[MAXCOLORNUM];	/* actually need only go up to maxrad */

/* fixed space overhead 256*4+256+256+256+256 words = 256*8 = 8kB */

/*
*
*/
#if defined(__STDC__) || defined(__cplusplus)
static void initnet(int numcolor)
#else
static void initnet(numcolor)
int numcolor;
#endif
{
int i;
int *p;
	
	for (i=0; i<numcolor; i++) {
		p = network[i];
		p[RED]   = i << netbiasshift;
		p[GREEN] = i << netbiasshift;
		p[BLUE]  = i << netbiasshift;
		freq[i]  = intbias / numcolor;
		bias[i]  = 0;
	}
}

/*
*
*/
#if defined(__STDC__) || defined(__cplusplus)
static void shownet(int numcolor)
#else
static void shownet(numcolor)
int numcolor;
#endif
{
int i,j;

	fprintf(stderr, "\n");
	for (i=0; i<numcolor; i++) {
		fprintf(stderr, "%d \t RGB= %d %d %d COL= %d", i,
			network[i][RED], network[i][GREEN],
			network[i][BLUE], network[i][COLOR]);
		for (j=0; j<numcolor; j++) {
			if (netindex[j] == i)
				fprintf(stderr, "  [%d]", j);
		}
		fprintf(stderr, "\n");
	}
	fprintf(stderr, "\n");
}


/*
*
*/
#if defined(__STDC__) || defined(__cplusplus)
static void inxbuild(int numcolor)
#else
static void inxbuild(numcolor)
int numcolor;
#endif
{
register int i,j,smallpos,smallval;
register int *p,*q;
int start,previous;

	for (previous=0,start=0, i=0; i<numcolor; i++) {
		p = network[i];
		smallpos = i;
		smallval = p[GREEN];	/* index on g */

		/* find smallest in i+1..numcolor-1 */
		for (j=i+1; j<numcolor; j++) {
			q = network[j];
			if (q[GREEN] < smallval) {	/* index on g */
				smallpos = j;
				smallval = q[GREEN]; /* index on g */
			}
		}
		if (i != smallpos) {
			q = network[smallpos];
			j = q[RED];   q[RED]   = p[RED];   p[RED]   = j;
			j = q[GREEN]; q[GREEN] = p[GREEN]; p[GREEN] = j;
			j = q[BLUE];  q[BLUE]  = p[BLUE];  p[BLUE]  = j;
			j = q[COLOR]; q[COLOR] = p[COLOR]; p[COLOR] = j;
		}

		/* smallval entry is now in position i */
		if (smallval != previous) {
			netindex[previous] = (start+i)>>1;
			for (j=previous+1; j<smallval; j++) netindex[j] = i;
			previous = smallval;
			start = i;
		}
	}

	netindex[previous] = (start+numcolor-1) >> 1;
	for (j=previous+1; j<numcolor; j++) netindex[j] = numcolor-1;
}


/*
*
*/
#if defined(__STDC__) || defined(__cplusplus)
static int inxsearch(register int numcolor,int r,int g,int b)
#else
static int inxsearch(numcolor,r,g,b)  /* accepts real RGB values after net is unbiased */
register int numcolor,r,g,b;
#endif
{
register int i,j,best,x,y,bestd;
register int *p;

	bestd = numcolor*3;	/* biggest possible dist is 256*3 */
	best = -1;
	i = netindex[g]; /* index on g */
	j = i-1;

	while ((i<numcolor) || (j>=0)) {
		if (i<numcolor) {
			p = network[i];
			x = p[GREEN] - g;	/* inx key */
			if (x >= bestd) i = numcolor; /* stop iter */
			else {
				i++;
				if (x<0) x = -x;
				y = p[BLUE] - b;
				if (y<0) y = -y;
				x += y;
				if (x<bestd) {
					y = p[RED] - r;	
					if (y<0) y = -y;
					x += y;	/* x holds distance */
					if (x<bestd) {bestd=x; best=p[COLOR];}
				}
			}
		}
		if (j>=0) {
			p = network[j];
			x = g - p[GREEN]; /* inx key - reverse dif */
			if (x >= bestd) j = -1; /* stop iter */
			else {
				j--;
				if (x<0) x = -x;
				y = p[BLUE] - b;
				if (y<0) y = -y;
				x += y;
				if (x<bestd) {
					y = p[RED] - r;	
					if (y<0) y = -y;
					x += y;	/* x holds distance */
					if (x<bestd) {bestd=x; best=p[COLOR];}
				}
			}
		}
	}
	return(best);
}


/*
*
*/
#if defined(__STDC__) || defined(__cplusplus)
static int contest(int numcolor,int r,int g,int b)
#else
static int contest(numcolor,r,g,b)	/* accepts biased RGB values */
int numcolor,r,g,b;
#endif
{
register int i,best,bestbias,x,y,bestd,bestbiasd;
register int *p,*q, *pp;

	bestd     = 0x7fffffff;
	bestbiasd = bestd;
	best      = -1;
	bestbias  = best;

	for (q=bias,p=freq,i=0; i<numcolor; i++) {
		pp = network[i];
		x = pp[BLUE] - b;
		if (x<0) x = -x;
		y = pp[GREEN] - g;
		if (y<0) y = -y;
		x += y;
		y = pp[RED] - r;
		if (y<0) y = -y;
		x += y;	/* x holds distance */
			/* >> netbiasshift not needed if funnyshift used */
		if (x<bestd) {bestd=x; best=i;}
		y = x - ((*q)>>funnyshift);  /* y holds biasd */
		if (y<bestbiasd) {bestbiasd=y; bestbias=i;}
		y = (*p >> betashift);	     /* y holds beta*freq */
		*p -= y;
		*q += (y<<gammashift);
		p++;
		q++;
	}
	freq[best] += beta;
	bias[best] -= betagamma;
	return(bestbias);
}


/*
*
*/
#if defined(__STDC__) || defined(__cplusplus)
static void alterneigh(int numcolor,int rad,int i,int r,int g,int b)
#else
static void alterneigh(numcolor,rad,i,r,g,b)	/* accepts biased RGB values */
int numcolor,rad,i;
int r,g,b;
#endif
{
register int j,k,lo,hi,a;
register int *p, *q;

	lo = i-rad;
	if(lo< -1) lo= -1;
	hi = i+rad;
	if(hi>numcolor) hi=numcolor;
	j  = i+1;
	k  = i-1;
	q  = radpower;
	while ((j<hi) || (k>lo)) {
		a = (*(++q));
		if (j<hi) {
			p = network[j];
			*p -= (a*(*p - r)) / alpharadbias; p++;
			*p -= (a*(*p - g)) / alpharadbias; p++;
			*p -= (a*(*p - b)) / alpharadbias; j++;
		}
		if (k>lo) {
			p = network[k];
			*p -= (a*(*p - r)) / alpharadbias; p++;
			*p -= (a*(*p - g)) / alpharadbias; p++;
			*p -= (a*(*p - b)) / alpharadbias; k--;
		}
	}
}


/*
*
*/
#if defined(__STDC__) || defined(__cplusplus)
static void altersingle(int alpha,int j,int r,int g,int b)
#else
static void altersingle(alpha,j,r,g,b)	/* accepts biased RGB values */
int alpha,j,r,g,b;
#endif
{
register int *q;

	q = network[j];		/* alter hit neuron */
	*q -= (alpha*(*q - r)) / initalpha; q++;
	*q -= (alpha*(*q - g)) / initalpha; q++;
	*q -= (alpha*(*q - b)) / initalpha;
}


/*
*
*/
#if defined(__STDC__) || defined(__cplusplus)
static void learn(int numcolor)
#else
static void learn(numcolor)
int numcolor;
#endif
{
register int i,j,r,g,b;
int radius,rad,alpha,step,delta,upto;
register byte *p;
byte *lim;

	upto = lengthcount/(3*samplefac);
	delta = upto/ncycles;
	if(delta==0) delta=1;
	lim = thepicture + lengthcount;
	p = thepicture;
	alpha = initalpha;
	radius = initradius;
	rad = radius >> radiusbiasshift;
	if(rad<=1) rad=0;

	for (i=0; i<rad; i++) 
		radpower[i] = alpha*(((rad*rad - i*i)*radbias)/(rad*rad));

	     if ((lengthcount%jump1) != 0) step = 3*jump1;
	else if ((lengthcount%jump2) != 0) step = 3*jump2;
	else if ((lengthcount%jump3) != 0) step = 3*jump3;
	else                               step = 3*jump4;

	if(app_data.verbose)
		fprintf(stderr,"xslideshow: nquant() upto = %d\n",upto);

	i = 0;
	while (i < upto) {
		r = p[RED]   << netbiasshift;
		g = p[GREEN] << netbiasshift;
		b = p[BLUE]  << netbiasshift;

		j = contest(numcolor,r,g,b);
		altersingle(alpha,j,r,g,b);
		if(rad) alterneigh(numcolor,rad,j,r,g,b); /* alter neighbours */

		p += step; if (p >= lim) p -= lengthcount;
	
		i++; if (i%delta == 0) {	
			alpha  -= alpha / alphadec;
			radius -= radius / radiusdec;
			rad     = radius >> radiusbiasshift;
			if (rad <= 1) rad = 0;
			for (j=0; j<rad; j++) 
				radpower[j] = alpha*(((rad*rad - j*j)*radbias)/(rad*rad));
		}
		myevent();
	}
	if(app_data.verbose)
		fprintf(stderr,"xslideshow: nquant() finished 1D learning: final alpha=%f !\n",((float)alpha)/initalpha);
}
	

/*
*
*/
#if defined(__STDC__) || defined(__cplusplus)
static void unbiasnet(int numcolor)
#else
static void unbiasnet(numcolor)
int numcolor;
#endif
{
int i,j;

	for (i=0; i<numcolor; i++) {
		for (j=0; j<3; j++) {
			network[i][j] >>= netbiasshift;
		}
		network[i][COLOR] = i; /* record colour no */
	}
}


/*
*
*/
#if defined(__STDC__) || defined(__cplusplus)
void nquant(int numcolor, int smpl)
#else
void nquant(numcolor, smpl)
int numcolor, smpl;
#endif
{
int i,j,k,r,g,b;
byte *p,*q;

	samplefac = smpl;
	if (samplefac < 0) samplefac = defaultsamplefac;	
	alphadec = 30 + ((samplefac-1)/3);
	if(app_data.verbose)
		fprintf(stderr,"xslideshow: nquant() sampling factor= %d, alphadec= %d\n",samplefac,alphadec);

	lengthcount = gim.subImageList->width * gim.subImageList->height * 3;
	if (lengthcount < 3*jump4) {
		fprintf(stderr,"xslideshow: nquant() pic too small: must have at least 600 pixels.\n");
		goodbyekiss();
	}

	if(gim.subImageList->bits_per_pixel == 8) {
		if((thepicture = (byte *)XtMalloc(sizeof(byte) * lengthcount)) == (byte *)NULL){
			fprintf(stderr,"xslideshow: nquant() memory allocation error.\n");
			goodbyekiss();
		}
		p = thepicture;
		for(q=gim.subImageList->image,j = 0;j<gim.subImageList->height;j++){
			for(i=0;i<gim.subImageList->width;i++) {
				*p++ = (byte)gim.subImageList->red[*q];
				*p++ = (byte)gim.subImageList->green[*q];
				*p++ = (byte)gim.subImageList->blue[*q++];
			}
			myevent();
		}
		XtFree((char *)gim.subImageList->image);
	}
	else {
		thepicture = gim.subImageList->image;
	}

	if(app_data.verbose)
		fprintf(stderr,"xslideshow: nquant() initnet()\n");
	initnet(numcolor);

	if(app_data.verbose)
		fprintf(stderr,"xslideshow: nquant() learn()\n");
	learn(numcolor);

	if(app_data.verbose)
		fprintf(stderr,"xslideshow: nquant() unbiasnet()\n");
	unbiasnet(numcolor);

	if(app_data.verbose)
		fprintf(stderr,"xslideshow: nquant() inxbuild()\n");
	inxbuild(numcolor);

#if defined (_DEBUG)
	if(app_data.verbose && app_data.debug)
		shownet(numcolor);
#endif

	for(i=0; i<numcolor; i++){
		gim.subImageList->red[network[i][COLOR]] = network[i][RED];
		gim.subImageList->green[network[i][COLOR]] = network[i][GREEN];
		gim.subImageList->blue[network[i][COLOR]] = network[i][BLUE];
	}
	gim.subImageList->mapsize = numcolor;

	if(app_data.verbose)
		fprintf(stderr,"xslideshow: nquant() inxsearch()\n");
	p = thepicture;
	for (k=0, i=0; i<gim.subImageList->height; i++) {
		for (j=0; j<gim.subImageList->width; j++) {
			r = *p++;
			g = *p++;
			b = *p++;
			thepicture[k++] = inxsearch(numcolor,r,g,b);
		}
		myevent();
	}

	gim.subImageList->image = (byte *)XtRealloc((char *)thepicture, sizeof(byte) * gim.subImageList->width * gim.subImageList->height);
	gim.subImageList->bits_per_pixel = 8;

	if(app_data.verbose)
		fprintf(stderr, "xslideshow: nquant() done\n");
}


