/*
Copyright (c) 1989-1990, University of Illinois board of trustees.  All
rights reserved.  Written by Michael Maciukenas at the Center for Prokaryote
Genome Analysis.  Design and implementation guidance by Steven Smith, Carl
Woese.
*/
/* File picker by Mike Maciukenas 
** Allows the user to search up and down the directory tree, and choose a 
**  file.  
** "Open" descends down into a directory, or chooses a file (depending **  on what is selected).  The user may also press return after choosing
**  a file or directory, to do the same thing.
** "Up Dir" ascends to the parent directory.  
** The user may also type a directory into the "Directory:" field.  When the
**  user presses return (or tab, or newline), the contents of the new directory
**  will be shown.
*/
#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>
#include <errno.h>

#include <xview/xview.h>
#include <xview/canvas.h>
#include <xview/scrollbar.h>
#include <xview/win_input.h>
#include "list.h"

#include "generic.h"

#include "canvas.h"



typedef enum {
	load_reg,
	load_link,
	load_exec,
	load_sock,
	load_dir
	} load_file_type;

typedef struct {
	ltgenericd g;
	tframe f;
	tlist l;
	tptext dirtext;
	tpanel p;
	char dirname[1024];
	tpbutton openb, upb, cancelb;
	tpmsg bmsg1, bmsg2;
	int (*fun)();
	int lastchosen;
	unsigned long lasttime;
	} *ltload, ltloadd;

tload_hide(s)
ltload s;
{
	tframe_pull_pin(s->f);
	tframe_hide(s->f);
}

tload_show(s)
ltload s;
{
	tload_make_filelist(s);
	tptext_set_current(s->dirtext);
	tframe_push_pin(s->f);
	tframe_show(s->f);
}

tload_push_pin(s)
ltload s;
{
	tframe_push_pin(s->f);
}

tload_pull_pin(s)
ltload s;
{
	tframe_pull_pin(s->f);
}

int tload_move(s, win, x, y)
ltload s;
Xv_Window win;
int x,y;
{
    if(s==NULL)
        return(0);
    if(titem_type(s)!=lt_save)
        return(0);

    tframe_move(s->f, win, x, y);
	return(1);
}

tload_checkdir(s)
char *s;
{
	DIR *dirp;
	dirp=opendir(s);
	if(dirp==NULL)
		return(0);
	else
	{
		closedir(dirp);
		return(1);
	}
}

load_file_type tload_readln(dirp, buf, dir)
DIR *dirp;
char *buf;
char *dir;
{
	struct dirent *entry;
	struct stat sbuf;
	int i;
	load_file_type result;

	result=load_reg;
	i=strlen(dir);
	entry=readdir(dirp);
	if(entry==NULL)
		strcpy(buf, "");
	else
	{
		strcpy(buf, entry->d_name);
		if(dir[i-1]!='/')
			strcat(dir, "/");
		strcat(dir, buf);
		if(stat(dir, &sbuf)!=0)
			strcpy(buf, "");
		else
		{
			if(S_ISDIR(sbuf.st_mode))
			{
				strcat(buf, "/");
				result=load_dir;
			}
			else if(S_ISLNK(sbuf.st_mode))
			{
				strcat(buf, "@");
				result=load_link;
			}
			else if(S_ISSOCK(sbuf.st_mode))
			{
				strcat(buf, "=");
				result=load_sock;
			}
			else if(S_ISREG(sbuf.st_mode))
			{
				if(sbuf.st_mode&(S_IXUSR|S_IXGRP|S_IXOTH))
				{
					strcat(buf, "*");
					result=load_exec;
				}
				else
					result=load_reg;
			}
		}
		dir[i]='\0';
	}
	return(result);
}

tload_make_filelist(s)
ltload s;
{
	char tempbuf[1024];
	char tmpname[1024];
	DIR *dirp;
	load_file_type type;


	strcpy(tmpname, s->dirname);
	dirp=opendir(tmpname);
	if(dirp==NULL)
	{
		fprintf(stderr, "tload_make_filelist got bad directory name\n:(\n");
		return;
	}
	else
	{
		tframe_set_busy(s->f);
		tlist_remove_all(s->l);
		for(type=tload_readln(dirp, tempbuf, s->dirname);tempbuf[0]!='\0';
				type=tload_readln(dirp, tempbuf, s->dirname))
		{
			if((strcmp(tempbuf, "./")!=0) &&
					(strcmp(tempbuf, "../")!=0))
				tlist_add(s->l, tempbuf, -2, 0, type);
		}
		closedir(dirp);
		tframe_set_notbusy(s->f);
		tlist_refresh(s->l);
	}
}

int tload_choose_open(b, e)
tpbutton b;
tevent e;
{
	ltload s;
	int selected;
	char namebuf[1024], thestr[1024];
	int x, y;
	load_file_type type;

	tevent_location(e, &x, &y);
	s=(ltload)titem_get_intdata(b);

	selected=tlist_selected(s->l);
	if(selected>=1)
	{
		strcpy(namebuf, tlist_nth_name(s->l, selected));
		type=(load_file_type)tlist_nth_data(s->l, selected);
		if(type==load_dir)
		{
			/* then it's a directory */
			strcpy(thestr, s->dirname);
			if(thestr[strlen(thestr)-1]!='/')
				strcat(thestr, "/");
			strcat(thestr, namebuf);
			thestr[strlen(thestr)-1]='\0';
			if(tload_checkdir(thestr))
			{
				strcpy(s->dirname, thestr);
				tload_make_filelist(s);
				tptext_set_val(s->dirtext, s->dirname);
				return(1);
			}
			else
			{
				if(errno==EACCES)
				{
					tnotice_no_choice_2line(s->f, x, y,
						"You don't have access to that directory.",
						"Please choose another.");
				}
				else
				{
					tnotice_no_choice_2line(s->f, x, y,
						"Invalid directory specified.",
						"Please choose another.");
				}
			}
		}
		else
		{
			/* it's a file */
			strcpy(thestr, tptext_get_val(s->dirtext));
			if(tload_checkdir(thestr))
			{
				/* valid directory */
				strcpy(thestr, s->dirname);
				switch(type)
				{
					case load_link:
					case load_sock:
					case load_exec:
						/* strip off id character */
						namebuf[strlen(namebuf)-1]='\0';
						break;
					case load_reg:
					default:
						break;
				}
				if(thestr[strlen(thestr)-1]!='/')
					strcat(thestr, "/");
				strcat(thestr, namebuf);
				if((*s->fun)(thestr, e))
					tload_hide(s);
				return(1);
			}
			else
			{
				if(errno==EACCES)
				{
					tnotice_no_choice_2line(s->f,
						x, y,
						"You don't have access to that directory.",
						"Please choose another.");
				}
				else
				{
					tnotice_no_choice_2line(s->f,
						x, y,
						"Invalid directory specified.",
						"Please choose another.");
				}
			}
		}
	}
}

int tload_choose_up(b, e)
tpbutton b;
tevent e;
{
	ltload s;
	int i;

	s=(ltload)titem_get_intdata(b);
	
	if(strlen(s->dirname)==1)
		return(1);
	if(s->dirname[strlen(s->dirname)-1]=='/')
		i=strlen(s->dirname)-2;
	else
		i=strlen(s->dirname)-1;
	for(;s->dirname[i]!='/' && i>0;i--);
	if(i==0) i++;
	s->dirname[i]='\0';
	tload_make_filelist(s);
	tptext_set_val(s->dirtext, s->dirname);
	return(0);
}

int tload_choose_cancel(b, e)
tpbutton b;
tevent e;
{
	ltload s;
	int i;

	s=(ltload)titem_get_intdata(b);
	
	if((*s->fun)(NULL, e))
		tload_hide(s);
	return(0);
}

int tload_choose_dirname(b, e)
tptext b;
tevent e;
{
	ltload s;
	char *keys;
	int x, y;

	tevent_location(e, &x, &y);
	s=(ltload)titem_get_intdata(b);

	if(tevent_type(e)==te_keydown)
	{
		keys=tevent_string(e);
		if(keys[0]=='\n' || keys[0]=='\r')
		{
			if(tload_checkdir(tptext_get_val(s->dirtext)))
			{
				strcpy(s->dirname, tptext_get_val(s->dirtext));
				tload_make_filelist(s);
			}
			else
			{
				if(errno==EACCES)
				{
					tnotice_no_choice_2line(s->f,
						x, y,
						"You don't have access to that directory.",
						"Please choose another.");
				}
				else
				{
					tnotice_no_choice_2line(s->f,
						x, y,
						"Invalid directory specified.",
						"Please choose another.");
				}
			}
		}
		return(tptext_insert);
	}
}

int handle_load_frame(f, e)
tframe f;
tevent e;
{
	int w, h;
	ltload s;

	s=(ltload)titem_get_intdata(f);

	if(tevent_type(e)==te_resize)
	{
		tevent_size(e, &w, &h);
		tpanel_set_width_to_frame(s->p, s->f);
		tlist_resize(s->l, 0, 0, w, h-tpanel_height(s->p), NULL, s->p);
	}
	else
		return(0);
}

int tload_choose_filelist(l, i, e)
tlist l;
int i;
tevent e;
{
	ltload s;
	char *keys;
	static unsigned long thistime;

	s=(ltload)titem_get_intdata(l);

	switch(tevent_type(e))
	{
		case te_mouseup:
			if(s->lastchosen==i && i!=-1)
			{
				/* may be double clicking */
				thistime=tevent_time(e);
				if(thistime-s->lasttime<500)
				{
					/* double clicking */
					tload_choose_open(s->openb, e);
					s->lastchosen=-1;
					return(0);
				}
				else
				{
					s->lastchosen=i;
					s->lasttime=thistime;
				}
			}
			else
			{
				s->lastchosen=i;
				s->lasttime=tevent_time(e);
			}
			break;
		case te_keydown:
			keys=tevent_string(e);
			if(keys[0]=='\n' || keys[0]=='\r')
				tload_choose_open(s->openb, e);
			s->lastchosen=-1;
			break;
		default:
			break;
	}
	return(1);
}

tload tload_new(parent, x, y, listfont, fun, args, title, dirname)
/* create a frame for choosing where to load a file */
tframe parent;
int x, y;
tfont listfont;
int (*fun)();
char *title;
char *dirname;
targs args;
{
	ltload tmp;

	if(parent==NULL || listfont==NULL)
		return(NULL);
	if(titem_type(parent)!=lt_frame || titem_type(listfont)!=lt_font)
		return(NULL);

	tmp=(ltload)titem_new(parent, lt_load, sizeof(ltloadd));
	if(tmp==NULL)
	{
		printf("Not enough memory for load frame\n");
		return(NULL);
	}

	tmp->f=tframe_new(parent, 50,50,500,500,0,1,0,title,args);
	if(tmp->f==NULL)
	{
		titem_free(tmp);
		return(NULL);
	}
	titem_set_intdata(tmp->f, tmp);
	tframe_set_event_procedure(tmp->f, handle_load_frame);

	tmp->p=tframe_dialog_panel(tmp->f);

	tmp->openb=tpbutton_new(tmp->p, 10, 10, 0, 0, NULL, NULL, "Open");
	tpbutton_set_event_procedure(tmp->openb, tload_choose_open);
	titem_set_intdata(tmp->openb, tmp);

	tmp->upb=tpbutton_new(tmp->p, 10, 10, 0, 0, tmp->openb, NULL, "Up Dir");
	tpbutton_set_event_procedure(tmp->upb, tload_choose_up);
	titem_set_intdata(tmp->upb, tmp);

	tmp->cancelb=tpbutton_new(tmp->p, 10, 10, 0, 0, tmp->upb, NULL,
		"Cancel");
	tpbutton_set_event_procedure(tmp->cancelb, tload_choose_cancel);
	titem_set_intdata(tmp->cancelb, tmp);

	tmp->bmsg1=(tpmsg)tpmsg_new(tmp->p,
		10, 10, 0, 0, NULL, tmp->openb, "    ");

	tmp->dirtext=tptext_new(tmp->p, 10, 60, 0, 0, NULL, tmp->bmsg1,
		"Directory:");
	tptext_set_displayed_characters(tmp->dirtext, 30);
	tptext_set_event_procedure(tmp->dirtext, tload_choose_dirname);
	titem_set_intdata(tmp->dirtext, tmp);
	tptext_set_val(tmp->dirtext, dirname);

	tmp->bmsg2=(tpmsg)tpmsg_new(tmp->p,
		10, 10, 0, 0, NULL, tmp->dirtext, "    ");

	tpanel_fit(tmp->p);

	tmp->fun=fun;
	strcpy(tmp->dirname, dirname);
	tmp->lasttime=0;
	tmp->lastchosen=-1;

	tmp->l=(tlist)tlist_new(tmp->f,
		0, 0, tpanel_width(tmp->p), 15*tfont_height(listfont),
		NULL, tmp->p, listfont, 0, args);
	titem_set_intdata(tmp->l, tmp);
	tlist_set_event_procedure(tmp->l, tload_choose_filelist);

	tload_make_filelist(tmp);

	tframe_fit(tmp->f);
	tframe_hide(tmp->f);
	tframe_pull_pin(tmp->f);

	return((tload)tmp);
}

void *tload_xview(s)
ltload s;
{
	if(s==NULL)
		return(NULL);
	if(titem_type(s)!=lt_load)
		return(NULL);
	
	return((void *)tframe_xview(s->f));
}

int tload_get_panel(s, p, b)
/* returns load file panel and bottom object */
ltload s;
tpanel *p;
titem *b;
{
	if(s==NULL)
		return(NULL);
	if(titem_type(s)!=lt_load)
		return(NULL);
	
	*p=s->p;
	*b=s->bmsg2;
	return(1);
}

int tload_refit_panel(s)
/* refits panel and moves list */
ltload s;
{
	if(s==NULL)
		return(NULL);
	if(titem_type(s)!=lt_load)
		return(NULL);
	
	tpanel_fit(s->p);
	tlist_move(s->l, 0, 0, NULL, s->p);
	tframe_fit(s->f);
	tpanel_set_width_to_frame(s->p, s->f);
	return(1);
}

int tload_set_busy(s)
/* refits panel and moves canvas */
ltload s;
{
    if(s==NULL)
        return(0);
    if(titem_type(s)!=lt_load)
        return(0);

    tframe_set_busy(s->f);
    return(1);
}

int tload_set_notbusy(s)
/* refits panel and moves canvas */
ltload s;
{
    if(s==NULL)
        return(0);
    if(titem_type(s)!=lt_load)
        return(0);

    tframe_set_notbusy(s->f);
    return(1);
}
