/*
 * Scene.java
 * the VRML scene
 *
 * created: mpichler, 19960923
 *
 * changed: kwagen, 19970910
 * changed: apesen, 19970910
 * changed: mpichler, 19970916
 *
 * $Id: Scene.java,v 1.41 1997/09/25 15:08:58 apesen Exp $
 */


package iicm.vrml.vrwave;

import iicm.vrml.pw.*;
import iicm.vrml.vrwave.pwdat.*;
import iicm.ge3d.GE3D;
import iicm.utils3d.Camera;
import iicm.utils3d.Ray;
import iicm.widgets.ApplyColour;
import iicm.widgets.DLGColourChoose;
import iicm.widgets.ImageButton;
import iicm.widgets.Progress;
import iicm.widgets.RadioMenuItem;
import iicm.widgets.StatusBar;

import java.awt.*;
import java.applet.*;
import java.io.*;
import java.net.*;
import java.util.*;


/**
 * Scene - the VRML scene
 * Copyright (c) 1996,97 IICM
 *
 * @author Michael Pichler, Karin Roschker, Andreas Pesendorfer
 * @version 0.9, latest change: 17 Sep 97
 */


public class Scene
  implements ApplyColour  // DLGColourChoose callback
{
  protected GroupNode root;  // the root node
  private Builder builder = new Builder ();
  private Drawer drawer = new Drawer ();
  private Picker picker = new Picker ();
  private boolean firstdraw_;
  private Applet applet_;
  protected Frame frame_;
  private MenuItem[] menuitems_;
  private SceneCanvas canvas_ = null;
  private Vector sensors_ = null;  // list of sensors which should be checked before each redraw
  private Hashtable nodenames_ = null;

  // timestamp of current frame (seconds since 1970, double precision)
  private double timestamp_ = 0.0;
  /** get frame timestamp */
  public double getTimestamp ()  { return timestamp_; }

  /** framerate */
  public void toggleShowframerate ()
  {
    showframerate_ = !showframerate_;
    clearStatusMessage ();
  }
  private boolean showframerate_ = false;
  // frames per second; will be 0 until framerate is displayed
  public float framespersecond_ = 0.0f;

  // drawing mode, interaction
  private int drawmode_ = GE3D.ge3d_smooth_shading;
  private int intdrawmode_ = GE3D.ge3d_wireframe;
  private boolean interact_ = false;
  /** set interaction flag */
  public void setInteraction (boolean flag)  { interact_ = flag; }
  /** interaction flag relevant? The case when interactive drawmode set differently from normal one */
  public boolean interactionRelevant ()
  { return (intdrawmode_ >= 0 && intdrawmode_ != drawmode_); }
  /** set (interactive) drawing mode */
  public void drawingMode (int mode, boolean interactive)
  { // caller responsible for updating menu state
    // System.out.println ("setting drawing mode to " + mode);
    if (interactive)
      intdrawmode_ = mode;
    else
      drawmode_ = mode;
  }
  /** get drawing mode (according to interaction flag) */
  public int curDrawingMode ()
  { return ((interact_ && intdrawmode_ >= 0) ? intdrawmode_ : drawmode_); }
  /** normal drawing mode */
  public int normalDrawingMode ()
  { return drawmode_; }
  /** interactive drawing mode */
  public int interDrawingMode ()
  { return intdrawmode_; }

  // viewing camera (includes Viewpoint management)
  private ViewCamera viewcam_ = new ViewCamera ();
  /** reset camera */
  protected void resetView ()  { viewcam_.reset (); }
  /** level view */
  protected void levelView ()  { viewcam_.levelize (); }
  /** untilt view */
  protected void untiltView ()  { viewcam_.untilt (); }
  /** set camera via GE3D */
  protected void setCamera ()  { viewcam_.setCamera (winaspect_); }
  /** get camera */
  public Camera getCamera ()  { return viewcam_; }
  /** get near clipping plane */
  float getNearClip ()  { return viewcam_.getNearClip (); }
  /** get far clipping plane */
  float getFarClip ()  { return viewcam_.getFarClip (); }

  /** default Viewpoint (first one encountered in file) */
  private Viewpoint defview_ = null;
  /** Viewpoint encountered in build */
  void hasViewpoint (Viewpoint vp)
  {
    if (defview_ == null)  // only of interest in initial build
      defview_ = vp;
  }

  /** set frame (for dialogs) */
  protected void setFrame (Frame frame)  { frame_ = frame; }

  /** set menuitems (created by frame) */
  protected void setMenus (MenuItem[] items)  { menuitems_ = items; }

  /** get menuitems (to update states) */
  MenuItem[] getMenus ()  { return menuitems_; }

  private ImageButton[] navbuttons_;
  private ImageButton behavbutton_, interactbutton_;

  /** set icon buttons (navigation et al., created by Iconbar, used to update states) */
  void setIconbuttons (ImageButton[] navbuttons, ImageButton behavbutton, ImageButton interactbutton)
  {
    navbuttons_ = navbuttons;
    behavbutton_ = behavbutton;
    interactbutton_ = interactbutton;
  }

  /** set active canvas (for repaint) */
  void setCanvas (SceneCanvas canvas)  { canvas_ = canvas; }

  /** redraw request */
  public void redraw ()
  { // repaint active canvas
    if (canvas_ != null)
      canvas_.repaint ();
  }

  // statusbar, progress indicator
  private StatusBar statusbar_;
  private Progress progress_;
  /** set statusbar and progress indicator */
  void setProgressbar (StatusBar statusbar, Progress progress)
  {
    statusbar_ = statusbar;
    progress_ = progress;
  }
  /** set status message */
  public void statusMessage (String label, int fnum)
  {
    if (statusbar_ != null)
      statusbar_.setLabel (0, fnum, label);
  }
  /** empty/default status message */
  public void clearStatusMessage ()
  {
    statusMessage ("VRwave", 0);
  }
  /** set working message */
  public void workingMessage (String label)
  {
    if (statusbar_ != null)
      statusbar_.setLabel (1, 0, label);
  }
  /** switch to working state */
  public void workingState (boolean flag)
  {
    if (statusbar_ != null)
    { // see SceneFrame constructor for card order
      if (flag)
        statusbar_.last ();  // working
      else
        statusbar_.first ();  // ready
    }
  }
  /** set progress (0.0 to 1.0) */
  public void setProgress (float level)
  {
    if (progress_ != null)
      progress_.setValue (level);
  }

  /* set cursor immediately
  protected void setFrameCursor (int type)
  {
    if (canvas_ != null && frame_ != null)
      canvas_.setFrameCursor (frame_, type);
  } */

  // window aspect
  private float winaspect_ = 1.33f;
  /** set window aspect */
  public void setWinAspect (float val)  { winaspect_ = val; }
  /** get window aspect */
  public float getWinAspect ()  { return winaspect_; }

  // scene center
  private float[] center_ = { 0.0f, 0.0f, 0.0f };
  /** set center of scene */
  public void setCenter (float[] c)  { center_ = c; }
  /** get center of scene */
  public float[] getCenter ()  { return center_; }

  // navigation mode
  protected int movemode_ = FLIP;
  // assert: same order as M_Flip etc.
  public final static int FLIP = 0;
  public final static int WALK = 1;
  public final static int FLY = 2;
  public final static int FLYTO = 3;
  public final static int HEADSUP = 4;
  public final static int NUMNAVMODES = 5;

  // widgets to be updated on changes (bitmask)
  public final static int UPDATE_MENU = 0x1;
  public final static int UPDATE_ICON = 0x2;
  public final static int UPDATE_ALL = 0x3;

  /** set navigation mode */
  public void setNavigationMode (int mode, int whatupdates)
  {
    movemode_ = mode;
    statusMessage (Translator.getLabel (Translator.L_HINTFLIP + mode), 0);  // assert: continuous labels
    if ((whatupdates & UPDATE_MENU) != 0 && menuitems_ != null)
      ((RadioMenuItem) menuitems_ [MenuDef.M_Flip + mode]).setState (true);
    if ((whatupdates & UPDATE_ICON) != 0 && navbuttons_ != null)
      navbuttons_ [mode].setState (true);
  }

  // picking purpose (set flags in Hitpoint)
  // public final static int PICK_POI = 0;

  // lighting
  private boolean haslight_ = false;
  /** scene has light source; no need for headlight */
  public void setHasLight ()  { haslight_ = true; }
  /** is there a light source? */
  public boolean hasLightSource ()  { return haslight_; }
  private boolean headlight_ = false;
  /** toggle headlight */
  public void toggleHeadlight ()  { headlight_ = !headlight_; }
  /** get headlight flag */
  public boolean getHeadlight ()  { return headlight_; };

  // colors
  public final static int COLBACKGROUND = 0;
  public final static int COLHEADLIGHT = 1;
  public final static int COLAMBIENT = 2;
  public final static int NUMCOLORS = 3;
  private int[] color_ = {  // 0xRRGGBB
    0xcccccc,  // background
    0xffffff,  // headlight
    0x333333   // ambient (OpenGL default: 0.2)
  };
  /** get a color (RGB values) */
  public int getColor (int i)  { return color_[i]; }
  /** change a color (RGB values) */
  public void setColor (int i, int rgb)  { color_[i] = rgb; }

  // dialogs
  private DialogFileOpen dlgfileopen_ = null;
  private DialogOpenLocation dlgopenlocation_ = null;
  String lastfilename_ = null;
  URL lasturl_ = null;
  URL baseurl_ = null;
  private DialogAbout dlgabout_ = null;
  private DLGColourChoose dlgcolchooser_ = null;
  private DialogSettings dlgsettings_ = null;

  // on/auto/off options
  public final static int TRISTATE_OFF = 0;  // assert: off is 0
  public final static int TRISTATE_ON = 1;
  public final static int TRISTATE_AUTO = 2;

  // backfaceculling
  private int backfaceculling_ = TRISTATE_AUTO;
  public int backfaceCulling ()
  { return backfaceculling_; }
  public void setBackfaceCulling (int val)
  { backfaceculling_ = val; }

  // line antialiasing
  private int supportsantialiasing_ = 0;  // not known before GE3D is loaded
  private int lineantialiasing_ = 0;
  public int getLineAntialiasing ()
  { return lineantialiasing_; }
  public void toggleLineAntialiasing ()
  { // no polygon anti-aliasing yet (requires ordered polygons for good results)
    if (supportsantialiasing_ != 0)
    {
      lineantialiasing_ = GE3D.AA_LINES - lineantialiasing_;
      GE3D.antialiasing (lineantialiasing_);
    }
  }

  // texture mipmapping
  private int texturemipmapping_ = 0;
  public int getTextureMipmapping ()
  { return texturemipmapping_; }
  public void setTextureMipmapping (int quality)
  { texturemipmapping_ = quality; }

  // lighting
  private int lighting_ = TRISTATE_AUTO;
  public int getLighting ()
  { return lighting_; }
  public void setLighting (int val)
  { lighting_ = val; }

  // texture lighting
  private boolean texlighting_ = true;
  public boolean getTexLighting ()
  { return texlighting_; }
  public void setTexLighting (boolean val)
  { texlighting_ = val; }

  // materials on/off
  private boolean materials_ = true;
  public boolean materials ()
  { return materials_; }
  public void setMaterials (boolean val)
  { materials_ = val; }

  // texture transparency
  private boolean texturetranspareny_ = true;
  public boolean getTextureTransparency ()
  { return texturetranspareny_; }
  public void setTextureTransparency (boolean val)
  { texturetranspareny_ = val; }

  // transparency method
  private int transpmethod_ = GE3D.TRANSP_STIPPLE;
  public int getTranspMethod ()
  { return transpmethod_; }
  public void setTranspMethod (int val)
  {
    if (val != transpmethod_)
    {
      transpmethod_ = val;
      GE3D.hint (GE3D.HINT_TRANSPARENCY, val);
      redraw ();
    }
  }

  // settings
  private int quadslices_ = 16;
  public void setQuadslices (int val)
  {
    if (val != quadslices_)
    {
      quadslices_ = val;
      redraw ();
    }
  }
  public int getQuadslices ()
  { return quadslices_; }

  // behavior (sensors, events and scripts)
  private boolean behavior_ = true;
  public void toggleBehavior (int whatupdates)
  {
    behavior_ = !behavior_;
    if ((whatupdates & UPDATE_MENU) != 0 && menuitems_ != null)
      ((CheckboxMenuItem) menuitems_[MenuDef.M_Behaviour]).setState (behavior_);
    if ((whatupdates & UPDATE_ICON) != 0 && behavbutton_ != null)
      behavbutton_.setState (behavior_);
  }
  /** check if behavior is enabled */
  public boolean getBehavior ()  { return behavior_; }
  /** check if behavior is running (continuous repaints driven by TimeSensor) */
  public boolean behavior ()  { return (behavior_ && (sensors_ != null)); }

  // interaction: mouse handles sensors/anchors instead of navigation
  private boolean interaction_ = false;
  public void toggleInteraction (int whatupdates)
  {
    interaction_ = !interaction_;
    if ((whatupdates & UPDATE_MENU) != 0 && menuitems_ != null)
      ((CheckboxMenuItem) menuitems_[MenuDef.M_Interaction]).setState (interaction_);
    if ((whatupdates & UPDATE_ICON) != 0 && interactbutton_ != null)
      interactbutton_.setState (interaction_);
  }
  /** check if interaction is enabled (otherwise navigation) */
  public boolean getInteraction ()  { return interaction_; }


  /** add a Sensornode to a list of sensors which are to check before a redraw */
  public void addSensor (Node sens)
  {
    if (sensors_ == null)
      sensors_ = new Vector ();
    sensors_.addElement (sens);
  }

  /**
   * get current time in <em>seconds</em> (double precision) since Jan 1 1970 00:00:00 GMT
   */
  public static double currentTime ()
  {
    return (((double) System.currentTimeMillis ()) / 1000.0);
  }


  /**
   * constructor
   * @arg applet applet (if run as applet) or null otherwise
   */

  public Scene (Applet applet)
  {
    applet_ = applet;
  }


  /**
   * clear current scene data (on replacing the scene)
   */

  private synchronized void clear ()
  {
    root = null;  // destroy old scene
    firstdraw_ = true;  // need build on next draw
    lastfilename_ = null;
    lasturl_ = null;
    baseurl_ = null;
    nodenames_ = null;

    haslight_ = false;

    sensors_ = null;  // clear list of sensors which are to check

    viewcam_.clear ();
    viewcam_.reset ();
    if (canvas_ != null)
      canvas_.reset ();

    clearStatusMessage ();  // clear old info (e.g. anchor descriptions)

    drawer.clearThreads ();
    builder.clearThreads ();
  } // clear


  // use with care
  public void clearScene ()
  {
    clear ();
  }


  /**
   * get a new, empty scene
   */

  protected void newScene ()  { clear (); }


  /**
   * readScene - read VRML stream from file
   * @param filename name of file to be read (non-null;
   * use InputStream variant to read from stdin)
   * @param baseurl base URL (set to file:filename if null)
   */

  public synchronized void readScene (String filename, String baseurl)
  {
    clear ();  // clear old data, reset flags
    lastfilename_ = filename;

    if (baseurl == null)
      baseurl = "file:" + (new File (filename)).getAbsolutePath ();

    try
    { baseurl_ = new URL (baseurl);
    } catch (MalformedURLException e) { }
    // else baseurl_ set to null in clear

    try
    {
      parseInput (new VRMLparser (Decompression.filter (filename)), filename);  // VRMLparser buffers stream
      // parseInput (new VRMLparser (new DecompressionStream (filename)), filename);  // equivalent
    }
    catch (IOException e)
    {
      System.out.println ("error on reading " + filename);
      // e.printStackTrace ();
      System.out.println (e.getMessage ());  // just prints file name
    }

  } // readScene (filename, baseurl)


  /**
   * readScene - read VRML stream from URL
   */

  public synchronized void readScene (URL url)
  {
    clear ();
    lasturl_ = url;
    baseurl_ = url;    

    String location = url.toString ();

    // TODO: handle compressed streams

    try
    { parseInput (new VRMLparser (url.openStream ()), location);
    }
    catch (IOException e)
    {
      System.out.println ("error on reading " + location);
      // e.printStackTrace ();
      System.out.println (e.getMessage ());  // just prints file name
    }
  } // readScene (URL)


  /**
   * readScene - read VRML input stream
   * @param location input name for error messages (e.g. baseurl or "<stdin>")
   */

  public synchronized void readScene (InputStream input, String baseurl, String location)
  {
    clear ();  // clear old data, reset flags
    // lastfilename_ is null, reload only possible via URL

    try
    {
      if (baseurl != null)
        baseurl_ = new URL (baseurl);
    } catch (MalformedURLException e) { }

    if (location == null)
      location = (baseurl != null) ? baseurl : "input stream";

    // TODO: handle compressed streams

    parseInput (new VRMLparser (input), location);

  } // readScene (stream)


  // do the parsing

  private void parseInput (VRMLparser parser, String location)
  {
    root = parser.readStream ();
    // does not throw IOException. try statements above catch IOExceptions on VRMLparser construction

    if (root != null)
    {
      nodenames_ = parser.getNodeNames ();
      if (VRwave.verbose)
        System.out.println ("parsing was ok.");
    }
    else
    {
      System.out.println ("error on parsing " + location);
      if (parser.getVersion () == 0.0f)
        System.out.println ("bad header");
    }

  } // parseInput


  /**
   * draw the scene (called by SceneCanvas.paint)
   */

  public synchronized void draw ()
  {
    if (firstdraw_)
    {
      GE3D.initGE3D ();

      // antialiasing is the only OpenGL feature missing in Mesa
      supportsantialiasing_ = GE3D.antialiasingSupport ();
      if (menuitems_ != null && supportsantialiasing_ == 0)
        menuitems_ [MenuDef.M_AALines].disable ();

      defview_ = null;

      if (VRwave.verbose)
        System.out.println ("*** build ***");
      builder.buildScene (this, root, baseurl_);

      if (defview_ != null)
        viewcam_.initialViewpoint (defview_);  // becomes active if first one

      // display some info (polygon count etc.)

      // default view in VRML 2.0 at (0, 0, 10)
      // (should offer a "view all" function)

      firstdraw_ = false;
    }

    timestamp_ = currentTime ();  // set timestamp_ to current time (is the same for for a whole frame) 
    // may measure time per frame

    // even Frame does not know that it is invisible when iconified
    // System.err.println ("Frame is valid, visible, showing, enabled: " +
    // frame_.isValid () + ", " + frame_.isVisible () + ", " + frame_.isShowing () + ", " + frame_.isEnabled ());

    if (behavior_)  // handle events and scripts only when behavior is ON
      if (sensors_ != null)
        for (int i = 0;  i < sensors_.size ();  i++)
          ((Sensor) sensors_.elementAt (i)).evaluate (timestamp_);

    // System.out.println ("*** draw ***");

    // may change cursor on "slow" draw here (only possible on Frame)

    drawer.drawScene (this, root);

  } // draw


  // finishedDraw (including swapBuffers and GUI drawings)
  // opportunity to measure time

  void finishedDraw ()
  {
    if (!showframerate_)  // might calculate fps for EAI
      return;

    float drawtime = (float) (currentTime () - timestamp_);  // seconds
    if (drawtime < 0.01f)
      drawtime = 0.01f;
    int drawsec = (int) drawtime;
    int dsechd = (int) (drawtime * 100.0f) - 100 * drawsec;
    framespersecond_ = (1.0f / drawtime);
    statusMessage ("last frame: " +
      drawsec + (dsechd < 10 ? ".0" : ".") + dsechd +
      " s (" + (int) framespersecond_ + " fps)", 0);  // ### translator
  }


  /**
   * build a new subgraph and add created nodes as children nodes to a grouping node (Inline)
   */

  public synchronized void buildInline (Inline node1, GroupNode node2)
  {
    builder.buildScene (this, node2, ((InlineData) node1.userdata).url_);
    // also for file inlines only one buildScene active (synchronized via Scene)
    ((InlineData) node1.userdata).inline_ = (GroupNode) node2;
    redraw ();
  }

  /**
   * build a new subgraph (needed for EAI's createVrmlFromString)
   */

  public synchronized void buildNode (GroupNode node)
  {
    builder.buildScene (this, node, null);  // TODO: pass correct base URL
    // TODO: turn off top-level node type checking
  }

  /**
   * pick the scenegraph. fills out Hitpoint.
   * @return node hit object
   */

  public synchronized Node pick (float fx, float fy, VHitpoint hit)
  {
    if (firstdraw_)  // no pick before build
      return null;

    Ray ray = getRay (fx, fy);
    return picker.pickScene (this, root, ray, hit, false, false);  // pick geometry, not sensors
  }

  public synchronized Ray getRay (float fx, float fy)
  {
    return (viewcam_.viewingRay (fx, fy, winaspect_));
  }

  /**
   * pick the scenegraph. fills out Hitpoint.
   * if flag dragsens is set, pick only dragsensors, otherwise pick geometry
   * @return node hit object
   */

  public synchronized Node pick (float fx, float fy, VHitpoint hit, boolean sensors, boolean keeptrf)
  {
    if (firstdraw_)  // no pick before build
      return null;

    Ray ray = viewcam_.viewingRay (fx, fy, winaspect_);

    // System.err.println ("pick (" + fx + ", " + fy + "), " + winaspect_);
    // System.err.println (ray);

    return picker.pickScene (this, root, ray, hit, sensors, keeptrf);
  }

  /**
   * activate an anchor, given by a URL string.
   * target in params passed to AppletContext (ignored otherwise)
   */

  public void activateAnchor (String urlstr, String[] params, int numparams)
  {
    if (applet_ != null)  // show document via applet context
    {
      URL url;
      try
      {
        url = new URL (urlstr);
      }
      catch (MalformedURLException e)
      {
        System.err.println ("bad URL: \"" + urlstr + "\"");
        return;
      }

      System.err.println ("activate Anchor \"" + urlstr + "\"");

      String target = null;  // only applet may handle target
      while (numparams-- > 0)
        if (params [numparams].startsWith ("target="))
        {
          target = params [numparams].substring (7);  // strlen "target="
          break;
        }

      // busy cursor set by browser
      AppletContext context = applet_.getAppletContext ();
      if (target != null)
      {
        System.err.println ("anchor target: " + target);
        context.showDocument (url, target);
      }
      else
        context.showDocument (url);

      return;

    } // via applet

    // remote browser call - Unix specific (see fetchNetscape of VRweb)
    urlstr = substChar (urlstr, '\'', "%27");  // get a shell safe URL encoding
    urlstr = substChar (urlstr, ',', "%2c");   // netscape interprets , as arg separator in openURL

    String[] command = { "/bin/sh", "-c", "netscape -remote 'openURL(" + urlstr + ")'" };  // Unix specific
    // target information cannot be passed via netscape remote calls

    if (frame_ != null)
    { // System.err.println ("--- finished work ---");
      frame_.setCursor (Frame.WAIT_CURSOR);  // will only show up with JDK 1.1
    }

    if (VRwave.verbose)
      System.out.println ("netscape remote call:\n" + iicm.vrml.pw.Decompression.cmdToString (command));
    try
    {
      Process netscape = Runtime.getRuntime ().exec (command);
      int exitval = netscape.waitFor ();
      if (exitval != 0)
      {
        System.err.println ("VRwave: could not execute " + iicm.vrml.pw.Decompression.cmdToString (command) +
          "\nnetscape must be running to handle remote requests");
      }
    }
    catch (Exception e)
    {
      System.err.println ("Error on netscape remote call: " + iicm.vrml.pw.Decompression.cmdToString (command));
      System.err.println ("got exception: " + e);  // SecurityException or WriteError
    }

    if (frame_ != null)
    { // System.err.println ("--- finished work ---");
      frame_.setCursor (Frame.DEFAULT_CURSOR);
    }

  } // activateAnchor

  /**
   * show help file via web browser
   * @see #activateAnchor
   */

  public void showHelpfile (String topic)
  {
    // System.err.println ("show help file " + topic);
    activateAnchor (VRwave.helpurl_ + topic, null, 0);

  } // showHelpFile

  /**
   * little helper to substitute each occurance of character c by s
   * in String str
   */

  public static String substChar (String str, char c, String s)
  {
    int pos = str.indexOf (c);
    if (pos < 0)
      return str;

    StringBuffer buf = new StringBuffer ();
    int from = 0;
    do
    {
      buf.append (str.substring (from, pos));
      buf.append (s);
      from = pos + 1;
      pos = str.indexOf (c, from);
    } while (pos >= 0);

    buf.append (str.substring (from));

    return buf.toString ();
  }

  /**
   * choose a file to open
   */

  public void openFile ()
  {
    if (dlgfileopen_ == null)
      dlgfileopen_ = new DialogFileOpen (frame_);

    DialogFileOpen dlg = dlgfileopen_;
    // width cannot be changed via resize

    dlg.show ();  // modal

    String file = dlg.getFile ();
    if (file == null)  // cancel
      return;

    String dir = dlgfileopen_.getDirectory ();
    String filename = dir + file;

    // System.out.println ("File: " + file);
    // System.out.println ("Dir : " + dlgfileopen_.getDirectory ());

    System.out.println ("reading scene " + filename);
    readScene (filename, null);
  } // openFile

  /**
   * choose a location to open
   */

  public void openLocation ()
  {
    if (dlgopenlocation_ == null)
      dlgopenlocation_ = new DialogOpenLocation (frame_);

    DialogOpenLocation dlg = dlgopenlocation_;
    dlg.show ();  // modal

    String url = dlg.getURLString ();
    if (url == null)  // cancel
      return;
    System.out.println ("opening location: " + url);

    try
    {
      readScene (new URL (url));
    }
    catch (MalformedURLException e)
    {
      System.err.println ("invalid URL <" + url + ">");
    }
  } // openLocation

  /**
   * reload file/URL opened last time
   */

  public boolean reloadFile ()
  {
    if (lastfilename_ != null)
      readScene (lastfilename_, (baseurl_ != null) ? baseurl_.toString () : null);
    else if (lasturl_ != null)
      readScene (lasturl_);
    else
    {
      System.out.println ("no scene loaded or input not available.");
      return false;
    }

    return true;
  }


  /** tiny helper to toggle a Frame */
  public static void toggleFrame (Frame f)
  {
    if (f.isVisible ())
      f.hide ();
    else
      f.show ();
  }

  /**
   * toggle color chooser
   */

  public void toggleColorChooser ()
  {
    if (dlgcolchooser_ == null)
      dlgcolchooser_ = new DLGColourChoose (Translator.L_ColChooser_Targets, this);
    dlgcolchooser_.setColours (color_, 0);
    toggleFrame (dlgcolchooser_);
  }

  /**
   * apply color callback
   */

  public void applyColour (DLGColourChoose dlg)
  {
    dlgcolchooser_.getColours (color_);
    // System.out.println ("got background " + new java.awt.Color (color_[0]));
    redraw ();
  }

  /**
   * toggle about dialog
   */

  public void toggleAbout ()
  {
    if (dlgabout_ == null)
      dlgabout_ = new DialogAbout ();
    toggleFrame (dlgabout_);
  }

  /**
   * toggle settings dialog
   */

  public void toggleSettings ()
  {
    if (dlgsettings_ == null)
      dlgsettings_ = new DialogSettings (this);
    toggleFrame (dlgsettings_);
  }

  /**
   * get the URL of the currently shown scene
   */

  public String getWorldURL ()
  {
    if (baseurl_ != null)
      return (baseurl_.toString ());
    else
      return ("");
  }

  public Hashtable getNodeNames ()
  {
    return nodenames_;
  }

  public synchronized void replaceScene (GroupNode node)
  {
    // TODO: baseurl now null, check if this is correct
    clear ();
    root = node;
    redraw ();
  }

} // Scene
