/*
 * Builder.java
 *
 * created: mpichler, 19960924
 *
 * changed: mpichler, 19970804
 * changed: apesen, 19970807
 * changed: kwagen, 19970917
 *
 * $Id: Builder.java,v 1.27 1997/09/17 08:06:41 kwagen Exp $
 */


package iicm.vrml.vrwave;

import iicm.vrml.pw.*;
import iicm.vrml.pwutils.*;
import iicm.ge3d.GE3D;
import iicm.vrml.vrwave.pwdat.*;
import iicm.utils3d.Vec3f;

import java.util.Enumeration;
import java.net.*;

/**
 * Builder - preprocessing traversal.
 * Copyright (c) 1996,97 IICM
 *
 * @author Michael Pichler, Karl Heinz Wagenbrunn
 * @version 0.3, latest change:  7 May 97
 */


public class Builder extends Traverser
{
  Scene scene_;  // the scene to be built
  URLServer urlserver_;
  Thread thread_;
  URL currbaseurl_;

  /**
   * build (preprocessing) of scene graph
   */

  void buildScene (Scene scene, GroupNode root, URL baseurl)
  {
    if (root == null)
      return;

    scene_ = scene;
    currbaseurl_ = baseurl;
    tGroupNode (root);
  }


  // Grouping nodes: check node types of children
  public void tGroupNode (GroupNode group)
  {
    Enumeration e = group.getChildrenEnumerator ();

    while (e.hasMoreElements ())
    {
      // check for legal children nodes (see Table 4.6 of VRML 97 "Concepts")
      // this should rather be in `pw' to have line number information
      Node child = (Node) e.nextElement ();
      if (! (
        child instanceof GroupNode ||
        child instanceof Light ||
        child instanceof Sensor ||
        child instanceof Interpolator ||
        child instanceof Background ||
        child instanceof Fog ||
        child instanceof NavigationInfo ||
        child instanceof Script ||
        child instanceof Shape ||
        child instanceof Sound ||
        child instanceof Viewpoint ||
        child instanceof WorldInfo ||
        child instanceof RouteNode ||
        child instanceof ProtoInstance  // TODO: check type of first PROTO child
         )  )
      {
        System.err.print ("Error in VRML data: Node of type " + child.nodeName () + " is not a valid child ");
        String groupname = group.nodeName ();
        if (groupname.charAt (0) == '<')
          System.err.println ("at root level");
        else
          System.err.println ("for group " + groupname);
      }
      child.traverse (this);
    }
  } // tGroupNode

  // Transform
  protected void tTransform (Transform trf)
  {
    TransformData dat = new TransformData (trf);
    trf.userdata = dat;  // holds transformation matrix for Drawer

    // should compute bounding box here
    tGroupNode (trf);  // traverse children
  }
  public native static void buildTransform (float[] t, float[] r, float[] s,
    float[] sr, float[] c, float[] trfmat);
  // args: translation, rotation (axis/angle), scale, scalerotation (axis/angle), center, trfmat (out)
  // needed by TransformData

  protected void tInline (Inline inline)
  {
    if (inline.userdata == null)
      inline.userdata = new InlineData ();
    InlineData dat = (InlineData) inline.userdata;

    Vec3f size = new Vec3f (inline.bboxSize.getValue ());  // bounding box: non-exposed field
    if (size.value_[0] >= 0 && size.value_[1] >= 0 && size.value_[2] >= 0)
    {
      dat.bboxmin_ = new Vec3f (inline.bboxCenter.getValue ());
      dat.bboxmin_.sincrease (-0.5f, size);

      dat.bboxmax_ = new Vec3f (inline.bboxCenter.getValue ());
      dat.bboxmax_.sincrease (0.5f, size);
    }

    if (dat.inline_ == null)
    {
      if (urlserver_ == null)
        urlserver_ = new URLServer (scene_);

      urlserver_.addRequest (inline, currbaseurl_);

      if (thread_ == null || !urlserver_.isActive ())
      {
        thread_ = new Thread (urlserver_);
        thread_.start ();
      }
    }
  } // Inline

  protected void tSwitch (Switch sw)
  {
    sw.userdata = new SwitchData ();

    // build all children
    MFNode nochildren = sw.children;
    sw.children = sw.choice;
    tGroupNode (sw);
    sw.children = nochildren;
  }

  protected void tLOD (LOD lod)
  {
    lod.userdata = new LODdata ();

    // build all children
    MFNode nochildren = lod.children;
    lod.children = lod.level;
    tGroupNode (lod);
    lod.children = nochildren;
  }

  protected void tBillboard (Billboard billb)
  {
    billb.userdata = new BillboardData ();

    tGroupNode (billb);
  }

  // Common Nodes
  protected void tAudioClip (AudioClip n)  { }

  // Light sources
  protected void tDirectionalLight (DirectionalLight lgt)
  { scene_.setHasLight ();
  }
  protected void tPointLight (PointLight lgt)
  { scene_.setHasLight ();
  }
  protected void tSpotLight (SpotLight lgt)
  { // scene_.setHasLight ();
  }

  protected void tShape (Shape shape)
  {
    // check types (should go into `pw')
    Node app = shape.appearance.getNode ();
    if (app != null && !(app instanceof Appearance || app instanceof ProtoInstance))  // TODO: check 1st PROTO child
      System.err.println ("Error in VRML data: Shape's appearance field should hold an Appearance node, not a " +
        app.nodeName ());
    Node geom = shape.geometry.getNode ();
    if (geom != null && !(geom instanceof Geometry || geom instanceof ProtoInstance))  // TODO: check 1st PROTO field
      System.err.println ("Error in VRML data: Shape's geometry field should hold any geometry node, not a " +
        geom.nodeName ());

    // TODO: shall all SFNode type children automatically be traversed
    // by Builder (not Drawer!)?
    Node.traverseNode (this, app);
    Node.traverseNode (this, geom);
  } // Shape

  protected void tSound (Sound n)  { }
  protected void tScript (Script n)  { }
  protected void tWorldInfo (WorldInfo n)  { }

  // Sensor Nodes
  protected void tCylinderSensor (CylinderSensor n)  
  {
    // scene_.addSensor (n);  // no need to evaluate, handled on event
  }
  protected void tPlaneSensor (PlaneSensor n)  
  { 
    // scene_.addSensor (n);  // no need to evaluate, handled on event
  }
  protected void tProximitySensor (ProximitySensor n)  
  {
    // scene_.addSensor (n);  // no need to evaluate, handled on event
  }
  protected void tSphereSensor (SphereSensor n)  
  {
    // scene_.addSensor (n);  // no need to evaluate, handled on event
  }
  protected void tTimeSensor (TimeSensor n)  
  {
    scene_.addSensor (n); 
  }
  protected void tTouchSensor (TouchSensor n)  
  {
    // scene_.addSensor (n);  // no need to evaluate, handled on event
  }
  protected void tVisibilitySensor (VisibilitySensor n)  
  {
    // scene_.addSensor (n);  // no need to evaluate, handled on event
  }


  // Geometry Nodes

  protected void tBox (Box box)
  {
    BoxData dat = new BoxData ();
    box.userdata = dat;
    float[] size = box.size.getValue ();
    float[] min = dat.min_;
    float[] max = dat.max_;

    max[0] = Math.abs (size [0]) / 2.0f;
    max[1] = Math.abs (size [1]) / 2.0f;
    max[2] = Math.abs (size [2]) / 2.0f;
    min[0] = - max[0];
    min[1] = - max[1];
    min[2] = - max[2];
  }

  protected void tCone (Cone cone)
  {
    ConeData dat = new ConeData ();
    cone.userdata = dat;
    dat.parts_ = 0;
    if (cone.bottom.getValue ())
      dat.parts_ |= GE3D.cyl_bottom;
    if (cone.side.getValue ())
      dat.parts_ |= GE3D.cyl_sides;
    // no top
  }

  protected void tCylinder (Cylinder cylinder)
  {
    CylinderData dat = new CylinderData ();
    cylinder.userdata = dat;
    dat.parts_ = 0;
    if (cylinder.bottom.getValue ())
      dat.parts_ |= GE3D.cyl_bottom;
    if (cylinder.side.getValue ())
      dat.parts_ |= GE3D.cyl_sides;
    if (cylinder.top.getValue ())
      dat.parts_ |= GE3D.cyl_top;
  }

  protected void tElevationGrid (ElevationGrid n)  { }

  protected void tExtrusion (Extrusion extr)
  {
    // nothing to be done if spine curve or cross section not defined
    if (extr.crossSection.getValueCount () < 1 || extr.spine.getValueCount () < 1)
      return;

    if (extr.userdata == null)
      extr.userdata = new ExtrusionData ();
    ExtrusionData dat = (ExtrusionData) extr.userdata;

    int numcrs = extr.crossSection.getValueCount ();
    int numspine = extr.spine.getValueCount ();

    int numscale = extr.scale.getValueCount ();
    float[] scale = {1, 1, 1};  // default if scale is empty
    if (numscale < 1)
      numscale = 1;
    else
      scale = extr.scale.getValueData ();

    int numrot = extr.orientation.getValueCount ();
    float[] rot = {1, 1, 1};  // default if orientation is empty
    if (numrot < 1)
      numrot = 1;
    else
      rot = extr.orientation.getValueData ();

    int caps = 0;  // bit 0: beginCap (not) drawn; bit 1: endCap (not) drawn
    int numfaces = (numcrs - 1) * (numspine - 1) * 2;
    int ncoordinds = numfaces * 4;

    if (extr.beginCap.getValue ())
    {
      numfaces++;
      ncoordinds += numcrs+1;
      caps |= 1;
    }
    
    if (extr.endCap.getValue ())
    {
      numfaces++;
      ncoordinds += numcrs+1;
      caps |= 2;
    }

    int numtexcoords = numcrs * numspine + (extr.beginCap.getValue () || extr.endCap.getValue () ? numcrs : 0);
    dat.coords = new float [numcrs * numspine * 3];
    dat.coordinds = new int [ncoordinds];
    dat.texcoords = new float [numtexcoords * 2];
    dat.texcoordinds = new int [ncoordinds];

    ncoordinds = buildExtrusionData (extr.spine.getValueData (), numspine,
      extr.crossSection.getValueData (), numcrs, scale, numscale, rot, numrot,
      caps, dat.coords, dat.coordinds, dat.texcoords, dat.texcoordinds);
    dat.numcoordinds = ncoordinds;
    dat.numfaces = numfaces;
    dat.facenormals = new float [3 * numfaces];
    buildFacenormals (dat.coords, dat.coordinds, ncoordinds, dat.facenormals, extr.ccw.getValue ());

    if (VRwave.convexify && !extr.convex.getValue () && caps > 0)
    {
      int num = numfaces + 2 * numcrs;
      dat.convexify = new IntArray (4 * num);
      dat.newfnormals = new FloatArray (3 * num);
      dat.newtexcinds = new IntArray (4 * num);
  
      convexify (dat.coords, dat.coordinds, ncoordinds, dat.facenormals, null, 0, dat.texcoordinds,
        dat.convexify, dat.newfnormals, null, dat.newtexcinds);
    }

    if (VRwave.asmooth)
    {
      dat.normalindex = new int [ncoordinds];
      float[] normals = new float [3 * ncoordinds];
      int numnormals = autosmooth (dat.coordinds, ncoordinds, dat.facenormals,
        numfaces, extr.creaseAngle.getValue (), normals, dat.normalindex);
      dat.normallist = new float [3 * numnormals];
      System.arraycopy (normals, 0, dat.normallist, 0, 3 * numnormals);
    }
  }

  native int buildExtrusionData (
    float[] spine, int numspine,  // in: spine point coordinates, number of spine points
    float[] crsection, int numcrs,  // in: cross section coordinates, number of cross section coordinates
    float[] scale, int numscale,  // in: scale values along the spine, number of scale values
    float[] rotation, int numrot,  // in: rotations of cross section plane along the spine, #rotations
    int caps,  // in: bit 0/1 set => draw also begin/end cap
    float[] coords, int[] coordinds,  // out: coords and coordindices to use for GE3D.drawFaceSet
    float[] texcoords, int[] texcinds  // out: texture coordinates and texture coordinate indices
  );

  protected void tIndexedFaceSet (IndexedFaceSet faceset)
  {
    if (faceset.userdata == null)
      faceset.userdata = new IndexedFaceSetData ();
    IndexedFaceSetData dat = (IndexedFaceSetData) faceset.userdata;

    dat.coords_ = null;
    Node cnode = faceset.coord.getNode ();
    if (cnode != null)
    {
      try
      {
        dat.coords_ = ((Coordinate) cnode).point;
      }
      catch (ClassCastException e)
      {
        System.out.println ("IndexedFaceSet. error: " + cnode.nodeName () + " coord (must be Coordinate)");
      }
    }
    // System.out.println ("coord: " + dat.coords_);

    dat.color_ = null;
    cnode = faceset.color.getNode ();
    if (cnode != null)
    {
      try
      {
        dat.color_ = ((Color) cnode).color;
      }
      catch (ClassCastException e)
      {
        System.out.println ("IndexedFaceSet. error: " + cnode.nodeName () + " color (must be Color)");
      }
    }

    cnode = faceset.texCoord.getNode ();
    if (cnode != null)
    {
      try
      {
        dat.texcoords_ = ((TextureCoordinate) cnode).point.getValueData ();
      }
      catch (ClassCastException e)
      {
        System.out.println ("IndexedFaceSet. error: " + cnode.nodeName () + " texCoord (must be TextureCoordinate)");
      }
    }
    else  // calculate default texture coordinates
    {
      float[] coords_ = null;
      if (dat.coords_ != null)
      {
        coords_ = dat.coords_.getValueData ();
        dat.texcoords_ = new float [2 * dat.coords_.getValueCount ()];
        buildDefaultTexcoords (coords_, 3 * dat.coords_.getValueCount (), dat.texcoords_);
      }
    }
    
    if (dat.numfaces_ == 0)
    { // calculate face normal data for the first time
      // or after they have been changed (exposedField)
      MFInt32 cindex = faceset.coordIndex;
      int numfaces = getNumfaces (cindex.getValueData (), cindex.getValueCount ());
      dat.numfaces_ = numfaces;
      float[] coords = null;
      if (dat.coords_ != null)
        coords = dat.coords_.getValueData ();
      if (numfaces != 0 && coords != null)
      {
        dat.fnormals_ = new FloatArray (3 * numfaces);
        boolean ccw = faceset.ccw.getValue ();
        buildFacenormals (coords, cindex.getValueData (), cindex.getValueCount (), dat.fnormals_.getData (), ccw);
      }
      else
        dat.fnormals_ = null;
    } // face normals

    // handle non convex polygons ("convexify", field convex set to FALSE)
    dat.convexify_ = null;
    dat.newcolorinds_ = null;
    dat.newtexcoordinds_ = null;

    if (!faceset.convex.getValue () && VRwave.convexify)
    {
      int num = dat.numfaces_ * dat.coords_.getValueCount ();
      IntArray coordinds = new IntArray (4 * num);
      FloatArray fnormals = new FloatArray (3 * num);
      MFInt32 cindex = faceset.coordIndex;
      int[] colinds = null;
      if (faceset.colorIndex.getValueCount () > 0)
        colinds = faceset.colorIndex.getValueData ();
      IntArray colorinds = null;
      int matb = 0;  // > 0 when have to update per face colors; indexed: 1, sequentially: 2

      if (dat.color_ != null && !faceset.colorPerVertex.getValue ())  // color per face(indexed)
      {
        matb = (colinds != null) ? 1 : 2;
        colorinds = new IntArray (num);
      }

      int[] texcindex = null;
      IntArray texcoordinds = null;
      if (faceset.texCoordIndex.getValueCount () > 0)
      {
        texcindex = faceset.texCoordIndex.getValueData ();
        texcoordinds = new IntArray (4 * num);
      }

      convexify (dat.coords_.getValueData (), cindex.getValueData (), cindex.getValueCount (),
        dat.fnormals_.getData (), colinds, matb, texcindex, coordinds, fnormals, colorinds, texcoordinds);

      dat.convexify_ = coordinds;
      dat.fnormals_ = fnormals;
      dat.numfaces_ = fnormals.getCount () / 3;
      if (matb > 0)
        dat.newcolorinds_ = colorinds;
      if (texcindex != null)
        dat.newtexcoordinds_ = texcoordinds;
    }

    dat.normallist_ = null;
    dat.normalindex_ = null;
    cnode = faceset.normal.getNode ();
    if (cnode == null)  // generate normals ("autosmooth") using creaseAngle
    {
      if (VRwave.asmooth)
      {
        MFInt32 cindex = faceset.coordIndex;
        dat.normalindex_ = new int [cindex.getValueCount ()];
        float[] normals = new float [3 * cindex.getValueCount ()];
        int numnormals = autosmooth (cindex.getValueData (), cindex.getValueCount (), dat.fnormals_.getData (),
          dat.numfaces_, faceset.creaseAngle.getValue (), normals, dat.normalindex_);
        dat.normallist_ = new float [3 * numnormals];
        System.arraycopy (normals, 0, dat.normallist_, 0, 3 * numnormals);
      }
    }
    else  // field normal not NULL
    {
      if (faceset.normalPerVertex.getValue ())  // ignore normals from Normal node, if per face
      {
        try
        {
          dat.normallist_ = ((Normal) cnode).vector.getValueData ();
          if (faceset.normalIndex.getValueCount () > 0)
            dat.normalindex_ = faceset.normalIndex.getValueData ();
          else
            dat.normalindex_ = faceset.coordIndex.getValueData ();
        }
        catch (ClassCastException e)
        {
          System.out.println ("IndexedFaceSet. error: " + cnode.nodeName () + " normal (must be Normal)");
        }
      }
    }
  } // IndexedFaceSet

  native int getNumfaces (int[] coordindex, int numcind);  // get no. of faces from coordIndex array
  native void buildFacenormals (  // build face normal vectors
    float[] coords, int[] coordindex, int numcind,  // in: coords and coordIndex arrays, no. of cinds
    float[] fnormals,  // out: face normal array (already allocated)
    boolean ccw
  );
  // default texture coordinates
  native void buildDefaultTexcoords (float[] coords, int numcoords, float[] texcoords);
  // autosmooth
  native int autosmooth (
    int[] coordindex, int numcind,  // in: coordIndex, no. of coordindices
    float[] fnormals, int numfaces,  // in: face normals (calculated by buildFacenormals), no. of faces
    float creaseangle,  // in: crease angle
    float[] normallist, int[] normalindex // out: list of normals and normal index (already allocated)
  );
  // convexify
  native void convexify (
    float[] coords, int[] coordindex, int numcind,  // in: coords and coordIndex arrays, no. of cinds
    float[] fnormals,  // in: face normal array
    int[] colorinds,  // in: color index array if defined per face(indexed), otherwise null 
    int matb,  // in: 2 if per face, 1 if per faceindexed, otherwise 0
    int[] texcoordindex,  // in: texture coordinates index array
    IntArray newcoordindex,  // out: new vertex index (triangulated polygons). data_ and count_ set in native code
    FloatArray newnormals,  // out: new face normals. data_ and count_ set in native code
    IntArray newcolorinds,  // out: new color index array if per face(indexed). data_ and count_ set in native code
    IntArray newtexcoordinds  // out: new texture coordinates index array. data_ and count_ set in native code
  );

  // IndexedLineSet
  protected void tIndexedLineSet (IndexedLineSet lineset)
  {
    if (lineset.userdata == null)
      lineset.userdata = new IndexedLineSetData ();
    IndexedLineSetData dat = (IndexedLineSetData) lineset.userdata;
    
    dat.coords_ = null;
    Node cnode = lineset.coord.getNode ();
    if (cnode != null)
    {
      try
      {
        dat.coords_ = ((Coordinate) cnode).point;
      }
      catch (ClassCastException e)
      {
        System.out.println ("IndexedLineSet. error: " + cnode.nodeName () + " coord (must be Coordinate)");
      }
    }

    dat.color_ = null;
    cnode = lineset.color.getNode ();
    if (cnode != null)
    {
      try
      {
        dat.color_ = ((Color) cnode).color;
      }
      catch (ClassCastException e)
      {
        System.out.println ("IndexedLineSet. error: " + cnode.nodeName () + " color (must be Color)");
      }
    }
  }

  // PointSet
  protected void tPointSet (PointSet pset)
  {
    if (pset.userdata == null)
      pset.userdata = new PointSetData ();
    PointSetData dat = (PointSetData) pset.userdata;
    
    dat.coords_ = null;
    Node cnode = pset.coord.getNode ();
    if (cnode != null)
    {
      try
      {
        dat.coords_ = ((Coordinate) cnode).point;
      }
      catch (ClassCastException e)
      {
        System.out.println ("PointSet. error: " + cnode.nodeName () + " coord (must be Coordinate)");
      }
    }

    dat.color_ = null;
    cnode = pset.color.getNode ();
    if (cnode != null)
    {
      try
      {
        dat.color_ = ((Color) cnode).color;
      }
      catch (ClassCastException e)
      {
        System.out.println ("PointSet. error: " + cnode.nodeName () + " color (must be Color)");
      }
    }
  }

  protected void tSphere (Sphere n)  { }  // empty
  protected void tText (Text n)  { }

  // Geometric Properties
  protected void tColor (Color n)  { }
  protected void tCoordinate (Coordinate n)  { }
  protected void tNormal (Normal n)  { }
  protected void tTextureCoordinate (TextureCoordinate n)  { }

  // Appearance Nodes
  protected void tAppearance (Appearance app)
  {
    // check types (should go into `pw')
    Node mat = app.material.getNode ();
    if (mat != null && !(mat instanceof Material || mat instanceof ProtoInstance))  // TODO: check 1st PROTO field
      System.err.println ("Error in VRML data: material field of Appearance should hold a Material node, not a " +
        mat.nodeName ());
    Node tex = app.texture.getNode ();
    if (tex != null && !(tex instanceof Texture || tex instanceof ProtoInstance))  // TODO: check 1st PROTO field
      System.err.println ("Error in VRML data: texture field of Appearance should hold any texture node, not a " +
        tex.nodeName ());
    Node ttrf = app.textureTransform.getNode ();
    if (ttrf != null && !(ttrf instanceof TextureTransform || ttrf instanceof ProtoInstance))  // TODO: check PROTO field
      System.err.println (
        "Error in VRML data: textureTransform field of Appearance should hold a TextureTransform node, not a " +
        ttrf.nodeName ());

    Node.traverseNode (this, mat);
    Node.traverseNode (this, tex);
    Node.traverseNode (this, ttrf);
  } // Appearance

  protected void tFontStyle (FontStyle n)  { }
  protected void tMaterial (Material n)  { }

  protected void tImageTexture (ImageTexture texture)
  {
    if (texture.userdata == null)
      texture.userdata = new ImageTextureData ();
  }

  protected void tMovieTexture (MovieTexture n)  { }

  protected void tPixelTexture (PixelTexture texture)
  {
    if (texture.userdata == null)
      texture.userdata = new PixelTextureData ();
  }

  protected void tTextureTransform (TextureTransform trf)
  {
    TextureTransformData dat = new TextureTransformData ();
    trf.userdata = dat;
    buildTextureMatrix (
      trf.center.getValue (), trf.rotation.getValue (),
      trf.scale.getValue (), trf.translation.getValue (), dat.trfmat_);
  }
  native void buildTextureMatrix (float[] c, float r, float[] s,
    float[] t, float[] trfmat);
  // args: center, rotation angle, scale, translation, trfmat (out)


  // Interpolator Nodes
  protected void tColorInterpolator (ColorInterpolator n)  { }
  protected void tCoordinateInterpolator (CoordinateInterpolator n)  { }
  protected void tNormalInterpolator (NormalInterpolator n)  { }
  protected void tOrientationInterpolator (OrientationInterpolator n)  { }
  protected void tPositionInterpolator (PositionInterpolator n)  { }
  protected void tScalarInterpolator (ScalarInterpolator n)  { }

  // Bindable Nodes
  protected void tBackground (Background n)  { }
  protected void tFog (Fog n)  { }
  protected void tNavigationInfo (NavigationInfo n)  { }

  protected void tViewpoint (Viewpoint vp)
  { scene_.hasViewpoint (vp);
  }

  // instance of a PROTO node
  protected void tProtoInstance (ProtoInstance n)  { }


  public void clearThreads ()
  {
    if (thread_ != null)
    {
      thread_.stop ();
      thread_ = null;
    }
    if (urlserver_ != null)
    {
      urlserver_.clearAll ();
      urlserver_ = null;
    }
  }

  // static { System.loadLibrary ("vrwbuild"); }
  // all java, right from the beginning

} // Builder
