/*********************************************************************
 * vi:set ts=4 nowrap:
 *
 * $Id: ipc-sliders.c,v 1.3 1998/12/26 21:14:26 nicb Exp $
 *
 * ipc sliders 
 *
 * These are supposed to be a set of functions directly interfacing
 * the control opcodes (see control.[ch] in the main distribution)
 * and using the System V IPC message passing to talk to a graphic
 * process that implement sliders; the graphic process is called as
 * follows:
 *
 * <graphic_process_path> -r <receive queue id> -s <send queue id>
 *
 * and it can be any process that handles these arguments and does something
 * sensible with them (so graphic processes can be redesigned separately
 * from this piece of code)
 *
 * Originally written by Nicola Bernardini [nicb@axnet.it]
 *
 * This module is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * CSOUND_SLIDERS defines the name of the environment variable which may
 * hold the path to the sliders graphic process; it may be redefined at
 * compile time
 */

#if !defined(CSOUND_SLIDERS)
#	define CSOUND_SLIDERS	"CSOUND_SLIDERS"
#endif /* !defined(CSOUND_SLIDERS) */

/*
 * CSOUND_SLIDERS_DEFAULT is the default pathname if CSOUND_SLIDERS is
 * not defined; it may be redefined at compile time
 */

#if !defined(CSOUND_SLIDERS_DEFAULT)
#	define CSOUND_SLIDERS_DEFAULT	"csound_sliders"
#endif /* !defined(CSOUND_SLIDERS_DEFAULT) */

/*
 * in the current implementation of the interface routines, the slider
 * value is an int; we want to keep this variable as much as possible,
 * so we redefine its type
 */

typedef int SliderValue;

/*
 * These are the interface routines
 */

#include "../cs.h"

static void flush_sliders();

void
DisplaySliders(int numsliders)
{
	static int manage_sliders(int);

	if (manage_sliders(numsliders) <= 0)
		die("could not start sliders\n");
}

SliderValue
GetSliderValue(int slider)
{
	static SliderValue read_slider(int);
	--slider;
	flush_sliders();
	return read_slider(slider);
}

void
SetSliderValue(int slider, SliderValue value)
{
	static void set_slider_value(int, SliderValue);
	--slider;
	flush_sliders();
	set_slider_value(slider, value);
}

void
SetSliderMin(int slider, SliderValue value)
{
	static void set_slider_min(int, SliderValue);
	--slider;
	flush_sliders();
	set_slider_min(slider, value);
}

void
SetSliderMax(int slider, SliderValue value)
{
	static void set_slider_max(int, SliderValue);
	--slider;
	flush_sliders();
	set_slider_max(slider, value);
}

/*
 * below here comes the pulp hardcore part :-)
 */

#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <errno.h>
#include "syserrs.h"

#define FORK_FAILED			(-10)
#define MSGGET_FAILED		(-11)
#define UNHANDLED_SIGNAL	(-12)

/*
 * manage_sliders()
 *
 * 1) check if sliders are up already, and if not, start communications
 * 2) call allocate sliders to check whether there are enough sliders
 *    up to satisfy this call
 * 3) return the number of currently allocated sliders
 */

inline static int num_sliders();

static int
manage_sliders(int nsliders)
{
	static void open_communication();
	static int get_sliders(int);
	inline static int squeue_id();
	inline static int rqueue_id();

	if (squeue_id() < 0 || rqueue_id() < 0)
		open_communication();

	if (nsliders > num_sliders() && get_sliders(nsliders) == nsliders)
	{
		static void build_sliders(int);
		build_sliders(nsliders);
	}

	return num_sliders();
}

/*
 * open_communication():
 *
 * 1) try open a msg queue for us
 * 2) if succeded, fork the graphic process
 * 3) set csound as process group leader, so that when we exit the graphic
 *    process gets killed too
 * 4) establish connections with it
 * 5) set a trap for termination signals to remove the queue upon exiting
 *
 */

static void
open_communication()
{
	static void start_messages();
	inline static int rqueue_id();
	inline static int squeue_id();

	start_messages();

	if (rqueue_id() > 0 && squeue_id() > 0)
	{
		pid_t father_pid = getpid();
		pid_t child_pid = 0;

		syscall_warning(setpgrp(), "setpgrp() - cannot set csound as process group leader");
		child_pid = syscall_fatal(vfork(), FORK_FAILED, "fork failed");

		if (!child_pid)				/* this is the child */
		{
			static char *graphic_process_name();
			static char * const *graphic_process_args(int, int);
			char *gp = graphic_process_name();
			char * const *args = graphic_process_args(squeue_id(), rqueue_id());

			syscall_warning(setpgid(0, father_pid),
				"cannot set graphic process to csound process group id");
			syscall_warning(execv(gp, args),
				"cannot start graphic process \"%s\" - running without sliders",
				gp);
			/* should never get to here if exec does not fail */
			exit(-1);	/* just in case exec fails... */
		}
	}
}

static char *
graphic_process_name()
{
	char *envname = getenv(CSOUND_SLIDERS);
	char *result = envname != (char *) NULL ? envname : CSOUND_SLIDERS_DEFAULT;
	
	return result;
}

/*
 * graphic_process_args:
 *
 * PLEASE NOTE: the client process has to be called with the send/receive
 * id pair inverted! (fairly obvious, or not? :-)
 */

static char * const *
graphic_process_args(int send, int receive)
{
	static char sendq[16] = { '\0' };	/* this must be send->receive */
	static char recvq[16] = { '\0' };	/* this must be receive->send */

	static char * const args[] =
	{
		"-r",
		sendq,							/* send queue to be received */
		"-s",
		recvq,							/* receive queue to be sent  */
		(char *) NULL
	};

	sprintf(sendq, "%d", send);
	sprintf(recvq, "%d", receive);

	return args;
}

/*
 * slider handling part
 */

static SliderValue *__sliders__ = (int *) NULL;
static int __numsliders__ = 0;

typedef enum
{
		FALSE = 0,
		TRUE = 1
} boolean;

inline static void
set_numsliders(int n)
{
	__numsliders__ = n;
}

inline static int
num_sliders()
{
	return __numsliders__;
}

/*
 * get_sliders():
 *
 * 1) should check if a sufficient number of sliders has been allocated,
 *    otherwise should either
 *
 *    a) allocate a set
 *    b) reallocate it
 *
 * 	  depending on whether there was some space already allocated
 *
 * 2) should return the number of sliders allocated in one way or another
 *
 */

static int
get_sliders(int nsliders)
{
		int result = nsliders;

		if (num_sliders() == 0)
		{
			static int allocate_sliders(int);
			result = allocate_sliders(nsliders);
		}
		else if (nsliders > num_sliders())
		{
			static int reallocate_sliders(int);
			result = reallocate_sliders(nsliders);
		}

		return nsliders;
}

static int
_allocate_sliders(int numsliders, SliderValue *core_memory)
{
	int result = 0;
	/*
	 * FIXME: we got no destroy handler (just a create one), so we won't
	 * be able to free the piece we're allocating right now; it should'nt
	 * matter much, since we're keeping everything until the end;
	 * ugly, though :-(
	 */
	__sliders__ = core_memory;

	if (__sliders__ != (SliderValue *) NULL)
	{
		set_numsliders(numsliders);
		result = num_sliders();
	}
	else
		die("memory exhausted - could not allocate slider memory\n");

	return result;
}

inline static int
allocate_sliders(int nsliders)
{
	return _allocate_sliders(nsliders,
		(SliderValue *) malloc(nsliders+1 * sizeof(SliderValue)));
}

inline static int
reallocate_sliders(int nsliders)
{
	return _allocate_sliders(nsliders,
		(SliderValue *) realloc(__sliders__, nsliders+1 * sizeof(SliderValue)));
}

inline static boolean
verify_slider(int slider)
{
	return (slider >= 0 && slider < num_sliders()) ? TRUE : FALSE;
}

inline static void
write_slider(int slider, SliderValue val)
{
	if (verify_slider(slider) == TRUE)
		__sliders__[slider] = val;
	else
		err_printf("write_slider: slider (%d) number out of range\n", slider);
}

inline static SliderValue
read_slider(int slider)
{
	int result = 0;

	if (verify_slider(slider) == TRUE)
		result = __sliders__[slider];
	else
		err_printf("read_slider: slider (%d) number out of range\n", slider);

	return result;
}

/*
 * message passing part
 */

/*
 * protocol: this has to be documented and followed by any graphic application
 * wishing to talk to the csound process
 */

#define	CSOUND_SLIDER_TYPE	(42)	/* any magic number can go here	*/

typedef enum
{
	NO_COMMAND = 0,
	BUILD_SLIDERS = 1,
	SET_SLIDER = 2,
	SET_SLIDER_MIN = 3,
	SET_SLIDER_MAX = 4,
	UNKNOWN_COMMAND
} SliderCommand;

typedef struct
{
	/*	protected data: required by IPC	*/
	long mtype;				/* for us, this is CSOUND_SLIDER_TYPE */
	/*	private data: our data portion	*/
	SliderCommand command;	/* command to be executed	*/
	int slider;				/* slider number			*/
	SliderValue value;		/* value					*/
} SliderMessage;

/*
 * general handling and setup
 *
 * There are two queues: the receiver and the sender (and they have
 * to be sent inverted to the client graphics process)
 */

int __rqid__ = -1;
int __sqid__ = -1;

inline static void
set_rqueue_id(int id)
{
	__rqid__ = id;
}

inline static int
rqueue_id()
{
	return __rqid__;
}

inline static void
set_squeue_id(int id)
{
	__sqid__ = id;
}

inline static int
squeue_id()
{
	return __sqid__;
}

/*
 * Admittedly, what follows is a bit of a hack and it's ugly as hell
 */

static void				/* cannot be inline, used as a function pointer */
stop_messages()
{
	syscall_warning(msgctl(rqueue_id(), IPC_RMID, (struct msqid_ds *) NULL),
				"receive message queue destruction failed");
	syscall_warning(msgctl(squeue_id(), IPC_RMID, (struct msqid_ds *) NULL),
				"send message queue destruction failed");
	set_rqueue_id(-1);
	set_squeue_id(-1);
}

static void
game_over(int sig)
{
	int pid = getpid();
	signal(sig, SIG_IGN);	/* block signals */

	/*
	 * This deletes the message queues - since we don't have a destructor
	 * for sliders, we'll use some signals to call this function
	 * and remove the queues
	 */
	stop_messages();

	signal(sig, SIG_DFL);	/* restore defaults */
	kill(-pid, sig);		/* and send the signal to the process group */
}

inline static void
start_messages()
{
	/*
	 * This creates the message queue - it is set with permissions for
	 * everybody to write in the queue, because the later unofficial
	 * linux csound versions are suid root in order to control the
	 * scheduler
	 */
	set_rqueue_id(syscall_fatal(msgget(IPC_PRIVATE, IPC_CREAT | IPC_EXCL | 0666),
		MSGGET_FAILED,
		"msgget failed: cannot create the receive message queue"));

	set_squeue_id(syscall_fatal(msgget(IPC_PRIVATE, IPC_CREAT | IPC_EXCL | 0666),
		MSGGET_FAILED,
		"msgget failed: cannot create the send message queue"));
	/*
	 * and set a number of signals to remove it
	 */
	signal(SIGTERM, game_over);
	signal(SIGINT, game_over);
	signal(SIGSEGV, game_over);		/* tee hee... */
	signal(SIGFPE, game_over);		/* tee hee... */
}

/*
 * Receiver functions
 */

static void
flush_sliders()
{
	int result = 0;
	SliderMessage received = { CSOUND_SLIDER_TYPE, NO_COMMAND, 0, 0 };

	while((result = msgrcv(rqueue_id(), (struct msgbuf *) &received,
			sizeof(received), CSOUND_SLIDER_TYPE, IPC_NOWAIT)) >= 0)
	{
		static void handle_received_message(const SliderMessage *);
		handle_received_message(&received);
	}

	/*
	 * errno == ENOMSG is the usual exit condition when IPC_NOWAIT is set:
	 * simply, there are no more messages in the queue; still we should
	 * check for all other error messages and issue a warning
	 */
	if (result < 0 && errno != ENOMSG)
		syscall_warning(result, "Unexpected error while receiving IPC message");
}

static void
handle_received_message(const SliderMessage *msg)
{
		static const char *receiver_message(SliderCommand);

		switch(msg->command)
		{
			case SET_SLIDER:
				write_slider(msg->slider, msg->value);
				break;
			default:
				err_printf("IPC message received: unrecognized command [%s]\n",
					receiver_message(msg->command));
				break;
		}
}

struct rmesg
{
	SliderCommand c;
	const char *message;
} receiver_messages[] =
{
	{	NO_COMMAND,			"NO COMMAND"	},
	{	BUILD_SLIDERS,		"BUILD SLIDERS" },
	{	SET_SLIDER,			"SET SLIDER" },
	{	SET_SLIDER_MIN, 	"SET SLIDER MIN" },
	{	SET_SLIDER_MAX,		"SET SLIDER MAX" },
	{	UNKNOWN_COMMAND,	"UNKNOWN COMMAND" }
};

static const char *
receiver_message(const SliderCommand c)
{
	register struct rmesg *p = receiver_messages;

	while (p->c != UNKNOWN_COMMAND)
	{
		if (c == p->c)
			break;
		++p;
	}

	return p->message;
}

/*
 * transmitter functions
 */

inline static void
send_slider(const SliderMessage *message)
{
	/*
	 * we use a blocking send, in the hope that the graphic process
	 * will be fast enough in picking it up... (but this might not
	 * be a good idea - to be checked); at any rate, in this way
	 * the msgsnd() call should not fail.
	 */
#if 0
	syscall_warning(msgsnd(squeue_id(), (struct msgbuf *) message,
					sizeof(*message), 0), "send_slider() failed");
#endif
}

inline static void
set_slider(SliderCommand command, int slider, int value)
{
	const SliderMessage sent =
	{
		CSOUND_SLIDER_TYPE,
		command,
		slider,
		value
	};

	send_slider(&sent);
}

inline static void
set_slider_value(int slider, int value)
{
	set_slider(SET_SLIDER, slider, value);
}

inline static void
set_slider_min(int slider, int value)
{
	set_slider(SET_SLIDER_MIN, slider, value);
}

inline static void
set_slider_max(int slider, int value)
{
	set_slider(SET_SLIDER_MAX, slider, value);
}

inline static void
build_sliders(int nsliders)
{
	set_slider(BUILD_SLIDERS, nsliders, 0);
}
