/* sliders and other MIDI opcodes by Gabriel Maldonado */

  /* Code shortened at the cost of a little data space by amalgamating 
     8, 16 32 and 64 options -- JPff  */

#include "cs.h" 
#include "midiops.h"
#include "midiops3.h"
#include <math.h>

#define f7bit  		(127.0f)
#define oneTOf7bit      (float)(1./127.)
#define f14bit  	(16383.0f)
#define oneTOf14bit     (float)(1/16383.)
#define f21bit 		(2097151.0f)
#define oneTOf21bit	(float)(1/2097151.)

extern MCHNBLK *m_chnbp[16];

static void slideri_init(SLIDER *p, int n)
{
    char chan = p->slchan = (char)((*p->ichan)-1);
    char sbuf[120];
    if (chan < 0 || chan  > 15)  {
      initerror("illegal channel");
      return;
    }
    { 
      float value;
      int j = 0;
      SLD *sld = p->s;
      char *slnum = p->slnum;
      float *min = p->min, *max= p->max;
      FUNC **ftp = p->ftp; 
      float *chanblock = (float *) m_chnbp[chan]->ctl_val;
      
      while (j++ < n) {
    	if ((*slnum = (char) *sld->ictlno) < 0 || *slnum > 127) {
          sprintf(sbuf, "illegal control number at position n.%d", j);
          initerror(sbuf);
          break;
        }
        if ((value=*sld->initvalue) < (*min=*sld->imin) ||
            value > (*max=*sld->imax) ) {
          sprintf(sbuf, "illegal initvalue at position n.%d", j);
          initerror(sbuf);
          break;
        }
        if (*sld->ifn > 0)   *ftp++ = ftfind(sld->ifn); 
        else                 *ftp++ = NULL;	
        value =  (*(sld++)->initvalue - *min) / (*max++ - *min++);
        chanblock[*slnum++] =  (float)((int)(value * f7bit + 0.5f));
      }
    }
}

static void slider_init(SLIDER *p, int n)
{
    float value;
    int j = 0;
    FUNC **ftp = p->ftp-1;
    float *chanblock = (float *) m_chnbp[p->slchan]->ctl_val; 
    char  *slnum = p->slnum;
    float *min = p->min, *max = p->max;
    float **result = p->r;

    while (j++ < n) {
      value = (float) (chanblock[*slnum++] * oneTOf7bit);
      if (*(++ftp))             /* if valid ftable,use value as index   */    
        value = *((*ftp)->ftable + (long)(value * (*ftp)->flen));
                                /* no interpolation */
      **result++ = value * (*max++ - *min) + *min++;   /* scales the output */
    }
}


/*--------------------------------------------------------*/

void slider_i8(SLIDER *p)
{
    slideri_init(p, 8);
}


void slider8(SLIDER *p)
{
    slider_init(p, 8);
}

void slider_i16(SLIDER *p)
{
    slideri_init(p, 16);
}


void slider16(SLIDER *p)
{
    slider_init(p, 16);
}

void slider_i32(SLIDER *p)
{
    slideri_init(p, 32);
}


void slider32(SLIDER *p)
{
    slider_init(p, 32);
}

void slider_i64(SLIDER *p)
{
    slideri_init(p, 64);
}

void slider64(SLIDER *p)
{
    slider_init(p, 64);
}


/*==============================*/
static void sliderif(SLIDERf *p, int n)
{
    char chan = p->slchan = (char)((*p->ichan)-1);
    char sbuf[120];
    if (chan < 0 || chan  > 15)  {
      initerror("illegal channel");
      return;
    }
    { 
      float value;
      int j = 0;
      SLDf *sld = p->s;
      char *slnum = p->slnum;
      float *min = p->min, *max= p->max;
      FUNC **ftp = p->ftp; 
      float	b;
      float *yt1 = p->yt1, *c1=p->c1, *c2=p->c2;
      float *chanblock = (float *) m_chnbp[chan]->ctl_val;
      while (j++ < 8) {
    	if ((*slnum = (char) *sld->ictlno) < 0 || *slnum > 127) {
          sprintf(sbuf, "illegal control number at position n.%d", j);
          initerror(sbuf);
          break;
        }
        if ((value=*sld->initvalue) < (*min=*sld->imin) ||
            value > (*max=*sld->imax) ) {
          sprintf(sbuf, "illegal initvalue at position n.%d", j);
          initerror(sbuf);
          break;
        }
        if (*sld->ifn > 0)   *ftp++ = ftfind(sld->ifn); 
        else                 *ftp++ = NULL;	
        value =  (*sld->initvalue - *min) / (*max++ - *min++);
        chanblock[*slnum++] =  (float)(int)(value * f7bit + .5);

		/*----- init filtering coeffs*/
        *yt1++ = 0.0f;
        b = (float)(2.0 - cos((double)(*(sld++)->ihp * tpidsr * ksmps)));
        *c2 = (float)(b - sqrt((double)(b * b - 1.0f)));
        *c1++ = 1.0f - *c2++;
      }
    }
}

void sliderf(SLIDERf *p, int n)
{
    float value;
    int j = 0;
    FUNC **ftp = p->ftp-1;
    float *chanblock = (float *) m_chnbp[p->slchan]->ctl_val; 
    char  *slnum = p->slnum;
    float *min = p->min, *max = p->max;
    float **result = p->r;
    float *yt1 = p->yt1, *c1=p->c1, *c2=p->c2;

    while (j++ < n) {
      value = chanblock[*slnum++] * oneTOf7bit;
      if (*(++ftp))             /* if valid ftable,use value as index   */    
        value = *( (*ftp)->ftable + (long)(value * (*ftp)->flen)); 
      value = value * (*max++ - *min) + *min++; 	/* scales the output */
      **result++ =
        *yt1++ = *c1++ * value + *c2++ * *yt1; /* filters the output */
      
      /* yt1++; c1++;  c2++; */
    }
}

void slider_i8f(SLIDERf *p)
{
    sliderif(p, 8);
}

void slider8f(SLIDERf *p)
{
    sliderf(p, 8);
}

void slider_i16f(SLIDERf *p)
{
    sliderif(p, 16);
}

void slider16f(SLIDERf *p)
{
    sliderf(p, 16);
}

void slider_i32f(SLIDERf *p)
{
    sliderif(p, 32);
}

void slider32f(SLIDERf *p)
{
    sliderf(p, 32);
}


void slider_i64f(SLIDERf *p)
{
    sliderif(p, 64);
}

void slider64f(SLIDERf *p)
{
    sliderf(p, 64);
}

/*===================================*/

static void islider(ISLIDER *p, int n)
{
    char chan= (char) ((*p->ichan)-1);
    char sbuf[120];
    if (chan < 0 || chan  > 15)  {
      initerror("illegal channel");
      return;
    }
    { 
      float value;
      int j = 0;
      ISLD *sld = p->s;
      char slnum;
      float *chanblock = (float *) m_chnbp[chan]->ctl_val; 
      FUNC *ftp;
      float **result = p->r;
   
      while (j++ < n) {
    	if ((slnum=(char) *sld->ictlno) < 0 || slnum > 127) {
          sprintf(sbuf, "illegal control number at position n.%d", j);
          initerror(sbuf);
          break;
        }
        value = chanblock[slnum] * oneTOf7bit;
        if (*sld->ifn > 0)  {
          ftp = ftfind(sld->ifn); 
          value = *( ftp->ftable + (long)(value * ftp->flen));
                                /* no interpolation */
        }
        **result++ = value * (*sld->imax - *sld->imin) + *sld->imin;   /* scales the output */
        sld++;
      }
    }
}


void islider8(ISLIDER *p)
{
    islider(p, 8);
}

void islider16(ISLIDER *p)
{
    islider(p, 16);
}

void islider32(ISLIDER *p)
{
    islider(p, 32);
}

void islider64(ISLIDER *p)
{
    islider(p, 64);
}



/*-------------------------------*/

static void slideri14(SLIDERBIT14 *p, int n)
{
    char chan= p->slchan = (char)((*p->ichan)-1);
    char sbuf[120];
    if (chan < 0 || chan  > 15)  {
      initerror("illegal channel");
      return;
    }
    { 
      float value;
      int intvalue, j = 0;
      SLD14 *sld = p->s;
      char *slnum_msb = p->slnum_msb;
      char *slnum_lsb = p->slnum_lsb;
      float *min = p->min, *max= p->max;
      FUNC **ftp = p->ftp; 
      float *chanblock = (float *) m_chnbp[chan]->ctl_val;

      while (j++ < n) {
    	if ((*slnum_msb = (char)*sld->ictlno_msb) < 0 || *slnum_msb > 127) {
          sprintf(sbuf, "illegal msb control number at position n.%d", j);
          initerror(sbuf);
          break;
        }
    	if ((*slnum_lsb = (char)*sld->ictlno_lsb) < 0 || *slnum_lsb > 127) {
          sprintf(sbuf, "illegal lsb control number at position n.%d", j);
          initerror(sbuf);
          break;
        }
        if ((value=*sld->initvalue) < (*min=*sld->imin) ||
            value > (*max=*sld->imax) ) {
          sprintf(sbuf, "illegal initvalue at position n.%d", j);
          initerror(sbuf);
          break;
        }
        if (*sld->ifn > 0)   *ftp++ = ftfind(sld->ifn); 
        else                 *ftp++ = NULL;	
        intvalue = (int) (((*(sld++)->initvalue - *min) / (*max++ - *min++))
                          * f14bit+0.5f);
        chanblock[*slnum_msb++] =  (float) (intvalue >> 7);
        chanblock[*slnum_lsb++] =  (float) (intvalue & 0x7f);
      }
    }
}

static void slider14(SLIDERBIT14 *p, int n)
{
    float value;
    int j = 0;
    FUNC **ftp = p->ftp-1;
    float *chanblock = (float *) m_chnbp[p->slchan]->ctl_val; 
    char  *slnum_msb = p->slnum_msb;
    char  *slnum_lsb = p->slnum_lsb;
    float *min = p->min, *max = p->max;
    float **result = p->r;

    while (j++ < n) {
      value = (float)((chanblock[*slnum_msb++]  * 128 
		       + chanblock[*slnum_lsb++]) * oneTOf14bit); 
      if (*(++ftp)) {           /* if valid ftable,use value as index   */    
        float phase = value * (*ftp)->flen;
        float *base = (*ftp)->ftable + (long)(phase);
        value = *base + (*(base+1) - *base) * (phase - (long) phase);
      }			
      **result++ = value * (*max++ - *min) + *min++;   /* scales the output */
    }
}

void slider_i16bit14(SLIDERBIT14 *p)
{
    slideri14(p, 16);
}

void slider16bit14(SLIDERBIT14 *p)
{
    slider14(p, 16);
}

void slider_i32bit14(SLIDERBIT14 *p)
{
    slideri14(p, 32);
}


void slider32bit14(SLIDERBIT14 *p)
{
    slider14(p, 32);
}

/*--------------------------------*/
void islider14(ISLIDERBIT14 *p, int n)
{
    char chan = (char)((*p->ichan)-1);
    char sbuf[120];
    if (chan < 0 || chan  > 15)  {
      initerror("illegal channel");
      return;
    }
    { 
      float value;
      int j = 0;
      ISLD14 *sld = p->s;
      char slnum_msb;
      char slnum_lsb;
      float *chanblock = (float *) m_chnbp[chan]->ctl_val; 
      float **result = p->r;
   
      while (j++ < n) {
    	if ((slnum_msb=(char)*sld->ictlno_msb) < 0 || slnum_msb > 127) {
          sprintf(sbuf, "illegal msb control number at position n.%d", j);
          initerror(sbuf);
          break;
        }
    	if ((slnum_lsb=(char)*sld->ictlno_lsb) < 0 || slnum_lsb > 127) {
          sprintf(sbuf, "illegal lsb control number at position n.%d", j);
          initerror(sbuf);
          break;
        }

        value =	(float)((chanblock[slnum_msb]  * 128 
                         + chanblock[slnum_lsb]) * oneTOf14bit); 
        if (*sld->ifn > 0) {    /* linear interpolation routine */    
          FUNC *ftp= ftfind(sld->ifn); 
          float phase = value * ftp->flen;
          float *base = ftp->ftable + (long)(phase);
          value = *base + (*(base + 1) - *base) * (phase - (long) phase);
        }
                                /* scales the output */
        **result++ = value * (*sld->imax - *sld->imin) + *sld->imin;
        sld++;
      }
    }
}

void islider16bit14(ISLIDERBIT14 *p)
{
    islider14(p, 16);
}

void islider32bit14(ISLIDERBIT14 *p)
{
    islider14(p, 16);
}
